/*

sshproxykey.c

Authors: Vesa Suontama     <vsuontam@ssh.fi>
         Jukka Aittokallio <jai@ssh.fi>
         Patrick Irwin     <irwin@ssh.fi>

Copyright (c) 1999-2002 SSH Communications Security Corp, Helsinki, Finland
              All rights reserved

*/

#include "sshincludes.h"
#include "sshproxykey.h"
#include "sshpk.h"
#include "sshrgf.h"
#include "sshgetput.h"

#define SSH_DEBUG_MODULE "SshProxyKey"

#define KEY_SIZE_TO_BYTES(x) ((((x) + 7) >> 3))

/* Declare the proxy keys defined later on this file. */
const SshPkType ssh_proxy_key_dl_modp;
const SshPkType ssh_proxy_key_dl_modp_dummy_rgf;
#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
const SshPkType ssh_proxy_key_if_modn;
const SshPkType ssh_proxy_key_if_modn_dummy_rgf;
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */

Boolean ssh_proxy_register(const SshPkType *type)
{
  if (ssh_pk_provider_register(type) != SSH_CRYPTO_OK)
    {
      SSH_DEBUG(0, ("Could not register proxy type"));
      return FALSE;
    }
    return TRUE;
}


#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
typedef struct ProxyRSAKeyRec {
  Boolean cb_does_rgf;
  SshUInt32 key_size;

  /* used for public key verify when the  callback does the rgf. */ 
  SshProxyVerifyOpCB rgf_verify_op;

  /* used to sign for private keys and to verify for public keys. */ 
  SshProxyKeyOpCB signature_op;

  /* used to decrypt for private keys and to encrypt for public keys. */ 
  SshProxyKeyOpCB encryption_op;

  SshProxyFreeOpCB free_op;

  SshUInt32 ref_count;
  void *context;
} *ProxyRSAKey;


typedef struct ProxyRSAKeySignContextRec {
  SshOperationHandle op;
  SshOperationHandle sub_op;
  ProxyRSAKey key;
  unsigned char *raw_data;
  size_t raw_data_len;
  SshPrivateKeySignCB callback;
  void *context;
} *ProxyRSAKeySignContext;


/* This function returns the padding mechanism used in a rgf scheme. 
   We just do this statically now.*/
static const char * get_pad_name(const char *rgf_name)
{
  if (strstr(rgf_name, "pkcs1v2-oaep"))
    return "rsa-pkcs1v2-oaep-none";
  else if  (strstr(rgf_name, "pkcs1"))
    return "rsa-pkcs1-none";
  else 
    return "rsa-none-none";
}

/************************ The RSA Signature Scheme. *********************/

void ssh_proxy_rsa_sign_abort(void *context)
{
  ProxyRSAKeySignContext ctx = context;

  ssh_operation_abort(ctx->sub_op);

  ssh_free(ctx->raw_data);
  ssh_free(ctx);
}

void ssh_proxy_rsa_sign_free(void *context)
{
  ProxyRSAKeySignContext ctx = context;
  
  ssh_operation_unregister(ctx->op);
  ssh_proxy_rsa_sign_abort(ctx);
}

void ssh_proxy_rsa_sign_op_done(SshCryptoStatus status,
                                const unsigned char *data,
                                size_t data_len,
                                void *context)
{
  ProxyRSAKeySignContext sign_ctx = context;

  sign_ctx->sub_op = NULL;

  (*sign_ctx->callback)(status, data, data_len, sign_ctx->context);
  
  ssh_proxy_rsa_sign_free(sign_ctx);
}

SshOperationHandle
ssh_proxy_rsa_sign_async(const void *private_key,
                         const char *rgf_name,
                         SshRGFHash hash,
                         SshPrivateKeySignCB callback,
                         void *context)
{
  ProxyRSAKeySignContext sign_ctx;
  SshOperationHandle sub_op;
  ProxyRSAKey key = (ProxyRSAKey)private_key;
  const char *name = "rsa-none-none";

  /* Check that the sign is supported */
  if (key->signature_op == NULL_FNPTR)
    {
      if (callback)
        (*callback)(SSH_CRYPTO_UNSUPPORTED, NULL, 0, context);
      return NULL;
    }
  
  if ((sign_ctx = ssh_calloc(1, sizeof(*sign_ctx))) != NULL)
    {
      sign_ctx->callback = callback;
      sign_ctx->context = context;
      sign_ctx->key = (ProxyRSAKey)private_key;

      /* This is a potential problem when hash is the rgf dummy. We have 
         to malloc again the data buffer. */
      if (sign_ctx->key->cb_does_rgf)
          sign_ctx->raw_data_len = ssh_rgf_hash_digest_length(hash);
      else
          sign_ctx->raw_data_len = KEY_SIZE_TO_BYTES(sign_ctx->key->key_size); 

      SSH_DEBUG(SSH_D_HIGHOK, ("Are now mallocating %d bytes.", 
                               sign_ctx->raw_data_len));   

      sign_ctx->raw_data = ssh_malloc(sign_ctx->raw_data_len);
      sign_ctx->op = NULL;
      sign_ctx->sub_op = NULL;

      /* The callback does the padding operation if cb_does_rgf is 
         TRUE in the proxykey. In addition, if the data in the RGF object 
         has not already been hashed the callback does the hashing operation 
         also. */ 
      if (sign_ctx->key->cb_does_rgf)
        {     
          /* If the hash is already done, we only want the pad name. */
          if (ssh_rgf_data_is_digest(hash))
            name = get_pad_name(rgf_name);
          else    
            name = rgf_name;
        }
      
      if (sign_ctx->raw_data == NULL ||
          ssh_rgf_hash_sign(hash,
                            sign_ctx->raw_data,
                            sign_ctx->raw_data_len) != SSH_RGF_OK)
        {
          (*callback)(SSH_CRYPTO_OPERATION_FAILED, NULL, 0, context);
          ssh_proxy_rsa_sign_free(sign_ctx);

          return NULL;
        }

      sign_ctx->op = ssh_operation_register(ssh_proxy_rsa_sign_abort,
                                            sign_ctx);

      sub_op = (*key->signature_op)(name, 
                                    sign_ctx->raw_data,
                                    sign_ctx->raw_data_len,
                                    ssh_proxy_rsa_sign_op_done,
                                    sign_ctx,
                                    key->context);

      if (sub_op)
        {
          sign_ctx->sub_op = sub_op;
          return sign_ctx->op;
        }
      return NULL;
    }
  else
    {
      (*callback)(SSH_CRYPTO_NO_MEMORY, NULL, 0, context);
      return NULL;
    }
}


SshOperationHandle
ssh_proxy_rsa_sign_none_none_async(const void *private_key,
                                   SshRGFHash hash,
                                   SshPrivateKeySignCB callback,
                                   void *context)
{
return ssh_proxy_rsa_sign_async(private_key,
                                "rsa-none-none",
                                hash,
                                callback,
                                context);

}    

SshOperationHandle
ssh_proxy_rsa_sign_pkcs1_sha1_async(const void *private_key,
                                   SshRGFHash hash,
                                   SshPrivateKeySignCB callback,
                                   void *context)
{
  return ssh_proxy_rsa_sign_async(private_key,
                                  "rsa-pkcs1-sha1",
                                  hash,
                                  callback,
                                  context);
}    


SshOperationHandle
ssh_proxy_rsa_sign_pkcs1_md5_async(const void *private_key,
                                   SshRGFHash hash,
                                   SshPrivateKeySignCB callback,
                                   void *context)
{
  return ssh_proxy_rsa_sign_async(private_key,
                                  "rsa-pkcs1-md5",
                                  hash,
                                  callback,
                                  context);
}    


SshOperationHandle
ssh_proxy_rsa_sign_pkcs1_md2_async(const void *private_key,
                                   SshRGFHash hash,
                                   SshPrivateKeySignCB callback,
                                   void *context)
{
  return ssh_proxy_rsa_sign_async(private_key,
                                  "rsa-pkcs1-md2",
                                  hash,
                                  callback,
                                  context);
}    

SshOperationHandle
ssh_proxy_rsa_sign_pkcs1_none_async(const void *private_key,
                                    SshRGFHash hash,
                                    SshPrivateKeySignCB callback,
                                    void *context)
{
  return ssh_proxy_rsa_sign_async(private_key,
                                  "rsa-pkcs1-none",
                                  hash,
                                  callback,
                                  context);
}    





/************************* The RSA Verification Scheme. *******************/


typedef struct ProxyRSAKeyVerifyContextRec {
  SshOperationHandle op;
  SshOperationHandle sub_op;
  ProxyRSAKey key;
  SshRGFHash hash;
  unsigned char *signature;
  size_t signature_len;
  unsigned char *raw_data;
  size_t raw_data_len;
  SshPublicKeyVerifyCB callback;
  void *context;
} *ProxyRSAKeyVerifyContext;


void ssh_proxy_rsa_verify_abort(void *context)
{
  ProxyRSAKeyVerifyContext ctx = context;

  ssh_operation_abort(ctx->sub_op);

  ssh_free(ctx->raw_data);
  ssh_free(ctx->signature);
  ssh_free(ctx);
}

void ssh_proxy_rsa_verify_free(void *context)
{
  ProxyRSAKeyVerifyContext ctx = context;
 
  ssh_operation_unregister(ctx->op);
  ssh_proxy_rsa_verify_abort(ctx);
}

/* Called if the proxy operation does not do the RGF. We have do the RGF 
   here and then validate the signature. */ 
void ssh_proxy_rsa_verify_op_done(SshCryptoStatus status,
                                  const unsigned char *data,
                                  size_t data_len,
                                  void *context)
{
  ProxyRSAKeyVerifyContext verify_ctx = context;
  unsigned char *output; 
  size_t output_len; 
 
  verify_ctx->sub_op = NULL; 
 
  if (status != SSH_CRYPTO_OK)
    { 
      SSH_DEBUG(SSH_D_FAIL, ("The RSA verify operation has failed."));
      (*verify_ctx->callback)(status, 
                              NULL, 
                              verify_ctx->context);
      
      ssh_proxy_rsa_verify_free(verify_ctx);
      return;  
    }

  if (ssh_rgf_hash_verify(verify_ctx->hash,
                          data,
                          data_len,
                          KEY_SIZE_TO_BYTES(verify_ctx->key->key_size), 
                          &output,
                          &output_len) != SSH_RGF_OK)
    {
      SSH_DEBUG(SSH_D_FAIL, ("The RGF hash verify operation has failed."));

      (*verify_ctx->callback)(SSH_CRYPTO_SIGNATURE_CHECK_FAILED,
                              NULL, 
                              verify_ctx->context);

      ssh_free(output);
      ssh_proxy_rsa_verify_free(verify_ctx);

      return;
    }
  else
    {   
      (*verify_ctx->callback)(SSH_CRYPTO_OK,
                              NULL, 
                              verify_ctx->context);

      ssh_free(output); 
      ssh_proxy_rsa_verify_free(verify_ctx);
    }
}

/* Called if the proxy operation does the RGF. The verification is already 
   completed. */ 
void ssh_proxy_rsa_verify_rgf_op_done(SshCryptoStatus status, void *context)
{
  ProxyRSAKeyVerifyContext verify_ctx = context;
  
  verify_ctx->sub_op = NULL; 
  
  (*verify_ctx->callback)(status,
                          NULL, 
                          verify_ctx->context);
  
  ssh_proxy_rsa_verify_free(verify_ctx);
}

SshOperationHandle
ssh_proxy_rsa_verify_async(const void *public_key,
                           const char *rgf_name,
                           const unsigned char *signature,
                           size_t signature_len,
                           SshRGFHash hash,
                           SshPublicKeyVerifyCB callback,
                           void *context)
{
  ProxyRSAKeyVerifyContext verify_ctx;
  SshOperationHandle sub_op;
  ProxyRSAKey key = (ProxyRSAKey) public_key;
  const char *name = "rsa-none-none";

  if ((key->cb_does_rgf == TRUE  && key->rgf_verify_op == NULL_FNPTR) ||
      (key->cb_does_rgf == FALSE && key->signature_op == NULL_FNPTR))
    {
      if (callback)
        (*callback)(SSH_CRYPTO_UNSUPPORTED, NULL, context);
      return NULL;
    }

  if ((verify_ctx = ssh_calloc(1, sizeof(*verify_ctx))) != NULL)
    {
      verify_ctx->callback = callback;
      verify_ctx->context = context;
      verify_ctx->hash = hash;
      verify_ctx->key = key;
      verify_ctx->signature_len = signature_len;
      verify_ctx->signature = ssh_memdup(signature, signature_len);
      verify_ctx->raw_data = NULL;
      verify_ctx->raw_data_len = 0;
      verify_ctx->op = NULL;
      verify_ctx->sub_op = NULL;
      
      if (verify_ctx->signature == NULL)
        {
          (*callback)(SSH_CRYPTO_NO_MEMORY, NULL, context);
          ssh_proxy_rsa_verify_free(verify_ctx);
       
          return NULL;
        }

      verify_ctx->op = ssh_operation_register(ssh_proxy_rsa_verify_abort,
                                              verify_ctx);
      
      if (key->cb_does_rgf == FALSE)
        {
        /* We pass the RGF Hash to the callback. The RGF is done in the 
           reply callback. */
        sub_op = (*key->signature_op)(name, 
                                      signature, 
                                      signature_len,
                                      ssh_proxy_rsa_verify_op_done,
                                      verify_ctx,
                                      key->context);
        }
      else
        {
          /* We pass the unhashed data to the callback. The RGF is the dummy
             RGF. This is a potential problem when hash is the rgf dummy. We 
             have to malloc again the data buffer. */
          verify_ctx->raw_data_len  =  ssh_rgf_hash_digest_length(hash);
          verify_ctx->raw_data = ssh_malloc(verify_ctx->raw_data_len);
          
          /* If the hash is already done, only pass the pad name to the cb. */
          if (ssh_rgf_data_is_digest(hash))
            name = get_pad_name(rgf_name);
          else    
            name = rgf_name;
          
          SSH_DEBUG(SSH_D_LOWSTART, ("The rgf name is %s", name));
          
          if (verify_ctx->raw_data == NULL || 
              ssh_rgf_hash_sign(hash, 
                                verify_ctx->raw_data, 
                                verify_ctx->raw_data_len) != SSH_RGF_OK)
            {
              (*callback)(SSH_CRYPTO_OPERATION_FAILED, NULL, context);
              ssh_proxy_rsa_verify_free(verify_ctx);

              return NULL;
            }
          
          sub_op = 
            (*key->rgf_verify_op)(name, 
                                  verify_ctx->raw_data, 
                                  verify_ctx->raw_data_len,
                                  signature, 
                                  signature_len,
                                  ssh_proxy_rsa_verify_rgf_op_done,
                                  verify_ctx, 
                                  key->context);
        }
      
      if (sub_op)
        {
          verify_ctx->sub_op = sub_op;
          return verify_ctx->op;
        }
      return NULL;
    }
  else
    {
      (*callback)(SSH_CRYPTO_NO_MEMORY, NULL, context);
      return NULL;
    }
}


SshOperationHandle
ssh_proxy_rsa_verify_none_none_async(const void *public_key,
                                     const unsigned char *signature,
                                     size_t signature_len,
                                     SshRGFHash hash,
                                     SshPublicKeyVerifyCB callback,
                                     void *context)
{
  return ssh_proxy_rsa_verify_async(public_key,
                                    "rsa-none-none",
                                    signature,
                                    signature_len,
                                    hash,
                                    callback,
                                    context);
}    

SshOperationHandle
ssh_proxy_rsa_verify_pkcs1_sha1_async(const void *public_key,
                                      const unsigned char *signature,
                                      size_t signature_len,
                                      SshRGFHash hash,
                                      SshPublicKeyVerifyCB callback,
                                      void *context)
{
  return ssh_proxy_rsa_verify_async(public_key,
                                    "rsa-pkcs1-sha1",
                                    signature,
                                    signature_len,
                                    hash,
                                    callback,
                                    context);
}    

SshOperationHandle
ssh_proxy_rsa_verify_pkcs1_md5_async(const void *public_key,
                                     const unsigned char *signature,
                                     size_t signature_len,
                                     SshRGFHash hash,
                                     SshPublicKeyVerifyCB callback,
                                     void *context)
{
  return ssh_proxy_rsa_verify_async(public_key,
                                    "rsa-pkcs1-md5",
                                    signature,
                                    signature_len,
                                    hash,
                                    callback,
                                    context);
}    
SshOperationHandle
ssh_proxy_rsa_verify_pkcs1_md2_async(const void *public_key,
                                     const unsigned char *signature,
                                     size_t signature_len,
                                     SshRGFHash hash,
                                     SshPublicKeyVerifyCB callback,
                                     void *context)
{
  return ssh_proxy_rsa_verify_async(public_key,
                                    "rsa-pkcs1-md2",
                                    signature,
                                    signature_len,
                                    hash,
                                    callback,
                                    context);
}    

SshOperationHandle
ssh_proxy_rsa_verify_pkcs1_none_async(const void *public_key,
                                      const unsigned char *signature,
                                      size_t signature_len,
                                      SshRGFHash hash,
                                      SshPublicKeyVerifyCB callback,
                                      void *context)
{
  return ssh_proxy_rsa_verify_async(public_key,
                                    "rsa-pkcs1-none",
                                    signature,
                                    signature_len,
                                    hash,
                                    callback,
                                    context);
}    
     


/************************* The RSA Encryption Scheme. *********************/

typedef struct ProxyRSAKeyEncryptContextRec {
  SshOperationHandle op;
  SshOperationHandle sub_op;
  ProxyRSAKey key;
  SshPublicKeyEncryptCB callback;
  unsigned char *raw_data;
  size_t raw_data_len;
  void *context;
} *ProxyRSAKeyEncryptContext;



void ssh_proxy_rsa_encrypt_abort(void *context)
{
  ProxyRSAKeyEncryptContext ctx = context;
 
  ssh_operation_abort(ctx->sub_op);
  ssh_free(ctx->raw_data);
  ssh_free(ctx);
}

void ssh_proxy_rsa_encrypt_free(void *context)
{
  ProxyRSAKeyEncryptContext ctx = context;

  ssh_operation_unregister(ctx->op);
  ssh_proxy_rsa_encrypt_abort(context);
}

void ssh_proxy_rsa_encrypt_op_done(SshCryptoStatus status,
                                   const unsigned char *data,
                                   size_t data_len,
                                   void *context)
{
  ProxyRSAKeyEncryptContext encrypt_ctx = context;
  
  encrypt_ctx->sub_op = NULL;
  
  (*encrypt_ctx->callback)(status,
                           data, 
                           data_len,
                           encrypt_ctx->context);

  ssh_proxy_rsa_encrypt_free(encrypt_ctx);

}

SshOperationHandle
ssh_proxy_rsa_encrypt_async(const void *public_key,
                            const char *rgf_name,
                            const unsigned char *plaintext,
                            size_t plaintext_len,
                            SshRGFHash hash,
                            SshPublicKeyEncryptCB callback,
                            void *context)
{
  ProxyRSAKeyEncryptContext encrypt_ctx;
  SshOperationHandle sub_op;
  ProxyRSAKey key = (ProxyRSAKey)public_key;

  if (key->encryption_op == NULL_FNPTR)
    {
      if (callback)
        (*callback)(SSH_CRYPTO_UNSUPPORTED, NULL, 0, context);
      return NULL;
    }
  
  if ((encrypt_ctx = ssh_calloc(1, sizeof(*encrypt_ctx))) != NULL)
    {
      encrypt_ctx->callback = callback;
      encrypt_ctx->context = context;
      encrypt_ctx->key = key;

      /* If the rgf is done in the callback, we don't want to pad the data 
         to the key size here, because the rgf usually adds extra padding. */ 
      if (encrypt_ctx->key->cb_does_rgf)
        {
          encrypt_ctx->raw_data_len = plaintext_len; 
        }
      else
        { 
          encrypt_ctx->raw_data_len = 
            KEY_SIZE_TO_BYTES(encrypt_ctx->key->key_size); 
        }      

      encrypt_ctx->raw_data = ssh_malloc(encrypt_ctx->raw_data_len);
      encrypt_ctx->op = NULL;
      encrypt_ctx->sub_op = NULL;
      
      /* The callback does the padding only if cb_does_rgf is TRUE 
         in the proxykey. */ 
      SSH_DEBUG(SSH_D_LOWSTART, ("The rgf name is %s\n", rgf_name));
      
      if (encrypt_ctx->raw_data == NULL ||
          ssh_rgf_hash_encrypt(hash, 
                               plaintext, 
                               plaintext_len, 
                               encrypt_ctx->raw_data,
                               encrypt_ctx->raw_data_len) != SSH_RGF_OK)
        {
          (*callback)(SSH_CRYPTO_OPERATION_FAILED, NULL, 0, context);
          ssh_proxy_rsa_encrypt_free(encrypt_ctx);

          return NULL;
        }
      
      encrypt_ctx->op = ssh_operation_register(ssh_proxy_rsa_encrypt_abort,
                                               encrypt_ctx);
      
      sub_op = (*key->encryption_op)(rgf_name,
                                     encrypt_ctx->raw_data,
                                     encrypt_ctx->raw_data_len,
                                     ssh_proxy_rsa_encrypt_op_done,
                                     encrypt_ctx,
                                     key->context);
      
      if (sub_op)
        {
          encrypt_ctx->sub_op = sub_op;
          return encrypt_ctx->op;
        }
      return NULL;
    }
  else
    {
      (*callback)(SSH_CRYPTO_NO_MEMORY, NULL, 0, context);
      return NULL;
    }
}

SshOperationHandle
ssh_proxy_rsa_encrypt_none_none_async(const void *public_key,
                                      const unsigned char *plaintext,
                                      size_t plaintext_len,
                                      SshRGFHash hash,
                                      SshPublicKeyEncryptCB callback,
                                      void *context)
{
  return ssh_proxy_rsa_encrypt_async(public_key,
                                     "rsa-none-none",
                                     plaintext,
                                     plaintext_len,
                                     hash,
                                     callback,
                                     context);
}


SshOperationHandle
ssh_proxy_rsa_encrypt_pkcs1v2_oaep_async(const void *public_key,
                                         const unsigned char *plaintext,
                                         size_t plaintext_len,
                                         SshRGFHash hash,
                                         SshPublicKeyEncryptCB callback,
                                         void *context)
{
  return ssh_proxy_rsa_encrypt_async(public_key,
                                     "rsa-pkcs1v2-oaep",
                                     plaintext,
                                     plaintext_len,
                                     hash,
                                     callback,
                                     context);
}

SshOperationHandle
ssh_proxy_rsa_encrypt_pkcs1_none_async(const void *public_key,
                                       const unsigned char *plaintext,
                                       size_t plaintext_len,
                                       SshRGFHash hash,
                                       SshPublicKeyEncryptCB callback,
                                       void *context)
{
  return ssh_proxy_rsa_encrypt_async(public_key,
                                     "rsa-pkcs1-none",
                                     plaintext,
                                     plaintext_len,
                                     hash,
                                     callback,
                                     context);
}

/************************* The RSA Decryption Scheme. ***********************/

typedef struct ProxyRSAKeyDecryptContextRec {
  SshOperationHandle op;
  SshOperationHandle sub_op;
  ProxyRSAKey key;
  SshRGFHash hash;
  SshPrivateKeyDecryptCB callback;
  unsigned char *raw_data;
  size_t raw_data_len;
  void *context;
} *ProxyRSAKeyDecryptContext;



void ssh_proxy_rsa_decrypt_abort(void *context)
{
  ProxyRSAKeyDecryptContext ctx = context;
 
  ssh_operation_abort(ctx->sub_op);

  ssh_free(ctx->raw_data);
  ssh_free(ctx);

}

void ssh_proxy_rsa_decrypt_free(void *context)
{
  ProxyRSAKeyDecryptContext ctx = context;

  ssh_operation_unregister(ctx->op);
  ssh_proxy_rsa_decrypt_abort(context);
}

void ssh_proxy_rsa_decrypt_op_done(SshCryptoStatus status,
                                   const unsigned char *data,
                                   size_t data_len,
                                   void *context)
{
  ProxyRSAKeyDecryptContext decrypt_ctx = context;
  unsigned char *output;
  size_t output_len;

  decrypt_ctx->sub_op = NULL;

  if (status != SSH_CRYPTO_OK)
    {
      SSH_DEBUG(SSH_D_FAIL, ("The RSA decrypt operation has failed"));
      (*decrypt_ctx->callback)(status,
                               NULL, 
                               0, 
                               decrypt_ctx->context);
 
      ssh_proxy_rsa_decrypt_free(decrypt_ctx);
      return;
    }

  if (ssh_rgf_hash_decrypt(decrypt_ctx->hash, 
                           data, 
                           data_len,
                           KEY_SIZE_TO_BYTES(decrypt_ctx->key->key_size),
                           &output, 
                           &output_len) != SSH_RGF_OK)
    {
      SSH_DEBUG(SSH_D_FAIL, ("The RGF decrypt operation has failed "));
      (*decrypt_ctx->callback)(SSH_CRYPTO_OPERATION_FAILED,
                               NULL, 
                               0, 
                               decrypt_ctx->context);

      ssh_free(output);
      ssh_proxy_rsa_decrypt_free(decrypt_ctx);

      return;
    }
  else
    {
      (*decrypt_ctx->callback)(SSH_CRYPTO_OK,
                               output, 
                               output_len,
                               decrypt_ctx->context);
      ssh_free(output);

      ssh_proxy_rsa_decrypt_free(decrypt_ctx);
    }
}

SshOperationHandle
ssh_proxy_rsa_decrypt_async(const void *private_key,
                            const char *rgf_name,
                            const unsigned char *ciphertext,
                            size_t ciphertext_len,
                            SshRGFHash hash,
                            SshPrivateKeyDecryptCB callback,
                            void *context)
{
  ProxyRSAKeyDecryptContext decrypt_ctx;
  SshOperationHandle sub_op;
  ProxyRSAKey key = (ProxyRSAKey)private_key;

  /* Check that the encryption callback was given */
  if (key->encryption_op == NULL_FNPTR)
    {
      if (callback)
        (*callback)(SSH_CRYPTO_UNSUPPORTED, NULL, 0, context);
      return NULL;
    }
                               
  
  if ((decrypt_ctx = ssh_calloc(1, sizeof(*decrypt_ctx))) != NULL)
    {
      decrypt_ctx->callback = callback;
      decrypt_ctx->context = context;
      decrypt_ctx->key = key;
      decrypt_ctx->hash = hash;
      decrypt_ctx->raw_data_len = ciphertext_len;
      decrypt_ctx->raw_data = ssh_memdup(ciphertext, ciphertext_len);
      decrypt_ctx->op = NULL;
      decrypt_ctx->sub_op = NULL;

      /* The callback does the padding operations 
         if cb_does_rgf is TRUE in the proxykey. */ 
      if (decrypt_ctx->key->cb_does_rgf)
          SSH_DEBUG(SSH_D_LOWSTART, ("The rgf name is %s\n", rgf_name));

      if (decrypt_ctx->raw_data == NULL)
        {
          (*callback)(SSH_CRYPTO_NO_MEMORY, NULL, 0, context);
          ssh_proxy_rsa_decrypt_free(decrypt_ctx);

          return NULL;
        }

      decrypt_ctx->op = ssh_operation_register(ssh_proxy_rsa_decrypt_abort,
                                               decrypt_ctx);
      
      sub_op = (*key->encryption_op)(rgf_name, 
                                     ciphertext, 
                                     ciphertext_len,
                                     ssh_proxy_rsa_decrypt_op_done,
                                     decrypt_ctx,
                                     key->context);
  
      if (sub_op)
        {
          decrypt_ctx->sub_op = sub_op;
          return decrypt_ctx->op;
        }
      return NULL;
    }
  else
    {
      (*callback)(SSH_CRYPTO_NO_MEMORY, NULL, 0, context);
      return NULL;
    }
}

SshOperationHandle
ssh_proxy_rsa_decrypt_none_none_async(const void *private_key,
                                      const unsigned char *ciphertext,
                                      size_t ciphertext_len,
                                      SshRGFHash hash,
                                      SshPrivateKeyDecryptCB callback,
                                      void *context)
{
return ssh_proxy_rsa_decrypt_async(private_key,
                                   "rsa-none-none",
                                   ciphertext,
                                   ciphertext_len,
                                   hash,
                                   callback,
                                   context);
}

SshOperationHandle
ssh_proxy_rsa_decrypt_pkcs1v2_oaep_async(const void *private_key,
                                         const unsigned char *ciphertext,
                                         size_t ciphertext_len,
                                         SshRGFHash hash,
                                         SshPrivateKeyDecryptCB callback,
                                         void *context)
{
return ssh_proxy_rsa_decrypt_async(private_key,
                                   "rsa-pkcs1v2-oaep",
                                   ciphertext,
                                   ciphertext_len,
                                   hash,
                                   callback,
                                   context);
}


SshOperationHandle
ssh_proxy_rsa_decrypt_pkcs1_none_async(const void *private_key,
                                       const unsigned char *ciphertext,
                                       size_t ciphertext_len,
                                       SshRGFHash hash,
                                       SshPrivateKeyDecryptCB callback,
                                       void *context)
{
return ssh_proxy_rsa_decrypt_async(private_key,
                                   "rsa-pkcs1-none",
                                   ciphertext,
                                   ciphertext_len,
                                   hash,
                                   callback,
                                   context);
}
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */

/* *************************** DSA Keys ***********************************/



typedef struct ProxyDSAKeyRec {
  Boolean cb_does_rgf;
  SshUInt32 key_size;
  SshProxyKeyOpCB sign_op;
  SshProxyVerifyOpCB verify_op;
  SshProxyFreeOpCB free_op;
  SshUInt32 ref_count;
  void *context;
} *ProxyDSAKey;


/************************* The DSA Signature Scheme. *********************/


typedef struct ProxyDSAKeySignContextRec {
  SshOperationHandle op;
  SshOperationHandle sub_op;
  ProxyDSAKey key;
  unsigned char *raw_data;
  size_t raw_data_len;
  SshPrivateKeySignCB callback;
  void *context;
} *ProxyDSAKeySignContext;

void ssh_proxy_dsa_sign_abort(void *context)
{
  ProxyDSAKeySignContext ctx = context;

  ssh_operation_abort(ctx->sub_op);

  ssh_free(ctx->raw_data);
  ssh_free(ctx);
}

void ssh_proxy_dsa_sign_free(void *context)
{
  ProxyDSAKeySignContext ctx = context;
  
  ssh_operation_unregister(ctx->op);
  ssh_proxy_dsa_sign_abort(ctx);
}

void ssh_proxy_dsa_sign_op_done(SshCryptoStatus status,
                                const unsigned char *data,
                                size_t data_len,
                                void *context)
{
  ProxyDSAKeySignContext sign_ctx = context;

  sign_ctx->sub_op = NULL;

  (*sign_ctx->callback)(status,
                        data, 
                        data_len,
                        sign_ctx->context);
  
  ssh_proxy_dsa_sign_free(sign_ctx);
}

SshOperationHandle
ssh_proxy_dsa_sign_async(const void *private_key,
                         const char *rgf_name,
                         SshRGFHash hash,
                         SshPrivateKeySignCB callback,
                         void *context)
{
  ProxyDSAKeySignContext sign_ctx;
  SshOperationHandle sub_op;
  ProxyDSAKey key = (ProxyDSAKey)private_key;

  if ((sign_ctx = ssh_calloc(1, sizeof(*sign_ctx))) != NULL)
    {
      sign_ctx->callback = callback;
      sign_ctx->context = context;
      sign_ctx->key = (ProxyDSAKey)private_key;

      /* This is a potential problem when hash is the rgf dummy. We have 
         to malloc again the data buffer. */
      if (sign_ctx->key->cb_does_rgf)
        {
          sign_ctx->raw_data_len = ssh_rgf_hash_digest_length(hash);
        }
      else
        {
          sign_ctx->raw_data_len = KEY_SIZE_TO_BYTES(sign_ctx->key->key_size); 
        }      

      SSH_DEBUG(SSH_D_HIGHOK, ("Are now mallocating %d bytes.", 
                               sign_ctx->raw_data_len));   
      
      sign_ctx->raw_data = ssh_malloc(sign_ctx->raw_data_len);
      sign_ctx->op = NULL;
      sign_ctx->sub_op = NULL;

      /* The callback does the hash only if cb_does_rgf is TRUE in the 
         proxykey and the data in the RGF object has not already been 
         hashed. */ 
      if (ssh_rgf_data_is_digest(hash))
        {
          rgf_name = "dsa-none-none";
          SSH_DEBUG(SSH_D_LOWSTART, ("RGF data is already a hash digest"));
        }

      if (sign_ctx->raw_data == NULL ||
          ssh_rgf_hash_sign(hash,
                            sign_ctx->raw_data,
                            sign_ctx->raw_data_len) != SSH_RGF_OK)
        {
          (*callback)(SSH_CRYPTO_OPERATION_FAILED, NULL, 0, context);
          ssh_proxy_dsa_sign_free(sign_ctx);

          return NULL;
        }

      sign_ctx->op = ssh_operation_register(ssh_proxy_dsa_sign_abort,
                                            sign_ctx);

      sub_op = (*key->sign_op)(rgf_name, 
                               sign_ctx->raw_data,
                               sign_ctx->raw_data_len,
                               ssh_proxy_dsa_sign_op_done,
                               sign_ctx,
                               key->context);


      if (sub_op)
        {
          sign_ctx->sub_op = sub_op;
          return sign_ctx->op;
        }
      return NULL;
    }
  else
    {
      (*callback)(SSH_CRYPTO_NO_MEMORY, NULL, 0, context);
      return NULL;
    }
}

SshOperationHandle
ssh_proxy_dsa_sign_none_none_async(const void *private_key,
                                   SshRGFHash hash,
                                   SshPrivateKeySignCB callback,
                                   void *context)
{
return ssh_proxy_dsa_sign_async(private_key,
                                "dsa-none-none",
                                hash,
                                callback,
                                context);

}    

SshOperationHandle
ssh_proxy_dsa_sign_nist_sha1_async(const void *private_key,
                                   SshRGFHash hash,
                                   SshPrivateKeySignCB callback,
                                   void *context)
{
  return ssh_proxy_dsa_sign_async(private_key,
                                  "dsa-nist-sha1",
                                  hash,
                                  callback,
                                  context);
}    

/************************* The DSA Verification Scheme. *******************/


typedef struct ProxyKeyDSAVerifyContextRec {
  SshOperationHandle op;
  SshOperationHandle sub_op;
  ProxyDSAKey key;
  unsigned char *signature;
  size_t signature_len;
  unsigned char *digest;
  size_t digest_len;
  SshPublicKeyVerifyCB callback;
  void *context;
} *ProxyDSAKeyVerifyContext;

void ssh_proxy_dsa_verify_abort(void *context)
{
  ProxyDSAKeyVerifyContext ctx = context;

  ssh_operation_abort(ctx->sub_op);
  ssh_free(ctx->digest);
  ssh_free(ctx->signature);
  ssh_free(ctx);
}

void ssh_proxy_dsa_verify_free(void *context)
{
  ProxyDSAKeyVerifyContext ctx = context;

  ssh_operation_unregister(ctx->op);
  ssh_proxy_dsa_verify_abort(ctx);
}

void ssh_proxy_dsa_verify_op_done(SshCryptoStatus status, void *context)
{
  ProxyDSAKeyVerifyContext verify_ctx = context;
  
  verify_ctx->sub_op = NULL; 
  
  (*verify_ctx->callback)(status,
                          NULL, 
                          verify_ctx->context);
  
  ssh_proxy_dsa_verify_free(verify_ctx);
}

SshOperationHandle
ssh_proxy_dsa_verify_async(const void *public_key,
                           const char *rgf_name,
                           const unsigned char *signature,
                           size_t signature_len,
                           SshRGFHash hash,
                           SshPublicKeyVerifyCB callback,
                           void *context)
{
  ProxyDSAKeyVerifyContext verify_ctx;
  SshOperationHandle sub_op;
  ProxyDSAKey key = (ProxyDSAKey)public_key;

  if (signature_len & 1 || 
      signature_len > 2 * KEY_SIZE_TO_BYTES(key->key_size)) 
    {
      SSH_DEBUG(SSH_D_FAIL, ("Signature length is incorrect"));

      ssh_rgf_hash_free(hash);
      (*callback)(SSH_CRYPTO_OPERATION_FAILED, NULL, context);

      return NULL;
    }

  if ((verify_ctx = ssh_calloc(1, sizeof(*verify_ctx))) != NULL)
    {
      verify_ctx->callback = callback;
      verify_ctx->context = context;
      verify_ctx->key = (ProxyDSAKey)public_key;

      /* We pass the unhashed data to the callback. This is a problem 
         when hash is the rgf dummy. We have to malloc again the data 
         buffer, no real solution at the moment */
      verify_ctx->digest_len =  ssh_rgf_hash_digest_length(hash);
      verify_ctx->digest = ssh_malloc(verify_ctx->digest_len);
      verify_ctx->signature = ssh_memdup(signature, signature_len);
      verify_ctx->signature_len = signature_len;
      verify_ctx->op = NULL;
      verify_ctx->sub_op = NULL;

      /* The callback does the hash only if cb_does_rgf is TRUE in the 
         proxykey and the data in the RGF object has not already been 
         hashed. */ 
      if (ssh_rgf_data_is_digest(hash))
        {
          rgf_name = "dsa-none-none";
          SSH_DEBUG(SSH_D_LOWSTART, ("RGF data is already a hash digest"));
        }
      
      if (verify_ctx->digest == NULL || verify_ctx->signature == NULL ||
          ssh_rgf_hash_sign(hash,
                            verify_ctx->digest,
                            verify_ctx->digest_len) != SSH_RGF_OK)
        {
          (*callback)(SSH_CRYPTO_OPERATION_FAILED, NULL, context);
          ssh_proxy_dsa_verify_free(verify_ctx);

          return NULL;
        }

      verify_ctx->op = ssh_operation_register(ssh_proxy_dsa_sign_abort,
                                              verify_ctx);

      sub_op = (*key->verify_op)(rgf_name, 
                                 verify_ctx->digest,
                                 verify_ctx->digest_len,
                                 verify_ctx->signature,
                                 verify_ctx->signature_len,
                                 ssh_proxy_dsa_verify_op_done,
                                 verify_ctx,
                                 key->context);
      
      if (sub_op)
        {
          verify_ctx->sub_op = sub_op;
          return verify_ctx->op;
        }
      return NULL;
    }
  else
    {
      (*callback)(SSH_CRYPTO_NO_MEMORY, NULL, context);
      return NULL;
    }

}


SshOperationHandle
ssh_proxy_dsa_verify_none_none_async(const void *public_key,
                                     const unsigned char *signature,
                                     size_t signature_len,
                                     SshRGFHash hash,
                                     SshPublicKeyVerifyCB callback,
                                     void *context)
{
  return ssh_proxy_dsa_verify_async(public_key,
                                    "dsa-none-none",
                                    signature,
                                    signature_len,
                                    hash,
                                    callback,
                                    context);
  
}    

SshOperationHandle
ssh_proxy_dsa_verify_nist_sha1_async(const void *public_key,
                                     const unsigned char *signature,
                                     size_t signature_len,
                                     SshRGFHash hash,
                                     SshPublicKeyVerifyCB callback,
                                     void *context)
{
  return ssh_proxy_dsa_verify_async(public_key,
                                    "dsa-nist-sha1",
                                    signature,
                                    signature_len,
                                    hash,
                                    callback,
                                    context);
}    






/* ****************** Diffie-Hellman Groups ***************************** */



/* The Proxy Group */

typedef struct ProxyGroupRec {
  SshUInt32 group_size;
  SshProxyDHSetupOpCB setup_op;
  SshProxyDHAgreeOpCB agree_op;
  SshProxyFreeOpCB free_op;
  SshUInt32 ref_count;
  void *context;
} *ProxyGroup;


/* *************** Diffie-Hellman Setup Scheme *************************** */

typedef struct ProxyDHSetupContextRec {
  SshOperationHandle op;
  SshOperationHandle sub_op;
  ProxyGroup group;
  SshPkGroupDHSetup callback;
  void *context;
} *ProxyDHSetupContext;


void ssh_proxy_dh_setup_abort(void *context)
{
  ProxyDHSetupContext ctx = context;

  ssh_operation_abort(ctx->sub_op);
  ssh_free(ctx);
}

void ssh_proxy_dh_setup_free(void *context)
{
  ProxyDHSetupContext ctx = context;

  ssh_operation_unregister(ctx->op);
  ssh_proxy_dh_setup_abort(ctx);
}

/* If the Diffie-Hellman private exponent is not returned to us by the 
   proxy setup operation, dh_len must be set to zero. */ 
void ssh_proxy_dh_setup_op_done(SshCryptoStatus status,
                                const unsigned char *exchange,
                                size_t exchange_len,
                                const unsigned char *dh,
                                size_t dh_len,
                                void *context)
{
  ProxyDHSetupContext setup_ctx = context;
  unsigned char *secret = NULL;

  setup_ctx->sub_op = NULL;

  if (status != SSH_CRYPTO_OK)
    {
      (*setup_ctx->callback)(status,
                             NULL, 
                             NULL, 
                             0, 
                             setup_ctx->context);

      ssh_proxy_dh_setup_free(setup_ctx);
      return;
    }

  if (dh_len)
  secret = ssh_malloc(dh_len + 4);

  if (exchange == NULL || secret == NULL)
    {
      ssh_free(secret);     
      (*setup_ctx->callback)(SSH_CRYPTO_OPERATION_FAILED,
                             NULL, 
                             NULL, 
                             0, 
                             setup_ctx->context);

      ssh_proxy_dh_setup_free(setup_ctx);
      return;  
    }

  if (dh_len)
    {
      SSH_PUT_32BIT(secret, dh_len);
      memcpy(secret + 4, dh, dh_len);
    }
  else
    secret = NULL;

  (*setup_ctx->callback)(SSH_CRYPTO_OK, 
                         (SshPkGroupDHSecret)secret, 
                         exchange, 
                         exchange_len,
                         setup_ctx->context);

  ssh_proxy_dh_setup_free(setup_ctx);
}

SshOperationHandle
ssh_proxy_dh_setup_async(void *pk_group,
                         SshPkGroupDHSetup callback,
                         void *context)
{
  ProxyDHSetupContext setup_ctx;
  SshOperationHandle sub_op;
  ProxyGroup group = (ProxyGroup)pk_group;
  
  if ((setup_ctx = ssh_calloc(1, sizeof(*setup_ctx))) != NULL)
    {
      setup_ctx->callback = callback;
      setup_ctx->context = context;
      setup_ctx->group = (ProxyGroup)pk_group;
      setup_ctx->sub_op = NULL;
    
      setup_ctx->op = ssh_operation_register(ssh_proxy_dh_setup_abort,
                                             setup_ctx);
 
      /* No raw data is given to the callback operation. It is the 
         responsibility of the callback operation to generate the 
         private Diffie-Hellman exponent. */ 
      sub_op = (*group->setup_op)(ssh_proxy_dh_setup_op_done,
                                  setup_ctx, group->context);
      
      if (sub_op)
        {
          setup_ctx->sub_op = sub_op;
          return setup_ctx->op;
        }
      return NULL;
    }
  else
    {
      (*callback)(SSH_CRYPTO_NO_MEMORY, NULL, NULL, 0, context);
      return NULL;
    }
}

/* ********************** Diffie-Hellman Agree Scheme ******************* */

typedef struct ProxyDHAgreeContextRec {
  SshOperationHandle op;
  SshOperationHandle sub_op;
  ProxyGroup group;
  unsigned char *exchange;
  size_t exchange_len;
  unsigned char *dh;
  size_t dh_len;
  SshPkGroupDHAgree callback;
  void *context;
} *ProxyDHAgreeContext;


void ssh_proxy_dh_agree_abort(void *context)
{
  ProxyDHAgreeContext ctx = context;

  ssh_operation_abort(ctx->sub_op);
  ssh_free(ctx->exchange);
  ssh_free(ctx->dh);
  ssh_free(ctx);
}

void ssh_proxy_dh_agree_free(void *context)
{
  ProxyDHAgreeContext ctx = context;

  ssh_operation_unregister(ctx->op);
  ssh_proxy_dh_agree_abort(ctx);
}


void ssh_proxy_dh_agree_op_done(SshCryptoStatus status,
                                const unsigned char *data,
                                size_t data_len,
                                void *context)
{
  ProxyDHAgreeContext agree_ctx = context;

  agree_ctx->sub_op = NULL;

  (*agree_ctx->callback)(status,
                         data, 
                         data_len,
                         agree_ctx->context);
  
  ssh_proxy_dh_agree_free(agree_ctx);
}

/* dh_extra should be NULL if the private Diffie-Hellman exponent is not 
   accessible to us. This function frees dh_extra. */
SshOperationHandle
ssh_proxy_dh_agree_async(const void *pk_group,
                         void *dh_extra,
                         const unsigned char *exchange_buffer,
                         size_t exchange_buffer_len,
                         SshPkGroupDHAgree callback,
                         void *context)
{
  ProxyDHAgreeContext agree_ctx;
  SshOperationHandle sub_op;
  ProxyGroup group = (ProxyGroup)pk_group;
  unsigned char *buf = dh_extra;

  if ((agree_ctx = ssh_calloc(1, sizeof(*agree_ctx))) != NULL)
    {
      agree_ctx->callback = callback;
      agree_ctx->context = context;
      agree_ctx->group = (ProxyGroup)pk_group;
      agree_ctx->op = NULL;
      agree_ctx->sub_op = NULL;
      agree_ctx->exchange_len = exchange_buffer_len; 
      agree_ctx->exchange = ssh_memdup(exchange_buffer, exchange_buffer_len);

      if (dh_extra) 
        {
          agree_ctx->dh_len =  SSH_GET_32BIT(buf); 
          agree_ctx->dh = ssh_memdup(buf + 4, agree_ctx->dh_len);
        }
      else
        {
          agree_ctx->dh_len = 0; 
          agree_ctx->dh = NULL;
        }

      /* Free the DH secret here. */
      ssh_free(dh_extra);
      dh_extra = NULL;

      if (agree_ctx->exchange == NULL || 
          (agree_ctx->dh == NULL && agree_ctx->dh_len > 0))
        {
          (*callback)(SSH_CRYPTO_NO_MEMORY, NULL, 0, context);
          ssh_proxy_dh_agree_free(agree_ctx);
          return NULL;
        }
      
      agree_ctx->op = ssh_operation_register(ssh_proxy_dh_agree_abort,
                                             agree_ctx);

      sub_op = (*group->agree_op)(agree_ctx->exchange,
                                  agree_ctx->exchange_len,
                                  agree_ctx->dh,
                                  agree_ctx->dh_len,
                                  ssh_proxy_dh_agree_op_done,
                                  agree_ctx,
                                  group->context);
      
      if (sub_op)
        {
          agree_ctx->sub_op = sub_op;
          return agree_ctx->op;
        }
      return NULL;
    }
  else
    {
      (*callback)(SSH_CRYPTO_NO_MEMORY, NULL, 0, context);
      return NULL;
    }
} 


/***************************  Utility Functions. ********************/ 

typedef struct ProxyKeyActionRec {
  void *proxykey;
} *ProxyKeyAction;

void *ssh_proxy_key_action_init(void)
{
  return ssh_calloc(1, sizeof(struct ProxyKeyActionRec));
}

const char *ssh_proxy_key_action_put(void *context,
                                         va_list ap,
                                         void *input_context,
                                         SshPkFormat format)
{
  ProxyKeyAction ctx = context;
  char *r;

  r = "p";
  switch (format)
    {
    case SSH_PKF_EXTERNALKEY:
      ctx->proxykey = va_arg(ap, void *);
      r = "p";
      break;
    default:
      r = NULL;
    }
  return r;
}

void *ssh_proxy_key_action_make(void *context)
{
  ProxyKeyAction act = context;
  return act->proxykey;
}

void ssh_proxy_key_action_free(void *context)
{
  ssh_free(context);
}

#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
void ssh_proxy_rsa_key_free(void *key)
{
  ProxyRSAKey proxykey = key;

  if (proxykey->ref_count == 0)
    {
      proxykey->free_op(proxykey->context);
      ssh_free(proxykey);
    }
  else
    {
      proxykey->ref_count--;
    }
}

void ssh_proxy_rsa_private_key_derive_public_key(const void *private_key,
                                                 void **public_key)
{
  *public_key = NULL;
}

void ssh_proxy_rsa_key_copy(void *op_src, void **op_dest)
{
  ProxyRSAKey key = op_src;
  key->ref_count++;
  *op_dest = key;
}
#endif /* SSHDIST_CRYPT_RSA */
#endif /* WITH_RSA */

void ssh_proxy_dsa_key_free(void *key)
{
  ProxyDSAKey proxykey = key;
  if (proxykey->ref_count == 0)
    {
      proxykey->free_op(proxykey->context);
      ssh_free(proxykey);
    }
  else
    {
      proxykey->ref_count--;
    }
}

void ssh_proxy_dsa_private_key_derive_public_key(const void *private_key,
                                                 void **public_key)
{
  *public_key = NULL;
}

void ssh_proxy_dsa_key_copy(void *op_src, void **op_dest)
{
  ProxyDSAKey src = op_src;

  src->ref_count++;
  *op_dest = src;
}



void ssh_proxy_group_free(void *group)
{
  ProxyGroup proxygroup = group;

  if (proxygroup->ref_count == 0)
    {
      proxygroup->free_op(proxygroup->context); 
      ssh_free(proxygroup);
    }
  else
    {
      proxygroup->ref_count--;
    }
}

void ssh_proxy_group_copy(void *op_src, void **op_dest)
{
  ProxyGroup src = op_src;

  src->ref_count++;
  *op_dest = src;
}


#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
/* Compute sizes needed in each RSA operation, these agree with those in 
   rsa-generate.c  */

#define SSH_RSA_MINIMUM_PADDING 10
#define SSH_RSA_MAX_BYTES       65535

size_t ssh_proxy_rsa_max_encrypt_input_len(const void *public_key)
{
  ProxyRSAKey ctx = (ProxyRSAKey)public_key;
  size_t len;
  
  len = KEY_SIZE_TO_BYTES(ctx->key_size) - 3 - SSH_RSA_MINIMUM_PADDING;
  
  if (len > 0 && len < SSH_RSA_MAX_BYTES)
    return len;
  return 0;
}

size_t ssh_proxy_rsa_max_oaep_encrypt_input_len(const void *public_key)
{
  ProxyRSAKey ctx = (ProxyRSAKey)public_key;
  size_t len;
 
  len = (KEY_SIZE_TO_BYTES(ctx->key_size) - 2 - 2*SSH_MAX_HASH_DIGEST_LENGTH);
  
  if (len > 0 && len < SSH_RSA_MAX_BYTES)
    return len;
  return 0;
}

size_t ssh_proxy_rsa_max_none_encrypt_input_len(const void *public_key)
{
  ProxyRSAKey ctx = (ProxyRSAKey)public_key;
  size_t len = KEY_SIZE_TO_BYTES(ctx->key_size);
  
  if (len > 0 && len < SSH_RSA_MAX_BYTES)
    return len;
  return 0;
}

size_t ssh_proxy_rsa_max_decrypt_input_len(const void *private_key)
{
  ProxyRSAKey ctx = (ProxyRSAKey)private_key;
  return KEY_SIZE_TO_BYTES(ctx->key_size);
}

size_t ssh_proxy_rsa_max_signature_input_len(const void *private_key)
{
  return (size_t)-1;
}

size_t ssh_proxy_rsa_max_signature_unhash_input_len(const void *private_key)
{
  ProxyRSAKey ctx = (ProxyRSAKey)private_key;
  size_t len = KEY_SIZE_TO_BYTES(ctx->key_size) - 3 - SSH_RSA_MINIMUM_PADDING; 

  if (len > 0 && len < SSH_RSA_MAX_BYTES)
    return len;
  return 0;
}

size_t ssh_proxy_rsa_max_encrypt_output_len(const void *public_key)
{
  ProxyRSAKey ctx = (ProxyRSAKey)public_key;
  return KEY_SIZE_TO_BYTES(ctx->key_size);
}

size_t ssh_proxy_rsa_max_decrypt_output_len(const void *private_key)
{
  ProxyRSAKey ctx = (ProxyRSAKey)private_key;
  return KEY_SIZE_TO_BYTES(ctx->key_size);
}


size_t ssh_proxy_rsa_max_signature_output_len(const void *private_key)
{
  ProxyRSAKey ctx = (ProxyRSAKey)private_key;
  return KEY_SIZE_TO_BYTES(ctx->key_size);
}
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */

/* Compute sizes needed in each DSA and DH operation, these agree with those 
   in dl-dh.c and dl-dsa.c */
size_t ssh_proxy_dsa_max_signature_input_len(const void *private_key)
{
  return (size_t)-1;
}

size_t ssh_proxy_dsa_max_signature_output_len(const void *private_key)
{
  ProxyDSAKey key = (ProxyDSAKey)private_key;
  return KEY_SIZE_TO_BYTES(key->key_size) * 2; 

}

size_t ssh_proxy_diffie_hellman_exchange_length(const void *parameters)
{
  ProxyGroup group = (ProxyGroup)parameters;
  return KEY_SIZE_TO_BYTES(group->group_size); 
}

size_t ssh_proxy_diffie_hellman_shared_secret_length(const void *parameters)
{
  ProxyGroup group = (ProxyGroup)parameters;
  return KEY_SIZE_TO_BYTES(group->group_size); 
}


#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
/************************ RSA Actions and Schemes *************************/

const SshPkSignature ssh_proxy_key_if_modn_signature_schemes[] =
{
  { "rsa-pkcs1-sha1", NULL_FNPTR,
    &ssh_rgf_pkcs1_sha1_def,
    ssh_proxy_rsa_max_signature_input_len,
    ssh_proxy_rsa_max_signature_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_verify_none_none_async,
    NULL_FNPTR,
    ssh_proxy_rsa_sign_none_none_async },
  { "rsa-pkcs1-md5", NULL_FNPTR,
    &ssh_rgf_pkcs1_md5_def,
    ssh_proxy_rsa_max_signature_input_len,
    ssh_proxy_rsa_max_signature_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_verify_none_none_async,
    NULL_FNPTR,
    ssh_proxy_rsa_sign_none_none_async },










  { "rsa-pkcs1-none", NULL_FNPTR,
    &ssh_rgf_pkcs1_none_def,
    ssh_proxy_rsa_max_signature_unhash_input_len,
    ssh_proxy_rsa_max_signature_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_verify_none_none_async,
    NULL_FNPTR,
    ssh_proxy_rsa_sign_none_none_async },
  { NULL }
};

const SshPkEncryption ssh_proxy_key_if_modn_encryption_schemes[] =
{
#ifdef SSHDIST_CRYPT_SHA
  { "rsa-pkcs1v2-oaep", NULL_FNPTR,
    &ssh_rgf_pkcs1v2_sha1_def,
    ssh_proxy_rsa_max_decrypt_input_len,
    ssh_proxy_rsa_max_decrypt_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_decrypt_none_none_async,
    ssh_proxy_rsa_max_oaep_encrypt_input_len,
    ssh_proxy_rsa_max_encrypt_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_encrypt_none_none_async,
    NULL_FNPTR },
  { "rsa-pkcs1-none", NULL_FNPTR,
    &ssh_rgf_pkcs1_none_def,
    ssh_proxy_rsa_max_decrypt_input_len,
    ssh_proxy_rsa_max_decrypt_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_decrypt_none_none_async,
    ssh_proxy_rsa_max_encrypt_input_len,
    ssh_proxy_rsa_max_encrypt_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_encrypt_none_none_async,
    NULL_FNPTR },
#endif /* SSHDIST_CRYPT_SHA */
  { "rsa-none-none", NULL_FNPTR,
    &ssh_rgf_dummy_def,
    ssh_proxy_rsa_max_decrypt_input_len,
    ssh_proxy_rsa_max_decrypt_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_decrypt_none_none_async,
    ssh_proxy_rsa_max_none_encrypt_input_len,
    ssh_proxy_rsa_max_encrypt_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_encrypt_none_none_async,
    NULL_FNPTR },
 
  { NULL }
};


const SshPkAction ssh_proxy_key_if_modn_actions[] =
{
  { SSH_PKF_KEY_TYPE, NULL,
    SSH_PK_FLAG_KEY_TYPE | SSH_PK_FLAG_PRIVATE_KEY |
    SSH_PK_FLAG_PUBLIC_KEY,
    SSH_PK_SCHEME_NONE, 0, NULL },

  { SSH_PKF_SIGN, "sign",
    SSH_PK_FLAG_SCHEME | SSH_PK_FLAG_PRIVATE_KEY |
    SSH_PK_FLAG_PUBLIC_KEY,
    SSH_PK_SCHEME_SIGN,
    sizeof(SshPkSignature),
    ssh_proxy_key_if_modn_signature_schemes, NULL_FNPTR },

  { SSH_PKF_ENCRYPT, "encrypt",
    SSH_PK_FLAG_SCHEME | SSH_PK_FLAG_PRIVATE_KEY |
    SSH_PK_FLAG_PUBLIC_KEY,
    SSH_PK_SCHEME_ENCRYPT,
    sizeof(SshPkEncryption),
    ssh_proxy_key_if_modn_encryption_schemes, NULL_FNPTR },

  { SSH_PKF_EXTERNALKEY, NULL,
    SSH_PK_FLAG_SPECIAL | SSH_PK_FLAG_PRIVATE_KEY |
    SSH_PK_FLAG_PUBLIC_KEY,
    SSH_PK_SCHEME_NONE, 0,
    NULL,
    ssh_proxy_key_action_put,
    NULL_FNPTR },

  { SSH_PKF_END }
};

const SshPkType ssh_proxy_key_if_modn =
{
  "proxy:if-modn",
  ssh_proxy_key_if_modn_actions,

  /* No group operations */ 
  NULL_FNPTR, NULL_FNPTR, NULL_FNPTR,
  NULL_FNPTR, NULL_FNPTR, NULL_FNPTR, NULL_FNPTR, NULL_FNPTR, NULL_FNPTR,
  NULL_FNPTR, NULL_FNPTR, NULL_FNPTR, NULL_FNPTR,

  /* Public key operations */
  ssh_proxy_key_action_init,
  ssh_proxy_key_action_make,
  ssh_proxy_key_action_free,

  NULL_FNPTR,
  NULL_FNPTR,
  ssh_proxy_rsa_key_free,
  ssh_proxy_rsa_key_copy,
  NULL_FNPTR,
  NULL_FNPTR,

  /* Private key operations */
  ssh_proxy_key_action_init,
  ssh_proxy_key_action_make,
  NULL_FNPTR,
  ssh_proxy_key_action_free,

  NULL_FNPTR,
  NULL_FNPTR,
  ssh_proxy_rsa_key_free,
  ssh_proxy_rsa_private_key_derive_public_key,
  ssh_proxy_rsa_key_copy,
  NULL_FNPTR,
  NULL_FNPTR,
};

/************************* RSA Actions and Schemes *******************/
/*************************  using the dummmy RGFs   ******************/


/*Use when the proxy callback wants to do the padding and hashing.*/
const SshPkSignature ssh_proxy_key_if_modn_signature_schemes_dummy_rgf[] =
{
  { "rsa-pkcs1-sha1", NULL_FNPTR,
    &ssh_rgf_dummy_def,
    ssh_proxy_rsa_max_signature_input_len,
    ssh_proxy_rsa_max_signature_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_verify_pkcs1_sha1_async,
    NULL_FNPTR,
    ssh_proxy_rsa_sign_pkcs1_sha1_async },
  { "rsa-pkcs1-md5", NULL_FNPTR,
    &ssh_rgf_dummy_def,
    ssh_proxy_rsa_max_signature_input_len,
    ssh_proxy_rsa_max_signature_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_verify_pkcs1_md5_async,
    NULL_FNPTR,
    ssh_proxy_rsa_sign_pkcs1_md5_async },










  { "rsa-pkcs1-none", NULL_FNPTR,
    &ssh_rgf_dummy_def,
    ssh_proxy_rsa_max_signature_unhash_input_len,
    ssh_proxy_rsa_max_signature_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_verify_pkcs1_none_async,
    NULL_FNPTR,
    ssh_proxy_rsa_sign_pkcs1_none_async },
  { NULL }
};

/* Use when the proxy callback wants to do the padding. */
const SshPkEncryption ssh_proxy_key_if_modn_encryption_schemes_dummy_rgf[] =
{
#ifdef SSHDIST_CRYPT_SHA
  { "rsa-pkcs1v2-oaep", NULL_FNPTR,
    &ssh_rgf_dummy_def,
    ssh_proxy_rsa_max_decrypt_input_len,
    ssh_proxy_rsa_max_decrypt_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_decrypt_pkcs1v2_oaep_async,
    ssh_proxy_rsa_max_oaep_encrypt_input_len,
    ssh_proxy_rsa_max_encrypt_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_encrypt_pkcs1v2_oaep_async,
    NULL_FNPTR },
  { "rsa-pkcs1-none", NULL_FNPTR,
    &ssh_rgf_dummy_def,
    ssh_proxy_rsa_max_decrypt_input_len,
    ssh_proxy_rsa_max_decrypt_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_decrypt_pkcs1_none_async,
    ssh_proxy_rsa_max_encrypt_input_len,
    ssh_proxy_rsa_max_encrypt_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_encrypt_pkcs1_none_async,
    NULL_FNPTR },
#endif /* SSHDIST_CRYPT_SHA */
  { "rsa-none-none", NULL_FNPTR,
    &ssh_rgf_dummy_def,
    ssh_proxy_rsa_max_decrypt_input_len,
    ssh_proxy_rsa_max_decrypt_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_decrypt_none_none_async,
    ssh_proxy_rsa_max_none_encrypt_input_len,
    ssh_proxy_rsa_max_encrypt_output_len,
    NULL_FNPTR,
    ssh_proxy_rsa_encrypt_none_none_async,
    NULL_FNPTR },
 
  { NULL }
};

const SshPkAction ssh_proxy_key_if_modn_actions_dummy_rgf[] =
{

  { SSH_PKF_KEY_TYPE, NULL,
    SSH_PK_FLAG_KEY_TYPE | SSH_PK_FLAG_PRIVATE_KEY |
    SSH_PK_FLAG_PUBLIC_KEY,
    SSH_PK_SCHEME_NONE, 0, NULL },

  { SSH_PKF_SIGN, "sign",
    SSH_PK_FLAG_SCHEME | SSH_PK_FLAG_PRIVATE_KEY |
    SSH_PK_FLAG_PUBLIC_KEY,
    SSH_PK_SCHEME_SIGN,
    sizeof(SshPkSignature),
    ssh_proxy_key_if_modn_signature_schemes_dummy_rgf, NULL_FNPTR },

  { SSH_PKF_ENCRYPT, "encrypt",
    SSH_PK_FLAG_SCHEME | SSH_PK_FLAG_PRIVATE_KEY |
    SSH_PK_FLAG_PUBLIC_KEY,
    SSH_PK_SCHEME_ENCRYPT,
    sizeof(SshPkEncryption),
    ssh_proxy_key_if_modn_encryption_schemes_dummy_rgf, NULL_FNPTR },

  { SSH_PKF_EXTERNALKEY, NULL,
    SSH_PK_FLAG_SPECIAL | SSH_PK_FLAG_PRIVATE_KEY |
    SSH_PK_FLAG_PUBLIC_KEY,
    SSH_PK_SCHEME_NONE, 0,
    NULL,
    ssh_proxy_key_action_put,
    NULL_FNPTR },


  { SSH_PKF_END }
};

const SshPkType ssh_proxy_key_if_modn_dummy_rgf =
{
  "proxy-dummy-rgf:if-modn",
  ssh_proxy_key_if_modn_actions_dummy_rgf,

  /* No group operations */ 
  NULL_FNPTR, NULL_FNPTR, NULL_FNPTR,
  NULL_FNPTR, NULL_FNPTR, NULL_FNPTR, NULL_FNPTR, NULL_FNPTR, NULL_FNPTR,
  NULL_FNPTR, NULL_FNPTR, NULL_FNPTR, NULL_FNPTR,

  /* Public key operations */
  ssh_proxy_key_action_init,
  ssh_proxy_key_action_make,
  ssh_proxy_key_action_free,

  NULL_FNPTR,
  NULL_FNPTR,
  ssh_proxy_rsa_key_free,
  ssh_proxy_rsa_key_copy,
  NULL_FNPTR,
  NULL_FNPTR,

  /* Private key operations */
  ssh_proxy_key_action_init,
  ssh_proxy_key_action_make,
  ssh_proxy_key_action_make,
  ssh_proxy_key_action_free,

  NULL_FNPTR,
  NULL_FNPTR,
  ssh_proxy_rsa_key_free,
  ssh_proxy_rsa_private_key_derive_public_key,
  ssh_proxy_rsa_key_copy,
  NULL_FNPTR,
  NULL_FNPTR,
};
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */

/******************************* DSA Schemes ********************************/

const SshPkSignature ssh_proxy_key_dl_modp_signature_schemes[] =
{
  { "dsa-nist-sha1", NULL_FNPTR,
    &ssh_rgf_std_sha1_def,
    ssh_proxy_dsa_max_signature_input_len,
    ssh_proxy_dsa_max_signature_output_len,
    NULL_FNPTR,
    ssh_proxy_dsa_verify_none_none_async,
    NULL_FNPTR,
    ssh_proxy_dsa_sign_none_none_async },

  { NULL }
};


/* No supported encryption schemes for dl-modp keys. */
const SshPkEncryption ssh_proxy_key_dl_modp_encryption_schemes[] =
{
  { NULL }
};


/******************************     DSA Schemes      ***********************/
/****************************** using the dummmy RGFs **********************/

const SshPkSignature ssh_proxy_key_dl_modp_signature_schemes_dummy_rgf[] =
{
  /* Use when the proxy callback wants to do the hashing. */
  { "dsa-nist-sha1", NULL_FNPTR,
    &ssh_rgf_dummy_def,
    ssh_proxy_dsa_max_signature_input_len,
    ssh_proxy_dsa_max_signature_output_len,
    NULL_FNPTR,
    ssh_proxy_dsa_verify_nist_sha1_async,
    NULL_FNPTR,
    ssh_proxy_dsa_sign_nist_sha1_async },

  { NULL }
};



/************************ Diffie_Hellman Schemes. ************************/

#ifdef SSHDIST_CRYPT_DH
/* Table of all supported diffie-hellman schemes for proxy modp keys. */
const SshPkDiffieHellman ssh_proxy_group_dl_modp_diffie_hellman_schemes[] =
{
  { "plain",
    NULL_FNPTR,
    ssh_proxy_diffie_hellman_exchange_length,
    ssh_proxy_diffie_hellman_shared_secret_length,
    NULL_FNPTR,
    ssh_proxy_dh_setup_async,
    NULL_FNPTR,
    ssh_proxy_dh_agree_async,
    NULL_FNPTR  },
  { NULL },
};
#endif /* SSHDIST_CRYPT_DH */




/**************************** DSA Actions and PkType *******************/

const SshPkAction ssh_proxy_key_dl_modp_actions[] =
{

  { SSH_PKF_KEY_TYPE, NULL,
    SSH_PK_FLAG_KEY_TYPE | SSH_PK_FLAG_PRIVATE_KEY |
    SSH_PK_FLAG_PUBLIC_KEY | SSH_PK_FLAG_PK_GROUP, 
    SSH_PK_SCHEME_NONE, 0, NULL },

  { SSH_PKF_SIGN, "sign",
    SSH_PK_FLAG_SCHEME | SSH_PK_FLAG_PRIVATE_KEY |
    SSH_PK_FLAG_PUBLIC_KEY,
    SSH_PK_SCHEME_SIGN,
    sizeof(SshPkSignature),
    ssh_proxy_key_dl_modp_signature_schemes, NULL_FNPTR },

#ifdef SSHDIST_CRYPT_DH
  { SSH_PKF_DH, "dh",
    SSH_PK_FLAG_SCHEME | SSH_PK_FLAG_PRIVATE_KEY | SSH_PK_FLAG_PUBLIC_KEY |
    SSH_PK_FLAG_PK_GROUP,
    SSH_PK_SCHEME_DH,
    sizeof(SshPkDiffieHellman),
    ssh_proxy_group_dl_modp_diffie_hellman_schemes, 
    NULL_FNPTR },
#endif /* SSHDIST_CRYPT_DH */

  { SSH_PKF_EXTERNALKEY, NULL,
    SSH_PK_FLAG_SPECIAL | SSH_PK_FLAG_PRIVATE_KEY |
    SSH_PK_FLAG_PUBLIC_KEY | SSH_PK_FLAG_PK_GROUP, 
    SSH_PK_SCHEME_NONE, 0,
    NULL,
    ssh_proxy_key_action_put,
    NULL_FNPTR },

  { SSH_PKF_END }
};

const SshPkType ssh_proxy_key_dl_modp =
{
  "proxy:dl-modp",
  ssh_proxy_key_dl_modp_actions,

  /* Group operations */ 
  ssh_proxy_key_action_init,
  ssh_proxy_key_action_make,
  ssh_proxy_key_action_free,
  
  /* Import, export */
  NULL_FNPTR, NULL_FNPTR, 
  
  ssh_proxy_group_free,
  ssh_proxy_group_copy,
 
  /* No predefined or precompute */
  NULL_FNPTR, NULL_FNPTR,

  /* No randomizer handling */
  NULL_FNPTR, NULL_FNPTR, NULL_FNPTR, NULL_FNPTR,

  /* Public key operations */
  ssh_proxy_key_action_init,
  ssh_proxy_key_action_make,
  ssh_proxy_key_action_free,

  NULL_FNPTR,
  NULL_FNPTR,
  ssh_proxy_dsa_key_free,
  ssh_proxy_dsa_key_copy,
  NULL_FNPTR,
  NULL_FNPTR,

  /* Private key operations */
  ssh_proxy_key_action_init,
  ssh_proxy_key_action_make,
  ssh_proxy_key_action_make,
  ssh_proxy_key_action_free,

  NULL_FNPTR,
  NULL_FNPTR,
  ssh_proxy_dsa_key_free,
  ssh_proxy_dsa_private_key_derive_public_key,
  ssh_proxy_dsa_key_copy,
  NULL_FNPTR,
  NULL_FNPTR,
};


/************************* DSA Actions and PkType ************************/
/***********************    using the dummmy RGFs  ***********************/


const SshPkAction ssh_proxy_key_dl_modp_actions_dummy_rgf[] =
{
  { SSH_PKF_KEY_TYPE, NULL,
    SSH_PK_FLAG_KEY_TYPE | SSH_PK_FLAG_PRIVATE_KEY |
    SSH_PK_FLAG_PUBLIC_KEY | SSH_PK_FLAG_PK_GROUP, 
    SSH_PK_SCHEME_NONE, 0, NULL },

  { SSH_PKF_SIGN, "sign",
    SSH_PK_FLAG_SCHEME | SSH_PK_FLAG_PRIVATE_KEY |
    SSH_PK_FLAG_PUBLIC_KEY,
    SSH_PK_SCHEME_SIGN,
    sizeof(SshPkSignature),
    ssh_proxy_key_dl_modp_signature_schemes_dummy_rgf, NULL_FNPTR },

#ifdef SSHDIST_CRYPT_DH
  { SSH_PKF_DH, "dh",
    SSH_PK_FLAG_SCHEME | SSH_PK_FLAG_PRIVATE_KEY | SSH_PK_FLAG_PUBLIC_KEY |
    SSH_PK_FLAG_PK_GROUP,
    SSH_PK_SCHEME_DH,
    sizeof(SshPkDiffieHellman),
    ssh_proxy_group_dl_modp_diffie_hellman_schemes, 
    NULL_FNPTR },
#endif /* SSHDIST_CRYPT_DH */

  { SSH_PKF_EXTERNALKEY, NULL,
    SSH_PK_FLAG_SPECIAL | SSH_PK_FLAG_PRIVATE_KEY |
    SSH_PK_FLAG_PUBLIC_KEY | SSH_PK_FLAG_PK_GROUP, 
    SSH_PK_SCHEME_NONE, 0,
    NULL,
    ssh_proxy_key_action_put,
    NULL_FNPTR },

  { SSH_PKF_END }
};

const SshPkType ssh_proxy_key_dl_modp_dummy_rgf =
{
  "proxy-dummy-rgf:dl-modp",
  ssh_proxy_key_dl_modp_actions_dummy_rgf,

  /* Group operations */ 
  ssh_proxy_key_action_init,
  ssh_proxy_key_action_make,
  ssh_proxy_key_action_free,

  /* Import, export */
  NULL_FNPTR, NULL_FNPTR, 
  
  ssh_proxy_group_free,
  ssh_proxy_group_copy,

  /* No predefined or precompute */
  NULL_FNPTR, NULL_FNPTR,

  /* No randomizer handling */
  NULL_FNPTR, NULL_FNPTR, NULL_FNPTR, NULL_FNPTR,

  /* Public key operations */
  ssh_proxy_key_action_init,
  ssh_proxy_key_action_make,
  ssh_proxy_key_action_free,

  NULL_FNPTR,
  NULL_FNPTR,
  ssh_proxy_dsa_key_free,
  ssh_proxy_dsa_key_copy,
  NULL_FNPTR,
  NULL_FNPTR,

  /* Private key operations */
  ssh_proxy_key_action_init,
  ssh_proxy_key_action_make,
  ssh_proxy_key_action_make,
  ssh_proxy_key_action_free,

  NULL_FNPTR,
  NULL_FNPTR,
  ssh_proxy_dsa_key_free,
  ssh_proxy_dsa_private_key_derive_public_key,
  ssh_proxy_dsa_key_copy,
  NULL_FNPTR,
  NULL_FNPTR,
};


/**************** Helper functions ***************************/
#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
Boolean ssh_register_rsa_proxy_key(Boolean cb_does_rgf)
{
  if (cb_does_rgf)
    {  
      /* Register the proxy-dummy-rgf:if-modn key type. */
      return ssh_proxy_register(&ssh_proxy_key_if_modn_dummy_rgf);
    }
  else
    {  
      /* Register the proxy:f-modn key type. */
      return ssh_proxy_register(&ssh_proxy_key_if_modn);
    } 
}

static char *ssh_make_rsa_proxy_key_name(Boolean cb_does_rgf, 
                                         const char *key_name)
{
  char *proxy_key_name;
  
  if (cb_does_rgf)
    {  
      /* If a key name is supplied, append it to the 
         "proxy-dummy-rgf" key_type to form the proxy_key_name. */
      if (key_name)
        {
          proxy_key_name = ssh_calloc(1, strlen("proxy-dummy-rgf:") 
                                      + strlen(key_name) + 1);
          if (proxy_key_name == NULL)
            {
              SSH_DEBUG(SSH_D_FAIL, ("No memory available"));
              return NULL;
            }
          
          strcat(proxy_key_name, "proxy-dummy-rgf:");
          strcat(proxy_key_name, key_name);
        }
      else
        {
          /* If a no key name is supplied, the key_name is 
             "proxy-dummy-rgf:if-modn". */
          proxy_key_name = ssh_strdup("proxy-dummy-rgf:if-modn");
        }
    }
  else
    {
      /* If a key name is supplied, append it to the 
         "proxy:if-modn" key_type to form the key_name. */
      if (key_name)
        {
          proxy_key_name = ssh_calloc(1, strlen("proxy:") + 
                                      strlen(key_name) + 1);
          if (proxy_key_name == NULL)
            {
              SSH_DEBUG(SSH_D_FAIL, ("No memory available"));
              return NULL;
            }
          
          strcat(proxy_key_name, "proxy:");
          strcat(proxy_key_name, key_name);
        }
      else
        {
          /* If no key name is supplied, the key_name is 
             "proxy:if-modn". */
          proxy_key_name = ssh_strdup("proxy:if-modn");
        }
    }

  SSH_DEBUG(SSH_D_MIDOK, 
            ("The proxy_key_name is %s", proxy_key_name));

  return proxy_key_name;
}
#endif /* SSHDIST_CRYPT_RSA */
#endif /* WITH_RSA */

Boolean ssh_register_dsa_proxy_key(Boolean cb_does_rgf)
{
  if (cb_does_rgf)
    {  
      /* Register the proxy-dummy-rgf:dl-modp key type. */
      return ssh_proxy_register(&ssh_proxy_key_dl_modp_dummy_rgf);
    }
  else
    {  
      /* Register the proxy:dl-modp key type. */
      return ssh_proxy_register(&ssh_proxy_key_dl_modp);
    } 
}

static char *ssh_make_dsa_proxy_key_name(Boolean cb_does_rgf, 
                                         const char *key_name)
{
  char *proxy_key_name;

  if (cb_does_rgf)
    {  
      /* If a scheme name is supplied, append it to the 
         "proxy-dummy-rgf:dl-modp" key_type to form the proxy_key_name. */
      if (key_name)
        {
          proxy_key_name = ssh_calloc(1, 
                                      strlen("proxy-dummy-rgf:") + 
                                      strlen(key_name) + 1);
          if (proxy_key_name == NULL)
            {
              SSH_DEBUG(SSH_D_FAIL, ("No memory available"));
              return NULL;
            }
          
          strcat(proxy_key_name, "proxy-dummy-rgf:");
          strcat(proxy_key_name, key_name);
        }
      else
        {
          /* If a no scheme name is supplied, the proxy_key_name is 
             "proxy:dummy-rgf:dl-modp". */
          proxy_key_name = ssh_strdup("proxy-dummy-rgf:dl-modp");
        }
    }
  else
    {
      /* If a scheme name is supplied, append it to the 
         "proxy:" key_type to form the proxy_key_name. */
      if (key_name)
        {
          proxy_key_name = ssh_calloc(1, strlen("proxy:") + 
                                      strlen(key_name) + 1);

          if (proxy_key_name == NULL)
            {
              SSH_DEBUG(SSH_D_FAIL, ("No memory available"));
              return NULL;
            }
          
          strcat(proxy_key_name, "proxy:");
          strcat(proxy_key_name, key_name);
        }
      else
        {
          /* If no scheme name is supplied, the proxy_key_name is 
             "proxy:dl-modp". */
          proxy_key_name = ssh_strdup("proxy:dl-modp");
        }
    }

  SSH_DEBUG(SSH_D_MIDOK, 
            ("The proxy_key_name is %s", proxy_key_name));

  return proxy_key_name;
}

static char *ssh_make_proxy_group_name(const char *group_name)
{
  char *proxy_group_name;

  /* If a group name is supplied, append it to the 
     "proxy:" to form the proxy_group_name. */
  if (group_name)
    {
      proxy_group_name = ssh_calloc(1, strlen("proxy:") + 
                                     strlen(group_name) + 1);

      if (proxy_group_name == NULL)
        {
          SSH_DEBUG(SSH_D_FAIL, ("No memory available"));
          return NULL;
        }
      
      strcat(proxy_group_name, "proxy:");
      strcat(proxy_group_name, group_name);
    }
      else
        {
          /* If no group name is supplied, the proxy_key_name is 
             "proxy:dl-modp". */
          proxy_group_name = ssh_strdup("proxy:dl-modp");
        }

  SSH_DEBUG(SSH_D_MIDOK, 
            ("The proxy_group_name is %s", proxy_group_name));

  return proxy_group_name;
}


/****************** The Proxy Key Generation Functions ********************/

#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
/* Create a proxy RSA private key. Calls the operation_cb with the data that 
   is being operated when the library is performing crypto operations with 
   the returned proxy key.

   The proxy key is freed with ssh_private_key_free. It is error to free 
   a key that is currently being used.

   The Boolean parameter cb_does_rgf if TRUE specifies that the proxykey 
   callbacks will do all padding and hashing operations. If name is 
   supplied, it will set the scheme of the proxy key.  
*/

SshPrivateKey ssh_private_rsa_key_create_proxy(Boolean cb_does_rgf,
                                               const char *name,
                                               SshUInt32 size_in_bits,
                                               SshProxyKeyOpCB sign_op,
                                               SshProxyKeyOpCB decrypt_op,
                                               SshProxyFreeOpCB free_op,
                                               void *context)
{
  SshPrivateKey key;
  ProxyRSAKey proxykey;
  char *proxy_name;
  
  /* Register the proxy key. */
  if (ssh_register_rsa_proxy_key(cb_does_rgf) == FALSE)
    return NULL; 
  
  /* Construct the key name. */
  proxy_name = ssh_make_rsa_proxy_key_name(cb_does_rgf, name);
  
  if (proxy_name == NULL)
    return NULL;

  if ((proxykey = ssh_calloc(1, sizeof(*proxykey))) != NULL)
    {
      proxykey->cb_does_rgf = cb_does_rgf;
      proxykey->context = context;
      proxykey->signature_op = sign_op;
      proxykey->encryption_op = decrypt_op;
      proxykey->free_op = free_op;
      proxykey->key_size = size_in_bits;
    
      if (ssh_private_key_define(&key,
                                 proxy_name,
                                 SSH_PKF_EXTERNALKEY, proxykey,
                                 SSH_PKF_END) != SSH_CRYPTO_OK)
       
        {
          SSH_DEBUG(SSH_D_FAIL, ("Error in defining the key"));
          
          ssh_free(proxy_name);
          return NULL;
        }
      
      ssh_free(proxy_name);
      return key;
    }
  else
    ssh_free(proxy_name);
  return NULL;
}

/* Create a proxy RSA public key. 

   Calls the operation_cb with the data that is being operated when
   the library is performing crypto operations with the returned proxy key.

   The proxy key is freed with ssh_public_key_free. It is error to free a 
   key that is currently being used.

   The Boolean parameter cb_does_rgf if TRUE specifies that the proxykey 
   callbacks will do all padding and hashing operations. If name is 
   supplied, it will set the scheme of the proxy key.  
*/
SshPublicKey 
ssh_public_rsa_key_create_proxy(Boolean cb_does_rgf,
                                const char *name,
                                SshUInt32 size_in_bits,
                                SshProxyVerifyOpCB rgf_verify_op,
                                SshProxyKeyOpCB verify_op,
                                SshProxyKeyOpCB encrypt_op,
                                SshProxyFreeOpCB free_op,
                                void *context)
{
  SshPublicKey key;
  ProxyRSAKey proxykey;
  char *proxy_name;

  /* Register the proxy key. */
  if (ssh_register_rsa_proxy_key(cb_does_rgf) == FALSE)
    return NULL; 

  /* Construct the key name. */  
  proxy_name = ssh_make_rsa_proxy_key_name(cb_does_rgf, name);
  
  if (proxy_name == NULL)
    return NULL;

  if ((proxykey = ssh_calloc(1, sizeof(*proxykey))) != NULL)
    {
      proxykey->cb_does_rgf = cb_does_rgf;
      proxykey->context = context;
      proxykey->rgf_verify_op = rgf_verify_op;
      proxykey->signature_op = verify_op;
      proxykey->encryption_op = encrypt_op;
      proxykey->free_op = free_op;
      proxykey->key_size = size_in_bits;

      if (ssh_public_key_define(&key,
                                proxy_name,
                                SSH_PKF_EXTERNALKEY, proxykey,
                                SSH_PKF_END) != SSH_CRYPTO_OK)
        {
          SSH_DEBUG(SSH_D_FAIL, ("Error in defining the key"));

          ssh_free(proxy_name);
          return NULL;
        }
      ssh_free(proxy_name);
      return key;
    }
  else
    ssh_free(proxy_name);
    return NULL;
}
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */

/* Create a proxy DSA private key. 

   Calls the operation_cb with the data that is being operated when
   the library is performing crypto operations with the returned proxy key.

   The proxy key is freed with ssh_private_key_free. It is error to free 
   a key that is currently being used. The Boolean parameter cb_does_rgf 
   if TRUE specifies that the proxykey callbacks will do all hashing 
   operations.
   
   If name is supplied, it will set the scheme of the proxy key.  
*/
SshPrivateKey ssh_private_dsa_key_create_proxy(Boolean cb_does_rgf,
                                               const char *name,
                                               SshUInt32 size_in_bits,
                                               SshProxyKeyOpCB sign_op,
                                               SshProxyFreeOpCB free_op,
                                               void *context)
{
  SshPrivateKey key;
  ProxyDSAKey proxykey;
  char *proxy_name;

  /* Register the proxy key. */
  if (ssh_register_dsa_proxy_key(cb_does_rgf) == FALSE)
    return NULL; 

  /* Construct the key name. */  
  proxy_name = ssh_make_dsa_proxy_key_name(cb_does_rgf, name);
  
if (proxy_name == NULL)
    return NULL;

  if ((proxykey = ssh_calloc(1, sizeof(*proxykey))) != NULL)
    {
      proxykey->cb_does_rgf = cb_does_rgf;
      proxykey->context = context;
      proxykey->sign_op = sign_op;
      proxykey->verify_op = NULL_FNPTR;
      proxykey->free_op = free_op;      
      proxykey->key_size = size_in_bits;

      if (ssh_private_key_define(&key,
                                 proxy_name,
                                 SSH_PKF_EXTERNALKEY, proxykey,
                                 SSH_PKF_END) != SSH_CRYPTO_OK)
        {
          SSH_DEBUG(SSH_D_FAIL, ("Error in defining the key"));

          ssh_free(proxy_name);
          return NULL;
        }
      ssh_free(proxy_name);
      return key;
    }
  else
    ssh_free(proxy_name);
    return NULL;
}

/* Create a proxy DSA public key. 

   Calls the operation_cb with the data that is being operated when
   the library is performing crypto operations with the returned proxy key.

   The proxy key is freed with ssh_public_key_free. It is error to free a key
   that is currently being used. The Boolean parameter cb_does_rgf if TRUE 
   specifies that the proxykey callbacks will do all hashing operations.

   If name is supplied, it will set the scheme of the proxy key.  
*/
SshPublicKey ssh_public_dsa_key_create_proxy(Boolean cb_does_rgf,
                                             const char *name,
                                             SshUInt32 size_in_bits,
                                             SshProxyVerifyOpCB verify_op,
                                             SshProxyFreeOpCB free_op,
                                             void *context)
{
  SshPublicKey key;
  ProxyDSAKey proxykey;
  char *proxy_name;

  /* Register the proxy key. */
  if (ssh_register_dsa_proxy_key(cb_does_rgf) == FALSE)
    return NULL; 
  
  /* Construct the key name. */
  proxy_name = ssh_make_dsa_proxy_key_name(cb_does_rgf, name);
   
  if (proxy_name == NULL)
    return NULL;
  
  if ((proxykey = ssh_calloc(1, sizeof(*proxykey))) != NULL)
    {
      proxykey->cb_does_rgf = cb_does_rgf;
      proxykey->context = context;
      proxykey->sign_op = NULL_FNPTR;
      proxykey->verify_op = verify_op;
      proxykey->free_op = free_op;      
      proxykey->key_size = size_in_bits;

      if (ssh_public_key_define(&key,
                                 proxy_name,
                                 SSH_PKF_EXTERNALKEY, proxykey,
                                 SSH_PKF_END) != SSH_CRYPTO_OK)
        {
          SSH_DEBUG(SSH_D_FAIL, ("Error in defining the key"));
       
          ssh_free(proxy_name);
          return NULL;
        }
      ssh_free(proxy_name);
      return key;
    }
  else
    ssh_free(proxy_name);
    return NULL;
}


/* Create a proxy group for Diffie-Hellman usage. The returned group is of 
   type group_type, which must be prefixed with "proxy:dl-modp".
   
   Calls the operation_cb with the data that is being operated when
   the library is performing crypto operations with the returned proxy group.
   
   The proxy group is freed with ssh_group_free. It is error to free a group
   that is currently being used. If name is supplied, it will set 
   the scheme of the proxy group.  
*/
SshPkGroup ssh_dh_group_create_proxy(const char *name,
                                     SshUInt32 size_in_bits,
                                     SshProxyDHSetupOpCB setup_op,
                                     SshProxyDHAgreeOpCB agree_op,
                                     SshProxyFreeOpCB free_op,
                                     void *context)
{
  SshPkGroup group;
  ProxyGroup proxygroup;
  char *proxy_name;

  /* Only dl-modp groups are supported. */  
  if (ssh_proxy_register(&ssh_proxy_key_dl_modp) == FALSE)
    return NULL;

  /* Construct the group name. */  
  proxy_name = ssh_make_proxy_group_name(name);

  if (proxy_name == NULL)
    return NULL;

  if ((proxygroup = ssh_calloc(1, sizeof(*proxygroup))) != NULL)
    {
      proxygroup->context = context;
      proxygroup->setup_op = setup_op;
      proxygroup->agree_op = agree_op;
      proxygroup->free_op = free_op;
      proxygroup->group_size = size_in_bits;
      
      if (ssh_pk_group_generate(&group,
                                proxy_name,
                                SSH_PKF_EXTERNALKEY, proxygroup,
                                SSH_PKF_END) != SSH_CRYPTO_OK)
        {
          SSH_DEBUG(SSH_D_FAIL, ("Cannot generate the group"));
          ssh_free(proxy_name);      
          return NULL;
        }

      ssh_free(proxy_name);
      return group;
    }
  else
    ssh_free(proxy_name);
  return NULL;
}

