/*

rsa.h

Author: Tatu Ylonen <ylo@cs.hut.fi>

Copyright (c) 1995, 2001 Tatu Ylonen <ylo@cs.hut.fi>, Espoo, Finland
                   All rights reserved

Created: Fri Mar  3 22:01:06 1995 ylo

RSA key generation, encryption and decryption.

*/

/*
 * $Id: rsa.h,v 1.2 2001/10/10 05:04:24 jai Exp $
 * $Log: rsa.h,v $ * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * $EndLog$
 */

#ifndef RSA_H
#define RSA_H

#include "sshmp.h"

typedef struct
{
  size_t bits;
  SshMPIntStruct p, q, e, d, n, u;
} SshRSAPrivateKey;

typedef struct
{
  size_t bits;
  SshMPIntStruct n, e;
} SshRSAPublicKey;

typedef struct
{
  unsigned int bits;
  SshMPIntStruct n, p, q, e, d, u;
} SshRSAInitCtx;

/* Function interface to genpkcs.c */

/* action handling. */

/* Initialization function for generation. And function that will free it. */

void *ssh_rsa_private_key_init_action(void);
void *ssh_rsa_public_key_init_action(void);
void ssh_rsa_private_key_init_ctx_free(void *context);

/* Function that makes the private key from information collected from
   vararg list. */
void *ssh_rsa_private_key_make_action(void *context);
void *ssh_rsa_private_key_define_action(void *context);
void *ssh_rsa_private_key_generate_action(void *context);
void *ssh_rsa_public_key_make_action(void *context);

/* Supported actions for RSA. Adding more actions is easy, but should be
   kept to minimum for ease of maintaining. And to clarity of usage. */

const char *
ssh_rsa_action_private_key_get(void *context, va_list ap,
                               void **output_context,
                               SshPkFormat format);
const char *
ssh_rsa_action_private_key_put(void *context, va_list ap,
                               void *input_context,
                               SshPkFormat format);

const char *
ssh_rsa_action_public_key_get(void *context, va_list ap,
                              void **output_context,
                              SshPkFormat format);
const char *
ssh_rsa_action_public_key_put(void *context, va_list ap,
                              void *input_context,
                              SshPkFormat format);

/* These functions import and export keys. */
Boolean ssh_rsa_public_key_import(const unsigned char *buf,
                              size_t len,
                              void **public_key);

Boolean ssh_rsa_public_key_export(const void *public_key,
                              unsigned char **buf,
                              size_t *length_return);

Boolean ssh_rsa_private_key_import(const unsigned char *buf,
                               size_t len,
                               void **private_key);

Boolean ssh_rsa_private_key_export(const void *private_key,
                               unsigned char **buf,
                               size_t *lenght_return);

/* Free keys */
void ssh_rsa_public_key_free(void *public_key);

void ssh_rsa_private_key_free(void *private_key);

/* Copying. */

void ssh_rsa_public_key_copy(void *op_src, void **op_dest);
void ssh_rsa_private_key_copy(void *op_src, void **op_dest);

/* Get maximum buffer lengths needed for specific operations. */
size_t ssh_rsa_public_key_max_encrypt_input_len(const void *public_key);
size_t ssh_rsa_public_key_max_oaep_encrypt_input_len(const void *public_key);
size_t ssh_rsa_public_key_max_none_encrypt_input_len(const void *public_key);

size_t ssh_rsa_public_key_max_encrypt_output_len(const void *public_key);

size_t ssh_rsa_private_key_max_signature_input_len(const void *private_key);

size_t
ssh_rsa_private_key_max_signature_unhash_input_len(const void *private_key);

size_t ssh_rsa_private_key_max_signature_output_len(const void *private_key);

size_t ssh_rsa_private_key_max_decrypt_input_len(const void *private_key);

size_t ssh_rsa_private_key_max_decrypt_output_len(const void *private_key);

/* Derive public key from the private key. */
void ssh_rsa_private_key_derive_public_key(const void *private_key,
                                           void **public_key);

/* Encrypt data. */
Boolean ssh_rsa_public_key_pkcs1_encrypt(const void *public_key,
                                         const unsigned char *plaintext,
                                         size_t plaintext_len,
                                         unsigned char *ciphertext_buffer,
                                         size_t ssh_buffer_len,
                                         size_t *ciphertext_len_return,
                                         const SshHashDefStruct *hash_def);

Boolean ssh_rsa_public_key_pkcs1v2_encrypt(const void *public_key,
                                           const unsigned char *plaintext,
                                           size_t plaintext_len,
                                           unsigned char *ciphertext_buffer,
                                           size_t ciphertext_buffer_len,
                                           size_t *ciphertext_len_return,
                                           const SshHashDefStruct *hash);
Boolean ssh_rsa_public_key_none_encrypt(const void *public_key,
                                        const unsigned char *plaintext,
                                        size_t plaintext_len,
                                        unsigned char *ciphertext_buffer,
                                        size_t ciphertext_buffer_len,
                                        size_t *ciphertext_len_return,
                                        const SshHashDefStruct *hash);
Boolean ssh_rsa_private_key_none_encrypt(const void *private_key,
                                         const unsigned char *data,
                                         size_t data_len,
                                         unsigned char *signature_buffer,
                                         size_t signature_buffer_len,
                                         size_t *signature_length_return);

/* Verify signature. */
Boolean ssh_rsa_public_key_pkcs1_verify(const void *public_key,
                                        const unsigned char *signature,
                                        size_t signature_len,
                                        Boolean need_hashing,
                                        const unsigned char *data,
                                        size_t data_len,
                                        const SshHashDefStruct *hash_def);

/* Verify signature. */
Boolean ssh_rsa_public_key_pkcs1_none_verify(const void *public_key,
                                             const unsigned char *signature,
                                             size_t signature_len,
                                             Boolean need_hashing,
                                             const unsigned char *data,
                                             size_t data_len,
                                             const SshHashDefStruct *hash_def);
/* Decrypt data. */
Boolean ssh_rsa_private_key_pkcs1_decrypt(const void *private_key,
                                          const unsigned char *ciphertext,
                                          size_t ciphertext_len,
                                          unsigned char *plaintext_buffer,
                                          size_t ssh_buffer_len,
                                          size_t *plaintext_length_return,
                                          const SshHashDefStruct *hash_def);

Boolean ssh_rsa_private_key_pkcs1v2_decrypt(const void *private_key,
                                            const unsigned char *ciphertext,
                                            size_t ciphertext_len,
                                            unsigned char *plaintext_buffer,
                                            size_t plaintext_buffer_len,
                                            size_t *plaintext_length_return,
                                            const SshHashDefStruct *hash);
Boolean ssh_rsa_public_key_none_decrypt(const void *public_key,
                                        const unsigned char *ciphertext,
                                        size_t ciphertext_len,
                                        unsigned char *plaintext,
                                        size_t plaintext_len,
                                        size_t *plaintext_length_return);
Boolean ssh_rsa_private_key_none_decrypt(const void *private_key,
                                         const unsigned char *ciphertext,
                                         size_t ciphertext_len,
                                         unsigned char *plaintext_buffer,
                                         size_t plaintext_buffer_len,
                                         size_t *plaintext_length_return,
                                         const SshHashDefStruct *hash);

/* Sign data. */
Boolean ssh_rsa_private_key_pkcs1_sign(const void *private_key,
                                       Boolean need_hashing,
                                       const unsigned char *data,
                                       size_t data_len,
                                       unsigned char *signature_buffer,
                                       size_t ssh_buffer_len,
                                       size_t *signature_length_return,
                                       const SshHashDefStruct *hash);
/* Sign data. */
Boolean ssh_rsa_private_key_pkcs1_none_sign(const void *private_key,
                                            Boolean need_hashing,
                                            const unsigned char *data,
                                            size_t data_len,
                                            unsigned char *signature_buffer,
                                            size_t ssh_buffer_len,
                                            size_t *signature_length_return,
                                            const SshHashDefStruct *hash);

Boolean ssh_rsa_private_key_none_none_sign(const void *private_key,
                                           Boolean need_hashing,
                                           const unsigned char *data,
                                           size_t data_len,
                                           unsigned char *signature_buffer,
                                           size_t signature_buffer_len,
                                           size_t *signature_length_return,
                                           const SshHashDefStruct *hash);

/* Encrypt data. */
Boolean ssh_rsa_public_key_encrypt(const void *public_key,
                                   const unsigned char *plaintext,
                                   size_t plaintext_len,
                                   unsigned char *ciphertext_buffer,
                                   size_t ssh_buffer_len,
                                   size_t *ciphertext_len_return,
                                   SshRGFHash rgf);

Boolean ssh_rsa_private_key_decrypt(const void *private_key,
                                    const unsigned char *ciphertext,
                                    size_t ciphertext_len,
                                    unsigned char *plaintext_buffer,
                                    size_t plaintext_buffer_len,
                                    size_t *plaintext_length_return,
                                    SshRGFHash rgf);

/* Sign data. */
Boolean ssh_rsa_private_key_sign(const void *private_key,
                                 SshRGFHash hash,
                                 unsigned char *signature_buffer,
                                 size_t ssh_buffer_len,
                                 size_t *signature_length_return);

/* Verify signature. */
Boolean ssh_rsa_public_key_verify(const void *public_key,
                                  const unsigned char *signature,
                                  size_t signature_len,
                                  SshRGFHash hash);

unsigned char *
ssh_rsa_pkcs1v2_default_explicit_param(const SshHashDefStruct *hash,
                                       size_t *param_len);

Boolean ssh_rsa_oaep_decode_with_mgf1(const SshHashDefStruct *hash,
                                      const unsigned char *emsg,
                                      size_t emsg_len,
                                      const unsigned char *param,
                                      size_t param_len,
                                      unsigned char **msg, size_t *msg_len);
Boolean ssh_rsa_oaep_encode_with_mgf1(const SshHashDefStruct *hash,
                                      const unsigned char *msg,
                                      size_t msg_len,
                                      const unsigned char *param,
                                      size_t param_len,
                                      unsigned char *emsg, size_t emsg_len);

void *ssh_rsa_make_private_key_of_all(SshMPInt p, SshMPInt q, SshMPInt n,
                                      SshMPInt e, SshMPInt d, SshMPInt u);

void ssh_rsa_private_key_init(SshRSAPrivateKey *private_key);

#endif /* RSA_H */
