/*

  genpkcs.c

  Author: Mika Kojo <mkojo@ssh.fi>

  Copyright (C) 1997-2002 SSH Communications Security Oy, Espoo, Finland
  All rights reserved.

  Created: Mon Jun  2 18:43:45 1997 [mkojo]

  Interface code for public key cryptosystems.

*/

/*
 * $Id: genpkcs.c,v 1.11 2002/02/28 11:16:19 kivinen Exp $
 *
 * $Log: genpkcs.c,v $ * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * $EndLog$
 */

#include "sshincludes.h"
#include "sshcrypt.h"
#include "sshrgf.h"
#include "sshpk.h"
#include "sshcryptocore/namelist.h"

/************************************************************************/

SshPkType const *ssh_pk_type_slots[SSH_PK_TYPE_MAX_SLOTS] =
{
  NULL, /* ... continued to the end. */
};

SshCryptoStatus
ssh_pk_provider_register(const SshPkType *type)
{
  int i;

  if (type == NULL)
    return SSH_CRYPTO_OPERATION_FAILED;

  for (i = 0; i < SSH_PK_TYPE_MAX_SLOTS; i++)
    {
      if (ssh_pk_type_slots[i] == NULL)
        {
          /* Empty slot detected. */
          ssh_pk_type_slots[i] = type;
          return SSH_CRYPTO_OK;
        }

      if (ssh_pk_type_slots[i] == type)
        /* Same type added already. */
        return SSH_CRYPTO_OK;
    }
  return SSH_CRYPTO_PROVIDER_SLOTS_EXHAUSTED;
}


/* Next: genpkcs.c functions ;) */

/* Find action with FLAG_SCHEME set on and matching given identifier. This
   is used when parsing names. */

const SshPkAction *ssh_pk_find_scheme_action(const SshPkAction *list,
                                             const char *identifier,
                                             const SshPkFlag given_flags)
{
  unsigned int i;
  SshPkFlag flags = given_flags | SSH_PK_FLAG_SCHEME;

  for (i = 0; identifier && list[i].format != SSH_PKF_END; i++)
    {
      if ((list[i].flags & flags) == flags)
        {
          /* Check for optimization. */
          if (strcmp(list[i].scheme_class, identifier) == 0)
            return &list[i];
        }
    }
  /* Failed to find a match. */
  return NULL;
}

/* Search from action list an entry that has atleast 'flags' on. */

const SshPkAction *ssh_pk_find_action(SshPkFormat format,
                                      const SshPkAction *list,
                                      const SshPkFlag flags)
{
  unsigned int i;
  Boolean prev = FALSE;

  for (i = 0; list[i].format != SSH_PKF_END; i++)
    {
      /* Check for optimization. */
      if (!((list[i].flags & SSH_PK_FLAG_LIST) && prev))
        {
          if (list[i].format == format)
            prev = TRUE;
          else
            continue;
        }

      /* Check whether flags match. */
      if ((list[i].flags & flags) == flags)
        {
          /* Found a correct match (because they are assumed to be unique
             this must be correct). */

          return &list[i];
        }
    }
  /* Failed to find a match. */
  return NULL;
}

/* Generic search for tables where the first element is const char *.
   How else can one do this? */
void *ssh_pk_find_generic(const char *name, const void *list, size_t msize)
{
  const unsigned char *buf = list;
  const char *buf_name;
  unsigned int i;

  if (name == NULL)
    return NULL;

  /* buf[i] points to start of a structure (which size is msize), buf_name
     is set to the const char * from start of the buf[i]. */
  for (i = 0;
       name && (buf_name = *((const char **)(buf + i)));
       i += msize)
    {
      if (strcmp(buf_name, name) == 0)
        {
          return (void *)(buf + i);
        }
    }
  return NULL;
}

/* Advance generically in scheme tables. Returns the first const char *
   pointer from scheme table (i.e. the name). */
const char *ssh_pk_next_generic(const void **list, size_t msize)
{
  const char *name = *((const char **)*list);

  *list = (void *)((unsigned char *)*list + msize);
  return name;
}

/* Routines for getting scheme names from private keys and public
   keys.  No other information is reasonable to expect to be gotten
   from schemes, although one could think getting descriptions
   etc... */

SshCryptoStatus ssh_private_key_get_scheme_name(SshPrivateKey key,
                                                const char **name,
                                                SshPkSchemeFlag flag)
{
  switch (flag)
    {
    case SSH_PK_SCHEME_SIGN:
      if (key->signature)
        *name = key->signature->name;
      else
        *name = NULL;
      break;
    case SSH_PK_SCHEME_ENCRYPT:
      if (key->encryption)
        *name = key->encryption->name;
      else
        *name = NULL;
      break;
    case SSH_PK_SCHEME_DH:
      if (key->diffie_hellman)
        *name = key->diffie_hellman->name;
      else
        *name = NULL;
      break;
    default:
      return SSH_CRYPTO_LIBRARY_CORRUPTED;
      break;
    }
  return SSH_CRYPTO_OK;
}

SshCryptoStatus ssh_public_key_get_scheme_name(SshPublicKey key,
                                               const char **name,
                                               SshPkSchemeFlag flag)
{
  switch (flag)
    {
    case SSH_PK_SCHEME_SIGN:
      if (key->signature)
        *name = key->signature->name;
      else
        *name = NULL;
      break;
    case SSH_PK_SCHEME_ENCRYPT:
      if (key->encryption)
        *name = key->encryption->name;
      else
        *name = NULL;
      break;
    case SSH_PK_SCHEME_DH:
      if (key->diffie_hellman)
        *name = key->diffie_hellman->name;
      else
        *name = NULL;
      break;
    default:
      return SSH_CRYPTO_LIBRARY_CORRUPTED;
      break;
    }
  return SSH_CRYPTO_OK;

}

/* Helpful function. */
SshNameNode ssh_pk_add_nnode(SshNameTree tree, SshNameNode node,
                             const char *scheme_type,
                             const char *scheme_identifier,
                             Boolean *flag)
{
  SshNameNode temp;

  if (*flag)
    {
      temp = ssh_ntree_add_next(tree, node, scheme_type);
    }
  else
    {
      temp = ssh_ntree_add_child(tree, node, scheme_type);
      *flag = TRUE;
    }
  ssh_ntree_add_child(tree, temp, scheme_identifier);
  return temp;
}

/* Generate the full name of a particular private key. */
char *
ssh_private_key_name(SshPrivateKey key)
{
  SshNameTree tree;
  SshNameNode node;
  char *tmp;
  Boolean flag = FALSE;

  ssh_ntree_allocate(&tree);

  node = ssh_ntree_add_child(tree, NULL, key->type->name);
  if (key->signature)
    node = ssh_pk_add_nnode(tree, node, "sign", key->signature->name, &flag);
  if (key->encryption)
    node = ssh_pk_add_nnode(tree, node, "encrypt", key->encryption->name,
                            &flag);
  if (key->diffie_hellman)
    node = ssh_pk_add_nnode(tree, node, "dh", key->diffie_hellman->name,
                            &flag);
  ssh_ntree_generate_string(tree, &tmp);
  ssh_ntree_free(tree);

  return tmp;
}

/* Generate the full name of a particular public key. */
char *
ssh_public_key_name(SshPublicKey key)
{
  SshNameTree tree;
  SshNameNode node;
  char *tmp = NULL;
  Boolean flag = FALSE;

  ssh_ntree_allocate(&tree);
  if (tree)
    {
      if ((node = ssh_ntree_add_child(tree, NULL, key->type->name))
          == NULL)
        return NULL;

      if (key->signature)
        {
          if ((node =
               ssh_pk_add_nnode(tree, node,
                                "sign", key->signature->name, &flag))
              == NULL)
            return NULL;
        }
      if (key->encryption)
        {
          if ((node =
               ssh_pk_add_nnode(tree, node,
                                "encrypt", key->encryption->name, &flag))
              == NULL)
            return NULL;
        }

      if (key->diffie_hellman)
        {
          if ((node = ssh_pk_add_nnode(tree, node,
                                       "dh", key->diffie_hellman->name, &flag))
              == NULL)
            return NULL;
        }

      ssh_ntree_generate_string(tree, &tmp);
      ssh_ntree_free(tree);
    }
  return tmp;
}

SshCryptoStatus
ssh_private_key_set_scheme(SshPrivateKey key,
                           void *scheme,
                           SshPkSchemeFlag flag)
{
  /* Set the corresponding scheme. */
  switch (flag)
    {
    case SSH_PK_SCHEME_SIGN:
      key->signature = scheme;
      break;
    case SSH_PK_SCHEME_ENCRYPT:
      key->encryption = scheme;
      break;
    case SSH_PK_SCHEME_DH:
      key->diffie_hellman = scheme;
      break;
    default:
      return SSH_CRYPTO_LIBRARY_CORRUPTED;
      break;
    }
  return SSH_CRYPTO_OK;
}

/* Set scheme to given void pointer. These routines should be used with
   caution because no checking is done to verify that given pointer is
   valid. */

SshCryptoStatus
ssh_public_key_set_scheme(SshPublicKey key, void *scheme, SshPkSchemeFlag flag)
{
  /* Set the corresponding scheme. */
  switch (flag)
    {
    case SSH_PK_SCHEME_SIGN:
      key->signature = scheme;
      break;
    case SSH_PK_SCHEME_ENCRYPT:
      key->encryption = scheme;
      break;
    case SSH_PK_SCHEME_DH:
      key->diffie_hellman = scheme;
      break;
    default:
      return SSH_CRYPTO_LIBRARY_CORRUPTED;
      break;
    }
  return SSH_CRYPTO_OK;
}


SshCryptoStatus
ssh_public_key_precompute(SshPublicKey key)
{
  if (key == NULL)
    return SSH_CRYPTO_OPERATION_FAILED;
  if (key->type->public_key_precompute)
    (*key->type->public_key_precompute)(key->context);
  return SSH_CRYPTO_OK;
}

SshCryptoStatus
ssh_private_key_precompute(SshPrivateKey key)
{
  if (key == NULL)
    return SSH_CRYPTO_OPERATION_FAILED;
  if (key->type->private_key_precompute)
    (*key->type->private_key_precompute)(key->context);
  return SSH_CRYPTO_OK;
}

/* Private key functions. */

/* Select new scheme to be used. That is assuming key supports many
   different schemes and/or padding types this can be of some
   use. Note however, that the key stays the same and some method
   assume keys to be of certain form. Such an example is DSA which by
   standard needs to have parameters of certain form, but this
   function could easily switch to DSA with key that is not of that
   form. Nevertheless I feel that such problems do not make switching
   to other methods unusable (even DSA would work with different
   parameters, although would not conform to the digital signature
   standard). */

SshCryptoStatus
ssh_private_key_select_scheme(SshPrivateKey key, ...)
{
  SshCryptoStatus status;
  const SshPkAction *action;
  SshPkFormat format;
  void *scheme;
  const char *name;
  va_list ap;

  if (key->type == NULL)
    return SSH_CRYPTO_KEY_UNINITIALIZED;

  va_start(ap, key);

  while ((format = va_arg(ap, SshPkFormat)) != SSH_PKF_END)
    {
      action = ssh_pk_find_action(format, key->type->action_list,
                                  SSH_PK_FLAG_SCHEME |
                                  SSH_PK_FLAG_PRIVATE_KEY);
      if (!action)
        {
          va_end(ap);
          return SSH_CRYPTO_UNSUPPORTED_IDENTIFIER;
        }

      /* Find the new scheme. */
      name = va_arg(ap, const char *);
      scheme = ssh_pk_find_generic(name, action->type,
                                   action->type_size);

      /* Awful error! Means that our scheme tables are either
         corrupted or application failed. */
      if (scheme == NULL)
        {
          va_end(ap);
          return SSH_CRYPTO_SCHEME_UNKNOWN;
        }

      status = ssh_private_key_set_scheme(key, scheme, action->scheme_flag);
      if (status != SSH_CRYPTO_OK)
        {
          va_end(ap);
          return status;
        }
    }
  va_end(ap);
  return SSH_CRYPTO_OK;
}

/* This function is needed in X.509 certificate routines. What is
   needed, is a way that creates from a bunch of stuff a valid
   SshPublicKey through sshcrypt header file. */

SshCryptoStatus
ssh_public_key_define(SshPublicKey *public_key,
                      const char *key_type, ...)
{
  SshCryptoStatus status = SSH_CRYPTO_UNKNOWN_KEY_TYPE;
  SshPublicKey pub_key;
  SshPkGroup group;
  const SshPkAction *action;
  SshPkFormat format;
  SshNameTree tree;
  SshNameNode node, child;
  SshNameTreeStatus nstat;
  const char *name, *r;
  void *wrapper;
  char *tmp, consumed[128];
  void *scheme;
  void *context;
  unsigned int i;
  va_list ap;

  /* Parse given group type. */
  ssh_ntree_allocate(&tree);
  nstat = ssh_ntree_parse(key_type, tree);
  if (nstat != SSH_NTREE_OK)
    {
      ssh_ntree_free(tree);
      return SSH_CRYPTO_UNKNOWN_KEY_TYPE;
    }
  node = ssh_ntree_get_root(tree);
  if (node == NULL)
    {
      ssh_ntree_free(tree);
      return SSH_CRYPTO_UNKNOWN_KEY_TYPE;
    }
  if ((tmp = ssh_nnode_get_identifier(node)) == NULL)
    {
      ssh_ntree_free(tree);
      return SSH_CRYPTO_NO_MEMORY;
    }

  /* Find out the key type. */
  for (i = 0; ssh_pk_type_slots[i] != NULL && ssh_pk_type_slots[i]->name; i++)
    {
      if (strcmp(ssh_pk_type_slots[i]->name, tmp) != 0)
        continue;

      /* Type matches i.e. we've found our key type, so continue with
         finding schemes and parameters. */
      ssh_free(tmp);
      node = ssh_nnode_get_child(node);

      /* Allocate private key context. */
      if ((pub_key = ssh_malloc(sizeof(*pub_key))) == NULL)
        {
          ssh_ntree_free(tree);
          return SSH_CRYPTO_OPERATION_FAILED;
        }
      pub_key->type = ssh_pk_type_slots[i];

      /* Clear pointers. */
      pub_key->signature = NULL;
      pub_key->encryption = NULL;
      pub_key->diffie_hellman = NULL;

      /* Initialize actions, and verify that context was allocated. */
      context = (*pub_key->type->public_key_action_init)();
      if (context == NULL)
        {
          ssh_ntree_free(tree);
          ssh_free(pub_key);
          va_end(ap);
          return SSH_CRYPTO_OPERATION_FAILED;
        }

      status = SSH_CRYPTO_OK;
      /* Run through all preselected schemes in the group_type. */
      while (node)
        {
          tmp = ssh_nnode_get_identifier(node);
          action =
            ssh_pk_find_scheme_action(pub_key->type->action_list,
                                      tmp,
                                      SSH_PK_FLAG_PUBLIC_KEY);
          ssh_free(tmp);
          if (!action)
            {
              status = SSH_CRYPTO_SCHEME_UNKNOWN;
              break;
            }
          child = ssh_nnode_get_child(node);
          if (child == NULL)
            tmp = SSH_PK_USUAL_NAME;
          else
            tmp = ssh_nnode_get_identifier(child);

          scheme = ssh_pk_find_generic(tmp, action->type, action->type_size);
          if (child)
            ssh_free(tmp);
          if (scheme == NULL)
            {
              status = SSH_CRYPTO_SCHEME_UNKNOWN;
              break;
            }

          /* Call action_scheme if not set to NULL. */
          if (((SshPkGen *)scheme)->action_scheme != NULL_FNPTR)
            (*((SshPkGen *)scheme)->action_scheme)(context);

          /* Set the corresponding scheme to the group. */
          status = ssh_public_key_set_scheme(pub_key, scheme,
                                             action->scheme_flag);

          if (status != SSH_CRYPTO_OK)
            {
              break;
            }
          /* Move to the next scheme. */
          node = ssh_nnode_get_next(node);
        }

      ssh_ntree_free(tree);
      if (status != SSH_CRYPTO_OK)
        {
          (*pub_key->type->public_key_action_free)(context);
          ssh_free(pub_key);
          va_end(ap);
          return status;
        }

      /* Parse vararg list. */
      consumed[0] = '\000';
      while (TRUE)
        {
          va_start(ap, key_type);
          PROCESS(ap, consumed);

          format = va_arg(ap, SshPkFormat);
          strcat(consumed, "i");
          if (format == SSH_PKF_END)
            break;

          /* Search name from command lists. */
          action = ssh_pk_find_action(format,
                                      pub_key->type->action_list,
                                      SSH_PK_FLAG_PUBLIC_KEY);
          if (!action)
            {
              (*pub_key->type->public_key_action_free)(context);
              ssh_free(pub_key);
              va_end(ap);
              return SSH_CRYPTO_UNSUPPORTED_IDENTIFIER;
            }

          /* Supported only scheme selection and special operations. */
          switch (action->flags & (SSH_PK_FLAG_SCHEME | SSH_PK_FLAG_SPECIAL))
            {
            case SSH_PK_FLAG_SCHEME:
              name = va_arg(ap, const char *);
              strcat(consumed, "p");
              scheme = ssh_pk_find_generic(name,
                                           action->type, action->type_size);
              if (scheme == NULL)
                {
                  (*pub_key->type->public_key_action_free)(context);
                  ssh_free(pub_key);
                  va_end(ap);
                  return SSH_CRYPTO_SCHEME_UNKNOWN;
                }

              /* Call the action_scheme function here if not
                 NULL. */
              if (((SshPkGen *)scheme)->action_scheme != NULL_FNPTR)
                (*((SshPkGen *)scheme)->action_scheme)(context);

              /* Set the corresponding scheme. */
              status = ssh_public_key_set_scheme(pub_key, scheme,
                                                 action->scheme_flag);
              if (status != SSH_CRYPTO_OK)
                {
                  (*pub_key->type->public_key_action_free)(context);
                  ssh_free(pub_key);
                  va_end(ap);
                  return status;
                }
              break;

            case SSH_PK_FLAG_SPECIAL:
              /* Assume we don't use wrappings. */
              wrapper = NULL;
              if (action->flags & SSH_PK_FLAG_WRAPPED)
                {
                  /* We assume that parameters are wrapped over group
                     structure. */
                  group = va_arg(ap, SshPkGroup);
                  strcat(consumed, "p");
                  wrapper = group->context;
                  /* For compatibility set also the Diffie-Hellman
                     field.  */
                  pub_key->diffie_hellman = group->diffie_hellman;
                }

              r = (*action->action_put)(context, ap, wrapper, format);
              if (r == NULL)
                {
                  (*pub_key->type->public_key_action_free)(context);
                  ssh_free(pub_key);
                  va_end(ap);
                  return SSH_CRYPTO_LIBRARY_CORRUPTED;
                }
              else
                strcat(consumed, r);
              break;
            default:
              ssh_fatal("ssh_public_key_define: internal error.");
              break;
            }
          va_end(ap);
        }

      /* Make the key and remove context. (One could incorporate
         making and freeing, however this way things seem to work
         also). */
      pub_key->context =
        (*pub_key->type->public_key_action_make)(context);
      (*pub_key->type->public_key_action_free)(context);

      /* Quit unhappily. */
      if (pub_key->context == NULL)
        {
          ssh_free(pub_key);
          va_end(ap);
          return SSH_CRYPTO_OPERATION_FAILED;
        }

      /* Quit happily. */
      *public_key = pub_key;
      va_end(ap);

      return SSH_CRYPTO_OK;
    }

  ssh_ntree_free(tree);
  va_end(ap);

  return SSH_CRYPTO_UNKNOWN_KEY_TYPE;
}

/* This is a little bit stupid, maybe same context for private and public
   key (internally) would be a good idea. */

SshCryptoStatus
ssh_public_key_select_scheme(SshPublicKey key, ...)
{
  SshCryptoStatus status;
  const SshPkAction *action;
  SshPkFormat format;
  const char *name;
  void *scheme;
  va_list ap;

  if (key->type == NULL)
    return SSH_CRYPTO_KEY_UNINITIALIZED;

  va_start(ap, key);

  while ((format = va_arg(ap, SshPkFormat)) != SSH_PKF_END)
    {
      action = ssh_pk_find_action(format, key->type->action_list,
                                  SSH_PK_FLAG_SCHEME |
                                  SSH_PK_FLAG_PUBLIC_KEY);

      if (!action)
        {
          va_end(ap);
          return SSH_CRYPTO_UNSUPPORTED_IDENTIFIER;
        }

      /* Find the new scheme. */
      name = va_arg(ap, const char *);
      scheme = ssh_pk_find_generic(name, action->type,
                                   action->type_size);

      /* Quit an awful error! Means that our scheme tables are either
         corrupted or application failed. */
      if (scheme == NULL)
        {
          va_end(ap);
          return SSH_CRYPTO_SCHEME_UNKNOWN;
        }

      status = ssh_public_key_set_scheme(key, scheme, action->scheme_flag);
      if (status != SSH_CRYPTO_OK)
        {
          va_end(ap);
          return status;
        }
    }
  va_end(ap);
  return SSH_CRYPTO_OK;
}

SshCryptoStatus
ssh_private_key_get_info(SshPrivateKey key, ...)
{
  SshCryptoStatus status;
  const SshPkAction *action;
  SshPkFormat format;
  const char **name_ptr, *r;
  char consumed[128];
  va_list ap;

  consumed[0] = '\000';
  while (TRUE)
    {
      va_start(ap, key);
      PROCESS(ap, consumed);

      format = va_arg(ap, SshPkFormat);
      strcat(consumed, "i");
      if (format == SSH_PKF_END)
        break;

      /* Seek for the action. */
      action = ssh_pk_find_action(format,
                                  key->type->action_list,
                                  SSH_PK_FLAG_PRIVATE_KEY);
      if (!action)
        {
          va_end(ap);
          return SSH_CRYPTO_UNSUPPORTED_IDENTIFIER;
        }

      switch (action->flags &
              (SSH_PK_FLAG_SCHEME|SSH_PK_FLAG_SPECIAL|SSH_PK_FLAG_KEY_TYPE))
        {
        case SSH_PK_FLAG_KEY_TYPE:
          name_ptr = va_arg(ap, const char **);
          strcat(consumed, "p");
          *name_ptr = strchr(key->type->name, ':');
          if (*name_ptr)
            (*name_ptr)++;
          else
            *name_ptr = key->type->name; /* ssh_private_key_name(key); */
          break;

        case SSH_PK_FLAG_SCHEME:
          name_ptr = va_arg(ap, const char **);
          strcat(consumed, "p");
          status = ssh_private_key_get_scheme_name(key,
                                                   name_ptr,
                                                   action->scheme_flag);
          if (status != SSH_CRYPTO_OK)
            {
              va_end(ap);
              return status;
            }
          break;

        case SSH_PK_FLAG_SPECIAL:
          if (action->flags & SSH_PK_FLAG_WRAPPED)
            {
              if (action->action_get)
                ssh_fatal("ssh_private_key_get_info: cannot wrap.");
              return SSH_CRYPTO_UNSUPPORTED_IDENTIFIER;
            }

          if (action->action_get == NULL_FNPTR)
            return SSH_CRYPTO_UNSUPPORTED;

          r = (*action->action_get)(key->context, ap, NULL, format);
          if (r == NULL)
            return SSH_CRYPTO_LIBRARY_CORRUPTED;
          else
            strcat(consumed, r);
          break;

        default:
          ssh_fatal("ssh_private_key_get_info: internal error.");
          break;
        }

      va_end(ap);
    }

  va_end(ap);
  return SSH_CRYPTO_OK;
}

SshCryptoStatus
ssh_public_key_get_info(SshPublicKey key, ...)
{
  SshCryptoStatus status;
  const SshPkAction *action;
  SshPkFormat format;
  const char **name_ptr, *r;
  char consumed[128];
  va_list ap;

  consumed[0] = '\000';
  while (TRUE)
    {
      va_start(ap, key);
      PROCESS(ap, consumed);

      format = va_arg(ap, SshPkFormat);
      strcat(consumed, "i");
      if (format == SSH_PKF_END)
        break;

      /* Seek for the action. */
      action = ssh_pk_find_action(format,
                                  key->type->action_list,
                                  SSH_PK_FLAG_PUBLIC_KEY);

      if (!action)
        {
          va_end(ap);
          return SSH_CRYPTO_UNSUPPORTED_IDENTIFIER;
        }

      switch (action->flags &
              (SSH_PK_FLAG_SCHEME|SSH_PK_FLAG_SPECIAL|SSH_PK_FLAG_KEY_TYPE))
        {
        case SSH_PK_FLAG_KEY_TYPE:
          name_ptr = va_arg(ap, const char **);
          strcat(consumed, "p");
          *name_ptr = strchr(key->type->name, ':');
          if (*name_ptr)
            (*name_ptr)++;
          else
            *name_ptr = key->type->name; /* ssh_private_key_name(key); */
          break;

        case SSH_PK_FLAG_SCHEME:
          name_ptr = va_arg(ap, const char **);
          strcat(consumed, "p");
          status = ssh_public_key_get_scheme_name(key,
                                                  name_ptr,
                                                  action->scheme_flag);
          if (status != SSH_CRYPTO_OK)
            {
              va_end(ap);
              return status;
            }
          break;

        case SSH_PK_FLAG_SPECIAL:
          if (action->flags & SSH_PK_FLAG_WRAPPED)
            {
              if (action->action_get)
                ssh_fatal("ssh_public_key_get_info: cannot wrap.");
              return SSH_CRYPTO_UNSUPPORTED_IDENTIFIER;
            }

          if (action->action_get == NULL_FNPTR)
            return SSH_CRYPTO_UNSUPPORTED;

          r = (*action->action_get)(key->context, ap, NULL, format);
          if (r == NULL)
            return SSH_CRYPTO_LIBRARY_CORRUPTED;
          else
            strcat(consumed, r);
          break;

        default:
          ssh_fatal("ssh_public_key_get_info: internal error.");
          break;
        }
      va_end(ap);
    }

  va_end(ap);
  return SSH_CRYPTO_OK;
}

/* Names could be given by gathering all possible combinations, however,
   it might be more useful for outsider to get names for some specific
   class of algorithms. Such as signature, encryption or some key exchange
   method. */
char *
ssh_public_key_get_supported(void)
{
  char *list;
  unsigned int i, j, k, l;
  const SshPkAction *action;
  const void *scheme_list;
  const char *scheme_list_name;
  SshNameTree tree;
  SshNameNode node;

  /* Allocate tree. */
  ssh_ntree_allocate(&tree);
  node = NULL;

  for (i = 0; ssh_pk_type_slots[i] != NULL && ssh_pk_type_slots[i]->name; i++)
    {
      /* Add key type node. */
      node = ssh_ntree_add_next(tree, node,
                                ssh_pk_type_slots[i]->name);

      for (action = ssh_pk_type_slots[i]->action_list, j = 0, l = 0;
           action[j].format != SSH_PKF_END; j++)
        {
          if ((action[j].flags & SSH_PK_FLAG_SCHEME) == SSH_PK_FLAG_SCHEME)
            {
              /* Add scheme identifier nodes. */
              if (l == 0)
                node = ssh_ntree_add_child(tree, node,
                                           action[j].scheme_class);
              else
                node = ssh_ntree_add_next(tree, node,
                                          action[j].scheme_class);
              l++;
              for (scheme_list = action[j].type, k = 0;
                   (scheme_list_name =
                    ssh_pk_next_generic(&scheme_list,
                                        action[j].type_size)) != NULL; k++)
                {
                  /* Add actual algorithm identifiers.

                  XXX Note, here we don't wonder about the *_USUAL_NAME
                  thing. It is more straight forward to just forget
                  it here. Although, it would make things easier to
                  read. */
                  if (k == 0)
                    node = ssh_ntree_add_child(tree, node,
                                               scheme_list_name);
                  else
                    node = ssh_ntree_add_next(tree, node,
                                              scheme_list_name);
                }
              /* Go up if one went down. */
              if (k)
                node = ssh_nnode_get_parent(node);
            }
        }
      /* Go up if one went down. */
      if (l)
        node = ssh_nnode_get_parent(node);
    }

  ssh_ntree_generate_string(tree, &list);
  ssh_ntree_free(tree);

  return list;
}

/* Public key routines copy and free. */

/* Doing copy of the key_src, so that both keys can be altered without
   affecting the other. Note, that although keys might seem to be totally
   separate some features might be implemeted with reference counting. */
SshCryptoStatus
ssh_public_key_copy(SshPublicKey key_src, SshPublicKey *key_dest)
{
  SshPublicKey created;

  if (key_src->type->public_key_copy == NULL_FNPTR)
    return SSH_CRYPTO_UNSUPPORTED;

  if ((created = ssh_malloc(sizeof(*created))) != NULL)
    {
      /* First copy all basic internal stuff and then the context
         explicitly. */
      memcpy(created, key_src, sizeof(*created));
      (*key_src->type->public_key_copy)(key_src->context, &created->context);
      *key_dest = created;
      return SSH_CRYPTO_OK;
    }
  else
    return SSH_CRYPTO_NO_MEMORY;
}

void
ssh_public_key_free(SshPublicKey key)
{
  if (key)
    {
      if (key->type->public_key_free)
        (*key->type->public_key_free)(key->context);
      key->context = NULL;
      ssh_free(key);
    }
}

/* Derive public key group for public key. */
SshPkGroup
ssh_public_key_derive_pk_group(SshPublicKey key)
{
  SshPkGroup group = NULL;

  if (key->type->public_key_derive_pk_group == NULL_FNPTR)
    return NULL;

  if ((group = ssh_malloc(sizeof(*group))) != NULL)
    {
      group->type = key->type;
      (*key->type->public_key_derive_pk_group)(key->context, &group->context);
      group->diffie_hellman = key->diffie_hellman;
    }
  return group;
}

/* Report the maximal length of bytes which may be encrypted with this
   public key. Return 0 if encryption not available for this public
   key. */
size_t
ssh_public_key_max_encrypt_input_len(SshPublicKey key)
{
  if (key->encryption == NULL)
    return 0;

  if (key->encryption->public_key_max_encrypt_input_len == NULL_FNPTR)
    return 0;

  return (*key->encryption->public_key_max_encrypt_input_len)(key->context);
}

/* This is similar to the previous one, but the maximal output length
   is returned instead the of the maximum input length. */
size_t
ssh_public_key_max_encrypt_output_len(SshPublicKey key)
{
  if (key->encryption == NULL)
    return 0;

  return (*key->encryption->public_key_max_encrypt_output_len)(key->context);
}

/* Private key routines copy and free. */

SshCryptoStatus
ssh_private_key_copy(SshPrivateKey key_src, SshPrivateKey *key_dest)
{
  SshPrivateKey created;

  if (key_src->type->private_key_copy == NULL_FNPTR)
    return SSH_CRYPTO_UNSUPPORTED;

  if ((created = ssh_malloc(sizeof(*created))) != NULL)
    {
      memcpy(created, key_src, sizeof(*created));
      (*key_src->type->private_key_copy)(key_src->context, &created->context);
      *key_dest = created;
      return SSH_CRYPTO_OK;
    }
  else
    return SSH_CRYPTO_NO_MEMORY;
}

/* Release a private key structure. */

void
ssh_private_key_free(SshPrivateKey key)
{
  if (key == NULL || key->context == NULL)
    ssh_fatal("ssh_private_key_free: undefined key.");
  if (key->type->private_key_free)
    (*key->type->private_key_free)(key->context);
  key->context = NULL;
  ssh_free(key);
}

SshPublicKey
ssh_private_key_derive_public_key(SshPrivateKey key)
{
  SshPublicKey pub;
  void *pub_context;

  if (key->type->private_key_derive_public_key == NULL_FNPTR)
    return NULL;

  (*key->type->private_key_derive_public_key)(key->context, &pub_context);
  if (pub_context == NULL)
    return NULL;

  if ((pub = ssh_malloc(sizeof(*pub))) != NULL)
    {
      pub->context = pub_context;
      pub->type = key->type;

      /* Set up all schemes for compatibility. */
      pub->signature = key->signature;
      pub->encryption = key->encryption;
      pub->diffie_hellman = key->diffie_hellman;
    }
  return pub;
}

SshPkGroup
ssh_private_key_derive_pk_group(SshPrivateKey key)
{
  SshPkGroup group;

  if (key->type->private_key_derive_pk_group == NULL_FNPTR)
    return NULL;

  if ((group = ssh_malloc(sizeof(*group))) != NULL)
    {
      group->type = key->type;
      (*key->type->private_key_derive_pk_group)(key->context, &group->context);
      /* Set up schemes for compatibility. */
      group->diffie_hellman = key->diffie_hellman;
    }
  return group;
}

/* Signature hash function derivation functions. */

SshRGFHash
ssh_public_key_derive_signature_hash(SshPublicKey key)
{
  if (key->signature == NULL)
    return NULL;
  return ssh_rgf_hash_allocate(key->signature->rgf_def);
}

SshRGFHash
ssh_private_key_derive_signature_hash(SshPrivateKey key)
{
  if (key->signature == NULL)
    return NULL;
  return ssh_rgf_hash_allocate(key->signature->rgf_def);
}

/* Perform the public key encryption operation. */

SshCryptoStatus
ssh_public_key_encrypt(SshPublicKey key,
                       const unsigned char *plaintext,
                       size_t plaintext_len,
                       unsigned char *ciphertext,
                       size_t buffer_len,
                       size_t *ciphertext_len_return)
{
  SshRGFHash hash;

  if (key->encryption == NULL ||
      key->encryption->public_key_encrypt == NULL_FNPTR)
    return SSH_CRYPTO_UNSUPPORTED;

  hash = ssh_rgf_hash_allocate(key->encryption->rgf_def);

  /* If true then encryption succeeded. */
  if (hash &&
      (*key->encryption->public_key_encrypt)(key->context,
                                             plaintext, plaintext_len,
                                             ciphertext, buffer_len,
                                             ciphertext_len_return,
                                             hash))
    return SSH_CRYPTO_OK;

  return SSH_CRYPTO_OPERATION_FAILED;
}

SshCryptoStatus
ssh_public_key_decrypt(SshPublicKey key,
                       const unsigned char *ciphertext,
                       size_t ciphertext_len,
                       unsigned char *plaintext,
                       size_t plaintext_len,
                       size_t *plaintext_len_return)
{
  if (key->encryption == NULL ||
      key->encryption->public_key_decrypt == NULL_FNPTR)
    return SSH_CRYPTO_UNSUPPORTED;

  /* If true then encryption succeeded. */
  if ((*key->encryption->public_key_decrypt)(key->context,
                                             ciphertext,
                                             ciphertext_len,
                                             plaintext,
                                             plaintext_len,
                                             plaintext_len_return))
    return SSH_CRYPTO_OK;

  return SSH_CRYPTO_OPERATION_FAILED;
}

/* Verify a signature. In fact, decrypt the given signature with the
   public key, and then compare the decrypted data to the given
   (supposedly original) data. If the decrypted data and the given
   data are identical (in the sense that they are of equal length and
   their contents are bit-wise same) the function returns TRUE,
   otherways FALSE. */

Boolean
ssh_public_key_verify_signature(SshPublicKey key,
                                const unsigned char *signature,
                                size_t signature_len,
                                const unsigned char *data,
                                size_t data_len)
{
  SshRGFHash hash;

  if (key == NULL ||
      key->signature == NULL ||
      key->signature->public_key_verify == NULL_FNPTR)
    return FALSE;

  /* We need to compute the RGFHash ourselves. */
  hash = ssh_rgf_hash_allocate(key->signature->rgf_def);
  if (hash == NULL)
    return FALSE;
  ssh_rgf_hash_update(hash, data, data_len);

  return (*key->signature->public_key_verify)(key->context,
                                              signature, signature_len,
                                              hash);
}

Boolean
ssh_public_key_verify_signature_with_digest(SshPublicKey key,
                                            const unsigned char *signature,
                                            size_t signature_len,
                                            const unsigned char *digest,
                                            size_t digest_len)
{
  SshRGFHash hash;

  if (key->signature == NULL ||
      key->signature->public_key_verify == NULL_FNPTR)
    return FALSE;

  hash = ssh_rgf_hash_allocate(key->signature->rgf_def);
  if (hash == NULL)
    return FALSE;
  if (ssh_rgf_hash_update_with_digest(hash,
                                      digest, digest_len) == FALSE)
    {
      ssh_rgf_hash_free(hash);
      return FALSE;
    }

  return (*key->signature->public_key_verify)(key->context,
                                              signature, signature_len,
                                              hash);
}

Boolean
ssh_public_key_verify_signature_with_hash(SshPublicKey key,
                                          const unsigned char *signature,
                                          size_t signature_len,
                                          SshRGFHash hash)
{
  if (key->signature == NULL ||
      key->signature->public_key_verify == NULL_FNPTR)
    return FALSE;

  return (*key->signature->public_key_verify)(key->context,
                                              signature, signature_len,
                                              hash);
}



size_t
ssh_private_key_max_signature_input_len(SshPrivateKey key)
{
  if (key->signature == NULL ||
      key->signature->private_key_max_signature_input_len == NULL_FNPTR)
    return 0;

  return (*key->signature->private_key_max_signature_input_len)(key->context);
}

size_t
ssh_private_key_max_signature_output_len(SshPrivateKey key)
{
  if (key->signature == NULL ||
      key->signature->private_key_max_signature_output_len == NULL_FNPTR)
    return 0;
  return (*key->signature->private_key_max_signature_output_len)(key->context);
}

/* Return the maximal lenght of bytes which may be decrypted with this
   private key. The result is queried from the corresponding private key
   cryptosystem package with a type-specific function. */

size_t
ssh_private_key_max_decrypt_input_len(SshPrivateKey key)
{
  if (key->encryption == NULL ||
      key->encryption->private_key_max_decrypt_input_len == NULL_FNPTR)
    return 0;

  return (*key->encryption->private_key_max_decrypt_input_len)(key->context);
}

/* Similar to the previous function except this will return the maximum
   output lenght with decryption. */

size_t
ssh_private_key_max_decrypt_output_len(SshPrivateKey key)
{
  if (key->encryption == NULL ||
      key->encryption->private_key_max_decrypt_output_len == NULL_FNPTR)
    return 0;

  return (*key->encryption->private_key_max_decrypt_output_len)(key->context);
}

/* Private key decrypt and encrypt */
SshCryptoStatus
ssh_private_key_decrypt(SshPrivateKey key,
                        const unsigned char *ciphertext,
                        size_t ciphertext_len,
                        unsigned char *plaintext,
                        size_t buffer_len,
                        size_t *plaintext_length_return)
{
  SshRGFHash hash;

  if (key->encryption == NULL ||
      key->encryption->private_key_decrypt == NULL_FNPTR)
    return SSH_CRYPTO_UNSUPPORTED;

  hash = ssh_rgf_hash_allocate(key->encryption->rgf_def);

  if (hash &&
      (*key->encryption->private_key_decrypt)(key->context,
                                              ciphertext,
                                              ciphertext_len,
                                              plaintext,
                                              buffer_len,
                                              plaintext_length_return,
                                              hash))
    return SSH_CRYPTO_OK;

  return SSH_CRYPTO_OPERATION_FAILED;
}

SshCryptoStatus
ssh_private_key_encrypt(SshPrivateKey key,
                        const unsigned char *plaintext,
                        size_t plaintext_len,
                        unsigned char *ciphertext,
                        size_t ciphertext_len,
                        size_t *ciphertext_length_return)
{
  if (key->encryption == NULL ||
      key->encryption->private_key_encrypt == NULL_FNPTR)
    return SSH_CRYPTO_UNSUPPORTED;

  if ((*key->encryption->private_key_encrypt)(key->context,
                                              plaintext,
                                              plaintext_len,
                                              ciphertext,
                                              ciphertext_len,
                                              ciphertext_length_return))

    return SSH_CRYPTO_OK;

  return SSH_CRYPTO_OPERATION_FAILED;
}

SshCryptoStatus
ssh_private_key_sign(SshPrivateKey key,
                     const unsigned char *data,
                     size_t data_len,
                     unsigned char *signature,
                     size_t signature_len,
                     size_t *signature_length_return)
{
  SshRGFHash hash;
  if (key->signature == NULL ||
      key->signature->private_key_sign == NULL_FNPTR)
    return SSH_CRYPTO_UNSUPPORTED;

  hash = ssh_rgf_hash_allocate(key->signature->rgf_def);
  if (hash == NULL)
    return SSH_CRYPTO_OPERATION_FAILED;
  ssh_rgf_hash_update(hash, data, data_len);

  if ((*key->signature->private_key_sign)(key->context,
                                          hash,
                                          signature, signature_len,
                                          signature_length_return))
    return SSH_CRYPTO_OK;

  return SSH_CRYPTO_OPERATION_FAILED;
}


SshCryptoStatus
ssh_private_key_sign_digest(SshPrivateKey key,
                            const unsigned char *digest,
                            size_t digest_len,
                            unsigned char *signature,
                            size_t signature_len,
                            size_t *signature_length_return)
{
  SshRGFHash hash;
  if (key->signature == NULL ||
      key->signature->private_key_sign == NULL_FNPTR)
    return SSH_CRYPTO_UNSUPPORTED;

  hash = ssh_rgf_hash_allocate(key->signature->rgf_def);
  if (hash == NULL)
    return SSH_CRYPTO_OPERATION_FAILED;

  if (ssh_rgf_hash_update_with_digest(hash,
                                      digest, digest_len) == FALSE)
    {
      ssh_rgf_hash_free(hash);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  if ((*key->signature->private_key_sign)(key->context,
                                          hash,
                                          signature, signature_len,
                                          signature_length_return))
    return SSH_CRYPTO_OK;

  return SSH_CRYPTO_OPERATION_FAILED;
}

SshCryptoStatus
ssh_private_key_sign_hash(SshPrivateKey key,
                          SshRGFHash    hash,
                          unsigned char *signature,
                          size_t signature_len,
                          size_t *signature_length_return)
{
  if (key->signature == NULL ||
      key->signature->private_key_sign == NULL_FNPTR)
    return SSH_CRYPTO_UNSUPPORTED;

  if ((*key->signature->private_key_sign)(key->context,
                                          hash,
                                          signature, signature_len,
                                          signature_length_return))
    return SSH_CRYPTO_OK;

  return SSH_CRYPTO_OPERATION_FAILED;
}


/* Diffie-Hellman key exchange method. */

size_t
ssh_pk_group_dh_setup_max_output_length(SshPkGroup group)
{
  if (group->diffie_hellman == NULL ||
      group->diffie_hellman->diffie_hellman_exchange_max_length == NULL_FNPTR)
    return 0;

  return (*group->diffie_hellman->
          diffie_hellman_exchange_max_length)(group->context);
}

size_t
ssh_pk_group_dh_agree_max_output_length(SshPkGroup group)
{
  if (group->diffie_hellman == NULL ||
      group->diffie_hellman->diffie_hellman_secret_value_max_length == NULL_FNPTR)
    return 0;

  return (*group->diffie_hellman->
          diffie_hellman_secret_value_max_length)(group->context);
}

void
ssh_pk_group_dh_secret_free(SshPkGroupDHSecret secret)
{
  /* Remark. Currently we can just free the secret using ssh_free.
     It is possible that in future more elaborate methods are needed. */
  ssh_free(secret);
}

SshCryptoStatus
ssh_pk_group_dh_setup(SshPkGroup group,
                      SshPkGroupDHSecret *secret,
                      unsigned char *exchange,
                      size_t exchange_length,
                      size_t *return_length)
{
  if (group->diffie_hellman == NULL ||
      group->diffie_hellman->diffie_hellman_setup == NULL_FNPTR)
    return SSH_CRYPTO_UNSUPPORTED;

  if ((*group->diffie_hellman->diffie_hellman_setup)(group->context,
                                                     (void **) secret,
                                                     exchange,
                                                     exchange_length,
                                                     return_length))
    return SSH_CRYPTO_OK;

  return SSH_CRYPTO_OPERATION_FAILED;
}

SshCryptoStatus
ssh_pk_group_dh_agree(SshPkGroup group,
                      SshPkGroupDHSecret secret,
                      const unsigned char *exchange,
                      size_t exchange_length,
                      unsigned char *secret_value_buffer,
                      size_t secret_value_buffer_length,
                      size_t *return_length)
{
  if (group->diffie_hellman == NULL ||
      group->diffie_hellman->diffie_hellman_agree == NULL_FNPTR)
    return SSH_CRYPTO_UNSUPPORTED;

  if ((*group->diffie_hellman->
       diffie_hellman_agree)(group->context,
                             (void *) secret,
                             exchange,
                             exchange_length,
                             secret_value_buffer,
                             secret_value_buffer_length,
                             return_length))
    return SSH_CRYPTO_OK;
  return SSH_CRYPTO_OPERATION_FAILED;
}

/* Unified Diffie-Hellman. */

size_t
ssh_pk_group_udh_agree_max_output_length(SshPkGroup group)
{
  if (group->diffie_hellman == NULL ||
      group->diffie_hellman->udh_secret_value_max_length == NULL_FNPTR)
    return 0;
  return (*group->diffie_hellman->
          udh_secret_value_max_length)(group->context);
}

SshCryptoStatus
ssh_pk_group_udh_agree(SshPublicKey public_key,
                       SshPrivateKey private_key,
                       void *secret,
                       const unsigned char *exchange,
                       size_t exchange_length,
                       unsigned char *secret_value_buffer,
                       size_t secret_value_buffer_length,
                       size_t *return_length)
{
  if (private_key->diffie_hellman == NULL ||
      public_key->diffie_hellman == NULL ||
      private_key->diffie_hellman->udh_agree == NULL_FNPTR)
    return SSH_CRYPTO_UNSUPPORTED;

  if ((*private_key->diffie_hellman->
       udh_agree)(public_key->context,
                  private_key->context,
                  secret,
                  exchange,
                  exchange_length,
                  secret_value_buffer,
                  secret_value_buffer_length,
                  return_length))
    return SSH_CRYPTO_OK;
  return SSH_CRYPTO_OPERATION_FAILED;
}

/* ... more to come. */

/* Asyncronous operations implemented using the syncronous software
   implementation if no asynchronous callback was defined for the key. */

/* Start asyncronous public key encryption operation. The library will call
   given callback when operation is done. Callback may be called immediately
   during this call. The function ssh_operation_abort function may be called to
   abort this operation before it finishes, in which case the callback is not
   called and the SshOperationHandle will be NULL. */
SshOperationHandle
ssh_public_key_encrypt_async(SshPublicKey key,
                             const unsigned char *plaintext,
                             size_t plaintext_length,
                             SshPublicKeyEncryptCB callback,
                             void *context)
{
  unsigned char *ciphertext;
  size_t return_length, ciphertext_length;
  SshCryptoStatus status;
  SshRGFHash hash;

  if (key->encryption &&
      key->encryption->public_key_encrypt_async &&
      (hash = ssh_rgf_hash_allocate(key->encryption->rgf_def)) != NULL)
    return (*key->encryption->
            public_key_encrypt_async)(key->context,
                                      plaintext, plaintext_length,
                                      hash,
                                      callback, context);

  /* Do it using the synchronous code. */
  ciphertext_length = ssh_public_key_max_encrypt_output_len(key);
  if ((ciphertext = ssh_malloc(ciphertext_length)) != NULL)
    {
      status = ssh_public_key_encrypt(key,
                                      plaintext, plaintext_length,
                                      ciphertext, ciphertext_length,
                                      &return_length);
      (*callback)(status, ciphertext, return_length, context);
      ssh_free(ciphertext);
    }
  return NULL;
}

/* Start asyncronous public key decrypt operation. The library will call given
   callback when operation is done. Callback may be called immediately during
   this call. The function ssh_operation_abort function may be called to abort
   this operation before it finishes, in which case the callback is not called
   and the SshOperationHandle will be NULL. Key and the ciphertext must remain
   constant during the operation, and can only be freed after the callback is
   called. This operation will  remove padding from the resultant decrypted
   data, if the scheme requires so. */
SshOperationHandle
ssh_public_key_decrypt_async(SshPublicKey key,
                             const unsigned char *ciphertext,
                             size_t ciphertext_len,
                             SshPublicKeyDecryptCB callback,
                             void *context)
{
  unsigned char *plaintext;
  size_t return_length, plaintext_length;
  SshCryptoStatus status;

  if (key->encryption &&
      key->encryption->public_key_decrypt_async != NULL_FNPTR)
    {
      /* Asyncronous operation. */
      return (*key->encryption->
              public_key_decrypt_async)(key->context,
                                        ciphertext,
                                        ciphertext_len,
                                        callback, context);
    }

  plaintext_length = ssh_public_key_max_encrypt_input_len(key);
  if ((plaintext = ssh_malloc(plaintext_length)) != NULL)
    {
      status = ssh_public_key_decrypt(key,
                                      ciphertext, ciphertext_len,
                                      plaintext, plaintext_length,
                                      &return_length);
      (*callback)(status, plaintext, return_length, context);
      ssh_free(plaintext);
    }
  return NULL;
}

/* Start asyncronous public key signature verify operation. The
   library will call given callback when operation is done. Callback
   may be called immediately during this call. The function
   ssh_operation_abort function may be called to abort this operation
   before it finishes, in which case the callback is not called and
   the SshOperationHandle will be NULL. */
SshOperationHandle
ssh_public_key_verify_async(SshPublicKey key,
                            const unsigned char *signature,
                            size_t signature_length,
                            const unsigned char *data,
                            size_t data_length,
                            SshPublicKeyVerifyCB callback,
                            void *context)
{
  if (key->signature && key->signature->public_key_verify_async)
    {
      SshRGFHash hash;

      hash = ssh_rgf_hash_allocate(key->signature->rgf_def);
      if (hash == NULL)
        {
          (*callback)(SSH_CRYPTO_OPERATION_FAILED, NULL, context);
          return NULL;
        }
      ssh_rgf_hash_update(hash, data, data_length);

      return (*key->signature->
              public_key_verify_async)(key->context,
                                       signature, signature_length,
                                       hash,
                                       callback, context);
    }

  if (ssh_public_key_verify_signature(key,
                                      signature, signature_length,
                                      data, data_length))
    {
      (*callback)(SSH_CRYPTO_OK, NULL, context);
    }
  else
    {
      (*callback)(SSH_CRYPTO_SIGNATURE_CHECK_FAILED, NULL, context);
    }
  return NULL;

}

/* Start asyncronous public key signature verify operation. As
   ssh_public_key_verify_asycn but with this interface one can give
   the exact digest one self. The library will call given callback
   when operation is done. Callback may be called immediately during
   this call. The function ssh_operation_abort function may be called to
   abort this operation before it finishes, in which case the callback
   is not called and the SshOperationHandle will be NULL. */
SshOperationHandle
ssh_public_key_verify_digest_async(SshPublicKey key,
                                   const unsigned char *signature,
                                   size_t signature_length,
                                   const unsigned char *digest,
                                   size_t digest_length,
                                   SshPublicKeyVerifyCB callback,
                                   void *context)
{
  if (key->signature && key->signature->public_key_verify_async)
    {
      SshRGFHash hash;
      hash = ssh_rgf_hash_allocate(key->signature->rgf_def);
      if (hash == NULL)
        {
          (*callback)(SSH_CRYPTO_OPERATION_FAILED, NULL, context);
          return NULL;
        }
      if (ssh_rgf_hash_update_with_digest(hash,
                                          digest, digest_length) == FALSE)
        {
          (*callback)(SSH_CRYPTO_OPERATION_FAILED, NULL, context);
          ssh_rgf_hash_free(hash);
          return NULL;
        }

      return (*key->signature->
              public_key_verify_async)(key->context,
                                       signature, signature_length,
                                       hash,
                                       callback, context);
    }

  if (ssh_public_key_verify_signature_with_digest(key,
                                                  signature,  signature_length,
                                                  digest, digest_length))
    {
      (*callback)(SSH_CRYPTO_OK, NULL, context);
    }
  else
    {
      (*callback)(SSH_CRYPTO_SIGNATURE_CHECK_FAILED, NULL, context);
    }
  return NULL;
}

SshOperationHandle
ssh_public_key_verify_hash_async(SshPublicKey key,
                                 const unsigned char *signature,
                                 size_t signature_length,
                                 SshRGFHash hash,
                                 SshPublicKeyVerifyCB callback,
                                 void *context)
{
  if (key->signature && key->signature->public_key_verify_async)
    {
      return (*key->signature->
              public_key_verify_async)(key->context,
                                       signature, signature_length,
                                       hash,
                                       callback, context);
    }

  if (ssh_public_key_verify_signature_with_hash(key,
                                                signature,
                                                signature_length,
                                                hash))
    {
      (*callback)(SSH_CRYPTO_OK, NULL, context);
    }
  else
    {
      (*callback)(SSH_CRYPTO_SIGNATURE_CHECK_FAILED, NULL, context);
    }
  return NULL;
}


/* Start asyncronous private key decryption operation. The library will call
   given callback when operation is done. Callback may be called immediately
   during this call. The function ssh_operation_abort function may be called to
   abort this operation before it finishes, in which case the callback is not
   called and the SshOperationHandle will be NULL. */
SshOperationHandle
ssh_private_key_decrypt_async(SshPrivateKey key,
                              const unsigned char *ciphertext,
                              size_t ciphertext_length,
                              SshPrivateKeyDecryptCB callback,
                              void *context)
{
  unsigned char *plaintext;
  size_t return_length, plaintext_length;
  SshCryptoStatus status;
  SshRGFHash hash;

  if (key->encryption &&
      (key->encryption->private_key_decrypt_async != NULL_FNPTR) &&
      (hash = ssh_rgf_hash_allocate(key->encryption->rgf_def)) != NULL)
    {
      /* Asyncronous operation. */
      return (*key->encryption->
              private_key_decrypt_async)(key->context,
                                         ciphertext, ciphertext_length,
                                         hash,
                                         callback, context);
    }


  plaintext_length = ssh_private_key_max_decrypt_output_len(key);
  if ((plaintext = ssh_malloc(plaintext_length)) != NULL)
    {
      status = ssh_private_key_decrypt(key,
                                       ciphertext, ciphertext_length,
                                       plaintext, plaintext_length,
                                       &return_length);
      (*callback)(status, plaintext, return_length, context);
      ssh_free(plaintext);
    }
  return NULL;
}

/* Start asyncronous private key encrypt operation. The library will call given
   callback when operation is done. Callback may be called immediately during
   this call. The function ssh_operation_abort function may be called to abort
   this operation before it finishes, in which case the callback is not called
   and the SshOperationHandle will be NULL. Key and the data must remain
   constant during the operation, and can only be freed after the callback is
   called. This operation does not do any kind of hashing for the data, it may
   do padding. */
SshOperationHandle
ssh_private_key_encrypt_async(SshPrivateKey key,
                              const unsigned char *data,
                              size_t data_len,
                              SshPrivateKeyEncryptCB callback,
                              void *context)
{
  unsigned char *ciphertext;
  size_t return_length, ciphertext_length;
  SshCryptoStatus status;

  if (key->encryption && key->encryption->private_key_encrypt_async)
    return (*key->encryption->
            private_key_encrypt_async)(key->context,
                                       data, data_len,
                                       callback, context);

  /* Do it using the synchronous code. */
  ciphertext_length = ssh_private_key_max_decrypt_input_len(key);
  if ((ciphertext = ssh_malloc(ciphertext_length)) != NULL)
    {
      status = ssh_private_key_encrypt(key,
                                       data, data_len,
                                       ciphertext, ciphertext_length,
                                       &return_length);
      (*callback)(status, ciphertext, return_length, context);
      ssh_free(ciphertext);
    }
  return NULL;
}

/* Start asyncronous private key signing operation. The library will
   call given callback when operation is done. Callback may be called
   immediately during this call. The function ssh_operation_abort
   function may be called to abort this operation before it finishes,
   in which case the callback is not called and the SshOperationHandle
   will be NULL. */
SshOperationHandle
ssh_private_key_sign_async(SshPrivateKey key,
                           const unsigned char *data,
                           size_t data_length,
                           SshPrivateKeySignCB callback,
                           void *context)
{
  unsigned char *signature;
  size_t return_length, signature_length;
  SshCryptoStatus status;

  if (key->signature &&
      key->signature->private_key_sign_async != NULL_FNPTR)
    {
      SshRGFHash hash;
      hash = ssh_rgf_hash_allocate(key->signature->rgf_def);
      if (hash == NULL)
        {
          (*callback)(SSH_CRYPTO_OPERATION_FAILED, NULL, 0, context);
          return NULL;
        }
      ssh_rgf_hash_update(hash, data, data_length);

      /* Asyncronous operation. */
      return (*key->signature->
              private_key_sign_async)(key->context,
                                      hash,
                                      callback, context);
    }

  signature_length = ssh_private_key_max_signature_output_len(key);
  if ((signature = ssh_malloc(signature_length)) != NULL)
    {
      status = ssh_private_key_sign(key,
                                    data, data_length,
                                    signature, signature_length,
                                    &return_length);
      (*callback)(status, signature, return_length, context);
      ssh_free(signature);
    }
  return NULL;
}

/* As ssh_private_key_sign but here one can give the hash digest directly. The
   hash which to use can be requested using
   ssh_private_key_derive_signature_hash function. */

SshOperationHandle
ssh_private_key_sign_digest_async(SshPrivateKey key,
                                  const unsigned char *digest,
                                  size_t digest_length,
                                  SshPrivateKeySignCB callback,
                                  void *context)
{
  unsigned char *signature;
  size_t return_length, signature_length;
  SshCryptoStatus status;

  if (key->signature &&
      key->signature->private_key_sign_async != NULL_FNPTR)
    {
      SshRGFHash hash;
      hash = ssh_rgf_hash_allocate(key->signature->rgf_def);
      if (hash == NULL)
        {
          (*callback)(SSH_CRYPTO_OPERATION_FAILED, NULL, 0, context);
          return NULL;
        }
      if (ssh_rgf_hash_update_with_digest(hash,
                                          digest, digest_length) == FALSE)
        {
          (*callback)(SSH_CRYPTO_OPERATION_FAILED, NULL, 0, context);
          ssh_rgf_hash_free(hash);
          return NULL;
        }

      /* Asyncronous operation. */
      return (*key->signature->
              private_key_sign_async)(key->context,
                                      hash,
                                      callback, context);
    }

  signature_length = ssh_private_key_max_signature_output_len(key);
  if ((signature = ssh_malloc(signature_length)) != NULL)
    {
      status =
        ssh_private_key_sign_digest(key,
                                    digest, digest_length,
                                    signature, signature_length,
                                    &return_length);
      (*callback)(status, signature, return_length, context);
      ssh_free(signature);
    }
  return NULL;
}

SshOperationHandle
ssh_private_key_sign_hash_async(SshPrivateKey key,
                                SshRGFHash hash,
                                SshPrivateKeySignCB callback,
                                void *context)
{
  unsigned char *signature;
  size_t return_length, signature_length;
  SshCryptoStatus status;

  if (key->signature &&
      key->signature->private_key_sign_async != NULL_FNPTR)
    {
      /* Asyncronous operation. */
      return (*key->signature->
              private_key_sign_async)(key->context,
                                      hash,
                                      callback, context);
    }

  signature_length = ssh_private_key_max_signature_output_len(key);
  if ((signature = ssh_malloc(signature_length)) != NULL)
    {
      status = ssh_private_key_sign_hash(key, hash,
                                         signature, signature_length,
                                         &return_length);
      (*callback)(status, signature, return_length, context);
      ssh_free(signature);
    }
  return NULL;
}

/* Start asyncronous Diffie-Hellman setup operation. The library will call
   given callback when operation is done. Callback may be called immediately
   during this call. The function ssh_operation_abort function may be called to
   abort this operation before it finishes, in which case the callback is not
   called and the SshOperationHandle will be NULL. */
SshOperationHandle
ssh_pk_group_dh_setup_async(SshPkGroup group,
                            SshPkGroupDHSetup callback,
                            void *context)
{
  SshPkGroupDHSecret secret;
  unsigned char *exchange;
  size_t exchange_length;
  size_t return_length;
  SshCryptoStatus status;

  if (group->diffie_hellman &&
      group->diffie_hellman->diffie_hellman_setup_async)
    {
      return (*group->diffie_hellman->diffie_hellman_setup_async)
        (group->context,callback, context);
    }

  exchange_length = ssh_pk_group_dh_setup_max_output_length(group);
  if ((exchange = ssh_malloc(exchange_length)) != NULL)
    {
      status = ssh_pk_group_dh_setup(group, &secret,
                                     exchange, exchange_length,
                                     &return_length);
      (*callback)(status, secret, exchange, return_length, context);
      ssh_free(exchange);
    }
  return NULL;
}


/* Start asyncronous Diffie-Hellman agree operation. The library will call
   given callback when operation is done. Callback may be called immediately
   during this call. The function ssh_operation_abort function may be called to
   abort this operation before it finishes, in which case the callback is not
   called and the SshOperationHandle will be NULL. */
SshOperationHandle
ssh_pk_group_dh_agree_async(SshPkGroup group,
                            SshPkGroupDHSecret secret,
                            const unsigned char *exchange,
                            size_t exchange_length,
                            SshPkGroupDHAgree callback,
                            void *context)
{
  unsigned char *secret_buffer;
  size_t secret_buffer_length;
  size_t return_length;
  SshCryptoStatus status;

  if (group->diffie_hellman &&
      group->diffie_hellman->diffie_hellman_agree_async)
    {
      return (*group->diffie_hellman->diffie_hellman_agree_async)
        (group->context,
         (void *)secret,
         exchange, exchange_length,
         callback, context);
    }

  secret_buffer_length = ssh_pk_group_dh_agree_max_output_length(group);
  if ((secret_buffer = ssh_malloc(secret_buffer_length)) != NULL)
    {
      status = ssh_pk_group_dh_agree(group, secret,
                                     exchange, exchange_length,
                                     secret_buffer, secret_buffer_length,
                                     &return_length);
      (*callback)(status, secret_buffer, return_length, context);
      ssh_free(secret_buffer);
    }

  return NULL;
}

/* Compute a secret value from an exchange value and a secret. Secret will
   be free and deleted (thus no need to free it otherways). This function
   destroys the secret even when an error occurs.

   Callback is called in the same way than in
   ssh_pk_group_diffie_hellman_agree_async.

   As with Diffie-Hellman the returned secret_value_buffer contains bits known
   only by participants of the exchange. However, the buffer contains lots of
   redundancy, thus some function should be run over it before use as keying
   material etc. */

SshOperationHandle
ssh_pk_group_udh_agree_async(SshPublicKey public_key,
                             SshPrivateKey private_key,
                             SshPkGroupDHSecret secret,
                             const unsigned char *exchange,
                             size_t exchange_length,
                             SshPkGroupDHAgree callback,
                             void *context)
{
  unsigned char *secret_buffer;
  size_t secret_buffer_length;
  size_t return_length;
  SshCryptoStatus status;
  SshPkGroup group;

  if (private_key->diffie_hellman &&
      private_key->diffie_hellman->udh_agree_async)
    {
      return (*private_key->diffie_hellman->udh_agree_async)
        (public_key->context, private_key->context,
         secret,
         exchange, exchange_length,
         callback, context);
    }

  group = ssh_public_key_derive_pk_group(public_key);
  secret_buffer_length = ssh_pk_group_udh_agree_max_output_length(group);
  if ((secret_buffer = ssh_malloc(secret_buffer_length)) != NULL)
    {
      status = ssh_pk_group_udh_agree(public_key, private_key,
                                      secret,
                                      exchange, exchange_length,
                                      secret_buffer, secret_buffer_length,
                                      &return_length);
      (*callback)(status, secret_buffer, return_length, context);
      ssh_free(secret_buffer);
    }
  return NULL;
}
