/*
  
  Authors: Antti Huima   <huima@ssh.fi>
           Mika Kojo     <mkojo@ssh.fi>
           Patrick Irwin <irwin@ssh.fi>
  
  Copyright (C) 2001 SSH Communications Security Corp, Finland
  All rights reserved.
  
  Created: Sun  Nov 25 19:10:02 2001 [irwin]
  
  This file contains generic functions to generate random 
  multiple-precision integers.
  
*/

#include "sshincludes.h"
#include "sshmp.h"
#include "sshgenmp.h"
#include "sshcrypt.h"

#define SSH_DEBUG_MODULE "SshGenMPInteger"

/* Generate a random integer (using the cryptographically strong
   random number generator). */

void ssh_mp_random_integer(SshMPInt ret, unsigned int bits)
{
  unsigned int i;
  SshUInt32 limb;

  ssh_mp_set_ui(ret, 0);
  /* Loop 32 bit limbs */
  for (i = 0; i < bits; i += 32)
    {
      /* Construct one limb */
      limb = (((SshUInt32)ssh_random_get_byte() << 24) |
              ((SshUInt32)ssh_random_get_byte() << 16) |
              ((SshUInt32)ssh_random_get_byte() << 8) |
              ((SshUInt32)ssh_random_get_byte() & 0xff));
      /* Shift and add */
      ssh_mp_mul_2exp(ret, ret, 32);
      ssh_mp_add_ui(ret, ret, limb);
    }
  /* Cut unneeded bits off */
  ssh_mp_mod_2exp(ret, ret, bits);
}


/* Get random number mod 'modulo' */

/* Random number with some sense in getting only a small number of
   bits. This will avoid most of the extra bits. However, we could
   do it in many other ways too. Like we could distribute the random bits
   in reasonably random fashion around the available size. This would
   ensure that cryptographical use would be slightly safer. */
void ssh_mp_mod_random_entropy(SshMPInt op, const SshMPInt modulo,
                               unsigned int bits)
{
  ssh_mp_random_integer(op, bits);
  ssh_mp_mod(op, op, modulo);
}

/* Just plain _modular_ random number generation. */
void ssh_mp_mod_random(SshMPInt op, const SshMPInt modulo)
{
  unsigned int bits;

  bits = ssh_mp_bit_size(modulo);
  ssh_mp_random_integer(op, bits);
  ssh_mp_mod(op, op, modulo);
}





