/*

  dlfix.c

  Author: Mika Kojo <mkojo@ssh.fi>

  Copyright (C) 1997-2001 SSH Communications Security Oy, Espoo, Finland
  All rights reserved.

  Created: Mon Jul 21 15:01:19 1997 [mkojo]

  Discrete logarithm predefined parameters.

  */

/*
 * $Id: dlfix.c,v 1.3 2001/11/13 13:42:59 irwin Exp $
 * $Log: dlfix.c,v $ * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * $EndLog$
 */

#include "sshincludes.h"
#include "sshmp.h"
#include "sshcrypt.h"
#include "sshbuffer.h"

/* Structure definition of parameters that are needed to store. */

typedef struct
{
  const char *name;
  const char *p;
  const char *q;
  const char *g;
} SshDLFixedParams;

/* All fixed parameters should satisfy atleast:

   - generate randomly (as much as possible)
   - p = q*c, where p and q are both prime
   - g^q (mod p) = 1

   Those restrictions can be ignored somewhat but then naming should not
   include "ssh". XXX */

#define SSH_DEFAULT_DL_MODP_PARAM_NAME "ssh-dl-modp-group-1024bit-1"

const SshDLFixedParams ssh_dlp_fixed_params[] =
{
  {
    /* 1024 bits with 160 bit order. */
    "ssh-dl-modp-group-dsa-1024bit-1",

     /* p */
    "18870663758990450276373834564151209634609185541696849681710763012"
    "47168050869633674293178701749077483982698046837347550634094699186"
    "01446907583820305901431915822615633146396545907952461810328182170"
    "35078613084684987462347222332164074368740586436373161810202065699"
    "4755732156870258013519880675646961814649650297159",

    /* q */
    "994432737688160994497917820104112624205251325913",

    /* g */
    "13752365807134022918504845143590215341528782407193227118193168220"
    "74331779978192018552914874329836913766220048777129900873815708450"
    "16796174527842910698477887372037694495736629521026242476079522482"
    "50227332682970988562298735692890934535992768521461586958206432475"
    "6777888883265989982517946734947352536810316486901"
  },

  {
    /* 1024 bits DSA style. */
    "ssh-dl-modp-group-1024bit-1",

    "179769313486231590770839156793787453197860296048756011706444"
    "423684197180216158519368947833795864925541502180565485980503"
    "646440548199239100050792877003355816639229553136239076508735"
    "759914822574862575007425302077447712589550957937778424442426"
    "617334727629299387668709205606050270810842907692932019128194"
    "467627007",
    "898846567431157953854195783968937265989301480243780058532222"
    "118420985901080792596844739168979324627707510902827429902518"
    "232202740996195500253964385016779083196147765681195382543678"
    "799574112874312875037126510387238562947754789688892122212133"
    "086673638146496938343546028030251354054214538464660095640972"
    "33813503",
     "2" },

  /* IKE groups. */
  {
    "ietf-ike-grp-modp-768",
    "0x"
    "FFFFFFFF FFFFFFFF C90FDAA2 2168C234 C4C6628B 80DC1CD1"
    "29024E08 8A67CC74 020BBEA6 3B139B22 514A0879 8E3404DD"
    "EF9519B3 CD3A431B 302B0A6D F25F1437 4FE1356D 6D51C245"
    "E485B576 625E7EC6 F44C42E9 A63A3620 FFFFFFFF FFFFFFFF",
    "0x"
    "7FFFFFFF FFFFFFFF E487ED51 10B4611A 62633145 C06E0E68"
    "94812704 4533E63A 0105DF53 1D89CD91 28A5043C C71A026E"
    "F7CA8CD9 E69D218D 98158536 F92F8A1B A7F09AB6 B6A8E122"
    "F242DABB 312F3F63 7A262174 D31D1B10 7FFFFFFF FFFFFFFF",
    "0x2"
  },
  {
    "ietf-ike-grp-modp-1024",
    /* prime */
    "0x"
    "FFFFFFFF FFFFFFFF C90FDAA2 2168C234 C4C6628B 80DC1CD1"
    "29024E08 8A67CC74 020BBEA6 3B139B22 514A0879 8E3404DD"
    "EF9519B3 CD3A431B 302B0A6D F25F1437 4FE1356D 6D51C245"
    "E485B576 625E7EC6 F44C42E9 A637ED6B 0BFF5CB6 F406B7ED"
    "EE386BFB 5A899FA5 AE9F2411 7C4B1FE6 49286651 ECE65381"
    "FFFFFFFF FFFFFFFF",
    /* LPF */
    "0x"
    "7FFFFFFF FFFFFFFF E487ED51 10B4611A 62633145 C06E0E68"
    "94812704 4533E63A 0105DF53 1D89CD91 28A5043C C71A026E"
    "F7CA8CD9 E69D218D 98158536 F92F8A1B A7F09AB6 B6A8E122"
    "F242DABB 312F3F63 7A262174 D31BF6B5 85FFAE5B 7A035BF6"
    "F71C35FD AD44CFD2 D74F9208 BE258FF3 24943328 F67329C0"
    "FFFFFFFF FFFFFFFF",
    /* generator */
    "0x2"
  },
  {
    "ietf-ike-grp-modp-1536",
    /* prime */
    "0x"
    "FFFFFFFF FFFFFFFF C90FDAA2 2168C234 C4C6628B 80DC1CD1"
    "29024E08 8A67CC74 020BBEA6 3B139B22 514A0879 8E3404DD"
    "EF9519B3 CD3A431B 302B0A6D F25F1437 4FE1356D 6D51C245"
    "E485B576 625E7EC6 F44C42E9 A637ED6B 0BFF5CB6 F406B7ED"
    "EE386BFB 5A899FA5 AE9F2411 7C4B1FE6 49286651 ECE45B3D"
    "C2007CB8 A163BF05 98DA4836 1C55D39A 69163FA8 FD24CF5F"
    "83655D23 DCA3AD96 1C62F356 208552BB 9ED52907 7096966D"
    "670C354E 4ABC9804 F1746C08 CA237327 FFFFFFFF FFFFFFFF",
    /* LPF */
    "0x"
    "7FFFFFFF FFFFFFFF E487ED51 10B4611A 62633145 C06E0E68"
    "94812704 4533E63A 0105DF53 1D89CD91 28A5043C C71A026E"
    "F7CA8CD9 E69D218D 98158536 F92F8A1B A7F09AB6 B6A8E122"
    "F242DABB 312F3F63 7A262174 D31BF6B5 85FFAE5B 7A035BF6"
    "F71C35FD AD44CFD2 D74F9208 BE258FF3 24943328 F6722D9E"
    "E1003E5C 50B1DF82 CC6D241B 0E2AE9CD 348B1FD4 7E9267AF"
    "C1B2AE91 EE51D6CB 0E3179AB 1042A95D CF6A9483 B84B4B36"
    "B3861AA7 255E4C02 78BA3604 6511B993 FFFFFFFF FFFFFFFF",
    /* generator */
    "0x2"
  },
  {
    "ietf-ike-grp-modp-2048",
    /* prime */
    "0x"
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1"
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD"
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245"
    "E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7ED"
    "EE386BFB5A899FA5AE9F24117C4B1FE649286651ECE45B3D"
    "C2007CB8A163BF0598DA48361C55D39A69163FA8FD24CF5F"
    "83655D23DCA3AD961C62F356208552BB9ED529077096966D"
    "670C354E4ABC9804F1746C08CA18217C32905E462E36CE3B"
    "E39E772C180E86039B2783A2EC07A28FB5C55DF06F4C52C9"
    "DE2BCBF6955817183995497CEA956AE515D2261898FA0510"
    "15728E5A8AACAA68FFFFFFFFFFFFFFFF",
    /* LPF */
    "0x"
    "7FFFFFFFFFFFFFFFE487ED5110B4611A62633145C06E0E68948127044533E63A0105"
    "DF531D89CD9128A5043CC71A026EF7CA8CD9E69D218D98158536F92F8A1BA7F09AB6"
    "B6A8E122F242DABB312F3F637A262174D31BF6B585FFAE5B7A035BF6F71C35FDAD44"
    "CFD2D74F9208BE258FF324943328F6722D9EE1003E5C50B1DF82CC6D241B0E2AE9CD"
    "348B1FD47E9267AFC1B2AE91EE51D6CB0E3179AB1042A95DCF6A9483B84B4B36B386"
    "1AA7255E4C0278BA3604650C10BE19482F23171B671DF1CF3B960C074301CD93C1D1"
    "7603D147DAE2AEF837A62964EF15E5FB4AAC0B8C1CCAA4BE754AB5728AE9130C4C7D"
    "02880AB9472D455655347FFFFFFFFFFFFFFF",
    /* generator */
    "0x2"
  },
  {
    "ietf-ike-grp-modp-3072",
    /* prime */
    "0x"
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1"
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD"
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245"
    "E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7ED"
    "EE386BFB5A899FA5AE9F24117C4B1FE649286651ECE45B3D"
    "C2007CB8A163BF0598DA48361C55D39A69163FA8FD24CF5F"
    "83655D23DCA3AD961C62F356208552BB9ED529077096966D"
    "670C354E4ABC9804F1746C08CA18217C32905E462E36CE3B"
    "E39E772C180E86039B2783A2EC07A28FB5C55DF06F4C52C9"
    "DE2BCBF6955817183995497CEA956AE515D2261898FA0510"
    "15728E5A8AAAC42DAD33170D04507A33A85521ABDF1CBA64"
    "ECFB850458DBEF0A8AEA71575D060C7DB3970F85A6E1E4C7"
    "ABF5AE8CDB0933D71E8C94E04A25619DCEE3D2261AD2EE6B"
    "F12FFA06D98A0864D87602733EC86A64521F2B18177B200C"
    "BBE117577A615D6C770988C0BAD946E208E24FA074E5AB31"
    "43DB5BFCE0FD108E4B82D120A93AD2CAFFFFFFFFFFFFFFFF",
    /* LPF */
    "0x"
    "7FFFFFFFFFFFFFFFE487ED5110B4611A62633145C06E0E68948127044533E63A0105"
    "DF531D89CD9128A5043CC71A026EF7CA8CD9E69D218D98158536F92F8A1BA7F09AB6"
    "B6A8E122F242DABB312F3F637A262174D31BF6B585FFAE5B7A035BF6F71C35FDAD44"
    "CFD2D74F9208BE258FF324943328F6722D9EE1003E5C50B1DF82CC6D241B0E2AE9CD"
    "348B1FD47E9267AFC1B2AE91EE51D6CB0E3179AB1042A95DCF6A9483B84B4B36B386"
    "1AA7255E4C0278BA3604650C10BE19482F23171B671DF1CF3B960C074301CD93C1D1"
    "7603D147DAE2AEF837A62964EF15E5FB4AAC0B8C1CCAA4BE754AB5728AE9130C4C7D"
    "02880AB9472D45556216D6998B8682283D19D42A90D5EF8E5D32767DC2822C6DF785"
    "457538ABAE83063ED9CB87C2D370F263D5FAD7466D8499EB8F464A702512B0CEE771"
    "E9130D697735F897FD036CC504326C3B01399F643532290F958C0BBD90065DF08BAB"
    "BD30AEB63B84C4605D6CA371047127D03A72D598A1EDADFE707E884725C16890549D"
    "69657FFFFFFFFFFFFFFF",
    /* generator */
    "0x2"
  },
  {
    "ietf-ike-grp-modp-4096",
    /* prime */
    "0x"
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1"
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD"
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245"
    "E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7ED"
    "EE386BFB5A899FA5AE9F24117C4B1FE649286651ECE45B3D"
    "C2007CB8A163BF0598DA48361C55D39A69163FA8FD24CF5F"
    "83655D23DCA3AD961C62F356208552BB9ED529077096966D"
    "670C354E4ABC9804F1746C08CA18217C32905E462E36CE3B"
    "E39E772C180E86039B2783A2EC07A28FB5C55DF06F4C52C9"
    "DE2BCBF6955817183995497CEA956AE515D2261898FA0510"
    "15728E5A8AAAC42DAD33170D04507A33A85521ABDF1CBA64"
    "ECFB850458DBEF0A8AEA71575D060C7DB3970F85A6E1E4C7"
    "ABF5AE8CDB0933D71E8C94E04A25619DCEE3D2261AD2EE6B"
    "F12FFA06D98A0864D87602733EC86A64521F2B18177B200C"
    "BBE117577A615D6C770988C0BAD946E208E24FA074E5AB31"
    "43DB5BFCE0FD108E4B82D120A92108011A723C12A787E6D7"
    "88719A10BDBA5B2699C327186AF4E23C1A946834B6150BDA"
    "2583E9CA2AD44CE8DBBBC2DB04DE8EF92E8EFC141FBECAA6"
    "287C59474E6BC05D99B2964FA090C3A2233BA186515BE7ED"
    "1F612970CEE2D7AFB81BDD762170481CD0069127D5B05AA9"
    "93B4EA988D8FDDC186FFB7DC90A6C08F4DF435C934063199"
    "FFFFFFFFFFFFFFFF",
    /* LPF */
    "0x"
    "7FFFFFFFFFFFFFFFE487ED5110B4611A62633145C06E0E68948127044533E63A0105"
    "DF531D89CD9128A5043CC71A026EF7CA8CD9E69D218D98158536F92F8A1BA7F09AB6"
    "B6A8E122F242DABB312F3F637A262174D31BF6B585FFAE5B7A035BF6F71C35FDAD44"
    "CFD2D74F9208BE258FF324943328F6722D9EE1003E5C50B1DF82CC6D241B0E2AE9CD"
    "348B1FD47E9267AFC1B2AE91EE51D6CB0E3179AB1042A95DCF6A9483B84B4B36B386"
    "1AA7255E4C0278BA3604650C10BE19482F23171B671DF1CF3B960C074301CD93C1D1"
    "7603D147DAE2AEF837A62964EF15E5FB4AAC0B8C1CCAA4BE754AB5728AE9130C4C7D"
    "02880AB9472D45556216D6998B8682283D19D42A90D5EF8E5D32767DC2822C6DF785"
    "457538ABAE83063ED9CB87C2D370F263D5FAD7466D8499EB8F464A702512B0CEE771"
    "E9130D697735F897FD036CC504326C3B01399F643532290F958C0BBD90065DF08BAB"
    "BD30AEB63B84C4605D6CA371047127D03A72D598A1EDADFE707E884725C168905490"
    "84008D391E0953C3F36BC438CD085EDD2D934CE1938C357A711E0D4A341A5B0A85ED"
    "12C1F4E5156A26746DDDE16D826F477C97477E0A0FDF6553143E2CA3A735E02ECCD9"
    "4B27D04861D1119DD0C328ADF3F68FB094B867716BD7DC0DEEBB10B8240E68034893"
    "EAD82D54C9DA754C46C7EEE0C37FDBEE48536047A6FA1AE49A0318CCFFFFFFFFFFFF"
    "FFFF",
    /* generator */
    "0x2"
  },
  {
    "ietf-ike-grp-modp-8192",
    /* prime */
    "0x"
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1"
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD"
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245"
    "E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7ED"
    "EE386BFB5A899FA5AE9F24117C4B1FE649286651ECE45B3D"
    "C2007CB8A163BF0598DA48361C55D39A69163FA8FD24CF5F"
    "83655D23DCA3AD961C62F356208552BB9ED529077096966D"
    "670C354E4ABC9804F1746C08CA18217C32905E462E36CE3B"
    "E39E772C180E86039B2783A2EC07A28FB5C55DF06F4C52C9"
    "DE2BCBF6955817183995497CEA956AE515D2261898FA0510"
    "15728E5A8AAAC42DAD33170D04507A33A85521ABDF1CBA64"
    "ECFB850458DBEF0A8AEA71575D060C7DB3970F85A6E1E4C7"
    "ABF5AE8CDB0933D71E8C94E04A25619DCEE3D2261AD2EE6B"
    "F12FFA06D98A0864D87602733EC86A64521F2B18177B200C"
    "BBE117577A615D6C770988C0BAD946E208E24FA074E5AB31"
    "43DB5BFCE0FD108E4B82D120A92108011A723C12A787E6D7"
    "88719A10BDBA5B2699C327186AF4E23C1A946834B6150BDA"
    "2583E9CA2AD44CE8DBBBC2DB04DE8EF92E8EFC141FBECAA6"
    "287C59474E6BC05D99B2964FA090C3A2233BA186515BE7ED"
    "1F612970CEE2D7AFB81BDD762170481CD0069127D5B05AA9"
    "93B4EA988D8FDDC186FFB7DC90A6C08F4DF435C934028492"
    "36C3FAB4D27C7026C1D4DCB2602646DEC9751E763DBA37BD"
    "F8FF9406AD9E530EE5DB382F413001AEB06A53ED9027D831"
    "179727B0865A8918DA3EDBEBCF9B14ED44CE6CBACED4BB1B"
    "DB7F1447E6CC254B332051512BD7AF426FB8F401378CD2BF"
    "5983CA01C64B92ECF032EA15D1721D03F482D7CE6E74FEF6"
    "D55E702F46980C82B5A84031900B1C9E59E7C97FBEC7E8F3"
    "23A97A7E36CC88BE0F1D45B7FF585AC54BD407B22B4154AA"
    "CC8F6D7EBF48E1D814CC5ED20F8037E0A79715EEF29BE328"
    "06A1D58BB7C5DA76F550AA3D8A1FBFF0EB19CCB1A313D55C"
    "DA56C9EC2EF29632387FE8D76E3C0468043E8F663F4860EE"
    "12BF2D5B0B7474D6E694F91E6DBE115974A3926F12FEE5E4"
    "38777CB6A932DF8CD8BEC4D073B931BA3BC832B68D9DD300"
    "741FA7BF8AFC47ED2576F6936BA424663AAB639C5AE4F568"
    "3423B4742BF1C978238F16CBE39D652DE3FDB8BEFC848AD9"
    "22222E04A4037C0713EB57A81A23F0C73473FC646CEA306B"
    "4BCBC8862F8385DDFA9D4B7FA2C087E879683303ED5BDD3A"
    "062B3CF5B3A278A66D2A13F83F44F82DDF310EE074AB6A36"
    "4597E899A0255DC164F31CC50846851DF9AB48195DED7EA1"
    "B1D510BD7EE74D73FAF36BC31ECFA268359046F4EB879F92"
    "4009438B481C6CD7889A002ED5EE382BC9190DA6FC026E47"
    "9558E4475677E9AA9E3050E2765694DFC81F56E880B96E71"
    "60C980DD98EDD3DFFFFFFFFFFFFFFFFF",
    /* LPF */
    "0x"
    "7FFFFFFFFFFFFFFFE487ED5110B4611A62633145C06E0E68948127044533E63A0105"
    "DF531D89CD9128A5043CC71A026EF7CA8CD9E69D218D98158536F92F8A1BA7F09AB6"
    "B6A8E122F242DABB312F3F637A262174D31BF6B585FFAE5B7A035BF6F71C35FDAD44"
    "CFD2D74F9208BE258FF324943328F6722D9EE1003E5C50B1DF82CC6D241B0E2AE9CD"
    "348B1FD47E9267AFC1B2AE91EE51D6CB0E3179AB1042A95DCF6A9483B84B4B36B386"
    "1AA7255E4C0278BA3604650C10BE19482F23171B671DF1CF3B960C074301CD93C1D1"
    "7603D147DAE2AEF837A62964EF15E5FB4AAC0B8C1CCAA4BE754AB5728AE9130C4C7D"
    "02880AB9472D45556216D6998B8682283D19D42A90D5EF8E5D32767DC2822C6DF785"
    "457538ABAE83063ED9CB87C2D370F263D5FAD7466D8499EB8F464A702512B0CEE771"
    "E9130D697735F897FD036CC504326C3B01399F643532290F958C0BBD90065DF08BAB"
    "BD30AEB63B84C4605D6CA371047127D03A72D598A1EDADFE707E884725C168905490"
    "84008D391E0953C3F36BC438CD085EDD2D934CE1938C357A711E0D4A341A5B0A85ED"
    "12C1F4E5156A26746DDDE16D826F477C97477E0A0FDF6553143E2CA3A735E02ECCD9"
    "4B27D04861D1119DD0C328ADF3F68FB094B867716BD7DC0DEEBB10B8240E68034893"
    "EAD82D54C9DA754C46C7EEE0C37FDBEE48536047A6FA1AE49A0142491B61FD5A693E"
    "381360EA6E593013236F64BA8F3B1EDD1BDEFC7FCA0356CF298772ED9C17A09800D7"
    "583529F6C813EC188BCB93D8432D448C6D1F6DF5E7CD8A76A267365D676A5D8DEDBF"
    "8A23F36612A5999028A895EBD7A137DC7A009BC6695FACC1E500E325C9767819750A"
    "E8B90E81FA416BE7373A7F7B6AAF3817A34C06415AD42018C8058E4F2CF3E4BFDF63"
    "F47991D4BD3F1B66445F078EA2DBFFAC2D62A5EA03D915A0AA556647B6BF5FA470EC"
    "0A662F6907C01BF053CB8AF7794DF1940350EAC5DBE2ED3B7AA8551EC50FDFF8758C"
    "E658D189EAAE6D2B64F617794B191C3FF46BB71E0234021F47B31FA43077095F96AD"
    "85BA3A6B734A7C8F36DF08ACBA51C937897F72F21C3BBE5B54996FC66C5F626839DC"
    "98DD1DE4195B46CEE9803A0FD3DFC57E23F692BB7B49B5D212331D55B1CE2D727AB4"
    "1A11DA3A15F8E4BC11C78B65F1CEB296F1FEDC5F7E42456C911117025201BE0389F5"
    "ABD40D11F8639A39FE3236751835A5E5E44317C1C2EEFD4EA5BFD16043F43CB41981"
    "F6ADEE9D03159E7AD9D13C53369509FC1FA27C16EF9887703A55B51B22CBF44CD012"
    "AEE0B2798E628423428EFCD5A40CAEF6BF50D8EA885EBF73A6B9FD79B5E18F67D134"
    "1AC8237A75C3CFC92004A1C5A40E366BC44D00176AF71C15E48C86D37E013723CAAC"
    "7223AB3BF4D54F1828713B2B4A6FE40FAB74405CB738B064C06ECC76E9EFFFFFFFFF"
    "FFFFFFFF",
    /* generator */
    "0x2"
  },
  { NULL }
};

char *ssh_dlp_param_get_predefined_groups(void)
{
  char *list = NULL;
  SshBufferStruct buffer;
  unsigned int i;

  ssh_buffer_init(&buffer);
  for (i = 0; ssh_dlp_fixed_params[i].name; i++)
    {
      if (ssh_buffer_len(&buffer) > 0)
        {
          if (ssh_buffer_append(&buffer, (unsigned char *) ",", 1)
              != SSH_BUFFER_OK)
            goto failure;
        }
      if (ssh_buffer_append(&buffer,
                            (unsigned char *) ssh_dlp_fixed_params[i].name,
                            strlen(ssh_dlp_fixed_params[i].name))
          != SSH_BUFFER_OK)
        goto failure;
    }

  if (ssh_buffer_append(&buffer, (unsigned char *) "\0", 1) == SSH_BUFFER_OK)
    list = ssh_strdup(ssh_buffer_ptr(&buffer));

 failure:
  ssh_buffer_uninit(&buffer);
  return list;
}

Boolean ssh_dlp_set_param(const char *name, const char **outname,
                          SshMPInteger p, SshMPInteger q, SshMPInteger g)
{
  int i;

  if (name == NULL)
    name = SSH_DEFAULT_DL_MODP_PARAM_NAME;

  for (i = 0; ssh_dlp_fixed_params[i].name; i++)
    {
      if (strcmp(ssh_dlp_fixed_params[i].name, name) == 0)
        break;
    }
  if (ssh_dlp_fixed_params[i].name == NULL)
    return FALSE;

  *outname = ssh_dlp_fixed_params[i].name;

  ssh_mprz_set_str(p, ssh_dlp_fixed_params[i].p, 0);
  ssh_mprz_set_str(q, ssh_dlp_fixed_params[i].q, 0);
  ssh_mprz_set_str(g, ssh_dlp_fixed_params[i].g, 0);
  
  return TRUE;
}

Boolean ssh_dlp_is_predefined_group(SshMPInteger p, SshMPInteger q, 
                                    SshMPInteger g)
{
  unsigned int i;
  SshMPIntegerStruct P, Q, G;
  
  ssh_mprz_init(&P);
  ssh_mprz_init(&Q);
  ssh_mprz_init(&G);
  
  for (i = 0; ssh_dlp_fixed_params[i].name; i++)
    {  
      ssh_mprz_set_str(&P, ssh_dlp_fixed_params[i].p, 0);
      ssh_mprz_set_str(&Q, ssh_dlp_fixed_params[i].q, 0);
      ssh_mprz_set_str(&G, ssh_dlp_fixed_params[i].g, 0);
      
      if ((ssh_mprz_cmp(p, &P) == 0) &&
          (ssh_mprz_cmp(q, &Q) == 0) &&
          (ssh_mprz_cmp(g, &G) == 0))     
        break;
    }
  
  ssh_mprz_clear(&P);
  ssh_mprz_clear(&Q);
  ssh_mprz_clear(&G);
  
  if (ssh_dlp_fixed_params[i].name == NULL)
    return FALSE;
  
  return TRUE;
}

/* dlfix.c */
