/*
   sshpubkey.c

   Author: Vesa Suontama <vsuontam@ssh.fi>

   Copyright (c) 2001-2002 SSH Communications Security Corp, Helsinki, Finland
   All rights reserved

   Created Tue May 08 11:59:24 2001.

   */

#include "sshincludes.h"
#include "sshcrypt.h"
#include "sshpubkey.h"
#include "sshkeyblob1.h"
#include "sshkeyblob2.h"
#include "ssh2pubkeyencode.h"







#define SSH_DEBUG_MODULE "SshPKB"

struct SshPkbTypeNameMap {
  SshPKBType type;
  char *name;
};

static struct SshPkbTypeNameMap type_name_map[] = {
  { SSH_PKB_SSH, "ssh-crypto-library-public-key@ssh.com" },
  { SSH_PKB_SSH_1, "secure-shell-1-public-key@ssh.com" },
  { SSH_PKB_SSH_2, "secure-shell-2-public-key@ssh.com" },
  { SSH_PKB_FROM_X509, "x509-certificate" },
  { SSH_PKB_PKCS12_BROWSER_KEY , "pkcs12-pfx" },
  { SSH_PKB_UNKNOWN, NULL }
};

/* Get type of the public key from the keyblob given as `data' string
   whose length is `len' bytes. The call detects type of the public
   key and fills corresponding value into `kind'. If comment is not
   NULL, and the key format supports visible comments, a copy of
   comment string is returned in `comment'.

   The function returns SSH_CRYPTO_OK if the key format was
   determined, and SSH_CRYPTO_UNKNOWN_KEY_TYPE if not. */
SshCryptoStatus
ssh_pkb_get_info(const unsigned char *data, size_t len,
                 unsigned char **unarmored_data, size_t *unarmored_len,
                 SshPKBType *kind, char **comment)
{



  SshPublicKey pk;
  unsigned long magic;
  unsigned char *tmp, *blob = NULL;
  char *tmpcomment = NULL;
  size_t bloblen;

  /* Check for SSH public key. */
  /* Should check for base64. XXX */
  if (ssh_public_key_import(data, len, &pk) == SSH_CRYPTO_OK)
    {
      ssh_public_key_free(pk);
      if (kind)
        *kind = SSH_PKB_SSH;
      if (comment)
        *comment = ssh_xstrdup("-");
      if (unarmored_len)
        *unarmored_len = len;
      if (unarmored_data)
        *unarmored_data = ssh_xmemdup(data, len);
      return SSH_CRYPTO_OK;
    }






































  tmp = ssh_xmemdup(data, len);
  magic = ssh2_key_blob_decode(tmp, len, FALSE,
                               NULL, &tmpcomment, &blob, &bloblen);

  if (magic == SSH_KEY_MAGIC_SSH1_PUBLIC)
    {
      if (kind)
        *kind = SSH_PKB_SSH_1;
      if (comment)
        *comment = tmpcomment;
      if (unarmored_len)
        *unarmored_len = bloblen;
      if (unarmored_data)
        {
          *unarmored_data = blob;
        }
      else
        {
          memset(blob, 0, bloblen);
          ssh_xfree(blob);
        }
      return SSH_CRYPTO_OK;
    }
  else if (magic == SSH_KEY_MAGIC_PUBLIC)
    {
      if (kind)
        *kind = SSH_PKB_SSH_2;
      if (comment)
        *comment = tmpcomment;
      if (unarmored_len)
        *unarmored_len = bloblen;
      if (unarmored_data)
        {
          *unarmored_data = blob;
        }
      else
        {
          memset(blob, 0, bloblen);
          ssh_xfree(blob);
        }
      return SSH_CRYPTO_OK;
    }
  else if (blob)
    {
      memset(blob, 0, bloblen);
      ssh_xfree(blob);
    }

  SSH_DEBUG(SSH_D_MIDOK, ("Could not deduce the public key type"));
  if (kind)
    *kind = SSH_PKB_UNKNOWN;
  return SSH_CRYPTO_UNKNOWN_KEY_TYPE;
}


/* Returns information about the key type. *Needs_secret is set to
   TRUE if decoding if the blob requires some secret code.  A
   printable name of the type is returned in
   key_type_name_ret. Returns TRUE on success and FALSE otherwise. */
Boolean
ssh_pkb_get_type_info(SshPKBType type,
                      Boolean *needs_secret,
                      const char **key_type_name_ret)
{
  /* XXX pretty basic with so few key types. :) */
  /* Table of key types and their 'properties' */
  static const struct SshPKBPropertiesRec
  {
    SshPKBType type;
    Boolean needs_secret;
    const char *name;
  } ssh_pkb_properties[] =
    {
      { SSH_PKB_UNKNOWN, FALSE, "Unknown"},
      { SSH_PKB_SSH, FALSE, "SSH Key"},
      { SSH_PKB_SSH_2, FALSE, "SSH2 public key." }




    };
  int i, l;

  /* Find the right type. */
  l = sizeof(ssh_pkb_properties) / sizeof(struct SshPKBPropertiesRec);
  for (i = 0; i < l; i++)
    {
      if (type == ssh_pkb_properties[i].type)
        {
          /* Type found. */
          if (needs_secret)
            *needs_secret = ssh_pkb_properties[i].needs_secret;
          if (key_type_name_ret)
            *key_type_name_ret = ssh_pkb_properties[i].name;
          return TRUE;
        }
    }
  /* Type was not found. */
  return FALSE;
}




























































/* Get public key out from the data blob given as `data' and `len'.
   If the `kind' argument specifies an encrypted key format, the
   password (or passphares) needed to decrypt the key is given as
   `password' and `password_len'.

   The function returns SSH_CRYPTO_OK if the public key was
   successfully decoded from the data. If the password is incorrect,
   the return value shall be SSH_CRYPTO_INVALID_PASSPHRASE, and if the
   key blob is not of specified type, the return value shall be
   SSH_CRYPTO_CORRUPTED_KEY_FORMAT.
*/
SshCryptoStatus
ssh_pkb_decode(SshPKBType kind,
               const unsigned char *data, size_t len,
               const unsigned char *password, size_t password_len,
               SshPublicKey *key)
{
  size_t bloblen;
  unsigned char *tmp, *blob;
  char *comment;
  SshPublicKey tmp_key;

  switch (kind)
    {
    case SSH_PKB_SSH:
      if (ssh_public_key_import(data, len, key) == SSH_CRYPTO_OK)
        return SSH_CRYPTO_OK;
      break;

    case SSH_PKB_SSH_1:
    case SSH_PKB_SSH_2:
      tmp = ssh_xmemdup(data, len);
      switch (ssh2_key_blob_decode(tmp, len, FALSE,
                                   NULL, NULL, &blob, &bloblen))
        {
        case SSH_KEY_MAGIC_PUBLIC:
          tmp_key = ssh_decode_pubkeyblob(blob, bloblen);
          if (tmp_key)
            {
              *key = tmp_key;
              return SSH_CRYPTO_OK;
            }
          return SSH_CRYPTO_CORRUPTED_KEY_FORMAT;
        case SSH_KEY_MAGIC_SSH1_PUBLIC:
          return ssh1_decode_pubkeyblob(blob, bloblen, &comment, key);
        default:
          /* It may still be unarmored format.  Let's try. */
          tmp_key = ssh_decode_pubkeyblob(data, len);
          if (tmp_key)
            {
              *key = tmp_key;
              return SSH_CRYPTO_OK;
            }
          return SSH_CRYPTO_CORRUPTED_KEY_FORMAT;
        }
      break;















    case SSH_PKB_UNKNOWN:
    default: SSH_DEBUG(SSH_D_FAIL, ("pkb decode with unknown key type"));
      break;
    }
  return SSH_CRYPTO_CORRUPTED_KEY_FORMAT;
}

/* Maps type identifier to canonical name that can be used in protocols. */
const char *ssh_pkb_type_to_name(SshPKBType kind)
{
  int i;

  for (i = 0; type_name_map[i].name != NULL; i++)
    {
      if (kind == type_name_map[i].type)
        return type_name_map[i].name;
    }
  return NULL;
}

/* Maps canonical name to type identifier. */
SshPKBType ssh_pkb_name_to_type(const char *name)
{
  int i;

  for (i = 0; type_name_map[i].name != NULL; i++)
    {
      if (strcasecmp(name, type_name_map[i].name) == 0)
        return type_name_map[i].type;
    }
  return SSH_PKB_UNKNOWN;
}
