/*

  sshkeyfile.h

  Authors:
        Tatu Ylnen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Simple functions that update user's (key)files.

*/

#ifndef _SSHKEYFILE_H_
#define _SSHKEYFILE_H_

#include "sshuser.h"
#include "sshcrypt.h"

#include "ssh2pubkeyencode.h"

/* the standard cipher used for passphrase encryption */

#ifndef SSH_PASSPHRASE_CIPHER
#define SSH_PASSPHRASE_CIPHER "3des-cbc"
#endif /* SSH_PASSPHRASE_CIPHER */

/* Reads a blob into a buffer. Return TRUE on failure.  The caller must free
   `*blob' with ssh_xfree when no longer needed. */
Boolean ssh_blob_read(SshUser user, const char *fname, unsigned char **blob,
                      size_t *bloblen, void *context);

/* Reads a blob in PEM format into a buffer and decodes it. Returns
   TRUE on failure.  The caller must free `*blob' with ssh_xfree when
   no longer needed. */
Boolean ssh_blob_read_pem(SshUser user, const char *fname, unsigned char **blob,
                          size_t *bloblen, void *context);

/* Write a blob. Return TRUE on failure. */
Boolean ssh_blob_write(SshUser user, const char *fname, mode_t mode,
                       const unsigned char *blob, size_t bloblen,
                       void *context);

/* Read a public/private key blob from a file. Return the magic code
   or SSH_KEY_MAGIC_FAIL on failure.  The caller should free comment
   with ssh_xfree when no longer needed. */
unsigned long ssh2_key_blob_read(SshUser user, const char *fname,
                                 Boolean try_convert_ssh1_cert,
                                 char **comment,
                                 unsigned char **blob,
                                 size_t *bloblen, void *context);

/* Write a key blob. Return TRUE on failure. */
Boolean ssh2_key_blob_write(SshUser user, const char *fname, mode_t mode,
                           unsigned long magic,
                           const char *comment, const unsigned char *key,
                           size_t keylen, void * context);

/* Read a public key from a file. Return NULL on failure.  The caller is
   responsible for freeing `comment' with ssh_xfree when no longer needed.
   `comment' can be NULL. */
SshPublicKey ssh_pubkey_read(SshUser user, const char *fname, char **comment,
                             void * context);

/* Write a public key to a file. Returns TRUE on error. */
Boolean ssh_pubkey_write(SshUser user, const char *fname, const char *comment,
                         SshPublicKey key, void *context);

/* Return value type for ssh_privkey_read(). */
typedef enum
{
  SSH_PRIVKEY_OK = 0,
  SSH_PRIVKEY_KEY_UNREADABLE,
  SSH_PRIVKEY_FAILURE
} SshPrivKeyResult;

/* Read a private key from a file. Return NULL on failure.  The caller
   should free `comment' with ssh_xfree when no longer needed.
   `comment' can be NULL. */
SshPrivKeyResult ssh_privkey_read(SshUser user, const char *fname,
                                  const char *passphrase,
                                  char **comment,
                                  SshPrivateKey *return_privkey);

/* Write a private key to a file with a passphrase. Return TRUE on error. */
Boolean ssh_privkey_write(SshUser user,
                          const char *fname, const char *passphrase,
                          const char *comment,
                          SshPrivateKey key,
                          void *context);

/* Generate a name string from any blob.  String consists of
   caller given string and space and sha1 hash of the blob in hex.
   String is allocated with ssh_xmalloc. */
char *ssh_generate_name_from_blob(char *name,
                                  unsigned char *blob,
                                  size_t bloblen);

#endif /* _SSHKEYFILE_H_ */
