/*

  ssh2pubkeyencode.c

  Authors:
        Tatu Ylonen <ylo@ssh.fi>
        Markku-Juhani Saarinen <mjos@ssh.fi>
        Timo J. Rinne <tri@ssh.fi>
        Sami Lehtinen <sjl@ssh.fi>

  Copyright (C) 1997-1998, 2001 SSH Communications Security Oy, Espoo, Finland
  All rights reserved.

  Encodes and decodes ssh2-format public key blobs.

*/

#include "sshincludes.h"
#include "ssh2pubkeyencode.h"
#include "sshcrypt.h"
#include "sshmp.h" /* was "gmp.h" */
#include "sshencode.h"
#include "sshbufaux.h"




#define SSH_DEBUG_MODULE "Ssh2KeyBlob"

/* define this to dump all keys encoded/decoded */
#undef DUMP_KEYS

/* define this to dump key blobs going in/out */
#undef DUMP_BLOBS

void
ssh_bufaux_put_mp_int_ssh2style(SshBuffer buffer, SshMPInt mp)
{
  unsigned char *data;
  size_t len, bytes;

  len = 4 + (8 + ssh_mp_get_size(mp, 2) + 7)/8;
  data = ssh_xmalloc(len);
  bytes = ssh_mprz_encode_ssh2style(mp, data, len);
  SSH_VERIFY(bytes <= len && bytes > 0);
  ssh_xbuffer_append(buffer, data, bytes);
  ssh_xfree(data);
}

Boolean
ssh_bufaux_get_mp_int_ssh2style(SshBuffer buffer, SshMPInt mp)
{
  unsigned char *data;
  size_t len, bytes;

  len = ssh_buffer_len(buffer);
  data = ssh_buffer_ptr(buffer);
  if ((bytes = ssh_mprz_decode_ssh2style(data, len, mp)) > 0)
    ssh_buffer_consume(buffer, bytes);

  return bytes != 0;
}

/* Encode a public key into a SSH2 format blob. Return size or 0 on
   failure. */

size_t ssh_encode_pubkeyblob(SshPublicKey pubkey, unsigned char **blob)
{
#ifdef SSHDIST_CRYPT_DSA
  SshMPIntC p, q, g, y;  /* DSS public parameters */
#endif /* SSHDIST_CRYPT_DSA */
#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
  SshMPIntC n, e;        /* RSA public parameters */
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */
  SshBuffer buf;
  size_t len;
  char *keytype;

  /* try to determine the exact type of the public key */

  if ((keytype = ssh_public_key_name(pubkey)) == NULL)
    {
      ssh_debug("ssh_encode_pubkeyblob: failed to extract "
                "key type information.");
      return 0;
    }

#ifdef SSHDIST_CRYPT_DSA
  /* -- DSS key type -- */

  /* this is sort of kludge-ish */
  if (strstr(keytype, "sign{dsa-nist") != NULL)
    {
      /* dig out the public parameters */

      ssh_mp_init(p);
      ssh_mp_init(q);
      ssh_mp_init(g);
      ssh_mp_init(y);

      if (ssh_public_key_get_info(pubkey,
                                  SSH_PKF_PRIME_P, p,
                                  SSH_PKF_PRIME_Q, q,
                                  SSH_PKF_GENERATOR_G, g,
                                  SSH_PKF_PUBLIC_Y, y,
                                  SSH_PKF_END)
          != SSH_CRYPTO_OK)
        {
          ssh_debug("ssh_encode_pubkeyblob: failed to get "
                    "internal parameters from a DSS public key.");

          ssh_xfree(keytype);
          return 0;
        }

      /* construct the public key string */

      buf = ssh_xbuffer_allocate();

      ssh_bufaux_put_uint32_string(buf, SSH_SSH_DSS, strlen(SSH_SSH_DSS));
      ssh_bufaux_put_mp_int_ssh2style(buf, p);
      ssh_bufaux_put_mp_int_ssh2style(buf, q);
      ssh_bufaux_put_mp_int_ssh2style(buf, g);
      ssh_bufaux_put_mp_int_ssh2style(buf, y);

#ifdef DUMP_KEYS
      printf("encode:\n p = ");
      ssh_mp_out_str(stdout, 16, p);
      printf("\n q = ");
      ssh_mp_out_str(stdout, 16, q);
      printf("\n g = ");
      ssh_mp_out_str(stdout, 16, g);
      printf("\n y = ");
      ssh_mp_out_str(stdout, 16, y);
      printf("\n\n");
#endif

      ssh_mp_clear(p);
      ssh_mp_clear(q);
      ssh_mp_clear(g);
      ssh_mp_clear(y);

      len = ssh_buffer_len(buf);
      *blob = ssh_xmalloc(len);
      memcpy(*blob, ssh_buffer_ptr(buf), len);
      ssh_buffer_free(buf);
      ssh_xfree(keytype);

      return len;
    }
#endif /* SSHDIST_CRYPT_DSA */

#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
  /* -- RSA key type -- */

  if (strstr(keytype, "sign{rsa-pkcs1") != NULL)
    {
      /* dig out the public parameters */

      ssh_mp_init(e);
      ssh_mp_init(n);

      if (ssh_public_key_get_info(pubkey,
                                  SSH_PKF_PUBLIC_E, e,
                                  SSH_PKF_MODULO_N, n,
                                  SSH_PKF_END)
          != SSH_CRYPTO_OK)
        {
          ssh_debug("ssh_encode_pubkeyblob: failed to get "
                    "internal parameters from a RSA public key.");
          ssh_xfree(keytype);

          return 0;
        }

      buf = ssh_xbuffer_allocate();
      ssh_bufaux_put_uint32_string(buf, SSH_SSH_RSA, strlen(SSH_SSH_RSA));
      ssh_bufaux_put_mp_int_ssh2style(buf, e);
      ssh_bufaux_put_mp_int_ssh2style(buf, n);

#ifdef DUMP_KEYS
      printf("encode:\n e = ");
      ssh_mp_out_str(stdout, 16, e);
      printf("\n n = ");
      ssh_mp_out_str(stdout, 16, n);
      printf("\n\n");
#endif

      len = ssh_buffer_len(buf);
      *blob = ssh_xmalloc(len);
      memcpy(*blob, ssh_buffer_ptr(buf), len);
      ssh_buffer_free(buf);

      ssh_mp_clear(e);
      ssh_mp_clear(n);

      ssh_xfree(keytype);
      return len;
    }
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */

  ssh_debug("ssh_encode_pubkeyblob: unrecognized key type %s", keytype);
  ssh_xfree(keytype);
  return 0;
}


/* Decode a public key blob. Return NULL on failure. */

SshPublicKey ssh_decode_pubkeyblob(const unsigned char *blob, size_t bloblen)
{
  unsigned char *keytype;
  SshPublicKey pubkey;
#ifdef SSHDIST_CRYPT_DSA
  SshMPIntC p, q, g, y;  /* DSS public parameters */
#endif /* SSHDIST_CRYPT_DSA */
#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
  SshMPIntC n, e;          /* RSA public parameters */
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */
  SshCryptoStatus code;
  SshBuffer buf;

#ifdef DUMP_BLOBS
  ssh_debug("ssh_decode_pubkeyblob:");
  ssh_debug_hexdump(0, blob, bloblen);
#endif

  buf = ssh_xbuffer_allocate();
  ssh_xbuffer_append(buf, blob, bloblen);

  if (ssh_decode_buffer(buf,
                        SSH_FORMAT_UINT32_STR, &keytype, NULL,
                        SSH_FORMAT_END) == 0)
    return NULL;

#ifdef SSHDIST_CRYPT_DSA
  /* -- DSS key type -- */

  if (strcmp(SSH_SSH_DSS, (char *) keytype) == 0)
    {
      ssh_mp_init(p);
      ssh_mp_init(q);
      ssh_mp_init(g);
      ssh_mp_init(y);

      if (!ssh_bufaux_get_mp_int_ssh2style(buf, p))
        goto fail1;
      if(!ssh_bufaux_get_mp_int_ssh2style(buf, q))
        goto fail1;
      if (!ssh_bufaux_get_mp_int_ssh2style(buf, g))
        goto fail1;
      if (!ssh_bufaux_get_mp_int_ssh2style(buf, y))
        goto fail1;

      /* ok, construct the public key */

      code = ssh_public_key_define(&pubkey,
                                   SSH_CRYPTO_DSS,
                                   SSH_PKF_PRIME_P, p,
                                   SSH_PKF_PRIME_Q, q,
                                   SSH_PKF_GENERATOR_G, g,
                                   SSH_PKF_PUBLIC_Y, y,
                                   SSH_PKF_END);
#ifdef DUMP_KEYS
      printf("decode:\n p = ");
      ssh_mp_out_str(stdout, 16, p);
      printf("\n q = ");
      ssh_mp_out_str(stdout, 16, q);
      printf("\n g = ");
      ssh_mp_out_str(stdout, 16, g);
      printf("\n y = ");
      ssh_mp_out_str(stdout, 16, y);
      printf("\n\n");
#endif

      ssh_mp_clear(p);
      ssh_mp_clear(q);
      ssh_mp_clear(g);
      ssh_mp_clear(y);

      if (code != SSH_CRYPTO_OK)
        {
          ssh_debug("ssh_decode_pubkeyblob: failed to set the "
                    "parameters of an DSS public key.");
          goto fail1;
        }

      ssh_buffer_free(buf);
      ssh_xfree(keytype);
      return pubkey;
    }
#endif /* SSHDIST_CRYPT_DSA */

#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
  /* -- RSA key type -- */

  if (strcmp(SSH_SSH_RSA, (char *) keytype) == 0)
    {
      ssh_mp_init(e);
      ssh_mp_init(n);

      if (!ssh_bufaux_get_mp_int_ssh2style(buf, e))
        goto fail1;
      if (!ssh_bufaux_get_mp_int_ssh2style(buf, n))
        goto fail1;

#ifdef DUMP_KEYS
      printf("decode:\n e = ");
      ssh_mp_out_str(stdout, 16, e);
      printf("\n n = ");
      ssh_mp_out_str(stdout, 16, n);
      printf("\n\n");
#endif

      code = ssh_public_key_define(&pubkey,
                                   SSH_CRYPTO_RSA,
                                   SSH_PKF_PUBLIC_E, e,
                                   SSH_PKF_MODULO_N, n,
                                   SSH_PKF_END);
      ssh_mp_clear(e);
      ssh_mp_clear(n);

      if (code != SSH_CRYPTO_OK)
        {
          ssh_debug("ssh_decode_pubkeyblob: failed to set the "
                    "parameters of an RSA public key.");
          goto fail1;
        }

      ssh_buffer_free(buf);
      ssh_xfree(keytype);

      return pubkey;
    }
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */

  /* could not identify key type */

  ssh_debug("ssh_decode_pubkeyblob: unrecognized key type %s",
            keytype);

fail1:
  ssh_buffer_free(buf);
  ssh_xfree(keytype);

  return NULL;
}

char *ssh_pubkeyblob_type(const unsigned char *blob, size_t bloblen)
{
  unsigned char *keytype = NULL;

  if (ssh_decode_array(blob, bloblen,
                       SSH_FORMAT_UINT32_STR, &keytype, NULL,
                       SSH_FORMAT_END) == 0)
    return NULL;

  return ((char *)keytype);
}

/* Returns TRUE if pk_format matches one of the defined plain pubkey
   formats */
Boolean
ssh_pubkeyblob_type_plain(const unsigned char *pk_format)
{
  if (strcmp(SSH_SSH_DSS, (char *)pk_format) == 0 ||
      strcmp(SSH_SSH_RSA, (char *)pk_format) == 0)
    return TRUE;
  return FALSE;
}










































































/* Decodes the public key from a public key blob or
   certificate. Returns NULL on failure.  pk_format must be set to
   "ssh_dss", "x509v3-sign-dss", etc. (as defined in ssh2 transport
   layer document).  If the type of the blob does not match the format
   given, returns NULL. */

SshPublicKey ssh_decode_pubkeyblob_general(const unsigned char *blob,
                                           size_t bloblen,
                                           const unsigned char *pk_format)
{
  SshPublicKey pk = NULL;
  char *pk_type = NULL;

  if (ssh_pubkeyblob_type_plain(pk_format))
    {
      /* The blob is an ordinary keyblob. Decode it in the usual way. */
      pk_type = ssh_pubkeyblob_type(blob, bloblen);

      pk = ssh_decode_pubkeyblob(blob, bloblen);
      if (!pk || !pk_type)
        return NULL;
    }









  else
    {
      return NULL;
    }

  /* Got the public key and format. Check that the format matches
     before returning the key. */

  if (strcmp((char *)pk_format, pk_type) != 0)
    {
      ssh_public_key_free(pk);
      pk = NULL;
    }

  ssh_xfree(pk_type);
  return pk;
}
