/*

sshserver.c

  Authors:
        Tatu Ylonen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  SSH server functionality for processing a connection.  Most of the
  implementation is actually shared with the client (in sshcommon.c).

*/

#include "ssh2includes.h"
#include "sshcommon.h"

#include "sshtrans.h"
#include "sshauth.h"
#include "sshconn.h"

#include "sshauthmethods.h"
#include "sshserver.h"
#include "sshuserfiles.h"
#include "sshcipherlist.h"
#include "sshtimeouts.h"
#include "sshnameserver.h"

#define SSH_DEBUG_MODULE "SshServer"


/* Fetches values for the transport parameters (e.g., encryption algorithms)
   from the config data. */

void ssh_server_update_transport_params(SshConfig config,
                                        SshTransportParams params)
{
  char *hlp;

  if (config->ciphers != NULL)
    {
      hlp = ssh_cipher_list_canonicalize(config->ciphers);

      if (hlp)
        {
          ssh_xfree(params->ciphers_c_to_s);
          params->ciphers_c_to_s = ssh_xstrdup(hlp);
          ssh_xfree(params->ciphers_s_to_c);
          params->ciphers_s_to_c = ssh_xstrdup(hlp);

          ssh_xfree(hlp);
        }
    }

  if (config->macs != NULL)
    {
      hlp = ssh_hash_list_canonicalize(config->macs);

      if (hlp)
        {
          ssh_xfree(params->macs_c_to_s);
          params->macs_c_to_s = ssh_xstrdup(hlp);
          ssh_xfree(params->macs_s_to_c);
          params->macs_s_to_c = ssh_xstrdup(hlp);
          ssh_xfree(hlp);
        }
    }



  params->disable_ssh2_compat = config->disable_ssh2_compat;
}


/* Forward declaration. */
void ssh_server_destroy_finalize(void *context);

/* Takes a stream, and creates an SSH server for processing that
   connection.  This closes the stream and returns NULL (without
   calling the destroy function) if an error occurs.  This does not
   free the given server key.  ``config'' must remain
   valid until the server is destroyed; it is not automatically freed.
     `stream'        the connection stream
     `config'        configuration data (not freed, must remain valid)
     `client_host_name'
                     name of the client host (reverse mapped, or if that
                     can't or won't be done, IP-address)
     `disconnect'    function to call on disconnect
     `debug'         function to call on debug message (may be NULL)
     `version_check' version check callback (may be NULL)
     `clean_up'      called during ssh_server_destroy() to let the
                     application cancel timeouts etc, which could cause
                     race conditions on disconnect (may be NULL)
     `context'       context to pass to the callbacks
   The object should be destroyed from the ``disconnect'' callback. */














SshServer ssh_server_wrap(SshStream stream,
                          SshConfig config,
                          const char *client_host_name,
                          SshConnDisconnectProc disconnect,
                          SshConnDebugProc debug,
                          SshVersionCallback version_check,
                          SshAuthPolicyProc auth_policy_proc,
                          SshCommonAuthenticationNotify authenticated_notify,
                          SshServerCleanupCB clean_up,
                          const char *protocol_version,
                          void *context)

{
  SshServer server;



  SshStream trans, auth;
  SshTransportParams params;
  SshTransportCompat compat_flags;
  SshAuthUninitCtx notify_ctx;

  
  /* Create the server object. */
  server = ssh_xcalloc(1, sizeof(*server));
  server->config = config;
  server->clean_up = clean_up;
  











































  /* Create parameters. */
  params = ssh_transport_create_params();
  ssh_server_update_transport_params(config, params);

  /* Create a transport layer protocol object. */
  ssh_debug("ssh_server_wrap: creating transport protocol");
  trans = ssh_transport_server_wrap(stream, 
                                    protocol_version,
                                    params, config->host_keys_ctx,
                                    version_check,
                                    (void *)context,
                                    ssh_common_rekey_notify_cb,
                                    /* SshCommon object is allocated a
                                       below. */
                                    (void *)&(server->common));
  
  ssh_transport_get_compatibility_flags(trans, &compat_flags);
  
  /* Create the authentication methods array for the server. */
  server->methods =
    ssh_server_authentication_initialize(config->allowed_authentications);

  ssh_debug("ssh_server_wrap: creating userauth protocol");

  notify_ctx = ssh_xcalloc(1, sizeof(*notify_ctx));
  notify_ctx->notify_callback = ssh_server_destroy_finalize;
  notify_ctx->notify_context = server;

  auth = ssh_auth_server_wrap(trans, auth_policy_proc, (void *)server,
                              server->methods, (void *)server,
                              ssh_server_authentication_uninitialize,
                              (void *)notify_ctx,
                              server->config->banner_msg,



                              &server->uc);

  /* Create the common part of client/server objects. */
  server->common = ssh_common_wrap(stream, auth, FALSE,
                                   server->config, client_host_name,
                                   disconnect, debug, authenticated_notify,
                                   NULL_FNPTR, NULL_FNPTR, compat_flags,
                                   context);

  /* Set common->ucp to point to the location that will receive the "SshUser"
     object for the authenticated user before SSH_CROSS_AUTHENTICATED is
     delivered. */
  server->common->ucp = &server->uc;
  return server;
}

/* Forcibly destroys the given server. */
  
void ssh_server_destroy(SshServer server)
{
  SSH_DEBUG(3, ("Destroying server."));

  /* Call application clean-up function. */
  if (server->clean_up)
    (*server->clean_up)(server->common->context);
  



  /* XXX kludge; sanitize. */
  if (server->config->host_keys_ctx)
    {
      ssh_host_key_list_free(server->config->host_keys_ctx);
      server->config->host_keys_ctx = NULL;
    }

  ssh_xfree(server->authenticated_client_user_name);
  ssh_xfree(server->authenticated_client_host);

  if (server->ext_auth_handle)
    ssh_operation_abort(server->ext_auth_handle);
  
  /* ssh_common_destroy() might trigger a destruction of the
     authentication layer _immediately_ (without going through the
     event loop), which will lead to the `server' struct being
     destroyed after this call (by
     ssh_server_destroy_finalize()). Because of this,
     ssh_common_destroy() must be called last here. //sjl */
  ssh_common_destroy(server->common);
}

/* This is called by ssh_server_authentication_uninitialize(), which
   is in turn called by the authentication layer, when the methods
   have been cleared. */
void ssh_server_destroy_finalize(void *context)
{
  SshServer server = (SshServer)context;
  SSH_PRECOND(server != NULL);
  
  SSH_DEBUG(3, ("Destroying server completed."));

  memset(server, 'F', sizeof(*server));
  ssh_xfree(server);
}
