/*

  sshrcmd.h

  Author:
        Timo J. Rinne <tri@ssh.com>

  Copyright (C) 2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

*/

#ifndef SSHRCMD_H_INCLUDED
#define SSHRCMD_H_INCLUDED 1

/*

  Execute command in remote host using Secure Shell connection.

  Secure Shell command initiated by this function can't perform any
  user interaction by itself.  User authentication can only be made
  using public key authentication with non-encrypted key or using the
  authentication agent.

  Function returns the process id number of the Secure Shell client
  process that is used in remote connection.  Process id can be passed
  for example to wait, wait3, wait4, waitpid, or kill system calls.
  The wait system call can be used to determine the exit value of the
  Secure Shell child process, which is same as the exit value of the
  remote command or 1 if Secure Shell couldn't authenticate the user
  and therefre failed to execute the remote command.

  Negative return value indicates that either file descriptor setup
  failed or Secure Shell client process couldn't be created.
  
   Arguments:

     remote_host    - Host that is to be connected.

     remote_port    - TCP port to be connected on the remote host.
                      Default port is used if this argument is NULL.

     remote_user    - Username that is used when logging into remote host
                      Current local username is used if this argument is NULL.

     remote_command - Command to be executed in the remote host.  
                      This one can't be NULL.

     fd_stdin       - Pointer to the integer in which the file descriptor
                      of the standard input stream of the child is returned.
                      User can write into this file descriptor and data
                      will appear in the standard input of the remote command.
                      If this argument is NULL, stream is not returned and
                      is closed instead so the remote command only receives
                      EOF as input.

     fd_stdout      - Pointer to the integer in which the file descriptor
                      of the standard output stream of the child is returned.
                      User can read from this file descriptor.  Data appearing
                      in this file descriptor is written by the remote
                      command into its standard output stream.
                      If this argument is NULL, stream is not returned and
                      is closed instead so and remote command can't write
                      to its standard output and output data gets discarded.

     fd_stderr      - Pointer to the integer in which the file descriptor
                      of the standard error stream of the child is returned.
                      User can read from this file descriptor.  Data appearing
                      in this file descriptor is written by the remote
                      command into its standard error stream.
                      If this argument is NULL, stream is not returned and
                      is closed instead so and remote command can't write
                      to its standard error and output data gets discarded.

*/

pid_t ssh_run_remote_command(const char *remote_host,
                             const char *remote_port,
                             const char *remote_user,
                             const char *remote_command,
                             int *fd_stdin,
                             int *fd_stdout,
                             int *fd_stderr);

#endif /* ! SSHRCMD_H_INCLUDED */
/* eof (sshrcmd.h) */
