/*

  sshhostkeyio.c

  Author: Tomi Salo <ttsalo@ssh.fi>

  Copyright (c) 2001 SSH Communications Security, Finland
  All rights reserved.

  Created Fri Feb 23 21:08:49 2001.

  Functions host key IO.

  */

#include "ssh2includes.h"
#include "sshhostkey.h"
#include "sshuserfiles.h"
#include "sshkeyfile.h"
#include "sshadt_list.h"
#include "ssh2pubkeyencode.h"
#include "sshappcommon.h"
#include "sshtimeouts.h"






#include "sshhostkeyi.h"

#define SSH_DEBUG_MODULE "SshHostKeyIO"

/* Forward declarations */















/* Informs the manager that no more keys will be specified and
   reads the keys in. */
void
ssh_host_key_read_keys(SshHostKeysContext ctx,
                       SshConfig config,
                       SshUser user,
                       Boolean read_private_keys,
                       Boolean dont_use_user_dir,
                       SshHostKeyReadCompletion completion,
                       void *completion_context)
{
  SshADTHandle handle;
  SshUser uc;
  Boolean keypair_read_successfully = FALSE;
  char *host_key_file;
  
  SSH_PRECOND(ctx);
  SSH_PRECOND(config);
  SSH_PRECOND(completion != NULL_FNPTR);

  ctx->completion = completion;
  ctx->completion_context = completion_context;
  ctx->log_facility = config->log_facility;
  
  uc = user;

  if (ssh_adt_num_objects(ctx->keylist) == 0)
    {
      /* No complete keypairs were specified in the configuration.
         Let's use the system defaults then... */
      if (!ctx->private_key_file)
        {
          ctx->private_key_file = ssh_xstrdup(SSH_HOSTKEY_FILE);
          if (!ctx->public_key_file)
            {
              ctx->public_key_file = ssh_xstrdup(SSH_PUBLIC_HOSTKEY); 
            }
        }
      else
        {
          /* Private key was specified, derive the name of the public
             key from it */
          ctx->public_key_file = ssh_xmalloc(strlen(ctx->private_key_file) +
                                             strlen(".pub") + 1);
          ssh_snprintf(ctx->public_key_file, strlen(ctx->private_key_file) +
                       strlen(".pub") + 1, "%s.pub", ctx->private_key_file);
        }
      ssh_host_key_try_add_keypair(ctx);

      ssh_debug("Host key pair is not specified, trying to use default '%s'.",
                SSH_HOSTKEY_FILE);
    }

  /* Read the keys */

  /* If `user' is not given, initialize uc. */
  if (!uc)
    if ((uc = ssh_user_initialize(NULL, FALSE)) == NULL)
      ssh_fatal("sshhostkey.c: ssh_user_initialize failed");
  
  handle = ssh_adt_enumerate_start(ctx->keylist);
  host_key_file = ssh_xmalloc(MAXPATHLEN);
  do {
    char *userdir = NULL, *comment;
    unsigned char *public_host_key_blob = NULL;
    SshHostKey hostkey;
    size_t public_host_key_blob_len;

    hostkey = ssh_adt_get(ctx->keylist, handle);
    hostkey->valid = FALSE;

    if (!ssh_app_is_file_name_absolute(hostkey->host_key_file))
      {

        if (ssh_user_uid(uc) == 0 || dont_use_user_dir)
          {
            userdir = ssh_xstrdup(SSH_SERVER_DIR);
          }
        else
          {
            if ((userdir = ssh_userdir(uc, config, TRUE)) == NULL)
              {
                ssh_warning("sshhostkey.c: no ssh2 user directory");
                goto try_next_key;
              }
          }







        ssh_snprintf(host_key_file, MAXPATHLEN, "%s/%s",
                     userdir, hostkey->host_key_file);
      }
    else
      {
        ssh_snprintf(host_key_file, MAXPATHLEN, "%s",
                     hostkey->host_key_file);
      }

    if (read_private_keys)
      {
        SSH_DEBUG(4, ("Reading private host key from %s", host_key_file));
        if (ssh_privkey_read(uc, host_key_file, "", &comment,
                             &hostkey->private_host_key) != SSH_PRIVKEY_OK)
          {
            ssh_warning("ssh_privkey_read from %s failed.", host_key_file);
            goto try_next_key;
          }

        if (comment != NULL)
          {
            if (strlen(comment) > 0)
              SSH_DEBUG(5, ("Key comment: %s", comment));
            ssh_xfree(comment);
          }
      }
    else
      {
        SSH_DEBUG(4, ("Skipping private key reading"));
        hostkey->private_host_key = NULL;
      }

    /* Now the public key blob */
    if (!ssh_app_is_file_name_absolute(hostkey->public_host_key_file))
      {
        ssh_snprintf(host_key_file, MAXPATHLEN, "%s/%s",
                     userdir, hostkey->public_host_key_file);
      }
    else
      {
        ssh_snprintf(host_key_file, MAXPATHLEN, "%s",
                     hostkey->public_host_key_file);
      }

    /* Try to read and decode the blob and extract the algorithm
       type. */




      {
        SSH_DEBUG(2, ("Reading public host key from %s", host_key_file));

        if (ssh2_key_blob_read(uc, host_key_file, TRUE, NULL,
                               &public_host_key_blob,
                               &public_host_key_blob_len, NULL)
            != SSH_KEY_MAGIC_PUBLIC)
          {
            ssh_warning("Unable to load public host key from %s",
                        host_key_file);
            goto try_next_key;
          }

        hostkey->public_key_algorithm =
          ssh_pubkeyblob_type(public_host_key_blob,
                              public_host_key_blob_len);
        if (hostkey->public_key_algorithm == NULL)
          {
            ssh_warning("Unable to get public key type from public key %s",
                        host_key_file);
            goto try_next_key;
          }
      }










































    
    hostkey->public_host_key = ssh_xbuffer_allocate();
    ssh_xbuffer_append(hostkey->public_host_key,
                       public_host_key_blob,
                       public_host_key_blob_len);
    
    keypair_read_successfully = TRUE;
    hostkey->valid = TRUE;

  try_next_key:
    ssh_xfree(public_host_key_blob);
    ssh_xfree(userdir);

    handle = ssh_adt_enumerate_next(ctx->keylist, handle);
  } while (handle != SSH_ADT_INVALID);

  if (keypair_read_successfully)
    {
      char *algs = ssh_host_key_get_algorithmlist(ctx);
      if (algs)
        {
          SSH_TRACE(2, ("Host key algorithms (from disk): %s", algs));
          ssh_xfree(algs);
        }
    }

  ctx->valid_keys = keypair_read_successfully;

  if (!user)
    ssh_user_free(uc, FALSE);
  ssh_xfree(host_key_file);
  












  
  (*completion)(keypair_read_successfully ?
                SSH_HOSTKEY_OK : SSH_HOSTKEY_ERROR,
                completion_context);
}






























































































































































































































































