/*
  sshfilecopy.h

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1999-2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Functions common for both scp and sftp.
 */

#ifndef SSHFILECOPY_H
#define SSHFILECOPY_H

#include "sshfilexfer.h"
#include "sshoperation.h"
#include "sshadt_list.h"
#include "sshglobals.h"





#define SSH_FC_DEFAULT_BUF_SIZE 0x8000
#define SSH_FC_DEFAULT_BUF_NUM 10

typedef struct SshFileCopyConnectionRec *SshFileCopyConnection;
typedef struct SshFileCopyFileRec *SshFileCopyFile;

/* Record that holds the parameters needed to make a connection. */
struct SshFileCopyConnectionRec
{
  char *user;
  char *host;
  char *port;
  SshFileClient client;
};

/* Return values for SshFileCopy functions. */
typedef enum
{
  /* operation was successful.*/
  SSH_FC_OK = 0,
  /* operation resulted in an undetermined error within sshfilecopy. */
  SSH_FC_ERROR,
  /* Destination is not directory, but it should be. */
  SSH_FC_ERROR_DEST_NOT_DIR,
  /* Maximum symlink level exceeded. */
  SSH_FC_ERROR_ELOOP,
  /* Connecting to host failed. */
  SSH_FC_ERROR_CONNECTION_FAILED,
  /* Connection broke for some reason. */
  SSH_FC_ERROR_CONNECTION_LOST,
  /* File doesn't exist. */
  SSH_FC_ERROR_NO_SUCH_FILE,
  /* No permission to access file. */
  SSH_FC_ERROR_PERMISSION_DENIED,
  /* Undetermined error from sshfilexfer. */
  SSH_FC_ERROR_FAILURE,
  /* File transfer protocol mismatch. */
  SSH_FC_ERROR_PROTOCOL_MISMATCH,
  /* Specific to ssh_file_copy_glob. No matches. */
  SSH_FC_ERROR_NO_MATCH
} SshFileCopyError;

typedef struct 
{
  SshFileCopyError error;
  char *error_name;
} SshFileCopyErrorName;

extern SshFileCopyErrorName ssh_file_copy_errors[];
  
/*****************************************************************************
 * Generic functions
 *****************************************************************************/

/* Parse newline convention. Returns approriate newline
   style. Recognizes "unix", "dos" and "mac". Return NULL on invalid
   input. */
char *ssh_file_copy_parse_newline_convention(const char *nl_conv);

/* XXX */
char *ssh_file_copy_strip_dot_dots(const char *filename);

/* XXX */
void ssh_file_copy_tokenize_path(const char *path,
                                 char **basedir_ret,
                                 char **filename_ret);

/*****************************************************************************
 * SshFileCopyFile structure manipulating functions.
 *****************************************************************************/

/* Allocate a new SshFileCopyFile structure. */
SshFileCopyFile ssh_file_copy_file_allocate(void);

/* Create an SshADTContainer for SshFileCopyFiles. */
SshADTContainer ssh_file_copy_file_list_create(SshADTCompareFunc cmp_func,
                                               void *context);

/* Traverse a SshFileCopyFile. Call the traversal callback for every
   encountered file (note if file is not a directory, the traversal cb
   will be once, for the file given).*/
typedef void (*SshFileCopyFileTraversalCB)(SshFileCopyFile file,
                                           void *context);

void ssh_file_copy_file_traverse(SshFileCopyFile file,
                                 SshFileCopyFileTraversalCB trav_cb,
                                 void *context);


/* Destroy SshFileCopyFile structure (including any dir entries
   etc. Closes file handle, if it is set. */
void ssh_file_copy_file_destroy(SshFileCopyFile file);

/* Generate a full path by traversing `file' and it's parents. The
   string is xmallocated and must be freed by the caller. */
char *ssh_file_copy_file_generate_full_path(SshFileCopyFile file);

/* Get filename from a SshFileCopyFile structure. */
const char *ssh_file_copy_file_get_name(SshFileCopyFile file);

/* Get the attributes struct of a SshFileCopyFile structure. If file
   has no attributes (it's not statted yet) returns NULL. */
SshFileAttributes
ssh_file_copy_file_get_attributes(SshFileCopyFile file);

/* Get long filename from a SshFileCopyFile structure. (It could very
   well be NULL) */
const char *ssh_file_copy_file_get_long_name(SshFileCopyFile file);

/* Get handle from a SshFileCopyFile structure. */
SshFileHandle ssh_file_copy_file_get_handle(SshFileCopyFile file);

/* Get the parent dir from a SshFileCopyFile structure. If it doesn't
   have one, returns NULL. */
SshFileCopyFile ssh_file_copy_file_get_parent(SshFileCopyFile file);

/* Get directory entries. If file doesn't have directory entries
   (either they have not been accepted when globbing or recursing, or
   the file is not a directory), return NULL. */
SshADTContainer ssh_file_copy_file_get_dir_entries(SshFileCopyFile file);

/* Registers a filename to a SshFileCopyFile structure. The filename
   is not copied to the struct, it's pointer is just registered. So it
   shouldn't be freed by itself. Use ssh_file_copy_file_destroy()
   instead. It is legal to call this functions multiple times for a
   single SshFileCopyFile structure _before_ it is used in any
   transfers. */
void ssh_file_copy_file_register_filename(SshFileCopyFile file, char *name);

/* Registers a long filename to a SshFileCopyFile structure. The filename
   is not copied to the struct, it's pointer is just registered. So it
   shouldn't be freed by itself. Use ssh_file_copy_file_destroy()
   instead. */
void ssh_file_copy_file_register_long_name(SshFileCopyFile file,
                                           char *long_name);

/* Register the attributes struct of a SshFileCopyFile structure. The
   attrs are not copied to the struct, the pointer is just
   registered. So the attrs shouldn't be freed by themselves. Use
   ssh_file_copy_file_destroy() instead.*/
void ssh_file_copy_file_register_attributes(SshFileCopyFile file,
                                            SshFileAttributes attrs);

/* Register directory entry to a file. ``file'' either shouldn't have
   attributes or should be a directory. */
void ssh_file_copy_file_register_dir_entry(SshFileCopyFile file,
                                           SshFileCopyFile dir_entry);

/*****************************************************************************
 * SshFileCopyConnection structure manipulating functions.
 *****************************************************************************/

/* Allocate a new SshFileCopyConnection structure. */
SshFileCopyConnection ssh_file_copy_connection_allocate(void);

/* Free a SshFileCopyConnection-structure. */
void ssh_file_copy_connection_destroy(SshFileCopyConnection connection);

/* Compare whether two SshFileCopyConnection structures are
   identical. Note that the 'client' part is not compared. Return TRUE
   if these match, or FALSE if not. Both arguments must be valid. */
Boolean ssh_file_copy_connection_compare(SshFileCopyConnection conn1,
                                         SshFileCopyConnection conn2);

/*****************************************************************************
 * Connection handling. .
 *****************************************************************************/

/* Callback, which will be called when connection is done. If 'stream'
   is NULL, connection was failed. */
typedef void (*SshFileCopyStreamReturnCB)(SshStream stream,
                                          void *context);

/* This callback is used to make the connection to the remote
   host. The stream is later wrapped to a SshFileClient, so it should
   not be used outside SshFileCopy. */
typedef void (*SshFileCopyConnectCB)(SshFileCopyConnection connection,
                                     void *context,
                                     SshFileCopyStreamReturnCB completion_cb,
                                     void *completion_context);

/* Register a callback, which is used to connect to the remote host by
   SshFileCopy. 'context' is given to the callback as an
   argument. This _must_ be called before using the other functions
   (which require the ability to form connections). */
void ssh_file_copy_register_connect_callback(SshFileCopyConnectCB callback,
                                             void *context);

/* Callback, which will be called when connection is done, and the
   stream is wrapped. NULL is returned, if connection failed. */
typedef void (*SshFileCopyConnectCompleteCB)(SshFileClient client,
                                             void *context);

/* Make a connection to the remote host, or, if 'host' is NULL,
   create streampair for local action. */
SshOperationHandle
ssh_file_copy_connect(SshFileCopyConnection connection,
                      SshFileCopyConnectCompleteCB completion_cb,
                      void *completion_context);

/*****************************************************************************
 * File transfer functions.
 *****************************************************************************/

/* Attributes, which control how the globbing is done. */
typedef struct SshFileCopyGlobAttrsRec
{
  SshADTCompareFunc compare_func;
  void *compare_ctx;
} *SshFileCopyGlobAttrs;

typedef void (*SshFileCopyErrorCB)(SshFileCopyError error,
                                   const char *error_message,
                                   void *context);

/* List of items of type SshFileCopyFile. */
typedef void (*SshFileCopyGlobReadyCB)(SshFileCopyError error,
                                       const char *error_message,
                                       SshADTContainer file_list,
                                       void *context);

/* XXX Takes a path (~pattern) as a string. */
void ssh_file_copy_glob(SshFileCopyConnection connection,
                        const char *pattern,
                        SshFileCopyGlobAttrs attrs,
                        SshFileCopyGlobReadyCB ready_cb,
                        SshFileCopyErrorCB error_cb,
                        void *context);

/* XXX Takes a list of strings. */
void ssh_file_copy_glob_multiple(SshFileCopyConnection connection,
                                 SshADTContainer pattern_list,
                                 SshFileCopyGlobAttrs attrs,
                                 SshFileCopyGlobReadyCB ready_cb,
                                 SshFileCopyErrorCB error_cb,
                                 void *context);

/* Attributes, which control how the directory recursion is done. */
typedef struct SshFileCopyRecurseAttrsRec
{
  /* If TRUE, ssh_file_copy_recurse_dirs() tries it's best to follow
     symlinks. */
  Boolean follow_symlinks;  
  SshADTCompareFunc compare_func;
  void *compare_ctx;
} *SshFileCopyRecurseAttrs, SshFileCopyRecurseAttrsStruct;

/* If not NULL, called for each file and directory.  The return value
   is a bit mask, and it determines whether file is added to the
   resultant list.  Also, if the file is a directory, then the return
   value is used to determine whether that directory will be
   recursed. */

typedef SshUInt32 (*SshFileCopyRecurseFileFunc)(const char *filename,
                                                const char *long_name,
                                                SshFileAttributes attrs,
                                                void *context);
/* Valid bits to set are defined here. */
/* Specifies that the file should be added to the resultant list. */
#define SSH_FCR_ADD      0x1
/* Specifies that the file should be recursed (if it is a
   directory).

   Note: if you specify this, you MUST also specify SSH_FCR_ADD. */
#define SSH_FCR_RECURSE  0x2


/* Returns list of items of type SshFileCopyFile. */
typedef void (*SshFileCopyRecurseReadyCB)(SshFileCopyError error,
                                          const char *error_message,
                                          SshADTContainer file_list,
                                          void *context);
/*
  XXX
  Recurse directory ``path''. Return items with ready_cb as
  SshFileCopyFiles in an SshADTContainer. */
void ssh_file_copy_recurse(SshFileCopyConnection source,
                           const char *path,
                           SshFileCopyRecurseAttrs attrs,
                           SshFileCopyRecurseFileFunc func,
                           SshFileCopyRecurseReadyCB ready_cb,
                           SshFileCopyErrorCB error_cb,
                           void *context);

/*
  As above, but takes a list of SshFileCopyFiles.
*/
void ssh_file_copy_recurse_multiple(SshFileCopyConnection source,
                                    SshADTContainer file_list,
                                    SshFileCopyRecurseAttrs attrs,
                                    SshFileCopyRecurseFileFunc func,
                                    SshFileCopyRecurseReadyCB ready_cb,
                                    SshFileCopyErrorCB error_cb,
                                    void *context);


typedef void (*SshFileCopyTransferReadyCB)(SshFileCopyError error,
                                           const char *error_message,
                                           void *context);

/* Transfer progress callback. Note that in ASCII transfer, written
   bytes can be lower or larger than read_bytes on successful transfer
   finish.  The 'finished' parameter is TRUE in the final progress
   callback (for the current file). ``skipped_bytes'' is the amount of
   bytes SshFileCopy could skip, because the blocks were identical in
   the source and destination files. ``written_bytes'' and
   ``read_bytes'' will include ``skipped_bytes''. */

typedef void (*SshFileCopyTransferProgressCB)(SshFileCopyConnection source,
                                              SshFileCopyFile source_file,
                                              SshFileCopyConnection dest,
                                              SshFileCopyFile dest_file,
                                              off_t read_bytes,
                                              off_t written_bytes,
                                              off_t skipped_bytes,
                                              SshUInt64 elapsed_time,
                                              Boolean finished,
                                              void *context);


/* SshFileCopyTransferOverwriteCB must call this when ready.  If
   ``overwrite'' is TRUE, the destination file will be overwritten, if
   FALSE, the file will be skipped.  The ``context'' argument must be
   the ``completion_context'' argument given to the OverwriteCB. */
typedef void (*SshFileCopyTransferOverwriteCompletionCB)
     (Boolean overwrite, void *context);

/* If this callback is set in TransferAttrs, transfer_file will call
   this if the destination file already exists and is a regular
   file. See above for ``completion_cb''. */
typedef void (*SshFileCopyTransferOverwriteCB)
     (SshFileCopyFile source_file,
      SshFileCopyFile destination_file,
      SshFileCopyTransferOverwriteCompletionCB completion_cb,
      void *completion_context,
      void *context);

typedef enum
{
  /* Binary transfer mode */
  SSH_FC_TRANSFER_MODE_BINARY = 0,

  /* Ascii transfer mode */
  SSH_FC_TRANSFER_MODE_ASCII = 1,

  /* Tranfer mode is selected automatically from the file extensions */
  SSH_FC_TRANSFER_MODE_AUTO = 2,

  SSH_FC_TRANSFER_MODE_NL_CONV_ASK = 4
} SshFileCopyTransferMode;

/* ``nl_conv'' must be valid (i.e. not NULL). */
typedef void (*SshFileCopyNLQueryCompletionCB)(const char *nl_conv,
                                               void *context);

typedef void (*SshFileCopyNLQueryCB)(Boolean destination,
                                     SshFileCopyNLQueryCompletionCB
                                     completion_cb,
                                     void *completion_context,
                                     void *context);

/* Attributes, which control how the transfer is done. */
typedef struct SshFileCopyTransferAttrsRec
{
  Boolean destination_must_be_dir;
  Boolean preserve_attributes;
  Boolean unlink_source;
  /* Whether to create the directories in file list. */
  Boolean create_dirs;










  





  size_t max_buffers;
  size_t max_buf_size;
  off_t write_offset;
  off_t read_offset;
  /* Stuff for mangling newlines. */
  /* If flag SSH_FC_TRANSFER_MODE_NL_CONV_ASK is ORed to
     ``transfer_mode'', newline conventions set to NULL will be queried
     via a callback ``newline_query_cb'', which must be set if this flag
     is used. */
  unsigned int transfer_mode;
  char *source_newline;
  char *dest_newline;
  char *ascii_extensions;
  SshFileCopyNLQueryCB newline_query_cb;
  SshFileCopyTransferProgressCB progress_cb;
  SshFileCopyTransferOverwriteCB overwrite_cb;
  void *context;  
} *SshFileCopyTransferAttrs;

SshOperationHandle
ssh_file_copy_transfer_multiple(SshFileCopyConnection source,
                                SshADTContainer file_list,
                                SshFileCopyConnection destination,
                                const char *dest_path,
                                SshFileCopyTransferAttrs attrs,
                                SshFileCopyTransferReadyCB ready_cb,
                                SshFileCopyErrorCB error_cb,
                                void *context);

/* Conveniency functions for ssh_file_copy_transfer_*. */

SSH_GLOBAL_DECLARE(int, ssh_fc_progcb_screen_width);
SSH_GLOBAL_DECLARE(int, ssh_fc_progcb_screen_height);
#define ssh_fc_progcb_screen_width SSH_GLOBAL_USE(ssh_fc_progcb_screen_width)
#define ssh_fc_progcb_screen_height SSH_GLOBAL_USE(ssh_fc_progcb_screen_height)

void ssh_file_copy_transfer_default_progress_cb_init(void);

/* The default transfer progress callback. Produces a relatively
   informational putput. Note that for you to use this, you must call
   ssh_file_copy_transfer_default_progress_cb_init() first. Note also,
   that this registers a signal handler for SIGWINCH. You can get the
   window dimension with the above globals
   (ssh_fc_progcb_screen_{width,height}. */
void ssh_file_copy_transfer_default_progress_cb(SshFileCopyConnection source,
                                                SshFileCopyFile source_file,
                                                SshFileCopyConnection dest,
                                                SshFileCopyFile dest_file,
                                                off_t read_bytes,
                                                off_t written_bytes,
                                                off_t skipped_bytes,
                                                SshUInt64 elapsed_time,
                                                Boolean finished,
                                                void *context);

#endif /* SSHFILECOPY_H */
