/*
  sshfilecopy.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1999-2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Functions common for both scp and sftp.
 */
/*
  TODO(among other things, loose order):

  - comment the code better
  
  - if umask is for example 077, then the execute bits will still be
    set, if they are set in the source. Fix.
  
  - new features (like what?)  */

#include "sshincludes.h"
#include "sshfilecopy.h"
#include "sshfilecopyi.h"
#include "sshappcommon.h"
#include "sshglob.h"
#include "sshstreampair.h"
#include "sshmiscstring.h"
#ifdef HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif /* HAVE_SYS_IOCTL_H */
#include "ssheloop.h"





#define SSH_DEBUG_MODULE "SshFileCopy"

/* Callback used in connecting remote hosts. */
static SshFileCopyConnectCB connect_callback = NULL_FNPTR;
static void *connect_context = NULL;

/* Initialize error-names. */
SshFileCopyErrorName ssh_file_copy_errors[] =
{
  { SSH_FC_OK,                      "SSH_FC_OK" },
  { SSH_FC_ERROR,                   "SSH_FC_ERROR" },
  { SSH_FC_ERROR_DEST_NOT_DIR,      "SSH_FC_ERROR_DEST_NOT_DIR" },
  { SSH_FC_ERROR_ELOOP,             "SSH_FC_ERROR_ELOOP" },
  { SSH_FC_ERROR_CONNECTION_FAILED, "SSH_FC_ERROR_CONNECTION_FAILED" },
  { SSH_FC_ERROR_CONNECTION_LOST,   "SSH_FC_ERROR_CONNECTION_LOST" },
  { SSH_FC_ERROR_NO_SUCH_FILE,      "SSH_FC_ERROR_NO_SUCH_FILE" },
  { SSH_FC_ERROR_PERMISSION_DENIED, "SSH_FC_ERROR_PERMISSION_DENIED" },
  { SSH_FC_ERROR_FAILURE,           "SSH_FC_ERROR_FAILURE" },
  { SSH_FC_ERROR_PROTOCOL_MISMATCH, "SSH_FC_ERROR_PROTOCOL_MISMATCH" },
  { SSH_FC_ERROR_NO_MATCH,          "SSH_FC_ERROR_NO_MATCH" },
  { -1, NULL}
};

/* Allocate a new SshFileCopyFile structure. */
SshFileCopyFile ssh_file_copy_file_allocate(void)
{
  SshFileCopyFile file;

  SSH_DEBUG(6, ("Allocating SshFileCopyFile structure..."));
  file = ssh_xcalloc(1, sizeof(*file));
  
  return file;
}

/* Destroy SshFileCopyFile structure. This should be of type
   SshAppListNodeDeleteProc, because it used to destroy list items.*/
void ssh_file_copy_file_destroy(SshFileCopyFile file)
{
  SSH_PRECOND(file != NULL);
  
  SSH_DEBUG(6, ("Destroying SshFileCopyFile structure..."));

  /* Close file (destroys handle) */
  if (file->handle)
    {
      SSH_DEBUG(7, ("Closing file-handle..."));
      /* XXX This will nevertheless cause an SSH_FXP_STATUS message to
         be sent by the file server. With remote connections, this
         message will be written to the stream by ssh2. Unfortunately,
         if scp2 (for example) has already exited, ssh2 will fail with
         SIGPIPE. Kludging this in scp2.c for the moment, but this
         probably should be fixed properly by catching the status
         packets. */
      ssh_file_client_close(file->handle, NULL_FNPTR, NULL);
      file->handle = NULL;
    }
  
  ssh_xfree(file->attributes);
  ssh_xfree(file->name);
  ssh_xfree(file->long_name);
  ssh_adt_destroy(file->dir_entries);
  
  memset(file, 'F', sizeof(*file));
  ssh_xfree(file);
}

/* This functions registers a new filename to 'file'. It is legal to
   call this functions multiple times for a single SshFileCopyFile
   structure _before_ it is used in any transfers. */
void ssh_file_copy_file_register_filename(SshFileCopyFile file, char *name)
{
  SSH_PRECOND(file != NULL);
  
  if (file->name)
    ssh_xfree(file->name);

  file->name = name;
}

/* Get the attributes struct of a SshFileCopyFile structure. If file
   has no attributes (it's not sttaed yet) returns NULL. */
SshFileAttributes ssh_file_copy_file_get_attributes(SshFileCopyFile file)
{
  SSH_PRECOND(file != NULL);
  return file->attributes;
}

/* This functions registers SshFileAttributes to 'file'.  */
void ssh_file_copy_file_register_attributes(SshFileCopyFile file,
                                            SshFileAttributes attrs)
{
  SSH_PRECOND(file != NULL);
  
  if (file->attributes)
    ssh_xfree(file->attributes);

  file->attributes = attrs;
}

void ssh_file_copy_file_traverse(SshFileCopyFile file,
                                 SshFileCopyFileTraversalCB trav_cb,
                                 void *context)
{
  SshADTHandle h;
  
  SSH_PRECOND(file);
  SSH_PRECOND(trav_cb != NULL_FNPTR);

  (*trav_cb)(file, context);

  if (file->dir_entries)
    {
      ssh_adt_list_sort(file->dir_entries);
      SSH_INVARIANT(file->dir_entries);
      
      for (h = ssh_adt_enumerate_start(file->dir_entries);
           h != SSH_ADT_INVALID;
           h = ssh_adt_enumerate_next(file->dir_entries, h))
        {
          SshFileCopyFile file2 = ssh_adt_get(file->dir_entries, h);
          ssh_file_copy_file_traverse(file2, trav_cb, context);
        }
    }
}

/* Register dir_entries to a SshFileCopyFile structure. */
void ssh_file_copy_file_register_dir_entry(SshFileCopyFile file,
                                           SshFileCopyFile dir_entry)
{
  SshADTHandle h;
  
  SSH_PRECOND(file != NULL);
  SSH_PRECOND(dir_entry != NULL);

  if (file->dir_entries == NULL)
    file->dir_entries = ssh_file_copy_file_list_create(NULL_FNPTR, NULL);

  dir_entry->parent_dir = file;
  
  h = ssh_adt_insert(file->dir_entries, dir_entry);
  SSH_ASSERT(h != SSH_ADT_INVALID);              
}

/* Get the parent dir from a SshFileCopyFile structure. */
SshFileCopyFile ssh_file_copy_file_get_parent(SshFileCopyFile file)
{
  SSH_PRECOND(file != NULL);
  return file->parent_dir;
}

/* Get dir_entries from a SshFileCopyFile structure. */
SshADTContainer ssh_file_copy_file_get_dir_entries(SshFileCopyFile file)
{
  SSH_PRECOND(file != NULL);
  return file->dir_entries;
}

/* Get filename from a SshFileCopyFile structure. */
const char *ssh_file_copy_file_get_name(SshFileCopyFile file)
{
  SSH_PRECOND(file != NULL);
  return file->name;
}

/* Get long filename from a SshFileCopyFile structure. */
const char *ssh_file_copy_file_get_long_name(SshFileCopyFile file)
{
  SSH_PRECOND(file != NULL);
  return file->long_name;
}

/* This functions registers a new long filename to 'file'.  */
void ssh_file_copy_file_register_long_name(SshFileCopyFile file,
                                           char *long_name)
{
  SSH_PRECOND(file != NULL);
  
  if (file->long_name)
    ssh_xfree(file->long_name);

  file->long_name = long_name;
}

/* Get handle from a SshFileCopyFile structure. */
SshFileHandle ssh_file_copy_file_get_handle(SshFileCopyFile file)
{
  SSH_PRECOND(file != NULL);
  return file->handle;
}

/* Duplicate a SshFileCopyFile structure. Notice, however, that if the
   file had bee opened in the original struct, it will be closed now
   (ie. the SshFileHandle is NULL in the copy). Also dir_entries are
   not copied. */
SshFileCopyFile ssh_file_copy_file_dup(SshFileCopyFile file)
{
  SshFileCopyFile new_file;

  new_file = ssh_file_copy_file_allocate();
  
  if (ssh_file_copy_file_get_name(file))
    ssh_file_copy_file_register_filename
      (new_file, ssh_xstrdup(ssh_file_copy_file_get_name(file)));

  if (ssh_file_copy_file_get_attributes(file))
    ssh_file_copy_file_register_attributes
      (new_file, ssh_file_attributes_dup
       (ssh_file_copy_file_get_attributes(file)));
  
  if (ssh_file_copy_file_get_long_name(file))
    ssh_file_copy_file_register_long_name
      (new_file, ssh_xstrdup(ssh_file_copy_file_get_long_name(file)));

  return new_file;
}

static void file_copy_file_destructor(void *item, void *context)
{
  SshFileCopyFile file = (SshFileCopyFile) item;
  ssh_file_copy_file_destroy(file);
}

static int file_copy_file_compare(const void *obj1, const void *obj2,
                                  void *context)
{
  SshFileCopyFile file1 = (SshFileCopyFile) obj1,
    file2 = (SshFileCopyFile) obj2;
  SSH_PRECOND(file1 != NULL);
  SSH_PRECOND(file2 != NULL);
  return strcmp(ssh_file_copy_file_get_name(file1),
                ssh_file_copy_file_get_name(file2));
}

SshADTContainer ssh_file_copy_file_list_create(SshADTCompareFunc cmp_func,
                                               void *context)
{
  SshADTContainer file_list;

  if (cmp_func == NULL_FNPTR)
    cmp_func = file_copy_file_compare;  

  file_list = ssh_adt_create_generic(SSH_ADT_LIST,
                                     SSH_ADT_COMPARE, cmp_func,
                                     SSH_ADT_DESTROY,
                                     file_copy_file_destructor,
                                     SSH_ADT_CONTEXT,
                                     context,
                                     SSH_ADT_ARGS_END);
  SSH_ASSERT(file_list);
  return file_list;
}

char *ssh_file_copy_file_generate_full_path(SshFileCopyFile file)
{
  SshFileCopyFile parent;
  char *full_path = ssh_xstrdup(ssh_file_copy_file_get_name(file));

  SSH_DEBUG(25, ("cur full path: %s", full_path));
  for (parent = file->parent_dir;
       parent != NULL;
       parent = parent->parent_dir)
    {
      char *hlp;
      const char *parent_dir;
      parent_dir = ssh_file_copy_file_get_name(parent);

      if (strcmp(parent_dir, "/") != 0)
        ssh_xdsprintf(&hlp, "%s/%s", parent_dir, full_path);
      else
        ssh_xdsprintf(&hlp, "/%s", full_path);

      ssh_xfree(full_path);
      full_path = hlp;
      SSH_DEBUG(25, ("full_path now: %s (cur parent: %s)", full_path,
                    parent_dir));
      
    }

  SSH_DEBUG(10, ("full path: %s", full_path));
  return full_path;
}

/* Allocate a new SshFileCopyConnection structure. */
SshFileCopyConnection ssh_file_copy_connection_allocate(void)
{
  SshFileCopyConnection connection;

  SSH_DEBUG(6, ("Allocating SshFileCopyConnection structure..."));
  connection =  ssh_xcalloc(1, sizeof(*connection));
  
  return connection;
}

/* Free a SshFileCopyConnection structure. */
void ssh_file_copy_connection_destroy(SshFileCopyConnection connection)
{
  SSH_DEBUG(6, ("Destroying SshFileCopyConnection structure..."));

  if (connection->client)
    ssh_file_client_destroy(connection->client);
  ssh_xfree(connection->user);
  ssh_xfree(connection->host);
  ssh_xfree(connection->port);
  
  memset(connection, 'F', sizeof(*connection));
  ssh_xfree(connection);
}

/* Compare whether two SshFileCopyConnection structures are
   identical. Note that the 'client' part is not compared. Return TRUE
   if these match, or FALSE if not. Both arguments must be valid. */
Boolean ssh_file_copy_connection_compare(SshFileCopyConnection conn1,
                                         SshFileCopyConnection conn2)
{
  SSH_PRECOND(conn1 != NULL);
  SSH_PRECOND(conn2 != NULL);
  
  if (conn1->user != NULL && conn2->user != NULL)
    {
      if (strcmp(conn1->user, conn2->user) != 0)
        return FALSE;
    }
  else
    {
      if (!(conn1->user == NULL && conn2->user == NULL))
        return FALSE;
    }

  if (conn1->host != NULL && conn2->host != NULL)
    {
      if (strcmp(conn1->host, conn2->host) != 0)
        return FALSE;
    }
  else
    {
      if (!(conn1->host == NULL && conn2->host == NULL))
        return FALSE;
    }

  if (conn1->port != NULL && conn2->port != NULL)
    {
      if (strcmp(conn1->port, conn2->port) != 0)
        return FALSE;
    }
  else
    {
      if (!(conn1->port == NULL && conn2->port == NULL))
        return FALSE;
    }

  return TRUE;
}

/* Removes escape characters before slashes ('\/' and the
   like). Returns a newly mallocated stripped string.*/
char *ssh_file_copy_strip_escapes_before_slashes(const char *filename)
{
  char *stripped = NULL;
  int len = 0, i = 0, str_ind = 0;
  int echar = '\\';
  
  SSH_PRECOND(filename != NULL);
  
  len = strlen(filename);
  
  stripped = ssh_xcalloc(len + 1, sizeof(char));

  for (i = 0, str_ind = 0; i < len - 1; i++, str_ind++)
    {

      if (filename[i] == echar && filename[i + 1] == '/' &&
          _ssh_glob_isescaped(&filename[i + 1], filename, echar))
        i++;


      stripped[str_ind] = filename[i];
    }

  stripped[str_ind] = filename[i];

  return stripped;
}

/* Removes extra slashes ('//' and the like) from filename. This also
   does the strip_escapes_before_slashes (those could be used to fool
   this). Returns a newly mallocated stripped string. */
char *ssh_file_copy_strip_extra_slashes(const char *filename)
{
  char *filename_copy = NULL, *ph = NULL;
  int i = 0, len = 0;
  SSH_PRECOND(filename != NULL);

  filename_copy = ssh_file_copy_strip_escapes_before_slashes(filename);


  while (filename_copy[i] != '\0')
    {
      for (/*EMPTY*/; filename_copy[i] != '\0' && filename_copy[i] != '/'; i++)
        ;

      if (filename_copy[i] != '\0')
        i++;
      
      ph = &filename_copy[i];
      
      for (/*EMPTY*/; filename_copy[i] != '\0' && filename_copy[i] == '/'; i++)
        ;
      
      len = strlen(&filename_copy[i]) + 1;
      memmove(ph, &filename_copy[i], len);
      
      i = ph - filename_copy;
    }


  return filename_copy;
}

/* Strip foo/../bar/../ combinations from filename to prevent
   malicious, or misguided, users from trashing some systems, and
   bringing the load up in others. */
char *ssh_file_copy_strip_dot_dots(const char *filename)
{
  char *filename_copy = NULL, *ph = NULL;
  int i = 0, len = 0;
  SSH_PRECOND(filename != NULL);

  filename_copy = ssh_file_copy_strip_extra_slashes(filename);

 restart:
  /* Skip first '/' characters. */
  for (/*EMPTY*/; filename_copy[i] == '/'; i++)
    ;

  /* ph points now to first non-'/' character. */
  ph = &filename_copy[i];

  
  if (strncmp(ph, "../", 3) == 0)
    {
      i += 3;
      goto restart;
    }

  if (strncmp(ph, "./", 2) == 0)
    {
      i += 2;
      goto restart;
    }

  /* Now we skip to next '/' character. */
  for (/*EMPTY*/; filename_copy[i] != '\0' && filename_copy[i] != '/'; i++)
    ;

  /* If at end, return. */
  if (filename_copy[i] == '\0')
    return filename_copy;
  
  i ++;
  
  if (strncmp(&filename_copy[i], "../", 3) == 0 ||
      (strncmp(&filename_copy[i], "..", 2) == 0 &&
       filename_copy[i + 2] == '\0'))
    {
      /* Here we have a strippable case. */
      if (filename_copy[i + 2] == '\0')
        {
          *ph = '\0';
          return filename_copy;
        }
      len = strlen(&filename_copy[i + 3]) + 1;
      memmove(ph, &filename_copy[i + 3], len);

      i = ph - filename_copy;
    }

  goto restart;
}

/* Get the basedir and filename. If either would be NULL, a "." is
   added instead. */
void ssh_file_copy_tokenize_path(const char *path,
                                 char **basedir_ret,
                                 char **filename_ret)
{
  char *hlp, *basedir, *file;




  basedir = ssh_xstrdup(path);

  
  if ((hlp = strrchr(basedir, '/')) != NULL)
    {
      if (hlp > &basedir[0])
        {
          *hlp = '\0';
          hlp++;
          if (strlen(hlp) == 0)
            {
              /* The case "foo/". It might seem backwards to return
                 "foo" as file, and no basedir, but it actually is
                 easier to use and more logical this way. */
              file = basedir;
              basedir = NULL;
            }
          else
            {
              file = ssh_xstrdup(hlp);
            }
        }
      else
        {
          /* The only '/' was at the beginning of the filename. */
          if (strlen(basedir) > 1)
            {
              file = ssh_xstrdup(&basedir[1]);
              basedir[1] = '\0';
            }
          else
            {
              /* If path is just "/". */
              file = basedir;
              basedir = NULL;
            }
        }
    }
  else
    {
      file = basedir;
      basedir = NULL;
    }

  if (filename_ret != NULL)
    *filename_ret = file;
  else
    ssh_xfree(file);
  
  if (basedir_ret != NULL)
    *basedir_ret = basedir;
  else
    ssh_xfree(basedir);
}

/* Register a callback, which is used to connect to the remote host by
   SshFileCopy. 'context' is given to the callback as an argument. */
void ssh_file_copy_register_connect_callback(SshFileCopyConnectCB callback,
                                             void *context)
{
  connect_callback = callback;
  connect_context = context;
}

typedef struct ConnectionCompletionCtxRec
{
  SshFileCopyConnection connection;
  SshFileClient client;
  SshFileCopyConnectCompleteCB completion_cb;
  void *completion_context;
  SshOperationHandle op_handle;
  Boolean aborted;
} *ConnectionCompletionCtx;

void connect_done_cb(void *context)
{
  ConnectionCompletionCtx ctx = (ConnectionCompletionCtx) context;
  if (ctx->aborted == TRUE)
    {
      ssh_file_client_destroy(ctx->client);
      ssh_xfree(ctx);
      return;
    }
  ctx->connection->client = ctx->client;
  ssh_operation_unregister(ctx->op_handle);  
  SSH_DEBUG(2, ("Connection to %s%s%s, ready to serve "
                "requests.",
                ctx->connection->host ? "remote host '" : "local",
                ctx->connection->host ? ctx->connection->host : "",
                ctx->connection->host ? "'" : ""));
  (*ctx->completion_cb)(ctx->connection->client,
                        ctx->completion_context);
  ssh_xfree(ctx);
}

void stream_return_cb(SshStream stream,
                      void *context)
{      
  ConnectionCompletionCtx ctx = (ConnectionCompletionCtx) context;

  if (ctx->aborted == TRUE)
    {
      if (stream)
        ssh_stream_destroy(stream);
      ssh_xfree(ctx);
      return;
    }
  
  if (stream == NULL)
    {
      SSH_TRACE(1, ("Connection failed."));
      (*ctx->completion_cb)(NULL, ctx->completion_context);
      return;
    }
  ctx->client = ssh_file_client_wrap(stream, connect_done_cb, ctx);
}

void connect_abort_cb(void *operation_context)
{
  ConnectionCompletionCtx ctx =
    (ConnectionCompletionCtx) operation_context;

  ctx->aborted = TRUE;
}

/* Make a connection to the remote host, or, if 'host' is NULL,
   create streampair for local action. */
SshOperationHandle
ssh_file_copy_connect(SshFileCopyConnection connection,
                      SshFileCopyConnectCompleteCB completion_cb,
                      void *completion_context)
{
  ConnectionCompletionCtx ctx = ssh_xcalloc(1, sizeof(*ctx));
      
  SSH_PRECOND(connection != NULL);
  SSH_PRECOND(completion_cb != NULL_FNPTR);
  ctx->connection = connection;
  ctx->completion_cb = completion_cb;
  ctx->completion_context = completion_context;
  ctx->op_handle = ssh_operation_register(connect_abort_cb, ctx);
  
  if (connection->host == NULL)
    {
      SshFileServer local_server = NULL;
      SshStream stream1 = NULL, stream2 = NULL;
      
      SSH_TRACE(2, ("Making local connection."));

      /* Make local connection. */
      ssh_stream_pair_create(&stream1, &stream2);
      /* The local_server is automatically destroyed with the stream. */
      local_server = ssh_file_server_wrap(stream1, -1);
      ctx->client = ssh_file_client_wrap(stream2, connect_done_cb, ctx);
    }
  else
    {
      SSH_TRACE(2, ("Connecting to remote host. (host = %s, "
                    "user = %s, port = %s)",
                    connection->host ? connection->host : "NULL",
                    connection->user ? connection->user : "NULL",
                    connection->port ? connection->port : "NULL"));
      /* use SshFileCopyConnectCB to connect. */
      SSH_ASSERT(connect_callback != NULL_FNPTR);
      
      (*connect_callback)(connection, connect_context,
                          stream_return_cb, ctx);
    }
  return ctx->op_handle;
}

SshFileCopyError fcc_fx_err_to_fc_err(SshFileClientError error)
{
  switch (error)
    {
    case SSH_FX_OK:
      return SSH_FC_OK;
    case SSH_FX_NO_SUCH_FILE:
      return SSH_FC_ERROR_NO_SUCH_FILE;
    case SSH_FX_PERMISSION_DENIED:
      return SSH_FC_ERROR_PERMISSION_DENIED;
    default:
      return SSH_FC_ERROR_FAILURE;
    }
}

/* This is similar to ssh_debug_format. */
char *fcc_format_error(const char *fmt, ...)
{
  va_list args;
  char *p;

  va_start(args, fmt);
  ssh_xdvsprintf(&p, fmt, args);
  va_end(args);

  return p;
}

char *ssh_file_copy_parse_newline_convention(const char *nl_conv)
{
  char *nl = NULL;
  
  if (!strcmp(nl_conv, "dos"))
    nl = ssh_xstrdup("\r\n");
  else if (!strcmp(nl_conv, "mac"))
    nl = ssh_xstrdup("\r");
  else if (!strcmp(nl_conv, "unix"))
    nl = ssh_xstrdup("\n");
  else
    ssh_warning("Ignoring unknown newline convention '%s'", nl_conv);
  
  return nl;
}

SSH_GLOBAL_DEFINE(int, ssh_fc_progcb_screen_width);
SSH_GLOBAL_DEFINE(int, ssh_fc_progcb_prev_screen_width);
SSH_GLOBAL_DEFINE(char *, ssh_fc_progcb_format_string);
SSH_GLOBAL_DEFINE(int, ssh_fc_progcb_screen_height);

SSH_GLOBAL_DECLARE(int, ssh_fc_progcb_prev_screen_width);
SSH_GLOBAL_DECLARE(char *, ssh_fc_progcb_format_string);

#define ssh_fc_progcb_prev_screen_width \
        SSH_GLOBAL_USE(ssh_fc_progcb_prev_screen_width)
#define ssh_fc_progcb_format_string SSH_GLOBAL_USE(ssh_fc_progcb_format_string)


#define SSH_FC_PROGCB_DEFAULT_SCREEN_WIDTH 78
#define SSH_FC_PROGCB_DEFAULT_SCREEN_HEIGHT 25

#if defined(TIOCGWINSZ)

/* SIGWINCH (window size change signal) handler.  This sends a window
   change request to the server. */

static void fc_progcb_win_dim_change(int sig, void *context)
{
  struct winsize ws;

  if (ioctl(fileno(stdin), TIOCGWINSZ, &ws) >= 0)
    {
      ssh_fc_progcb_screen_width = ws.ws_col;
      ssh_fc_progcb_screen_height = ws.ws_row;
    }
  else
    {
      SSH_TRACE(2, ("unable to get window size parameters."));
    }
}
#endif /* TIOCGWINSZ */

void ssh_file_copy_transfer_default_progress_cb_init(void)
{
  SSH_GLOBAL_INIT(ssh_fc_progcb_screen_width,
                  SSH_FC_PROGCB_DEFAULT_SCREEN_WIDTH);
  SSH_GLOBAL_INIT(ssh_fc_progcb_screen_height,
                  SSH_FC_PROGCB_DEFAULT_SCREEN_HEIGHT);
  SSH_GLOBAL_INIT(ssh_fc_progcb_prev_screen_width, -1);
  SSH_GLOBAL_INIT(ssh_fc_progcb_format_string, NULL);

  if (isatty(fileno(stdin)))
    {
#if defined(SIGWINCH) && defined(TIOCGWINSZ)
      /* Register a signal handler for SIGWINCH to send window change
         notifications to the server. */
      ssh_register_signal(SIGWINCH, fc_progcb_win_dim_change, NULL);
#endif /* SIGWINCH && TIOCGWINSZ*/
#ifdef TIOCGWINSZ
      /* SIGWINCH not defined, call the handler now, once */
      fc_progcb_win_dim_change(0, NULL);
#endif /* TIOCGWINSZ */
    }











}

void ssh_file_copy_transfer_default_progress_cb(SshFileCopyConnection source,
                                                SshFileCopyFile source_file,
                                                SshFileCopyConnection dest,
                                                SshFileCopyFile dest_file,
                                                off_t read_bytes,
                                                off_t written_bytes,
                                                off_t skipped_bytes,
                                                SshUInt64 elapsed_time,
                                                Boolean finished,
                                                void *context)
{
  SshFileAttributes attrs;
  double transfer_rate;
  char *file_name;
  char num_buffer[10], tr_rate_buffer[10], eta_string[10];
  int name_space = -1;
  
  attrs = ssh_file_copy_file_get_attributes(source_file);

  SSH_VERIFY(attrs != NULL);
  SSH_ASSERT(attrs->flags & SSH_FILEXFER_ATTR_SIZE);

  if (read_bytes == skipped_bytes && skipped_bytes > 0)
    {
      fprintf(stdout, "\r%s: complete md5 match -> "
              "transfer skipped\r\n",
              ssh_file_copy_file_get_name(source_file));
      return;
    }

  if (elapsed_time <= 0)
    elapsed_time = 1L;

  transfer_rate = (double) (SshInt64)written_bytes / (SshInt64)elapsed_time;

  if (!finished)
    {
      ssh_format_time(eta_string, sizeof(eta_string),
                      (SshUInt64)((SshInt64)(attrs->size -
                                             written_bytes) /
                                  (transfer_rate < 1.0 ? 1.0 :
                                   transfer_rate)));
    }
  else
    {
      ssh_format_time(eta_string, sizeof(eta_string), elapsed_time);
    }

  if (ssh_fc_progcb_screen_width < SSH_FC_PROGCB_DEFAULT_SCREEN_WIDTH)
    ssh_fc_progcb_screen_width = SSH_FC_PROGCB_DEFAULT_SCREEN_WIDTH ;

  name_space = ssh_fc_progcb_screen_width - 41;

#define FC_FORMAT_STRING_SIZE 1024
  if (ssh_fc_progcb_format_string == NULL)
    {
      /* Make sure that the format string is recalculated. */
      ssh_fc_progcb_prev_screen_width = -1;
      /* XXX Leaks memory on exit. Minor problem. */
      ssh_fc_progcb_format_string = ssh_xcalloc(FC_FORMAT_STRING_SIZE,
                                                sizeof(char));
      
    }
  
  /* If screen width is changed, reconstruct the format string. */
  if (ssh_fc_progcb_prev_screen_width != ssh_fc_progcb_screen_width)
    {
      ssh_fc_progcb_prev_screen_width = ssh_fc_progcb_screen_width;
      
      ssh_snprintf(ssh_fc_progcb_format_string,
                   FC_FORMAT_STRING_SIZE,
                   "\r%%-%d.%ds | %%4sB | %%4sB/s | %%3.3s: %%s | "
                   "%%3d%%%%", name_space, name_space);
    }

  file_name = ssh_xstrdup(ssh_file_copy_file_get_name(source_file));
  if (strlen(file_name) > name_space)
    {
      char *hlp;
      SSH_ASSERT(strlen(file_name) - name_space + 2 > 0);
      SSH_ASSERT(name_space > 3);
      
      ssh_xdsprintf(&hlp, "..%s",
                    &file_name[strlen(file_name) - name_space + 2]);
      ssh_xfree(file_name);
      file_name = hlp;
    }
  
  fprintf(stdout, ssh_fc_progcb_format_string, file_name,
          ssh_format_number(num_buffer, sizeof(num_buffer),
                            written_bytes, 1024),
          ssh_format_number(tr_rate_buffer, sizeof(tr_rate_buffer),
                            (SshUInt64)transfer_rate, 1024),
          !finished ? "ETA" : "TOC",
          eta_string,
          (attrs->size && !finished) ?
          (int) (100.0 * (double) (SshInt64)written_bytes /
                 (SshInt64) attrs->size) : 100);

  ssh_xfree(file_name);
  
  if (finished)
    {
      /* This file is at it's end. */
      fprintf(stdout, "\r\n");
    }
}


