/*
  sshfc_trcore.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 2001,2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  The core transfer step functions and callbacks.

  Created Wed Jan 17 09:58:53 2001.
*/

#include "ssh2includes.h"
#include "sshfilecopy.h"
#include "sshfilecopyi.h"
#include "sshfc_transferi.h"
#include "sshmatch.h"
#include "sshadt_list.h"
#include "sshoperation.h"
#include "sshappcommon.h"

#define SSH_DEBUG_MODULE "SshFCTransferCore"

typedef struct TCRequestRec
{
  /* Operation handle for the ssh_file_client_* operations, so that we
     can abort them, if we get aborted. */
  SshOperationHandle op_handle;

  /* offset, where the read/write started. */
  off_t req_offset;
  /* Requested bytes. */
  size_t req_bytes;







  
  /* Associated buffer. */
  SshBuffer buffer;

  /* Associated thread. */
  SshFSMThread thread;
} *TCRequest, TCRequestStruct;

typedef struct TransferCoreThreadContextRec
{
  SshFileCopyFile source_file;
  SshFileCopyFile dest_file;

  SshADTContainer req_queue;
  SshADTContainer in_progress_queue;






  Boolean dealing_with_source;

  off_t read_bytes;
  off_t written_bytes;
  off_t skipped_bytes;
  
  SshFSMThread parent;

} *TransferCoreThreadContext;


















typedef struct TransferManglerThreadContextRec
{
  SshADTContainer req_queue;

  off_t read_offset;
  off_t write_offset;

  off_t file_len;
  
  SshBuffer window;
} *TransferManglerThreadContext;

void transfer_core_thread_destructor(SshFSM fsm, void *tdata)
{
  TransferCoreThreadContext thread_context =
    (TransferCoreThreadContext) tdata;

  SSH_PRECOND(tdata != NULL);

  SSH_DEBUG(4, ("Destroying request queues.."));

  ssh_adt_destroy(thread_context->in_progress_queue);
  ssh_adt_destroy(thread_context->req_queue);

  ssh_xfree(thread_context);
}

void transfer_mangler_destructor(SshFSM fsm, void *tdata)
{
  TransferManglerThreadContext thread_context =
    (TransferManglerThreadContext) tdata;

  SSH_PRECOND(tdata != NULL);

  ssh_adt_destroy(thread_context->req_queue);
  ssh_buffer_free(thread_context->window);
  ssh_xfree(thread_context);
}













int tc_req_comp(const void *obj1, const void *obj2, void *context)
{
  SshUInt64 diff;
  TCRequest req1 = (TCRequest) obj1;
  TCRequest req2 = (TCRequest) obj2;
  
  diff = req1->req_offset - req2->req_offset;
  if (diff < 0)
    return -1;
  else if (diff == 0)
    return 0;
  else
    return 1;
}

void tc_req_destroy(void *obj, void *context)
{
  TCRequest req = (TCRequest) obj;

  SSH_DEBUG(6, ("Destroying request."));
  if (req->buffer)
    {
      ssh_buffer_free(req->buffer);
      req->buffer = NULL;
    }




  ssh_xfree(req);
}

void tc_inprogress_req_destroy(void *obj, void *context)
{
  TCRequest req = (TCRequest) obj;

  SSH_DEBUG(5, ("Destroying request in progress."));

  ssh_operation_abort(req->op_handle);
  req->op_handle = NULL;

  tc_req_destroy(req, context);
}

void tc_buffer_destroy(void *obj, void *context)
{
  SshBuffer buf = (SshBuffer) obj;

  SSH_DEBUG(4, ("Freeing buffer from queue..."));

  ssh_buffer_free(buf);
}

/* The transfer loop. */
SSH_FSM_STEP(transfer_start)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  gdata->mangle_this_file = FALSE;

  if (gdata->attrs->transfer_mode & (SSH_FC_TRANSFER_MODE_AUTO |
                                     SSH_FC_TRANSFER_MODE_ASCII))
    {
      if (gdata->src_newline == NULL ||
          gdata->dest_newline == NULL ||
          strcmp(gdata->src_newline,
                 gdata->dest_newline) != 0)
        {
          if (gdata->attrs->transfer_mode & SSH_FC_TRANSFER_MODE_AUTO)
            {
              char *rest, *current, *temp;
              SshRegexMatcher rex;
              
              rest = gdata->attrs->ascii_extensions;
              while (gdata->mangle_this_file == FALSE &&
                     strlen(rest) > 0 &&
                     (current = ssh_app_param_list_get_next(rest)) != NULL)
                {
                  char *temp_file = ssh_xstrdup(ssh_file_copy_file_get_name
                                                (tdata->cur_loc));
                  char *filen = strrchr(temp_file, '/');
                  if (filen == NULL)
                    filen = temp_file;
                  else
                    filen++;
                  
                  rest += strlen(current);
                  if (*rest == ',')
                    rest++;
                  ssh_xdsprintf(&temp, "*.%s", current);
                  rex = ssh_regex_create(ssh_app_get_global_regex_context(),
                                         temp, SSH_REGEX_SYNTAX_ZSH_FILEGLOB);
                  if (rex == NULL)
                    {
                      ssh_debug("Regex library couldn't parse ASCII "
                                "extension '%s'", current);
                    }
                  if (ssh_regex_match_cstr(rex, filen))
                    gdata->mangle_this_file = TRUE;
                  else
                    SSH_DEBUG(3, ("'%s' didn't match with '%s'.", temp,
                                  filen));
                  ssh_regex_free(rex);
                  ssh_xfree(current);
                }
            }
          else
            {
              gdata->mangle_this_file = TRUE;
            }

          if (gdata->mangle_this_file)
            SSH_DEBUG(3, ("File ``%s'' will be mangled.",
                          ssh_file_copy_file_get_name(tdata->cur_loc)));
        }
    }















































  /* Default state after mangling checks. */
  SSH_FSM_SET_NEXT(transfer_start_finalize);

  if (gdata->mangle_this_file)
    {
      if (gdata->src_newline == NULL ||
          gdata->dest_newline == NULL)
        {
          if (gdata->attrs->transfer_mode & SSH_FC_TRANSFER_MODE_NL_CONV_ASK)
            {
              SSH_FSM_SET_NEXT(transfer_get_source_newline_conv);
            }
          else
            {
              const char *direction;

              if (gdata->src_newline == NULL &&
                  gdata->dest_newline == NULL)
                direction = "source and destination";
              else if (gdata->src_newline == NULL)
                direction = "source";
              else
                direction = "destination";
              
              FCC_ERROR(SSH_FC_ERROR,
                        ("%s (src): Failed to get %s newline "
                         "convention, consult software manual on how to set "
                         "it manually. File will NOT be converted.",
                         ssh_file_copy_file_get_name(tdata->cur_loc),
                         direction));
              gdata->mangle_this_file = FALSE;
            }
        }
    }
  return SSH_FSM_CONTINUE;
}

void nl_query_completion_cb(const char *nl_conv, void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  SshFileCopyTransferContext gdata =
    (SshFileCopyTransferContext)ssh_fsm_get_gdata(thread);
  TransferThreadContext tdata =
    (TransferThreadContext)ssh_fsm_get_tdata(thread);
  
  SSH_ASSERT(nl_conv != NULL);

  if (tdata->dealing_with_source == TRUE)
    gdata->src_newline = ssh_xstrdup(nl_conv);
  else
    gdata->dest_newline = ssh_xstrdup(nl_conv);

  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_FSM_STEP(transfer_get_source_newline_conv)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  SSH_PRECOND(gdata->attrs->transfer_mode & SSH_FC_TRANSFER_MODE_NL_CONV_ASK);

  SSH_FSM_SET_NEXT(transfer_get_dest_newline_conv);
  
  if (gdata->src_newline)
    return SSH_FSM_CONTINUE;
  
  SSH_DEBUG(5, ("Querying newline convention from user."));

  SSH_ASSERT(gdata->attrs->newline_query_cb != NULL_FNPTR);
  
  tdata->dealing_with_source = TRUE;
  SSH_FSM_ASYNC_CALL((*gdata->attrs->newline_query_cb)(FALSE,
                                                       nl_query_completion_cb,
                                                       thread,
                                                       gdata->attrs->context));
}

SSH_FSM_STEP(transfer_get_dest_newline_conv)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  SSH_PRECOND(gdata->attrs->transfer_mode & SSH_FC_TRANSFER_MODE_NL_CONV_ASK);

  SSH_FSM_SET_NEXT(transfer_start_finalize);

  if (gdata->dest_newline)
    return SSH_FSM_CONTINUE;

  SSH_DEBUG(5, ("Querying newline convention from user."));

  SSH_ASSERT(gdata->attrs->newline_query_cb != NULL_FNPTR);
  
  tdata->dealing_with_source = FALSE;  
  SSH_FSM_ASYNC_CALL((*gdata->attrs->newline_query_cb)(TRUE,
                                                       nl_query_completion_cb,
                                                       thread,
                                                       gdata->attrs->context));
}

SSH_FSM_STEP(transfer_start_finalize)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  TransferCoreThreadContext reader_tdata, writer_tdata;
  TransferManglerThreadContext mangler_tdata;
  SshFileAttributes attrs;

  if (gdata->mangle_this_file && (gdata->attrs->transfer_mode &
                                  SSH_FC_TRANSFER_MODE_NL_CONV_ASK))
    gdata->mangle_this_file =
      strcmp(gdata->src_newline,
             gdata->dest_newline) ? TRUE : FALSE;
  
  /* Allocate buffers etc. */
  if (!gdata->in_blocking)
    gdata->in_blocking = ssh_fsm_condition_create(gdata->fsm);
  if (!gdata->out_blocking)
    gdata->out_blocking = ssh_fsm_condition_create(gdata->fsm);

  SSH_TRACE(2, ("Starting transfer for file %s, destination %s",
                ssh_file_copy_file_get_name(tdata->cur_loc),
                ssh_file_copy_file_get_name(tdata->cur_dest)));

  reader_tdata = ssh_xcalloc(1, sizeof(*reader_tdata));
  gdata->reader = ssh_fsm_thread_create(gdata->fsm,
                                        transfer_shred,
                                        NULL_FNPTR,
                                        transfer_core_thread_destructor,
                                        reader_tdata);
  SSH_VERIFY(gdata->reader != NULL);
  ssh_fsm_set_thread_name(gdata->reader,
                          "reader");

  writer_tdata = ssh_xcalloc(1, sizeof(*writer_tdata));
  gdata->writer = ssh_fsm_thread_create(gdata->fsm,
                                        transfer_write_out,
                                        NULL_FNPTR,
                                        transfer_core_thread_destructor,
                                        writer_tdata);
  SSH_VERIFY(gdata->writer != NULL);
  ssh_fsm_set_thread_name(gdata->writer, "writer");

  reader_tdata->source_file = writer_tdata->source_file = tdata->cur_loc;
  reader_tdata->dest_file = writer_tdata->dest_file = tdata->cur_dest;






  reader_tdata->req_queue = ssh_adt_create_generic(SSH_ADT_LIST,
                                                   SSH_ADT_DESTROY,
                                                   tc_req_destroy,
                                                   SSH_ADT_ARGS_END);
  reader_tdata->in_progress_queue =
    ssh_adt_create_generic(SSH_ADT_LIST,
                           SSH_ADT_DESTROY,
                           tc_inprogress_req_destroy,
                           SSH_ADT_ARGS_END);

  writer_tdata->req_queue = ssh_adt_create_generic(SSH_ADT_LIST,
                                                   SSH_ADT_DESTROY,
                                                   tc_req_destroy,
                                                   SSH_ADT_ARGS_END);
  writer_tdata->in_progress_queue =
    ssh_adt_create_generic(SSH_ADT_LIST,
                           SSH_ADT_DESTROY,
                           tc_inprogress_req_destroy,
                           SSH_ADT_ARGS_END);
  
  if (gdata->mangle_this_file)
    {
      SSH_DEBUG(2, ("File '%s' will be mangled.",
                    ssh_file_copy_file_get_name(tdata->cur_loc)));
      attrs = ssh_file_copy_file_get_attributes(tdata->cur_loc);
      if (!gdata->mangle_blocking)
        gdata->mangle_blocking = ssh_fsm_condition_create(gdata->fsm);
      
      mangler_tdata = ssh_xcalloc(1, sizeof(*mangler_tdata));
      gdata->mangler = ssh_fsm_thread_create(gdata->fsm,
                                             transfer_mangle,
                                             NULL_FNPTR,
                                             transfer_mangler_destructor,
                                             mangler_tdata);
      SSH_VERIFY(gdata->mangler != NULL);
      mangler_tdata->req_queue = ssh_adt_create_generic(SSH_ADT_LIST,
                                                        SSH_ADT_DESTROY,
                                                        tc_req_destroy,
                                                        SSH_ADT_COMPARE,
                                                        tc_req_comp,
                                                        SSH_ADT_ARGS_END);
      mangler_tdata->window = ssh_xbuffer_allocate();
      mangler_tdata->file_len = attrs->size;

      SSH_VERIFY(mangler_tdata->req_queue != NULL);
    }
  
  /* If not already allocated, allocate buffer queue. */
  if (!gdata->buf_queue)
    gdata->buf_queue = ssh_adt_create_generic(SSH_ADT_LIST,
                                              SSH_ADT_DESTROY,
                                              tc_buffer_destroy,
                                              SSH_ADT_ARGS_END);

  reader_tdata->parent = writer_tdata->parent = thread;
  
  if (gdata->attrs->progress_cb)
    {
      ssh_time_measure_reset(gdata->transfer_timer);
      ssh_time_measure_start(gdata->transfer_timer);

      /* Initialize timeout for progress callback. */
      ssh_register_timeout(1L, 0L, progress_timeout_cb, gdata);
    }

  SSH_FSM_SET_NEXT(transfer_finish_thread);
  SSH_FSM_WAIT_THREAD(gdata->writer);
}

/* "Shred" a file to multiple requests. */
SSH_FSM_STEP(transfer_shred)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferCoreThreadContext tdata = (TransferCoreThreadContext)thread_context;

  TCRequest new_req = NULL;
  SshFileAttributes src_attrs, dest_attrs;
  off_t bytes = 0L;

  src_attrs = ssh_file_copy_file_get_attributes(tdata->source_file);
  dest_attrs = ssh_file_copy_file_get_attributes(tdata->dest_file);

  SSH_ASSERT(src_attrs != NULL);
  SSH_ASSERT(src_attrs->flags & SSH_FILEXFER_ATTR_SIZE);
  SSH_ASSERT(gdata->attrs->max_buf_size);

  for (bytes = 0L; bytes < src_attrs->size;)
    {
      new_req = ssh_xcalloc(1, sizeof(*new_req));
      new_req->req_offset = bytes;
      new_req->req_bytes = gdata->attrs->max_buf_size <
        src_attrs->size - bytes ?
        gdata->attrs->max_buf_size : src_attrs->size - bytes;
      bytes += new_req->req_bytes;
      SSH_DEBUG(9, ("offset = %qd.", new_req->req_offset));
      SSH_VERIFY(ssh_adt_insert_to(tdata->req_queue, SSH_ADT_BEGINNING,
                                   new_req) != SSH_ADT_INVALID);
    }

  SSH_DEBUG(9, ("File was shredded to %ld buffers of size %ld (or less)",
                ssh_adt_num_objects(tdata->req_queue),
                gdata->attrs->max_buf_size));
  
  SSH_FSM_SET_NEXT(transfer_read_in);






















  
  return SSH_FSM_CONTINUE;
}

void read_in_cb(SshFileClientError error, const unsigned char *data,
                size_t len, const char *error_msg, const char *lang_tag,
                void *context)
{
  TCRequest req = (TCRequest) context;
  SshFSMThread thread = req->thread;
  SshFileCopyError error_code;
  const char *error_string;
  SshADTHandle h;
  SshFileCopyTransferContext gdata =
    (SshFileCopyTransferContext)ssh_fsm_get_gdata(thread);
  TransferCoreThreadContext tdata =
    (TransferCoreThreadContext)ssh_fsm_get_tdata(thread);

  req->op_handle = NULL;

  switch (error)
    {
    case SSH_FX_OK:
      SSH_DEBUG(7, ("Got %ld bytes of data.", len));

      ssh_xbuffer_append(req->buffer, data, len);

      if (req->req_bytes > len)
        {
          TCRequest new_req;
          /* If we get fewer bytes than we requested, we schedule a
             new request for the missing bit, and wake up the reader. */
          new_req = ssh_xcalloc(1, sizeof(*new_req));
          new_req->req_offset = req->req_offset + len;
          new_req->req_bytes = req->req_bytes - len;
          SSH_VERIFY(ssh_adt_insert_to(tdata->req_queue, SSH_ADT_BEGINNING,
                                       new_req) != SSH_ADT_INVALID);
        }
      req->req_bytes = len;
      tdata->read_bytes += len;

      h = ssh_adt_get_handle_to(tdata->in_progress_queue, req);
      SSH_VERIFY(h != SSH_ADT_INVALID);
      (void) ssh_adt_detach(tdata->in_progress_queue, h);

      if (gdata->mangle_this_file)
        {
          TransferManglerThreadContext mangler_tdata;
          SSH_DEBUG(7, ("Adding a write request to mangler's request queue."));
          mangler_tdata = ssh_fsm_get_tdata(gdata->mangler);
          h = ssh_adt_insert_to(mangler_tdata->req_queue, SSH_ADT_BEGINNING,
                                req);
          SSH_VERIFY(h != SSH_ADT_INVALID);

          SSH_FSM_CONDITION_SIGNAL(gdata->mangle_blocking);
        }
      else
        {
          TransferCoreThreadContext writer_tdata;
          SSH_DEBUG(7, ("Adding a write request to writer's request queue."));
          writer_tdata = ssh_fsm_get_tdata(gdata->writer);
          h = ssh_adt_insert_to(writer_tdata->req_queue, SSH_ADT_BEGINNING,
                                req);
          SSH_VERIFY(h != SSH_ADT_INVALID);
          
          SSH_FSM_CONDITION_SIGNAL(gdata->out_blocking);
        }
      return;
    case SSH_FX_EOF:
      error_code = SSH_FC_ERROR; error_string = "got EOF reading file";
      goto error_common;
    case SSH_FX_OP_UNSUPPORTED:
      error_code = SSH_FC_ERROR; error_string = "unsupported operation";
      goto error_common;
    case SSH_FX_NO_SUCH_FILE:
      error_code = SSH_FC_ERROR_NO_SUCH_FILE; error_string = "no such file";
      goto error_common;
    case SSH_FX_PERMISSION_DENIED:
      error_code = SSH_FC_ERROR_PERMISSION_DENIED;
      error_string = "permission denied";
      goto error_common;
    case SSH_FX_BAD_MESSAGE:
      error_code = SSH_FC_ERROR_PROTOCOL_MISMATCH;
      error_string = "protocol error";
      goto error_common;
    case SSH_FX_FAILURE:
      error_code = SSH_FC_ERROR_FAILURE; error_string = "undefined error";
      goto error_common;
    case SSH_FX_NO_CONNECTION:
      SSH_NOTREACHED;
      break;
    case SSH_FX_CONNECTION_LOST:
      error_code = SSH_FC_ERROR_CONNECTION_LOST;
      error_string = "connection lost";
      goto error_common;
    case SSH_FX_OUT_OF_MEMORY:
      error_code = SSH_FC_ERROR_FAILURE;
      error_string = "out of memory";
      break;
    error_common:
      ssh_adt_detach(tdata->in_progress_queue,
                     ssh_adt_get_handle_to(tdata->in_progress_queue, req));
      SSH_VERIFY(ssh_adt_insert_to(gdata->buf_queue, SSH_ADT_BEGINNING,
                                   req->buffer) != SSH_ADT_INVALID);
      /* Report error. */
      FCC_ERROR(error_code, ("%s (src): %s %s%s%s",
                             ssh_file_copy_file_get_name(tdata->source_file),
                             error_string,
                             error_msg ? "(server msg: '" : "",
                             error_msg ? error_msg : "",
                             error_msg ? "')" : ""));

      /* Cancel rest of the requests. */
      ssh_adt_destroy(tdata->in_progress_queue);
      tdata->in_progress_queue = NULL;
      /* Finish the threads. */
      SSH_FSM_SET_NEXT(transfer_readwrite_error);
      break;
    }

  req->buffer = NULL;
  tc_req_destroy(req, NULL);
  ssh_fsm_continue(thread);
}

SSH_FSM_STEP(transfer_read_in)
{
  SshBuffer buf;
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferCoreThreadContext tdata = (TransferCoreThreadContext)thread_context;
  TransferCoreThreadContext writer_tdata = ssh_fsm_get_tdata(gdata->writer);
  TCRequest req;

  if (ssh_adt_num_objects(tdata->req_queue) == 0 &&
      ssh_adt_num_objects(tdata->in_progress_queue) == 0 &&
      ssh_adt_num_objects(writer_tdata->req_queue) == 0 &&
      ssh_adt_num_objects(writer_tdata->in_progress_queue) == 0)
    {
      /* If all queues are empty, we can exit. */
      SSH_FSM_CONDITION_SIGNAL(gdata->out_blocking);
      gdata->reader = NULL;
      return SSH_FSM_FINISH;
    }

  if (ssh_adt_num_objects(tdata->req_queue) == 0)
    {
      SSH_DEBUG(8, ("No more requests at this time."));
      SSH_FSM_CONDITION_WAIT(gdata->in_blocking);
    }

  req = ssh_adt_detach_from(tdata->req_queue, SSH_ADT_END);
  SSH_ASSERT(req != SSH_ADT_INVALID);































  if (!ssh_adt_num_objects(gdata->buf_queue))
    {
      SSH_ASSERT(gdata->attrs->max_buffers);
      if (gdata->num_bufs < gdata->attrs->max_buffers)
        {
          buf = ssh_xbuffer_allocate();
          gdata->num_bufs++;
        }
      else
        {
          SshADTHandle h;
          /* Put req back to req_queue. */
          h = ssh_adt_insert_to(tdata->req_queue, SSH_ADT_END, req);
          SSH_ASSERT(h != SSH_ADT_INVALID);
          SSH_FSM_CONDITION_WAIT(gdata->in_blocking);
        }
    }
  else
    {
      buf = ssh_adt_detach_from(gdata->buf_queue, SSH_ADT_END);
      SSH_ASSERT(buf != NULL);
    }

  req->buffer = buf;
  req->thread = thread;

  SSH_DEBUG(6, ("Reading %ld bytes from offset %qd.",
                req->req_bytes, req->req_offset + gdata->attrs->read_offset));

  SSH_VERIFY(ssh_adt_insert_to(tdata->in_progress_queue, SSH_ADT_BEGINNING,
                               req) != SSH_ADT_INVALID);

  /* Result comes in a data callback. */
  req->op_handle = ssh_file_client_read
    (ssh_file_copy_file_get_handle(tdata->source_file),
     req->req_offset + gdata->attrs->read_offset, req->req_bytes,
     read_in_cb, (void *)req);

  return SSH_FSM_CONTINUE;
}

void write_out_cb(SshFileClientError error,
                  const char *error_msg,
                  const char *lang_tag,
                  void *context)
{
  TCRequest req = (TCRequest) context;
  SshFSMThread thread = req->thread;
  SshFileCopyError error_code;
  char *error_string;
  SshFileCopyTransferContext gdata =
    (SshFileCopyTransferContext)ssh_fsm_get_gdata(thread);
  TransferCoreThreadContext tdata =
    (TransferCoreThreadContext)ssh_fsm_get_tdata(thread);

  req->op_handle = NULL;

  switch (error)
    {
    case SSH_FX_OK:
      ssh_buffer_clear(req->buffer);
      tdata->written_bytes += req->req_bytes;
      SSH_VERIFY(ssh_adt_detach_object(tdata->in_progress_queue, req) != NULL);
      SSH_VERIFY(ssh_adt_insert_to(gdata->buf_queue, SSH_ADT_BEGINNING,
                                   req->buffer) != SSH_ADT_INVALID);
      SSH_FSM_CONDITION_SIGNAL(gdata->in_blocking);
      req->buffer = NULL;
      tc_req_destroy(req, NULL);
      return;
    case SSH_FX_EOF:
      error_code = SSH_FC_ERROR; error_string = "got EOF writing file";
      goto error_common;
    case SSH_FX_OP_UNSUPPORTED:
      error_code = SSH_FC_ERROR; error_string = "unsupported operation";
      goto error_common;
    case SSH_FX_NO_SUCH_FILE:
      error_code = SSH_FC_ERROR_NO_SUCH_FILE; error_string = "no such file";
      goto error_common;
    case SSH_FX_PERMISSION_DENIED:
      error_code = SSH_FC_ERROR_PERMISSION_DENIED;
      error_string = "permission denied";
      goto error_common;
    case SSH_FX_BAD_MESSAGE:
      error_code = SSH_FC_ERROR_PROTOCOL_MISMATCH;
      error_string = "protocol error";
      goto error_common;
    case SSH_FX_FAILURE:
      error_code = SSH_FC_ERROR_FAILURE; error_string = "undefined error";
      goto error_common;
    case SSH_FX_NO_CONNECTION:
      SSH_NOTREACHED;
      break;
    case SSH_FX_CONNECTION_LOST:
      error_code = SSH_FC_ERROR_CONNECTION_LOST;
      error_string = "connection lost";
      goto error_common;
    case SSH_FX_OUT_OF_MEMORY:
      error_code = SSH_FC_ERROR_FAILURE;
      error_string = "out of memory";
      break;
    error_common:
      ssh_adt_detach(tdata->in_progress_queue,
                     ssh_adt_get_handle_to(tdata->in_progress_queue, req));
      SSH_VERIFY(ssh_adt_insert_to(gdata->buf_queue, SSH_ADT_BEGINNING,
                                   req->buffer) != SSH_ADT_INVALID);
      /* Report error. */
      FCC_ERROR(error_code, ("%s (dst): %s %s%s%s",
                             ssh_file_copy_file_get_name(tdata->dest_file),
                             error_string,
                             error_msg ? "(server msg: '" : "",
                             error_msg ? error_msg : "",
                             error_msg ? "')" : ""));
      /* Cancel rest of the requests. */
      ssh_adt_destroy(tdata->in_progress_queue);
      tdata->in_progress_queue = NULL;
      /* Finish the threads. */
      SSH_FSM_SET_NEXT(transfer_readwrite_error);
      break;
    }

  req->buffer = NULL;
  tc_req_destroy(req, NULL);
  ssh_fsm_continue(thread);
}

SSH_FSM_STEP(transfer_write_out)
{
  TCRequest req;
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferCoreThreadContext tdata = (TransferCoreThreadContext)thread_context;

  if (gdata->reader == NULL && ssh_adt_num_objects(tdata->req_queue) == 0 &&
      gdata->mangler == NULL)
    {
      SshFileAttributes dest_attrs;
      
      if (gdata->attrs->progress_cb)
        {
          SshUInt64 seconds;
          off_t read_bytes;
          TransferCoreThreadContext reader_tdata;
          reader_tdata = gdata->reader ?
            ((TransferCoreThreadContext)ssh_fsm_get_tdata(gdata->reader)) :
            NULL;
          
          /* Unregister any progress indicator timeouts. Call the
             progress-callback directly to indicate that the
             transfer ended. */
          ssh_cancel_timeouts(progress_timeout_cb, gdata);
          ssh_time_measure_stop(gdata->transfer_timer);
          ssh_time_measure_get_value(gdata->transfer_timer,
                                     &seconds, NULL);
          read_bytes = reader_tdata ? reader_tdata->read_bytes :
            tdata->written_bytes;
          
          (*gdata->attrs->progress_cb)(gdata->src_conn,
                                       tdata->source_file,
                                       gdata->dest_conn,
                                       tdata->dest_file,
                                       read_bytes,
                                       tdata->written_bytes,
                                       tdata->skipped_bytes,
                                       seconds,
                                       TRUE,
                                       gdata->attrs->context);
        }

      dest_attrs = ssh_file_copy_file_get_attributes(tdata->dest_file);
      /* This is set, because upper layer needs to truncate the file to
         the correct length. */
      dest_attrs->permissions |= SSH_FILEXFER_ATTR_SIZE;
      dest_attrs->size = tdata->written_bytes;
      SSH_DEBUG(2, ("Writer finished."));
      gdata->writer = NULL;
      SSH_FSM_SET_NEXT(transfer_core_finish);
      return SSH_FSM_CONTINUE;
    }

  if (ssh_adt_num_objects(tdata->req_queue) == 0)
    {
      SSH_DEBUG(8, ("No more requests at this time."));
      SSH_FSM_CONDITION_WAIT(gdata->out_blocking);
    }

  req = ssh_adt_detach_from(tdata->req_queue, SSH_ADT_END);
  req->thread = thread;

  SSH_ASSERT(req != SSH_ADT_INVALID);

  SSH_DEBUG(6, ("Writing %ld bytes to offset %qd.",
                ssh_buffer_len(req->buffer), req->req_offset +
                gdata->attrs->write_offset));

  SSH_VERIFY(ssh_adt_insert_to(tdata->in_progress_queue, SSH_ADT_BEGINNING,
                               req) != SSH_ADT_INVALID);

  /* Result comes in a status callback, thus written bytes has to be
     stored. */
  req->op_handle = ssh_file_client_write
    (ssh_file_copy_file_get_handle(tdata->dest_file),
     req->req_offset + gdata->attrs->write_offset,
     ssh_buffer_ptr(req->buffer), ssh_buffer_len(req->buffer),
     write_out_cb, (void *)req);

  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(transfer_readwrite_error)
{
  SSH_FSM_SET_NEXT(transfer_core_finish);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(transfer_core_finish)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;

  if (gdata->reader == thread)
    gdata->reader = NULL;
  
  if (gdata->writer == thread)
    gdata->writer = NULL;

  /* We have to kill the other thread, because if we aborted because of
     an error, the other thread would just hang in the condition
     variable. */
  if (gdata->reader)
    {
      ssh_fsm_kill_thread(gdata->reader);
      gdata->reader = NULL;
    }
  if (gdata->writer)
    {
      ssh_fsm_kill_thread(gdata->writer);
      gdata->writer = NULL;
    }








  
  if (gdata->mangler)
    {
      ssh_fsm_kill_thread(gdata->mangler);
      gdata->mangler = NULL;
    }
  ssh_cancel_timeouts(progress_timeout_cb, gdata);
  
  return SSH_FSM_FINISH;
}

void progress_timeout_cb(void *context)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)context;
  TransferCoreThreadContext reader_tdata;
  TransferCoreThreadContext writer_tdata;
  SshUInt64 seconds;

  reader_tdata = gdata->reader ? ssh_fsm_get_tdata(gdata->reader) : NULL;
  writer_tdata = ssh_fsm_get_tdata(gdata->writer);

  ssh_time_measure_get_value(gdata->transfer_timer,
                             &seconds, NULL);
  /* The result may be a little off; if we're doing newline mangling,
     the exact filesize may be a little off. There is no good solution
     for this; only way to calculate the resulting filesize would be to
     count the number of newlines in the source file before the actual
     transfer, and believe me, we are not going to do that. */

  (*gdata->attrs->progress_cb)(gdata->src_conn,
                               writer_tdata->source_file,
                               gdata->dest_conn,
                               writer_tdata->dest_file,
                               reader_tdata ?
                               reader_tdata->read_bytes
                               : writer_tdata->written_bytes,
                               writer_tdata->written_bytes,
                               writer_tdata->skipped_bytes,
                               seconds,
                               FALSE,
                               gdata->attrs->context);

  ssh_register_timeout(1L, 0L, progress_timeout_cb, gdata);
}


















































































































































































































SSH_FSM_STEP(transfer_mangle)
{
  TransferManglerThreadContext tdata =
    (TransferManglerThreadContext)thread_context;
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  SshADTContainer l;
  SshADTHandle h;
  TCRequest req;
  SshBufferStruct tmp_buf;
  SshBuffer req_buf;
  TransferCoreThreadContext writer_tdata;
  Boolean done_something = FALSE;
  size_t window_len, bytes;
  unsigned char *window_ptr;
  TCRequest new_req;
  size_t newline_len;
  
  SSH_PRECOND(gdata->mangle_this_file);

  l = tdata->req_queue;
  if (ssh_adt_num_objects(l) == 0)
    {
      if (tdata->file_len > 0)
        {
          SSH_FSM_CONDITION_WAIT(gdata->mangle_blocking);
        }
      else
        {
          SSH_TRACE(2, ("Nothing to mangle, finishing mangler."));
          gdata->mangler = NULL;
          SSH_FSM_CONDITION_SIGNAL(gdata->out_blocking);
          return SSH_FSM_FINISH;
        }
    }
  
  /* Sort request queue to the order of offsets, so we can append them
     to the buffer in one go. */
  ssh_adt_list_sort(l);

  /* Append requests to the window. */
  while ((h = ssh_adt_get_handle_to_location(l, SSH_ADT_BEGINNING)) !=
         SSH_ADT_INVALID)
    {
      req = ssh_adt_get(l, h);
      SSH_ASSERT(req != NULL);
      if (tdata->read_offset + ssh_buffer_len(tdata->window) ==
          req->req_offset)
        {
          SSH_DEBUG(2, ("Adding buffer contents to mangling window."));
          SSH_VERIFY(ssh_adt_detach(l, h) == req);
          ssh_xbuffer_append(tdata->window, ssh_buffer_ptr(req->buffer),
                             ssh_buffer_len(req->buffer));
          ssh_buffer_clear(req->buffer);
          SSH_VERIFY(ssh_adt_insert_to(gdata->buf_queue, SSH_ADT_BEGINNING,
                                       req->buffer) != SSH_ADT_INVALID);
          ssh_xfree(req);
          done_something = TRUE;
        }
      else
        {
          if (done_something == FALSE)
            {
              /* Requests are not sequential; we have to wait for
                 more. */
              SSH_DEBUG(6, ("Waiting for more requests."));
              SSH_FSM_CONDITION_WAIT(gdata->mangle_blocking);
            }
        }
    }
  /* Mangle. */
  SSH_ASSERT(strcmp(gdata->src_newline,
                    gdata->dest_newline) != 0);

  window_len = ssh_buffer_len(tdata->window);
  window_ptr = ssh_buffer_ptr(tdata->window);
  ssh_buffer_init(&tmp_buf);

  bytes = 0;
  newline_len = strlen(gdata->src_newline);
  if (window_len >= newline_len)
    {
      for (bytes = 0; bytes <= window_len - newline_len;)
        {
          if (memcmp(window_ptr + bytes, gdata->src_newline,
                     strlen(gdata->src_newline)) == 0)
            {
              ssh_xbuffer_append(&tmp_buf, gdata->dest_newline,
                                 strlen(gdata->dest_newline));
              bytes += strlen(gdata->src_newline);
            }
          else
            {
              ssh_xbuffer_append(&tmp_buf, window_ptr + bytes, 1);
              bytes++;
            }
        }
    }
  ssh_buffer_consume(tdata->window, bytes);
  tdata->read_offset += bytes;
  if (tdata->read_offset + ssh_buffer_len(tdata->window) >= tdata->file_len)
    {
      /* No room for additional newlines; add rest of window to be
         written. */
      SSH_DEBUG(6, ("Adding rest of file to write buffer (no room for "
                    "newline (%ld)).", ssh_buffer_len(tdata->window)));
      ssh_buffer_append(&tmp_buf, window_ptr + bytes, window_len - bytes);
      tdata->read_offset += window_len - bytes;
      ssh_buffer_clear(tdata->window);
    }
  
  writer_tdata = ssh_fsm_get_tdata(gdata->writer);
  for (bytes = 0; bytes < ssh_buffer_len(&tmp_buf);)
    {
      if (!ssh_adt_num_objects(gdata->buf_queue))
        {
          req_buf = ssh_xbuffer_allocate();
          gdata->num_bufs++;
        }
      else
        {
          req_buf = ssh_adt_detach_from(gdata->buf_queue, SSH_ADT_END);
        }
      new_req = ssh_xcalloc(1, sizeof(*new_req));
      new_req->req_offset = tdata->write_offset + bytes;
      new_req->req_bytes = gdata->attrs->max_buf_size <
        ssh_buffer_len(&tmp_buf) - bytes ?
        gdata->attrs->max_buf_size : ssh_buffer_len(&tmp_buf) - bytes;
      ssh_buffer_clear(req_buf);
      ssh_buffer_append(req_buf, ssh_buffer_ptr(&tmp_buf) + bytes,
                        new_req->req_bytes);
      bytes += new_req->req_bytes;
      new_req->buffer = req_buf;
      SSH_VERIFY(ssh_adt_insert_to(writer_tdata->req_queue, SSH_ADT_END,
                                   new_req) != SSH_ADT_INVALID);
      SSH_FSM_CONDITION_SIGNAL(gdata->out_blocking);
    }
  ssh_buffer_uninit(&tmp_buf);
  tdata->write_offset += bytes;
  SSH_DEBUG(8, ("file_len = %qd, write_offset = %qd, "
                "tdata->read_offset = %qd.", tdata->file_len,
                tdata->write_offset, tdata->read_offset));
  if (tdata->read_offset >= tdata->file_len)
    {
      gdata->mangler = NULL;
      /* We've mangled everything. */
      return SSH_FSM_FINISH;
    }
  else
    {
      return SSH_FSM_CONTINUE;
    }
}
