/*
  sshfc_transfer.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 2000-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Functions that perform the actual transfer.
*/

#include "ssh2includes.h"
#include "sshfilecopy.h"
#include "sshfilecopyi.h"
#include "sshbuffer.h"
#include "sshtimemeasure.h"
#include "sshfc_transferi.h"
#include "sshglob.h"




#define SSH_DEBUG_MODULE "SshFCTransfer"

/**********
 * ssh_file_copy_transfer_files() related stuff.
 **********/

/* Forward declarations. */
SSH_FSM_STEP(transfer_check_src_conn);
SSH_FSM_STEP(transfer_check_dest_conn);
SSH_FSM_STEP(transfer_conn_failed);
SSH_FSM_STEP(transfer_get_newline_conventions);
SSH_FSM_STEP(transfer_stat_dest_loc);
SSH_FSM_STEP(transfer_walk_file_list);
SSH_FSM_STEP(transfer_prepare_base_dir);
SSH_FSM_STEP(transfer_prepare_file);
SSH_FSM_STEP(transfer_start_file_ops);
SSH_FSM_STEP(transfer_walk_tree);
SSH_FSM_STEP(transfer_stat_src);
SSH_FSM_STEP(transfer_mkdir);
SSH_FSM_STEP(transfer_check_overwrite);
SSH_FSM_STEP(transfer_check_cmp_sizes);
SSH_FSM_STEP(transfer_open_dest);
SSH_FSM_STEP(transfer_stat_dest);
SSH_FSM_STEP(transfer_dir_descent);
SSH_FSM_STEP(transfer_chmod_dest_dir);
SSH_FSM_STEP(transfer_rm_src_dir);
SSH_FSM_STEP(transfer_open_src);
SSH_FSM_STEP(transfer_chmod_dest);





SSH_FSM_STEP(transfer_core_start);
SSH_FSM_STEP(transfer_set_attrs);
SSH_FSM_STEP(transfer_close_src);
SSH_FSM_STEP(transfer_close_dest);
SSH_FSM_STEP(transfer_rm_src);
SSH_FSM_STEP(transfer_done);
SSH_FSM_STEP(transfer_cleanup);

void fct_stat_cb(SshFileClientError error, SshFileCopyFile file,
                 SshFileAttributes attrs, const char *error_msg,
                 SshFSMThread thread);

/*
  Prototype for

    SSH_FSM_STEP(transfer_finish_thread);

  is in sshfc_transferi.h .
*/

#define FCC_CLEANUP(error, message)                             \
do                                                              \
{                                                               \
  (*gdata->completion_callback)((error), (message),             \
                                gdata->callback_context);       \
  /* stop transfer and abort */                                 \
  ssh_fsm_set_next(gdata->main_thread, transfer_cleanup);       \
  ssh_fsm_continue(gdata->main_thread);                         \
}                                                               \
while (0)

FCC_GEN_BF_CALLBACK(fct, SshFileCopyTransferContext, TransferThreadContext)
FCC_GEN_STATUS(fct)
FCC_GEN_HANDLE(fct)
FCC_GEN_ATTR(fct)




void transfer_thread_context_destroy(TransferThreadContext thread_context)
{
  if (thread_context->op_handle)
    {
      ssh_operation_abort(thread_context->op_handle);
      thread_context->op_handle = NULL;
    }
  if (thread_context->parent)
    {
      TransferThreadContext parent_tdata;
      parent_tdata = ssh_fsm_get_tdata(thread_context->parent);
      parent_tdata->child = NULL;
    }

  if (thread_context->child)
    {
      TransferThreadContext child_tdata;
      child_tdata = ssh_fsm_get_tdata(thread_context->child);
      child_tdata->parent = NULL;
      ssh_fsm_kill_thread(thread_context->child);
      thread_context->child = NULL;
    }
  /* XXX The rest. */
}

void transfer_core_helper_destructor(SshFSM fsm, void *tdata)
{
  TransferThreadContext thread_context = (TransferThreadContext) tdata;
  
  SSH_PRECOND(tdata != NULL);

  transfer_thread_context_destroy(thread_context);
  ssh_xfree(thread_context);
}


void transfer_thread_destructor(SshFSM fsm, void *tdata)
{
  TransferThreadContext thread_context = (TransferThreadContext) tdata;

  SSH_PRECOND(tdata != NULL);

  transfer_thread_context_destroy(thread_context);
  
  if (thread_context->cur_dest)
    ssh_file_copy_file_destroy(thread_context->cur_dest);
  if (thread_context->cur_loc_allocated)
    ssh_file_copy_file_destroy(thread_context->cur_loc);
  
  ssh_xfree(thread_context->src_file_full_path);
  ssh_xfree(thread_context);
}

void fct_conn_completion_cb(SshFileClient client, void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  SshFileCopyTransferContext gdata =
    (SshFileCopyTransferContext)ssh_fsm_get_gdata(thread);
  TransferThreadContext tdata =
    (TransferThreadContext)ssh_fsm_get_tdata(thread);

  tdata->op_handle = NULL;

  if (!client)
    SSH_FSM_SET_NEXT(transfer_conn_failed);
  
  if (tdata->dealing_with_source)
    gdata->src_conn->client = client;
  else
    gdata->dest_conn->client = client;
  
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_FSM_STEP(transfer_check_src_conn)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  tdata->dealing_with_source = TRUE;

  SSH_FSM_SET_NEXT(transfer_check_dest_conn);

  if (gdata->src_conn->client)
    {
      SSH_DEBUG(4, ("Source connection OK."));
      return SSH_FSM_CONTINUE;
    }
  SSH_DEBUG(2, ("Establishing source connection."));
  SSH_FSM_ASYNC_CALL(tdata->op_handle =
                     ssh_file_copy_connect(gdata->src_conn,
                                           fct_conn_completion_cb,
                                           thread));

}

SSH_FSM_STEP(transfer_check_dest_conn)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  tdata->dealing_with_source = FALSE;

  SSH_FSM_SET_NEXT(transfer_get_newline_conventions);
  
  if (gdata->dest_conn->client)
    {
      SSH_DEBUG(4, ("Destination connection OK."));
      return SSH_FSM_CONTINUE;
    }
  SSH_DEBUG(2, ("Establishing destination connection."));
  SSH_FSM_ASYNC_CALL(tdata->op_handle =
                     ssh_file_copy_connect(gdata->dest_conn,
                                           fct_conn_completion_cb,
                                           thread));
}

SSH_FSM_STEP(transfer_get_newline_conventions)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  unsigned char *nl = NULL;

  SSH_FSM_SET_NEXT(transfer_stat_dest_loc);

  SSH_DEBUG(2, ("Fething source newline convention extension data."));
  if (!ssh_file_client_get_extension_data
      (gdata->src_conn->client, "newline@vandyke.com",
       &nl, NULL))
    {
      SSH_DEBUG(2, ("Source sftp-server didn't give newline "
                    "convention."));
      nl = gdata->attrs->source_newline;
    }

  if (nl)
    gdata->src_newline = ssh_xstrdup(nl);

  nl = NULL;
  
  SSH_DEBUG(2, ("Fething dest newline convention extension data."));
  if (!ssh_file_client_get_extension_data
      (gdata->dest_conn->client, "newline@vandyke.com",
       &nl, NULL))
    {
      SSH_DEBUG(2, ("Destination sftp-server didn't give newline "
                    "convention."));
      nl = gdata->attrs->dest_newline;
    }

  if (nl)
    gdata->dest_newline = ssh_xstrdup(nl);

  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(transfer_conn_failed)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  const char *msg;

  if (tdata->dealing_with_source)
    msg = "Connecting to source failed.";
  else
    msg = "Connecting to destination failed.";
    
  (*gdata->completion_callback)(SSH_FC_ERROR_CONNECTION_FAILED,
                                msg, gdata->callback_context);
  SSH_FSM_SET_NEXT(transfer_cleanup);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(transfer_stat_dest_loc)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  SSH_DEBUG(3, ("Statting destination location ``%s'' before transfer...",
                ssh_file_copy_file_get_name(gdata->dest_loc)));

  FCC_START_ATTRS(tdata, gdata->dest_loc, FALSE, fct_stat_cb);
  SSH_FSM_SET_NEXT(transfer_walk_file_list);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_stat
                     (gdata->dest_conn->client,
                      ssh_file_copy_file_get_name(gdata->dest_loc),
                      transfer_attribute_cb, thread));
}

SSH_FSM_STEP(transfer_walk_file_list)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  SshFileAttributes dest_attrs;
  dest_attrs = ssh_file_copy_file_get_attributes(gdata->dest_loc);

  if (gdata->attrs->destination_must_be_dir &&
      (!dest_attrs || ((dest_attrs->permissions & S_IFMT) != S_IFDIR)))
    {
      char *str =
        fcc_format_error("destination (%s) is not a directory.",
                         ssh_file_copy_file_get_name(gdata->dest_loc));
      (*gdata->completion_callback)(SSH_FC_ERROR_DEST_NOT_DIR,
                                    str, gdata->callback_context);
      ssh_xfree(str);
      SSH_FSM_SET_NEXT(transfer_cleanup);
      return SSH_FSM_CONTINUE;
    }
  
  SSH_DEBUG(3, ("Setting next source file item..."));

  if (tdata->h != SSH_ADT_INVALID)
    {
      TransferThreadContext ct_ctx;
      ct_ctx = ssh_xcalloc(1, sizeof(*ct_ctx));
      ct_ctx->parent = thread;
      tdata->child = ssh_fsm_thread_create(fsm,
                                           transfer_prepare_base_dir,
                                           NULL_FNPTR,
                                           transfer_thread_destructor,
                                           ct_ctx);
      SSH_VERIFY(tdata->child != NULL);
      ct_ctx->cur_loc = (SshFileCopyFile) ssh_adt_get(gdata->file_list,
                                                      tdata->h);
      SSH_ASSERT(ct_ctx->cur_loc != NULL);
      tdata->h = ssh_adt_enumerate_next(gdata->file_list, tdata->h);
      SSH_FSM_WAIT_THREAD(tdata->child);
    }
  else
    {
      SSH_DEBUG(3, ("No more source files in this list."));
      SSH_FSM_SET_NEXT(transfer_done);
      return SSH_FSM_CONTINUE;
    }
  SSH_NOTREACHED;
}

/* Some background:

   The file list can have two types of files:

   1) A `lone' file, which has been added to the list, and has not
   been recursed or globbed in anyway. It doesn't have a parent file
   object. It should be transferred to the destination in it's own
   right.

   2) A directory tree, which has either been globbed or recursed, or
   both. The file objects have relations (except for the upmost one,
   all files have a parent), and the upmost file (i.e. a directory)
   should not created (it is used to make the source file path).
*/
SSH_FSM_STEP(transfer_prepare_base_dir)
{
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  TransferThreadContext ct_ctx;
  char *hlp;
  
  ct_ctx = ssh_xcalloc(1, sizeof(*ct_ctx));
  ct_ctx->parent = thread;
  ct_ctx->basedir = ssh_xstrdup(ssh_file_copy_file_get_name(tdata->cur_loc));

  SSH_DEBUG(2, ("file name: ``%s''", ct_ctx->basedir));
  
  if (tdata->cur_loc->dir_entries == NULL)
    {
      SshFileCopyFile new_file = ssh_file_copy_file_allocate();
      SshADTHandle h;
      char *file_name;

      SSH_DEBUG(2, ("file ``%s'' doesn't have dir entries", ct_ctx->basedir));
      /* The lone file case. */
      
      /* This mangling is needed, so that
         ssh_file_copy_file_generate_full_path() will return the
         correct filename. */
      tdata->cur_loc = ssh_file_copy_file_dup(tdata->cur_loc);
      tdata->cur_loc->dir_entries = ssh_file_copy_file_list_create(NULL_FNPTR,
                                                                   NULL);
      tdata->cur_loc_allocated = TRUE;
      new_file->parent_dir = tdata->cur_loc;
      
      if ((hlp = strrchr(ct_ctx->basedir, '/')) != NULL)
        {
          *hlp = '\0';
          hlp++;
          file_name = ssh_xstrdup(hlp);
          
          if (strlen(ct_ctx->basedir) == 0)
            {
              ssh_xfree(ct_ctx->basedir);
              ct_ctx->basedir = ssh_xstrdup("/");
            }
        }
      else
        {
          file_name = ssh_xstrdup(ct_ctx->basedir);
          ssh_xfree(ct_ctx->basedir);
          ct_ctx->basedir = ssh_xstrdup(".");
        }
      SSH_ASSERT(file_name != NULL);
      ssh_file_copy_file_register_filename(new_file, file_name);
      ssh_file_copy_file_register_filename(tdata->cur_loc,
                                           ssh_xstrdup(ct_ctx->basedir));
      h = ssh_adt_insert(tdata->cur_loc->dir_entries, new_file);
      
      SSH_ASSERT(h != SSH_ADT_INVALID);
    }

  ct_ctx->cur_loc = tdata->cur_loc;

  /* Let's walk the tree*/
  ct_ctx->h = ssh_adt_enumerate_start(tdata->cur_loc->dir_entries);
  
  hlp = ssh_glob_strip_escapes(ct_ctx->basedir);
  ssh_xfree(ct_ctx->basedir);
  ct_ctx->basedir = hlp;
  
  SSH_TRACE(2, ("basedir = %s", ct_ctx->basedir));
  
  tdata->child = ssh_fsm_thread_create(fsm, transfer_walk_tree, NULL_FNPTR,
                                       transfer_thread_destructor, ct_ctx);
  SSH_VERIFY(tdata->child != NULL);

  SSH_FSM_SET_NEXT(transfer_finish_thread);
  SSH_FSM_WAIT_THREAD(tdata->child);
}

SSH_FSM_STEP(transfer_walk_tree)
{
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  if (tdata->h != SSH_ADT_INVALID)
    {
      TransferThreadContext ct_ctx;

      ct_ctx = ssh_xcalloc(1, sizeof(*ct_ctx));      
      ct_ctx->cur_loc = ssh_adt_get(tdata->cur_loc->dir_entries, tdata->h);
      ct_ctx->basedir = ssh_xstrdup(tdata->basedir);
      ct_ctx->parent = thread;
      
      SSH_ASSERT(ct_ctx->cur_loc != NULL);
      
      if (ct_ctx->cur_loc->dir_entries)
        {
          ct_ctx->h = ssh_adt_enumerate_start(ct_ctx->cur_loc->dir_entries);
        }
      
      tdata->h = ssh_adt_enumerate_next(tdata->cur_loc->dir_entries, tdata->h);
      tdata->child = ssh_fsm_thread_create(fsm, transfer_prepare_file,
                                           NULL_FNPTR,
                                           transfer_thread_destructor, ct_ctx);
      SSH_ASSERT(tdata->child != NULL);
      SSH_FSM_WAIT_THREAD(tdata->child);
    }
  return SSH_FSM_FINISH;
}

SSH_FSM_STEP(transfer_prepare_file)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  char *hlp, *hlp2;
  const char *dest_path;
  SshFileAttributes dest_attrs =
    ssh_file_copy_file_get_attributes(gdata->dest_loc);
  
  SSH_FSM_SET_NEXT(transfer_start_file_ops);
  
  /* Remove escapes. */
  tdata->src_file_full_path =
    ssh_file_copy_file_generate_full_path(tdata->cur_loc);
  
  /* Mangle with dest_loc. */
  tdata->cur_dest = ssh_file_copy_file_allocate();
  dest_path = ssh_file_copy_file_get_name(gdata->dest_loc);
  
  if (gdata->attrs->destination_must_be_dir ||
      (dest_attrs && (dest_attrs->permissions & S_IFMT) == S_IFDIR))
    {
      SSH_ASSERT(tdata->basedir[0] != '\0');

      if (gdata->attrs->create_dirs &&
          !strncmp(tdata->src_file_full_path, tdata->basedir,
                   strlen(tdata->basedir)) &&
          (tdata->src_file_full_path[strlen(tdata->basedir)] == '/' ||
           (!strcmp(tdata->basedir, "/") &&
            tdata->basedir[0] == '/')))
        {
          hlp = &(tdata->src_file_full_path[strlen(tdata->basedir)]);
          if (strcmp(tdata->basedir, "/") != 0)
            hlp++;          
        }
      else
        {
          hlp = (char *)ssh_file_copy_file_get_name(tdata->cur_loc);
        }
      
      ssh_xdsprintf(&hlp2, "%s%s%s",
                    dest_path,
                    (dest_path[0] != '\0' && strcmp(dest_path, "/") != 0) ?
                    "/" : "",
                    hlp);
      hlp = hlp2;
    }
  else
    {
      hlp = ssh_xstrdup(dest_path);
    }
























  
  ssh_file_copy_file_register_filename(tdata->cur_dest, hlp);
  SSH_DEBUG(3, ("destination file: %s", hlp));
  SSH_DEBUG(3, ("source file: %s", tdata->src_file_full_path));
  
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(transfer_start_file_ops)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  SshFileAttributes attrs;
  
  attrs = ssh_file_copy_file_get_attributes(tdata->cur_loc);

  if (attrs == NULL ||
      !(attrs->flags & SSH_FILEXFER_ATTR_PERMISSIONS ) ||
      !(attrs->flags & SSH_FILEXFER_ATTR_ACMODTIME) ||
      (attrs->permissions & S_IFMT) == S_IFLNK)
    {
      if (tdata->stat_done)
        {
          SSH_DEBUG(2, ("File server didn't give us all attributes for "
                        "file ``%s'', trying to continue without them.",
                        ssh_file_copy_file_get_name(tdata->cur_loc)));
        }
      else
        {
          SSH_FSM_SET_NEXT(transfer_stat_src);
          return SSH_FSM_CONTINUE;
        }
    }
  
  if (!tdata->stat_done)
    tdata->stat_done = TRUE;

  switch (attrs->permissions & S_IFMT)
    {
    case S_IFDIR:
      /* stat_dest -> mkdir -> walk_tree (spawn) -> chmod_dest_dir ->
         rm_src_dir -> fin */
      if (!gdata->attrs->create_dirs)
        {
          if (!tdata->cur_loc->dir_entries)
            {
              /* Print warning, if we've been asked to specifically
                 transfer this file (and not some under it). */
              FCC_ERROR(SSH_FC_ERROR,
                        ("File ``%s'' is a directory, and directory creation "
                         "(recursion) has not been enabled. ",
                         ssh_file_copy_file_get_name(tdata->cur_loc)));
              return SSH_FSM_FINISH;
            }
          else
            {
              SSH_FSM_SET_NEXT(transfer_dir_descent);
              return SSH_FSM_CONTINUE;
            }
        }

      tdata->after_stat_state = transfer_mkdir;      
      break;
    case S_IFREG:
      /* 
         stat_dest -> check_overwrite -> open_dest -> open_src ->
         (stat_dest_after_creation ->) chmod_dest



         transfer_start (core) -> set_attrs
         -> close_src -> rm_src -> close_dest -> fin
      */
      SSH_ASSERT(tdata->cur_loc->dir_entries == NULL);
      tdata->after_stat_state = transfer_check_overwrite;
      break;
    default:
      FCC_ERROR(SSH_FC_ERROR, ("File ``%s'' not a regular file, directory or "
                               "symlink.",
                               ssh_file_copy_file_get_name(tdata->cur_loc)));
      return SSH_FSM_FINISH;
    }

  SSH_ASSERT(tdata->after_stat_state != NULL_FNPTR);
  
  SSH_FSM_SET_NEXT(transfer_stat_dest);
  
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(transfer_finish_thread)
{
  return SSH_FSM_FINISH;
}


/************************************************************
 * Common operations.
 ************************************************************/

void fct_stat_cb(SshFileClientError error, SshFileCopyFile file,
                      SshFileAttributes attrs, const char *error_msg,
                      SshFSMThread thread)
{
  FCC_DATA(SshFileCopyTransferContext, TransferThreadContext);

  if (error == SSH_FX_OK)
    {
      /* Store attributes */
      ssh_file_copy_file_register_attributes(file,
                                             ssh_file_attributes_dup(attrs));
    }
  else if (error != SSH_FX_NO_SUCH_FILE && !tdata->dealing_with_source)
    {
      /* We don't care if the destination file didn't previously
         exist. (but output error in other cases) */
      FCC_ERROR(fcc_fx_err_to_fc_err(error), ("stat: %s", error_msg));
    }
  else if (tdata->dealing_with_source)
    {
      /* If the source file doesn't exist (or we encounter some other
         error), we drop this file. */
      SSH_FSM_SET_NEXT(transfer_finish_thread);
      FCC_ERROR(fcc_fx_err_to_fc_err(error), ("stat: %s", error_msg));
    }
}

SSH_FSM_STEP(transfer_stat_src)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  SSH_DEBUG(3, ("Statting source file ``%s''...", tdata->src_file_full_path));

  FCC_START_ATTRS(tdata, tdata->cur_loc, TRUE, fct_stat_cb);
  SSH_FSM_SET_NEXT(transfer_start_file_ops);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_stat
                         (gdata->src_conn->client, tdata->src_file_full_path,
                          transfer_attribute_cb, thread));
}

SSH_FSM_STEP(transfer_stat_dest)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  
  SSH_DEBUG(2, ("Statting destination file ``%s''...",
                ssh_file_copy_file_get_name(tdata->cur_dest)));
  FCC_START_ATTRS(tdata, tdata->cur_dest, FALSE, fct_stat_cb);
  SSH_FSM_SET_NEXT(tdata->after_stat_state);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_stat
                     (gdata->dest_conn->client,
                      ssh_file_copy_file_get_name(tdata->cur_dest),
                      transfer_attribute_cb, thread));
}

void fct_stat_after_cb(SshFileClientError error, SshFileCopyFile file,
                       SshFileAttributes attrs, const char *error_msg,
                       SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyTransferContext);

  if (error == SSH_FX_OK)
    {
      /* Store attributes */
      ssh_file_copy_file_register_attributes(file,
                                             ssh_file_attributes_dup(attrs));
    }
  else if (error != SSH_FX_NO_SUCH_FILE)
    {
      SSH_FSM_SET_NEXT(transfer_finish_thread);
      FCC_ERROR(fcc_fx_err_to_fc_err(error), ("stat_after: %s", error_msg));
    }
}

SSH_FSM_STEP(transfer_stat_dest_after_creation)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  
  SSH_DEBUG(3, ("Statting destination file ``%s'' again...",
                ssh_file_copy_file_get_name(tdata->cur_dest)));
  FCC_START_ATTRS(tdata, tdata->cur_dest, FALSE, fct_stat_after_cb);
  SSH_FSM_SET_NEXT(tdata->after_stat_state);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_stat
                     (gdata->dest_conn->client,
                      ssh_file_copy_file_get_name(tdata->cur_dest),
                      transfer_attribute_cb, thread));
}

/************************************************************
 * Operations on directories.
 ************************************************************/

void fct_mkdir_cb(SshFileClientError error, SshFileCopyFile file,
                  const char *error_msg, SshFSMThread thread)
{
  FCC_DATA(SshFileCopyTransferContext, TransferThreadContext);

  if (error == SSH_FX_OK)
    {
      SSH_DEBUG(3, ("Successfully created directory %s.",
                    ssh_file_copy_file_get_name(file)));
      return;
    }
  else if (error == SSH_FX_FAILURE)
    {
      /* When creating directories, this error occurs if the directory
         already existed. So, the destination directory is statted to
         see if it exists, to avoid giving the user false messages. */
      if (ssh_file_copy_file_get_attributes(tdata->cur_dest) != NULL)
        {
          /* File (dir) exists. */
          SSH_TRACE(2, ("Destination directory ``%s'' already existed.",
                        ssh_file_copy_file_get_name(file)));
          return;
        }
      /* else FALL TO BOTTOM */
    }

  SSH_FSM_SET_NEXT(transfer_finish_thread);
  FCC_ERROR(fcc_fx_err_to_fc_err(error), ("mkdir: %s", error_msg));
}

SSH_FSM_STEP(transfer_mkdir)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;











  FCC_START_STATUS(tdata, tdata->cur_dest, FALSE, fct_mkdir_cb);
  SSH_FSM_SET_NEXT(transfer_dir_descent);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_mkdir
                     (gdata->dest_conn->client,
                      ssh_file_copy_file_get_name(tdata->cur_dest),
                      NULL, transfer_status_cb, thread));
}

SSH_FSM_STEP(transfer_dir_descent)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  TransferThreadContext ct_ctx;

  /* If directory creation is not enabled, and we encounter a dir, we
     come here directly from transfer_file_ops. */
  if (gdata->attrs->create_dirs)
    SSH_FSM_SET_NEXT(transfer_chmod_dest_dir);
  else
    SSH_FSM_SET_NEXT(transfer_finish_thread);
  
  ct_ctx = ssh_xcalloc(1, sizeof(*ct_ctx));
  ct_ctx->cur_loc = tdata->cur_loc;
  ct_ctx->h = tdata->h;
  ct_ctx->basedir = ssh_xstrdup(tdata->basedir);
  ct_ctx->parent = thread;
  tdata->child = ssh_fsm_thread_create(fsm,
                                       transfer_walk_tree,
                                       NULL_FNPTR,
                                       transfer_thread_destructor,
                                       ct_ctx);
  SSH_ASSERT(tdata->child != NULL);
  SSH_FSM_WAIT_THREAD(tdata->child);
}

void fct_chmod_dest_dir_cb(SshFileClientError error,
                           SshFileCopyFile file,
                           const char *error_msg, SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyTransferContext);

  if (error != SSH_FX_OK)
    {
      /* This is not fatal. */
      FCC_ERROR(fcc_fx_err_to_fc_err(error),
                ("chmod_dest_dir: %s", error_msg));
    }
}

SSH_FSM_STEP(transfer_chmod_dest_dir)
{
  SshFileAttributesStruct attrs;
  SshFileAttributes orig_attrs;
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;


  if (!gdata->attrs->preserve_attributes)
    {
      SSH_FSM_SET_NEXT(transfer_rm_src_dir);
      return SSH_FSM_CONTINUE;
    }


  orig_attrs = ssh_file_copy_file_get_attributes(tdata->cur_loc);

  attrs = *orig_attrs;

  /* Remove size from flags. */
  attrs.flags &= ~SSH_FILEXFER_ATTR_SIZE;
  /* Remove uidgid from flags. */
  attrs.flags &= ~SSH_FILEXFER_ATTR_UIDGID;















  FCC_START_STATUS(tdata, tdata->cur_dest, FALSE,
                   fct_chmod_dest_dir_cb);
  SSH_FSM_SET_NEXT(transfer_rm_src_dir);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_setstat
                     (gdata->dest_conn->client,
                      ssh_file_copy_file_get_name(tdata->cur_dest),
                      &attrs, transfer_status_cb, thread));
}

void fct_rm_src_dir_cb(SshFileClientError error,
                       SshFileCopyFile file,
                       const char *error_msg, SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyTransferContext);

  if (error == SSH_FX_OK)
    {
      SSH_DEBUG(2, ("Successfully removed source directory ``%s''",
                    ssh_file_copy_file_get_name(file)));
    }
  else
    {
      /* For future changes; we're already going to this state. */
      SSH_FSM_SET_NEXT(transfer_finish_thread);
      FCC_ERROR(fcc_fx_err_to_fc_err(error),
                ("rm_src_dir: %s", error_msg));
    }
}

SSH_FSM_STEP(transfer_rm_src_dir)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  if (!gdata->attrs->unlink_source)
    return SSH_FSM_FINISH;

  FCC_START_STATUS(tdata, tdata->cur_loc, TRUE, fct_rm_src_dir_cb);
  SSH_FSM_SET_NEXT(transfer_finish_thread);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_rmdir
                     (gdata->src_conn->client, tdata->src_file_full_path,
                      transfer_status_cb, thread));
}

/************************************************************
 * Operations on regular files.
 ************************************************************/

void fct_overwrite_completion_cb(Boolean overwrite, void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  SSH_PRECOND(thread);
  
  SSH_DEBUG(2, ("Got result for overwrite: %s.",
                overwrite ? "TRUE" : "FALSE"));
  if (!overwrite)
    {
      SSH_DEBUG(1, ("This file should not be overwritten. Skipping."));
      SSH_FSM_SET_NEXT(transfer_finish_thread);
    }
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_FSM_STEP(transfer_check_overwrite)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  SshFileAttributes attrs = ssh_file_copy_file_get_attributes(tdata->cur_dest);

  SSH_FSM_SET_NEXT(transfer_check_cmp_sizes);
  








  if (attrs != NULL &&
      (attrs->flags & SSH_FILEXFER_ATTR_PERMISSIONS) &&
      ((attrs->permissions & S_IFMT) == S_IFREG) &&
      (gdata->attrs->overwrite_cb != NULL_FNPTR))
    {
      SSH_DEBUG(2, ("Querying overwrite (dest file: '%s')...",
                    ssh_file_copy_file_get_name(tdata->cur_dest)));
      SSH_FSM_ASYNC_CALL((*gdata->attrs->overwrite_cb)
                         (tdata->cur_loc, tdata->cur_dest,
                          fct_overwrite_completion_cb, thread,
                          gdata->attrs->context));
    }
  
  return SSH_FSM_CONTINUE;
}

void fct_remove_dest_cb(SshFileClientError error, SshFileCopyFile file,
                        const char *error_msg, SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyTransferContext);

  if (error != SSH_FX_OK)
    {
      FCC_ERROR(fcc_fx_err_to_fc_err(error), ("open: %s", error_msg));
    }
}

SSH_FSM_STEP(transfer_check_cmp_sizes)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  SshFileAttributes dest_attrs =
    ssh_file_copy_file_get_attributes(tdata->cur_dest);
  SshFileAttributes src_attrs =
    ssh_file_copy_file_get_attributes(tdata->cur_loc);

  SSH_ASSERT(src_attrs);
  
  FCC_START_STATUS(tdata, tdata->cur_dest, FALSE, fct_remove_dest_cb);
  SSH_FSM_SET_NEXT(transfer_open_dest);

  if (dest_attrs == NULL ||
      (((dest_attrs->flags & SSH_FILEXFER_ATTR_SIZE) &&
        (src_attrs->flags & SSH_FILEXFER_ATTR_SIZE)) &&
       src_attrs->size >= dest_attrs->size))
    return SSH_FSM_CONTINUE;

  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_remove
                     (gdata->dest_conn->client,
                      ssh_file_copy_file_get_name(tdata->cur_dest),
                      transfer_status_cb, thread));
}

void fct_open_cb(SshFileClientError error, SshFileCopyFile file,
                 SshFileHandle handle, const char *error_msg,
                 SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyTransferContext);

  if (error == SSH_FX_OK)
    {
      /* Store handle */
      SSH_DEBUG(4, ("File `%s' opened successfully.",
                    ssh_file_copy_file_get_name(file)));
      file->handle = handle;
    }
  else
    {
      /* Drop file, and report error. */
      SSH_FSM_SET_NEXT(transfer_finish_thread);
      FCC_ERROR(fcc_fx_err_to_fc_err(error), ("open: %s", error_msg));
    }
}

SSH_FSM_STEP(transfer_open_dest)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  SSH_DEBUG(4, ("Opening dest file %s...",
                ssh_file_copy_file_get_name(tdata->cur_dest)));

  FCC_START_HANDLE(tdata, tdata->cur_dest, FALSE, fct_open_cb);
  SSH_FSM_SET_NEXT(transfer_open_src);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_open
                     (gdata->dest_conn->client,
                      ssh_file_copy_file_get_name(tdata->cur_dest),
                      O_RDWR | O_CREAT /* XXX | O_TRUNC */,
                      NULL, transfer_handle_cb, thread));
}

SSH_FSM_STEP(transfer_open_src)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  
  SSH_DEBUG(4, ("Opening source file %s...", tdata->src_file_full_path));

  FCC_START_HANDLE(tdata, tdata->cur_loc, TRUE, fct_open_cb);
  SSH_FSM_SET_NEXT(transfer_chmod_dest);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_open
                     (gdata->src_conn->client, tdata->src_file_full_path,
                      O_RDONLY, NULL, transfer_handle_cb, thread));
}

void fct_chmod_dest(SshFileClientError error,
                    SshFileCopyFile file,
                    const char *error_msg, SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyTransferContext);

  if (error == SSH_FX_FAILURE &&
      ssh_file_client_get_version(gdata->dest_conn->client) <= 0)
    {
      /* If the server is of older breed, these warnings are probably
         bogus. (atleast if the server doesn't have futimes function,
         but has utimes function) */
      return;
    }
  if (error != SSH_FX_OK)
    {
      FCC_ERROR(fcc_fx_err_to_fc_err(error),
                ("chmod_dest_before_transfer: %s", error_msg));
    }
}

SSH_FSM_STEP(transfer_chmod_dest)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  SshFileAttributesStruct attrs;
  SshFileAttributes dest_attrs, src_attrs;
  
  memset(&attrs, 0, sizeof(attrs));

  dest_attrs = ssh_file_copy_file_get_attributes(tdata->cur_dest);
  src_attrs = ssh_file_copy_file_get_attributes(tdata->cur_loc);

  if (dest_attrs == NULL)
    {
      tdata->after_stat_state = transfer_chmod_dest;
      SSH_FSM_SET_NEXT(transfer_stat_dest_after_creation);
      return SSH_FSM_CONTINUE;
    }
  
  SSH_ASSERT(dest_attrs != NULL);
  SSH_ASSERT(src_attrs != NULL);

  /* These are here because of the SSH_FSM_CONTINUE in the following
     if-clauses. */
  FCC_START_STATUS(tdata, tdata->cur_dest, FALSE, fct_chmod_dest);




  SSH_FSM_SET_NEXT(transfer_core_start);

  
  if ((dest_attrs->permissions & S_IFMT) != S_IFREG)
    {
      SSH_DEBUG(2, ("Destination file is not a regular file, not changing "
                    "attributes."));
      return SSH_FSM_CONTINUE;
    }


  if (src_attrs->flags & SSH_FILEXFER_ATTR_PERMISSIONS)
    {
      if (gdata->attrs->preserve_attributes)
        {
          SSH_DEBUG(4, ("Setting permissions."));
          attrs.permissions = src_attrs->permissions;
          attrs.flags |= SSH_FILEXFER_ATTR_PERMISSIONS;
        }
      else
        {
          /* This has to be done anyway, so if these operations are not
             supported, it still valid to tell that to the user. */
          if (src_attrs->permissions & 0111)
            {
              /* Umask. */
              attrs.permissions = dest_attrs->permissions;

              /* exec-bit. */
              attrs.permissions |= (src_attrs->permissions & 0111);

              attrs.flags |= SSH_FILEXFER_ATTR_PERMISSIONS;
            }
          else
            {
              /* The source file doesn't have exec bits set, so we don't
                 have to touch it. */
              return SSH_FSM_CONTINUE;
            }
        }
    }









  
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_fsetstat
                     (tdata->cur_dest->handle, &attrs,
                      transfer_status_cb, thread));
}



















































































































SSH_FSM_STEP(transfer_core_start)
{
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  TransferThreadContext ct_ctx;
  
  ct_ctx = ssh_xcalloc(1, sizeof(*ct_ctx));
  *ct_ctx = *tdata;
  ct_ctx->parent = thread;
  
  tdata->child = ssh_fsm_thread_create(fsm,
                                       transfer_start,
                                       NULL_FNPTR,
                                       transfer_core_helper_destructor,
                                       ct_ctx);
  SSH_ASSERT(tdata->child != NULL);

  SSH_FSM_SET_NEXT(transfer_set_attrs);
  
  SSH_FSM_WAIT_THREAD(tdata->child);
}

SSH_FSM_STEP(transfer_set_attrs)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  SshFileAttributesStruct attrs;
  SshFileAttributes dest_attrs, src_attrs;

  memset(&attrs, 0, sizeof(attrs));

  dest_attrs = ssh_file_copy_file_get_attributes(tdata->cur_dest);
  src_attrs = ssh_file_copy_file_get_attributes(tdata->cur_loc);
  
  SSH_ASSERT(dest_attrs != NULL);
  SSH_ASSERT(src_attrs != NULL);

  /* These are here because of the SSH_FSM_CONTINUEs in the following
     if-clauses. */
  FCC_START_STATUS(tdata, tdata->cur_dest, FALSE, fct_chmod_dest);
  SSH_FSM_SET_NEXT(transfer_close_src);

  if ((dest_attrs->permissions & S_IFMT) != S_IFREG)
    {
      SSH_DEBUG(2, ("Destination file is not a regular file, not changing "
                    "attributes."));
      return SSH_FSM_CONTINUE;
    }

  if ((gdata->attrs->preserve_attributes &&
       !(src_attrs->flags & SSH_FILEXFER_ATTR_ACMODTIME)) &&
      !(dest_attrs->flags & SSH_FILEXFER_ATTR_SIZE))
    {
      SSH_DEBUG(2, ("Source file attributes do not contain times and dest file "
                    "attrs don't contain size, skipping."));
      return SSH_FSM_CONTINUE;
    }
  
  if (gdata->attrs->preserve_attributes &&
      src_attrs->flags & SSH_FILEXFER_ATTR_ACMODTIME)
    {
      SSH_DEBUG(4, ("Setting modification time."));
      attrs.mtime = src_attrs->mtime;
      attrs.atime = src_attrs->atime;
      attrs.flags |= SSH_FILEXFER_ATTR_ACMODTIME;
    }
  /* These are set to the correct (new) values in sshfc_trcore.c . */
  if (dest_attrs->flags & SSH_FILEXFER_ATTR_SIZE)
    {
      SSH_DEBUG(4, ("Truncating."));
      attrs.size = dest_attrs->size;
      attrs.flags |= SSH_FILEXFER_ATTR_SIZE;
    }
  
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_fsetstat
                     (tdata->cur_dest->handle, &attrs,
                      transfer_status_cb, thread));
}

SSH_FSM_STEP(transfer_close_src)
{
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  SSH_TRACE(3, ("Closing source file ``%s''...",
                ssh_file_copy_file_get_name(tdata->cur_loc)));

  SSH_FSM_SET_NEXT(transfer_rm_src);

  ssh_file_client_close(tdata->cur_loc->handle, NULL_FNPTR, NULL);
  tdata->cur_loc->handle = NULL;
  return SSH_FSM_CONTINUE;
}

void fct_rm_src_cb(SshFileClientError error,
                   SshFileCopyFile file,
                   const char *error_msg, SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyTransferContext);
  
  if (error == SSH_FX_OK)
    SSH_DEBUG(3, ("Source file '%s' removed successfully.",
                  ssh_file_copy_file_get_name(file)));
  else
    FCC_ERROR(fcc_fx_err_to_fc_err(error), ("rm_src: %s", error_msg));
}

SSH_FSM_STEP(transfer_rm_src)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  FCC_START_STATUS(tdata, tdata->cur_loc, TRUE, fct_rm_src_cb);
  SSH_FSM_SET_NEXT(transfer_close_dest);

  if (!gdata->attrs->unlink_source)
    {
      return SSH_FSM_CONTINUE;
    }

  SSH_TRACE(2, ("Removing source file ``%s''...", tdata->src_file_full_path));

  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_remove
                     (gdata->src_conn->client, tdata->src_file_full_path,
                      transfer_status_cb, thread));
}

SSH_FSM_STEP(transfer_close_dest)
{
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  SSH_TRACE(3, ("Closing dest file %s...",
                ssh_file_copy_file_get_name
                (tdata->cur_dest)));

  ssh_file_client_close(tdata->cur_dest->handle, NULL_FNPTR, NULL);

  tdata->cur_dest->handle = NULL;

  SSH_TRACE(1, ("Finished with file ``%s''.",
                ssh_file_copy_file_get_name(tdata->cur_dest)));

  return SSH_FSM_FINISH;
}

typedef struct CompletionTimeoutCtxRec
{
  SshFileCopyTransferReadyCB completion_callback;
  void *callback_context;
} *CompletionTimeoutCtx;

/* The calling of the completion callback from the bottom of the event
   loop is a fix for a persistent bug in Windows. */
void completion_timeout(void *context)
{
  CompletionTimeoutCtx ctx = (CompletionTimeoutCtx) context;

  SSH_PRECOND(ctx != NULL);

  (*ctx->completion_callback)(SSH_FC_OK, "", ctx->callback_context);

  memset(ctx, 'F', sizeof(*ctx));
  ssh_xfree(ctx);
}

SSH_FSM_STEP(transfer_done)
{
  CompletionTimeoutCtx completion_timeout_ctx = NULL;
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  
  completion_timeout_ctx = ssh_xcalloc(1, sizeof(*completion_timeout_ctx));
  completion_timeout_ctx->completion_callback = gdata->completion_callback;
  completion_timeout_ctx->callback_context = gdata->callback_context;

  ssh_register_timeout(0L, 0L, completion_timeout, completion_timeout_ctx);

  SSH_FSM_SET_NEXT(transfer_cleanup);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(transfer_abort)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  /* Abort the transfer. */

  SSH_TRACE(2, ("Aborting transfer..."));

  ssh_cancel_timeouts(progress_timeout_cb, gdata);

  gdata->operation_handle = NULL;

  SSH_FSM_SET_NEXT(transfer_cleanup);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(transfer_cleanup)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  if (tdata->child)
    {
      TransferThreadContext child_tdata;
      child_tdata = ssh_fsm_get_tdata(tdata->child);
      child_tdata->parent = NULL;
      ssh_fsm_kill_thread(tdata->child);
      tdata->child = NULL;
    }

  if (gdata->reader)
    {
      ssh_fsm_kill_thread(gdata->reader);
      gdata->reader = NULL;
    }

  if (gdata->writer)
    {
      ssh_fsm_kill_thread(gdata->writer);
      gdata->writer = NULL;
    }

  if (gdata->mangler)
    {
      ssh_fsm_kill_thread(gdata->mangler);
      gdata->mangler = NULL;
    }

  ssh_fsm_destroy(gdata->fsm);

  if (gdata->operation_handle)
    ssh_operation_unregister(gdata->operation_handle);

  if (gdata->in_blocking)
    {
      /* Everybody should be killed already, this is just for sanity. */
      SSH_FSM_CONDITION_BROADCAST(gdata->in_blocking);
      ssh_fsm_condition_destroy(gdata->in_blocking);
      gdata->in_blocking = NULL;
    }
  if (gdata->out_blocking)
    {
      /* Everybody should be killed already, this is just for sanity. */
      SSH_FSM_CONDITION_BROADCAST(gdata->out_blocking);
      ssh_fsm_condition_destroy(gdata->out_blocking);
      gdata->out_blocking = NULL;
    }
  if (gdata->mangle_blocking)
    {
      /* Everybody should be killed already, this is just for sanity. */
      SSH_FSM_CONDITION_BROADCAST(gdata->mangle_blocking);
      ssh_fsm_condition_destroy(gdata->mangle_blocking);
      gdata->mangle_blocking = NULL;
    }

  if (gdata->buf_queue)
    ssh_adt_destroy(gdata->buf_queue);

  ssh_time_measure_free(gdata->transfer_timer);
  ssh_time_measure_free(gdata->total_transfer_timer);

  ssh_xfree(gdata->src_newline);
  ssh_xfree(gdata->dest_newline);
  ssh_xfree(gdata);

  return SSH_FSM_FINISH;
}

SshFSMStateDebugStruct transfer_states_array[] =
{
  { "fct_check_src_conn", "Check source connection", transfer_check_src_conn },
  { "fct_check_dest_conn", "Check destination connection",
    transfer_check_dest_conn },
  { "fct_conn_failed", "Connection has failed", transfer_conn_failed },

  { "fct_stat_dest_loc", "Stat destination location.",
    transfer_stat_dest_loc },
  { "fct_stat_src", "Stat source file", transfer_stat_src },
  { "fct_stat_dest", "Stat destination file", transfer_stat_dest },
  { "fct_rm_src", "Remove source file", transfer_rm_src },
  { "fct_open_dest", "Open destination file", transfer_open_dest },
  { "fct_open_src", "Open source file", transfer_open_src },
  { "fct_chmod_dest", "Change permissions of dest file", transfer_chmod_dest },
  { "fct_set_attrs", "Truncate and change access and modification times",
    transfer_set_attrs },
  
#include "sshfc_trcore_states.h"

  { "fct_mkdir", "Make a new directory", transfer_mkdir },
  { "fct_chmod_dest_dir",
    "Change permissions and fileattributes of destination directory",
    transfer_chmod_dest_dir },
  { "fct_rm_src_dir", "Remove source directory", transfer_rm_src_dir },

  { "fct_walk_file_list", "Get next file in file list",
    transfer_walk_file_list },
  { "fct_prepare_base_dir", "Prepare base directory",
    transfer_prepare_base_dir },
  { "fct_start_file_ops", "Start operations on file",
    transfer_start_file_ops },
  { "fct_walk_tree", "Get next file in directory entry list",
    transfer_walk_tree },
  { "fct_dir_descent", "Go down on directory hierarchy",
    transfer_dir_descent },
  { "fct_core_start", "Start the actual transfer", transfer_core_start },
  { "fct_close_src", "Close source file", transfer_close_src },
  { "fct_close_dest", "Close destination file", transfer_close_dest },
  { "fct_done", "Transfer done", transfer_done },
  { "fct_cleanup", "Do cleanup", transfer_cleanup },

  { "fct_finish_thread", "Finish this thread", transfer_finish_thread },
  { "fct_abort", "Abort file transfer exchange", transfer_abort }
};

static int file_counter(SshFileCopyFile file)
{
  int ret = 0;
  SshADTHandle h;
  SSH_PRECOND(file);  

  ret++;

  if (file->dir_entries)
    {    
      for (h = ssh_adt_enumerate_start(file->dir_entries);
           h != SSH_ADT_INVALID;
           h = ssh_adt_enumerate_next(file->dir_entries, h))
        ret += file_counter(ssh_adt_get(file->dir_entries, h));
    }
  
  return ret;
}

int transfer_count_files(SshADTContainer c)
{
  SshADTHandle h;
  int count = 0;
  
  for (h = ssh_adt_enumerate_start(c);
       h != SSH_ADT_INVALID;
       h = ssh_adt_enumerate_next(c, h))
    {
      SshFileCopyFile file = (SshFileCopyFile)ssh_adt_get(c, h);
      SSH_ASSERT(file);
      count += file_counter(file);

      /* basedirs of files with dir_entries should not be counted in. */
      if (file->dir_entries)
        {
          count --;
        }
    }

  return count;
}

void transfer_operation_abort_cb(void *operation_context)
{
  SshFileCopyTransferContext gdata =
    (SshFileCopyTransferContext) operation_context;
  TransferThreadContext tdata;

  if (gdata->main_thread == NULL)
    return;
  
  tdata = (TransferThreadContext)ssh_fsm_get_tdata(gdata->main_thread);

  ssh_operation_abort(tdata->op_handle);
  tdata->op_handle = NULL;
  
  ssh_fsm_set_next(gdata->main_thread, transfer_abort);
  ssh_fsm_continue(gdata->main_thread);
}

SshOperationHandle
ssh_file_copy_transfer_multiple(SshFileCopyConnection source,
                                SshADTContainer file_list,
                                SshFileCopyConnection destination,
                                const char *dest_path,
                                SshFileCopyTransferAttrs attrs,
                                SshFileCopyTransferReadyCB ready_cb,
                                SshFileCopyErrorCB error_cb,
                                void *context)
{
  SshFileCopyTransferContext transfer_context;
  TransferThreadContext child_tdata;
  int count;

  SSH_PRECOND(source != NULL);
  SSH_PRECOND(file_list != NULL);
  SSH_PRECOND(destination != NULL);
  SSH_PRECOND(dest_path != NULL);
  SSH_PRECOND(ready_cb != NULL_FNPTR);
  SSH_PRECOND(error_cb != NULL_FNPTR);

  /* Count files in file_list (if more than one, destination must
     be directory). */
  count = transfer_count_files(file_list);


  SSH_DEBUG(2, ("File list has %d files.", count));

  if (count <= 0)
    {
      (*ready_cb)(SSH_FC_OK, "No files to transfer.", context);
      return NULL;
    }

  if (attrs == NULL)
    {
      /* Use default attrs. */
      attrs = ssh_xcalloc(1, sizeof(*attrs));
    }

  if (count > 1)
    attrs->destination_must_be_dir = TRUE;

  transfer_context = ssh_xcalloc(1, sizeof(*transfer_context));
  transfer_context->fsm = ssh_fsm_create(transfer_context);
  SSH_VERIFY(transfer_context->fsm != NULL);
  ssh_fsm_register_debug_names(transfer_context->fsm, transfer_states_array,
                               SSH_FSM_NUM_STATES(transfer_states_array));
                               
  transfer_context->src_conn = source;
  transfer_context->file_list = file_list;
  transfer_context->dest_conn = destination;
  transfer_context->dest_loc = ssh_file_copy_file_allocate();
  ssh_file_copy_file_register_filename(transfer_context->dest_loc,
                                       ssh_xstrdup(dest_path));
  transfer_context->attrs = attrs;
  transfer_context->completion_callback = ready_cb;
  transfer_context->error_callback = error_cb;
  transfer_context->callback_context = context;

  if (attrs->transfer_mode & (SSH_FC_TRANSFER_MODE_ASCII |
                              SSH_FC_TRANSFER_MODE_AUTO))
    {
      if (attrs->transfer_mode & SSH_FC_TRANSFER_MODE_AUTO)
        SSH_ASSERT(attrs->ascii_extensions != NULL);
      if (!(attrs->transfer_mode & SSH_FC_TRANSFER_MODE_NL_CONV_ASK))
        {
          if (attrs->source_newline == NULL)
            SSH_DEBUG(2, ("ASCII mangling selected, source newline not set, "
                          "determining at runtime."));
          if (attrs->dest_newline == NULL)
            SSH_DEBUG(2, ("ASCII mangling selected, dest newline not set, "
                          "determining at runtime."));
        }
      else
        {
          SSH_ASSERT(attrs->newline_query_cb != NULL_FNPTR);
        }
    }






  
  child_tdata = ssh_xcalloc(1, sizeof(*child_tdata));
  child_tdata->h = ssh_adt_enumerate_start(file_list);
  
  transfer_context->main_thread =
    ssh_fsm_thread_create(transfer_context->fsm,
                          transfer_check_src_conn,
                          NULL_FNPTR,
                          transfer_thread_destructor,
                          child_tdata);
  SSH_VERIFY(transfer_context->main_thread != NULL);
  
  ssh_fsm_set_thread_name(transfer_context->main_thread, "main_thread");
  transfer_context->transfer_timer = ssh_time_measure_allocate();
  transfer_context->total_transfer_timer = ssh_time_measure_allocate();

  transfer_context->operation_handle =
    ssh_operation_register(transfer_operation_abort_cb, transfer_context);

  return transfer_context->operation_handle;
}
