/*
  sshfc_recurse.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 2000-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Functions that perform the recursing of subdirectories.
 */

#include "sshincludes.h"
#include "sshfilecopy.h"
#include "sshfilecopyi.h"

#define SSH_DEBUG_MODULE "SshFCRecurse"

/**********
 * ssh_file_copy_recurse_dirs() related stuff.
 **********/

#define SSH_FCR_MAX_SYMLINK_LEVEL_DEPTH 64

/* Global data. */
typedef struct SshFileCopyRecurseContextRec
{
  SshFSM fsm;

  SshFileCopyErrorCB error_callback;
  SshFileCopyRecurseReadyCB completion_callback;
  void *callback_context;

  SshFSMThread main_thread;

  /* File list given by application. */
  SshADTContainer file_list;

  /* Optional function to be executed on files, given by
     application. */
  SshFileCopyRecurseFileFunc func;

  /* Recurse attrs, given by application. */
  SshFileCopyRecurseAttrs attrs;

  /* New file list to be returned to application. */
  SshADTContainer new_file_list;

  /* New directory, which we are currently recursing. */
  SshFileCopyFile new_dir;

  /* Helper variable, used to retain pointer to a dynamically
     malloced string, which is used to lstat a file. The string is
     freed in a callback. */
  char *filename_to_be_statted;

  /* Connection handling. */
  SshFileCopyConnection source;
  SshFileCopyFile location;
  
  /* Error message given to application, when fatal error encountered. */
  char *error_message;

} *SshFileCopyRecurseContext;

typedef struct RecurseThreadContextRec *RecurseThreadContext,
  RecurseThreadContextStruct;

struct RecurseThreadContextRec
{
  /* Parent thread, that will be waken when child is ready. */
  SshFSMThread parent;
  /* Child thread of the thread. */
  SshFSMThread child;

  /* The dir we are currently working on. */
  SshFileCopyFile current_dir;

  SshFileCopyFile parent_dir;
  
  SshFileCopyFile cur_loc;
  /* If TRUE, the thread's destructor will destroy ``cur_loc''. */
  Boolean cur_loc_allocated;
  
  /* Current file list, where this thread should add it's files. */
  SshADTContainer list_to_add;
  SshADTHandle h;
  
  char *cur_file_fullpath;

  RecurseThreadContext readdir_child_ctx;
  
  /* File, which we got from the last call to readdir. */
  SshFileCopyFile new_file;

  /* The count of symlinks we have traversed (in depth). */
  int symlink_level;

  Boolean cur_file_is_symlink;
  
  /* For unified callback handling macros. */
  Boolean dealing_with_source;
  
  SshOperationHandle op_handle;
  SshFSMStepCB current_state;
  SshFileCopyFile current_file;
  FCCommonStatusCB status_cb;
  FCCommonHandleCB handle_cb;
  FCCommonDataCB data_cb;
  FCCommonNameCB name_cb;
  FCCommonAttributeCB attrs_cb;
};

/* Main thread. */
void recurse_child_thread_destructor(SshFSM fsm, void *tdata)
{
  RecurseThreadContext thread_context = (RecurseThreadContext) tdata;

  SSH_PRECOND(thread_context != NULL);

  if (thread_context->op_handle)
    {
      ssh_operation_abort(thread_context->op_handle);
      thread_context->op_handle = NULL;
    }
  if (thread_context->parent)
    {
      RecurseThreadContext parent_tdata;
      parent_tdata = ssh_fsm_get_tdata(thread_context->parent);
      parent_tdata->child = NULL;
    }

  if (thread_context->child)
    {
      RecurseThreadContext child_tdata;
      child_tdata = ssh_fsm_get_tdata(thread_context->child);
      child_tdata->parent = NULL;
      ssh_fsm_kill_thread(thread_context->child);
      thread_context->child = NULL;
    }

  if (thread_context->cur_loc_allocated && thread_context->cur_loc)
    {
      /* The dir_entries are not copied, but are part of the file list
         given to us. Therefore, we shouldn't touch it. */
      thread_context->cur_loc->dir_entries = NULL;
      ssh_file_copy_file_destroy(thread_context->cur_loc);
    }
  
  
  if (thread_context->current_dir)
    ssh_file_copy_file_destroy(thread_context->current_dir);

  if (thread_context->new_file)
    ssh_file_copy_file_destroy(thread_context->new_file);

  ssh_xfree(thread_context);
}

/* Forward declarations. */
SSH_FSM_STEP(recurse_check_conn);
SSH_FSM_STEP(recurse_walk_file_list);
SSH_FSM_STEP(recurse_process_location);
SSH_FSM_STEP(recurse_parse_raw);
SSH_FSM_STEP(recurse_walk_tree);
SSH_FSM_STEP(recurse_opendir);
SSH_FSM_STEP(recurse_readdir);
SSH_FSM_STEP(recurse_lstat_file);
SSH_FSM_STEP(recurse_stat_file);
SSH_FSM_STEP(recurse_finish_thread);
SSH_FSM_STEP(recurse_done);
SSH_FSM_STEP(recurse_cleanup);

#define FCC_CLEANUP(error, message)                             \
do                                                              \
{                                                               \
  (*gdata->completion_callback)((error), (message), NULL,       \
                                gdata->callback_context);       \
  /* stop transfer and abort */                                 \
  ssh_fsm_set_next(gdata->main_thread, recurse_cleanup);        \
  ssh_fsm_continue(gdata->main_thread);                         \
}                                                               \
while (0)

FCC_GEN_BF_CALLBACK(fcr, SshFileCopyRecurseContext, RecurseThreadContext)
FCC_GEN_HANDLE(fcr)
FCC_GEN_NAME(fcr)
FCC_GEN_ATTR(fcr)

void fcr_conn_completion_cb(SshFileClient client, void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  SshFileCopyRecurseContext gdata =
    (SshFileCopyRecurseContext)ssh_fsm_get_gdata(thread);
  RecurseThreadContext tdata = (RecurseThreadContext)ssh_fsm_get_tdata(thread);

  tdata->op_handle = NULL;

  if (!client)
    {
      (*gdata->completion_callback)(SSH_FC_ERROR_CONNECTION_FAILED,
                                    "Connecting to source failed",
                                    NULL, gdata->callback_context);
      ssh_fsm_set_next(gdata->main_thread, recurse_cleanup);
    }
  gdata->source->client = client;
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_FSM_STEP(recurse_check_conn)
{
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;

  SSH_FSM_SET_NEXT(recurse_walk_file_list);
  
  if (gdata->source->client)
    {
      SSH_DEBUG(4, ("Source connection OK."));
      return SSH_FSM_CONTINUE;
    }

  SSH_DEBUG(2, ("Establishing source connection."));
  SSH_FSM_ASYNC_CALL(tdata->op_handle =
                     ssh_file_copy_connect(gdata->source,
                                           fcr_conn_completion_cb,
                                           thread));

}

/* This is very similar to recurse_walk_tree */
SSH_FSM_STEP(recurse_walk_file_list)
{
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;

  if (tdata->h != SSH_ADT_INVALID)
    {
      SshFSMThread ct;
      RecurseThreadContext ct_ctx;
      
      /* Spawn child to process item. */
      ct_ctx = ssh_xcalloc(1, sizeof(*ct_ctx));
      ct_ctx->list_to_add = gdata->new_file_list;
      ct_ctx->cur_loc =
        (SshFileCopyFile) ssh_adt_get(gdata->file_list, tdata->h);
      SSH_ASSERT(ct_ctx->cur_loc != NULL);
      ct = ssh_fsm_thread_create(fsm, recurse_process_location, NULL_FNPTR,
                                 recurse_child_thread_destructor, ct_ctx);
      SSH_ASSERT(ct != NULL);
      tdata->h = ssh_adt_enumerate_next(gdata->file_list, tdata->h);
      
      SSH_FSM_WAIT_THREAD(ct);
    }

  SSH_DEBUG(3, ("No more files in this list."));
  SSH_FSM_SET_NEXT(recurse_done);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(recurse_process_location)
{
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;
  SshFileAttributes attrs;
  SshFileCopyFile new_file = NULL;
  SshADTHandle h;
  SshUInt32 ret_val = 0L;
  
  SSH_PRECOND(tdata->list_to_add);
  
  attrs = ssh_file_copy_file_get_attributes(tdata->cur_loc);

  SSH_DEBUG(4, ("Processing file ``%s''...",
                ssh_file_copy_file_get_name(tdata->cur_loc)));
  
  if (attrs == NULL)
    {
      SSH_DEBUG(2, ("File ``%s'' is \"raw\", and it needs to be parsed.",
                    ssh_file_copy_file_get_name(tdata->cur_loc)));
      SSH_FSM_SET_NEXT(recurse_parse_raw);
      return SSH_FSM_CONTINUE;
    }

  SSH_ASSERT(attrs != NULL);

  if (!(attrs->flags & SSH_FILEXFER_ATTR_PERMISSIONS))
    {
      SSH_DEBUG(3, ("File %s needs to be statted.",
                    ssh_file_copy_file_get_name(tdata->cur_loc)));

      SSH_FSM_SET_NEXT(recurse_lstat_file);
      return SSH_FSM_CONTINUE;
    }

  /* symlinks. */
  if (tdata->cur_file_is_symlink)
    {
      tdata->cur_file_is_symlink = FALSE;
      tdata->symlink_level++;

      if (tdata->symlink_level > SSH_FCR_MAX_SYMLINK_LEVEL_DEPTH)
        {
          FCC_ERROR(SSH_FC_ERROR_ELOOP,
                    ("fcr_process: %s: Max symlink count for depth exceeded "
                     "(we have very probably encountered a symlink loop. See "
                     "man-page for scp2).",
                     ssh_file_copy_file_get_name(tdata->cur_loc)));
          return SSH_FSM_FINISH;
        }
    }
      
  if ((attrs->permissions & S_IFMT) == S_IFLNK &&
      (gdata->attrs->follow_symlinks ||
       /* If this is the topmost file, we want to follow the symlink. */
       !tdata->parent_dir))
    {
      /* Check whether symlink points to a directory. */
      tdata->cur_file_is_symlink = TRUE;
      SSH_FSM_SET_NEXT(recurse_stat_file);
      return SSH_FSM_CONTINUE;
    }
  
  if (gdata->func)
    {
      char *temp_filename =
        ssh_file_copy_file_generate_full_path(tdata->cur_loc);

      ret_val =
        (*gdata->func)(temp_filename,
                       ssh_file_copy_file_get_long_name(tdata->cur_loc),
                       ssh_file_copy_file_get_attributes(tdata->cur_loc),
                       gdata->callback_context);
      
      if (ret_val == 0x0)
        {
          SSH_TRACE(7, ("RecurseFileFunc returned 0 (zero), ignoring `%s'.",
                        temp_filename));
          ssh_xfree(temp_filename);
          return SSH_FSM_FINISH;
        }
      ssh_xfree(temp_filename);
    }
  else
    {
      /* If no func is defined, default action is to recurse and add
         file. */
      ret_val |= SSH_FCR_ADD;
      ret_val |= SSH_FCR_RECURSE;
    }

  if (ret_val & SSH_FCR_RECURSE)
    /* If you want to recurse it, you _have_ to add it. */
    SSH_ASSERT(ret_val & SSH_FCR_ADD);
  
  /* Currently, if we've managed this far, also the following if
     statement will always be TRUE. Just preparing for the future. */
  if (ret_val & SSH_FCR_ADD)
    {
      new_file = ssh_file_copy_file_dup(tdata->cur_loc);
      new_file->parent_dir = tdata->parent_dir;

      h = ssh_adt_insert_to(tdata->list_to_add, SSH_ADT_END, new_file);
      SSH_ASSERT(h != SSH_ADT_INVALID);
    }
  
  if (ret_val & SSH_FCR_RECURSE &&
      (tdata->cur_loc->dir_entries != NULL ||
       (attrs->permissions & S_IFMT) == S_IFDIR))
    {
      SshFSMStepCB first_state = NULL_FNPTR;
      SshFSMThread ct;
      RecurseThreadContext ct_ctx;

      SSH_ASSERT(new_file != NULL);
      
      /* Spawn thread to recurse through subdirectory. */

      ct_ctx = ssh_xcalloc(1, sizeof(*ct_ctx));
      ct_ctx->cur_loc = new_file;
      ct_ctx->parent_dir = tdata->parent_dir;
      
      if ((attrs->permissions & S_IFMT) == S_IFDIR &&
          /* If directory has dir_entries, then it has been
             globbed/whatever and it shouldn't be recursed again. */
          !tdata->cur_loc->dir_entries)
        {
          SSH_DEBUG(7, ("File ``%s'' is a directory, descending...",
                        ssh_file_copy_file_get_name(tdata->cur_loc)));
          first_state = recurse_opendir;
        }
      else
        {
          SSH_DEBUG(7, ("File ``%s'' has directory entries, descending...",
                        ssh_file_copy_file_get_name(tdata->cur_loc)));
          SSH_ASSERT(tdata->cur_loc->dir_entries != NULL);
          ct_ctx->h = ssh_adt_enumerate_start(tdata->cur_loc->dir_entries);
          first_state = recurse_walk_tree;
        }

      new_file->dir_entries =
        ssh_file_copy_file_list_create(gdata->attrs->compare_func,
                                       gdata->attrs->compare_ctx);
      ct_ctx->list_to_add = new_file->dir_entries;

      ct = ssh_fsm_thread_create(fsm, first_state, NULL_FNPTR,
                                 recurse_child_thread_destructor, ct_ctx);
      SSH_ASSERT(ct);
      
      SSH_FSM_SET_NEXT(recurse_finish_thread);
      SSH_FSM_WAIT_THREAD(ct);
    }
  else
    {
      /* File is already added to the list, so we're done with this file. */
      return SSH_FSM_FINISH;
    }
}

SSH_FSM_STEP(recurse_walk_tree)
{
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;

  if (tdata->h != SSH_ADT_INVALID)
    {
      SshFSMThread ct;
      RecurseThreadContext ct_ctx;

      /* Spawn child to process item. */
      ct_ctx = ssh_xcalloc(1, sizeof(*ct_ctx));
      ct_ctx->list_to_add = tdata->list_to_add;

      ct_ctx->parent_dir = tdata->cur_loc;
      ct_ctx->cur_loc =
        (SshFileCopyFile) ssh_adt_get(tdata->cur_loc->dir_entries, tdata->h);
      SSH_ASSERT(ct_ctx->cur_loc != NULL);
      
      tdata->h = ssh_adt_enumerate_next(tdata->cur_loc->dir_entries, tdata->h);
      
      ct = ssh_fsm_thread_create(fsm, recurse_process_location, NULL_FNPTR,
                                 recurse_child_thread_destructor, ct_ctx);
      SSH_ASSERT(ct != NULL);
      
      SSH_FSM_WAIT_THREAD(ct);
    }

  SSH_DEBUG(3, ("No more files in this list."));
  return SSH_FSM_FINISH;
}

void fcr_parse_stat_cb(SshFileClientError error,
                       SshFileCopyFile file,
                       SshFileAttributes attrs,
                       const char *error_msg,
                       SshFSMThread thread)
{
  FCC_DATA(SshFileCopyRecurseContext, RecurseThreadContext);

  ssh_xfree(tdata->cur_file_fullpath);

  if (error == SSH_FX_OK)
    {
      SSH_PRECOND(file != NULL);

      tdata->cur_loc = ssh_file_copy_file_dup(file);
      tdata->cur_loc->dir_entries = file->dir_entries;

      ssh_file_copy_file_register_attributes(tdata->cur_loc,
                                             ssh_file_attributes_dup(attrs));
      
      
      /* Normally, cur_loc is a member from the file list given to
         ssh_file_copy_recurse_dirs() and therefore cannot be freed
         when the thread is destroyed. By enabling this flag, the
         thread's cleanup function knows that it should destroy
         cur_loc. */
      tdata->cur_loc_allocated = TRUE;
    }
  else
    {
      SSH_FSM_SET_NEXT(recurse_finish_thread);
      FCC_ERROR(fcc_fx_err_to_fc_err(error),
                ("fcr_parse_raw: %s", error_msg));
    }
}

SSH_FSM_STEP(recurse_parse_raw)
{
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;

  SSH_PRECOND(tdata->cur_loc != NULL);

  tdata->cur_file_fullpath =
    ssh_file_copy_file_generate_full_path(tdata->cur_loc);
  /* lstat file */
  FCC_START_ATTRS(tdata, tdata->cur_loc, TRUE, fcr_parse_stat_cb);
  SSH_FSM_SET_NEXT(recurse_process_location);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_lstat
                        (gdata->source->client,
                         tdata->cur_file_fullpath,
                         transfer_attribute_cb, thread));
}

void fcr_stat_cb(SshFileClientError error,
                 SshFileCopyFile file,
                 SshFileAttributes attrs,
                 const char *error_msg,
                 SshFSMThread thread)
{
  FCC_DATA(SshFileCopyRecurseContext, RecurseThreadContext);

  ssh_xfree(tdata->cur_file_fullpath);

  if (error == SSH_FX_OK)
    {
      ssh_file_copy_file_register_attributes(file,
                                             ssh_file_attributes_dup(attrs));
    }
  else
    {
      SSH_FSM_SET_NEXT(recurse_finish_thread);
      FCC_ERROR(fcc_fx_err_to_fc_err(error), ("fcr_stat: %s", error_msg));
    }
}

SSH_FSM_STEP(recurse_lstat_file)
{
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;

  tdata->cur_file_fullpath =
    ssh_file_copy_file_generate_full_path(tdata->cur_loc);
  FCC_START_ATTRS(tdata, tdata->cur_loc, TRUE, fcr_stat_cb);
  SSH_FSM_SET_NEXT(recurse_process_location);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_lstat
                     (gdata->source->client, tdata->cur_file_fullpath,
                      transfer_attribute_cb, thread));
}

SSH_FSM_STEP(recurse_stat_file)
{
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;

  tdata->cur_file_fullpath =
    ssh_file_copy_file_generate_full_path(tdata->cur_loc);
  FCC_START_ATTRS(tdata, tdata->cur_loc, TRUE, fcr_stat_cb);
  SSH_FSM_SET_NEXT(recurse_process_location);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_stat
                     (gdata->source->client, tdata->cur_file_fullpath,
                      transfer_attribute_cb, thread));
}

void fcr_opendir_cb(SshFileClientError error,
                    SshFileCopyFile file,
                    SshFileHandle handle, const char *error_msg,
                    SshFSMThread thread)
{
  FCC_DATA(SshFileCopyRecurseContext, RecurseThreadContext);

  ssh_xfree(tdata->cur_file_fullpath);

  if (error == SSH_FX_OK)
    {
      file->handle = handle;
    }
  else
    {
      SSH_FSM_SET_NEXT(recurse_finish_thread);
      FCC_ERROR(fcc_fx_err_to_fc_err(error), ("fcr_opendir: %s", error_msg));
    }
}

SSH_FSM_STEP(recurse_opendir)
{
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;
  SSH_PRECOND(tdata->cur_loc->dir_entries != NULL);
  
  SSH_DEBUG_INDENT;
  
  tdata->cur_file_fullpath =
    ssh_file_copy_file_generate_full_path(tdata->cur_loc);
  
  SSH_TRACE(3, ("Opening directory ``%s''...", tdata->cur_file_fullpath));

  FCC_START_HANDLE(tdata, tdata->cur_loc, TRUE, fcr_opendir_cb);
  SSH_FSM_SET_NEXT(recurse_readdir);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_opendir
                     (gdata->source->client, tdata->cur_file_fullpath,
                      transfer_handle_cb, thread));
}

void fcr_readdir_cb(SshFileClientError error, SshFileCopyFile file,
                    const char *name, const char *long_name,
                    SshFileAttributes attrs, const char *error_msg,
                    SshFSMThread thread)
{
  FCC_DATA(SshFileCopyRecurseContext, RecurseThreadContext);




  if (error == SSH_FX_OK)
    {
      RecurseThreadContext ct_ctx;
      
      if (strcmp(name, ".") == 0 || strcmp(name, "..") == 0)
        {
          SSH_FSM_SET_NEXT(recurse_readdir);
          return;
        }
      ct_ctx = ssh_xcalloc(1, sizeof(*ct_ctx));
      ct_ctx->parent_dir = file;
      ct_ctx->cur_loc = ssh_file_copy_file_allocate();
      ct_ctx->cur_loc->parent_dir = file;
      ct_ctx->list_to_add = tdata->cur_loc->dir_entries;
      ssh_file_copy_file_register_filename(ct_ctx->cur_loc,
                                           ssh_xstrdup(name));

      ssh_file_copy_file_register_long_name(ct_ctx->cur_loc,
                                            ssh_xstrdup(long_name));






      ssh_file_copy_file_register_attributes(ct_ctx->cur_loc,
                                             ssh_file_attributes_dup(attrs));
      tdata->readdir_child_ctx = ct_ctx;
    }
  else if (error == SSH_FX_EOF)
    {
      /* We are at the end of the directory. */
      ssh_file_client_close(tdata->cur_loc->handle, NULL_FNPTR, NULL);
      tdata->cur_loc->handle = NULL;
      SSH_FSM_SET_NEXT(recurse_finish_thread);
    }
  else
    {
      /* Drop current dir. */
      SSH_FSM_SET_NEXT(recurse_finish_thread);
      FCC_ERROR(fcc_fx_err_to_fc_err(error), ("fcr_readdir: %s", error_msg));
    }
}

SSH_FSM_STEP(recurse_readdir)
{
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;

  SSH_PRECOND(tdata->cur_loc != NULL);
  SSH_PRECOND(tdata->cur_loc->handle != NULL);

  if (tdata->readdir_child_ctx)
    {
      SshFSMThread ct;
      ct = ssh_fsm_thread_create(fsm, recurse_process_location,
                                 NULL_FNPTR, recurse_child_thread_destructor,
                                 tdata->readdir_child_ctx);
      tdata->readdir_child_ctx = NULL;
      SSH_FSM_WAIT_THREAD(ct);
    }
  
  FCC_START_NAME(tdata, tdata->cur_loc, TRUE, fcr_readdir_cb);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_readdir
                     (tdata->cur_loc->handle, transfer_name_cb, thread));
}

SSH_FSM_STEP(recurse_finish_thread)
{
  return SSH_FSM_FINISH;
}

SSH_FSM_STEP(recurse_done)
{
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  
  (*gdata->completion_callback)(SSH_FC_OK,
                                "Recursion complete.",
                                gdata->new_file_list,
                                gdata->callback_context);
  ssh_fsm_destroy(gdata->fsm);
  gdata->new_file_list = NULL;
  
  /* XXX What else should be freed ? */
  ssh_xfree(gdata->attrs);
  ssh_xfree(gdata);
  return SSH_FSM_FINISH;
}

SSH_FSM_STEP(recurse_cleanup)
{
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;

  /* XXX testing */
  if (tdata->child)
    {
      RecurseThreadContext child_tdata;
      child_tdata = ssh_fsm_get_tdata(tdata->child);
      child_tdata->parent = NULL;
      ssh_fsm_kill_thread(tdata->child);
      tdata->child = NULL;
    }
  ssh_fsm_destroy(gdata->fsm);
  ssh_adt_destroy(gdata->new_file_list);
  
#if 0
  if (gdata->operation_handle)
    ssh_operation_unregister(gdata->operation_handle);
#endif
  
  ssh_xfree(gdata);

  return SSH_FSM_FINISH;
}

SshFSMStateDebugStruct recurse_states_array[] =
{ { "fcr_check_conn", "Check connection", recurse_check_conn },
  { "fcr_walk_file_list", "Traverse given file list", recurse_walk_file_list },
  { "fcr_process_location", "Process file", recurse_process_location },
  { "fcr_parse_raw", "Parse \"raw\" file", recurse_parse_raw },
  { "fcr_walk_tree", "Go through items in file list", recurse_walk_tree },
  { "fcr_opendir", "Open a directory", recurse_opendir },
  { "fcr_readdir", "Take next directory element", recurse_readdir },
  { "fcr_lstat", "lstat file", recurse_lstat_file },
  { "fcr_stat", "lstat file", recurse_stat_file },
  { "fcr_finish_thread", "Finish this thread", recurse_finish_thread },
  { "fcr_recurse_done", "We're done recursing", recurse_done },
  { "fcr_cleanup", "Clean up", recurse_cleanup },

  /* Child threads check directories "recursively". */
  /* XXX Rest of the states. */

};

void ssh_file_copy_recurse_multiple(SshFileCopyConnection source,
                                    SshADTContainer file_list,
                                    SshFileCopyRecurseAttrs attrs,
                                    SshFileCopyRecurseFileFunc func,
                                    SshFileCopyRecurseReadyCB ready_cb,
                                    SshFileCopyErrorCB error_cb,
                                    void *context)
{
  SshFileCopyRecurseContext recurse_context;
  RecurseThreadContext tdata;

  SSH_PRECOND(source != NULL);
  SSH_PRECOND(file_list != NULL);
  SSH_PRECOND(ready_cb != NULL_FNPTR);
  SSH_PRECOND(error_cb != NULL_FNPTR);

  recurse_context = ssh_xcalloc(1, sizeof(*recurse_context));
  recurse_context->fsm = ssh_fsm_create(recurse_context);
  SSH_VERIFY(recurse_context->fsm != NULL);
  ssh_fsm_register_debug_names(recurse_context->fsm, recurse_states_array,
                               SSH_FSM_NUM_STATES(recurse_states_array));
  recurse_context->completion_callback = ready_cb;
  recurse_context->error_callback = error_cb;
  recurse_context->callback_context = context;
  recurse_context->source = source;
  recurse_context->func = func;

  recurse_context->attrs = ssh_xcalloc(1, sizeof(*attrs));

  if (attrs)
    *recurse_context->attrs = *attrs;
  
  recurse_context->file_list = file_list;

  recurse_context->new_file_list =
    ssh_file_copy_file_list_create(recurse_context->attrs->compare_func,
                                   recurse_context->attrs->compare_ctx);

  tdata = ssh_xcalloc(1, sizeof(*tdata));
  tdata->h = ssh_adt_enumerate_start(recurse_context->file_list);
  
  recurse_context->main_thread = ssh_fsm_thread_create(recurse_context->fsm,
                                                       recurse_check_conn,
                                                       NULL_FNPTR,
                                                       NULL_FNPTR, /* XXX mem
                                                                      leaks? */
                                                       tdata);
  SSH_VERIFY(recurse_context->main_thread != NULL);
  ssh_fsm_set_thread_name(recurse_context->main_thread, "main_thread");
}

typedef struct RecurseHelperCtxRec
{
  SshADTContainer list;
  void *completion_ctx;
  SshFileCopyRecurseFileFunc func;
  SshFileCopyRecurseReadyCB ready_cb;
  SshFileCopyErrorCB error_cb;
} *RecurseHelperCtx, RecurseHelperCtxStruct;

static SshUInt32 fcr_helper_rec_func(const char *filename,
                                     const char *long_name,
                                     SshFileAttributes attrs,
                                     void *context)
{
  RecurseHelperCtx ctx = (RecurseHelperCtx)context;
  return (*ctx->func)(filename, long_name, attrs, ctx->completion_ctx);
}

static void fcr_helper_ready_cb(SshFileCopyError error,
                                const char *error_message,
                                SshADTContainer file_list,
                                void *context)
{
  RecurseHelperCtx ctx = (RecurseHelperCtx)context;
  (*ctx->ready_cb)(error, error_message, file_list, ctx->completion_ctx);
  ssh_adt_destroy(ctx->list);
  ssh_xfree(ctx);
}


static void fcr_helper_error_cb(SshFileCopyError error,
                                const char *error_message,
                                void *context)
{
  RecurseHelperCtx ctx = (RecurseHelperCtx)context;
  (*ctx->error_cb)(error, error_message, ctx->completion_ctx);
}

void ssh_file_copy_recurse(SshFileCopyConnection source,
                           const char *path,
                           SshFileCopyRecurseAttrs attrs,
                           SshFileCopyRecurseFileFunc func,
                           SshFileCopyRecurseReadyCB ready_cb,
                           SshFileCopyErrorCB error_cb,
                           void *context)
{
  RecurseHelperCtx ctx = ssh_xcalloc(1, sizeof(*ctx));
  SshFileCopyFile file;
  SshADTHandle h;
  
  ctx->list = ssh_file_copy_file_list_create(NULL_FNPTR, NULL);
  ctx->completion_ctx = context;
  ctx->func = func;
  ctx->ready_cb = ready_cb;
  ctx->error_cb = error_cb;
  
  file = ssh_file_copy_file_allocate();
  ssh_file_copy_file_register_filename(file, ssh_xstrdup(path));
  h = ssh_adt_insert(ctx->list, file);

  ssh_file_copy_recurse_multiple(source, ctx->list, attrs,
                                 fcr_helper_rec_func,
                                 fcr_helper_ready_cb,
                                 fcr_helper_error_cb,
                                 ctx);
}
