/*
  sshfc_glob.c

  Author: Sami Lehtinen <sjl@ssh.com>
  
          Regex help from Antti Huima <huima@ssh.com>

  Copyright (C) 2000-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Functions that perform the globbing in file transfer.
 */

#include "sshincludes.h"
#include "sshfilecopy.h"
#include "sshfilecopyi.h"
#include "sshappcommon.h"
#include "sshregex.h"
#include "sshglob.h"





#define SSH_DEBUG_MODULE "SshFCGlob"








/**********
 * ssh_file_copy_glob() related stuff.
 **********/

SSH_FSM_STEP(glob_next_file);
SSH_FSM_STEP(glob_ready);
SSH_FSM_STEP(glob_finish);

typedef struct SshFileCopyGlobContextRec
{
  SshFSM fsm;

  /* The attrs, that can be used to affect the globbing behaviour. */
  SshFileCopyGlobAttrs attrs;
  /* Callback that is called when the globbing is ready (whether
     successfully or not), and if the glob has been successful, return
     the list. */
  SshFileCopyGlobReadyCB ready_cb;
  /* Callback that is called when a non-fatal error is
     encountered. (can't open some file etc.) */
  SshFileCopyErrorCB error_callback;
  /* Context, which is passed to the above callabacks.*/
  void *callback_context;

  /* The list of files. */
  SshADTContainer file_list;
  
  /* Connection used. */
  SshFileCopyConnection connection;

  SshFSMThread main_thread;

  /* 'Raw' filenames are stored here. */
  SshADTContainer orig_file_list;
  SshADTHandle h;  

  Boolean match_found;
} *SshFileCopyGlobContext;

typedef struct GlobThreadContextRec
{
  char *pattern;
  char *current_basedir;

  /* Globbin related data structures. */
  SshRegexContext regex_context;
  SshRegexMatcher pattern_matcher;

  Boolean match_found_with_cur;
} *GlobThreadContext;

void glob_thread_destructor(SshFSM fsm, void *tdata)
{
  GlobThreadContext thread_data = (GlobThreadContext)tdata;

  SSH_PRECOND(thread_data != NULL);

  SSH_DEBUG(3, ("Destroying thread data..."));

  ssh_xfree(thread_data->current_basedir);
  if (thread_data->pattern_matcher)
    ssh_regex_free(thread_data->pattern_matcher);
  ssh_xfree(thread_data->pattern);
  if (thread_data->regex_context)
    ssh_regex_free_context(thread_data->regex_context);
  ssh_xfree(thread_data);
}

SshUInt32 glob_recurse_file_func(const char *filename,
                                 const char *long_name,
                                 SshFileAttributes file_attributes,
                                 void *context)
{



  SshFSMThread thread = (SshFSMThread) context;
  SshUInt32 ret_val = SSH_FCR_ADD | SSH_FCR_RECURSE;
  
  FCC_TDATA(GlobThreadContext);
  









  SSH_DEBUG(5, ("Got file %s.", filename));

  if (strcmp(tdata->current_basedir, filename) == 0)
    return ret_val;

  if ((file_attributes->permissions & S_IFMT) == S_IFDIR)
    {
      /* If we match the whole filename, we don't want to recurse the
         directory. */
      if (ssh_regex_match_cstr(tdata->pattern_matcher, filename))
        {
          ret_val &= ~SSH_FCR_RECURSE;
        }
      else
        {
          char *temp_filename;
          /* If subdirectory name is not postfixed with a slash, do it here,
             so that regex-matcher won't unnecessarily match.

             Example:

             {(asterisk) means *}
             Pattern: tsap/foo/(asterisk).txt 

             filename: tsap/foo/tsup (a directory)

             would match with prefix matching (encounters end of string
             before match succeeds or fails), but

             filename: tsap/foo/tsup/

             won't, because '*' won't match a '/'.
         
          */
          if (filename[strlen(filename) - 1] != '/')
            ssh_xdsprintf(&temp_filename, "%s/", filename);
          else
            temp_filename = ssh_xstrdup(filename);
      
          if (!(ssh_regex_match_cstr_prefix(tdata->pattern_matcher,
                                            temp_filename)))
            {
              ssh_xfree(temp_filename);
              return 0L;
            }
          ssh_xfree(temp_filename);
        }      
    }
  else
    {
      if (!(ssh_regex_match_cstr(tdata->pattern_matcher, filename)))
        return 0L;

    }

  SSH_DEBUG(2, ("Accepting file ``%s''.", filename));
  /* We have found at least one real match. */
  tdata->match_found_with_cur = TRUE;
  ret_val |= SSH_FCR_ADD;
  
  /* Accept this file. */
  return ret_val;
}


void glob_recurse_ready_cb(SshFileCopyError error,
                           const char *error_message,
                           SshADTContainer file_list,
                           void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  SshADTHandle h, h2;
  FCC_DATA(SshFileCopyGlobContext, GlobThreadContext);  
  
  SSH_FSM_SET_NEXT(glob_next_file);

  switch(error)
    {
    case SSH_FC_OK:
      /* Merge received list to ours */
      if (tdata->match_found_with_cur)
        {
          /* Indicates at least one match with whole file_list. */
          gdata->match_found = TRUE;
          h = ssh_adt_get_handle_to_location(file_list, SSH_ADT_BEGINNING);
  
          while (h != SSH_ADT_INVALID)
            {
              SshFileCopyFile file =
                (SshFileCopyFile) ssh_adt_detach(file_list, h);
              SSH_ASSERT(file != NULL);
              h2 = ssh_adt_insert_to(gdata->file_list, SSH_ADT_END, file);
              SSH_ASSERT(h2 != SSH_ADT_INVALID);
              h = ssh_adt_get_handle_to_location(file_list, SSH_ADT_BEGINNING);
            }
        }
      else
        {
          FCC_ERROR(SSH_FC_ERROR_NO_MATCH, ("No match (pattern ``%s'')",
                                            tdata->pattern));
        }
      ssh_adt_destroy(file_list);
      break;
    case SSH_FC_ERROR:
    case SSH_FC_ERROR_DEST_NOT_DIR:
    case SSH_FC_ERROR_NO_SUCH_FILE:
    case SSH_FC_ERROR_PERMISSION_DENIED:
    case SSH_FC_ERROR_FAILURE:
    case SSH_FC_ERROR_ELOOP:
      FCC_ERROR(error, (error_message));
      break;
    case SSH_FC_ERROR_CONNECTION_FAILED:
    case SSH_FC_ERROR_CONNECTION_LOST:
    case SSH_FC_ERROR_PROTOCOL_MISMATCH:
      (*gdata->ready_cb)(error, error_message, NULL, gdata->callback_context);
      SSH_FSM_SET_NEXT(glob_finish);
      /* Only glob can give out NO_MATCH errors. */
    case SSH_FC_ERROR_NO_MATCH:
      ssh_fatal("Got NO_MATCH from recurse. (this is a bug.)");
      break;
    }

  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

void glob_recurse_error_callback(SshFileCopyError error,
                                 const char *error_message,
                                 void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  SshFileCopyGlobContext gdata =
    (SshFileCopyGlobContext)ssh_fsm_get_gdata(thread);

  (*gdata->error_callback)(error, error_message, gdata->callback_context);
}





























SSH_FSM_STEP(glob_next_file)
{
  SshFileCopyGlobContext gdata = (SshFileCopyGlobContext)fsm_context;
  GlobThreadContext tdata = (GlobThreadContext)thread_context;
  SshADTHandle h;
  SshFileCopyRecurseAttrsStruct rec_attrs;  
  char *temp_filename, *orig_filename, *p, *hlp;





  memset(&rec_attrs, 0, sizeof(rec_attrs));
  
  tdata->match_found_with_cur = FALSE;
  ssh_xfree(tdata->pattern);
  tdata->pattern = NULL;
  
  if (gdata->h == SSH_ADT_INVALID)
    {
      SSH_DEBUG(3, ("File list is at it's end."));
      SSH_FSM_SET_NEXT(glob_ready);
      return SSH_FSM_CONTINUE;
    }

  /* Get next file from list. */
  orig_filename = ssh_adt_get(gdata->orig_file_list, gdata->h);
  SSH_ASSERT(orig_filename);
  orig_filename = ssh_xstrdup(orig_filename);









  SSH_DEBUG(2, ("orig_filename = %s", orig_filename));
  
  /* Advance placeholder to next. */
  gdata->h = ssh_adt_enumerate_next(gdata->orig_file_list, gdata->h);
  




















    {
      temp_filename = ssh_xstrdup(orig_filename);
    }







  if ((p = ssh_glob_next_unescaped_wildchar(temp_filename)) != NULL)

    {








      *p = '\0';

      if ((p = strrchr(temp_filename, '/')) != NULL)
        {
          /* Path has more than one component at basepart */
          if (p > &temp_filename[0])
            {
              *p = '\0';
              tdata->current_basedir = temp_filename;
              temp_filename =
                ssh_xstrdup(&orig_filename[strlen
                                           (tdata->current_basedir) + 1]);
            }
          else
            {
              /* The only '/' was at the beginning of the filename. */
              tdata->current_basedir = ssh_xstrdup("/");
              ssh_xfree(temp_filename);
              temp_filename = ssh_xstrdup(&orig_filename[1]);
            }
        }
      else
        {
          /* No '/' characters at basepart. */
          tdata->current_basedir = ssh_xstrdup("");
          ssh_xfree(temp_filename);
          temp_filename = ssh_xstrdup(orig_filename);
        }
    }
  else
    {
      SshFileCopyFile new_loc, new_file;
      char *temp_basedir;
      
      /* Doesn't contain wildchars. */
      SSH_TRACE(3, ("Adding file %s to file list without globbing...",
                    temp_filename));

      new_loc = ssh_file_copy_file_allocate();
      ssh_file_copy_tokenize_path(temp_filename, &temp_basedir, &hlp);
      ssh_xfree(temp_filename);
      temp_filename = ssh_glob_strip_escapes(hlp);
      ssh_xfree(hlp);
      
      SSH_DEBUG(1, ("basedir: %s, file: %s", temp_basedir, temp_filename));
      
      gdata->match_found = TRUE;

      if (!temp_basedir)
        temp_basedir = ssh_xstrdup(".");

      new_file = ssh_file_copy_file_allocate();      
      
      ssh_file_copy_file_register_filename(new_loc, temp_basedir);
      ssh_file_copy_file_register_filename(new_file, temp_filename);
      ssh_file_copy_file_register_dir_entry(new_loc, new_file);
      
      h = ssh_adt_insert_to(gdata->file_list, SSH_ADT_END, new_loc);
      SSH_ASSERT(h != SSH_ADT_INVALID);
      
      return SSH_FSM_CONTINUE;

    }

  if (tdata->pattern_matcher)
    ssh_regex_free(tdata->pattern_matcher);

  if (strlen(tdata->current_basedir) == 0)
    {
      ssh_xfree(tdata->current_basedir);
      tdata->current_basedir = ssh_xstrdup(".");

      /* This is because recurse would return "./.zshrc", and our
         pattern would be e.g. ".zshr*". So we need to patch the
         pattern. */
      ssh_xdsprintf(&hlp, "./%s", orig_filename);
      ssh_xfree(orig_filename);
      orig_filename = hlp;
    }

  tdata->pattern = orig_filename;
  tdata->pattern_matcher = ssh_regex_create(tdata->regex_context,
                                            orig_filename,
                                            SSH_REGEX_SYNTAX_ZSH_FILEGLOB);

  if (!tdata->pattern_matcher)
    {
      FCC_ERROR(SSH_FC_ERROR_FAILURE, ("Pattern ``%s'' was deemed invalid "
                                       "by the sshregex-library.",
                                       tdata->pattern));
      return SSH_FSM_CONTINUE;
    }

  ssh_xfree(temp_filename);

  /* Strip escaped from basedir before using it. */
  hlp = tdata->current_basedir;
  tdata->current_basedir = ssh_glob_strip_escapes(tdata->current_basedir);
  ssh_xfree(hlp);

  
  SSH_TRACE(2, ("Starting glob of %s...", orig_filename));
  SSH_DEBUG(3, ("basedir: %s", tdata->current_basedir));

  rec_attrs.compare_func = gdata->attrs->compare_func;
  rec_attrs.compare_ctx = gdata->attrs->compare_ctx;
  
  SSH_FSM_ASYNC_CALL(ssh_file_copy_recurse(gdata->connection,
                                           tdata->current_basedir,
                                           &rec_attrs,
                                           glob_recurse_file_func,
                                           glob_recurse_ready_cb,
                                           glob_recurse_error_callback,
                                           thread));
}

SSH_FSM_STEP(glob_ready)
{
  SshFileCopyGlobContext gdata = (SshFileCopyGlobContext)fsm_context;

  SSH_FSM_SET_NEXT(glob_finish);

  if (gdata->match_found)
    {      
      (*gdata->ready_cb)(SSH_FC_OK, "Globbing successful.", gdata->file_list,
                         gdata->callback_context);
      gdata->file_list = NULL;
    }
  else
    {
      (*gdata->ready_cb)(SSH_FC_ERROR_NO_MATCH, "No matches.",
                         NULL, gdata->callback_context);
    }
  
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(glob_finish)
{
  SshFileCopyGlobContext gdata = (SshFileCopyGlobContext)fsm_context;

  ssh_fsm_destroy(gdata->fsm);
  /* If something went wrong, free the unfinished list. */
  ssh_adt_destroy(gdata->file_list);
  
  SSH_DEBUG(3, ("Destroying global data..."));
  ssh_xfree(gdata->attrs);
  ssh_xfree(gdata);
  
  return SSH_FSM_FINISH;
}

SshFSMStateDebugStruct glob_states_array[] =
{ { "fcg_next_file", "Set next file to glob", glob_next_file },
  { "fcg_ready", "Globbing ready", glob_ready },
  { "fcg_finish", "Finish", glob_finish }
};

void ssh_file_copy_glob_multiple(SshFileCopyConnection connection,
                                 SshADTContainer file_list,
                                 SshFileCopyGlobAttrs attrs,
                                 SshFileCopyGlobReadyCB ready_cb,
                                 SshFileCopyErrorCB error_callback,
                                 void *context)
{
  SshFileCopyGlobContext glob_context;
  GlobThreadContext tdata;

  SSH_PRECOND(connection != NULL);
  SSH_PRECOND(ready_cb != NULL_FNPTR);
  SSH_PRECOND(error_callback != NULL_FNPTR);

  SSH_DEBUG(2, ("Starting globbing, file_list has %ld entries.",
                ssh_adt_num_objects(file_list)));
  
  glob_context = ssh_xcalloc(1, sizeof(*glob_context));
  glob_context->fsm = ssh_fsm_create(glob_context);
  SSH_VERIFY(glob_context->fsm != NULL);
  ssh_fsm_register_debug_names(glob_context->fsm, glob_states_array,
                               SSH_FSM_NUM_STATES(glob_states_array));
  if (attrs == NULL)
    attrs = ssh_xcalloc(1, sizeof(*attrs));
    
  glob_context->connection = connection;
  glob_context->attrs = ssh_xcalloc(1, sizeof(*attrs));
  if (attrs != NULL)
    *glob_context->attrs = *attrs;

  glob_context->ready_cb = ready_cb;
  glob_context->error_callback = error_callback;
  glob_context->callback_context = context;
  glob_context->file_list =
    ssh_file_copy_file_list_create(glob_context->attrs->compare_func,
                                   glob_context->attrs->compare_ctx);
  glob_context->orig_file_list = file_list;
  glob_context->h = ssh_adt_enumerate_start(file_list);

  glob_context->match_found = FALSE;
  
  tdata = ssh_xcalloc(1, sizeof(*tdata));
  glob_context->main_thread = ssh_fsm_thread_create(glob_context->fsm,
                                                    glob_next_file,
                                                    NULL_FNPTR,
                                                    glob_thread_destructor,
                                                    tdata);
  SSH_VERIFY(glob_context->main_thread != NULL);
  tdata->regex_context = ssh_regex_create_context();
  SSH_VERIFY(tdata->regex_context != NULL);
  
  ssh_fsm_set_thread_name(glob_context->main_thread, "main_thread");
}









