/*

  sshconfig.h

  Authors:
        Tatu Ylnen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Processing configuration data in SSH (both client and server).

*/

#ifndef SSHCONFIG_H
#define SSHCONFIG_H

typedef struct SshConfigRec *SshConfig;

#include "sshuser.h"
#include "sshcrypt.h"
#include "sshhostkey.h"
#include "sshadt.h"
#include "sshadt_conv.h"
#include "sshregex.h"








#define SUBSYSTEM_PREFIX "subsystem-"
#define SUBSYSTEM_PREFIX_LEN 10

/* Definition for SshForward */

typedef struct SshForwardRec {
  struct SshForwardRec *next;
  char *local_addr;
  char *port;
  char *connect_to_host;
  char *connect_to_port;
  char *protocol;
} *SshForward;

typedef enum {
  /* No ssh1 agent compatibility */
  SSH_AGENT_COMPAT_NONE = 0, 
  /* Forward connections for old ssh1 agent.  Also ssh2 agent works with 
     this mode, but no agent forwarding path is added to the data. */
  SSH_AGENT_COMPAT_TRADITIONAL = 1,
  /* Forward connections for ssh2 agent emulating ssh1 agent.  Ssh1 agent
     do not work with this mode. */
  SSH_AGENT_COMPAT_SSH2 = 2
} SshAgentSsh1CompatMode;

typedef enum {
  /* Don't allow root login's. */
  SSH_ROOTLOGIN_FALSE = 0,
  /* Allow root login's. */
  SSH_ROOTLOGIN_TRUE = 1,
  /* Don't allow if using password authentication. */ 
  SSH_ROOTLOGIN_NOPWD = 2
} SshPermitRootLogin;

typedef enum {
  SSH_STRICT_HOSTKEY_CHECKING_NO = 0,
  SSH_STRICT_HOSTKEY_CHECKING_YES = 1,
  SSH_STRICT_HOSTKEY_CHECKING_ASK = 2
} SshStrictHostKeyChecking;

typedef struct SshPatternHolderRec
{
  char *pattern;
  SshRegexSyntax regex_syntax;
} SshPatternHolderStruct, *SshPatternHolder;

typedef struct SshSubConfigRec
{
  SshPatternHolderStruct pattern;
  char *config_file;
} SshSubConfigStruct, *SshSubConfig;

typedef struct SshMetaConfigRec
{
  int version_major;
  int version_minor;
  SshRegexSyntax regex_syntax;
} SshMetaConfigStruct, *SshMetaConfig;

typedef struct SshForwardACLRec {
  Boolean allow;
  Boolean local;
  SshPatternHolderStruct user_pattern;
  SshPatternHolderStruct forward_pattern;
  SshPatternHolderStruct originator_pattern;
} SshForwardACLStruct, *SshForwardACL;

/* Data type for SSH server configuration data. */
struct SshConfigRec
{
  /* TRUE if the config data is for a client. */
  Boolean client;

  /* Hash tables with parsing information of all the configuration
     variables. */
  SshADTContainer set_param;
  SshADTContainer set_param_aliases;
  
  /* The list of host keys */
  SshHostKeysContext host_keys_ctx;

  /* common for both client and server */
  Boolean agent_forwarding;
  Boolean x11_forwarding;

  Boolean x11_application_trusted;
  /* Path to the "xauth" program. */
  char *xauth_path;
  
  /* For server. */
  Boolean allow_tcp_forwarding;

  /* List of subconfig files. */
  SshADTContainer user_configurations;
  SshADTContainer host_configurations;
  
  Boolean force_ptty_allocation;
  Boolean verbose_mode;
  Boolean compression;
  int compression_level;
  SshADTContainer allowed_authentications;
  SshADTContainer required_authentications;
  Boolean user_known_hosts;
  
  char *port;
  char *ciphers;
  char *macs;
  char *user_conf_dir;
  char *identity_file;
  char *authorization_file;
  char *random_seed_file;





  char *pgp_public_key_file;
  char *pgp_secret_key_file;

  char *external_authorization_prog;

  char *passwd_path;
  char *password_prompt;
  int password_guesses;
  int number_of_password_prompts;
  
  int auth_interactive_failure_timeout;

  /* Minimum and maximum size of publickey used in publickey auth. */
  unsigned int auth_publickey_min_size;
  unsigned int auth_publickey_max_size;




#ifdef SSHDIST_SSH2_AUTH_KBDINTERACTIVE
#ifdef SSH_SERVER_WITH_KEYBOARD_INTERACTIVE
  /* Keyboard interactive for server */
  int auth_kbd_int_optional_needed;

  SshADTContainer auth_kbd_int_required;
  SshADTContainer auth_kbd_int_optional;

  int auth_kbd_int_retries;

  char *auth_kbd_int_plugin_prog;
#endif /* SSH_SERVER_WITH_KEYBOARD_INTERACTIVE */
#endif /* SSHDIST_SSH2_AUTH_KBDINTERACTIVE */
  
  /* Environment variables to be set in the remote server. */
  SshADTContainer remote_env;
  /* Settable environment vars (in server). */
  SshADTContainer settable_env_vars;

  SshUInt32 rekey_interval_bytes;
  SshUInt32 rekey_interval_seconds;

  int max_connections;
  
  char *host_to_connect;
  char *socks_server;
  Boolean use_socks5;
  
  char *login_as_user;
  char *password; /* used on Windows */
  SshForward local_forwards;
  SshForward remote_forwards;

  SshADTContainer allowed_hosts;
  SshADTContainer denied_hosts;
  SshADTContainer allowed_shosts;
  SshADTContainer denied_shosts;

  SshADTContainer allowed_users;
  SshADTContainer denied_users;
  SshADTContainer allowed_groups;
  SshADTContainer denied_groups;
  
  Boolean require_reverse_mapping;  /* is reverse DNS mapping required */
  Boolean try_reverse_mapping; /* if DNS mapping is not required, this
                                  specifies whether it is done anyway */

  SshLogFacility log_facility;
  /* This should be SshLogFacility, but it is initialized to -1, so we
     use int. */
  int sftp_server_log_facility;
  char *debug_log_file_name;

  Boolean batch_mode;
  Boolean authentication_notify;
  SshStrictHostKeyChecking strict_host_key_checking;
  Boolean go_background;
  Boolean one_shot_forwarding;
  Boolean dont_read_stdin;
  Boolean gateway_ports;
  char *escape_char;
  Boolean ignore_rhosts;
  /* This is int because we assign it to -1 first; just a style
     issue. */
  int ignore_root_rhosts;
  SshPermitRootLogin permit_root_login;
  Boolean permit_empty_passwords;
  Boolean try_empty_password;
  Boolean strict_modes;
  Boolean quiet_mode;
  Boolean fascist_logging;
  Boolean print_motd;
  Boolean check_mail;
  Boolean auth_success_msg;
  Boolean keep_alive;
  Boolean no_delay;
  Boolean inetd_mode;

  Boolean hostbased_force_client_hostname_dns_match;

  char *listen_address;
  int login_grace_time;

  SshADTContainer chroot_users;
  SshADTContainer chroot_groups;

  SshADTContainer allowed_tcp_forwarding_users;
  SshADTContainer denied_tcp_forwarding_users;
  SshADTContainer allowed_tcp_forwarding_groups;
  SshADTContainer denied_tcp_forwarding_groups;

  SshADTContainer forward_acls;
  
  SshADTContainer subsystems;

#ifdef SSH_SERVER_WITH_SECURID
  int securid_guesses; /* number of guesses at passphrase allowed */
#endif /* SSH_SERVER_WITH_SECURID */





  /* How many UDP broadcast commands are handled by server each second.
     If this is zero, no broadcasts are received at all. */
  int broadcasts_allowed_per_second;

  /* Flag specifying whether to enable ssh1 compatibility. */
  Boolean ssh1compatibility;

  /* Ssh1 agent forwarding compatibility mode */
  SshAgentSsh1CompatMode ssh_agent_compat;

  /* Path to ssh1/sshd1.*/
  char *ssh1_path;

  /* Path to alternate sshd1 configuration file. */
  char *sshd1_config_file_path;

#ifdef SSHDIST_SSH2_INTERNAL_SSH1_EMULATION
#ifdef WITH_INTERNAL_SSH1_EMULATION
  /* Use internal ssh1 protocol emulation (client only). */
  Boolean ssh1_emulation_internal;

  /* If TRUE, don't try to mask password length. */
  Boolean ssh1_no_ignore_packets_in_password_auth;
#endif /* WITH_INTERNAL_SSH1_EMULATION */
#endif /* SSHDIST_SSH2_INTERNAL_SSH1_EMULATION */

  /* Ssh1 arguments for compatibility. If this is NULL, there will be no
     ssh1 compatibility.  The first argument should be the program name. */
  char **ssh1_argv;
  int ssh1_argc;

  /* The file descriptor for the connection to the remote ssh1 server/client.
     This is used in ssh1 compatibility code. */
  int ssh1_fd;






  /* Whether to disable compatibility with old ssh2 versions (for
     paranoid reasons) */
  Boolean disable_ssh2_compat;

  /* Path to ssh-signer2. */
  char *signer_path;

  /* The default domain, which should be set if, for example
     'hostname' returns only basepart of the FQDN. */
  char *default_domain;

  /* The path to the banner message, if any. */
  char *banner_message_file_path;
  /* The banner message. */
  char *banner_msg;

  /* Server idle timeout in seconds for the connection. 0 = disabled */
  SshUInt32 idle_timeout;

  /* Protocols that client uses to make connection.  See protocol_mask
     in SshTcpConnectParams in sshtcp.h */
  SshUInt32 protocol_mask;

  /* Protocol version string used by the server. (After the "SSH-2.0-") */
  char *protocol_version_string;
  


















































































};

/* This a template for a function which is used to check whether a
   given parameter (represented as a standard C-string) is valid by
   certain qualifiers. Return FALSE if so, TRUE if not.*/
typedef Boolean (*SshParameterValidityProc)(const char *param,
                                            void *data);

/* Parse a configuration/authorization file into an array of variable
   name <-> value pairs. Return the number of variables or -1 on
   error. Pointers to tables of pointers to null-terminated strings are
   placed at *variables and *values. ``metaconfig'' will store
   "configuration for the configuration" and can be NULL, if caller
   doesn't care. Stanzas with headings matching ``instance'' will be
   parsed. If ``instance'' is NULL, this function will warn about
   headings, and the stanzas following headings will be ignored. */
int ssh2_parse_config(SshUser user, const char *instance, const char *path, 
                      char ***variables, char ***values,
                      SshMetaConfig metaconfig,
                      Boolean remove_quotes);

/* Return TRUE if match found. */
typedef Boolean (*SshConfigMatchCB)(const char *heading,
                                    SshMetaConfig metaconfig,
                                    void *context);

/* As ssh2_parse_config(), but more generic. If ``match_cb'' is not
   NULL_FNPTR, this function will ignore ``instance'' and will use the
   callback to check whether a heading matches. If ``match_cb'' is
   NULL_FNPTR and instance is NULL, will warn about headings, and the
   stanzas following headings will be ignored. */
int ssh2_parse_config_ext(SshUser user, const char *instance,
                          SshConfigMatchCB match_cb,
                          void *match_ctx,
                          const char *path,
                          char ***variables, char ***values,
                          SshMetaConfig metaconfig,
                          Boolean remove_quotes);

/* Free the "vars" and "vals" arrays */
void ssh_free_varsvals(int n, char **vars, char **vals);

/* Returns default configuration information for the server. */
SshConfig ssh_server_create_config(void);

/* Returns default configuration information for the client. */
SshConfig ssh_client_create_config(void);

/* This should be called after initializing the config-struct
   (ie. after command-line variables have been parsed. This checks
   that some required members are initialized properly.*/
void ssh_config_init_finalize(SshConfig config);

/* Frees client configuration data. */
void ssh_config_free(SshConfig config);






/* The different config types. XXX these are not yet used. */
#define SSH_CONFIG_TYPE_CLIENT        0x0001
#define SSH_CONFIG_TYPE_SERVER_GLOBAL 0x0002
#define SSH_CONFIG_TYPE_SERVER_USER   0x0004
#define SSH_CONFIG_TYPE_SERVER_HOST   0x0008

#define SSH_CONFIG_TYPE_ALL (SSH_CONFIG_TYPE_CLIENT |           \
                             SSH_CONFIG_TYPE_SERVER_GLOBAL |    \
                             SSH_CONFIG_TYPE_SERVER_USER |      \
                             SSH_CONFIG_TYPE_SERVER_HOST)
#define SSH_CONFIG_TYPE_SERVER_ALL (SSH_CONFIG_TYPE_SERVER_GLOBAL |     \
                                    SSH_CONFIG_TYPE_SERVER_USER |       \
                                    SSH_CONFIG_TYPE_SERVER_HOST)

/* Reads config data from the given file.  Stanzas with headings that
   match ``instance'' will be parsed. If ``instance'' is NULL, informs
   ssh_config_read_file() that it should warn about stanzas (they won't
   ever be parsed, therefore they will only confuse).  Returns FALSE if
   an error occurs (displays error messages with ssh_warning()) */
Boolean ssh_config_read_file(SshUser user, SshConfig config,
                             const char *instance, const char *filename,
                             int config_type);

/* As ssh_config_read_file(), but more generic.  If ``match_cb'' is not
   NULL_FNPTR, this function will ignore ``instance'' and will use the
   callback to check whether a heading matches. If ``match_cb'' is
   NULL_FNPTR and instance is NULL, will warn about headings, and the
   stanzas following headings will be ignored. */
Boolean ssh_config_read_file_ext(SshUser user, SshConfig config,
                                 const char *instance,
                                 SshConfigMatchCB match_cb, void *match_ctx,
                                 const char *filename,
                                 int config_type);

/* Set the variable corresponding to `var' to `val' in config. Return
   TRUE if an error occurs. */
Boolean ssh_config_set_parameter(SshConfig config, SshMetaConfig metaconfig,
                                 char *var, char *val, int config_type);

/* Parse a line of input. Return TRUE if an error occurs. */
Boolean ssh_config_parse_line(char *line, char **var, char **val);

/* Parse a line of input, and set variables in the config
   struct. Return TRUE if an error occurs. */
Boolean ssh_config_parse_line_and_set_params(SshConfig config, char *line,
                                             int config_type);

/* Reads the host key that is defined in the config data. Returns
   TRUE if succesful. */
Boolean ssh_server_load_host_key(SshConfig config,
                                 SshPrivateKey *private_host_key,
                                 unsigned char **public_host_key_blob,
                                 size_t *public_host_key_blob_len);

/* Allocatar function for SshPatternHolder */
SshPatternHolder ssh_config_pattern_alloc(char *token,
                                          SshMetaConfig metaconfig);

/* Destroys an SshPatternHolder object (this is a valid callback to give
   to ssh_adt_create_generic()) */
void ssh_config_pattern_destructor(SshPatternHolder holder, void *context);

/* Parse forwarding definitions. Format is port:remotehost:remoteport */
Boolean ssh_parse_forward(SshForward *forwards, const char *spec);

/* Return idle-timeout value. Return -1 if parsing fails. */
SshInt32 ssh_config_parse_timeout(const char *value);

/* Get log facility configuration value as an xmallocated string
   (e.g. with SSH_LOGFACILITY_AUTH this function returns
   "AUTH"). Returns NULL if given log facility is not found. */
char *ssh_config_log_facility_name(SshLogFacility fac);

/* Read in banner message file (moved from ssh_config_init_finalize()
   for HostSpecificConfig). */
void ssh_config_read_banner_message(SshConfig config);

#endif /* SSHCONFIG_H */
