/*

sshchtcpfwd.c

  Authors:
        Tatu Ylonen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Code implementing TCP/IP forwarding channels for SSH2 servers and clients.

*/


#include "ssh2includes.h"
#include "sshtcp.h"
#include "sshencode.h"
#include "sshmsgs.h"
#include "sshuser.h"
#include "sshcommon.h"
#include "sshappcommon.h"
#include "sshdsprintf.h"
#include "sshnameserver.h"
#ifdef SSHDIST_SSH2_FTP_FORWARDING
#include "sshftpfilter.h"
#endif /* SSHDIST_SSH2_FTP_FORWARDING */
#ifdef SSHDIST_SSH2_SOCKS_FILTER
#include "sshsocksfilter.h"
#endif /* SSHDIST_SSH2_SOCKS_FILTER */

#ifndef VXWORKS
#include <syslog.h>
#endif /* VXWORKS */
#ifdef NEED_SYS_SYSLOG_H
#include <sys/syslog.h>
#endif /* NEED_SYS_SYSLOG_H */






#ifdef SSH_CHANNEL_TCPFWD
#include "sshchtcpfwd.h"

#ifdef HAVE_LIBWRAP
#include <netdb.h>
#include <tcpd.h>
#include "sshfdstream.h"
#endif /* HAVE_LIBWRAP */
#include "auths-common.h"

#define SSH_DEBUG_MODULE "Ssh2ChannelTcpFwd"

#define SSH_TCPIP_WINDOW         30000
#define SSH_TCPIP_PACKET_SIZE     4096

typedef struct SshChannelTcpFwdConnectRec
{
  SshRemoteTcpForward fwd;
  int channel_id;



  SshConnOpenCompletionProc completion;

  char *originator_ip;
  void *completion_context;
} *SshChannelTcpFwdConnect;


#ifndef SSH_WIN_CLIENT
/* Helper function to match a forward acl with user+host+forward. Return
   TRUE if match found. */
Boolean match_forward_acl(SshCommon common,
                          SshForwardACL forw_acl,
                          const char *uid_str,
                          const char *address_fqdn,
                          const char *address_ip,
                          const char *port_str,
                          const char *originator_address,
                          SshRegexContext rex_ctx)
{
  char *port_pat, *copy;
  Boolean match;
  SshPatternHolderStruct holder;
  
  SSH_PRECOND(common != NULL);
  
  /* user pattern */
  match = ssh_match_user_groups(ssh_user_name(common->user_data), uid_str,
                                common->num_groups, common->group_strs,
                                common->gid_strs,
                                common->remote_host,
                                common->remote_ip,
                                &forw_acl->user_pattern, rex_ctx);
  if (!match)
    return FALSE;
  
  /* forward target pattern */
  copy = ssh_xstrdup(forw_acl->forward_pattern.pattern);
  if ((port_pat = strchr(copy, '%')) != NULL)
    {
      *port_pat = '\0';
      port_pat++;
    }
  holder.pattern = copy;
  holder.regex_syntax = forw_acl->forward_pattern.regex_syntax;
  match = ssh_match_host_id(address_fqdn, address_ip,
                            &holder, rex_ctx);
  if (match && port_pat)
    match = ssh_match_string(port_str, port_pat,
                             forw_acl->forward_pattern.regex_syntax,
                             rex_ctx);
  ssh_xfree(copy);
  if (!match)
    return FALSE;
  
  if (originator_address && forw_acl->originator_pattern.pattern)
    {
      /* originator pattern */
      match = ssh_match_host_id(originator_address, originator_address,
                                &forw_acl->originator_pattern, rex_ctx);
    }
  return match;
}

/* returns FALSE if user is allowed to forward. */
Boolean is_tcp_forwarding_denied_for_user(SshCommon common, Boolean local,
                                          const char *address_fqdn,
                                          const char *address_ip,
                                          const char *port,
                                          const char *originator_address)
{
  SshConfig config = common->config;
  int num_allowed = 0, num_denied = 0;

  SshGroup *groups = NULL;

  size_t num_groups = 0;
  char **group_strs = NULL, **gid_strs = NULL, *uid_str = NULL;
  int port_num = 0;
  SshForwardACL forw_acl;
  SshADTContainer c;
  SshADTHandle h;
  Boolean denied = FALSE, match;
  SshRegexContext rex_ctx = ssh_app_get_global_regex_context();
  
  if (local)
    SSH_ASSERT(originator_address != NULL);
  
  match =
    ssh_server_auth_check_user_generic(common->user_data, common->user,
                                       "is allowed TCP forwarding",
                                       "in denied TCP forwarding",
                                       common->remote_host, common->remote_ip,
                                       config->allowed_tcp_forwarding_users,
                                       config->denied_tcp_forwarding_users,
                                       config->allowed_tcp_forwarding_groups,
                                       config->denied_tcp_forwarding_groups);

  /* If the user is denied forwarding by base conf, they override ACLs. */
  if (match)
    return TRUE;

  port_num = ssh_inet_get_port_by_service(port, "tcp");
  if (port_num < 1 || port_num > 65535)
    {
      SSH_DEBUG(2, ("Invalid port %ld.", port));
      return TRUE;
    }

  c = config->forward_acls;
  
  for (h = ssh_adt_enumerate_start(c);
       h != SSH_ADT_INVALID;
       h = ssh_adt_enumerate_next(c, h))
    {
      forw_acl = ssh_adt_get(c, h);
      SSH_ASSERT(forw_acl != NULL);
      if (forw_acl->local != local)
        continue;

      if (forw_acl->allow)
        num_allowed++;
      else
        num_denied++;
    }

  /* If forward ACLs are empty, user is allowed to forward. */
  if (num_allowed == 0 && num_denied == 0)
    return FALSE;

  /* If group_strs,gid_strs are not allocated yet, do it now. */
  if (common->num_groups == 0)
    {

      int i;
      groups = ssh_user_get_groups(common->user_data);
      SSH_VERIFY(groups != NULL);
      for (i = 0; groups[i]; i++)
        ;
      num_groups = i;
      group_strs = ssh_xcalloc(i, sizeof(char *));
      gid_strs = ssh_xcalloc(i, sizeof(char *));
      for (i = 0; groups[i]; i++)
        {
          group_strs[i] = ssh_xstrdup(ssh_group_get_name(groups[i]));
          ssh_xdsprintf(&gid_strs[i], "%ld", ssh_group_get_gid(groups[i]));
        }

      common->num_groups = num_groups;
      common->group_strs = group_strs;
      common->gid_strs = gid_strs;
    }

  ssh_xdsprintf(&uid_str, "%ld", ssh_user_uid(common->user_data));
  
  if (num_denied > 0)
    {
      for (h = ssh_adt_enumerate_start(c);
           h != SSH_ADT_INVALID;
           h = ssh_adt_enumerate_next(c, h))
        {
          forw_acl = ssh_adt_get(c, h);
          SSH_ASSERT(forw_acl != NULL);
          if (forw_acl->local != local)
            continue;
      
          match = match_forward_acl(common, forw_acl, uid_str, address_fqdn,
                                    address_ip, port, originator_address,
                                    rex_ctx);
          if (match)
            break;
        }
      if (match)
        {
          denied = TRUE;
          goto free_and_return;
        }
    }
  
  if (num_allowed > 0)
    {
      for (h = ssh_adt_enumerate_start(c);
           h != SSH_ADT_INVALID;
           h = ssh_adt_enumerate_next(c, h))
        {
          forw_acl = ssh_adt_get(c, h);
          SSH_ASSERT(forw_acl != NULL);
          if (forw_acl->local != local)
            continue;
      
          match = match_forward_acl(common, forw_acl, uid_str, address_fqdn,
                                    address_ip, port, originator_address,
                                    rex_ctx);
          if (match)
            break;
        }

      if (match)
        denied = FALSE;
      else
        denied = TRUE;
    }

 free_and_return:
  
  return denied;
}
#endif /* SSH_WIN_CLIENT */


/***********************************************************************
 * Glue functions for creating/destroying channel type and session
 * contexts.
 ***********************************************************************/

/* This function is called once when a SshCommon object is created. */

void *ssh_channel_ftcp_create(SshCommon common)
{
  SshChannelTypeTcpForward ct;

  ct = ssh_xcalloc(1, sizeof(*ct));
  ct->common = common;
  return ct;
}

/* This function is called once when an SshCommon object is being
   destroyed.  This should destroy all remote forwarded TCP/IP
   channels and listeners and free the context. */

void ssh_channel_ftcp_destroy(void *context)
{
  SshChannelTypeTcpForward ct = (SshChannelTypeTcpForward)context;
  SshRemoteTcpForward remote_fwd, remote_next;

  /* Destroy all existing channels.
     XXX not implemented. */

  /* Free any remote forwarding records. */
  for (remote_fwd = ct->remote_forwards; remote_fwd;
       remote_fwd = remote_next)
    {
      remote_next = remote_fwd->next;
      if (remote_fwd->listener)
        ssh_tcp_destroy_listener(remote_fwd->listener);
      ssh_xfree(remote_fwd->address_to_bind);
      ssh_xfree(remote_fwd->port);
      ssh_xfree(remote_fwd->protocol);
      ssh_xfree(remote_fwd->connect_to_host);
      ssh_xfree(remote_fwd->connect_to_port);
      memset(remote_fwd, 'F', sizeof(*remote_fwd));
      ssh_xfree(remote_fwd);
    }

  /* Destroy the channel type context. */
  memset(ct, 'F', sizeof(*ct));
  ssh_xfree(ct);
}

/* Returns the channel type context from the SshCommon object. */

SshChannelTypeTcpForward ssh_channel_ftcp_ct(SshCommon common)
{
  return (SshChannelTypeTcpForward)
    ssh_common_get_channel_type_context(common, "forwarded-tcpip");
}

/* This function is called once when a SshCommon object is created. */

void *ssh_channel_dtcp_create(SshCommon common)
{
  SshChannelTypeTcpDirect ct;

  ct = ssh_xcalloc(1, sizeof(*ct));
  ct->common = common;
  return ct;
}

/* This function is called once when an SshCommon object is being
   destroyed.  This should destroy all locally forwarded TCP/IP
   channels and listeners and free the context. */

void ssh_channel_dtcp_destroy(void *context)
{
  SshChannelTypeTcpDirect ct = (SshChannelTypeTcpDirect)context;
  SshLocalTcpForward local_fwd, local_next;

  /* Destroy all existing channels.
     XXX not implemented. */

  /* Free local forwarding records. */
  for (local_fwd = ct->local_forwards; local_fwd;
       local_fwd = local_next)
    {
      local_next = local_fwd->next;
      if (local_fwd->listener)
        ssh_tcp_destroy_listener(local_fwd->listener);
      ssh_xfree(local_fwd->connect_to_host);
      ssh_xfree(local_fwd->connect_to_port);
      ssh_xfree(local_fwd->protocol);
      ssh_xfree(local_fwd->port);
      memset(local_fwd, 'F', sizeof(*local_fwd));
      ssh_xfree(local_fwd);
    }

  /* Destroy the channel type context. */
  memset(ct, 'F', sizeof(*ct));
  ssh_xfree(ct);
}

/* Returns the channel type context from the SshCommon object. */

SshChannelTypeTcpDirect ssh_channel_dtcp_ct(SshCommon common)
{
  return (SshChannelTypeTcpDirect)
    ssh_common_get_channel_type_context(common, "direct-tcpip");
}

/***********************************************************************
 * Handling destruction of a TCP/IP channel.
 ***********************************************************************/

/* Function to be called when a forwarded TCP/IP connection is closed.
   This function is used for all types of TCP/IP channels. */

void ssh_channel_tcp_connection_destroy(void *context)
{
  SshCommon common = (SshCommon)context;

  /* Inform the common code that a channel has been destroyed. */
  ssh_common_destroy_channel(common);
}

/***********************************************************************
 * Processing a channel open request for a remote-forwarded TCP/IP
 * channel, and connecting to the destination address/port.
 ***********************************************************************/

/* Called when a connection to the real TCP/IP port (that the
   connection was forwarded to) has been established. */

void ssh_channel_ftcp_open_connected(SshTcpError error,
                                     SshStream stream,
                                     void *context)
{
  SshChannelTcpFwdConnect c = (SshChannelTcpFwdConnect)context;
  SshTcpForwardActive active_fwd_context;

  if (error != SSH_TCP_OK)
    {
      ssh_warning("Connecting to %s:%s failed (remote forward, port %s)",
                  c->fwd->connect_to_host, c->fwd->connect_to_port,
                  c->fwd->port);
      (*c->completion)(SSH_OPEN_CONNECT_FAILED,
                       NULL, FALSE, FALSE, 0, NULL, 0, NULL_FNPTR, NULL_FNPTR,
                       NULL, c->completion_context);
      memset(c, 'F', sizeof(*c));
      ssh_xfree(c);
      return;
    }

  if (strcasecmp(c->fwd->protocol, "tcp") != 0)
    {
      if (strcasecmp(c->fwd->protocol, "http") == 0)
        {
          ssh_warning("forwarding protocol \"http\" defaults to \"tcp\"");
        }
#ifdef SSHDIST_SSH2_FTP_FORWARDING
      else if (strcasecmp(c->fwd->protocol, "ftp") == 0)
        {
          SshFtpFilter filter_ctx;

          filter_ctx = ssh_xcalloc(1, sizeof (*filter_ctx));
          filter_ctx->local_fwd = NULL;
          filter_ctx->remote_fwd = c->fwd;
          filter_ctx->common = c->fwd->common;
          filter_ctx->remote_fwd_creation_in_progress = FALSE;
          stream = ssh_stream_filter_create(stream,
                                            1024,
                                            ssh_ftp_output_filter,
                                            ssh_ftp_input_filter,
                                            ssh_ftp_filter_destroy,
                                            (void *)filter_ctx);
        }
      else if (strcasecmp(c->fwd->protocol, "ftp-data") == 0)
        {
          /* This tunnel is a dynamically created ftp data channel. The
             forwarding is not needed now that the tunnel is running, so
             send a request to destroy it. */
          ssh_channel_cancel_remote_tcp_forward(c->fwd->common,
                                                c->fwd->address_to_bind,
                                                c->fwd->port, NULL_FNPTR,
                                                NULL);
        }
#endif /* SSHDIST_SSH2_FTP_FORWARDING */
      else
        {
          ssh_warning("forwarding protocol \"%s\" defaults to \"tcp\"",
                      c->fwd->protocol);
        }
    }

  SSH_DEBUG(0, ("remote open host=\"%s\" port=\"%s\" proto=\"%s\"",
                c->fwd->connect_to_host,
                c->fwd->connect_to_port,
                c->fwd->protocol));

  /* Record that we now have a new channel. */
  ssh_common_new_channel(c->fwd->common);

  /* init the new tcp forward struct */
  active_fwd_context = ssh_xcalloc(1, sizeof(*active_fwd_context));

  active_fwd_context->connect_from_host = ssh_xstrdup(c->originator_ip);
  active_fwd_context->port = ssh_xstrdup(c->fwd->port);
  active_fwd_context->connect_to_port = ssh_xstrdup(c->fwd->connect_to_port);
  active_fwd_context->connect_to_host = ssh_xstrdup(c->fwd->connect_to_host);
  active_fwd_context->common = c->fwd->common;
  active_fwd_context->channel_type = SSH_TFACT_REMOTE;

  /* add the channel to the active list in the common object */
  ssh_common_add_active_forward(active_fwd_context);

  /* Call the completion procedure to indicate that we are done. */
  (*c->completion)(SSH_OPEN_OK,
                   stream, TRUE, TRUE, SSH_TCPIP_WINDOW, NULL, 0,
                   NULL_FNPTR, ssh_common_delete_active_forward,
                   (void *)active_fwd_context, c->completion_context);

  if (c->originator_ip != NULL)
    ssh_xfree(c->originator_ip);

  memset(c, 'F', sizeof(*c));
  ssh_xfree(c);
}

/* Processes an open request for a remote-forwarded TCP/IP channel. */

void ssh_channel_ftcp_open_request(const char *type, int channel_id,
                                   const unsigned char *data, size_t len,




                                   SshConnOpenCompletionProc completion,

                                   void *completion_context, void *context)
{
  SshCommon common = (SshCommon)context;
  SshUInt32 port, originator_port;
  char *address_to_bind, *originator_ip;
  char port_string[20];
  SshRemoteTcpForward fwd;
  SshChannelTcpFwdConnect c;
  SshChannelTypeTcpForward ct;

  SSH_DEBUG(2, ("open request for remote forwarded TCP/IP channel"));

  ct = ssh_channel_ftcp_ct(common);

  if (ssh_decode_array(data, len,
                       SSH_FORMAT_UINT32_STR, &address_to_bind, NULL,
                       SSH_FORMAT_UINT32, &port,
                       SSH_FORMAT_UINT32_STR, &originator_ip, NULL,
                       SSH_FORMAT_UINT32, &originator_port,
                       SSH_FORMAT_END) != len || len == 0)
    {
      /* XXX should disconnect? */
      SSH_DEBUG(0, ("bad data"));
      (*completion)(SSH_OPEN_RESOURCE_SHORTAGE,
                    NULL, FALSE, FALSE, 0, NULL, 0, NULL_FNPTR, NULL_FNPTR,
                    NULL, completion_context);
      return;
    }

  ssh_snprintf(port_string, sizeof(port_string), "%ld", (long) port);

  ssh_debug("Received remote TCP/IP forward connect for port %s from %s:%ld",
            port_string, originator_ip, (long)originator_port);

  for (fwd = ct->remote_forwards; fwd; fwd = fwd->next)
    if (strcmp(fwd->address_to_bind, address_to_bind) == 0 &&
        strcmp(fwd->port, port_string) == 0)
      {
        c = ssh_xcalloc(1, sizeof(*c));
        c->fwd = fwd;
        c->channel_id = channel_id;
        c->completion = completion;
        c->completion_context = completion_context;
        c->originator_ip = ssh_xstrdup(originator_ip);
        ssh_tcp_connect(fwd->connect_to_host, fwd->connect_to_port,
                        NULL, ssh_channel_ftcp_open_connected,
                        (void *)c);

        ssh_xfree(address_to_bind);
        ssh_xfree(originator_ip);
        return;
      }

  ssh_warning("Received remote TCP/IP connect for non-forwarded port %s "
              "from %s:%ld",
              port_string, originator_ip, (long)originator_port);

  ssh_xfree(address_to_bind);
  ssh_xfree(originator_ip);

  (*completion)(SSH_OPEN_ADMINISTRATIVELY_PROHIBITED,
                NULL, FALSE, FALSE, 0, NULL, 0, NULL_FNPTR, NULL_FNPTR, NULL,
                completion_context);
}

/***********************************************************************
 * Processing a channel open request for a direct tcp/ip connection
 * to some port.  This is typically used for local forwards.
 ***********************************************************************/

typedef struct {
  SshCommon common;
  int channel_id;
  char *connect_to_host;
  char *connect_to_host_ip;
  char *connect_to_port;
  char *originator_ip;
  long originator_port;




  SshConnOpenCompletionProc completion;

  void *completion_context;
} *SshDirectTcp;

/* Called when connecting to the real destination port is complete. */

void ssh_channel_dtcp_connected(SshTcpError error,
                                SshStream stream, void *context)
{
  SshDirectTcp tcp = (SshDirectTcp)context;

  SSH_DEBUG(5, ("direct connected: %d", (int)error));

  /* Check result. */
  if (error != SSH_TCP_OK)
    {
      /* Connection failed. */
      (*tcp->completion)(SSH_OPEN_CONNECT_FAILED,
                         NULL, FALSE, FALSE, 0, NULL, 0, NULL_FNPTR,
                         NULL_FNPTR, NULL,
                         tcp->completion_context);

      ssh_xfree(tcp);
      return;
    }

  /* Record that we have a new channel. */
  ssh_common_new_channel(tcp->common);

  /* Connection was successful.  Establish the channel. */
  (*tcp->completion)(SSH_OPEN_OK,
                     stream, TRUE, TRUE, SSH_TCPIP_WINDOW, NULL, 0,
                     NULL_FNPTR, ssh_channel_tcp_connection_destroy,
                     (void *)tcp->common, tcp->completion_context);
  ssh_xfree(tcp);
}

#ifndef SSH_WIN_CLIENT
/* Processes an open request for a TCP/IP forwarding to given address. */









































































void dtcp_finalize_open_proc(SshDirectTcp tcp)
{
  char *host_to_connect, *port_to_connect;

  ssh_debug("Direct TCP/IP connect to %s:%s from %s:%ld",
            tcp->connect_to_host, tcp->connect_to_port, tcp->originator_ip,
            (long)tcp->originator_port);
  ssh_xfree(tcp->originator_ip);
  /* These variables (or pointers to them) have to copied away, as the
     callback to ssh_tcp_connect() might get called _immediately_ during
     the call to ssh_tcp_connect(). */
  host_to_connect = tcp->connect_to_host;
  port_to_connect = tcp->connect_to_port;
  ssh_xfree(tcp->connect_to_host_ip);
  /* Connect to the given host/port. */
  ssh_tcp_connect(host_to_connect, port_to_connect,
                  NULL, ssh_channel_dtcp_connected,
                  (void *)tcp);
  ssh_xfree(host_to_connect);
  ssh_xfree(port_to_connect);
}

void dtcp_addr_lookup_cb(SshTcpError error, const char *result, void *context)
{
  SshDirectTcp tcp = (SshDirectTcp) context;
  SshConfig config = tcp->common->config;
  SSH_PRECOND(config != NULL);
  
  if (error == SSH_TCP_OK)
    {
      SSH_DEBUG(3, ("lookup result is \"%s\".", result));
      if (tcp->connect_to_host_ip != NULL)
        tcp->connect_to_host = ssh_xstrdup(result);
      else
        tcp->connect_to_host_ip = ssh_xstrdup(result);
    }
  else
    {
      if (tcp->connect_to_host_ip != NULL)
        {
          if (config->try_reverse_mapping)
            {
              ssh_warning("DNS lookup failed for \"%s\".",
                          tcp->connect_to_host_ip);
            }
          else
            {
              SSH_DEBUG(2, ("DNS lookup disabled in the configuration"));
            }
          tcp->connect_to_host = ssh_xstrdup(tcp->connect_to_host_ip);
        }
      else
        {
          SSH_DEBUG(2, ("Hostname `%s' didn't resolve to an IP-address.",
                        tcp->connect_to_host));
          tcp->connect_to_host_ip = ssh_xstrdup("NO_SUCH_IP");
        }
    }
  
  if (is_tcp_forwarding_denied_for_user(tcp->common, TRUE,
                                        tcp->connect_to_host,
                                        tcp->connect_to_host_ip,
                                        tcp->connect_to_port,
                                        tcp->originator_ip))
    {
      /* Denied. */
      ssh_log_event(tcp->common->config->log_facility,
                    SSH_LOG_INFORMATIONAL,
                    "Direct TCP/IP forwarding request (%s:%s) "
                    "denied for user in configuration.", tcp->connect_to_host,
                    tcp->connect_to_port);
      (*tcp->completion)(SSH_OPEN_ADMINISTRATIVELY_PROHIBITED,
                    NULL, FALSE, FALSE, 0, NULL, 0, NULL_FNPTR, NULL_FNPTR,
                    NULL, tcp->completion_context);
      return;
    }

  dtcp_finalize_open_proc(tcp);
}

void ssh_channel_dtcp_open_request(const char *type, int channel_id,
                                   const unsigned char *data, size_t len,




                                   SshConnOpenCompletionProc completion,

                                   void *completion_context, void *context)
{
  SshCommon common = (SshCommon)context;
  char *connect_to_host, *connect_to_port, *originator_ip;
  SshUInt32 port, originator_port;
  SshDirectTcp tcp;








  SSH_DEBUG(5, ("direct TCP/IP channel open request"));

  /* Parse packet data. */
  if (ssh_decode_array(data, len,
                       SSH_FORMAT_UINT32_STR, &connect_to_host, NULL,
                       SSH_FORMAT_UINT32, &port,
                       SSH_FORMAT_UINT32_STR, &originator_ip, NULL,
                       SSH_FORMAT_UINT32, &originator_port,
                       SSH_FORMAT_END) != len || len == 0)
    {
      /* XXX disconnect? */
      SSH_DEBUG(0, ("bad data"));
      (*completion)(SSH_OPEN_ADMINISTRATIVELY_PROHIBITED,
                    NULL, FALSE, FALSE, 0, NULL, 0, NULL_FNPTR, NULL_FNPTR,
                    NULL, completion_context);
      return;
    }

  /* We do not currently allow direct connections from server to client. */
  if (common->client)
    {
      ssh_warning("Direct TCP/IP connection request from server "
                  "to %s:%ld denied.",
                  connect_to_host, (long)port);
      /* Free dynamically allocated data. */
      ssh_xfree(originator_ip);
      ssh_xfree(connect_to_host);
      (*completion)(SSH_OPEN_ADMINISTRATIVELY_PROHIBITED,
                    NULL, FALSE, FALSE, 0, NULL, 0, NULL_FNPTR, NULL_FNPTR,
                    NULL, completion_context);
      return;
    }




































    {
      if (!common->config->allow_tcp_forwarding)
        {
          ssh_log_event(common->config->log_facility,
                        SSH_LOG_INFORMATIONAL,
                        "Direct TCP/IP forwarding request "
                        "denied in configuration.");
          (*completion)(SSH_OPEN_ADMINISTRATIVELY_PROHIBITED,
                        NULL, FALSE, FALSE, 0, NULL, 0, NULL_FNPTR, NULL_FNPTR,
                        NULL, completion_context);
          return;
        }
    }

  /* Convert port number to string. */
  ssh_dsprintf(&connect_to_port, "%ld", (long) port);

  /* Save data for callback. */
  tcp = ssh_xcalloc(1, sizeof(*tcp));
  tcp->common = common;
  if (ssh_inet_is_valid_ip_address(connect_to_host))
    tcp->connect_to_host_ip = connect_to_host;
  else
    tcp->connect_to_host = connect_to_host;
  
  tcp->connect_to_port = connect_to_port;
  tcp->originator_ip = originator_ip;
  tcp->originator_port = originator_port;
  tcp->channel_id = channel_id;
  tcp->completion = completion;
  tcp->completion_context = completion_context;












    {
      if (common->config->try_reverse_mapping &&
          tcp->connect_to_host_ip != NULL)
        {
          /* Look up host name. */
          ssh_tcp_get_host_by_addr(tcp->connect_to_host_ip,
                                   dtcp_addr_lookup_cb,
                                   tcp);
        }
      else
        {
          if (tcp->connect_to_host_ip != NULL)
            /* We don't want reverse mapping, use forward target host
               IP-address as host name. */
            dtcp_addr_lookup_cb(SSH_TCP_FAILURE, tcp->connect_to_host_ip, tcp);
          else
            /* Whether we want to reverse map or not, we want to find
               out the forward target's IP-address. */
            ssh_tcp_get_host_addrs_by_name(tcp->connect_to_host,
                                           dtcp_addr_lookup_cb, tcp);
        }
    }
}

#endif /* SSH_WIN_CLIENT */

/***********************************************************************
 * Processing an incoming connection to a remotely forwarded socket.
 ***********************************************************************/

/* This function is called whenever a connection is received at a remotely
   forwarded socket.  This sends a channel open request to the other
   side. */

void ssh_channel_ftcp_incoming_connection(SshTcpError error, SshStream stream,
                                          void *context)
{
  SshRemoteTcpForward fwd = (SshRemoteTcpForward)context;
  char ip[20], port[20];
  SshBufferStruct buffer;

  SSH_DEBUG(5, ("connection to forwarded TCP/IP port"));

  /* We should only receive new connection notifications. */
  if (error != SSH_TCP_NEW_CONNECTION)
    ssh_fatal("ssh_channel_ftcp_incoming_connection: error %d", (int)error);

  /* Get remote ip address and port. */
  if (!ssh_tcp_get_remote_address(stream, ip, sizeof(ip)))
    strcpy(ip, "UNKNOWN");
  if (!ssh_tcp_get_remote_port(stream, port, sizeof(port)))
    strcpy(port, "UNKNOWN");

  ssh_debug("Connection to forwarded port %s from %s:%s",
            fwd->port, ip, port);
  ssh_log_event(fwd->common->config->log_facility,
                SSH_LOG_INFORMATIONAL,
                "Connection to forwarded port %s from %s:%s",
                fwd->port, fwd->common->remote_host, port);

#ifdef HAVE_LIBWRAP
  {
    struct request_info req;
    struct servent *serv;
    char fwdportname[32];
    static RETSIGTYPE (*old_handler)(int sig) = NULL;

    old_handler = signal(SIGCHLD, SIG_DFL);

    if (old_handler == SIG_ERR)
      {
        ssh_warning("ssh_channel_ftcp_incoming_connection: Could not set "
                    "SIGCHLD signal handler.");
        old_handler = SIG_IGN;
      }

    /* try to find port's name in /etc/services */
    serv = getservbyport(atoi(fwd->port), "tcp");
    if (serv == NULL)
      {
        /* not found (or faulty getservbyport) -
           use the number as a name */
        ssh_snprintf(fwdportname, sizeof(fwdportname), "sshdfwd-%s",
                     fwd->port);
      }
    else
      {
        ssh_snprintf(fwdportname, sizeof(fwdportname), "sshdfwd-%.20s",
                     serv->s_name);
      }
    /* fill req struct with port name and fd number */
    request_init(&req, RQ_DAEMON, fwdportname,
                 RQ_FILE, ssh_stream_fd_get_readfd(stream), NULL);
    fromhost(&req);
    if (!hosts_access(&req))
      {
        ssh_conn_send_debug(fwd->common->conn, TRUE,
                            "Fwd connection from %.500s to local port "
                            "%s refused by tcp_wrappers.",
                            eval_client(&req), fwdportname);
        ssh_stream_destroy(stream);
        signal(SIGCHLD, old_handler);

        return;
      }
    if (signal(SIGCHLD, old_handler) == SIG_ERR && old_handler != SIG_IGN)
      {
        ssh_warning("ssh_channel_ftcp_incoming_connection: Could not reset "
                    "SIGCHLD signal handler.");
      }

    ssh_log_event(fwd->common->config->log_facility, SSH_LOG_INFORMATIONAL,
                  "Remote fwd connect from %.500s to local port %s",
                  eval_client(&req), fwdportname);
  }
#endif /* HAVE_LIBWRAP */

#ifndef SSH_WIN_CLIENT
  if (is_tcp_forwarding_denied_for_user(fwd->common, FALSE,
                                        fwd->address_to_bind,
                                        fwd->address_to_bind,
                                        fwd->port, ip))
    {
      ssh_conn_send_debug(fwd->common->conn, TRUE,
                          "Fwd connection from %.500s to local port "
                          "%s refused by forward ACLs.", ip, fwd->port);
      ssh_stream_destroy(stream);
      return;
    }
#endif /* SSH_WIN_CLIENT */

  /* Register that we have an open channel. */
  ssh_common_new_channel(fwd->common);

  /* Send a request to open a channel and connect it to the given port. */
  ssh_buffer_init(&buffer);
  ssh_encode_buffer(&buffer,
                    SSH_FORMAT_UINT32_STR,
                    fwd->address_to_bind, strlen(fwd->address_to_bind),
                    SSH_FORMAT_UINT32, (SshUInt32) atol(fwd->port),
                    SSH_FORMAT_UINT32_STR, ip, strlen(ip),
                    SSH_FORMAT_UINT32, (SshUInt32) atol(port),
                    SSH_FORMAT_END);










  ssh_conn_send_channel_open(fwd->common->conn, "forwarded-tcpip",
                             stream, TRUE, FALSE, SSH_TCPIP_WINDOW,
                             SSH_TCPIP_PACKET_SIZE,
                             ssh_buffer_ptr(&buffer), ssh_buffer_len(&buffer),
                             NULL_FNPTR,
                             ssh_channel_tcp_connection_destroy,
                             (void *)fwd->common, NULL_FNPTR, NULL);

  ssh_buffer_uninit(&buffer);
}

#ifndef SSH_WIN_CLIENT

/***********************************************************************
 * Processes a request to set up TCP/IP forwarding.  This is typically
 * used in the server.
 ***********************************************************************/

/* Processes a received request to set up remote TCP/IP forwarding. */
typedef struct RemoteFwdCompletionCtxRec
{
  char *address_to_bind;
  SshUInt32 port;
  SshCommon common;



  SshConnGlobalRequestCompletionProc completion_cb;

  void *req_completion_context;
} RemoteFwdCompletionCtxStruct, *RemoteFwdCompletionCtx;

void remote_tcp_forward_request_completion_proc(Boolean success,
                                                RemoteFwdCompletionCtx ctx)
{
  SshRemoteTcpForward fwd;
  SshChannelTypeTcpForward ct;
  SshCommon common;
  char *port_string = NULL;

  SSH_PRECOND(ctx != NULL);
  SSH_PRECOND(ctx->common != NULL);
  SSH_PRECOND(ctx->address_to_bind != NULL);

  common = ctx->common;

  SSH_DEBUG(3, ("success: %s", success ? "TRUE" : "FALSE"));

  if (!success)
    {
      (*ctx->completion_cb)(FALSE, ctx->req_completion_context);
      goto free_and_return;
    }

  /* Convert port number to a string. */
  ssh_dsprintf(&port_string, "%ld", (unsigned long) ctx->port);

  ct = ssh_channel_ftcp_ct(common);

  /* Create a socket listener. */
  fwd = ssh_xcalloc(1, sizeof(*fwd));
  fwd->listener = ssh_tcp_make_listener(ctx->address_to_bind, port_string,
                                        NULL,
                                        ssh_channel_ftcp_incoming_connection,
                                        (void *)fwd);
  if (fwd->listener == NULL)
    {
      ssh_debug("Creating remote listener for %s:%s failed.",
                ctx->address_to_bind, port_string);
      ssh_log_event(common->config->log_facility,
                    SSH_LOG_NOTICE,
                    "Creating remote listener for %s:%s failed.",
                    ctx->address_to_bind, port_string);

      (*ctx->completion_cb)(FALSE, ctx->req_completion_context);

      ssh_xfree(fwd);
      goto free_and_return;
    }

  /* Fill the remaining fields. */
  fwd->common = common;
  fwd->address_to_bind = ctx->address_to_bind;
  ctx->address_to_bind = NULL;
  fwd->port = ssh_xstrdup(port_string);
  fwd->connect_to_host = NULL;
  fwd->connect_to_port = NULL;
  fwd->protocol = NULL;

  /* Add to list of forwardings. */
  fwd->next = ct->remote_forwards;
  ct->remote_forwards = fwd;

  ssh_log_event(common->config->log_facility,
                SSH_LOG_INFORMATIONAL,
                "Port %ld set up for remote forwarding.",
                (unsigned long) ctx->port);

  (*ctx->completion_cb)(TRUE, ctx->req_completion_context);
 free_and_return:
  ssh_xfree(ctx->address_to_bind);
  ssh_xfree(ctx);
}


























































void
ssh_channel_remote_tcp_forward_request(const char *type,
                                       const unsigned char *data,
                                       size_t len,




                                       SshConnGlobalRequestCompletionProc
                                       completion,

                                       void *req_completion_context,
                                       void *context)
{
  SshCommon common = (SshCommon)context;
  char *address_to_bind = NULL;
  SshUInt32 port;
  RemoteFwdCompletionCtx completion_ctx = NULL;







  SSH_DEBUG(5, ("remote TCP/IP forwarding request received"));
  ssh_log_event(common->config->log_facility,
                SSH_LOG_INFORMATIONAL,
                "Remote TCP/IP forwarding request received from host \"%s\", "
                "by authenticated user \"%s\".",
                common->remote_host,
                ssh_user_name(common->user_data));

#define SEND_FAILURE goto failure

  /* Don't allow a server to send remote forwarding requests to the client. */
  if (common->client)
    {
      ssh_warning("Remote TCP/IP forwarding request from server denied.");
      SEND_FAILURE;
    }

  /* Parse the request. */
  if (ssh_decode_array(data, len,
                       SSH_FORMAT_UINT32_STR, &address_to_bind, NULL,
                       SSH_FORMAT_UINT32, &port,
                       SSH_FORMAT_END) != len || len == 0)
    {
      SSH_DEBUG(0, ("bad data"));
      address_to_bind = NULL;
      SEND_FAILURE;
    }

  /* If user is not logged in as a privileged user, don't allow
     forwarding of privileged ports. */
  if (port < 1024)
    {
      if (ssh_user_uid(common->user_data))
        {
          ssh_warning("User \"%s\" not root, tried to forward privileged "
                      "port %ld.", ssh_user_name(common->user_data),
                      (unsigned long) port);
          SEND_FAILURE;
        }
      else
        {
          ssh_log_event(common->config->log_facility,
                        SSH_LOG_NOTICE,
                        "Privileged user \"%s\" forwarding a privileged port.",
                        ssh_user_name(common->user_data));
        }
    }

  if (port >= 65536)
    {
      ssh_warning("User \"%s\" tried to forward port above 65535 (%ld).",
                  ssh_user_name(common->user_data), (unsigned long) port);
      SEND_FAILURE;
    }



































    {
      char *port_str;
      
      if (!common->config->allow_tcp_forwarding)
        {
          ssh_log_event(common->config->log_facility,
                        SSH_LOG_INFORMATIONAL,
                        "Remote TCP/IP forwarding request "
                        "denied in configuration.");
          SEND_FAILURE;
        }

      ssh_xdsprintf(&port_str, "%ld", port);
      if (is_tcp_forwarding_denied_for_user(common, FALSE, address_to_bind,
                                            address_to_bind, port_str, NULL))
        {
          /* Denied. */
          ssh_log_event(common->config->log_facility,
                        SSH_LOG_INFORMATIONAL,
                        "Remote TCP/IP forwarding (%s:%s) request "
                        "denied for user in configuration.",
                        address_to_bind, port_str);
          ssh_xfree(port_str);
          SEND_FAILURE;
        }
      ssh_xfree(port_str);
    }

  completion_ctx = ssh_xcalloc(1, sizeof(*completion_ctx));
  completion_ctx->address_to_bind = address_to_bind;
  completion_ctx->port = port;
  completion_ctx->common = common;
  completion_ctx->completion_cb = completion;
  completion_ctx->req_completion_context = req_completion_context;












    {
      remote_tcp_forward_request_completion_proc(TRUE, completion_ctx);
    }

  return;

 failure:
  ssh_xfree(address_to_bind);
  (*completion)(FALSE, req_completion_context);
#undef SEND_FAILURE
}

/* Processes a received request to cancel remote TCP/IP forwarding. */

void
ssh_channel_tcp_forward_cancel(const char *type,
                               const unsigned char *data,
                               size_t len,



                               SshConnGlobalRequestCompletionProc completion,

                               void *completion_context,
                               void *context)
{
  SshCommon common = (SshCommon)context;
  char *address_to_bind;
  SshUInt32 port;
  char port_string[20];
  SshRemoteTcpForward fwd, *fwdp;
  SshChannelTypeTcpForward ct;

  SSH_DEBUG(5, ("remote TCP/IP cancel request received"));

  ct = ssh_channel_ftcp_ct(common);

#define SEND_FAILURE                            \
  do {                                          \
    (*completion)(FALSE, completion_context);   \
    return;                                     \
  }while(0)

  /* Don't allow a server to send remote forwarding requests to the client. */
  if (common->client)
    {
      ssh_warning("Remote TCP/IP forwarding cancel from server denied.");
      SEND_FAILURE;
    }

  /* Parse the request. */
  if (ssh_decode_array(data, len,
                       SSH_FORMAT_UINT32_STR, &address_to_bind, NULL,
                       SSH_FORMAT_UINT32, &port,
                       SSH_FORMAT_END) != len || len == 0)
    {
      SSH_DEBUG(0, ("bad data"));
      SEND_FAILURE;
    }

  /* Convert port number to a string. */
  ssh_snprintf(port_string, sizeof(port_string), "%ld", (unsigned long) port);

  for (fwdp = &ct->remote_forwards; *fwdp; fwdp = &fwd->next)
    {
      fwd = *fwdp;
      if (strcmp(port_string, fwd->port) == 0 &&
          strcmp(address_to_bind, fwd->address_to_bind) == 0)
        {
          ssh_tcp_destroy_listener(fwd->listener);
          ssh_xfree(fwd->address_to_bind);
          ssh_xfree(fwd->port);
          ssh_xfree(fwd->protocol);
          *fwdp = fwd->next;
          ssh_xfree(fwd);
          ssh_xfree(address_to_bind);
          (*completion)(TRUE, completion_context);
          return;
        }
    }

  SSH_DEBUG(1, ("port %s address_to_bind %s not found",
                port_string, address_to_bind));
  ssh_xfree(address_to_bind);
  SEND_FAILURE;
#undef SEND_FAILURE
}
#endif /* SSH_WIN_CLIENT */

/***********************************************************************
 * Sending a request to start remote TCP/IP forwarding.
 ***********************************************************************/

/* Requests forwarding of the given remote TCP/IP port.  If the completion
   procedure is non-NULL, it will be called when done. */

void ssh_channel_start_remote_tcp_forward(SshCommon common,
                                          const char *address_to_bind,
                                          const char *port,
                                          const char *connect_to_host,
                                          const char *connect_to_port,
                                          const char *protocol,
                                          void (*completion)(Boolean ok,
                                                             void *context),
                                          void *context)
{
  SshRemoteTcpForward fwd;
  SshBufferStruct buffer;
  SshChannelTypeTcpForward ct;

  SSH_DEBUG(2, ("requesting remote forwarding for port %s", port));

  ct = ssh_channel_ftcp_ct(common);

  /* Create a context for the forwarding. */
  fwd = ssh_xcalloc(1, sizeof(*fwd));
  fwd->common = common;
  fwd->address_to_bind = ssh_xstrdup(address_to_bind);
  fwd->port = ssh_xstrdup(port);
  fwd->connect_to_host = ssh_xstrdup(connect_to_host);
  fwd->connect_to_port = ssh_xstrdup(connect_to_port);
  fwd->protocol = ssh_xstrdup(protocol ? protocol : "tcp");
  /* Add it to the list of remote forwardings. */
  fwd->next = ct->remote_forwards;
  ct->remote_forwards = fwd;

  /* Send a forwarding request to the remote side. */
  ssh_buffer_init(&buffer);
  ssh_encode_buffer(&buffer,
                    SSH_FORMAT_UINT32_STR,
                      address_to_bind, strlen(address_to_bind),
                    SSH_FORMAT_UINT32, (SshUInt32) atol(port),
                    SSH_FORMAT_END);






  ssh_conn_send_global_request(common->conn, "tcpip-forward",
                               ssh_buffer_ptr(&buffer),
                               ssh_buffer_len(&buffer),
                               completion, context);

  ssh_buffer_uninit(&buffer);
}

/* Requests cancellation of the given remote TCP/IP port forwarding.
   If the completion procedure is non-NULL, it will be called when
   done. */

void ssh_channel_cancel_remote_tcp_forward(SshCommon common,
                                           const char *address_to_bind,
                                           const char *port,
                                           void (*completion)(Boolean ok,
                                                              void *context),
                                           void *context)
{
  SshBufferStruct buffer;

  SSH_DEBUG(5, ("cancelling remote forwarding for port %s", port));

  /* Send a forwarding request to the remote side. */
  ssh_buffer_init(&buffer);
  ssh_encode_buffer(&buffer,
                    SSH_FORMAT_UINT32_STR,
                      address_to_bind, strlen(address_to_bind),
                    SSH_FORMAT_UINT32, (SshUInt32) atol(port),
                    SSH_FORMAT_END);






  ssh_conn_send_global_request(common->conn, "cancel-tcpip-forward",
                               ssh_buffer_ptr(&buffer),
                               ssh_buffer_len(&buffer),
                               completion, context);

  ssh_buffer_uninit(&buffer);
}

/***********************************************************************
 * Handling incoming connections to a locally forwarded port
 ***********************************************************************/

#ifdef SSHDIST_SSH2_SOCKS_FILTER
void socks_filter_completion(void *context)
{
  SshTcpForwardActive active_fwd_context = (SshTcpForwardActive) context;
  active_fwd_context->handle = NULL;
}
#endif /* SSHDIST_SSH2_SOCKS_FILTER */

/* This function is called whenever a locally forwarded TCP/IP port is
   connected. */

void ssh_channel_dtcp_incoming_connection(SshTcpError op,
                                          SshStream stream, void *context)
{
  SshLocalTcpForward fwd = (SshLocalTcpForward)context;
  char ip[20], port[20];
  SshTcpForwardActive active_fwd_context;

  /* We should only receive new connection notifications. */
  if (op != SSH_TCP_NEW_CONNECTION)
    ssh_fatal("ssh_channel_dtcp_incoming_connection: op %d", (int)op);

  /* Get remote ip address and port. */
  if (!ssh_tcp_get_remote_address(stream, ip, sizeof(ip)))
    strcpy(ip, "UNKNOWN");
  if (!ssh_tcp_get_remote_port(stream, port, sizeof(port)))
    strcpy(port, "UNKNOWN");

#ifdef HAVE_LIBWRAP
  {
    struct request_info req;
    struct servent *serv;
    char fwdportname[32];
    static RETSIGTYPE (*old_handler)(int sig) = NULL;

    old_handler = signal(SIGCHLD, SIG_DFL);

    if (old_handler == SIG_ERR)
      {
        ssh_warning("ssh_channel_dtcp_incoming_connection: Could not set "
                    "SIGCHLD signal handler.");
        old_handler = SIG_IGN;
      }

    /* try to find port's name in /etc/services */
    serv = getservbyport(atoi(fwd->port), "tcp");
    if (serv == NULL)
      {
        /* not found (or faulty getservbyport) -
           use the number as a name */
        ssh_snprintf(fwdportname, sizeof(fwdportname), "sshdfwd-%s",
                     fwd->port);
      }
    else
      {
        ssh_snprintf(fwdportname, sizeof(fwdportname), "sshdfwd-%.20s",
                     serv->s_name);
      }
    /* fill req struct with port name and fd number */
    request_init(&req, RQ_DAEMON, fwdportname,
                 RQ_FILE, ssh_stream_fd_get_readfd(stream), NULL);
    fromhost(&req);
    if (!hosts_access(&req))
      {
        ssh_conn_send_debug(fwd->common->conn, TRUE,
                            "Fwd connection from %.500s to local port %s "
                            "refused by tcp_wrappers.",
                            eval_client(&req), fwdportname);
        ssh_warning("Fwd connection from %.500s to local port %s "
                    "refused by tcp_wrappers.",
                    eval_client(&req), fwdportname);
        ssh_stream_destroy(stream);
        signal(SIGCHLD, old_handler);

        return;
      }

    if (signal(SIGCHLD, old_handler) == SIG_ERR && old_handler != SIG_IGN)
      {
        ssh_warning("ssh_channel_dtcp_incoming_connection: Could not reset "
                    "SIGCHLD signal handler.");
      }

    ssh_log_event(fwd->common->config->log_facility, SSH_LOG_INFORMATIONAL,
                  "direct fwd connect from %.500s to local port %s",
                  eval_client(&req), fwdportname);
  }
#endif /* HAVE_LIBWRAP */

  /* Initialize the tcp forward context. This provides us with
     identity of the channel when the connection is closed.  We have
     to add this to the list now, since in dtcp_open_to_remote we lose
     some information (the local port the connection is going
     through). */

  active_fwd_context = ssh_xcalloc(1, sizeof(*active_fwd_context));

  active_fwd_context->connect_from_host = ssh_xstrdup(ip);
  active_fwd_context->port = ssh_xstrdup(fwd->port);
  active_fwd_context->connect_to_host = ssh_xstrdup(fwd->connect_to_host);
  active_fwd_context->connect_to_port = ssh_xstrdup(fwd->connect_to_port);
  active_fwd_context->common = fwd->common;
  active_fwd_context->channel_type = SSH_TFACT_LOCAL;

  if (strcasecmp(fwd->protocol, "tcp") != 0)
    {
      if (strcasecmp(fwd->protocol, "http") == 0)
        {
          ssh_warning("forwarding protocol \"http\" defaults to \"tcp\"");
        }
#ifdef SSHDIST_SSH2_FTP_FORWARDING
      else if (strcasecmp(fwd->protocol, "ftp") == 0)
        {
          SshFtpFilter filter_ctx;

          filter_ctx = ssh_xcalloc(1, sizeof (*filter_ctx));
          filter_ctx->local_fwd = fwd;
          filter_ctx->remote_fwd = NULL;
          filter_ctx->common = fwd->common;
          filter_ctx->remote_fwd_creation_in_progress = FALSE;
          stream = ssh_stream_filter_create(stream,
                                            1024,
                                            ssh_ftp_output_filter,
                                            ssh_ftp_input_filter,
                                            ssh_ftp_filter_destroy,
                                            (void *)filter_ctx);
        }
      else if (strcasecmp(fwd->protocol, "ftp-data") == 0)
        {
          /* This tunnel is a dynamically created ftp data channel and
             the listener needs to be destroyed right away. The rest
             of the fwd structure will be freed when the client exits. */
          ssh_tcp_destroy_listener(fwd->listener);
          fwd->listener = NULL;
        }
#endif /* SSHDIST_SSH2_FTP_FORWARDING */
#ifdef SSHDIST_SSH2_SOCKS_FILTER
      else if (strcasecmp(fwd->protocol, "socks") == 0)
        {
          active_fwd_context->handle =
            ssh_socks_filter_create(stream, fwd->common, active_fwd_context,
                                    socks_filter_completion,
                                    (void *)active_fwd_context);
          /* The SOCKS filter will handle the creation of forwards. */
          return;
        }
#endif /* SSHDIST_SSH2_SOCKS_FILTER */
      else
        {
          ssh_warning("forwarding protocol \"%s\" defaults to \"tcp\"",
                      fwd->protocol);
        }
    }

  SSH_DEBUG(0, ("local tunnel open, host=\"%s\" port=\"%s\" proto=\"%s\"",
                fwd->connect_to_host,
                fwd->connect_to_port,
                fwd->protocol));

  /* add active channel to the common object's active channel list */
  ssh_common_add_active_forward(active_fwd_context);

  /* Send a request to open a channel and connect it to the given port. */
  ssh_channel_dtcp_open_to_remote(fwd->common, stream,
                                  fwd->connect_to_host,
                                  fwd->connect_to_port,
                                  ip, port);
}

/***********************************************************************
 * Starting local TCP/IP forwarding for a port
 ***********************************************************************/

/* Requests forwarding of the given local TCP/IP port.  Returns TRUE if
   forwarding was successfully started, FALSE otherwise. */

Boolean ssh_channel_start_local_tcp_forward(SshCommon common,
                                            const char *address_to_bind,
                                            const char *port,
                                            const char *connect_to_host,
                                            const char *connect_to_port,
                                            const char *protocol)
{
  SshLocalTcpForward fwd;
  SshChannelTypeTcpDirect ct;
  long portnumber;
  SshUser user;

  SSH_DEBUG(2, ("requesting local forwarding for port %s to %s:%s",
                port, connect_to_host, connect_to_port));

  portnumber = atol(port);

#ifndef SSH_WIN_CLIENT
  user = ssh_user_initialize(NULL, FALSE);
    /* If user is not logged in as a privileged user, don't allow
     forwarding of privileged ports. */
  if (portnumber < 1024)
    {
      if (ssh_user_uid(user))
        {
          ssh_warning("Tried to forward privileged port %d as an ordinary "
                      "user.", portnumber);
          goto error;
        }
    }
#endif /* SSH_WIN_CLIENT */

  if (portnumber >= 65536)
    {
      ssh_warning("Tried to forward port above 65535 (%d).", portnumber);
      goto error;
    }

  ct = ssh_channel_dtcp_ct(common);

  fwd = ssh_xcalloc(1, sizeof(*fwd));
  fwd->common = common;
  fwd->listener = ssh_tcp_make_listener(address_to_bind, port,
                                        NULL,
                                        ssh_channel_dtcp_incoming_connection,
                                        (void *)fwd);
  if (!fwd->listener)
    {
      SSH_DEBUG(1, ("creating listener failed"));
      ssh_xfree(fwd);
      goto error;
    }

  fwd->port = ssh_xstrdup(port);
  fwd->connect_to_host = ssh_xstrdup(connect_to_host);
  fwd->connect_to_port = ssh_xstrdup(connect_to_port);
  fwd->protocol = ssh_xstrdup(protocol ? protocol : "tcp");

  fwd->next = ct->local_forwards;
  ct->local_forwards = fwd;

#ifndef SSH_WIN_CLIENT
  ssh_user_free(user, FALSE);
#endif /* SSH_WIN_CLIENT */
  return TRUE;

 error:
#ifndef SSH_WIN_CLIENT
  ssh_user_free(user, FALSE);
#endif /* SSH_WIN_CLIENT */
  return FALSE;
}

/***********************************************************************
 * Sending a direct open request to a TCP/IP port from the remote side.
 * This is direct forwarding, and is primarily used for local
 * forwardings.
 ***********************************************************************/

/* Opens a direct connection to the given TCP/IP port at the remote side.
   The originator values should be set to useful values and are passed
   to the other side.  ``stream'' will be used to transfer channel data.
   The stream will be closed when the channel is closed, or if opening
   the channel fails. */

void ssh_channel_dtcp_open_to_remote(SshCommon common, SshStream stream,
                                     const char *connect_to_host,
                                     const char *connect_to_port,
                                     const char *originator_ip,
                                     const char *originator_port)
{
  SshBufferStruct buffer;
  void* active_forward;

  SSH_DEBUG(5, ("opening direct TCP/IP connection to %s:%s originator %s:%s",
                connect_to_host, connect_to_port,
                originator_ip, originator_port));

  /* Register that we have a new channel. */
  ssh_common_new_channel(common);

  /* Format the channel open request in a buffer. */
  ssh_buffer_init(&buffer);
  ssh_encode_buffer(&buffer,
                    SSH_FORMAT_UINT32_STR,
                      connect_to_host, strlen(connect_to_host),
                    SSH_FORMAT_UINT32, (SshUInt32) atol(connect_to_port),
                    SSH_FORMAT_UINT32_STR,
                      originator_ip, strlen(originator_ip),
                    SSH_FORMAT_UINT32, (SshUInt32) atol(originator_port),
                    SSH_FORMAT_END);

  /* Since we've just added the active forward to the end of the
     active forward list (in dtcp_incoming_connection) we take this
     struct to pass to the callback function */

  SSH_ASSERT(ssh_adt_num_objects(common->active_local_forwards) > 0);

  /* copy the last item for the callback */
  active_forward =
    ssh_adt_get_object_from_location(common->active_local_forwards,
                                     SSH_ADT_END);

  /* Send the channel open request. */










  ssh_conn_send_channel_open(common->conn, "direct-tcpip",
                             stream, TRUE, FALSE, SSH_TCPIP_WINDOW,
                             SSH_TCPIP_PACKET_SIZE,
                             ssh_buffer_ptr(&buffer), ssh_buffer_len(&buffer),
                             NULL_FNPTR,
                             ssh_common_delete_active_forward,
                             (void *)active_forward, NULL_FNPTR, NULL);

  ssh_buffer_uninit(&buffer);
}

#endif /* SSH_CHANNEL_TCPFWD */
