/*

  ssh-signer2.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1999-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Hostbased authentication, client-side. This program is supposed to
  be suid, as it should have access to the private part of the host
  key.

*/
/*
  ssh-signer2 communicates with the ssh2-client with a very simple protocol.

  First, ssh2 will fork and exec ssh-signer2. Then it wraps
  ssh-signer2's stdin and stdout to a sshpipestream. This stream is
  then wrapped to a sshpacketstream. Using this stream it is very easy
  to implement a packet-based protocol.

  Possible messages from ssh2 to ssh-signer2:

  #define SSH_AUTH_HOSTBASED_PACKET    (SshPacketType)1

    If this is received, ssh-signer2 begins to check for the packet's
    validity. As ssh-signer2 has (should have) access to the client
    host's hostkey, it shouldn't sign everything it
    receives. Particularly the client host's name and user's name in
    the client host are important. This packet MUST be the first, or
    the next right after SSH_AUTH_HOSTBASED_COMPAT packet.

    This messages payload is packet, which is formatted according to
    the ssh-userauth draft, under hostbased-authentication.

  #define SSH_AUTH_HOSTBASED_COMPAT    (SshPacketType)5

    This MUST be sent first, if at all. Packet payload is a
    comma-separated list of compatibility flags needed to interoperate
    with older versions of servers.

  #define SSH_AUTH_HOSTBASED_END       (SshPacketType)2

    This notifies ssh-signer2 that ssh2 client no longer needs ssh-signer2.
    This message has no payload.

  Possible messages from ssh-signer2 to ssh2:

  #define SSH_AUTH_HOSTBASED_SIGNATURE (SshPacketType)3

    This message is sent by ssh-signer2 to ssh2 when it has checked
    the packet and signed it. Payload is the signature.

  Common messages:

  #define SSH_AUTH_HOSTBASED_ERROR     (SshPacketType)4

    Payload:
        byte    error_code (type SshAuthHostbasedError)
        string  human readable error message
        string  language tag (as per [RFC-1776])

     After this packet nothing else should sent to the stream, and
     both should close it.
*/

#include "ssh2includes.h"
#include "sshpacketstream.h"
#include "ssheloop.h"
#include "sshfdstream.h"
#include "sshencode.h"
#include "sshhostkeyio.h"
#include "ssh-signer2.h"
#include "sshmsgs.h"
#include "sshauth.h"
#include "sshconfig.h"
#include "sshuserfiles.h"
#include "sshuserfile.h"
#include "sshtcp.h"
#include "sshnameserver.h"
#include "sshconfig.h"
#include "sshtimeouts.h"
#include "sshdsprintf.h"
#include "sshappcommon.h"
#include "sshsnlist.h"
#include "sshfsm.h"
#include "ssh2compat.h"
#ifdef HAVE_SYS_UTSNAME_H
#include <sys/utsname.h>
#endif /* HAVE_SYS_UTSNAME_H */
#include "sshglobals.h"




#define SSH_DEBUG_MODULE "SshSigner2"

/* Define this to get hexdumps */
/* #define HEXDUMPS */

/* Define this to get debug messages */
/* #define SIGNER_DEBUG */

/* Define this to make ssh-signer2 sleep for 30 seconds after it's
   start. (Only useful in debugging.)*/
/* #define SLEEP_AFTER_STARTUP */

/* Define this to make ssh-signer2 really quiet. */
#ifndef SIGNER_DEBUG
#define SIGNER_QUIET
#endif /* SIGNER_DEBUG */

/* Stored program name for ssh-signer (taken from argv[0]). */
static char *progname;

typedef enum
{
  SIGNER_START = 0,
  SIGNER_COMPAT_FLAGS_RECEIVED,
  SIGNER_HOSTBASED_PACKET_RECEIVED,
  SIGNER_DEAD
} SignerState;

typedef struct SignerCompatRec
{
  Boolean hostbased_requested_service_draft_incompat;
  Boolean signature_encode_draft_incompat;
  Boolean rsa_hash_scheme_draft_incompat;
  Boolean cert_rsa_hash_scheme_incompat;
} *SignerCompat;

typedef struct SshSignerRec
{
  /* Parameters for callbacks*/
  unsigned char *packet_payload;
  size_t packet_payload_len;

  SshPacketType packet_type;

  Boolean packet_pending;

  /* Internal stuff */
  SshConfig serv_config;
  SshConfig global_config;
  SshUser effective_user_data;
  SshUser real_user;
  Boolean quiet;
  SshPacketWrapper wrapper;
  SshFSM fsm;
  SshFSMThread main_thread;
  SignerState state;
  SshAuthHostBasedError error_code;

  /* Operation handle for the async signing operation. */
  SshOperationHandle op_handle;

  /* compat flags. */
  SignerCompat compat;

  char *pubkeytype;

  /* If we need to send an error message, this field will contain the
     "human readable error string". */
  char *error_message_to_ssh2;
  /* ... and this will contain the error code.  */
  SshAuthHostBasedError error_code_to_ssh2;
} *SshSigner;

/* Forward declarations. */
void signer_can_send(void *context);
void signer_hostkey_read_completion(SshHostKeyStatus success,
                                    void *context);
SSH_FSM_STEP(signer_suspend);
SSH_FSM_STEP(signer_process_error_packet);
SSH_FSM_STEP(signer_process_end);
SSH_FSM_STEP(signer_send_error);
SSH_FSM_STEP(signer_read_hostkeys);
SSH_FSM_STEP(signer_check_hostbased_packet);
SSH_FSM_STEP(signer_sign_hostbased_packet);
SSH_FSM_STEP(signer_process_compat_flags);
SSH_FSM_STEP(signer_send_signature);
SSH_FSM_STEP(signer_finish);

SSH_FSM_STEP(signer_suspend)
{
  return SSH_FSM_SUSPENDED;
}

SSH_FSM_STEP(signer_process_error_packet)
{
  unsigned int error_code;
  char *error_message = NULL, *lang_tag = NULL;
  size_t parsed_len = 0L;

  SshSigner gdata = (SshSigner) fsm_context;

  SSH_FSM_SET_NEXT(signer_finish);

  /* Tell packet wrapper we don't want more packets, and output EOF to
     the stream. */
  ssh_packet_wrapper_send_eof(gdata->wrapper);

  if ((parsed_len = ssh_decode_array(gdata->packet_payload,
                                     gdata->packet_payload_len,
                                     SSH_FORMAT_CHAR, &error_code,
                                     SSH_FORMAT_UINT32_STR,
                                     &error_message, NULL,
                                     SSH_FORMAT_UINT32_STR, &lang_tag, NULL,
                                     SSH_FORMAT_END)) == 0)
    {
      ssh_warning("Invalid error packet received (what an irony...).");
      gdata->error_code = SIGNER_ERROR_PROTOCOL_ERROR;
    }

  if (parsed_len < gdata->packet_payload_len)
    {
      ssh_warning("Extra data after SSH_AUTH_HOSTBASED_ERROR packet.");
    }

  ssh_warning("Received error %d, message \"%s\", lang_tag \"%s\".",
              error_code, error_message, lang_tag);

  switch (error_code)
    {
    case SIGNER_ERROR_GENERIC:
    case SIGNER_ERROR_PROTOCOL_ERROR:
      gdata->error_code = error_code;
      break;
    default:
      ssh_warning("Received unknown error code %d.", error_code);
      gdata->error_code = SIGNER_ERROR_PROTOCOL_ERROR;
    }

  ssh_xfree(error_message);
  ssh_xfree(lang_tag);

  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(signer_process_end)
{
  SSH_FSM_SET_NEXT(signer_finish);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(signer_send_error)
{
  SshSigner gdata = (SshSigner) fsm_context;

  gdata->state = SIGNER_DEAD;

  if (gdata->packet_payload)
    {
      ssh_xfree(gdata->packet_payload);
      gdata->packet_payload = NULL;
    }

  gdata->packet_type = SSH_AUTH_HOSTBASED_ERROR;

  gdata->packet_payload_len =
    ssh_encode_array_alloc(&gdata->packet_payload,
                           SSH_FORMAT_CHAR,
                           (unsigned int) gdata->error_code_to_ssh2,
                           SSH_FORMAT_UINT32_STR, gdata->error_message_to_ssh2,
                           strlen(gdata->error_message_to_ssh2),
                           SSH_FORMAT_UINT32_STR, "en", 2,
                           SSH_FORMAT_END);

  ssh_packet_wrapper_can_receive(gdata->wrapper, FALSE);

  SSH_FSM_SET_NEXT(signer_finish);

  gdata->packet_pending = TRUE;

  if (ssh_packet_wrapper_can_send(gdata->wrapper))
    ssh_register_timeout(0L, 0L, signer_can_send, gdata);

  return SSH_FSM_SUSPENDED;
}

SSH_FSM_STEP(signer_read_hostkeys)
{
  SshSigner gdata = (SshSigner) fsm_context;

  /* Go get hostkeys. */
  SSH_FSM_ASYNC_CALL(ssh_host_key_read_keys(gdata->serv_config->host_keys_ctx,
                                            gdata->serv_config,
                                            gdata->effective_user_data,
                                            TRUE, TRUE,
                                            signer_hostkey_read_completion,
                                            thread));
}

void signer_hostkey_read_completion(SshHostKeyStatus success,
                                    void *context)
{
  SshFSMThread thread = (SshFSMThread)context;
  SshSigner gdata = (SshSigner) ssh_fsm_get_gdata(thread);
  SSH_PRECOND(gdata != NULL);

  if (success == SSH_HOSTKEY_ERROR)
    {
      SSH_TRACE(0, ("Unable to load server hostkeys."));
      gdata->error_code_to_ssh2 = SIGNER_ERROR_GENERIC;
      gdata->error_message_to_ssh2 = ssh_xstrdup("Unable to load host keys.");
      SSH_FSM_SET_NEXT(signer_send_error);
    }
  else
    {
      SSH_FSM_SET_NEXT(signer_check_hostbased_packet);
    }
  
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_FSM_STEP(signer_check_hostbased_packet)
{
  /* Received. */
  unsigned int msg_byte; /* This is unsigned int because
                            SSH_FORMAT_CHAR expects uint; caused a
                            rather nasty bug during development (I
                            used SshUInt8, which wasn't long
                            enough => ssh_decode_array blew the
                            stack).*/
  char *service_str = NULL, *hostbased_str = NULL;
  char *recv_pubkey_alg = NULL, *recv_hostname = NULL;
  char *recv_username = NULL;
  unsigned char *recv_pubkeyblob = NULL;
  size_t recv_pubkeyblob_len;
  /* Dug up by us. */
  char *hostname = NULL, *username = NULL;
  SshBuffer pubkeyblob_buf = NULL;
  char *used_service;
  size_t hostname_len;

  /* Internal stuff*/
  char *hostkeyfile = NULL;

  SshSigner gdata = (SshSigner) fsm_context;
  
  SSH_PRECOND(gdata->packet_payload != NULL);

  /* If check is failed, go to this state. */
  SSH_FSM_SET_NEXT(signer_send_error);

  if (ssh_decode_array(gdata->packet_payload, gdata->packet_payload_len,
                       /* session id */
                       SSH_FORMAT_UINT32_STR, NULL, NULL,
                       /* SSH_MSG_USERAUTH_REQUEST (must be checked)*/
                       SSH_FORMAT_CHAR, &msg_byte,
                       /* user name (on remote host) */
                       SSH_FORMAT_UINT32_STR, NULL, NULL,
                       /* service (must be checked)*/
                       SSH_FORMAT_UINT32_STR, &service_str, NULL,
                       /* "hostbased" (must be checked)*/
                       SSH_FORMAT_UINT32_STR, &hostbased_str, NULL,
                       /* public key algorithm for hostkey (must
                          be checked)*/
                       SSH_FORMAT_UINT32_STR, &recv_pubkey_alg, NULL,
                       /* public hostkey and certificates (must be
                          checked)*/
                       SSH_FORMAT_UINT32_STR, &recv_pubkeyblob,
                       &recv_pubkeyblob_len,
                       /* client host name (must be checked)*/
                       SSH_FORMAT_UINT32_STR, &recv_hostname, NULL,
                       /* user name on client host (must be checked) */
                       SSH_FORMAT_UINT32_STR, &recv_username, NULL,
                       SSH_FORMAT_END) != gdata->packet_payload_len ||
      gdata->packet_payload_len == 0)
    {
      /* There was an error. */
      SSH_TRACE(0, ("Invalid packet."));
      gdata->error_code_to_ssh2 = SIGNER_ERROR_PROTOCOL_ERROR;
      gdata->error_message_to_ssh2 = ssh_xstrdup("Decoding supplied packet "
                                                 "failed.");
      goto error;
    }

  /* Getting hostname. */
  hostname = ssh_xmalloc(MAXHOSTNAMELEN*2);

  ssh_tcp_get_host_name(hostname, MAXHOSTNAMELEN*2);
  hostname_len = strlen(hostname);
  /* Sanity check */
  if (hostname_len + 2 >= MAXHOSTNAMELEN)
    ssh_fatal("Local hostname too long!");
  if (strchr(hostname, '.'))
    {
      /* Append trailing period to the name. */
      hostname[hostname_len] = '.';
      hostname[hostname_len + 1] = '\0';
    }
  else
    {
      char *temp_name;
      if (!gdata->global_config->default_domain)
        {
          SSH_TRACE(2, ("hostname doesn't contain '.' and DefaultDomain "
                        "isn't set. Hostbased authentication will "
                        "very likely fail."));
        }
      else
        {
          /* Append domain name and trailing period in the name. */
          ssh_xdsprintf(&temp_name, "%s.%s.", hostname,
                       gdata->global_config->default_domain);
          ssh_xfree(hostname);
          hostname = temp_name;
        }
    }

  /* Getting username. */
  username = ssh_xstrdup(ssh_user_name(gdata->real_user));

  /* Check all parameters. */
  if (msg_byte != SSH_MSG_USERAUTH_REQUEST)
    {
      SSH_TRACE(1, ("Invalid packet."));
      SSH_DEBUG(1, ("msg_byte != SSH_MSG_USERAUTH_REQUEST "
                    "(msg_byte = %d)", msg_byte));
      gdata->error_code_to_ssh2 = SIGNER_ERROR_DATA_VERIFICATION_FAILED;
      gdata->error_message_to_ssh2 =
        ssh_xstrdup("Supplied packet is not a SSH_MSG_USERAUTH_REQUEST "
                    "packet.");
      goto error;
    }
  if (gdata->compat->hostbased_requested_service_draft_incompat)
    used_service = SSH_USERAUTH_SERVICE;
  else
    used_service = SSH_CONNECTION_SERVICE;

  if (strcmp(service_str, used_service) != 0)
    {
      SSH_TRACE(1, ("Invalid packet."));
      SSH_DEBUG(1, ("service_str != \"%s\" (it was '%s')",
                    used_service, service_str));
      gdata->error_code_to_ssh2 = SIGNER_ERROR_DATA_VERIFICATION_FAILED;
      ssh_xdsprintf(&gdata->error_message_to_ssh2, "Supplied packet is not "
                   "used to request %s", used_service);
      goto error;
    }
  if (strcmp(hostbased_str, SSH_AUTH_HOSTBASED) != 0)
    {
      SSH_TRACE(1, ("Invalid packet."));
      SSH_DEBUG(1, ("hostbased_str != \"hostbased\" (it was '%s')",
                    hostbased_str));
      gdata->error_code_to_ssh2 = SIGNER_ERROR_DATA_VERIFICATION_FAILED;
      gdata->error_message_to_ssh2 =
        ssh_xstrdup("Supplied packet is not for \"hostbased\" "
                    "authentication.");
      goto error;
    }

  {
    char *alg_list = ssh_host_key_get_algorithmlist
      (gdata->serv_config->host_keys_ctx);
    const char *temp_list = alg_list;
    Boolean match = FALSE;

    do {
      char *next_alg = ssh_snlist_get_name(temp_list);
      if (strcmp(next_alg, recv_pubkey_alg) == 0)
        {
          match = TRUE;
          ssh_xfree(next_alg);
          break;
        }
      ssh_xfree(next_alg);
      temp_list = ssh_snlist_step_forward(temp_list);
    } while (temp_list);

    ssh_xfree(alg_list);

    if (!match)
      {
        SSH_TRACE(1, ("Invalid packet."));
        SSH_DEBUG(1, ("Client gave us invalid pubkey-algorithms for our "
                      "hostkey."));
        gdata->error_code_to_ssh2 = SIGNER_ERROR_DATA_VERIFICATION_FAILED;
        ssh_xdsprintf(&gdata->error_message_to_ssh2,
                      "Supplied packet contains wrong public "
                      "key algorithm (%s).",recv_pubkey_alg);
        goto error;
      }
  }

  gdata->pubkeytype = ssh_xstrdup(recv_pubkey_alg);

  pubkeyblob_buf = ssh_host_key_get_pubkey_by_algname
    (gdata->pubkeytype, gdata->serv_config->host_keys_ctx);

  if (!pubkeyblob_buf)
    ssh_fatal("Internal error fetching publickey.");

  if (recv_pubkeyblob_len != ssh_buffer_len(pubkeyblob_buf) ||
      memcmp(recv_pubkeyblob, ssh_buffer_ptr(pubkeyblob_buf),
             ssh_buffer_len(pubkeyblob_buf)) != 0)
    {
      SSH_TRACE(1, ("Invalid packet."));
      SSH_DEBUG(1, ("Client gave us wrong (or corrupted) "
                    "public key."));
      if (recv_pubkeyblob_len != ssh_buffer_len(pubkeyblob_buf))
        SSH_DEBUG(1, ("Lengths differ (received: %d ; ours: %d)",
                      recv_pubkeyblob_len, ssh_buffer_len(pubkeyblob_buf)));
#ifdef HEXDUMPS
      SSH_DEBUG_HEXDUMP(3, ("client gave us:"),
                        recv_pubkeyblob, pubkeyblob_len);
      SSH_DEBUG_HEXDUMP(3, ("our pubkey:"),
                        recv_pubkeyblob, pubkeyblob_len);
#endif /* HEXDUMPS */
      gdata->error_code_to_ssh2 = SIGNER_ERROR_DATA_VERIFICATION_FAILED;
      gdata->error_message_to_ssh2 =
        ssh_xstrdup("Supplied packet contains malformed public key.");
      goto error;
    }

  if (strcasecmp(recv_hostname, hostname) != 0)
    {
      SSH_TRACE(1, ("Invalid packet."));
      SSH_DEBUG(1, ("Wethinks the client gave us the wrong hostname. "
                    "(client's opinion: '%s' ours: '%s'",
                    recv_hostname, hostname));
      gdata->error_code_to_ssh2 = SIGNER_ERROR_DATA_VERIFICATION_FAILED;
      ssh_xdsprintf(&gdata->error_message_to_ssh2, "Supplied packet had the "
                    "wrong hostname (packet contained \"%s\", ssh-signer "
                    "got \"%s\").", recv_hostname, hostname);
      goto error;
    }
  if (strcmp(recv_username, username) != 0)
    {
      SSH_TRACE(1, ("Invalid packet."));
      SSH_DEBUG(1, ("Client definitely gave us the wrong user name. "
                    "(it says: '%s' we know: '%s')", recv_username,
                    username));
      gdata->error_code_to_ssh2 = SIGNER_ERROR_DATA_VERIFICATION_FAILED;
      ssh_xdsprintf(&gdata->error_message_to_ssh2, "Supplied packet had the "
                    "wrong client username (packet contained \"%s\", "
                    "ssh-signer got \"%s\").", recv_username, username);
      goto error;
    }

  /* Success. */
  SSH_TRACE(0, ("Received packet was OK."));
  SSH_FSM_SET_NEXT(signer_sign_hostbased_packet);
 error:
  ssh_xfree(service_str);
  ssh_xfree(hostbased_str);
  ssh_xfree(recv_pubkey_alg);
  ssh_xfree(recv_hostname);
  ssh_xfree(recv_username);
  ssh_xfree(recv_pubkeyblob);
  ssh_xfree(hostname);
  ssh_xfree(username);
  ssh_xfree(hostkeyfile);
  return SSH_FSM_CONTINUE;
}

void signer_signature_cb(SshCryptoStatus result,
                         const unsigned char *signature_buffer,
                         size_t sig_len,
                         void *context)
{
  SshFSMThread thread = (SshFSMThread)context;
  SshSigner gdata = (SshSigner) ssh_fsm_get_gdata(thread);
  SSH_PRECOND(gdata != NULL);

  gdata->op_handle = NULL;

  if (result != SSH_CRYPTO_OK)
    {
      SSH_TRACE(0, ("ssh_private_key_sign() returned %d.", result));
      gdata->error_code_to_ssh2 = SIGNER_ERROR_SIGNING_OPERATION_FAILED;
      gdata->error_message_to_ssh2 = ssh_xstrdup("Signature operation failed.");
      goto error;
    }

#ifdef HEXDUMPS
  SSH_DEBUG_HEXDUMP(5, ("Signature:"), signature_buffer, sig_len);
#endif /* HEXDUMPS */
  /* Send it to client. */
  gdata->packet_type = SSH_AUTH_HOSTBASED_SIGNATURE;

  if (!gdata->compat->signature_encode_draft_incompat)
    {
      gdata->packet_payload_len =
        ssh_encode_array_alloc(&gdata->packet_payload,
                               SSH_FORMAT_UINT32_STR,
                               gdata->pubkeytype, strlen(gdata->pubkeytype),
                               SSH_FORMAT_UINT32_STR,
                               signature_buffer,
                               sig_len,
                               SSH_FORMAT_END);
    }
  else
    {
      gdata->packet_payload_len =
        ssh_encode_array_alloc(&gdata->packet_payload,
                               SSH_FORMAT_DATA,
                               signature_buffer,
                               sig_len,
                               SSH_FORMAT_END);
    }

  SSH_FSM_SET_NEXT(signer_send_signature);
 error:
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

/* Sign the packet and send it to client. */
SSH_FSM_STEP(signer_sign_hostbased_packet)
{
  SshSigner gdata = (SshSigner) fsm_context;
  char *comment = NULL;
  size_t sig_len;
  char *hostkeyfile = NULL;
  SshPrivateKey privkey;
  Boolean is_rsa = FALSE;
  Boolean use_sha1_with_rsa = TRUE;
  
  /* If signature op fails, go to this state. */
  SSH_FSM_SET_NEXT(signer_send_error);

  /* If we've gotten this far, the packet is ok, and it can be
     signed. */

  privkey = ssh_host_key_get_privkey_by_algname
    (gdata->pubkeytype, gdata->serv_config->host_keys_ctx);

  SSH_ASSERT(privkey != NULL);

  /* Check how big a chunk our private key can sign (this is
     purely a sanity check, as both of our signature schemas do
     their own hashing) */
  sig_len = ssh_private_key_max_signature_input_len(privkey);

  SSH_TRACE(2, ("Max input length for signing: %d", sig_len));

  if (sig_len == 0)
    {
      SSH_TRACE(0, ("Private key not capable of signing! "
                    "(definitely an error)"));
      gdata->error_code_to_ssh2 = SIGNER_ERROR_SIGNING_OPERATION_FAILED;
      gdata->error_message_to_ssh2 = ssh_xstrdup("Private host key not capable "
                                                 "of signing.");
      goto error;
    }
  else if (sig_len != -1 && sig_len < gdata->packet_payload_len)
    {
      SSH_TRACE(0, ("Private key can't sign our data. (too much "
                    "data (data_len %d, max input len for signing "
                    "%d))", gdata->packet_payload_len, sig_len));
      gdata->error_code_to_ssh2 = SIGNER_ERROR_SIGNING_OPERATION_FAILED;
      gdata->error_message_to_ssh2 = ssh_xstrdup("Private host key can't sign "
                                                 "packet (too much data).");
      goto error;
    }

  if (!strncmp(gdata->pubkeytype, "ssh-rsa", 7))
    {
      is_rsa = TRUE;
      if (gdata->compat->rsa_hash_scheme_draft_incompat)
        use_sha1_with_rsa = FALSE;
    }
  if (!strncmp(gdata->pubkeytype, "x509v3-sign-rsa", 15))
    {
      is_rsa = TRUE;

      if (gdata->compat->cert_rsa_hash_scheme_incompat)
        use_sha1_with_rsa = FALSE;
    }
  /* RSA hash scheme compatibility. */
  if (is_rsa &&
      !ssh_compat_rsa_private_key_change_scheme(privkey, use_sha1_with_rsa))
    {
      SSH_TRACE(0, ("Changing private key scheme failed."));
      gdata->error_code_to_ssh2 = SIGNER_ERROR_SIGNING_OPERATION_FAILED;
      gdata->error_message_to_ssh2 =
        ssh_xstrdup("Changing private host key hash scheme failed.");
      goto error;
    }

  /* Do the actual signing. */

#ifdef HEXDUMPS
  SSH_DEBUG_HEXDUMP(5, ("Signing following data"),
                    gdata->packet_payload + 4, gdata->packet_payload_len - 4);
#endif /* HEXDUMPS */

  ssh_xfree(hostkeyfile);
  ssh_xfree(comment);

  SSH_FSM_ASYNC_CALL((gdata->op_handle =
                      ssh_private_key_sign_async(privkey,
                                                 gdata->packet_payload,
                                                 gdata->packet_payload_len,
                                                 signer_signature_cb,
                                                 thread)));
 error:
  ssh_xfree(hostkeyfile);
  ssh_xfree(comment);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(signer_process_compat_flags)
{
  SshSigner gdata = (SshSigner) fsm_context;
  size_t len = 0;
  char *compat_flags = NULL;

  SSH_PRECOND(gdata->packet_payload != NULL);
  SSH_PRECOND(gdata->packet_payload_len);

  len = ssh_decode_array(gdata->packet_payload, gdata->packet_payload_len,
                         SSH_FORMAT_UINT32_STR, &compat_flags, NULL,
                         SSH_FORMAT_END);

  if (len == 0 || len != gdata->packet_payload_len)
    {
      /* Malformed packet. */
      ssh_xfree(compat_flags);
      gdata->error_code_to_ssh2 = SIGNER_ERROR_PROTOCOL_ERROR;
      gdata->error_message_to_ssh2 =
        ssh_xstrdup("Malformed SSH_AUTH_HOSTBASED_COMPAT packet.");
      SSH_FSM_SET_NEXT(signer_send_error);
      return SSH_FSM_CONTINUE;
    }

  /* Do the actual processing. */
  {
    char *rest, *current;

    rest = compat_flags;

    while (strlen(rest) > 0 &&
           (current = ssh_app_param_list_get_next(rest)) != NULL)
      {
        rest += strlen(current);
        if (*rest == ',')
          rest++;

        if (strcmp(current, HOSTBASED_REQUESTED_SERVICE_DRAFT_INCOMPAT) == 0)
          gdata->compat->hostbased_requested_service_draft_incompat = TRUE;
        else if (strcmp(current, SIGNATURE_ENCODE_DRAFT_INCOMPAT) == 0)
          gdata->compat->signature_encode_draft_incompat = TRUE;
        else if (strcmp(current, RSA_HASH_SCHEME_DRAFT_INCOMPAT) == 0)
          gdata->compat->rsa_hash_scheme_draft_incompat = TRUE;




        
        /* If more compat-flags are needed, add them here. */
        ssh_xfree(current);
      }
  }

  ssh_xfree(compat_flags);

  /* We are now prepared to receive more packets. */
  ssh_packet_wrapper_can_receive(gdata->wrapper, TRUE);

  return SSH_FSM_SUSPENDED;
}

SSH_FSM_STEP(signer_send_signature)
{
  SshSigner gdata = (SshSigner) fsm_context;

  SSH_FSM_SET_NEXT(signer_suspend);

  gdata->packet_pending = TRUE;

  if (ssh_packet_wrapper_can_send(gdata->wrapper))
    ssh_register_timeout(0L, 0L, signer_can_send, gdata);

  return SSH_FSM_SUSPENDED;
}

SSH_FSM_STEP(signer_finish)
{
  SshSigner gdata = (SshSigner) fsm_context;
  ssh_fsm_destroy(gdata->fsm);

  SSH_DEBUG(3, ("Destroying gdata..."));
  if (gdata->op_handle)
    {
      ssh_operation_abort(gdata->op_handle);
      gdata->op_handle = NULL;
    }
  ssh_packet_wrapper_destroy(gdata->wrapper);
  ssh_xfree(gdata->packet_payload);
  ssh_config_free(gdata->serv_config);
  ssh_config_free(gdata->global_config);
  ssh_user_free(gdata->effective_user_data, FALSE);
  ssh_user_free(gdata->real_user, FALSE);
  ssh_xfree(gdata->compat);
  ssh_xfree(gdata->error_message_to_ssh2);
  ssh_xfree(gdata->pubkeytype);
  ssh_xfree(gdata);
  SSH_DEBUG(3, ("done."));
  
  return SSH_FSM_FINISH;
}

SshFSMStateDebugStruct signer_states[] =
{
  { "signer_suspend", "Suspend thread", signer_suspend },

  { "signer_process_compat_flags", "Process compatibility flags",
    signer_process_compat_flags },
  { "signer_process_error_packet", "Process received error packet",
    signer_process_error_packet },
  { "signer_process_end", "Process received END packet",
    signer_process_end },
  { "signer_send_error", "Send error packet", signer_send_error },

  { "signer_check_hostbased_packet",
    "Check hostbased-authentication packet validity",
    signer_check_hostbased_packet },
  { "signer_read_hostkeys",
    "Read the hostkeys to memory",
    signer_read_hostkeys },
  { "signer_sign_hostbased_packet", "Sign the hostbased packet",
    signer_sign_hostbased_packet },
  { "signer_send_signature", "Send the signature to the application",
    signer_send_signature },

  { "signer_finish", "We're ready", signer_finish }
};

void signer_received_packet(SshPacketType type,
                            const unsigned char *data, size_t len,
                            void *context)
{
  SshSigner signer = (SshSigner) context;

  SSH_DEBUG(2, ("Received packet, length = %ld", len));
#ifdef HEXDUMPS
  SSH_DEBUG_HEXDUMP(3, ("packet:"), data, len);
#endif /* HEXDUMPS */

  ssh_packet_wrapper_can_receive(signer->wrapper, FALSE);

  SSH_DEBUG(3, ("Previous state: %d", signer->state));

  switch (type)
    {
    case SSH_AUTH_HOSTBASED_PACKET:
      if (signer->state != SIGNER_START &&
          signer->state != SIGNER_COMPAT_FLAGS_RECEIVED &&
          signer->state != SIGNER_HOSTBASED_PACKET_RECEIVED)
        {
          /* This must be sent first or after compat flags. */
          signer->error_code_to_ssh2 = SIGNER_ERROR_PROTOCOL_ERROR;
          signer->error_message_to_ssh2 = ssh_xstrdup("Protocol error.");
          goto send_error;
        }

      signer->state = SIGNER_HOSTBASED_PACKET_RECEIVED;
      ssh_fsm_set_next(signer->main_thread, signer_read_hostkeys);
      break;
    case SSH_AUTH_HOSTBASED_END:
      if (signer->state == SIGNER_DEAD)
        {
          /* This must be sent only once. */
          ssh_fatal("Internal error (state HOSTBASED_END again)");
        }
      signer->state = SIGNER_DEAD;
      ssh_fsm_set_next(signer->main_thread, signer_process_end);
      break;
    case SSH_AUTH_HOSTBASED_ERROR:
      if (signer->state == SIGNER_DEAD)
        {
          /* This must be sent only once. */
          ssh_fatal("Internal error (state HOSTBASED_ERROR again)");
        }
      signer->state = SIGNER_DEAD;
      ssh_fsm_set_next(signer->main_thread, signer_process_error_packet);
      break;
    case SSH_AUTH_HOSTBASED_COMPAT:
      if (signer->state != SIGNER_START)
        {
          /* This has to be sent first, if at all. */
          signer->error_code_to_ssh2 = SIGNER_ERROR_PROTOCOL_ERROR;
          signer->error_message_to_ssh2 = ssh_xstrdup("Protocol error (compat "
                                                      "flags sent too late).");
          goto send_error;
        }
      signer->state = SIGNER_COMPAT_FLAGS_RECEIVED;
      ssh_fsm_set_next(signer->main_thread, signer_process_compat_flags);
      break;
    default:
      SSH_TRACE(0, ("Invalid packet type %d received from ssh2-client.",
                    type));
      signer->error_code_to_ssh2 = SIGNER_ERROR_PROTOCOL_ERROR;
      signer->error_message_to_ssh2 = ssh_xstrdup("Invalid packet type "
                                                  "received.");
    send_error:
      ssh_fsm_set_next(signer->main_thread, signer_send_error);
      ssh_fsm_continue(signer->main_thread);
      return;
    }

  ssh_fsm_continue(signer->main_thread);

  if (signer->packet_payload != NULL)
    ssh_xfree(signer->packet_payload);

  signer->packet_payload = ssh_xmemdup(data, len);
  signer->packet_payload_len = len;
}

void signer_received_eof(void *context)
{
  SshSigner signer = (SshSigner) context;

  SSH_DEBUG(3, ("Received EOF from packetstream."));

  if (signer->state != SIGNER_DEAD)
    {
      ssh_warning("Received EOF from packetstream before transactions were "
                  "complete.");
      signer->state = SIGNER_DEAD;
    }

  ssh_fsm_set_next(signer->main_thread, signer_finish);
  ssh_fsm_continue(signer->main_thread);
}

void signer_can_send(void *context)
{
  SshSigner signer = (SshSigner)context;

  if (!signer->packet_pending)
    return;

  signer->packet_pending = FALSE;

  SSH_PRECOND(signer != NULL);
  SSH_PRECOND(signer->packet_type);
  SSH_PRECOND(signer->packet_payload != NULL);
  SSH_PRECOND(signer->packet_payload_len);

  ssh_packet_wrapper_send(signer->wrapper,
                          signer->packet_type,
                          signer->packet_payload,
                          signer->packet_payload_len);

  signer->packet_type = (SshPacketType)0;
  ssh_xfree(signer->packet_payload);
  signer->packet_payload = NULL;
  signer->packet_payload_len = 0L;

  ssh_packet_wrapper_can_receive(signer->wrapper, TRUE);

  ssh_fsm_continue(signer->main_thread);
}


void signer_ssh_fatal(const char *message, void *context)
{
  fprintf(stderr, "%s:FATAL:%s\n", progname, message);
  fflush(stderr);
}

void signer_ssh_warning(const char *message, void *context)
{
  SshSigner signer = (SshSigner) context;

  if (!signer->quiet)
    fprintf(stderr, "%s:%s\n", progname, message);
  fflush(stderr);
}

void signer_ssh_debug(const char *message, void *context)
{
  SshSigner signer = (SshSigner) context;

  if (!signer->quiet)
    fprintf(stderr, "%s:%s\n", progname, message);
  fflush(stderr);
}

int main(int argc, char **argv)
{
  SshStream stdio_stream;
  SshSigner signer;
  char *config_filename;
  char *temp_name;

#ifdef SLEEP_AFTER_STARTUP
  sleep(30);
#endif /* SLEEP_AFTER_STARTUP */

  ssh_user_close_fds();
  
  /* Get program name (without path). */
  if ((temp_name = strrchr(argv[0], '/')) != NULL)
    progname = ssh_xstrdup(temp_name + 1);
  else
    progname = ssh_xstrdup(argv[0]);

  /* XXX there should be a way to give command-line parameters to this
     program, but, they should only be used if the uid is the same as
     euid. */
  ssh_event_loop_initialize();

  ssh_global_init();



#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
  ssh_pk_provider_register(&ssh_pk_if_modn_generator);
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */
#ifdef SSHDIST_CRYPT_DSA
  ssh_pk_provider_register(&ssh_pk_dl_modp_generator);
#endif /* SSHDIST_CRYPT_DSA */

  signer = ssh_xcalloc(1, sizeof(*signer));
  signer->fsm = ssh_fsm_create(signer);
  SSH_VERIFY(signer->fsm != NULL);
  ssh_fsm_register_debug_names(signer->fsm, signer_states,
                               SSH_FSM_NUM_STATES(signer_states));
  signer->compat = ssh_xcalloc(1, sizeof(*signer->compat));
#ifdef SIGNER_QUIET
  signer->quiet = TRUE;
#else /* SIGNER_QUIET */
  signer->quiet = FALSE;
#endif /* SIGNER_QUIET */

  ssh_debug_register_callbacks(signer_ssh_fatal, signer_ssh_warning,
                               signer_ssh_debug, (void *)signer);
#ifdef SIGNER_DEBUG
  ssh_debug_set_global_level(5);
#endif /* SIGNER_DEBUG */

  /* Act as server. */
  signer->serv_config = ssh_server_create_config();

  signer->global_config = ssh_client_create_config();
  /* Initialize user context with euid. This is used to dig up the
     hostkey and such. */
  signer->effective_user_data = ssh_user_initialize_with_uid(geteuid(), FALSE);

  signer->real_user = ssh_user_initialize(NULL, FALSE);

  ssh_randseed_open(signer->effective_user_data,
                    signer->serv_config);

  /* XXX what about alternative config files? This should be possible
     to configure somehow. An option for configure is probably a good
     idea. */
  ssh_xdsprintf(&config_filename, "%s/%s",
                SSH_SERVER_DIR, SSH_SERVER_CONFIG_FILE);

  if (!ssh_config_read_file(signer->effective_user_data, signer->serv_config,
                            NULL, config_filename,
                            SSH_CONFIG_TYPE_SERVER_GLOBAL))
    ssh_warning("%s: Failed to read config file %s", argv[0],
                config_filename);

  ssh_xfree(config_filename);

  if (!ssh_config_read_file(signer->real_user, signer->global_config,
                            "", SSH_CLIENT_GLOBAL_CONFIG_FILE,
                            SSH_CONFIG_TYPE_CLIENT))
    ssh_warning("%s: Failed to read config file %s", argv[0],
                SSH_CLIENT_GLOBAL_CONFIG_FILE);

  SSH_TRACE(2, ("randomseed file: %s", signer->serv_config->random_seed_file));

  stdio_stream = ssh_stream_fd_wrap2(fileno(stdin), fileno(stdout),
                                     TRUE);

  signer->wrapper = ssh_packet_wrap(stdio_stream,
                                    signer_received_packet,
                                    signer_received_eof,
                                    signer_can_send,
                                    signer);

  signer->main_thread = ssh_fsm_thread_create(signer->fsm,
                                              signer_suspend,
                                              NULL_FNPTR, NULL_FNPTR, NULL);
  SSH_VERIFY(signer->main_thread != NULL);
  ssh_fsm_set_thread_name(signer->main_thread, "main_thread");

  ssh_packet_wrapper_can_receive(signer->wrapper, TRUE);

  ssh_event_loop_run();

  ssh_event_loop_uninitialize();




  ssh_global_uninit();

  return 0;
}
