/*

  sshsftpcwd.c

  Author: Tomi Salo <ttsalo@ssh.com>
          Sami Lehtinen <sjl@ssh.com>
          
  Copyright (C) 1999-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Created Wed Jan  5 21:52:44 2000.

  Auxiliary stuff implementing CWD in sftp2. (When using
  sshfilecopy).
  
  */

#include "sshincludes.h"
#include "sftpcwd.h"
#include "sshfilexfer.h"
#include "sshdsprintf.h"
#include "sshfilecopy.h"

#define SSH_DEBUG_MODULE "SftpCwd"

struct SshSftpCwdContextRec
{
  char *cwd;
  /* The path currently checked for validity. */
  char *new_cwd;  
  SshFileClient file_client;
  void *user_context;
  SshSftpChangeCwdCb change_cwd_cb;
};

/* Initialize the cwd stuff and get your very own context! */
SshSftpCwdContext
ssh_sftp_cwd_init(char *path, SshFileClient file_client)
{
  SshSftpCwdContext ctx;
  SSH_PRECOND(path);
  SSH_PRECOND(file_client);
  
  ctx = ssh_xcalloc(1, sizeof(*ctx));
  ctx->file_client = file_client;
  ctx->cwd = ssh_xstrdup(path);
  
  return ctx;
}

void
ssh_sftp_cwd_uninit(SshSftpCwdContext cwd_context)
{
  ssh_xfree(cwd_context->cwd);
  ssh_xfree(cwd_context->new_cwd);  
  ssh_xfree(cwd_context);
}

/* XXX
   
   To strip the foo/../ constructs from paths, it would probably be
   easiest to add the stripping to cwd_add and cwd_strip.
 */
/* Adds the cwd in front of the given string.
   If `path' is absolute, returns it.
   'path' is permitted to be NULL.*/
char *
ssh_sftp_cwd_add(const char *path,
                 SshSftpCwdContext cwd_context)
{
  char *new_path;

  /* If we haven't changed working directory. */
  if (!cwd_context->cwd)
    return ssh_xstrdup(path);
  
  if (path)
    {
      if (*path == '/')
        new_path = ssh_xstrdup(path);
      else
        {
          if (strcmp(cwd_context->cwd, "/") == 0)
            ssh_xdsprintf(&new_path, "/%s", path);
          else
            ssh_xdsprintf(&new_path, "%s/%s", cwd_context->cwd, path);
        }
    }
  else
    {
      new_path = ssh_xstrdup(cwd_context->cwd);
    }
  return new_path;
}

/* Strips the cwd from the front of the given path.
   Returns an xmallocated string or NULL if the given
   path did not contain the cwd. */
char *
ssh_sftp_cwd_strip(const char *path,
                   SshSftpCwdContext cwd_context)
{
  char *cwd = cwd_context->cwd, *p;
  
  if (!cwd)
    return ssh_xstrdup(path);
  
  if (!strncmp(path, cwd, strlen(cwd)))
    {
      p = (char *)&path[strlen(cwd)];
      if (*p == '/')
        p++;
      if (*p == '\0')
        p = ".";
      p = ssh_xstrdup(p);
    }
  else
    {
      p = NULL; 
    }
  return p;
}

void
ssh_sftp_cwd_change_cb(SshFileClientError error,
                       SshFileAttributes attributes,
                       const char *error_msg,
                       const char *lang_tag,
                       void *context)
{
  SshSftpCwdContext cwd_ctx = (SshSftpCwdContext)context;

  /* If the proposed path exists, we'll replace the old
     path stored in the list with it. */
  if (error == SSH_FX_OK)
    {
      ssh_xfree(cwd_ctx->cwd);
      cwd_ctx->cwd = cwd_ctx->new_cwd;
      SSH_DEBUG(2, ("Path OK: %s", cwd_ctx->cwd));

      cwd_ctx->change_cwd_cb(SSH_SFTP_CWD_OK, cwd_ctx->user_context);
    }
  else
    {
      ssh_xfree(cwd_ctx->new_cwd);
      cwd_ctx->change_cwd_cb(SSH_SFTP_CWD_ERROR, cwd_ctx->user_context);
    }
  cwd_ctx->new_cwd = NULL;      
}

void
ssh_sftp_cwd_change(char *path,
                    SshSftpCwdContext cwd_ctx,
                    SshSftpChangeCwdCb cwd_cb,
                    void *context)
{
  char *hlp, *hlp2;

  SSH_PRECOND(path);
  
  if (*path == '\0')
    {
      /* return with error */
      (*cwd_cb)(SSH_SFTP_CWD_ERROR, context);
      return;
    }
  
  cwd_ctx->user_context = context;
  cwd_ctx->change_cwd_cb = cwd_cb;

  /* Add the cwd (or if absolute, use that) */
  hlp = ssh_sftp_cwd_add(path, cwd_ctx);
  /* Get rid of crud. */
  hlp2 = ssh_file_copy_strip_dot_dots(hlp);
  ssh_xfree(hlp);

  if (strlen(hlp2) >= 2)
    {
      /* Check for "/." at end. */
      hlp = &hlp2[strlen(hlp2) - 2];
      /* If path ends in "/." and is not empty after that, remove.  */
      if (!strcmp(hlp, "/.") && hlp != hlp2)
        *hlp = '\0';
      /* For next phase... */
      hlp++;
    }
  else
    {
      /* Here the path MUST be one character long. */
      SSH_ASSERT(hlp2[1] == '\0');
      hlp = hlp2;
    }
  
  /* If path ends in a slash, and is not empty after that, remove. */
  if (*hlp == '/' && hlp != hlp2)
    *hlp = '\0';

  cwd_ctx->new_cwd = hlp2;
  ssh_xdsprintf(&hlp, "%s/.", cwd_ctx->new_cwd);
  
  ssh_file_client_stat(cwd_ctx->file_client, hlp,
                       ssh_sftp_cwd_change_cb, cwd_ctx);
  ssh_xfree(hlp);
}
