/*

  sftp2.c

  Author: Tomi Salo <ttsalo@ssh.com>
          Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1999-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  SFTP. Similar to scp, but uses an ftp-like interface.

 */
/*
  TODO:

  - it is stupid, that all commands, which takes files as arguments,
    do globbing separately. Modify so, that if a command needs a
    filelist, it will globbed as default, and the command can then use
    the resultant filelist.
    - e.g. ls and get/put implement globbing separately, even though
    they could do use a common globbing mechanism.
  - tab completion could be improved to be even more context
    sensitive (ex. complete commands, options, hosts etc)
  - rl_cb is not the correct place to launch commands. Change.
  - some things (e.g. get/put) would be much more cleaner, if
    implemented in their own threads.
  - pager has a bit skewed logic. Because it operates on the same
    FSM as the threads here, it could use common condition variables
    with these.
  - ls should be able to enable a "long" output, even without
    recursion. (why it is like this now, is that we use the server
    provided long string, which can only be gotten with
    SSH_FXP_READDIR)
  - implement more ls options, to be more like GNU ls.
  
*/
#include "sshincludes.h"
#include "sshfilecopy.h"
#include "ssh2includes.h"

#include "sshtty.h"

#include "ssheloop.h"
#include "sshgetopt.h"
#include "sshappcommon.h"
#include "sshreadline.h"
#include "sshfsm.h"

#include "sshunixpipestream.h"
#include "sigchld.h"

#include "sshgetopt.h"
#include "sftpcwd.h"
#include "sftppager.h"
#include "sshfdstream.h"
#include "sshfileio.h"
#include "sshtimeouts.h"
#include "sshdsprintf.h"
#include "sshuser.h"
#include "sshencode.h"
#include "sshglobals.h"
#include "sshadt_list.h"
#include "sshadt_conv.h"












#define SSH_DEBUG_MODULE "Sftp2"

int exit_value = 0;

/* XXX Filexfer errors and filecopy errors are mixed. Well, at least
   doesn't return 0 (zero) in batchmode anymore... */
#define SFTP_SET_EXIT_VAL(error)                                \
do {                                                            \
  if (gdata->batchmode)                                         \
    exit_value = exit_value < error ? error : exit_value;       \
} while (0)

typedef enum
{
  SSH_SFTP_GET,
  SSH_SFTP_PUT
} SshSftpGetputDirection;

/* Flags that determine how sftp_file_compare should sort the files. */
/* The default, by name. */
#define SFTP_FILESORT_NAME    0x0000
/* By size. */
#define SFTP_FILESORT_SIZE    0x0001
/* The order should be reversed. */
#define SFTP_FILESORT_REVERSE 0xF000

/* Global data structure */
typedef struct SshSftp2GDataRec {
  char *debug_level;
  Boolean verbose_mode;

  char *ssh_path; /* Main ssh2 binary */
  SshFSM fsm;
  SshFSMThread main_thread;
  Boolean need_rl_initialization;
  char *initial_connect_target;

  /* Required for ssh_readline_eloop(). */
  SshReadLineCtx rl;
  SshStream stdio_stream;

  /* Handle for list operations in the main thread. */
  SshADTHandle h;
  
  /* Of the variables below, the ones starting with "r" (remote) or
     "l" (local) are the real ones and are copied into the other two,
     depending on whether we are doing a remote or local
     command. lls/ls, lpwd/pwd, lcd/cd etc. will simply use connection
     and cwd_ctx. */
  SshSftpCwdContext rcwd_ctx;
  SshSftpCwdContext lcwd_ctx;
  SshSftpCwdContext cwd_ctx;
  SshFileCopyConnection rconn;
  SshFileCopyConnection lconn;
  SshFileCopyConnection conn;

  /* Which side are we opening? */
  Boolean cmd_doing_remote;

  /* command line options passed to sftp2 to be passed to ssh2,
     such as ciphers or macs */
  SshADTContainer command_line_options;

  SshSftpPagerCtx pager_ctx;

  SshFileCopyGlobAttrs glob_attrs;
  SshFileCopyRecurseAttrs recurse_attrs;
  /* Persistent transfer attributes. */
  SshFileCopyTransferAttrs global_tr_attrs;
  /* Per-transfer attributes. */
  SshFileCopyTransferAttrs tr_attrs;
  
  /* for aborting the transfer. */
  SshOperationHandle transfer_op_handle;
  Boolean exit_attempted;

  /* The getopt context used with internal commands. */
  SshGetOptData getopt_data;

  /* Container for file names given in the sftp command line. */
  SshADTContainer cl_file_list;
  /* Container for file lists returned by ssh_file_copy_{glob,recurse}*. */
  SshADTContainer file_list;

  /* These next two variables are used by commands, which only take
     one or two filenames. */
  char *cmd_file;
  char *cmd_file_target;
  
  /* Flags, that determine how the file list should be sorted. See
     sftp_file_compare() below, and the flags, above. */
  SshUInt32 file_sort_flags;

  /* Batchmode */
  char *batchfile_name;
  Boolean batchmode;
  char *batchfile_data;
  char *batchfile_ptr;
  size_t batchfile_len;

  /* Stuff for command_ls */
  Boolean cmd_ls_recursive;
  Boolean cmd_ls_long_format;
  Boolean cmd_ls_no_args;
  
  /* Stuff for command_rm */
  Boolean cmd_rm_directory;
  /* Stuff for command_mkdir */
  struct SshFileAttributesRec cmd_mkdir_attrs;
  /* Stuff for command_getput */
  SshSftpGetputDirection cmd_getput_direction;

  /* Stuff for Tab Completion */
  char *tabcompl_constant_part;
  char *tabcompl_word_to_complete;
  char *tabcompl_original_word_to_complete;

  char *tabcompl_resultline;

  /* XXX Generic solution, please. */
  Boolean command_cd;
  
  /* File transfer mode */
  char *remote_newline;
  char *client_newline;

  SshFileCopyNLQueryCompletionCB newline_query_completion_cb;
  void *newline_query_completion_context;

  /* From what state to continue (for those callbacks that need it). */
  SshFSMStepCB next_state;
} *Sftp2GData;


/* FUNCTION PROTOTYPES */

SSH_FSM_STEP(sftp_connect_local);
SSH_FSM_STEP(sftp_setup_cwd);
SSH_FSM_STEP(sftp_get_command);
SSH_FSM_STEP(sftp_cmd_open);
SSH_FSM_STEP(sftp_cmd_close);
SSH_FSM_STEP(sftp_cmd_finalize_open);
SSH_FSM_STEP(sftp_cmd_cd_start);
SSH_FSM_STEP(sftp_cmd_pwd_start);
SSH_FSM_STEP(sftp_cmd_rm_start);
SSH_FSM_STEP(sftp_cmd_mkdir_start);
SSH_FSM_STEP(sftp_cmd_ls_glob);
SSH_FSM_STEP(sftp_cmd_ls_recurse);
SSH_FSM_STEP(sftp_cmd_ls_walk_list);
SSH_FSM_STEP(sftp_cmd_ls_walk_list_prepare);
SSH_FSM_STEP(sftp_cmd_ls_step_wait_cv);
SSH_FSM_STEP(sftp_cmd_ls_step);
SSH_FSM_STEP(sftp_cmd_ls_step_finish);
SSH_FSM_STEP(sftp_cmd_ls_check);
SSH_FSM_STEP(sftp_cmd_ls_check_recurse);
SSH_FSM_STEP(sftp_cmd_ls_finish);
SSH_FSM_STEP(sftp_cmd_getput_glob);
SSH_FSM_STEP(sftp_cmd_getput_recurse);
SSH_FSM_STEP(sftp_cmd_getput_transfer);
SSH_FSM_STEP(sftp_cmd_rename_check);
SSH_FSM_STEP(sftp_cmd_rename);
SSH_FSM_STEP(sftp_cmd_readlink);
SSH_FSM_STEP(sftp_cmd_symlink);
SSH_FSM_STEP(sftp_tabcompl_start);
SSH_FSM_STEP(sftp_tabcompl_results);
SSH_FSM_STEP(sftp_tabcompl_results2);
SSH_FSM_STEP(sftp_tabcompl_finish);
SSH_FSM_STEP(sftp_finish);
SSH_FSM_STEP(sftp_finish_thread);

void ssh_sftp_print_help(char *topic);

/* FUNCTION IMPLEMENTATIONS */

/* XXX This has to go. see sshdebug.c (and ssh_debug_output_i(),
   ssh_debug() for a better explanation... */
static int client_debug_output(FILE *stream, const char *format, ...)
{
  int bytes;
  va_list ap;

  int flags, fd;

  /* stream might be in non-blocking mode, so we set it to blocking for
     as long as we write our message. */
  fd = fileno(stream);
  flags = fcntl(fd, F_GETFL, 0);
#ifdef O_ASYNC
  fcntl(fd, F_SETFL, flags & ~(O_ASYNC | O_NONBLOCK));
#else /* O_ASYNC */
  fcntl(fd, F_SETFL, flags & ~(O_NONBLOCK));
#endif /* O_ASYNC */

  va_start(ap, format);
  bytes = vfprintf(stream, format, ap);
  va_end(ap);

  fcntl(fd, F_SETFL, flags);

  return bytes;
}

void
sftp_debug(const char *msg, void *context)
{
  Sftp2GData sftp_gdata = (Sftp2GData) context;

  SSH_PRECOND(sftp_gdata != NULL);

  if (sftp_gdata->debug_level != NULL)
    client_debug_output(stderr, "%s\r\n", msg);
}

void
sftp_warning(const char *msg, void *context)
{
  client_debug_output(stderr, "Warning: %s\r\n", msg);
}

void
sftp_fatal(const char *msg, void *context)
{

  Sftp2GData sftp_gdata = (Sftp2GData) context;
  if (sftp_gdata->rl)
    ssh_readline_eloop_uninitialize(sftp_gdata->rl);
  ssh_leave_raw_mode(-1);
  ssh_leave_non_blocking(-1);


  client_debug_output(stderr, "FATAL: %s\r\n", msg);

  exit(-1);
}

void
sftp_fatal_signal_handler(int signal, void *context)
{
  ssh_fatal("Received signal %d.", signal);
}

int sftp_file_compare(const void *obj1, const void *obj2, void *context)
{
  Sftp2GData gdata = (Sftp2GData) context;
  SshFileCopyFile file1, file2;
  if (gdata->file_sort_flags & SFTP_FILESORT_REVERSE)
    {
      file1 = (SshFileCopyFile) obj2;
      file2 = (SshFileCopyFile) obj1;
    }
  else
    {
      file1 = (SshFileCopyFile) obj1;
      file2 = (SshFileCopyFile) obj2;
    }
  
  SSH_PRECOND(file1 != NULL);
  SSH_PRECOND(file2 != NULL);

  if (gdata->file_sort_flags & SFTP_FILESORT_SIZE)
    {
      SshFileAttributes attrs1, attrs2;
      attrs1 = ssh_file_copy_file_get_attributes(file1);
      attrs2 = ssh_file_copy_file_get_attributes(file2);
      if (attrs1 && attrs2 &&
          attrs1->flags & SSH_FILEXFER_ATTR_SIZE &&
          attrs2->flags & SSH_FILEXFER_ATTR_SIZE)
        {
          return attrs2->size - attrs1->size;
        }
    }
  
  return strcmp(ssh_file_copy_file_get_name(file1),
                ssh_file_copy_file_get_name(file2));
}

void
sftp_sigint_handler(int sig, void *context)
{
  Sftp2GData gdata = (Sftp2GData) context;

  if (gdata->transfer_op_handle)
    {
      SSH_TRACE(2, ("Aborting the transfer..."));
      ssh_operation_abort(gdata->transfer_op_handle);
      gdata->transfer_op_handle = NULL;
      ssh_fsm_set_next(gdata->main_thread, sftp_get_command);
      SSH_FSM_CONTINUE_AFTER_CALLBACK(gdata->main_thread);
    }
  else
    {
      if (gdata->rconn->client == NULL)
        {
          ssh_fsm_set_next(gdata->main_thread, sftp_finish);
        }
      else
        {
          if (gdata->exit_attempted == TRUE)
            {
              ssh_fsm_set_next(gdata->main_thread, sftp_finish);
              ssh_fsm_continue(gdata->main_thread);
            }
          else
            {
              printf("\r\nYou have an active connection. Press again "
                     "to terminate.\r\n");
              gdata->exit_attempted = TRUE;
            }
        }
    }
}

/* Finalizing the initialization, eg. async stuff, which can't be done
   in main(). */
void local_connect_complete_cb(SshFileClient client,
                               void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);

  SSH_DEBUG(2, ("Connection ready."));

  if (!client)
    ssh_fatal("Local connection failed.");

  gdata->lconn->client = client;


  /* ReadLine stuff. */
  gdata->stdio_stream = ssh_stream_fd_stdio();

  ssh_enter_non_blocking(-1);
  ssh_enter_raw_mode(-1, TRUE);




  /* Ugly, maybe, but I wouldn't add a state for this. -sjl */
  gdata->lcwd_ctx =
    ssh_sftp_cwd_init(".", gdata->lconn->client);

  gdata->rl = ssh_readline_eloop_initialize(gdata->stdio_stream);

  gdata->need_rl_initialization = FALSE;

  if (!gdata->rl)
    ssh_fatal("Couldn't initialize SshReadLine. termcap entrys are "
              "probably wrong, or missing for your terminal type.");

  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}


SSH_FSM_STEP(sftp_connect_local)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;

  SSH_FSM_SET_NEXT(sftp_get_command);
  gdata->lconn->host = NULL;

  SSH_FSM_ASYNC_CALL(ssh_file_copy_connect(gdata->lconn,
                                           local_connect_complete_cb,
                                           thread));
}

/* If condition is true, sets the cwd and connection variables
   for remote action, if it is false, for local action. Returns
   true if there is an error. */
int
sftp_set_cwd_and_conn(Sftp2GData gdata, int condition)
{
  if (condition)
    {
      if (gdata->rconn->client == NULL)
        {
          printf("Error: Not connected.\r\n");
          SFTP_SET_EXIT_VAL(SSH_FC_ERROR);
          return -1;
        }
      gdata->conn = gdata->rconn;
      gdata->cwd_ctx = gdata->rcwd_ctx;
    }
  else
    {
      if (gdata->lconn->client == NULL)
        {
          printf("Error: Local not connected (use lopen).\r\n");
          SFTP_SET_EXIT_VAL(SSH_FC_ERROR);
          return -1;
        }
      gdata->conn = gdata->lconn;
      gdata->cwd_ctx = gdata->lcwd_ctx;
    }
  return 0;
}

/* Returns FALSE if command operates on local files and
   TRUE if on remote files. */
Boolean
sftp_check_for_remote_commands(const char *command)
{
  if (!strcmp(command, "lcd") ||
      !strcmp(command, "lls") ||
      !strcmp(command, "put") ||
      !strcmp(command, "mput") ||
      !strcmp(command, "lrename") ||
      !strcmp(command, "lreadlink") ||
      !strcmp(command, "lsymlink") ||
      !strcmp(command, "lpwd") ||
      !strcmp(command, "lrm") ||
      !strcmp(command, "lmkdir") ||
      !strcmp(command, "lrmdir") ||
      !strcmp(command, "llsroots"))
    return FALSE;
  else
    return TRUE;
}

#define SFTP_CMD_ASCII              1
#define SFTP_CMD_AUTO               2
#define SFTP_CMD_BINARY             3
#define SFTP_CMD_CD                 4
#define SFTP_CMD_CLOSE              5
#define SFTP_CMD_DEBUG              6
#define SFTP_CMD_GET                7
#define SFTP_CMD_GETEXT             8
#define SFTP_CMD_HELP               9
#define SFTP_CMD_LS                11
#define SFTP_CMD_LSROOTS           12
#define SFTP_CMD_MKDIR             13
#define SFTP_CMD_OPEN              14
#define SFTP_CMD_PUT               15
#define SFTP_CMD_PWD               16
#define SFTP_CMD_QUIT              17
#define SFTP_CMD_READLINK          18
#define SFTP_CMD_RENAME            19
#define SFTP_CMD_RM                20
#define SFTP_CMD_RMDIR             21
#define SFTP_CMD_SETEXT            22
#define SFTP_CMD_SYMLINK           23
#define SFTP_CMD_VERBOSE           24



/* Specials. */
/* This flag is ORed in, if the command is "local" in nature. */
#define SFTP_CMD_LOCAL         0x1000
#define SFTP_CMD_NO_CONN       0x2000
#define SFTP_NO_SUCH_COMMAND   0x10000

struct 
{
  char *name;
  SshUInt32 cmd_type;
} sftp_cmds[] =
  {
    { "rm",        SFTP_CMD_RM },
    { "lrm",       SFTP_CMD_RM | SFTP_CMD_LOCAL },
    { "rmdir",     SFTP_CMD_RMDIR },
    { "lrmdir",    SFTP_CMD_RMDIR | SFTP_CMD_LOCAL },
    { "ls",        SFTP_CMD_LS },
    { "lls",       SFTP_CMD_LS | SFTP_CMD_LOCAL },
    { "cd",        SFTP_CMD_CD },
    { "lcd",       SFTP_CMD_CD | SFTP_CMD_LOCAL },
    { "rename",    SFTP_CMD_RENAME },
    { "lrename",   SFTP_CMD_RENAME | SFTP_CMD_LOCAL },
    { "get",       SFTP_CMD_GET },
    { "mget",      SFTP_CMD_GET },
    { "put",       SFTP_CMD_PUT | SFTP_CMD_LOCAL },
    { "mput",      SFTP_CMD_PUT | SFTP_CMD_LOCAL },
    { "readlink",  SFTP_CMD_READLINK },
    { "lreadlink", SFTP_CMD_READLINK | SFTP_CMD_LOCAL },
    { "symlink",   SFTP_CMD_SYMLINK },
    { "lsymlink",  SFTP_CMD_SYMLINK | SFTP_CMD_LOCAL },
    { "mkdir",     SFTP_CMD_MKDIR },
    { "lmkdir",    SFTP_CMD_MKDIR | SFTP_CMD_LOCAL },
    { "pwd",       SFTP_CMD_PWD },
    { "lpwd",      SFTP_CMD_PWD | SFTP_CMD_LOCAL },
    { "lsroots",   SFTP_CMD_LSROOTS },
    { "llsroots",  SFTP_CMD_LSROOTS | SFTP_CMD_LOCAL },
    { "open",      SFTP_CMD_OPEN | SFTP_CMD_NO_CONN},
    { "lopen",     SFTP_CMD_OPEN | SFTP_CMD_LOCAL | SFTP_CMD_NO_CONN},
    { "localopen", SFTP_CMD_OPEN | SFTP_CMD_LOCAL | SFTP_CMD_NO_CONN},
    { "close",     SFTP_CMD_CLOSE },
    { "lclose",    SFTP_CMD_CLOSE | SFTP_CMD_LOCAL },
    { "quit",      SFTP_CMD_QUIT | SFTP_CMD_NO_CONN },
    { "ascii",     SFTP_CMD_ASCII | SFTP_CMD_NO_CONN },
    { "binary",    SFTP_CMD_BINARY | SFTP_CMD_NO_CONN },
    { "auto",      SFTP_CMD_AUTO | SFTP_CMD_NO_CONN },
    { "setext",    SFTP_CMD_SETEXT | SFTP_CMD_NO_CONN },
    { "getext",    SFTP_CMD_GETEXT | SFTP_CMD_NO_CONN },







    { "debug",     SFTP_CMD_DEBUG | SFTP_CMD_NO_CONN },
    { "verbose",   SFTP_CMD_VERBOSE | SFTP_CMD_NO_CONN },
    { "help",      SFTP_CMD_HELP | SFTP_CMD_NO_CONN },
    { NULL,        SFTP_NO_SUCH_COMMAND }
  };

SshLongOptionStruct sftp_getput_longopts[] =
{
  { "preserve-attributes", SSH_GETOPT_LONG_OPTIONAL_ARGUMENT, NULL, 'p'},




  { NULL, 0, NULL, 0}
};

SshUInt32 sftp_cmd_descriptor(const char *command)
{
  int i;
  
  for (i = 0; sftp_cmds[i].name != NULL; i++)
    {
      if (!strcmp(sftp_cmds[i].name, command))
        return sftp_cmds[i].cmd_type;
    }
  return SFTP_NO_SUCH_COMMAND;
}

/* Callback from the readline library. Receives a line from the user
   and the thread in the context. Parses the arguments, and if
   a valid command is recognized, stores relevant information in the
   gdata structure and passes control to a command-specific state.
   */
void
rl_cb(const char *line, unsigned int flags, void *context)
{
#define SSH_SFTP_CMDLINE_WORDS 256
  char *cmdline = NULL, *cmdline_words[SSH_SFTP_CMDLINE_WORDS];
  int words = 0, k = 0, end_of_word = 0, end_of_previous_word = 0;
  int i = 0, j = 0, ch;
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);
  Boolean quote_active, backslash_active, is_remote;
  SshADTHandle h;
  SshUInt32 cmd, cmd_no_opts;
  
  if (!gdata->batchmode)
    SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);

  for (k = 0; k < SSH_SFTP_CMDLINE_WORDS; k++)
    cmdline_words[k] = NULL;

  /* Cleanup after last commands. */
  ssh_xfree(gdata->tabcompl_resultline);
  gdata->tabcompl_resultline = NULL;
  ssh_xfree(gdata->cmd_file);
  gdata->cmd_file = NULL;
  ssh_xfree(gdata->cmd_file_target);
  gdata->cmd_file_target = NULL;
  ssh_adt_clear(gdata->cl_file_list);        
  ssh_getopt_init_data(gdata->getopt_data);
  
  if (!line)
    {
      printf("\r\n");
      SSH_FSM_SET_NEXT(sftp_finish);
      return;
    }
  
  
  SSH_TRACE(10, ("Got line: %s", line));

  printf("\r\n");

  if (strlen(line) == 0)
    return;
  
  /* Do some command line processing: split the line into words,
     taking quotation marks and backslashes into account.

     cmdline is a tmp var for storing the processed words.
     i is index into cmdline_words (array of pointers)
     j is index into cmdline
     k is index into line
  */




  cmdline = ssh_xmalloc(strlen(line) + 32);
  i = 0; j = 0; k = 0;
  quote_active = FALSE;
  backslash_active = FALSE;
  while (i < SSH_SFTP_CMDLINE_WORDS)
    {
      /* Skip through the whitespace until something
         is encountered. */
      while (line[k] == ' ')
        k++;

      /* Now, whatever we have here, it'll cause a new word
         to be recorded. */
      cmdline_words[i++] = &(cmdline[j]);
      /* Copy the characters until the end of the word. */
      do
        {
          backslash_active = FALSE;
          if (line[k] == '\\')
            {
              backslash_active = TRUE;
              k++;
            }
          if (!backslash_active && line[k] == '"')
            {
              if (quote_active)
                quote_active = FALSE;
              else
                quote_active = TRUE;
              k++;
              continue;
            }
          if (line[k] != 0 && (quote_active || backslash_active ||
                               line[k] != ' '))
            cmdline[j++] = line[k++];
        }
      while (line[k] != 0 && (quote_active || backslash_active ||
                              line[k] != ' '));
      cmdline[j++] = 0;
      end_of_previous_word = end_of_word;
      end_of_word = k;
      if (line[k] == 0)
        {
          if (quote_active == TRUE)
            {
              printf("Error: Unterminated quotes\r\n");
              SFTP_SET_EXIT_VAL(SSH_FC_ERROR);
              goto exit;
            }
          break;
        }
    }

  /* KLUDGE-WARNING: */
  /* wipe out the empty strings from the end (whitespaces after cmd) */
  while (i>1 && strlen(cmdline_words[i-1]) == 0)
    i--;

  cmdline_words[i] = NULL;
  words = i;

  cmd = sftp_cmd_descriptor(cmdline_words[0]);
  
  is_remote = (cmd & SFTP_CMD_LOCAL) ? FALSE : TRUE;
  
  if (!(cmd & SFTP_CMD_NO_CONN))
    {
      /* Check that needed connection is valid. */
      if (sftp_set_cwd_and_conn(gdata, is_remote))
        {
          goto exit;
        }
    }

  /* TAB COMPLETION */

  if (flags & SSH_READLINE_CB_FLAG_TAB)
    {
      gdata->command_cd = FALSE;
          
      /* Store the last word of the line, which we will try to complete,
         and the beginning of the line, which will remain the same, and
         branch to a state 'tabcompl_start'. */
      gdata->tabcompl_word_to_complete =
        ssh_xstrdup(cmdline_words[words - 1]);
      if (words > 1)
        {
          gdata->tabcompl_constant_part = ssh_xstrdup(line);
          gdata->tabcompl_constant_part[end_of_previous_word] = '\0';
          gdata->command_cd = !strcmp(cmdline_words[0], "cd") ||
            !strcmp(cmdline_words[0], "lcd");
        }
      else
        {
          gdata->tabcompl_constant_part = NULL;
        }

      if (!(cmd & SFTP_CMD_NO_CONN) &&
          sftp_set_cwd_and_conn(gdata, is_remote) != -1)
        {
          SSH_FSM_SET_NEXT(sftp_tabcompl_start);
        }
      else
        {
          /* Error: not connected. Give user the same line
             back. */
          gdata->tabcompl_resultline = ssh_xstrdup(line);
          SFTP_SET_EXIT_VAL(SSH_FC_ERROR);
          SSH_FSM_SET_NEXT(sftp_get_command);
        }
      goto exit;
    }


  /* NORMAL COMMANDS */

  gdata->exit_attempted = FALSE;
  gdata->file_sort_flags = 0L;
      
  cmd_no_opts = cmd & ~(SFTP_CMD_LOCAL | SFTP_CMD_NO_CONN);
  
  SSH_DEBUG(2, ("Command descriptor 0x%x (0x%x)", cmd,
                cmd_no_opts));
  
  switch (cmd & ~(SFTP_CMD_LOCAL | SFTP_CMD_NO_CONN))
    {
    case SFTP_CMD_QUIT:
      SSH_FSM_SET_NEXT(sftp_finish);
      break;      
    case SFTP_CMD_DEBUG:
      if (cmdline_words[1] == NULL)
        {
          printf("Error: no debug level given.\r\n");
          goto exit;
        }
      ssh_xfree(gdata->debug_level);
      gdata->debug_level = NULL;
      if (!strcmp(cmdline_words[1], "no") ||
          !strcmp(cmdline_words[1], "disable"))
        {
          ssh_debug_clear_modules();
          gdata->verbose_mode = FALSE;
        }
      else
        {
          gdata->debug_level = ssh_xstrdup(cmdline_words[1]);
          ssh_debug_set_level_string(gdata->debug_level);
          gdata->verbose_mode = TRUE;
        }
      break;
    case SFTP_CMD_VERBOSE:
      ssh_xfree(gdata->debug_level);
      gdata->debug_level = ssh_xstrdup("2");
      ssh_debug_set_level_string(gdata->debug_level);
      gdata->verbose_mode = TRUE;
      break;
    case SFTP_CMD_OPEN:
      /* Handle the local connection (no hostname given)
         and remote connection (dup the given hostname) */
      if (is_remote)
        {
          gdata->conn = gdata->rconn;
          gdata->cwd_ctx = gdata->rcwd_ctx;
          gdata->rcwd_ctx = NULL;
          gdata->cmd_doing_remote = TRUE;
        }
      else
        {
          gdata->conn = gdata->lconn;
          gdata->cwd_ctx = gdata->lcwd_ctx;
          gdata->lcwd_ctx = NULL;
          gdata->cmd_doing_remote = FALSE;
        }

      if (cmdline_words[1] == NULL)
        {
          printf("Error: no hostname given.\r\n");
          SFTP_SET_EXIT_VAL(SSH_FC_ERROR);
          break;
        }
      else if (!strcmp(cmdline_words[1], "-l"))
        {
          ssh_xfree(gdata->conn->host);
          gdata->conn->host = NULL;
          printf("Opening a local connection\r\n");
        }
      else
        {
          printf("Opening connection to %s\r\n", cmdline_words[1]);
          ssh_xfree(gdata->conn->host);
          gdata->conn->host = ssh_xstrdup(cmdline_words[1]);
        }

      if (gdata->conn->client)
        {
          SSH_DEBUG(2, ("Closing active %s connection...",
                        is_remote ? "remote" : "local"));
          gdata->next_state = sftp_cmd_open;
          SSH_FSM_SET_NEXT(sftp_cmd_close);
        }
      else
        {
          SSH_FSM_SET_NEXT(sftp_cmd_open);
        }
      break;
    case SFTP_CMD_CLOSE:
      SSH_FSM_SET_NEXT(sftp_cmd_close);
      gdata->next_state = sftp_get_command;
      if (is_remote)
        gdata->rcwd_ctx = NULL;
      else
        gdata->lcwd_ctx = NULL;
      break;  
    case SFTP_CMD_HELP:
      ssh_sftp_print_help(cmdline_words[1]);
      break;
    case SFTP_CMD_CD:
      if (cmdline_words[1] == NULL)
        {
          printf("Error: No directory given to command 'cd'.\r\n");
          SFTP_SET_EXIT_VAL(SSH_FC_ERROR);
          goto exit;
        }








      gdata->cmd_file = ssh_xstrdup(cmdline_words[1]);
      SSH_FSM_SET_NEXT(sftp_cmd_cd_start);
      break;
    case SFTP_CMD_RM:
    case SFTP_CMD_RMDIR:
      if (cmdline_words[1] == NULL)
        {
          printf("Error: No argument given.\r\n");
          SFTP_SET_EXIT_VAL(SSH_FC_ERROR);
          break;
        }
      if (cmdline_words[2] != NULL)
        printf("Warning: This command only takes one argument (rest "
               "are ignored).\r\n");

      gdata->cmd_file = ssh_sftp_cwd_add(cmdline_words[1], gdata->cwd_ctx);

      if ((cmd & ~SFTP_CMD_LOCAL) == SFTP_CMD_RMDIR)
        gdata->cmd_rm_directory = TRUE;
      else
        gdata->cmd_rm_directory = FALSE;

      SSH_FSM_SET_NEXT(sftp_cmd_rm_start);
      break;
    case SFTP_CMD_MKDIR:
      if (cmdline_words[1] == NULL)
        {
          printf("Error: No argument given to command 'mkdir'.\r\n");
          SFTP_SET_EXIT_VAL(SSH_FC_ERROR);
          break;
        }
      if (cmdline_words[2] != NULL)
        printf("Warning: This command only takes one argument (the "
               "rest are ignored).\r\n");

      gdata->cmd_file = ssh_sftp_cwd_add(cmdline_words[1], gdata->cwd_ctx);
      gdata->cmd_mkdir_attrs.flags = 0;
      SSH_FSM_SET_NEXT(sftp_cmd_mkdir_start);
      break;
    case SFTP_CMD_RENAME:
      if (cmdline_words[1] == NULL || cmdline_words[2] == NULL)
        {
          printf("Error: Not enough arguments given to command "
                 "'rename'.\r\n");
          SFTP_SET_EXIT_VAL(SSH_FC_ERROR);
          break;
        }
      if (cmdline_words[3] != NULL)
        printf("Warning: This command only takes two arguments (the "
               "rest are ignored).\r\n");

      gdata->cmd_file = ssh_sftp_cwd_add(cmdline_words[1], gdata->cwd_ctx);
      gdata->cmd_file_target =
        ssh_sftp_cwd_add(cmdline_words[2], gdata->cwd_ctx);
      SSH_FSM_SET_NEXT(sftp_cmd_rename_check);
      break;
    case SFTP_CMD_READLINK:
      if (cmdline_words[1] == NULL)
        {
          printf("Error: No enough arguments given to command "
                 "'readlink'.\r\n");
          SFTP_SET_EXIT_VAL(SSH_FC_ERROR);
          break;
        }
      if (cmdline_words[2] != NULL)
        printf("Warning: This command only takes one argument (the "
               "rest are ignored).\r\n");
      
      gdata->cmd_file =
        ssh_sftp_cwd_add(cmdline_words[1], gdata->cwd_ctx);
      SSH_FSM_SET_NEXT(sftp_cmd_readlink);
      break;
    case SFTP_CMD_SYMLINK:
      if (cmdline_words[1] == NULL || cmdline_words[2] == NULL)
        {
          printf("Error: No enough arguments given to command "
                 "'symlink'.\r\n");
          SFTP_SET_EXIT_VAL(SSH_FC_ERROR);
          break;
        }
      if (cmdline_words[3] != NULL)
        printf("Warning: This command only takes two arguments (the "
               "rest are ignored).\r\n");

      gdata->cmd_file_target = ssh_xstrdup(cmdline_words[1]);
      gdata->cmd_file = ssh_sftp_cwd_add(cmdline_words[2], gdata->cwd_ctx);
      SSH_FSM_SET_NEXT(sftp_cmd_symlink);
      goto exit;
    case SFTP_CMD_PWD:
      SSH_FSM_SET_NEXT(sftp_cmd_pwd_start);
      break;
    case SFTP_CMD_LS:
      /* Go to state command_ls_glob unless option parsing below
         reports an error and directs us back to get_command. */
      SSH_FSM_SET_NEXT(sftp_cmd_ls_glob);

      /* Parse the options and construct the filelist (given
         after the command). Put these in the gdata structure. */
      gdata->cmd_ls_recursive = FALSE;
      gdata->cmd_ls_long_format = FALSE;
      gdata->cmd_ls_no_args = FALSE;
        
      while ((ch = ssh_getopt(words, cmdline_words, "RlSr",
                              gdata->getopt_data)) != -1)
        {
          switch(ch)
            {
            case 'R':
              gdata->cmd_ls_recursive = TRUE;
              break;
            case 'l':
              gdata->cmd_ls_long_format = TRUE;
              break;
            case 'S':
              gdata->file_sort_flags |= SFTP_FILESORT_SIZE;
              break;
            case 'r':
              gdata->file_sort_flags |= SFTP_FILESORT_REVERSE;
              break;
            default:
              SSH_FSM_SET_NEXT(sftp_get_command);
              fprintf(stderr, "\r");
              goto exit;
              break;
            }
        }

      /* If there are no arguments, * is the directory to list. */
      if (gdata->getopt_data->ind == words)
        {
          char *tmp_str = ssh_sftp_cwd_add("*", gdata->cwd_ctx);
          ssh_adt_insert(gdata->cl_file_list, tmp_str);
          gdata->cmd_ls_no_args = TRUE;
        }

      while (gdata->getopt_data->ind < words)
        {
          char *tmp_str;
          tmp_str = ssh_sftp_cwd_add(cmdline_words[gdata->getopt_data->ind],
                                     gdata->cwd_ctx);

          ssh_adt_insert(gdata->cl_file_list, tmp_str);
          gdata->getopt_data->ind++;
        }
      break;
    case SFTP_CMD_GET:
    case SFTP_CMD_PUT:
      /* Mark the direction of the transfer. */
      if (cmd_no_opts == SFTP_CMD_GET)
        gdata->cmd_getput_direction = SSH_SFTP_GET;
      else
        gdata->cmd_getput_direction = SSH_SFTP_PUT;

      /* XXX We'll have to check that the other connection is OK, too. */
        
      /* Go to state command_ls_glob unless option parsing below
         reports an error and directs us back to get_command. */
      SSH_FSM_SET_NEXT(sftp_cmd_getput_glob);

      *gdata->tr_attrs = *gdata->global_tr_attrs;
      
      /* Parse the options and construct the filelist (given
         after the command). Put these in the gdata structure. */
      while ((ch = ssh_getopt_long(words, cmdline_words, "p::W::c::",
                                   sftp_getput_longopts, NULL,
                                   gdata->getopt_data)) != -1)
        {
          Boolean bool_opt = TRUE;
          
          if (gdata->getopt_data->arg &&
              (!strcmp(gdata->getopt_data->arg, "no") ||
               !strcmp(gdata->getopt_data->arg, "disable")))
            bool_opt = FALSE;
              
          switch(ch)
            {
              /* Currently, get and put don't take options. */
            case 'p':
              gdata->tr_attrs->preserve_attributes = bool_opt;
              break;








            default:
              SSH_FSM_SET_NEXT(sftp_get_command);
              goto exit;
              break;
            }
        }

      /* If there are no arguments, complain. */
      if (gdata->getopt_data->ind == words)
        {
          printf("Error: No arguments given to %s.\r\n", cmdline_words[0]);
          SFTP_SET_EXIT_VAL(SSH_FC_ERROR);
          SSH_FSM_SET_NEXT(sftp_get_command);
          break;
        }

      while (gdata->getopt_data->ind < words)
        {
          char *tmp_str;
              
          tmp_str = ssh_sftp_cwd_add(cmdline_words[gdata->getopt_data->ind],
                                     gdata->cwd_ctx);
          SSH_DEBUG(2, ("Adding %s to file list.", tmp_str));
          h = ssh_adt_insert(gdata->cl_file_list, tmp_str);
          SSH_ASSERT(h != SSH_ADT_INVALID);
          gdata->getopt_data->ind++;
        }
      break;
    case SFTP_CMD_ASCII:
      if (cmdline_words[1] != NULL)
        {
          char *newline;

          i = 1;
#define DUMP_NL_CONV(str)                       \
do {                                            \
  if ((str) == NULL)                            \
    printf(" <ask>");                           \
  else                                          \
    for (i = 0; i < strlen(str); i++)           \
      printf(" 0x%x", (str)[i]);                \
} while (0)
          if (!strcmp(cmdline_words[i], "-s"))
            {
              unsigned char *nl;
              
              printf("Newline conventions:\r\n");
              printf("Actual:\r\n");
              printf("\r\nremote:");
              if (gdata->rconn->client == NULL)
                printf(" <not connected>");
              else if (ssh_file_client_get_extension_data(gdata->rconn->client,
                                                          "newline@vandyke.com",
                                                          &nl, NULL))
                DUMP_NL_CONV(nl);
              else
                printf(" <not available>");
              
              printf("\r\nlocal:");
              if (gdata->lconn->client == NULL)
                printf(" <not connected>");
              else if (ssh_file_client_get_extension_data(gdata->lconn->client,
                                                          "newline@vandyke.com",
                                                          &nl, NULL))
                DUMP_NL_CONV(nl);
              else
                printf(" <not available>");
              printf("\r\n");

              printf("\r\nHints:\r\n");
              printf("\r\nremote:");
              DUMP_NL_CONV(gdata->remote_newline);
              printf("\r\nlocal:");
              DUMP_NL_CONV(gdata->client_newline);
              printf("\r\n");

              goto exit;
            }

          if (i < words)
            {
              if (!strcmp(cmdline_words[i], "ask"))
                {
                  if (gdata->batchmode)
                    {
                      printf("In batchmode, ignoring \"ask\" newline "
                             "convention (using default value)\r\n");
                      goto exit;
                    }
                  newline = NULL;
                }
              else if ((newline = ssh_file_copy_parse_newline_convention
                        (cmdline_words[i])) == NULL)
                {
                  /* Unknown newline convention. */
                  goto exit;
                }
              ssh_xfree(gdata->remote_newline);
              gdata->remote_newline = newline;
              i++;
            }
          if (i < words)
            {
              if (!strcmp(cmdline_words[i], "ask"))
                {
                  if (gdata->batchmode)
                    {
                      printf("In batchmode, ignoring \"ask\" newline "
                             "convention (using default value)\r\n");
                      goto exit;
                    }
                  newline = NULL;
                }
              else if ((newline = ssh_file_copy_parse_newline_convention
                        (cmdline_words[i])) == NULL)
                {
                  /* Unknown newline convention. */
                  goto exit;
                }
              ssh_xfree(gdata->client_newline);
              gdata->client_newline = newline;
              i++;
            }
          printf("Newline conventions updated\r\n");
        }

      gdata->global_tr_attrs->transfer_mode =
        SSH_FC_TRANSFER_MODE_ASCII;
      printf("File transfer mode is now ascii\r\n");

      break;
    case SFTP_CMD_BINARY:
      gdata->global_tr_attrs->transfer_mode = SSH_FC_TRANSFER_MODE_BINARY;
      printf("File transfer mode is now binary\r\n");
      break;
    case SFTP_CMD_AUTO:
      gdata->global_tr_attrs->transfer_mode = SSH_FC_TRANSFER_MODE_AUTO;
      printf("File transfer mode is now auto\r\n");
      break;
    case SFTP_CMD_SETEXT:
      {
        int i;
        if (cmdline_words[1] == NULL)
          {
            printf("Error: No argument given to command 'setext'.\r\n");
            SFTP_SET_EXIT_VAL(SSH_FC_ERROR);
            break;
          }
        ssh_xfree(gdata->global_tr_attrs->ascii_extensions);
        gdata->global_tr_attrs->ascii_extensions =
          ssh_xstrdup(cmdline_words[1]);

        for (i = 2; cmdline_words[i] != NULL; i++)
          {
            char *tmp = gdata->global_tr_attrs->ascii_extensions;
            ssh_xdsprintf(&gdata->global_tr_attrs->ascii_extensions,
                          "%s,%s", gdata->global_tr_attrs->ascii_extensions,
                          cmdline_words[i]);
            ssh_xfree(tmp);
          }

        printf("Ascii extensions are : %s\r\n",
               gdata->global_tr_attrs->ascii_extensions);
      }
      break;
    case SFTP_CMD_GETEXT:
      printf("Ascii extensions are : %s\r\n",
             gdata->global_tr_attrs->ascii_extensions);
      break;
























    case SFTP_CMD_LSROOTS:
      {
        unsigned char *ext_data;
        size_t ext_data_len;
        size_t ret_val, bytes = 0L;
        unsigned int type;
        const char *type_str = NULL;
        char *root_name;

        if (!ssh_file_client_get_extension_data(gdata->conn->client,
                                                "fs-roots@vandyke.com",
                                                &ext_data, &ext_data_len))
          {
            printf("No file system roots received from server.\r\n");
            goto exit;
          }

        printf("The %s filesystem has the following roots:\r\n",
               !strcmp(cmdline_words[0], "lsroots") ? "remote" : "local");

        while (1)
          {
            ret_val = ssh_decode_array(ext_data + bytes,
                                       ext_data_len - bytes,
                                       SSH_FORMAT_UINT32_STR,
                                       &root_name, NULL,
                                       SSH_FORMAT_END);
            if (ret_val == 0 || strlen(root_name) == 0)
              break;

            bytes += ret_val;
            ret_val = ssh_decode_array(ext_data + bytes,
                                       ext_data_len - bytes,
                                       SSH_FORMAT_CHAR, &type,
                                       SSH_FORMAT_END);
            bytes += ret_val;

            switch (type)
              {
              case 1:
                type_str = "Unknown drive type";
                break;
              case 2:
                type_str = "Removable";
                break;
              case 3:
                type_str = "Fixed";
                break;
              case 4:
                type_str = "Remote";
                break;
              case 5:
                type_str = "CDROM";
                break;
              case 6:
                type_str = "RAM disk";
                break;
              case 7:
                type_str = "Other";
                break;
              default:
                type_str = "Unknown type value";
              }
            /* XXX official type names. */
            printf("%9s\t%s\r\n", root_name, type_str);
            ssh_xfree(root_name);
          }
        printf("\r\n");
      }
      break;
      
    default:
      printf("Unrecognized command line: '%s'\r\n", line);
      break;
    }


#if 0
  /* Command: MD5SUM */
  if (!strcmp(cmdline_words[0], "md5sum") ||
      !strcmp(cmdline_words[0], "lmd5sum"))
    {
      if (cmdline_words[1] == NULL)
        {
          printf("Error: No enough arguments given to command "
                 "'readlink'.\r\n");
          SFTP_SET_EXIT_VAL(SSH_FC_ERROR);
          goto exit;
        }
      if (cmdline_words[2] != NULL)
        printf("Warning: This command only takes one argument (the "
               "rest are ignored).\r\n");
      /* Set the connection and cwd variables. */
      if (sftp_set_cwd_and_conn(gdata,
                                !strcmp(cmdline_words[0], "readlink")))
        goto exit;

      gdata->cmd_file =
        ssh_sftp_cwd_add(cmdline_words[1], gdata->cwd_ctx);
      SSH_FSM_SET_NEXT(sftp_cmd_readlink);
    }
#endif

 exit:
  if (cmdline)
    ssh_xfree(cmdline);



  return;
}

SSH_FSM_STEP(sftp_get_command)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;
  char *tmp_ptr;

  /* If the initial connection target was specified on the command
     line, try to connect before reading any commands from the user. */
  if (gdata->initial_connect_target != NULL)
    {
      gdata->rconn->host = ssh_xstrdup(gdata->initial_connect_target);
      gdata->initial_connect_target = NULL;
      gdata->conn = gdata->rconn;
      gdata->cmd_doing_remote = TRUE;
      SSH_FSM_SET_NEXT(sftp_cmd_open);
      return SSH_FSM_CONTINUE;
    }

  if (gdata->batchmode)
    {
      if (gdata->batchfile_data == NULL)
        {
          ssh_read_file(gdata->batchfile_name,
                        (unsigned char **)&gdata->batchfile_data,
                        &gdata->batchfile_len);
          if (gdata->batchfile_data == NULL)
            {
              printf("Error: Could not read the batchfile.\r\n");
              SFTP_SET_EXIT_VAL(SSH_FC_ERROR);
              SSH_FSM_SET_NEXT(sftp_finish);
              return SSH_FSM_CONTINUE;
            }
          gdata->batchfile_ptr = gdata->batchfile_data;
        }
      /* Now the batchfile_ptr points to the start of the line to be
         executed. */
      tmp_ptr = gdata->batchfile_ptr;
      while (gdata->batchfile_ptr <
             gdata->batchfile_data + gdata->batchfile_len

             && *gdata->batchfile_ptr != '\n'




             )
        {
          gdata->batchfile_ptr++;
        }
      if (gdata->batchfile_ptr == gdata->batchfile_data + gdata->batchfile_len)
        {
          SSH_FSM_SET_NEXT(sftp_finish);
          return SSH_FSM_CONTINUE;
        }
      *gdata->batchfile_ptr = 0;
      gdata->batchfile_ptr++;
      printf("sftp> %s", tmp_ptr);





      rl_cb(ssh_xstrdup(tmp_ptr), 0, thread);
      return SSH_FSM_CONTINUE;
    }
  else
    {

      SSH_FSM_ASYNC_CALL(ssh_readline_eloop_ext
                         ("sftp> ", gdata->tabcompl_resultline,
                          gdata->rl, rl_cb, SSH_READLINE_CB_FLAG_TAB,
                          thread));





    }
}

/* CD IMPLEMENTATION */

void
command_cd_cb(SshSftpCwdResult result, void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);

  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
  SSH_FSM_SET_NEXT(sftp_cmd_pwd_start);

  if (result == SSH_SFTP_CWD_ERROR)
    {
      printf("CD failed.\r\n");
      /* Do not try to continue if cd failed in batchmode. */
      if (gdata->batchmode)
        {
          printf("Aborting batchmode.\r\n");
          /* We don't know whether the directory didn't exist or whether
             the file just wasn't a directory, so to simplify... */
          SFTP_SET_EXIT_VAL(SSH_FC_ERROR_DEST_NOT_DIR);
          SSH_FSM_SET_NEXT(sftp_finish);
          return;
        }
    }
  return;
}

SSH_FSM_STEP(sftp_cmd_cd_start)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;

  SSH_FSM_ASYNC_CALL(ssh_sftp_cwd_change(gdata->cmd_file,
                                         gdata->cwd_ctx,
                                         command_cd_cb, thread));
}

/* PWD IMPLEMENTATION */

void
command_pwd_cb(SshFileClientError error,
               const char *name,
               const char *long_name,
               SshFileAttributes attrs,
               const char *error_msg,
               const char *lang_tag,
               void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);

  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
  SSH_FSM_SET_NEXT(sftp_get_command);

  if (error != SSH_FX_OK)
    {
      printf("PWD failed.\r\n");
      SFTP_SET_EXIT_VAL(error);
      return;
    }
  printf("%s\r\n", name);
  return;
}

SSH_FSM_STEP(sftp_cmd_pwd_start)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;

  /* Get the sftp-internal cwd and feed it to the realpath
     resolver, and print the results in the callback. */

  gdata->cmd_file = ssh_sftp_cwd_add(NULL, gdata->cwd_ctx);
  SSH_FSM_ASYNC_CALL(ssh_file_client_realpath(gdata->conn->client,
                                              gdata->cmd_file,
                                              command_pwd_cb, thread));
}

/* OPEN IMPLEMENTATION */

void open_completion_cb(SshFileClient client,
                        void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);

  gdata->conn->client = client;

  /* Allocate a SftpCwd context if it does not exist yet and the
     connection has been opened. (If the connection is not open, every
     command that needs the SftpCwd ctx will complain about connection
     not being open). */
  if (gdata->cmd_doing_remote)
    {
      if (gdata->rcwd_ctx)
        ssh_sftp_cwd_uninit(gdata->rcwd_ctx);
      gdata->rcwd_ctx = ssh_sftp_cwd_init(".", gdata->rconn->client);
    }
  else
    {
      if (gdata->lcwd_ctx)
        ssh_sftp_cwd_uninit(gdata->lcwd_ctx);
      gdata->lcwd_ctx = ssh_sftp_cwd_init(".", gdata->lconn->client);
    }

  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_FSM_STEP(sftp_cmd_open)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;

  SSH_FSM_SET_NEXT(sftp_cmd_finalize_open);

  SSH_FSM_ASYNC_CALL(ssh_file_copy_connect(gdata->conn, open_completion_cb,
                                           thread));
}

SSH_FSM_STEP(sftp_cmd_close)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;

  if (gdata->conn->client != NULL)
    {
      ssh_file_client_destroy(gdata->conn->client);
      gdata->conn->client = NULL;
    }
  if (gdata->cwd_ctx != NULL)
    {
      ssh_sftp_cwd_uninit(gdata->cwd_ctx);
      gdata->cwd_ctx = NULL;
    }

  SSH_FSM_SET_NEXT(gdata->next_state);
  gdata->next_state = NULL_FNPTR;
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(sftp_cmd_finalize_open)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;

  if (gdata->conn->client == NULL)
    {
      if (gdata->rconn->host == NULL)
        printf("Local connection failed.\r\n");
      else
        printf("Connecting to %s failed.\r\n",
               gdata->rconn->host);
    }

  SSH_FSM_SET_NEXT(sftp_get_command);
  return SSH_FSM_CONTINUE;
}

/* RM AND RMDIR IMPLEMENTATION */

void
command_rm_cb(SshFileClientError error,
              const char *error_msg,
              const char *lang_tag,
              void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);

  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
  SSH_FSM_SET_NEXT(sftp_get_command);

  if (error != SSH_FX_OK)
    {
      SFTP_SET_EXIT_VAL(error);
      printf("Command failed.\r\n");
    }
  return;
}

SSH_FSM_STEP(sftp_cmd_rm_start)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;
  if (gdata->cmd_rm_directory)
    SSH_FSM_ASYNC_CALL(ssh_file_client_rmdir(gdata->conn->client,
                                             gdata->cmd_file,
                                             command_rm_cb, thread));
  else
    SSH_FSM_ASYNC_CALL(ssh_file_client_remove(gdata->conn->client,
                                              gdata->cmd_file,
                                              command_rm_cb, thread));
}

/* MKDIR IMPLEMENTATION */

void
command_mkdir_cb(SshFileClientError error,
                 const char *error_msg,
                 const char *lang_tag,
                 void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);

  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
  SSH_FSM_SET_NEXT(sftp_get_command);

  if (error != SSH_FX_OK)
    {
      SFTP_SET_EXIT_VAL(error);
      printf("Command failed.\r\n");
    }
  return;
}

SSH_FSM_STEP(sftp_cmd_mkdir_start)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;
  SSH_FSM_ASYNC_CALL(ssh_file_client_mkdir(gdata->conn->client,
                                           gdata->cmd_file,
                                           &gdata->cmd_mkdir_attrs,
                                           command_mkdir_cb, thread));
}

/* RENAME IMPLEMENTATION */

void
command_rename_cb(SshFileClientError error,
                  const char *error_msg,
                  const char *lang_tag,
                  void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);

  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
  SSH_FSM_SET_NEXT(sftp_get_command);

  if (error != SSH_FX_OK)
    {
      if (error == SSH_FX_OP_UNSUPPORTED)
        {
          printf("Server does not have rename capability.\r\n");
        }
      else
        {
          printf("Rename failed.\r\n");
        }
      SFTP_SET_EXIT_VAL(error);
    }
}

SSH_FSM_STEP(sftp_cmd_rename)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;

  SSH_FSM_ASYNC_CALL(ssh_file_client_rename(gdata->conn->client,
                                            gdata->cmd_file,
                                            gdata->cmd_file_target,
                                            command_rename_cb,
                                            thread));
}

void
command_rename_stat_cb(SshFileClientError error,
                       SshFileAttributes attributes,
                       const char *error_msg,
                       const char *lang_tag,
                       void *context)
{
  SshFSMThread thread = (SshFSMThread) context;

  if (error == SSH_FX_NO_SUCH_FILE)
    {
      /* Destination does not exist, so perform the actual rename. */
      SSH_FSM_SET_NEXT(sftp_cmd_rename);
    }
  else
    {
      printf("Target already exists, operation aborted.\r\n");
      SSH_FSM_SET_NEXT(sftp_get_command);
    }
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_FSM_STEP(sftp_cmd_rename_check)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;

  SSH_FSM_ASYNC_CALL(ssh_file_client_stat(gdata->conn->client,
                                          gdata->cmd_file_target,
                                          command_rename_stat_cb,
                                          thread));
}

/* READLINK IMPLEMENTATION */
void
command_readlink_cb(SshFileClientError error,
                    const char *name,
                    const char *long_name,
                    SshFileAttributes attrs,
                    const char *error_msg,
                    const char *lang_tag,
                    void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);

  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
  SSH_FSM_SET_NEXT(sftp_get_command);

  if (error != SSH_FX_OK)
    {
      if (error_msg)
        {
          SSH_DEBUG(2, ("Received error %d, msg \"%s\", lang_tag \"%s\"",
                        error, error_msg, lang_tag));
        }
      if (error == SSH_FX_OP_UNSUPPORTED)
        {
          printf("Server does not have readlink capability.\r\n");
        }
      else
        {
          printf("Readlink failed.\r\n");
        }
      SFTP_SET_EXIT_VAL(error);
    }
  else
    {
      printf("%s -> %s\r\n", gdata->cmd_file, name);
    }
}

SSH_FSM_STEP(sftp_cmd_readlink)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;

  SSH_FSM_ASYNC_CALL(ssh_file_client_readlink(gdata->conn->client,
                                              gdata->cmd_file,
                                              command_readlink_cb,
                                              thread));
}

/* SYMLINK IMPLEMENTATION */
void
command_symlink_cb(SshFileClientError error,
                    const char *error_msg,
                    const char *lang_tag,
                    void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);

  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
  SSH_FSM_SET_NEXT(sftp_get_command);

  if (error != SSH_FX_OK)
    {
      if (error_msg)
        {
          SSH_DEBUG(2, ("Received error %d, msg \"%s\", lang_tag \"%s\"",
                        error, error_msg, lang_tag));
        }
      if (error == SSH_FX_OP_UNSUPPORTED)
        {
          printf("Server does not have symlink capability.\r\n");
        }
      else
        {
          printf("Symlink failed.\r\n");
        }
      SFTP_SET_EXIT_VAL(error);
    }
}

SSH_FSM_STEP(sftp_cmd_symlink)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;

  SSH_FSM_ASYNC_CALL(ssh_file_client_symlink(gdata->conn->client,
                                             gdata->cmd_file,
                                             gdata->cmd_file_target,
                                             command_symlink_cb,
                                             thread));
}

/* GETPUT IMPLEMENTATION */

/* Receive the munged filelist from the globbing function. Go to
   command_getput_recurse and call the recurse_dirs function
   from there. */
void
command_getput_op_ready_cb(SshFileCopyError error,
                           const char *error_message,
                           SshADTContainer file_list,
                           void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
  if (error == SSH_FC_OK)
    {
      ssh_adt_destroy(gdata->file_list);
      gdata->file_list = file_list;
    }
  else
    {
      printf("error message: \"%s\"\r\n", error_message);
      SFTP_SET_EXIT_VAL(error);
      SSH_FSM_SET_NEXT(sftp_get_command);
    }
  return;
}

void
command_getput_op_error_cb(SshFileCopyError error,
                             const char *error_message,
                             void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);
  SFTP_SET_EXIT_VAL(error);
  printf("%s\r\n", error_message);
}


SSH_FSM_STEP(sftp_cmd_getput_glob)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;

  SSH_FSM_SET_NEXT(sftp_cmd_getput_recurse);

  SSH_FSM_ASYNC_CALL(ssh_file_copy_glob_multiple(gdata->conn,
                                                 gdata->cl_file_list,
                                                 NULL,
                                                 command_getput_op_ready_cb,
                                                 command_getput_op_error_cb,
                                                 thread));
}

SSH_FSM_STEP(sftp_cmd_getput_recurse)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;
  SSH_FSM_SET_NEXT(sftp_cmd_getput_transfer);
  SSH_FSM_ASYNC_CALL(ssh_file_copy_recurse_multiple
                     (gdata->conn,
                      gdata->file_list,
                      gdata->recurse_attrs,
                      NULL_FNPTR,
                      command_getput_op_ready_cb,
                      command_getput_op_error_cb,
                      thread));
}

void
command_getput_transfer_ready_cb(SshFileCopyError error,
                                const char *error_message,
                                void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
  gdata->transfer_op_handle = NULL;
  SFTP_SET_EXIT_VAL(error);
  SSH_FSM_SET_NEXT(sftp_get_command);
  ssh_adt_destroy(gdata->file_list);
  gdata->file_list = NULL;
}

void
command_getput_transfer_error_cb(SshFileCopyError error,
                                const char *error_message,
                                void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);
  SFTP_SET_EXIT_VAL(error);
  printf("%s\r\n", error_message);
}

char *stat_eta(SshUInt64 secs)
{
  static char stat_result[9];
  int hours, mins;

   hours = secs / 3600L;
   secs %= 3600L;
   mins = secs / 60L;
   secs %= 60L;

   ssh_snprintf(stat_result, sizeof(stat_result),
                "%02d:%02d:%02d", hours, mins, (int)secs);
   return(stat_result);
}

SSH_FSM_STEP(sftp_cmd_getput_transfer)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;
  SshFileCopyConnection conn1, conn2;
  
  /* Choose the connections depending on the command. */
  ssh_xfree(gdata->tr_attrs->source_newline);
  ssh_xfree(gdata->tr_attrs->dest_newline);

  if (gdata->tr_attrs->transfer_mode &
      (SSH_FC_TRANSFER_MODE_AUTO | SSH_FC_TRANSFER_MODE_ASCII))
    {
      SSH_DEBUG(2, ("ASCII transfer specified."));
      if (gdata->remote_newline == NULL || gdata->client_newline == NULL)
        gdata->tr_attrs->transfer_mode |=
          SSH_FC_TRANSFER_MODE_NL_CONV_ASK;
    }

  if (gdata->cmd_getput_direction == SSH_SFTP_GET)
    {
      gdata->cmd_file = ssh_sftp_cwd_add(".", gdata->lcwd_ctx);

      gdata->tr_attrs->source_newline =
        gdata->remote_newline ? ssh_xstrdup(gdata->remote_newline) : NULL;
      gdata->tr_attrs->dest_newline =
        gdata->client_newline ? ssh_xstrdup(gdata->client_newline) : NULL;

      conn1 = gdata->rconn;
      conn2 = gdata->lconn;
    }
  else
    {
      /* SSH_SFTP_PUT */
      gdata->cmd_file = ssh_sftp_cwd_add(".", gdata->rcwd_ctx);

      gdata->tr_attrs->source_newline =
        gdata->client_newline ? ssh_xstrdup(gdata->client_newline) : NULL;
      gdata->tr_attrs->dest_newline =
        gdata->remote_newline ? ssh_xstrdup(gdata->remote_newline) : NULL;

      conn1 = gdata->lconn;
      conn2 = gdata->rconn;
    }
      
  SSH_FSM_ASYNC_CALL(gdata->transfer_op_handle =
                     ssh_file_copy_transfer_multiple
                     (conn1,
                      gdata->file_list,
                      conn2,
                      gdata->cmd_file,
                      gdata->tr_attrs,
                      command_getput_transfer_ready_cb,
                      command_getput_transfer_error_cb,
                      thread));
}

/* LS IMPLEMENTATION */

/* The state-callback-mess may be a little hard to follow. */

void command_ls_op_error_cb(SshFileCopyError error,
                            const char *error_message,
                            void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);
  SFTP_SET_EXIT_VAL(error);
  SSH_DEBUG(2, ("Non-fatal error %d: msg: %s", error, error_message));
}

void command_ls_op_ready_cb(SshFileCopyError error,
                            const char *error_message,
                            SshADTContainer file_list,
                            void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);
  SSH_TRACE(2, ("Operation complete."));
  
  ssh_adt_destroy(gdata->file_list);
  gdata->file_list = file_list;

  if (error == SSH_FC_ERROR_NO_MATCH && gdata->cmd_ls_no_args)
    {
      SSH_DEBUG(3, ("No dir entries, pattern ``*'' returned no matches "
                    "(error_msg = ``%s''", error_message));
      SSH_FSM_SET_NEXT(sftp_get_command);
    }
  else if (error != SSH_FC_OK)
    {
      printf("error message: \"%s\"\r\n", error_message);
      SFTP_SET_EXIT_VAL(error);
      SSH_FSM_SET_NEXT(sftp_get_command);
    }
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_FSM_STEP(sftp_cmd_ls_glob)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;

  if (gdata->cmd_ls_recursive)
    SSH_FSM_SET_NEXT(sftp_cmd_ls_recurse);
  else
    SSH_FSM_SET_NEXT(sftp_cmd_ls_walk_list_prepare);

  SSH_DEBUG(2, ("ls: starting glob."));
  
  SSH_FSM_ASYNC_CALL(ssh_file_copy_glob_multiple(gdata->conn,
                                                 gdata->cl_file_list,
                                                 gdata->glob_attrs,
                                                 command_ls_op_ready_cb,
                                                 command_ls_op_error_cb,
                                                 thread));
}

SSH_FSM_STEP(sftp_cmd_ls_recurse)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;

  SSH_FSM_SET_NEXT(sftp_cmd_ls_walk_list_prepare);

  SSH_DEBUG(2, ("ls: starting recurse."));

  SSH_FSM_ASYNC_CALL(ssh_file_copy_recurse_multiple(gdata->conn,
                                                    gdata->file_list,
                                                    gdata->recurse_attrs,
                                                    NULL_FNPTR,
                                                    command_ls_op_ready_cb,
                                                    command_ls_op_error_cb,
                                                    thread));
}

typedef struct ListWalkerTDataRec
{
  SshADTHandle h;
  SshADTContainer c;
  SshFSMCondition child_cv;
  SshFSMCondition parent_cv;
  SshFileCopyFile parent_dir;
  SshFSMThread last_ct;
} *ListWalkerTData, ListWalkerTDataStruct;

void list_walker_destroy(SshFSM fsm, void *context)
{
  ListWalkerTData tdata = (ListWalkerTData) context;
  if (tdata->child_cv)
    ssh_fsm_condition_destroy(tdata->child_cv);
  ssh_xfree(tdata);
}

typedef struct LsCheckerTDdataRec
{
  SshFileCopyFile cur_file;
  SshADTContainer top_c;
  SshADTHandle* top_h;
  char *full_path;
} * LsCheckerTDdata, LsCheckerTDdataStruct;

void ls_checker_destroy(SshFSM fsm, void *context)
{
  LsCheckerTDdata tdata = (LsCheckerTDdata) context;
  ssh_xfree(tdata->full_path);
  ssh_xfree(tdata);  
}

SSH_FSM_STEP(sftp_cmd_ls_walk_list_prepare)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;
  SshFSMThread ct;
  ListWalkerTData ct_ctx;
  
  SSH_FSM_SET_NEXT(sftp_cmd_ls_finish);
  
  gdata->pager_ctx = ssh_sftp_pager_init(gdata->stdio_stream,
                                         fsm,
                                         ssh_fc_progcb_screen_width,
                                         ssh_fc_progcb_screen_height);

  if (gdata->batchmode)
    ssh_sftp_pager_dont_ask(gdata->pager_ctx);

  ssh_adt_list_sort(gdata->file_list);

  ct_ctx = ssh_xcalloc(1, sizeof(*ct_ctx));
  ct_ctx->c = gdata->file_list;
  ct_ctx->h = ssh_adt_enumerate_start(gdata->file_list);
  ct = ssh_fsm_thread_create(fsm, sftp_cmd_ls_walk_list,
                             NULL_FNPTR, list_walker_destroy, ct_ctx);
  SSH_ASSERT(ct != NULL);
  SSH_FSM_WAIT_THREAD(ct);  
}

void sftp_ls_dump_file(Sftp2GData gdata, SshFileCopyFile file)
{
  const char *name, *long_name;
  SshFileAttributes attrs;
  
  name = ssh_file_copy_file_get_name(file);
  long_name = ssh_file_copy_file_get_long_name(file);
  attrs = ssh_file_copy_file_get_attributes(file);
  
  SSH_PRECOND(name);
  
  if (gdata->cmd_ls_long_format == TRUE && long_name)
    {      
      ssh_sftp_pager_write_string(gdata->pager_ctx, long_name);
    }
  else
    {
      ssh_sftp_pager_write_string(gdata->pager_ctx, name);

      /* Recurse should get us these. */
      if (attrs && (attrs->flags & SSH_FILEXFER_ATTR_PERMISSIONS))
        {
          /* Output fancy chars after filename, ls-style. */
          if ((attrs->permissions & S_IFMT) == S_IFDIR)
            {
              ssh_sftp_pager_write_string(gdata->pager_ctx, "/");          
            }
          else if ((attrs->permissions & S_IFMT) == S_IFLNK)
            {
              ssh_sftp_pager_write_string(gdata->pager_ctx, "@");
            }
          else if ((attrs->permissions & S_IFMT) == S_IFREG &&
                   attrs->permissions & 0111)
            {
              ssh_sftp_pager_write_string(gdata->pager_ctx, "*");
            }
        }
    }  
}

SSH_FSM_STEP(sftp_cmd_ls_walk_list)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;
  ListWalkerTData tdata = (ListWalkerTData) thread_context;
  
  if (tdata->h != SSH_ADT_INVALID)
    {
      SshFSMThread ct;
      SshFileCopyFile file = ssh_adt_get(tdata->c, tdata->h);
      SshFileAttributes attrs;
      SshADTContainer de;
      SSH_ASSERT(file);
      
      attrs = ssh_file_copy_file_get_attributes(file);
      de = ssh_file_copy_file_get_dir_entries(file);
      
      if (ssh_file_copy_file_get_parent(file) == NULL && de && !attrs)
        {
          /* File is just a basedir, and the dir entries should be
             walked just as the topmost file list. */
          SshFSMThread ct;
          ListWalkerTData ct_ctx;
          
          ssh_adt_list_sort(de);
          ct_ctx = ssh_xcalloc(1, sizeof(*ct_ctx));
          ct_ctx->c = de;
          ct_ctx->h = ssh_adt_enumerate_start(de);
          ct = ssh_fsm_thread_create(fsm, sftp_cmd_ls_walk_list,
                                     NULL_FNPTR, list_walker_destroy, ct_ctx);
          
          SSH_ASSERT(ct != NULL);
          tdata->h = ssh_adt_enumerate_next(tdata->c, tdata->h);
          SSH_FSM_WAIT_THREAD(ct);  
        }
      
      if (de)
        {
          ListWalkerTData ct_ctx = ssh_xcalloc(1, sizeof(*ct_ctx));

          SSH_DEBUG(5, ("Dir has %d files", ssh_adt_num_objects(de)));
          /* This is done so that "root" dirs for these files get
             printed correctly. */
          ssh_adt_list_sort(de);
          ct_ctx->h = ssh_adt_enumerate_start(de);
          ct_ctx->c = de;
          ct_ctx->parent_dir = file;
          ct = ssh_fsm_thread_create(fsm, sftp_cmd_ls_step,
                                     NULL_FNPTR, list_walker_destroy, ct_ctx);
          SSH_ASSERT(ct != NULL);

          tdata->h = ssh_adt_enumerate_next(tdata->c, tdata->h);
          SSH_FSM_WAIT_THREAD(ct);
        }
      if (attrs)
        {
          sftp_ls_dump_file(gdata, file);
          ssh_sftp_pager_write_string(gdata->pager_ctx, "\r\n");
          tdata->h = ssh_adt_enumerate_next(tdata->c, tdata->h);
        }
      else
        {
          /* Additional checks are needed, as filecopy didn't have
             to parse this file. */
          /* Spawn a child to perform these. */
          /* It will lstat the file, and if the file is a directory, a
             one-level recurse will be performed. The SshFileCopyFile
             elements of the file_list returned by that recurse will
             replace this current file. */
          LsCheckerTDdata ct_ctx = ssh_xcalloc(1, sizeof(*ct_ctx));

          /* Note that we don't update the ADT handle here, as we need
             it if we have to replace the file (if it is a
             directory). */
          ct_ctx->cur_file = file;
          ct_ctx->top_h = &tdata->h;
          ct_ctx->top_c = tdata->c;
          
          ct = ssh_fsm_thread_create(fsm,
                                     sftp_cmd_ls_check,
                                     NULL_FNPTR, ls_checker_destroy,
                                     ct_ctx);
          SSH_FSM_WAIT_THREAD(ct);
        }
      return SSH_FSM_CONTINUE;
    }
  return SSH_FSM_FINISH;
}

SSH_FSM_STEP(sftp_cmd_ls_step_wait_cv)
{
  ListWalkerTData tdata = (ListWalkerTData) thread_context;
  SSH_FSM_SET_NEXT(sftp_cmd_ls_step);
  SSH_DEBUG(4, ("Waiting for parent to signal our condition variable..."));
  SSH_FSM_CONDITION_WAIT(tdata->parent_cv);
}

/* XXX Broken. */
SSH_FSM_STEP(sftp_cmd_ls_step)
{  
  Sftp2GData gdata = (Sftp2GData) fsm_context;
  ListWalkerTData tdata = (ListWalkerTData) thread_context;

  SSH_DEBUG(10, ("step (thread: 0x%p)", thread));
  if (tdata->parent_dir)
    {
      char *hlp, *hlp2;

      hlp2 = ssh_file_copy_file_generate_full_path(tdata->parent_dir);
      hlp = ssh_sftp_cwd_strip(hlp2, gdata->cwd_ctx);
      if (hlp != NULL)
        ssh_xfree(hlp2);
      else
        hlp = hlp2;
      
      ssh_sftp_pager_write_string(gdata->pager_ctx, "\r\n");
      ssh_sftp_pager_write_string(gdata->pager_ctx, hlp);
      ssh_sftp_pager_write_string(gdata->pager_ctx, ":\r\n\r\n");
      tdata->parent_dir = NULL;
      ssh_xfree(hlp);
    }
  
  if (tdata->h != SSH_ADT_INVALID)
    {
      SshFileCopyFile file = (SshFileCopyFile) ssh_adt_get(tdata->c, tdata->h);
      SshADTContainer de;
      
      de = ssh_file_copy_file_get_dir_entries(file);

      if (de != NULL)
        {
          /* Spawn child to handle directory. */
          ListWalkerTData ct_ctx = ssh_xcalloc(1, sizeof(*ct_ctx));

          ssh_adt_list_sort(de);
          SSH_DEBUG(5, ("Dir has %d files", ssh_adt_num_objects(de)));

          if (!tdata->child_cv)
            {
              tdata->child_cv = ssh_fsm_condition_create(fsm);
              SSH_ASSERT(tdata->child_cv != NULL);
            }

          ct_ctx->c = de;
          ct_ctx->parent_cv = tdata->child_cv;
          ct_ctx->h = ssh_adt_enumerate_start(de);
          ct_ctx->parent_dir = file;
          tdata->last_ct = ssh_fsm_thread_create(fsm,
                                                 sftp_cmd_ls_step_wait_cv,
                                                 NULL_FNPTR,
                                                 list_walker_destroy, ct_ctx);
          SSH_ASSERT(tdata->last_ct != NULL);
        }
      
      sftp_ls_dump_file(gdata, file);
      
      ssh_sftp_pager_write_string(gdata->pager_ctx, "\r\n");
      ssh_sftp_pager_wakeup(gdata->pager_ctx, thread);

      tdata->h  = ssh_adt_enumerate_next(tdata->c, tdata->h);
    }
  else
    {
      /* The list is at end. */
      SSH_FSM_SET_NEXT(sftp_cmd_ls_step_finish);

      if (tdata->last_ct != NULL)
        {
          SSH_FSM_CONDITION_SIGNAL(tdata->child_cv);
          SSH_FSM_WAIT_THREAD(tdata->last_ct);
        }
      else
        {
          return SSH_FSM_CONTINUE;
        }
    }
  /* Give the pager a chance to do it's job. */
  return SSH_FSM_YIELD;
}

SSH_FSM_STEP(sftp_cmd_ls_step_finish)
{
  ListWalkerTData tdata = (ListWalkerTData) thread_context;
  /* Signal condition variable to wake possible siblings. */
  if (tdata->parent_cv)
    {
      SSH_FSM_CONDITION_SIGNAL(tdata->parent_cv);
    }
  return SSH_FSM_FINISH;
}

static void sftp_ls_chk_attr_cb(SshFileClientError error,
                                    SshFileAttributes attrs,
                                    const char *error_msg,
                                    const char *lang_tag,
                                    void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  LsCheckerTDdata tdata = (LsCheckerTDdata) ssh_fsm_get_tdata(thread);
  
  if (error == SSH_FX_OK)
    {
      ssh_file_copy_file_register_attributes(tdata->cur_file,
                                             ssh_file_attributes_dup(attrs));
      
    }
  else
    {
      if (error_msg)
        printf("Couldn't stat file ``%s'', error msg: %s\r\n",
               tdata->full_path, error_msg);
      else
        printf("Couldn't stat file ``%s'' (%d)\r\n",
               tdata->full_path, error);
    }
  
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_FSM_STEP(sftp_cmd_ls_check)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;
  LsCheckerTDdata tdata = (LsCheckerTDdata) thread_context;

  tdata->full_path = ssh_file_copy_file_generate_full_path(tdata->cur_file);
  
  SSH_FSM_SET_NEXT(sftp_cmd_ls_check_recurse);
  
  SSH_FSM_ASYNC_CALL(ssh_file_client_lstat(gdata->conn->client,
                                           tdata->full_path,
                                           sftp_ls_chk_attr_cb,
                                           thread));
}

SshUInt32 sftp_ls_chk_recurse_func(const char *filename,
                                   const char *long_name,
                                   SshFileAttributes attrs,
                                   void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  LsCheckerTDdata tdata = (LsCheckerTDdata) ssh_fsm_get_tdata(thread);

  if (!strcmp(filename, tdata->full_path))
    return SSH_FCR_ADD | SSH_FCR_RECURSE;
  else
    return SSH_FCR_ADD; 
}

void sftp_ls_chk_recurse_ready_cb(SshFileCopyError error,
                                  const char *error_message,
                                  SshADTContainer file_list,
                                  void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);
  LsCheckerTDdata tdata = (LsCheckerTDdata) ssh_fsm_get_tdata(thread);

  if (error == SSH_FC_OK)
    {
      if (ssh_adt_num_objects(file_list) < 1)
        {
          /* The recursion resulted in an empty list (the file to
             recurse didn't exist etc. So, it should be skipped. */
          tdata->top_h = ssh_adt_enumerate_next(tdata->top_c, *tdata->top_h);
        }
      else
        {
          /* Get the matched file from the list. */
          SshFileCopyFile file;
          SshADTHandle h = ssh_adt_get_handle_to_location(file_list,
                                                          SSH_ADT_BEGINNING);
          SSH_ASSERT(ssh_adt_num_objects(file_list) == 1);
      
          file = ssh_adt_detach(file_list, h);
          ssh_adt_destroy(file_list);
          ssh_adt_delete(tdata->top_c, *tdata->top_h);
          h = ssh_adt_insert(tdata->top_c, file);
          SSH_ASSERT(h != SSH_ADT_INVALID);
          ssh_adt_list_sort(tdata->top_c);
          /* Find the inserted object, and move the main handle to that
             location. */
          *tdata->top_h = SSH_ADT_INVALID;
          for (h = ssh_adt_enumerate_start(tdata->top_c);
               h != SSH_ADT_INVALID;
               h = ssh_adt_enumerate_next(tdata->top_c, h))
            {
              SshFileCopyFile hlp = ssh_adt_get(tdata->top_c, h);
              if (hlp == file)
                { 
                  *tdata->top_h = h;
                  break;
                }
            }
          SSH_POSTCOND(*tdata->top_h != SSH_ADT_INVALID);
        }
    }
  else
    {
      printf("error message: \"%s\"\r\n", error_message);
      SFTP_SET_EXIT_VAL(error);
      SSH_FSM_SET_NEXT(sftp_get_command);
    }
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

void sftp_ls_chk_recurse_error_cb(SshFileCopyError error,
                                  const char *error_message,
                                  void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);
  SFTP_SET_EXIT_VAL(error);
  printf("%s\r\n", error_message);
}

SSH_FSM_STEP(sftp_cmd_ls_check_recurse)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;
  LsCheckerTDdata tdata = (LsCheckerTDdata) thread_context;
  SshFileAttributes attrs = ssh_file_copy_file_get_attributes(tdata->cur_file);

  SSH_FSM_SET_NEXT(sftp_finish_thread);
  
  if (!attrs)
    {
      /* File doesn't exist. Skip it and we're finished. */
      *tdata->top_h = ssh_adt_enumerate_next(tdata->top_c, *tdata->top_h);
      return SSH_FSM_FINISH;
    }
  
  if (attrs->flags & SSH_FILEXFER_ATTR_PERMISSIONS)
    {
      /* Check for directory. */
      if ((attrs->permissions & S_IFMT) == S_IFDIR ||
          ((attrs->permissions & S_IFMT) == S_IFLNK &&
           ssh_file_copy_file_get_parent(tdata->cur_file) == NULL))
        {
          SSH_FSM_ASYNC_CALL(ssh_file_copy_recurse
                             (gdata->conn,
                              tdata->full_path,
                              gdata->recurse_attrs,
                              sftp_ls_chk_recurse_func,
                              sftp_ls_chk_recurse_ready_cb,
                              sftp_ls_chk_recurse_error_cb,
                              thread));
        }
    }
  else
    {
      SSH_TRACE(0, ("We didn't receive file permissions from server. Bug "
                    "in other end."));
    }  
  return SSH_FSM_FINISH;
}

void
command_ls_pager_finish_cb(SshSftpPagerResult result,
                           void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  SSH_TRACE(4, ("Pager Finished"));
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
  SSH_FSM_SET_NEXT(sftp_get_command);
}

SSH_FSM_STEP(sftp_cmd_ls_finish)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;

  ssh_adt_destroy(gdata->file_list);
  gdata->file_list = NULL;
  
  ssh_sftp_pager_wakeup(gdata->pager_ctx, thread);

  SSH_FSM_SET_NEXT(sftp_get_command);
  SSH_TRACE(4, ("Asking the pager to finish."));
  SSH_FSM_ASYNC_CALL(ssh_sftp_pager_uninit(gdata->pager_ctx,
                                           command_ls_pager_finish_cb,
                                           thread));
}

void
tabcompl_error_cb(SshFileCopyError error,
                    const char *error_message,
                    void *context)
{
  SSH_DEBUG(2, ("tab_compl: %s", error_message));
}

void
tabcompl_ready_cb(SshFileCopyError error,
                    const char *error_message,
                    SshADTContainer file_list,
                    void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  Sftp2GData gdata = (Sftp2GData) ssh_fsm_get_gdata(thread);

  if (error == SSH_FC_OK || error == SSH_FC_ERROR_NO_MATCH)
    {
      ssh_adt_destroy(gdata->file_list);
      gdata->file_list = file_list;
    }
  else
    {
      SSH_FSM_SET_NEXT(sftp_tabcompl_finish);
    }
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_FSM_STEP(sftp_tabcompl_start)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;
  char *tmp_str;
  SshADTHandle h;
  
  SSH_FSM_SET_NEXT(sftp_tabcompl_results);

  SSH_TRACE(5, ("constant part: %s, part to be completed: %s",
                gdata->tabcompl_constant_part ?
                gdata->tabcompl_constant_part : "(NULL)",
                gdata->tabcompl_word_to_complete));

  gdata->tabcompl_original_word_to_complete =
    ssh_xstrdup(gdata->tabcompl_word_to_complete);

  /* Add the current directory to the string we are trying to complete. */
  tmp_str = ssh_sftp_cwd_add(gdata->tabcompl_word_to_complete,
                             gdata->cwd_ctx);
  
  ssh_xfree(gdata->tabcompl_word_to_complete);
  gdata->tabcompl_word_to_complete = tmp_str;
  /* XXX Change to use ssh_file_copy_glob() as soon as it is done. */
  ssh_adt_clear(gdata->cl_file_list);
  h = ssh_adt_insert(gdata->cl_file_list,
                     ssh_xstrdup(gdata->tabcompl_word_to_complete));
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_FSM_ASYNC_CALL(ssh_file_copy_glob_multiple(gdata->conn,
                                        gdata->cl_file_list,
                                        gdata->glob_attrs,
                                        tabcompl_ready_cb,
                                        tabcompl_error_cb,
                                        thread));
}

char *sftp_apply_quotation(const char *str)
{
  char *tmp_str, *tmp_ptr;
  int i, len;
  /* Apply quotes to word_to_complete. */
  len = strlen(str);
  tmp_str = ssh_xmalloc(2 * len + 1);
  tmp_ptr = tmp_str;
  for (i = 0; i < len; i++)
    {
      if (str[i] == ' ' || str[i] == '"')
        *tmp_ptr++ = '\\';
      *tmp_ptr++ = str[i];
    }
  *tmp_ptr = '\0';
  return tmp_str;
}

typedef struct TabcomplPathListCtxRec
{
  Sftp2GData gdata;
  SshADTContainer path_list;
} *TabcomplPathListCtx, TabcomplPathListCtxStruct;

void sftp_tabcompl_path_list_gen(SshFileCopyFile file, void *context)
{
  TabcomplPathListCtx ctx = (TabcomplPathListCtx) context;
  SshFileAttributes attrs;
  SshADTHandle h;
  char *full_path, *hlp;
  Boolean is_dir = FALSE, accept_this = FALSE;
  
  /* If file has dir_entries, it was not matched completely, i.e. it
     shouldn't be displayed. */
  if (ssh_file_copy_file_get_dir_entries(file))
    return;

  attrs = ssh_file_copy_file_get_attributes(file);
  if (attrs && (attrs->flags & SSH_FILEXFER_ATTR_PERMISSIONS))
    {
      if ((attrs->permissions & S_IFMT) == S_IFDIR)
        is_dir = TRUE;
    }
  else
    {
      accept_this = TRUE;
    }
  
  if (ctx->gdata->command_cd)
    {
      if (is_dir)
        {
          /* If command is cd, we only accept files with no valid
             attributes or directories. */
          accept_this = TRUE;
        }
    }
  else
    {
      accept_this = TRUE;
    }
  
  if (!accept_this)
    return;
  
  full_path = ssh_file_copy_file_generate_full_path(file);
  
  hlp = ssh_sftp_cwd_strip(full_path, ctx->gdata->cwd_ctx);
  if (hlp != NULL)
    ssh_xfree(full_path);
  else
    /* Didn't contain CWD. */
    hlp = full_path;
  
  if (is_dir)
    {
      full_path = hlp;
      ssh_xdsprintf(&hlp, "%s/", full_path);
      ssh_xfree(full_path);
    }
  
  h = ssh_adt_insert(ctx->path_list, hlp);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  
  SSH_DEBUG(10, ("added ``%s''", hlp));
}

SshADTContainer ssh_sftp_get_path_list(Sftp2GData gdata,
                                       SshFileCopyFile file)
{
  TabcomplPathListCtx ctx = ssh_xcalloc(1, sizeof(*ctx));
  SshADTContainer path_list;
  
  path_list = ssh_adt_create_generic(SSH_ADT_LIST,
                                     SSH_ADT_DESTROY,
                                     ssh_adt_callback_destroy_free,
                                     SSH_ADT_COMPARE,
                                     ssh_adt_callback_compare_str,
                                     SSH_ADT_ARGS_END);
  SSH_ASSERT(path_list != NULL);
  ctx->path_list = path_list;
  ctx->gdata = gdata;
  
  ssh_file_copy_file_traverse(file, sftp_tabcompl_path_list_gen,
                              ctx);
  ssh_xfree(ctx);
  
  return path_list;
}

char *
sftp_get_longest_common_prefix(Sftp2GData gdata, SshADTContainer path_list)
{
  SshADTHandle h;
  char *prefix, *cur_str;
  size_t prefix_len, i;
  
  h = ssh_adt_enumerate_start(path_list);

  if (h == NULL)
    {
      /* Items in the path list didn't match our criteria (for cd, for
         example). */
      return NULL;
    }
  
  prefix = (char *) ssh_adt_get(path_list, h);
  SSH_ASSERT(h != NULL);

  prefix_len = strlen(prefix);
  
  for (h = ssh_adt_enumerate_next(path_list, h);
       h != SSH_ADT_INVALID && prefix_len > 0;
       h = ssh_adt_enumerate_next(path_list, h))
    {
      cur_str = ssh_adt_get(path_list, h);

      for (i = 0; i < prefix_len; i++)
        if (cur_str[i] != prefix[i])
          {
            prefix_len = i;
            break;
          }
    }

  if (prefix_len > 0)
    {
      prefix[prefix_len] = '\0';
      prefix = ssh_xstrdup(prefix);
    }
  else
    {
      prefix = NULL;
    }
  
  return prefix;
}

void
sftp_subst_filelistitems_to_tabcompl_results(Sftp2GData gdata,
                                             SshFileCopyFile file)
{
  SshBuffer tmp_buf;
  char *hlp, *hlp2;  
  SshADTContainer path_list;
  SshADTHandle h;
  
  path_list = ssh_sftp_get_path_list(gdata, file);
  h = ssh_adt_enumerate_start(path_list);
  SSH_ASSERT(h != NULL);
  
  tmp_buf = ssh_xbuffer_allocate();
  if (gdata->tabcompl_constant_part != NULL)
    {
      ssh_xbuffer_append(tmp_buf,
                         (unsigned char *)gdata->tabcompl_constant_part,
                         strlen(gdata->tabcompl_constant_part));
      ssh_xbuffer_append(tmp_buf, (unsigned char *)" ", 1);
    }
  
  while (h != SSH_ADT_INVALID)
    {
      hlp2 = ssh_adt_get(path_list, h);
      SSH_ASSERT(hlp2 != NULL);
      
      hlp = sftp_apply_quotation(hlp2);
      ssh_xfree(hlp2);
      ssh_xbuffer_append(tmp_buf, (unsigned char *)hlp, strlen(hlp));
      ssh_xfree(hlp);

      h = ssh_adt_enumerate_next(path_list, h);
      if (h != SSH_ADT_INVALID)
        ssh_xbuffer_append(tmp_buf, (unsigned char *)" ", 1);        
    }
  ssh_xbuffer_append(tmp_buf, (unsigned char *)"\0", 1);
  gdata->tabcompl_resultline = ssh_xstrdup(ssh_buffer_ptr(tmp_buf));
  ssh_buffer_free(tmp_buf);
}

typedef struct TabcomplCheckCtxRec
{
  Sftp2GData gdata;
  Boolean orig_pattern_only;
  int num;
} *TabcomplCheckCtx, TabcomplCheckCtxStruct;

void tabcompl_check_orig_cb(SshFileCopyFile file, void *context)
{
  TabcomplCheckCtx ctx = (TabcomplCheckCtx) context;
  char *full_path;

  if (ssh_file_copy_file_get_dir_entries(file) != NULL)
    return;

  if (ctx->num > 1)
    {
      if (ctx->orig_pattern_only)
        ctx->orig_pattern_only = FALSE;
      
      return;
    }
  
  ctx->num++;
  
  full_path = ssh_file_copy_file_generate_full_path(file);

  SSH_DEBUG(20, ("full_path: %s, word: %s, file: %s", full_path,
                 ctx->gdata->tabcompl_word_to_complete,
                 ssh_file_copy_file_get_name(file)));
  
  if (!strcmp(ctx->gdata->tabcompl_word_to_complete, full_path) ||
      /* The original path might have also been "full_path/", in which
         case, ssh_file_copy_glob would return only "full_path". */
      (strlen(full_path) ==
       strlen(ctx->gdata->tabcompl_word_to_complete) - 1 &&       
       /* Whether original ends in a slash. */
       ctx->gdata->tabcompl_word_to_complete
       [strlen(ctx->gdata->tabcompl_word_to_complete) - 1] == '/' &&
       /* .. and if so, does the new path match everything but the slash. */
       !strncmp(full_path, ctx->gdata->tabcompl_word_to_complete,
                strlen(full_path))))
    ctx->orig_pattern_only = TRUE;
  
  ssh_xfree(full_path);
}

SSH_FSM_STEP(sftp_tabcompl_results)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;
  char *tmp_str;
  SshFileCopyFile file = NULL;
  SshADTHandle h;
  TabcomplCheckCtxStruct check_ctx;
  
  /* Got results from the first round of globbing.
     1) NULL list - the string was a regexp and didn't match anything.
       - Repeat the globbing with an added asterisk
     2) List with the original string - string was not a regexp, whether the
        file exists is unknown.
       - Repeat the globbing with an added asterisk, but if multiple
         results are returned, do not substitute them on the command line,
         only show them.
     3) List with element(s) other than the original string - string was a
        regexp and matched to files.
       - Return the results.
  */

  /* Cases 1 and 2 */
  if (gdata->file_list != NULL)
    {
      /* Must contain only one object, as we gave only one pattern. */
      SSH_ASSERT(ssh_adt_num_objects(gdata->file_list) == 1);
      h = ssh_adt_get_handle_to_location(gdata->file_list,
                                         SSH_ADT_BEGINNING);
      SSH_ASSERT(h != SSH_ADT_INVALID);
      file = ssh_adt_get(gdata->file_list, h);
    }

  /* Check whether file has only one "real" filename, and whether it
     matches the original filename. */
  memset(&check_ctx, 0, sizeof(check_ctx));
  check_ctx.gdata = gdata;  
  ssh_file_copy_file_traverse(file, tabcompl_check_orig_cb, &check_ctx);
  
  if (gdata->file_list == NULL || check_ctx.orig_pattern_only)
    {
      SSH_FSM_SET_NEXT(sftp_tabcompl_results2);

      ssh_adt_destroy(gdata->file_list);
      gdata->file_list = NULL;

      if (gdata->tabcompl_word_to_complete
          [strlen(gdata->tabcompl_word_to_complete) - 1] == '*')
        {
          /* If last character is '*', don't append, as that would
             lead to recursing all subdirectories. Instead, this is
             treated as failure. */
          SSH_DEBUG(2, ("String already had a '*' at end. Going directly to "
                        "next phase."));
          return SSH_FSM_CONTINUE;
        }
      
      ssh_xdsprintf(&tmp_str, "%s*", gdata->tabcompl_word_to_complete);
      
      SSH_TRACE(5, ("Trying with %s\r\n", tmp_str));
      ssh_xfree(gdata->tabcompl_word_to_complete);
      gdata->tabcompl_word_to_complete = tmp_str;

      /* XXX Change to use ssh_file_copy_glob() as soon as it is done. */
      ssh_adt_clear(gdata->cl_file_list);
      h = ssh_adt_insert(gdata->cl_file_list,
                         ssh_xstrdup(gdata->tabcompl_word_to_complete));
      SSH_ASSERT(h != SSH_ADT_INVALID);
      SSH_FSM_ASYNC_CALL(ssh_file_copy_glob_multiple(gdata->conn,
                                                     gdata->cl_file_list,
                                                     gdata->glob_attrs,
                                                     tabcompl_ready_cb,
                                                     tabcompl_error_cb,
                                                     thread));
    }
  
  /* Case 3 - substitute the files on the command line. */
  sftp_subst_filelistitems_to_tabcompl_results(gdata, file);
  SSH_FSM_SET_NEXT(sftp_tabcompl_finish);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(sftp_tabcompl_results2)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;
  char *hlp = NULL, *hlp2, *tmp_str;

  tmp_str = gdata->tabcompl_original_word_to_complete;
  
  if (gdata->file_list != NULL)
    {
      SshADTContainer path_list;
      SshADTHandle h;
      
      SshFileCopyFile file =
        ssh_adt_get_object_from_location(gdata->file_list,
                                         SSH_ADT_BEGINNING);
      SSH_ASSERT(ssh_adt_num_objects(gdata->file_list) == 1);
      
      /* Show the results but do not substitute them on the
         command line. */      
      path_list = ssh_sftp_get_path_list(gdata, file);
      ssh_adt_list_sort(path_list);
      
      for (h = ssh_adt_enumerate_start(path_list);
           h != SSH_ADT_INVALID;
           h = ssh_adt_enumerate_next(path_list, h))
        {
          hlp2 = ssh_adt_get(path_list, h);
          SSH_ASSERT(hlp2);
          printf("%s\r\n", hlp2);
        }
      
      /* Replace the original string with the longest common
         prefix of the results printed above. */
      hlp = sftp_get_longest_common_prefix(gdata, path_list);

      if (hlp)
        {          
          SSH_DEBUG(2, ("prefix = %s", hlp));
      
          tmp_str = hlp;
        }
      ssh_adt_destroy(path_list);
    }

  hlp = sftp_apply_quotation(tmp_str);
  ssh_xfree(gdata->tabcompl_original_word_to_complete);
  gdata->tabcompl_original_word_to_complete = hlp;

  /* No results OR got multiple matches. Give back the original string. */
  if (gdata->tabcompl_constant_part)
    ssh_xdsprintf(&gdata->tabcompl_resultline, "%s %s",
                  gdata->tabcompl_constant_part,
                  gdata->tabcompl_original_word_to_complete);
  else
    gdata->tabcompl_resultline =
      ssh_xstrdup(gdata->tabcompl_original_word_to_complete);
  
  SSH_FSM_SET_NEXT(sftp_tabcompl_finish);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(sftp_tabcompl_finish)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;
  ssh_adt_destroy(gdata->file_list);
  gdata->file_list = NULL;
  ssh_xfree(gdata->tabcompl_word_to_complete);
  ssh_xfree(gdata->tabcompl_original_word_to_complete);
  ssh_xfree(gdata->tabcompl_constant_part);
  SSH_FSM_SET_NEXT(sftp_get_command);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(sftp_finish)
{
  Sftp2GData gdata = (Sftp2GData) fsm_context;

  /* Do sftp2-specific cleanup */
  ssh_xfree(gdata->ssh_path);
  ssh_xfree(gdata->recurse_attrs);
  if (gdata->rcwd_ctx)
    ssh_sftp_cwd_uninit(gdata->rcwd_ctx);
  if (gdata->lcwd_ctx)
    ssh_sftp_cwd_uninit(gdata->lcwd_ctx);
  if (gdata->rconn)
    ssh_file_copy_connection_destroy(gdata->rconn);
  if (gdata->lconn)
    ssh_file_copy_connection_destroy(gdata->lconn);
  ssh_xfree(gdata->batchfile_data);
  ssh_xfree(gdata->batchfile_name);
  ssh_xfree(gdata->global_tr_attrs->ascii_extensions);
  ssh_xfree(gdata->global_tr_attrs->source_newline);
  ssh_xfree(gdata->global_tr_attrs->dest_newline);
  ssh_xfree(gdata->global_tr_attrs);
  ssh_xfree(gdata->tr_attrs);
  ssh_xfree(gdata->remote_newline);
  ssh_xfree(gdata->client_newline);

  
  ssh_adt_destroy(gdata->file_list);
  ssh_adt_destroy(gdata->cl_file_list);
  
  /* delete any command line options passed to ssh2 */
  ssh_adt_destroy(gdata->command_line_options);
  

  if (gdata->rl)
    ssh_readline_eloop_uninitialize(gdata->rl);

  if (gdata->stdio_stream)
    ssh_stream_destroy(gdata->stdio_stream);

  ssh_leave_raw_mode(-1);


  ssh_fsm_destroy(gdata->fsm);

  ssh_xfree(gdata);

  return SSH_FSM_FINISH;
}

SSH_FSM_STEP(sftp_finish_thread)
{
  return SSH_FSM_FINISH;
}

SshFSMStateDebugStruct sftp_states [] =
{
#include "sftppager_states.h"
  { "connect_local", "Finalize initialization",
    sftp_connect_local },
  { "get_command", "Prepare to read a command from user", sftp_get_command },
  { "command_open", "Open a connection to destination host",
    sftp_cmd_open },
  { "command_finalize_open", "Finalize open", sftp_cmd_finalize_open },
  { "command_cd_start", "Cd: Start the operation",
    sftp_cmd_cd_start },
  { "command_pwd_start", "Pwd: Start the operation",
    sftp_cmd_pwd_start },
  { "command_rm_start", "Rm: Start the operation",
    sftp_cmd_rm_start },
  { "command_mkdir_start", "Mkdir: Start the operation",
    sftp_cmd_mkdir_start },
  { "command_ls_glob", "Ls: prepare for globbing",
    sftp_cmd_ls_glob },
  { "command_ls_recurse", "Ls: prepare for recurse_dirs",
    sftp_cmd_ls_recurse },
  { "command_ls_finish", "Ls: State after recurse_dirs",
    sftp_cmd_ls_finish },
  { "command_getput_glob", "Getput: prepare for globbing",
    sftp_cmd_getput_glob },
  { "command_getput_recurse", "Getput: prepare for recurse_dirs",
    sftp_cmd_getput_recurse },
  { "command_getput_transfer", "Getput: prepare for transfer of files",
    sftp_cmd_getput_transfer },
  { "command_rename_check", "Rename: prepare for statting the destination",
    sftp_cmd_rename_check },
  { "command_rename", "Rename: prepare for the actual operation",
    sftp_cmd_rename },
  { "command_readlink", "Readlink", sftp_cmd_readlink },
  { "command_symlink", "Symlink", sftp_cmd_symlink },
  { "tabcompl_start", "Tab Completion: start",
    sftp_tabcompl_start },
  { "tabcompl_results", "Tab Completion: get the first results",
    sftp_tabcompl_results },
  { "tabcompl_results2", "Tab Completion: get the second results",
    sftp_tabcompl_results2 },
  { "tabcompl_finish", "Tab Completion: finish and cleanup",
    sftp_tabcompl_finish },
  { "finish", "Finish", sftp_finish }
};

void
sftp_usage(Sftp2GData sftp_gdata)
{
  SSH_PRECOND(sftp_gdata != NULL);

  fprintf(stderr,
          "Usage: sftp2 [-D debug_level_spec] [-B batchfile] [-S path] "
          "[-h]\r\n"
          "             [-V] [-P port] [-b buffer_size] [-N max_requests]\r\n"

          "             [-c cipher] [-m mac] [-4] [-6] [-o option_to_ssh2]\r\n"





          "             [user@]host[#port]\r\n");

  exit(1);
}

typedef struct SftpConnectionCtxRec
{
  SshStream stream;
  Sftp2GData gdata;
  pid_t child_pid;
  char buffer[200];
  int read_bytes;
  SshFileCopyStreamReturnCB completion_cb;
  void *completion_context;
} *SftpConnectionCtx;

void
sftp_sigchld_handler(pid_t pid, int status, void *context)
{
  SftpConnectionCtx ctx = (SftpConnectionCtx) context;

  SSH_PRECOND(ctx != NULL);

  ssh_warning("child process (%s) exited with code %d.",
              ctx->gdata->ssh_path, status);

  if (ctx->gdata->rl)
    {
      ssh_readline_eloop_uninitialize(ctx->gdata->rl);
      ctx->gdata->rl = NULL;
    }
  ssh_leave_raw_mode(-1);

  exit(SSH_FC_ERROR);
}

void new_stream_callback(SshStreamNotification notification, void *context)
{
  SftpConnectionCtx ctx = (SftpConnectionCtx) context;
  int ret = 0;

  SSH_TRACE(4, ("stream notification: %d", notification));

  switch (notification)
    {
    case SSH_STREAM_INPUT_AVAILABLE:

      while ((ret = ssh_stream_read
              (ctx->stream, (unsigned char *)&(ctx->buffer[ctx->read_bytes]),
               1)) > 0 &&
             ctx->read_bytes < sizeof(ctx->buffer) - 1)
        {
          ctx->read_bytes += ret;

          SSH_TRACE(5, ("read char: %c, read_bytes: %d, buf len: %d",
                        ctx->buffer[strlen(ctx->buffer) - 1],
                        ctx->read_bytes, strlen(ctx->buffer)));
          SSH_DEBUG_HEXDUMP(5, ("received message:"),
                            (unsigned char *)ctx->buffer, ctx->read_bytes);
          fflush(stderr);
          if (ctx->buffer[strlen(ctx->buffer) - 1] == '\n')
            {
              SSH_TRACE(2, ("buffer: '%s'", ctx->buffer));
              if (strcmp(ctx->buffer, "AUTHENTICATED YES\n") == 0)
                {

                  ssh_enter_non_blocking(-1);
                  ssh_enter_raw_mode(-1, TRUE);

                  (*ctx->completion_cb)(ctx->stream, ctx->completion_context);
                  return;
                }
              else
                {
                  SSH_DEBUG_HEXDUMP(3, ("received message:"),
                                    (unsigned char *)ctx->buffer,
                                    ctx->read_bytes);
                  ssh_fatal("ssh2 client failed to authenticate. (or you "
                            "have too old ssh2 installed, check with "
                            "ssh2 -V)");
                }
            }

        }
      if (ret == 0)
        {
          ssh_fatal("EOF received from ssh2. ");
        }
      if (ctx->read_bytes >= sizeof(ctx->buffer) - 1)
        {
          ssh_fatal("Received corrupted (or wrong type of) data from "
                    "ssh2-client.");
        }
      break;
    case SSH_STREAM_CAN_OUTPUT:
      break;
    case SSH_STREAM_DISCONNECTED:
      SSH_TRACE(2, ("Received SSH_STREAM_DISCONNECTED from wrapped stream."));
      ssh_fatal("ssh2 client failed to authenticate. (or you "
                "have too old ssh2 installed, check with ssh2 -V)");
      /* XXX */
      break;
    }
}

void connect_callback(SshFileCopyConnection connection, void *context,
                      SshFileCopyStreamReturnCB completion_cb,
                      void *completion_context)
{
#define SSH_ARGV_SIZE   64
  Sftp2GData gdata = (Sftp2GData) context;
  SftpConnectionCtx new_ctx;
  struct stat st;
  SshStream client_stream;
  char *ssh_argv[SSH_ARGV_SIZE];
  int i = 0;
  SshADTHandle h;




  if (stat(gdata->ssh_path, &st) < 0)
    {
      SSH_TRACE(0, ("Couldn't find ssh2 on path specified (%s). "
                    "Trying default PATH...", gdata->ssh_path));
      ssh_xfree(gdata->ssh_path);
      gdata->ssh_path = ssh_xstrdup("ssh2");
    }

  ssh_argv[i++] = gdata->ssh_path;

  if (connection->user != NULL)
    {
      ssh_argv[i++] = "-l";
      ssh_argv[i++] = connection->user;
    }
  if (connection->port != NULL)
    {
      ssh_argv[i++] = "-p";
      ssh_argv[i++] = connection->port;
    }

  if (gdata->verbose_mode)
    ssh_argv[i++] = "-v";

  ssh_argv[i++] = "-x";
  ssh_argv[i++] = "-a";


  ssh_argv[i++] = "-opasswordprompt=%U@%H's password:";



  ssh_argv[i++] = "-oauthenticationnotify=yes";
  if (gdata->batchmode)
    {
      ssh_argv[i++] = "-oBatchMode=yes";
    }

  /* put any command line options passed to the sftp command line to
     the argument list */
  for (h = ssh_adt_enumerate_start(gdata->command_line_options);
       h != SSH_ADT_INVALID;
       h = ssh_adt_enumerate_next(gdata->command_line_options, h))
    {
      char *str = ssh_adt_get(gdata->command_line_options, h);
      SSH_ASSERT(str != NULL);
      
      if (i >= SSH_ARGV_SIZE - 4)
        ssh_fatal("Too many arguments for %s (at least %d given).",
                  gdata->ssh_path, i + 3);
      ssh_argv[i++] = str;
    }

  ssh_argv[i++] = connection->host;

  ssh_argv[i++] = "-s";
  ssh_argv[i++] = "sftp";

  ssh_argv[i] = NULL;


  ssh_leave_non_blocking(-1);
  ssh_leave_raw_mode(-1);


  SSH_ASSERT(i < SSH_ARGV_SIZE);
  if (gdata->verbose_mode)
    {
      SSH_DEBUG_INDENT;
      for (i = 0; ssh_argv[i]; i++)
        ssh_debug("argv[%d] = %s", i, ssh_argv[i]);
      SSH_DEBUG_UNINDENT;
    }


  switch (ssh_pipe_create_and_fork(&client_stream, NULL))











    {
    case SSH_PIPE_ERROR:
      ssh_fatal("ssh_pipe_create_and_fork() failed");

    case SSH_PIPE_PARENT_OK:
      new_ctx = ssh_xcalloc(1, sizeof(*new_ctx));
      new_ctx->stream = client_stream;
      new_ctx->gdata = gdata;
      new_ctx->completion_cb = completion_cb;
      new_ctx->completion_context = completion_context;

      new_ctx->child_pid = ssh_pipe_get_pid(client_stream);
      ssh_sigchld_register(new_ctx->child_pid, sftp_sigchld_handler,
                           new_ctx);


      ssh_stream_set_callback(client_stream, new_stream_callback,
                              new_ctx);
      new_stream_callback(SSH_STREAM_INPUT_AVAILABLE,
                          new_ctx);
#ifdef SUSPEND_AFTER_FORK
      kill(getpid(), SIGSTOP);
#endif /* SUSPEND_AFTER_FORK */



      return;


    case SSH_PIPE_CHILD_OK:
      execvp(ssh_argv[0], ssh_argv);
      fprintf(stderr, "Executing ssh2 failed. Command:'");
      for (i = 0;ssh_argv[i] != NULL; i++)
        fprintf(stderr," %s", ssh_argv[i]);
      fprintf(stderr,"' System error message: '%s'\r\n", strerror(errno));
      exit(-1);

    }
  SSH_NOTREACHED;
}

void newline_query_rl_cb(const char *line, void *context)
{
  Sftp2GData gdata = (Sftp2GData) context;
  char *newline;

  printf("\r\n");
  newline = ssh_file_copy_parse_newline_convention(line);
  if (newline == NULL)
    {
      ssh_readline_eloop("Please answer \"unix\", \"dos\", or \"mac\" "
                         "(without the quotes) :", NULL, gdata->rl,
                         newline_query_rl_cb,
                         gdata);
      return;
    }
  else
    {
      if (gdata->cmd_doing_remote)
        {
          ssh_xfree(gdata->remote_newline);
          gdata->remote_newline = newline;
        }
      else
        {
          ssh_xfree(gdata->client_newline);
          gdata->client_newline = newline;
        }
      (*gdata->newline_query_completion_cb)
        (newline, gdata->newline_query_completion_context);
    }
}

void sftp_newline_query_cb(Boolean destination,
                            SshFileCopyNLQueryCompletionCB completion_cb,
                            void *completion_context,
                            void *context)
{
  Sftp2GData gdata = (Sftp2GData) context;
  char str[256];
  gdata->newline_query_completion_cb = completion_cb;
  gdata->newline_query_completion_context = completion_context;

  if (destination)
    gdata->cmd_doing_remote = TRUE;
  else
    gdata->cmd_doing_remote = FALSE;

  ssh_snprintf(str, sizeof(str), "Specify %s newline type (\"unix\", "
               "\"dos\", or \"mac\"): ", destination ? "remote" : "local");
  ssh_readline_eloop(str, NULL, gdata->rl, newline_query_rl_cb,
                     gdata);
}

int
main(int argc, char **argv)
{
  Sftp2GData gdata = ssh_xcalloc(1, sizeof(*gdata));
  SshFSM fsm;
  int ch;
  char opt_buf[10];

  ssh_event_loop_initialize();






  ssh_app_set_debug_format(NULL);

  /* Initialize the FSM */
  fsm = ssh_fsm_create(gdata);

  ssh_fsm_register_debug_names(fsm, sftp_states,
                               SSH_FSM_NUM_STATES(sftp_states));
  SSH_ASSERT(fsm != NULL);

  gdata->fsm = fsm;

  /* Initialize gdata */
  gdata->ssh_path = ssh_xstrdup(SSH_SSH2_PATH);
  gdata->rconn = ssh_file_copy_connection_allocate();
  gdata->lconn = ssh_file_copy_connection_allocate();
  gdata->glob_attrs =
    ssh_xcalloc(1, sizeof(*gdata->glob_attrs));
  gdata->recurse_attrs =
    ssh_xcalloc(1, sizeof(*gdata->recurse_attrs));
  gdata->global_tr_attrs =
    ssh_xcalloc(1, sizeof(*gdata->global_tr_attrs));
  
  gdata->batchmode = FALSE;
  gdata->tabcompl_resultline = NULL;

  gdata->command_line_options =
    ssh_adt_create_generic(SSH_ADT_LIST,
                           SSH_ADT_DESTROY,
                           ssh_adt_callback_destroy_free,
                           SSH_ADT_ARGS_END);
  SSH_ASSERT(gdata->command_line_options != NULL);

  gdata->cl_file_list =
    ssh_adt_create_generic(SSH_ADT_LIST,
                           SSH_ADT_DESTROY,
                           ssh_adt_callback_destroy_free,
                           SSH_ADT_ARGS_END);
  SSH_ASSERT(gdata->cl_file_list != NULL);

  gdata->glob_attrs->compare_func = gdata->recurse_attrs->compare_func =
    sftp_file_compare;
  gdata->glob_attrs->compare_ctx = gdata->recurse_attrs->compare_ctx =
    gdata;
  gdata->global_tr_attrs->destination_must_be_dir = TRUE;
  gdata->global_tr_attrs->preserve_attributes = FALSE;
  gdata->global_tr_attrs->unlink_source = FALSE;
  gdata->global_tr_attrs->create_dirs = TRUE;
  gdata->global_tr_attrs->max_buf_size = SSH_FC_DEFAULT_BUF_SIZE;
  gdata->global_tr_attrs->max_buffers = SSH_FC_DEFAULT_BUF_NUM;
  gdata->global_tr_attrs->progress_cb =
    ssh_file_copy_transfer_default_progress_cb;

  /* Initializes global variables used in the default progress cb. */
  ssh_file_copy_transfer_default_progress_cb_init();
  






  /* Check for --help in the command line */
  if (argc > 1)
    {
      if (!strcmp(argv[1], "--help"))
        {
          sftp_usage(gdata);
          exit(0);
        }
    }

#define ADD_OPT(opt)                                                    \
do {                                                                    \
  SshADTHandle h;                                                       \
  h = ssh_adt_insert_to(gdata->command_line_options, SSH_ADT_END,       \
                        ssh_xstrdup(opt));                              \
  SSH_ASSERT(h != SSH_ADT_INVALID);                                     \
} while (0)

  /* Handle the command line options. */

  while ((ch = ssh_getopt(argc, argv, "D:vS:Vh?B:b:N:P:c:m:o:64", NULL)) != -1)



    {
      if (!ssh_optval)
        {
          sftp_usage(gdata);
        }
      switch(ch)
        {
        case 'D':
          ssh_xfree(gdata->debug_level);
          gdata->debug_level = ssh_xstrdup(ssh_optarg);
          ssh_debug_set_level_string(gdata->debug_level);
          gdata->verbose_mode = TRUE;
          break;
        case 'v':
          ssh_xfree(gdata->debug_level);
          gdata->debug_level = ssh_xstrdup("2");
          ssh_debug_set_level_string(gdata->debug_level);
          gdata->verbose_mode = TRUE;
          break;
        case 'S':
          ssh_xfree(gdata->ssh_path);
          gdata->ssh_path = ssh_xstrdup(ssh_optarg);
          break;
        case 'V':
          ssh2_version(argv[0]);
          exit(0);
          break;
        case 'h':
        case '?':
          sftp_usage(gdata);
          exit(0);
          break;
        case 'B':
          gdata->batchmode = TRUE;
          gdata->batchfile_name = ssh_xstrdup(ssh_optarg);
          break;
        case 'b':
          gdata->global_tr_attrs->max_buf_size = atoi(ssh_optarg);
          if (gdata->global_tr_attrs->max_buf_size <= 0)
            {
              printf("-b requires an argument greater than zero.\r\n");
              exit(0);
            }
          break;
        case 'N':
          gdata->global_tr_attrs->max_buffers = atoi(ssh_optarg);
          if (gdata->global_tr_attrs->max_buffers <= 0)
            {
              printf("-N requires an argument greater than zero.\r\n");
              exit(0);
            }
          break;
        case 'P':
          ssh_xfree(gdata->rconn->port);
          gdata->rconn->port = ssh_xstrdup(ssh_optarg);
          break;
        case 'c':
        case 'm':
        case '4':
        case '6':
        case 'o':
          ssh_snprintf(opt_buf, sizeof(opt_buf), "-%c", ch);
          ADD_OPT(opt_buf);
          if (ssh_optarg)
            ADD_OPT(ssh_optarg);
          break;







        default:
          sftp_usage(gdata);
          break;
        }
    }

  gdata->initial_connect_target = NULL;
  if (ssh_optind < argc - 1)
    {
      printf("Too many arguments.\n");
      sftp_usage(gdata);
    }
  if (ssh_optind == argc - 1)
    gdata->initial_connect_target = argv[ssh_optind];

  ssh_debug_register_callbacks(sftp_fatal, sftp_warning, sftp_debug,
                               (void *)(gdata));

  ssh_file_copy_register_connect_callback(connect_callback,
                                          gdata);





  
  /* Default file transfer mode is binary */
  gdata->global_tr_attrs->transfer_mode = SSH_FC_TRANSFER_MODE_BINARY;
  gdata->global_tr_attrs->ascii_extensions = ssh_xstrdup("txt,htm*,pl,php*");

  gdata->global_tr_attrs->newline_query_cb = sftp_newline_query_cb;
  gdata->global_tr_attrs->context = gdata;

  /* Per-transfer attrs. */
  gdata->tr_attrs = ssh_xcalloc(1, sizeof(*gdata->tr_attrs));
  
  gdata->getopt_data = ssh_xcalloc(1, sizeof(*gdata->getopt_data));
  
  gdata->remote_newline = ssh_xstrdup("\n");



  gdata->client_newline = ssh_xstrdup("\n");



  ssh_register_signal(SIGHUP, sftp_fatal_signal_handler,
                      (void *)gdata);
  /*   Capture Ctrl-Cs, so that we can abort transfers. */
  ssh_register_signal(SIGINT, sftp_sigint_handler,
                      (void *)gdata);

  ssh_register_signal(SIGTERM, sftp_fatal_signal_handler,
                      (void *)gdata);
  ssh_register_signal(SIGABRT, sftp_fatal_signal_handler,
                      (void *)gdata);

  
  gdata->exit_attempted = FALSE;

  gdata->main_thread = ssh_fsm_thread_create(fsm,
                                             sftp_connect_local,
                                             NULL_FNPTR, NULL_FNPTR, NULL);

  ssh_event_loop_run();

  ssh_global_uninit();

  ssh_event_loop_uninitialize();
  
  return exit_value;
}
