/*

  scp2.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1999-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Secure Copy. Designed to transfer files between hosts, using ssh2.

*/









#include "sshincludes.h"
#include "ssh2includes.h"
#include "ssheloop.h"
#include "sshfsm.h"
#include "sshmiscstring.h"
#include "sshgetopt.h"
#include "sshappcommon.h"
#include "sshfilecopy.h"
#include "sshglob.h"
#include "sshdsprintf.h"
#include "sshgetcwd.h"
#include "sshadt_conv.h"
#include "sshfdstream.h"
#include "sshreadline.h"

#undef interface
#include "sshtty.h"
#include "sshunixpipestream.h"
#include "sigchld.h"






#define SSH_DEBUG_MODULE "Scp2"

/* #define SUSPEND_AFTER_FORK */

/* #define TEST_TRANSFER_ABORT */


/* Forward declaration. */
typedef struct ScpFileListItemRec *ScpFileListItem;

/* What kind of statistics to output during transfer, if any. */
typedef enum
  {
    SCP_STATISTICS_YES = 0,
    SCP_STATISTICS_NO,
    SCP_STATISTICS_SIMPLE
  } ScpStatistics;

/* Scp2 session record. Holds all relevant information which must be
   "globally" known. (This is passed to callbacks etc.) */
typedef struct ScpSessionRec
{
  /* program name (by which we were called), not argv[0]. */
  char *scp2_name;

  /* Set, if user wants debug output. */
  Boolean debug_mode;

  /* Set, if user wants scp to remain quiet (fatal errors are still
     displayed). */
  Boolean quiet_mode;

  /* Width of terminal screen. */
  int screen_width;

  /* Remote port (if given on the command line with '-P'). */
  char *remote_port;

  /* Optional path to ssh. */
  char *ssh_path;

  /* Additional options for ssh. */
  SshADTContainer ssh_options;

  /* If set, indicates that scp is operating in a tty, and therefore
     can print things. */
  Boolean have_tty;

  /* If set, scp doesn't show progress indicator. */
  ScpStatistics statistics;

  /* Whether directories should be recursed (and created to the other end). */
  Boolean recurse_dirs;
  
  /* If set, scp only shows what would've happened if copying had
     taken place. */
  Boolean do_not_copy;

  /* Are we in batch mode? */
  Boolean batch_mode;

  /* Whether to prompt the user before overwriting a file. */
  Boolean interactive_mode;
  
  /* Exit value of scp2. */
  SshFileCopyError exit_value;

} *ScpSession;

typedef struct ScpGDataRec
{
  /* Attributes for ssh_file_copy_transfer_multiple(). */
  SshFileCopyTransferAttrs attrs;

  /* pid of the ssh child scp executes. */
  pid_t ssh_child_pid;

  /* Contains the original (given by the user, to the command line)
     file list. These are ScpFileListItems. */
  SshADTContainer file_list;
  SshADTHandle h;
  
  /* This is what will be given to the
     ssh_file_copy_{glob,recurse,transfer}* functions, and what was
     returned by them. */
  SshADTContainer cur_list;

  ScpFileListItem cur_item;
  
  /* The destination file. */
  SshFileCopyConnection dest_conn;
  char *dest_path;
  
  SshFSM fsm;
  SshFSMThread main_thread;

  /* Operation handle for ssh_file_copy_transfer_multiple, which can be
     used to abort the operation. */
  SshOperationHandle transfer_op_handle;

  /* Pointer to the Session object. */
  ScpSession session;

  /* For interactive mode. */
  SshStream stdio_stream;
  SshReadLineCtx rl;
  SshFileCopyTransferOverwriteCompletionCB overwrite_completion_cb;
  void *overwrite_completion_ctx;
  /* This is also used by '--overwrite'. */
  enum {SCP_OVERWRITE_UNDEFINED = 0,
        SCP_OVERWRITE_YES_TO_ALL,
        SCP_OVERWRITE_NO_TO_ALL} overwrite_persistent_answer;
} *ScpGData, ScpGDataStruct;

/* Structure, which holds a connection and the files to be transferred
   to/from that location. */
struct ScpFileListItemRec
{
  SshFileCopyConnection connection;
  SshADTContainer file_list;
};

typedef struct ScpConnectionCtxRec
{
  SshStream stream;
  ScpSession session;
  pid_t child_pid;
  char buffer[200];
  int read_bytes;
  SshFileCopyStreamReturnCB completion_cb;
  void *completion_context;
} *ScpConnectionCtx;

/* Callback used by sshfilecopy to get a stream (by connecting with
   ssh and wrapping that to stream). */
void connect_callback(SshFileCopyConnection connection, void *context,
                      SshFileCopyStreamReturnCB completion_cb,
                      void *completion_context);

/* Parse a string to a SshFileCopyConnection structure. String is of
   form [user@]host[#port] . In case of an error or if the given
   string is malformed, return TRUE. Otherwise, return FALSE. filename
   is returned in return_filename, and it is malloced inside
   scp_location_parse(). */
Boolean scp_location_parse(SshFileCopyConnection connection,
                           char **return_filename,
                           const char *orig_string);

/* Find a matching SshFileCopyConnection from file_list. file_list
   MUST contain ScpFileListItems. Return matching
   SshFileCopyConnection if found, or NULL if not. */
ScpFileListItem
scp_file_list_find_matching_connection(SshADTContainer file_list,
                                       SshFileCopyConnection connection);

/* Destroy a ScpFileListItem structure. */
void scp_file_list_item_destroy(void *item);

/* Destroy a ScpFinalFileListItem structure. */
void scp_final_file_list_item_destroy(void *item);

/* Adds the given file name to the list of transferred files.
   On Windows replaces backslashes with slashes before addition. */
void scp_add_raw_file(SshADTContainer file_list, char *filename);

#ifdef TEST_TRANSFER_ABORT
void scp_interrupt_signal_handler(int sig, void *context)
{
  ScpGData gdata = (ScpGData) context;

  SSH_PRECOND(sig == SIGINT);
  SSH_PRECOND(gdata != NULL);

  ssh_debug("SIGINT received.");

  if (gdata->transfer_op_handle)
    {
      ssh_operation_abort(gdata->transfer_op_handle);
      gdata->transfer_op_handle = NULL;
    }

  ssh_unregister_signal(SIGINT);
}
#endif /* TEST_TRANSFER_ABORT */

/* Initializes an ScpSession context. Does not allocate the main
   struct. */
void scp_session_initialize(ScpSession session)
{
  session->ssh_path = ssh_xstrdup(SSH_SSH2_PATH);
  session->ssh_options = ssh_adt_create_generic(SSH_ADT_LIST,
                                                SSH_ADT_DESTROY,
                                                ssh_adt_callback_destroy_free,
                                                SSH_ADT_ARGS_END);
}

void scp_sigchld_handler(pid_t pid, int status, void *context)
{
  ScpConnectionCtx new_ctx = (ScpConnectionCtx) context;

  SSH_PRECOND(new_ctx != NULL);

  ssh_warning("child process (%s) exited with code %d.",
              new_ctx->session->ssh_path, status);

  exit(SSH_FC_ERROR);
}

/* Frees ScpSession contents. Does not free the main struct. */
void scp_session_destroy(ScpSession session)
{
  ssh_xfree(session->ssh_path);
  ssh_xfree(session->remote_port);

  ssh_adt_destroy(session->ssh_options);
}

/* Allocate a file list item.*/
ScpFileListItem scp_file_list_item_allocate(void)
{
  ScpFileListItem item;

  item = ssh_xcalloc(1, sizeof(*item));
  item->file_list = ssh_adt_create_generic(SSH_ADT_LIST,
                                           SSH_ADT_DESTROY,
                                           ssh_adt_callback_destroy_free,
                                           SSH_ADT_ARGS_END);
  SSH_ASSERT(item->file_list != NULL);

  return item;
}

void scp_file_list_item_destroy(void *item)
{
  ScpFileListItem list_item = (ScpFileListItem) item;

  SSH_DEBUG(4, ("Destroying ScpFileListItem..."));
  if (list_item->connection)
    {
      ssh_file_copy_connection_destroy(list_item->connection);
    }
  ssh_adt_destroy(list_item->file_list);

  memset(list_item, 'F', sizeof(*list_item));

  ssh_xfree(list_item);
}

void scp_debug(const char *msg, void *context)
{
  ScpSession session = (ScpSession) context;

  SSH_PRECOND(session != NULL);

  if (!session->debug_mode || session->quiet_mode)
    return;

  fprintf(stderr, "%s:%s\r\n", session->scp2_name, msg);
}

void scp_warning(const char *msg, void *context)
{
  ScpSession session = (ScpSession) context;

  SSH_PRECOND(session != NULL);

  if (!session->quiet_mode)
    fprintf(stderr, "%s: warning: %s\r\n", session->scp2_name, msg);
}

void scp_fatal(const char *msg, void *context)
{
  ScpSession session = (ScpSession) context;
  /* If we would call SSH_ASSERT here for the session struct, we could
     go to a infinite loop. The reason is left as an exercise for the
     reader. */
  fprintf(stderr, "%s: FATAL: %s\r\n", session->scp2_name, msg);

  exit(-1);
}

#define INFO ssh_informational
void scp_usage(ScpSession session)
{
  SSH_PRECOND(session != NULL);




  ssh_informational("usage: %s [options]\r\n",
                    session->scp2_name);
  INFO("            [[user@]host[#port]:]file ...\r\n");
  INFO("            [[user@]host[#port]:]file_or_dir\r\n");
  INFO("\r\n");
  INFO("Options:\r\n");
  INFO("  -D debug_level_spec  Set debug level. (Syntax is `module=level')\r\n");
  INFO("  -d                   Force target to be a directory.\r\n");
  INFO("  -q                   Make scp quiet (only fatal errors "
       "are displayed).\r\n");
  INFO("  -Q                   Don't show progress indicator.\r\n");

  INFO("  -p                   Preserve file attributes and "
       "timestamps.\r\n");




  /*  INFO("  -n                   Show what would've been done "
      "without actually copying\r\n");
      INFO("                       any files.\r\n");*/
  INFO("  -u                   Remove source files after "
       "copying.\r\n");
  INFO("  -B                   Sets batch-mode on.\r\n");
  INFO("  --interactive\r\n");
  INFO("  -I                   Prompt whether to overwrite an existing\r\n"
       "                       destination file. (doesn't work with "
       "`-B')\r\n");
  INFO("  --overwrite[=no]     Whether to overwrite existing destination "
       "file(s)\r\n"
       "                       (default: yes).\r\n");
  INFO("  -r                   Recurse subdirectories.\r\n");








  INFO("  -a[arg]              Transfer files in ascii mode. See manual page\r\n"
       "                       for description of optional argument.\r\n");
  INFO("  --verbose\r\n");
  INFO("  -v                   Verbose mode; equal to `-D 2'.\r\n");

  INFO("  -1                   Engage scp1 compatibility.\r\n");
  INFO("  -4                   Use IPv4 to connect.\r\n");
  INFO("  -6                   Use IPv6 to connect.\r\n");

  INFO("  -c cipher            Select encryption algorithm. "
       "Multiple -c options are \r\n"
       "                       allowed and a single -c flag "
       "can have only one cipher.\r\n");
  INFO("  -S ssh2-path         Tell scp2 where to find ssh2.\r\n");
  INFO("  -P ssh2-port         Tell scp2 which port sshd2 "
       "listens on the remote machine.\r\n");
  INFO("  -b buffer-size       Define maximum buffer size for "
       "one request\r\n"
       "                       (default 32768 bytes).\r\n");
  INFO("  -N max_requests      Define maximum number of "
       "concurrent requests\r\n"
       "                       (default 10).\r\n");




  INFO("  -o ssh2-opt          Specify additional options for ssh2.\r\n");





  INFO("  --version\r\n");
  INFO("  -V                   Display version.\r\n");
  INFO("  --help\r\n");
  INFO("  -h                   Display this help.\r\n");
  INFO("\r\n");
}
#undef INFO

SSH_FSM_STEP(scp_glob);
SSH_FSM_STEP(scp_recurse);
SSH_FSM_STEP(scp_transfer);
SSH_FSM_STEP(scp_finish);

void op_ready_cb(SshFileCopyError error,
                   const char *error_message,
                   SshADTContainer file_list,
                   void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  ScpGData gdata = (ScpGData) ssh_fsm_get_gdata(thread);

  SSH_DEBUG(2, ("Received error \"%s\"., msg: %s",
                ssh_file_copy_errors[error].error_name, error_message));

  if (gdata->session->exit_value < error)
    gdata->session->exit_value = error;

  ssh_adt_destroy(gdata->cur_item->file_list);
  gdata->cur_item->file_list = file_list;
  
  switch (error)
    {
    case SSH_FC_OK:
      /* Everything went just fine. */
      break;      
    default:
      ssh_fsm_set_next(thread, scp_finish);
      break;
    }

  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

void glob_error_cb(SshFileCopyError error,
                   const char *error_message,
                   void *context)
{
  SSH_TRACE(2, ("Received error %s", ssh_file_copy_errors[error].error_name));
  ssh_informational("%s\r\r\n", error_message);
}

SSH_FSM_STEP(scp_glob)
{
  ScpGData gdata = (ScpGData) fsm_context;
  
  if (gdata->h == SSH_ADT_INVALID)
    {
      SSH_FSM_SET_NEXT(scp_finish);
      return SSH_FSM_CONTINUE;
    }

  SSH_FSM_SET_NEXT(scp_recurse);
  
  gdata->cur_item = (ScpFileListItem) ssh_adt_get(gdata->file_list, gdata->h);
  SSH_ASSERT(gdata->cur_item);
  gdata->h = ssh_adt_enumerate_next(gdata->file_list, gdata->h);

  /* Local files shouldn't be globbed, as they are globbed by
     the shell. If wildcards survive to here, then the user has
     deliberately escaped them, and we should use them
     literalily. */

  /* Here we make a list of the file names given on the command line,
     wrapped in SshFileCopyFiles. */
  SSH_DEBUG(2, ("Current source host: %s", gdata->cur_item->connection->host ?
                gdata->cur_item->connection->host : "(local)"));
  
  if (gdata->cur_item->connection->host == NULL)
    {
      SshADTHandle h, h2;
      SshADTContainer new_list;
      
      new_list = ssh_file_copy_file_list_create(NULL_FNPTR, NULL);
      
      SSH_DEBUG(4, ("Adding local file(s) without globbing."));

      SSH_DEBUG(2, ("file list has %d files",
                    ssh_adt_num_objects(gdata->cur_item->file_list)));
      /* File is local. */
      for (h = ssh_adt_enumerate_start(gdata->cur_item->file_list);
           h != SSH_ADT_INVALID;
           h = ssh_adt_enumerate_next(gdata->cur_item->file_list, h))
        {
          SshFileCopyFile new_file = ssh_file_copy_file_allocate();
          SshFileCopyFile sub_file;
          char *basedir, *file_name, *p;

          p = (char *) ssh_adt_get(gdata->cur_item->file_list, h);
          SSH_ASSERT(p != NULL);

          ssh_file_copy_tokenize_path(p, &basedir, &file_name);
          new_file = ssh_file_copy_file_allocate();

          if (basedir)
            {
              sub_file = ssh_file_copy_file_allocate();
              p = ssh_glob_strip_escapes(basedir);
              ssh_xfree(basedir);
              
              SSH_DEBUG(3, ("Adding local file ``%s'', basedir ``%s''",
                            file_name, p));
              ssh_file_copy_file_register_filename (new_file, p);
              ssh_file_copy_file_register_filename(sub_file, file_name);    
              ssh_file_copy_file_register_dir_entry(new_file, sub_file);
            }
          else
            {
              SSH_DEBUG(3, ("Adding local file ``%s''", file_name));
              ssh_file_copy_file_register_filename (new_file, file_name);
            }
          
          h2 = ssh_adt_insert_to(new_list, SSH_ADT_END, new_file);
          SSH_ASSERT(h2 != SSH_ADT_INVALID);          
        }
      ssh_adt_destroy(gdata->cur_item->file_list);
      gdata->cur_item->file_list = new_list;
      return SSH_FSM_CONTINUE;
    }




  SSH_FSM_ASYNC_CALL(ssh_file_copy_glob_multiple(gdata->cur_item->connection,
                                                 gdata->cur_item->file_list,
                                                 NULL,
                                                 op_ready_cb,
                                                 glob_error_cb,
                                                 thread));
}

void recurse_nonfatal_error_cb(SshFileCopyError error,
                               const char *error_message,
                               void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  ScpGData gdata = (ScpGData) ssh_fsm_get_gdata(thread);

  SSH_TRACE(2, ("Received error %s.", ssh_file_copy_errors[error].error_name));
  if (!gdata->session->quiet_mode)
    ssh_warning("%s", error_message);
  /* XXX scp2 shouldn't return 0 after these. */
}

#ifdef DEBUG_LIGHT
void scp_file_trav_cb(SshFileCopyFile file, void *context)
{
  char *full_path = ssh_file_copy_file_generate_full_path(file);
  
  ssh_debug("%s", full_path);
  ssh_xfree(full_path);
}

void scp_dump_file_list(SshADTContainer file_list)
{
  SshADTHandle h;

  ssh_debug("file_list:");
  
  for (h = ssh_adt_enumerate_start(file_list);
       h != SSH_ADT_INVALID;
       h = ssh_adt_enumerate_next(file_list, h))
    {
      SshFileCopyFile file = (SshFileCopyFile)ssh_adt_get(file_list, h);

      ssh_file_copy_file_traverse(file, scp_file_trav_cb, NULL);
    }
}
#endif /* DEBUG_LIGHT */

SSH_FSM_STEP(scp_recurse)
{
  ScpGData gdata = (ScpGData) fsm_context;

  SSH_FSM_SET_NEXT(scp_transfer);

  if (!gdata->session->recurse_dirs)
    return SSH_FSM_CONTINUE;

#ifdef DEBUG_LIGHT
  if (SSH_DEBUG_ENABLED(6))
    scp_dump_file_list(gdata->cur_item->file_list);
#endif /* DEBUG_LIGHT */
  SSH_FSM_ASYNC_CALL(ssh_file_copy_recurse_multiple
                     (gdata->cur_item->connection,
                      gdata->cur_item->file_list,
                      NULL, NULL_FNPTR,
                      op_ready_cb,
                      recurse_nonfatal_error_cb,
                      thread));
}

void transfer_ready_cb(SshFileCopyError error,
                       const char *error_message,
                       void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  ScpGData gdata = (ScpGData) ssh_fsm_get_gdata(thread);

  SSH_TRACE(0, ("Received error %s, error message %s.",
                ssh_file_copy_errors[error].error_name,
                error_message));

  if (gdata->session->exit_value < error)
    gdata->session->exit_value = error;

  ssh_adt_destroy(gdata->cur_item->file_list);
  gdata->cur_item->file_list = NULL;
  
  switch (error)
    {
    case SSH_FC_OK:
      /* Everything went just fine. */
      break;
    default:
      if (!gdata->session->quiet_mode)
        ssh_warning("%s", error_message);
      break;
    }

  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

void overwrite_rl_cb(const char *line, void *context)
{
  ScpGData gdata = (ScpGData) context;
  Boolean ret = TRUE;
  
  ssh_informational("\r\n");
  
  if (line == NULL)
    goto abort_transfer;

  switch (*line)
    {
    case 'a':
    abort_transfer:
      SSH_ASSERT(gdata->transfer_op_handle);
      ssh_operation_abort(gdata->transfer_op_handle);
      ssh_fsm_set_next(gdata->main_thread, scp_finish);
      ssh_fsm_continue(gdata->main_thread);
      return;
    case 'N':
      gdata->overwrite_persistent_answer = SCP_OVERWRITE_NO_TO_ALL;
      /* FALL THROUGH */
    case 'n':
      ret = FALSE;
      break;
    case 'Y':
      gdata->overwrite_persistent_answer = SCP_OVERWRITE_YES_TO_ALL;
      /* FALL THROUGH */
    case 'y':
      /* The default case. */
      break;
    default:
      ssh_readline_eloop("Please answer [y/Y/n/N/a]:", "", gdata->rl,
                         overwrite_rl_cb, gdata);
      return;
    }
  (*gdata->overwrite_completion_cb)(ret, gdata->overwrite_completion_ctx);
}

void transfer_overwrite_cb(SshFileCopyFile source_file,
                           SshFileCopyFile destination_file,
                           SshFileCopyTransferOverwriteCompletionCB
                           completion_cb,
                           void *completion_context,
                           void *context)
{
  ScpGData gdata = (ScpGData) context;
  char *prompt;
  SSH_PRECOND(gdata);
  SSH_PRECOND(gdata->rl);

  if (gdata->overwrite_persistent_answer != SCP_OVERWRITE_UNDEFINED)
    {
      Boolean ret = FALSE;
      
      if (gdata->overwrite_persistent_answer == SCP_OVERWRITE_YES_TO_ALL)
        ret = TRUE;

      (*completion_cb)(ret, completion_context);
      return;
    }
  ssh_xdsprintf(&prompt, "Overwrite destination file '%s' with '%s' "
                "(yes/yes to all/no/no to all/abort) [y/Y/n/N/a]:",
                ssh_file_copy_file_generate_full_path(destination_file),
                ssh_file_copy_file_generate_full_path(source_file));
  gdata->overwrite_completion_cb = completion_cb;
  gdata->overwrite_completion_ctx = completion_context;
  ssh_readline_eloop(prompt, "", gdata->rl,
                     overwrite_rl_cb, gdata);
  ssh_xfree(prompt);
}

void transfer_overwrite_no_cb(SshFileCopyFile source_file,
                              SshFileCopyFile destination_file,
                              SshFileCopyTransferOverwriteCompletionCB
                              completion_cb,
                              void *completion_context,
                              void *context)
{
  ScpGData gdata = (ScpGData) context;
  SSH_PRECOND(gdata);
  SSH_PRECOND(gdata->overwrite_persistent_answer == SCP_OVERWRITE_NO_TO_ALL);
  (*completion_cb)(FALSE, completion_context);
}

void transfer_nonfatal_error_cb(SshFileCopyError error,
                                const char *error_string,
                                void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  ScpGData gdata = (ScpGData) ssh_fsm_get_gdata(thread);

  SSH_TRACE(2, ("Received error %s.", ssh_file_copy_errors[error].error_name));

  if (!gdata->session->quiet_mode)
    ssh_warning("%s", error_string);

  if (gdata->session->exit_value < error)
    gdata->session->exit_value = error;
}

SSH_FSM_STEP(scp_transfer)
{
  ScpGData gdata = (ScpGData) fsm_context;

  SSH_FSM_SET_NEXT(scp_glob);

#ifdef DEBUG_LIGHT
  if (SSH_DEBUG_ENABLED(6))
    scp_dump_file_list(gdata->cur_item->file_list);
#endif /* DEBUG_LIGHT */
  
  SSH_FSM_ASYNC_CALL(gdata->transfer_op_handle =
                     ssh_file_copy_transfer_multiple
                     (gdata->cur_item->connection,
                      gdata->cur_item->file_list,
                      gdata->dest_conn, gdata->dest_path,
                      gdata->attrs,
                      transfer_ready_cb, transfer_nonfatal_error_cb,
                      thread));
}

SSH_FSM_STEP(scp_finish)
{
  ScpGData gdata = (ScpGData) fsm_context;
  /* XXX Do we need to do something else here? */
  ssh_fsm_destroy(gdata->fsm);

  ssh_adt_destroy(gdata->file_list);

  ssh_xfree(gdata->attrs->source_newline);
  ssh_xfree(gdata->attrs->dest_newline);
  ssh_xfree(gdata->attrs);
  ssh_file_copy_connection_destroy(gdata->dest_conn);
  ssh_xfree(gdata->dest_path);
  if (gdata->rl)
    ssh_readline_eloop_uninitialize(gdata->rl);
  if (gdata->stdio_stream)
    ssh_stream_destroy(gdata->stdio_stream);
  
  ssh_xfree(gdata);
  return SSH_FSM_FINISH;
}

void scp_thread_destructor(SshFSM fsm, void *tdata)
{
  /* Nothing here yet. */
}

SshFSMStateDebugStruct scp_states[] =
{ { "scp_glob", "Glob locations", scp_glob },
  { "scp_recurse", "Recurse source directories", scp_recurse },
  { "scp_transfer", "Tranfer files", scp_transfer },
  { "scp_finish", "Finish", scp_finish }
};

void scp_simple_progress_cb(SshFileCopyConnection source,
                            SshFileCopyFile source_file,
                            SshFileCopyConnection dest,
                            SshFileCopyFile dest_file,
                            off_t read_bytes,
                            off_t written_bytes,
                            off_t skipped_bytes,
                            SshUInt64 elapsed_time,
                            Boolean finished,
                            void *context)
{
  double transfer_rate;
  char num_buffer[10], tr_rate_buffer[10], toc_buffer[10], *file_name;
  
  if (!finished)
    /* Only print info when transfer finished. */
    return;

  if (read_bytes == skipped_bytes && skipped_bytes > 0)
    {
      fprintf(stdout, "\r%s: complete md5 match -> "
              "transfer skipped\r\n",
              ssh_file_copy_file_get_name(source_file));
      return;
    }

  if (elapsed_time <= 0)
    elapsed_time = 1L;

  transfer_rate = (double) (SshInt64)written_bytes / (SshInt64)elapsed_time;
  ssh_format_number(num_buffer, sizeof(num_buffer),
                    written_bytes, 1024);
  ssh_format_number(tr_rate_buffer, sizeof(tr_rate_buffer),
                    (SshUInt64)transfer_rate, 1024);
  ssh_format_time(toc_buffer, sizeof(toc_buffer), elapsed_time);

  file_name = ssh_file_copy_file_generate_full_path(source_file);
  
  fprintf(stdout, "%s | %4sB | %4sB/s | TOC: %s\r\n",
          file_name, num_buffer, tr_rate_buffer, toc_buffer);
  ssh_xfree(file_name);
}

off_t parse_offset_value(const char *value)
{
  char ch, *str;
  off_t num;

  str = ssh_xstrdup(value);

  ch = str[strlen(str) - 1];

  if (!isdigit(ch))
    {
      str[strlen(str) - 1] = '\0';
      ch = tolower(ch);
    }

  num = strtol(str, NULL, 10);

  if (num < 0)
    {
      num = -1;
      goto error;
    }

  if (!isdigit(ch))
    {
      switch (ch)
        {
        case 'k': /* kilos. */
          num *= 1024;
          break;
        case 'm': /* megas. */
          num *= 1024 * 1024;
          break;
        case 'g': /* gigas. */
          num *= 1024 * 1024 * 1024;
          break;
        default:
          num = -1;
        }
    }

 error:
  ssh_xfree(str);
  return num;
}

char *backslashify(char *str)
{
  char *new_str = ssh_xcalloc(strlen(str)*2+1, sizeof(char));
  int i = 0, j = 0;

  /* Escape every character except '/' of the string. */
  /* XXX Bound to cause problems with MULTIBYTE. Yuichi, take a look. */
  for (i = 0; i < strlen(str); i++)
    {
      if (str[i] != '/')
        {
          new_str[j++] = '\\';
        }
      new_str[j++] = str[i];
    }
  new_str[j] = '\0';
  return new_str;
}




#define SCP_LONGOPT_OVERWRITE  0x102
#define SCP_LONGOPT_STATISTICS 0x103

SshLongOptionStruct scp_longopts[] =
{
  { "help",    SSH_GETOPT_LONG_NO_ARGUMENT, NULL, 'h'},
  { "version", SSH_GETOPT_LONG_NO_ARGUMENT, NULL, 'V'},
  { "verbose", SSH_GETOPT_LONG_NO_ARGUMENT, NULL, 'v'},






  { "interactive", SSH_GETOPT_LONG_OPTIONAL_ARGUMENT, NULL,
    'I' },
  { "overwrite", SSH_GETOPT_LONG_OPTIONAL_ARGUMENT, NULL,
    SCP_LONGOPT_OVERWRITE },
  { "statistics", SSH_GETOPT_LONG_REQUIRED_ARGUMENT, NULL,
    SCP_LONGOPT_STATISTICS },
  { NULL, 0, NULL, 0}
};

int main(int argc, char **argv)
{
  int ch, i;
  char *filename, *getopt_str;
  ScpSession session = ssh_xcalloc(1, sizeof(*session));
  ScpGData gdata = NULL;
  SshFileCopyError exit_value = SSH_FC_OK;
  char opt_buf[10];
  int longind = -1;
  Boolean bool_arg;
  
  ssh_debug_register_callbacks(scp_fatal, scp_warning, scp_debug,
                               (void *)(session));
  ssh_app_set_debug_format(NULL);

  gdata = ssh_xcalloc(1, sizeof(*gdata));
  gdata->fsm = ssh_fsm_create(gdata);
  ssh_fsm_register_debug_names(gdata->fsm,
                               scp_states,
                               SSH_FSM_NUM_STATES(scp_states));

  scp_session_initialize(session);
  gdata->session = session;

  gdata->file_list = ssh_adt_create_generic(SSH_ADT_LIST,
                                            SSH_ADT_DESTROY,
                                            scp_file_list_item_destroy,
                                            SSH_ADT_ARGS_END);
  gdata->attrs = ssh_xcalloc(1, sizeof(*gdata->attrs));
  gdata->attrs->max_buf_size = SSH_FC_DEFAULT_BUF_SIZE;
  gdata->attrs->max_buffers = SSH_FC_DEFAULT_BUF_NUM;





  /* Default file transfer mode is binary */
  gdata->attrs->transfer_mode = SSH_FC_TRANSFER_MODE_BINARY;







  ssh_event_loop_initialize();


  if ((filename = strrchr(argv[0], '/')) != NULL)
    filename++;
  else
    filename = argv[0];
  
  session->scp2_name = ssh_xstrdup(filename);
  ssh_register_program_name(filename);





  filename = NULL;


  gdata->attrs->source_newline = ssh_xstrdup("\n");



  gdata->attrs->dest_newline = ssh_xstrdup("\n");
  
  getopt_str = "qQdpvua::h?P:c:D:rVBN:b:O:S:o:i:I"




    "64tf1";








#define ADD_OPT(opt)                                            \
do {                                                            \
  SshADTHandle h;                                               \
  h = ssh_adt_insert_to(session->ssh_options, SSH_ADT_END,      \
                        ssh_xstrdup(opt));                      \
  SSH_ASSERT(h != SSH_ADT_INVALID);                             \
} while (0)

  while ((ch = ssh_getopt_long(argc, argv, getopt_str,
                               scp_longopts, &longind, NULL)) != -1)
    {
      bool_arg = TRUE;
      
      if (ssh_optarg && (!strcmp(ssh_optarg, "no") ||
                         !strcmp(ssh_optarg, "disable")))
        bool_arg = FALSE;

      switch(ch)
        {

        case 't':
        case 'f':
          /* Scp 1 compatibility mode, this is remote server for ssh-1 scp,
             exec old scp here. */
          {
            ssh_warning("Executing scp1.");
            execvp("scp1", argv);
            ssh_fatal("Executing ssh1 in compatibility mode failed (Check "
                      "that scp1 is in your PATH).");
          }
          break;
        case '1':
          /* Scp 1 compatibility in the client */
          {
            char **av;
            int j;
            int removed_flag = FALSE;

            av = ssh_xcalloc(argc, sizeof(char *));
            ssh_warning("Executing scp1 compatibility.");
            for (i = 0, j = 0 ; i < argc ; i++)
              {
                SSH_DEBUG(3, ("argv[%d] = %s", i, argv[i]));

                /* skip first "-1" argument */
                if (strcmp("-1", argv[i]) || removed_flag)
                  {
                    av[j] = argv[i];
                    j++;
                  }
                else
                  {
                    removed_flag = TRUE;
                  }
              }

            execvp("scp1", av);
            ssh_fatal("Executing ssh1 in compatibility mode failed (Check "
                      "that scp1 is in your PATH).");
          }
          break;





















        case 'p':
          gdata->attrs->preserve_attributes = TRUE;
          break;
        case 'r':
          gdata->session->recurse_dirs = TRUE;
          gdata->attrs->create_dirs = TRUE;
          break;
        case 'P':
          session->remote_port = ssh_xstrdup(ssh_optarg);

          {
            int port = atoi(session->remote_port);

            if ((port <= 0) || (port > 65535))
              {
                ssh_warning("Port specification '%s' is invalid or out of "
                            "range.",
                            session->remote_port);
                scp_usage(session);
                exit(SSH_FC_ERROR);
              }
          }

          break;
        case 'i':
        case 'o':
        case 'c':
        case '4':
        case '6':
          ssh_snprintf(opt_buf, sizeof(opt_buf), "-%c", ch);
          ADD_OPT(opt_buf);
          if (ssh_optarg)
            ADD_OPT(ssh_optarg);
          break;
        case 'B':
          ADD_OPT("-o");
          ADD_OPT("batchmode=yes");
          session->batch_mode = TRUE;
          break;
        case 'S':
          ssh_xfree(session->ssh_path);
          session->ssh_path = ssh_xstrdup(ssh_optarg);
          break;
        case 'd':
          gdata->attrs->destination_must_be_dir = TRUE;
          break;
        case 'D':
          ssh_debug_set_level_string(ssh_optarg);
          ADD_OPT("-d");
          ADD_OPT(ssh_optarg);          
          session->debug_mode = TRUE;
          break;
        case 'v':          
          ssh_debug_set_global_level(2);
          ADD_OPT("-v");
          session->debug_mode = TRUE;
          break;
        case 'q':
          session->quiet_mode = TRUE;
          break;
        case 'Q':
          session->statistics = SCP_STATISTICS_NO;
          break;
        case 'u':
          gdata->attrs->unlink_source = TRUE;
          break;
        case 'n':
          session->do_not_copy = TRUE;
          break;
        case 'h':
        case '?':
          scp_usage(session);
          exit(SSH_FC_OK);
          break;
        case 'a':
          gdata->attrs->transfer_mode = SSH_FC_TRANSFER_MODE_ASCII;
          if (ssh_optarg)
            {
              /* This is hint to filecopy; it uses server given newline
                 convention, if any. */
              char *nl;
#define DST_STR "dest:"
#define SRC_STR "src:"
              if (!strncmp(ssh_optarg, DST_STR, strlen(DST_STR)))
                {
                  nl = ssh_file_copy_parse_newline_convention
                    (ssh_optarg + strlen(DST_STR));
                  gdata->attrs->dest_newline = nl;
                }
              else if (!strncmp(ssh_optarg, SRC_STR, strlen(SRC_STR)))
                {
                  nl = ssh_file_copy_parse_newline_convention
                    (ssh_optarg + strlen(SRC_STR));
                  gdata->attrs->source_newline = nl;
                }
              else
                {
                  nl = ssh_file_copy_parse_newline_convention(ssh_optarg);
                  /* Default mode of operation. */
                  gdata->attrs->dest_newline = nl;
                }
#undef DST_STR
#undef SRC_STR
              if (!nl)
                ssh_fatal("Invalid argument to -a (%s)", ssh_optarg);
            }
          break;
        case 'V':
          ssh2_version(NULL);
          exit(SSH_FC_OK);
          break;
        case 'N':
          gdata->attrs->max_buffers = atoi(ssh_optarg);
          if (!gdata->attrs->max_buffers)
            ssh_fatal("-N requires an argument greater than zero.");
          break;
        case 'b':
          gdata->attrs->max_buf_size = atoi(ssh_optarg);
          if (!gdata->attrs->max_buf_size)
            ssh_fatal("-b requires an argument greater than zero.");
          break;
        case 'O':
          /* read and write offsets. */
          if (strlen(ssh_optarg) < 2)
            ssh_fatal("invalid -O parameter");

          switch (ssh_optarg[0])
            {
            case 'r':
            case 'R':
              gdata->attrs->read_offset = parse_offset_value(&ssh_optarg[1]);
              break;
            case 'w':
            case 'W':
              gdata->attrs->write_offset =
                parse_offset_value(&ssh_optarg[1]);
              break;
            default:
              ssh_fatal("invalid -O parameter (requires -O[rRwW]<bytes> "
                        "format).");
              break;
            }
          break;
        case 'I':
          gdata->session->interactive_mode = (bool_arg == TRUE);
          break;








        case SCP_LONGOPT_OVERWRITE:
          if (!ssh_optarg)
            /* Default is to overwrite. */
            break;

          if (!strcmp(ssh_optarg, "no"))
            gdata->overwrite_persistent_answer = SCP_OVERWRITE_NO_TO_ALL;
          
          break;
        case SCP_LONGOPT_STATISTICS:
          if (!strcmp(ssh_optarg, "no"))
            session->statistics = SCP_STATISTICS_NO;
          else if (!strcmp(ssh_optarg, "yes"))
            session->statistics = SCP_STATISTICS_YES;
          else if (!strcmp(ssh_optarg, "simple"))
            session->statistics = SCP_STATISTICS_SIMPLE;
          break;
        default:
          scp_usage(session);
          exit(SSH_FC_ERROR);
          break;
        }
    }

  if (isatty(fileno(stdout)))
    session->have_tty = TRUE;

  if (session->have_tty == FALSE && session->statistics == SCP_STATISTICS_YES)
    session->statistics = SCP_STATISTICS_SIMPLE;

  if (argc - ssh_optind < 2)
    {
      if (argc > 1)
        ssh_warning("Missing destination file.");
      scp_usage(session);
      exit(SSH_FC_ERROR);
    }

#ifndef HAVE_SIGNAL
  ssh_sigchld_initialize();
#endif /* HAVE_SIGNAL */


  /* Parse source locations */
  for (i = ssh_optind; i < argc - 1; i++)
    {
      SshFileCopyConnection source_connection;
      ScpFileListItem temp_item;

      source_connection = ssh_file_copy_connection_allocate();

      if (scp_location_parse(source_connection, &filename, argv[i]))
        {
          /* An error occurred. */
          /* XXX Free everything that can be freed, and exit. */

          ssh_warning("Invalid source specification '%s'.", argv[i]);
          scp_usage(session);
          exit(SSH_FC_ERROR);
        }

      /* If 'connection' is local, and filename isn't absolute, it
         should be preceded by current working directory. */
      if ((source_connection)->host == NULL)
        {
          if (!ssh_app_is_file_name_absolute(filename))
              {
                char *new_filename, *cwd;

                cwd = ssh_getcwd();
                /* We need to escape the characters from getcwd, so that
                   they are not interpreted as regexs by the globber. */

                new_filename = backslashify(cwd);
                ssh_xfree(cwd);
                cwd = new_filename;

                ssh_xdsprintf(&new_filename, "%s/%s", cwd, filename);

                ssh_xfree(cwd);
                ssh_xfree(filename);
                filename = new_filename;
              }
        }

      /* Check if 'connection' is identical to a previous one, and
         if so, just add the 'filename' to the previous. */
      if ((temp_item =
           scp_file_list_find_matching_connection(gdata->file_list,
                                                  source_connection)) != NULL)
        {
          ssh_file_copy_connection_destroy(source_connection);
          scp_add_raw_file(temp_item->file_list, filename);
        }
      else
        {
          ScpFileListItem source = scp_file_list_item_allocate();
          SshADTHandle h;
          
          source->connection = source_connection;

          scp_add_raw_file(source->file_list, filename);

          /* Add the file item to the list. */
          h = ssh_adt_insert_to(gdata->file_list, SSH_ADT_END, source);
          SSH_ASSERT(h != SSH_ADT_INVALID);
        }

      ssh_xfree(filename);
    }

  gdata->dest_conn = ssh_file_copy_connection_allocate();
  
  /* Parse destination location. */
  if (scp_location_parse(gdata->dest_conn, &filename, argv[i]))
    {
      /* An error occurred. */
      /* XXX Free everything that can be freed, and exit. */

      ssh_warning("Invalid destination specification '%s'.", argv[i]);
      scp_usage(session);
      exit(SSH_FC_ERROR);
    }

  if (gdata->dest_conn->host == NULL)
    {
      SshRegexMatcher rex;
      SshRegexContext rex_ctx = ssh_app_get_global_regex_context();

      if (!session->quiet_mode)
        {
          /* Test whether the user typoed the destination file. */
          rex = ssh_regex_create(rex_ctx,
                                 /* user */
                                 "^{({\\@|[-@]}+)@}/1/"
                                 /* host */
                                 "({{\\#|\\:}|[-~:#]}+)"
                                 /* port */
                                 "{#({\\:|[-~:]}+)}?"
                                 /* no filename */
                                 "$",
                                 SSH_REGEX_SYNTAX_SSH);
          SSH_VERIFY(rex != NULL);
          if (ssh_regex_match_cstr(rex, filename))
            ssh_informational("Destination will be local file `%s' (this "
                              "might not be what you\r\n"
                              "intended).\r\n", filename);
          ssh_regex_free(rex);
        }

      if (!ssh_app_is_file_name_absolute(filename))
          {
            char *new_filename, *cwd;

            cwd = ssh_getcwd();

            if (cwd == NULL)
              ssh_fatal("Couldn't get path of current working directory");

            ssh_xdsprintf(&new_filename, "%s/%s", cwd, filename);

            ssh_xfree(cwd);
            ssh_xfree(filename);
            filename = new_filename;
          }
    }

  switch (gdata->session->statistics)
    {
    case SCP_STATISTICS_NO:
      gdata->attrs->progress_cb = NULL_FNPTR;
      break;
    case SCP_STATISTICS_SIMPLE:
      gdata->attrs->progress_cb = scp_simple_progress_cb;
      break;
    default:
      gdata->attrs->progress_cb = ssh_file_copy_transfer_default_progress_cb;
      ssh_file_copy_transfer_default_progress_cb_init();
      break;
    }

  if (gdata->session->interactive_mode && !gdata->session->batch_mode)
    {
      gdata->stdio_stream = ssh_stream_fd_stdio();
      gdata->rl = ssh_readline_eloop_initialize(gdata->stdio_stream);
      if (gdata->rl)
        {
          gdata->attrs->overwrite_cb = transfer_overwrite_cb;
        }
      else
        {
          ssh_fatal("Interactive mode specified, but failed to allocate "
                    "readline.");
        }
    }

  if (gdata->overwrite_persistent_answer == SCP_OVERWRITE_NO_TO_ALL)
    {
      /* This might override interactive mode, but it is completely
         valid. */
      gdata->attrs->overwrite_cb = transfer_overwrite_no_cb;
    }
  

  gdata->dest_path = filename;





  /* Register connection callback for sshfilecopy. */
  ssh_file_copy_register_connect_callback(connect_callback, session);

  gdata->h = ssh_adt_enumerate_start(gdata->file_list);
  
  gdata->main_thread = ssh_fsm_thread_create(gdata->fsm, scp_glob, NULL_FNPTR,
                                             NULL_FNPTR, NULL);

  gdata->attrs->context = gdata;
  
#ifdef TEST_TRANSFER_ABORT
  ssh_register_signal(SIGINT, scp_interrupt_signal_handler, gdata);
#endif /* TEST_TRANSFER_ABORT */

  ssh_event_loop_run();

  ssh_global_uninit();
  
  ssh_event_loop_uninitialize();

  exit_value = session->exit_value;

  ssh_app_free_global_regex_context();

  scp_session_destroy(session);

  return exit_value;
}

void new_stream_callback(SshStreamNotification notification, void *context)
{
  ScpConnectionCtx ctx = (ScpConnectionCtx) context;
  int ret = 0;

  SSH_TRACE(3, ("notification: %d", notification));

  switch (notification)
    {
    case SSH_STREAM_INPUT_AVAILABLE:

      while ((ret =
              ssh_stream_read(ctx->stream,
                              (unsigned char *)&(ctx->buffer[ctx->read_bytes]),
                              1)) > 0 &&
             ctx->read_bytes < sizeof(ctx->buffer) - 1)
        {
          ctx->read_bytes += ret;

          SSH_TRACE(20, ("read char: %c",
                        ctx->buffer[strlen(ctx->buffer) - 1]));
          SSH_TRACE(20, ("read_bytes: %d, buffer len: %d",
                        ctx->read_bytes, strlen(ctx->buffer)));
          SSH_DEBUG_HEXDUMP(25, ("received message:"),
                            (unsigned char *)ctx->buffer, ctx->read_bytes);
          fflush(stderr);
          if (ctx->buffer[strlen(ctx->buffer) - 1] == '\n')
            {
              SSH_DEBUG(3, ("buffer: '%s'", ctx->buffer));
              if (strcmp(ctx->buffer, "AUTHENTICATED YES\n") == 0)
                {
                  (*ctx->completion_cb)(ctx->stream, ctx->completion_context);
                  return;
                }
              else
                {
                  SSH_DEBUG_HEXDUMP(3, ("received message:"),
                                    (unsigned char *)ctx->buffer,
                                    ctx->read_bytes);
                  ssh_warning("ssh2 client failed to authenticate. (or you "
                              "have too old ssh2 installed, check "
                              "with ssh2 -V)");
                }
            }

        }
      if (ret == -1)
        return;
      if (ret == 0)
        {

          ssh_warning("EOF received from ssh2. ");




        }
      if (ctx->read_bytes >= sizeof(ctx->buffer) - 1)
        {
          ssh_warning("Received corrupted (or wrong type of) data from "
                      "ssh2-client.");
        }
      break;
    case SSH_STREAM_CAN_OUTPUT:
      return;
    case SSH_STREAM_DISCONNECTED:
      SSH_TRACE(2, ("Received SSH_STREAM_DISCONNECTED from wrapped stream."));
      ssh_warning("ssh2 client failed to authenticate. (or you "
                  "have too old ssh2 installed, check with ssh2 -V)");
      /* XXX */
      break;
    }
  ssh_stream_destroy(ctx->stream);
  (*ctx->completion_cb)(NULL, ctx->completion_context);
}

void connect_callback(SshFileCopyConnection connection, void *context,
                      SshFileCopyStreamReturnCB completion_cb,
                      void *completion_context)
{
  ScpSession session = (ScpSession) context;
  ScpConnectionCtx new_ctx;
#define SSH_ARGV_SIZE   128
  struct stat st;
  SshStream client_stream;
  char *ssh_argv[SSH_ARGV_SIZE];
  int i = 0;



  SshADTHandle h;
  
  new_ctx = ssh_xcalloc(1, sizeof(*new_ctx));
  new_ctx->session = session;
  new_ctx->completion_cb = completion_cb;
  new_ctx->completion_context = completion_context;


  if (stat(session->ssh_path, &st) < 0)
    {
      SSH_TRACE(0, ("Couldn't find ssh2 on path specified (%s). "
                    "Trying default PATH...", session->ssh_path));
      ssh_xfree(session->ssh_path);
      session->ssh_path = ssh_xstrdup("ssh2");
    }


  ssh_argv[i++] = session->ssh_path;

  if (connection->user != NULL)
    {
      ssh_argv[i++] = "-l";
      ssh_argv[i++] = connection->user;
    }
  if (connection->port != NULL)
    {
      ssh_argv[i++] = "-p";
      ssh_argv[i++] = connection->port;
    }
  else if (session->remote_port != NULL)
    {
      ssh_argv[i++] = "-p";
      ssh_argv[i++] = session->remote_port;
    }
  if (session->quiet_mode)
    {
      ssh_argv[i++] = "-q";
    }

  ssh_argv[i++] = "-x";
  ssh_argv[i++] = "-a";

  ssh_argv[i++] = "-oclearallforwardings=yes";
  ssh_argv[i++] = "-oforcepttyallocation=no";

  ssh_argv[i++] = "-opasswordprompt=%U@%H's password:";



  ssh_argv[i++] = "-onodelay=yes";
  ssh_argv[i++] = "-oauthenticationnotify=yes";

  for (h = ssh_adt_enumerate_start(session->ssh_options);
       h != SSH_ADT_INVALID;
       h = ssh_adt_enumerate_next(session->ssh_options, h))
    {
      char *str = ssh_adt_get(session->ssh_options, h);
      SSH_ASSERT(str != NULL);
      
      if (i >= SSH_ARGV_SIZE - 4)
        ssh_fatal("Too many arguments for %s (at least %d given).",
                  session->ssh_path, i + 2 + 1); /* Starts from zero... */
      ssh_argv[i++] = str;
    }
  
  ssh_argv[i++] = connection->host;

  ssh_argv[i++] = "-s";
  ssh_argv[i++] = "sftp";

  ssh_argv[i] = NULL;

  SSH_ASSERT(i < SSH_ARGV_SIZE);

  if (session->debug_mode)
    {
      SSH_DEBUG_INDENT;
      for (i = 0; ssh_argv[i]; i++)
        SSH_DEBUG(2, ("argv[%d] = %s", i, ssh_argv[i]));
      SSH_DEBUG_UNINDENT;
    }


  switch (ssh_pipe_create_and_fork(&client_stream, NULL))














    {
    case SSH_PIPE_ERROR:
      ssh_fatal("ssh_pipe_create_and_fork() failed");

    case SSH_PIPE_PARENT_OK:
      new_ctx->stream = client_stream;
      new_ctx->session = session;
      new_ctx->completion_cb = completion_cb;
      new_ctx->completion_context = completion_context;

      new_ctx->child_pid = ssh_pipe_get_pid(client_stream);
      ssh_sigchld_register(new_ctx->child_pid, scp_sigchld_handler,
                           new_ctx);


      ssh_stream_set_callback(client_stream, new_stream_callback,
                              new_ctx);
      new_stream_callback(SSH_STREAM_INPUT_AVAILABLE,
                          new_ctx);
#ifdef SUSPEND_AFTER_FORK
      kill(getpid(), SIGSTOP);
#endif /* SUSPEND_AFTER_FORK */




      return;


    case SSH_PIPE_CHILD_OK:
      execvp(ssh_argv[0], ssh_argv);
      fprintf(stderr, "Executing ssh2 failed. Command:'");
      for (i = 0;ssh_argv[i] != NULL; i++)
        fprintf(stderr," %s", ssh_argv[i]);

      fprintf(stderr,"' System error message: '%s'\r\n", strerror(errno));
      exit(-1);

    }
  SSH_NOTREACHED;
}

/* Find a matching SshFileCopyConnection from file_list. file_list
   MUST contain ScpFileListItems. Return matching
   ScpFileListItem if found, or NULL if not. */
ScpFileListItem
scp_file_list_find_matching_connection(SshADTContainer file_list,
                                       SshFileCopyConnection connection)
{
  ScpFileListItem item;
  SshADTHandle h;
  
  SSH_PRECOND(file_list != NULL);
  SSH_PRECOND(connection != NULL);

  for(h = ssh_adt_enumerate_start(file_list);
      h != SSH_ADT_INVALID;
      h = ssh_adt_enumerate_next(file_list, h))
    {
      item = ssh_adt_get(file_list, h);
      SSH_ASSERT(item != NULL);
      
      if (ssh_file_copy_connection_compare(item->connection,
                                           connection))
        {
          /* Found. */
          SSH_DEBUG(6, ("Found matching connection (host: %s)",
                        connection->host));
          return item;
        }
    }

  SSH_DEBUG(4, ("No matching connection (host: %s)", connection->host));
  
  /* Matching item not found. */
  return NULL;
}

/* Parse a string to a SshFileCopyConnection structure. String is of form
   [[user@]host[#port]:]filename . In case of an error or if the given
   string is malformed, return TRUE. Otherwise, return FALSE. */
Boolean scp_location_parse(SshFileCopyConnection connection,
                           char **return_filename,
                           const char *orig_string)
{
  char *host = NULL, *user = NULL, *port = NULL, *filename = NULL;
  SshRegexContext rex_ctx = ssh_app_get_global_regex_context();
  SshRegexMatcher rex;

  SSH_ASSERT(orig_string != NULL);

  if (strlen(orig_string) == 0)
    {
      return TRUE;
    }




















  /* Match the special case, where host is enclosed in brackets. */
  rex = ssh_regex_create(rex_ctx,
                         /* user */
                         "^{({\\@|[-@]}+)@}?"
                         /* [host] */
                         "~[({\\~]|[-~]]}+)~]"
                         /* port */
                         "{#({\\:|[-~:]}+)}?"
                         /* filename */
                         ":(.*)$",
                         SSH_REGEX_SYNTAX_SSH);

  if (rex == NULL)
    {
      ssh_warning("Unable to create regex 1 in location parser.");
      return TRUE;
    }

  if (ssh_regex_match_cstr(rex, orig_string))
    {
      SSH_DEBUG(4, ("Match with special case."));
      user = (char *)ssh_regex_get_submatch(rex, 1);
      host = (char *)ssh_regex_get_submatch(rex, 2);
      port = (char *)ssh_regex_get_submatch(rex, 3);
      filename = (char *)ssh_regex_get_submatch(rex, 4);

      goto match;
    }
  ssh_regex_free(rex);

  /* Match the normal case. */
  rex = ssh_regex_create(rex_ctx,
                         /* user */
                         "^{({\\@|[-@]}+)@}?"
                         /* host */
                         "({{\\#|\\:}|[-~:#]}+)"
                         /* port */
                         "{#({\\:|[-~:]}+)}?"
                         /* filename */
                         ":(.*)$",
                         SSH_REGEX_SYNTAX_SSH);

  if (rex == NULL)
    {
      ssh_warning("Unable to create regex 2 in location parser.");
      return TRUE;
    }

  if (ssh_regex_match_cstr(rex, orig_string))
    {
      SSH_DEBUG(4, ("Match with normal case."));
      user = (char *)ssh_regex_get_submatch(rex, 1);
      host = (char *)ssh_regex_get_submatch(rex, 2);
      port = (char *)ssh_regex_get_submatch(rex, 3);
      filename = (char *)ssh_regex_get_submatch(rex, 4);
      goto match;
    }

  SSH_DEBUG(4, ("Falltrough, whole string is a filename."));
  filename = (char *)orig_string;

 match:
  SSH_DEBUG(4, ("user: %s, host: %s, port: %s, filename: %s",
                user ? user : "NULL", host ? host : "NULL",
                port ? port : "NULL", filename ? filename : "NULL"));

  /* If filename is empty, default to dot. */
  if (strlen(filename) == 0)
    filename = ".";


  filename = ssh_xstrdup(filename);





  connection->user = ssh_glob_strip_escapes(user);
  connection->host = ssh_glob_strip_escapes(host);
  connection->port = ssh_glob_strip_escapes(port);

  ssh_regex_free(rex);

  SSH_DEBUG(3, ("user = %s, host = %s, port = %s, filename = %s",
                connection->user ?
                connection->user : "NULL",
                connection->host ?
                connection->host : "NULL",
                connection->port ?
                connection->port : "NULL",
                filename));

  *return_filename = filename;

  return FALSE;
}

void scp_add_raw_file(SshADTContainer file_list, char *filename)
{
  SshADTHandle h;






  h = ssh_adt_insert_to(file_list, SSH_ADT_END,
                        ssh_file_copy_strip_dot_dots(filename));
  SSH_ASSERT(h != NULL);
}


