/*

ssh2compat.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 2001 SSH Communications Security Oy, Espoo, Finland
  All rights reserved.

  Conveniency functions for compatibility with older ssh-versions.
  (intended for compat code that touches both sshproto and apps/ssh)
*/

#include "sshincludes.h"
#include "ssh2compat.h"

#define SSH_DEBUG_MODULE "Ssh2Compat"

/* Change public key hashing scheme as needed. Only used with
   RSA-keys. Returns FALSE on success. */
Boolean ssh_compat_rsa_public_key_change_scheme(SshPublicKey pubkey,
                                                Boolean hash_is_sha1)
{
#ifdef SSHDIST_CRYPT_RSA
  char *type = NULL, *new_scheme = NULL;
  
  SSH_PRECOND(pubkey != NULL);

  if (ssh_public_key_get_info(pubkey,
                              SSH_PKF_KEY_TYPE, &type,
                              SSH_PKF_END) != SSH_CRYPTO_OK)
    {
      SSH_DEBUG(4, ("ssh_public_key_get_info failed."));
      return FALSE;
    }

  if (strncmp(type, "if-modn", 7) != 0)
    {
      SSH_DEBUG(4, ("Public key is not an RSA key, so nothing "
                    "needs to be done. (type = '%s')", type));      
      return TRUE;
    }
  SSH_DEBUG(4, ("Public key scheme = '%s')", type));

  if (hash_is_sha1)
    {
      new_scheme = "rsa-pkcs1-sha1";
    }
  else
    {
      new_scheme = "rsa-pkcs1-md5";
    }
      
  if (ssh_public_key_select_scheme(pubkey,
                                   SSH_PKF_SIGN, new_scheme,
                                   SSH_PKF_END) != SSH_CRYPTO_OK)
    {
      SSH_DEBUG(4, ("Couldn't change scheme to '%s'.", new_scheme));
      return FALSE;
    }



#endif /* SSHDIST_CRYPT_RSA */
  return TRUE;
}

/* Change private key hashing scheme as needed. Only used with
   RSA-keys. Returns FALSE on success. */
Boolean ssh_compat_rsa_private_key_change_scheme(SshPrivateKey privkey,
                                                 Boolean hash_is_sha1)
{
#ifdef SSHDIST_CRYPT_RSA
  char *type = NULL, *new_scheme = NULL;
  
  SSH_PRECOND(privkey != NULL);

  if (ssh_private_key_get_info(privkey,
                               SSH_PKF_KEY_TYPE, &type,
                               SSH_PKF_END) != SSH_CRYPTO_OK)
    {
      SSH_DEBUG(4, ("ssh_private_key_get_info failed."));
      return FALSE;
    }

  if (strncmp(type, "if-modn", 7) != 0)
    {
      SSH_DEBUG(4, ("Private key is not an RSA key, so nothing "
                    "needs to be done. (type = '%s')", type));      
      return TRUE;
    }

  if (hash_is_sha1)
    {
      new_scheme = "rsa-pkcs1-sha1";
    }
  else
    {
      new_scheme = "rsa-pkcs1-md5";
    }
      
  if (ssh_private_key_select_scheme(privkey,
                                    SSH_PKF_SIGN, new_scheme,
                                    SSH_PKF_END) != SSH_CRYPTO_OK)
    {
      SSH_DEBUG(4, ("Couldn't change scheme to '%s'.", new_scheme));
      return FALSE;
    }



#endif /* SSHDIST_CRYPT_RSA */
  return TRUE;
}

Boolean ssh_compat_is_rsa_public_key(SshPublicKey pubkey)
{
  char *type = NULL;
  Boolean is_rsa;
  SSH_PRECOND(pubkey != NULL);

  if (ssh_public_key_get_info(pubkey,
                              SSH_PKF_KEY_TYPE, &type,
                              SSH_PKF_END) != SSH_CRYPTO_OK)
    {
      SSH_DEBUG(4, ("ssh_public_key_get_info failed."));
      return FALSE;
    }

  if (strncmp(type, "if-modn", 7) != 0)
    is_rsa = FALSE;
  else
    is_rsa = TRUE;

  ssh_xfree(type);
  return is_rsa;
}
