/*

  auths-pubkey.c

  Authors:
        Tatu Ylonen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Public key authentication, server-side.

*/

#include "ssh2includes.h"
#include "sshencode.h"
#include "sshauth.h"
#include "sshmsgs.h"
#include "sshuser.h"
#include "sshkeyfile.h"
#include "sshconfig.h"
#include "sshgetput.h"
#include "sshserver.h"
#include "sshdebug.h"
#include "sshuserfile.h"
#include "sshuserfiles.h"
#include "auths-common.h"
#include "auths-pubkey.h"
#include "ssh2compat.h"
#include "ssh2pubkeyencode.h"
#include "sshdlex.h"
#include "sshappcommon.h"
#include "sshadt_list.h"
#ifdef WITH_PGP
#include "ssh2pgp.h"
#endif /* WITH_PGP */







#define SSH_DEBUG_MODULE "Ssh2AuthPubKeyServer"


/* Uncomment this to put a delay to a successful authentication attempt
   (to test the authentication protocol). */
/* #define TEST_ASYNC_SERVER_METHODS */

#ifdef TEST_ASYNC_SERVER_METHODS
#include "sshtimeouts.h"

typedef struct SshPublicKeyAuthCtxRec
{
  SshBuffer packet;
  SshAuthServerResult result;
  SshAuthServerCompletionProc completion_proc;
  void *completion_context;
} *SshPublicKeyAuthCtx, SshPublicKeyAuthCtxStruct;

void publickey_auth_finalize_callback(void *context)
{
  SshPublicKeyAuthCtx publickey_auth_ctx = (SshPublicKeyAuthCtx)context;
  SSH_PRECOND(publickey_auth_ctx);
  SSH_PRECOND(publickey_auth_ctx->completion_proc);
  SSH_PRECOND(publickey_auth_ctx->completion_context);

  SSH_TRACE(2, ("Delay ended."));

  (*publickey_auth_ctx->completion_proc)
    (publickey_auth_ctx->result,
     publickey_auth_ctx->packet,
     publickey_auth_ctx->completion_context);

  ssh_xfree(publickey_auth_ctx);
}

#endif /* TEST_ASYNC_SERVER_METHODS */

















































































































































































































































































































































#ifdef DEBUG_LIGHT
void ssh_dump_publickey_options(SshCommon common)
{
  SshADTHandle handle;
  SshAuthPublicKeyOptions opts = common->publickey_options;
  Boolean first = TRUE;

  if (opts == NULL)
    return;

#define DUMP_LVL 4
#define DUMP_LIST(container, name)                                      \
  do {                                                                  \
    if ((container) != NULL)                                            \
      {                                                                 \
        for (handle = ssh_adt_enumerate_start((container));             \
             handle != SSH_ADT_INVALID;                                 \
             handle = ssh_adt_enumerate_next((container), handle))      \
          {                                                             \
            char *obj;                                                  \
                                                                        \
            if (first)                                                  \
              {                                                         \
                SSH_DEBUG(DUMP_LVL, ("%s:", (name)));                   \
                first = FALSE;                                          \
              }                                                         \
                                                                        \
            obj = (char *)ssh_adt_get((container), handle);             \
            SSH_DEBUG(DUMP_LVL, ("  %s", obj));                         \
          }                                                             \
        first = TRUE;                                                   \
      }                                                                 \
  } while (0)

  DUMP_LIST(opts->use_allowed_from, "allow-from");
  DUMP_LIST(opts->use_denied_from, "deny-from");

  SSH_DEBUG(DUMP_LVL, ("command             = %s", common->forced_command ?
                       common->forced_command : "NULL"));

  DUMP_LIST(opts->environment_vars, "environment");

  SSH_DEBUG(DUMP_LVL, ("idle-timeout        = %d", opts->idle_timeout));
  SSH_DEBUG(DUMP_LVL, ("no-port-forwarding  = %s", opts->no_port_forwarding ?
                       "TRUE" : "FALSE"));
  SSH_DEBUG(DUMP_LVL, ("no-x11-forwarding   = %s", opts->no_x11_forwarding ?
                       "TRUE" : "FALSE"));
  SSH_DEBUG(DUMP_LVL, ("no-agent-forwarding = %s", opts->no_agent_forwarding ?
                       "TRUE" : "FALSE"));
  SSH_DEBUG(DUMP_LVL, ("no-pty              = %s", opts->no_pty ?
                       "TRUE" : "FALSE"));
#undef DUMP_LVL
#undef DUMP_LIST
}
#endif /* DEBUG_LIGHT */

/* Return TRUE if parsing fails, FALSE if it
   succeeds. ``pubkey_options'' has to xmallocated before this call, and
   it is not freed by this call. The ``option_str'' should contain the
   options of the publickey, separated by commas.

   string ::= "([^"\\]|\\.)*"
   non-string ::= [^",]*
   parameter-name ::= [a-zA-Z]+
   item ::= <parameter-name>=(<string>|<non-string>)
   option ::= [-a-z[:digit:]]+
   list ::= ((<item>,[ \t]*)*<item>)?

   Ex:
   command="echo kukkuu", from="*.hut\\,.fi,*.ssh.com", no-x11-forwarding */
Boolean ssh_parse_publickey_options(SshCommon common,
                                    SshAuthPublicKeyOptions pubkey_options,
                                    SshMetaConfig metaconfig,
                                    const char *option_str)
{
  int len, token;
  char *ptr = (char *)option_str;
  char *end = (char *)option_str + strlen(option_str);
  const char *regexs[10];
  SshDLexer lexer;
  int i = 0;
  Boolean ret = FALSE;

  regexs[i++] = "[-a-zA-Z]+=(\"([^\"\\\\]|\\\\.)*\"|[^\",]*)";
  regexs[i++] = "[-a-z[:digit:]]+";
  regexs[i++] = ",[ \\t]*";
  regexs[i++] = NULL;

  lexer = ssh_dlex_create(ssh_app_get_global_regex_context(),
                          regexs,
                          3,
                          SSH2_REGEX_SYNTAX,
                          SSH_DLEX_FIRST_MATCH);

  while (ptr < end && ssh_dlex_next(lexer, ptr, end - ptr, &len, &token))
    {
      switch (token)
        {
        case 0:
          /* options with arguments. */
          {
            SshDLexer sub_lexer;
            int sub_len, sub_token;
            char *to_be_deleted, *sub_ptr, *sub_end;
            char *option, *value;

            sub_ptr = ssh_xstrdup(ptr);
            sub_ptr[len] = '\0';
            to_be_deleted = sub_ptr;
            sub_end = sub_ptr + strlen(sub_ptr);

            i = 0;

            regexs[i++] = "[-a-zA-Z]+=";
            regexs[i++] = "(\"([^\"\\\\]|\\\\.)*\"|[^\",]*)";
            regexs[i++] = NULL;

            sub_lexer = ssh_dlex_create(ssh_app_get_global_regex_context(),
                                        regexs,
                                        2,
                                        SSH2_REGEX_SYNTAX,
                                        SSH_DLEX_FIRST_MATCH);

            ssh_dlex_next(sub_lexer, sub_ptr, sub_end - sub_ptr,
                          &sub_len, &sub_token);

            SSH_DEBUG(6, ("Got token '%s' (token = %d).", sub_ptr, sub_token));
            SSH_ASSERT(sub_token == 0);
            option = ssh_xstrdup(sub_ptr);
            /* Get rid of "=" at the end. */
            option[sub_len - 1] = '\0';

            sub_ptr += sub_len;

            ssh_dlex_next(sub_lexer, sub_ptr, sub_end - sub_ptr,
                          &sub_len, &sub_token);

            SSH_DEBUG(6, ("Got token '%s' (token = %d).", sub_ptr, sub_token));
            SSH_ASSERT(sub_token == 1);
            value = ssh_xstrdup(sub_ptr);
            value[sub_len] = '\0';

            sub_ptr += sub_len;

            ssh_xfree(to_be_deleted);
            ssh_dlex_destroy(sub_lexer);

            if (value[0] == '\"')
              {
                /* Strip first and last char (they are '"'
                   characters). */
                to_be_deleted = value;
                value = ssh_xstrdup(value + 1);
                value[strlen(value) - 1] = '\0';
                ssh_xfree(to_be_deleted);
              }

            SSH_DEBUG(6, ("option = '%s', value = '%s'", option, value));

            if (strlen(value) == 0)
              {
                SSH_TRACE(2, ("Public key option '%s' has zero-length "
                              "value."));
                ret = TRUE;
                goto invalid_value;
              }

            if (strcmp(option, "allow-from") == 0)
              {
                if (pubkey_options->use_allowed_from == NULL)
                  {
                    pubkey_options->use_allowed_from =
                      ssh_adt_create_generic(SSH_ADT_LIST,
                                             SSH_ADT_DESTROY,
                                             ssh_config_pattern_destructor,
                                             SSH_ADT_ARGS_END);
                  }
                SSH_VERIFY(ssh_adt_insert(pubkey_options->use_allowed_from,
                                          ssh_config_pattern_alloc
                                          (value, metaconfig)) !=
                           SSH_ADT_INVALID);
                value = NULL;
              }
            else if (strcmp(option, "deny-from") == 0)
              {
                if (pubkey_options->use_denied_from == NULL)
                  {
                    pubkey_options->use_denied_from =
                      ssh_adt_create_generic(SSH_ADT_LIST,
                                             SSH_ADT_DESTROY,
                                             ssh_config_pattern_destructor,
                                             SSH_ADT_ARGS_END);
                  }
                SSH_VERIFY(ssh_adt_insert(pubkey_options->use_denied_from,
                                          ssh_config_pattern_alloc
                                          (value, metaconfig)) !=
                           SSH_ADT_INVALID);
                value = NULL;
              }
            else if (strcmp(option, "command") == 0)
              {
                ssh_xfree(common->forced_command);
                common->forced_command = value;
                value = NULL;
              }
            else if (strcmp(option, "environment") == 0)
              {
                if (pubkey_options->environment_vars == NULL)
                  {
                    pubkey_options->environment_vars =
                      ssh_adt_create_generic(SSH_ADT_LIST,
                                             SSH_ADT_DESTROY,
                                             ssh_adt_callback_destroy_free,
                                             SSH_ADT_ARGS_END);
                    SSH_VERIFY(pubkey_options->environment_vars != NULL);
                  }
                SSH_VERIFY(ssh_adt_insert(pubkey_options->environment_vars,
                                          value) != SSH_ADT_INVALID);
                value = NULL;
              }
            else if (strcmp(option, "idle-timeout") == 0)
              {
                SshInt32 num;

                if ((num = ssh_config_parse_timeout(value)) < 0)
                  {
                    SSH_DEBUG(3, ("Invalid idle-timeout value in "
                                  "public-key options (value = '%s')", value));
                    ret = TRUE;
                  }
                else
                  {
                    pubkey_options->idle_timeout = num;
                  }
              }
            else
              {
                ret = TRUE;
              }
          invalid_value:
            ssh_xfree(option);
            ssh_xfree(value);
          }
          break;
        case 1:
          /* Boolean values. */
          if (strncmp(ptr, "no-port-forwarding", len) == 0)
            pubkey_options->no_port_forwarding = TRUE;
          else if (strncmp(ptr, "no-x11-forwarding", len) == 0)
            pubkey_options->no_x11_forwarding = TRUE;
          else if (strncmp(ptr, "no-agent-forwarding", len) == 0)
            pubkey_options->no_agent_forwarding = TRUE;
          else if (strncmp(ptr, "no-pty", len) == 0)
            pubkey_options->no_pty = TRUE;
          else
            {
              SSH_TRACE(2, ("Unknown public key option '%.*s'.", len, ptr));
              ret = TRUE;
            }
          break;
        case 2:
          /* Separator. */
          break;
        default:
          /* Cannot happen. */
          SSH_NOTREACHED;
        }
      ptr += len;
    }

  if (ptr < end)
    {
      SSH_DEBUG(2, ("Garbage at the end of the string: `%s'.\n", ptr));
      ret = TRUE;
    }

  ssh_dlex_destroy(lexer);
  return ret;
}

/* Check whether the key is authorized for login as the specified user
   from specified host.  If check_signatures if FALSE, this is not
   required to verify signatures on authorization certificates.  This
   may use a local database or the certificates to determine
   authorization. */
Boolean ssh_server_auth_pubkey_verify(SshUser uc, char *remote_ip,
                                      unsigned char *certs,
                                      size_t certs_len,
                                      unsigned char *certs_type,
                                      unsigned char *sig,
                                      size_t sig_len,
                                      const unsigned char *session_id,
                                      size_t session_id_len,
                                      SshServer server,
                                      Boolean check_signatures)
{
  unsigned char *tblob;
  size_t tbloblen;
  SshPublicKey pubkey;
  char *pubkeytype = NULL;
  Boolean sig_ok;
  SshBuffer buf;
  char filen[1024];
  int i, n;
  unsigned int pk_size = 0;
  char *service;
  SshEncodingFormat format_session_id;
  unsigned long magic;
  char *userdir, **vars = NULL, **vals = NULL;
  SshMetaConfig metaconfig = NULL;
#ifdef WITH_PGP
  char *pgp_public_key_file;
#endif /* WITH_PGP */

#ifdef HAVE_SIGNAL
  void (*old_sigchld_handler)(int);
  old_sigchld_handler = signal(SIGCHLD, SIG_DFL);
#endif /* HAVE_SIGNAL */


  SSH_PRECOND(server != NULL);
  SSH_PRECOND(server->common != NULL);
  SSH_PRECOND(server->common->config != NULL);

#ifdef WITH_PGP
  pgp_public_key_file =
    ssh_xstrdup(server->common->config->pgp_public_key_file);
#endif /* WITH_PGP */

  /* The publickey options will be usually parsed twice for the key
     used in the authentication, because we must check whether
     {allow,deny}-from attributes in options match the host the user
     is logging in from. This way it is more to the spirit of the
     spec, as we won't return SSH_MSG_USERAUTH_PK_OK for a key whose
     usage is not allowed by publickey options. */
#define SIGCHECK                                                           \
  do {                                                                     \
    if (tbloblen == certs_len && memcmp(certs, tblob, tbloblen) == 0)      \
      {                                                                    \
        if (i + 1 < n)                                                     \
          {                                                                \
            if (strcasecmp(vars[i + 1], "command") == 0)                   \
              {                                                            \
                /* Backwards compatibility with old "command"              \
                   keyword. */                                             \
                if (strlen(vals[i + 1]) > 2 &&                             \
                    vals[i + 1][0] == '\"' &&                              \
                    vals[i + 1][strlen(vals[i + 1]) - 1] == '\"')          \
                  {                                                        \
                    /* Remove quotes, if any. */                           \
                    char *new_val = ssh_xstrdup(vals[i + 1] + 1);          \
                    new_val[strlen(new_val) - 1] = '\0';                   \
                    ssh_xfree(vals[i + 1]);                                \
                    vals[i + 1] = new_val;                                 \
                  }                                                        \
                ssh_xfree(server->common->forced_command);                 \
                server->common->forced_command = ssh_xstrdup(vals[i + 1]); \
              }                                                            \
            else if (strcasecmp(vars[i + 1], "options") == 0)              \
              {                                                            \
                ssh_publickey_options_free(server->common->                \
                                           publickey_options);             \
                server->common->publickey_options =                        \
                  ssh_xcalloc(1, sizeof(*server->common->                  \
                                        publickey_options));               \
                if (ssh_parse_publickey_options                            \
                    (server->common, server->common->publickey_options,    \
                     metaconfig, vals[i + 1]))                             \
                  ssh_warning("User %s has malformed options with %s=%s.", \
                              ssh_user_name(uc), vars[i], vals[i]);        \
              }                                                            \
          }                                                                \
        ssh_xfree(tblob);                                                  \
        goto match;                                                        \
      }                                                                    \
    ssh_xfree(tblob);                                                      \
  } while(0)

  ssh_userfile_init(ssh_user_name(uc), ssh_user_uid(uc), ssh_user_gid(uc),
                    NULL_FNPTR, NULL);

  SSH_DEBUG(6, ("auth_pubkey_verify user = %s  check_sig = %s  type = %s",
                ssh_user_name(uc), check_signatures ? "yes" : "no",
                certs_type));

  /* open and read the user's authorization file */

  if (certs_len < 16)  /* ever seen a 12-byte public key ? */
    goto exit_false;

  if ((userdir = ssh_userdir(uc, server->config, FALSE)) == NULL)
    goto exit_false;

  if (server->common->config->authorization_file &&
      *server->common->config->authorization_file == '/')
    {
      /* Filename is absolute. */
      ssh_snprintf(filen, sizeof(filen), "%s",
                   server->common->config->authorization_file);
    }
  else
    {
      ssh_snprintf(filen, sizeof(filen), "%s/%s", userdir,
                   server->common->config->authorization_file == NULL ?
                   SSH_AUTHORIZATION_FILE :
                   server->common->config->authorization_file);
    }
  
  metaconfig = ssh_xcalloc(1, sizeof(*metaconfig));
  n = ssh2_parse_config(uc, remote_ip ? remote_ip : "",
                        filen, &vars, &vals, metaconfig, FALSE);

  /* now see if we find matching "key" - definitions */

  for (i = 0; i < n; i++)
    {
      if (strcasecmp(vars[i], "key") == 0)
        {
          if (*vals[i] == '/')
            ssh_snprintf(filen, sizeof(filen), "%s", vals[i]);
          else
            ssh_snprintf(filen, sizeof(filen), "%s/%s", userdir, vals[i]);

          SSH_DEBUG(6, ("key %d, %s", i, filen));
          tblob = NULL;
          magic = ssh2_key_blob_read(uc, filen, TRUE, NULL,
                                     &tblob, &tbloblen, NULL);
          if (magic == SSH_KEY_MAGIC_PUBLIC)
            {
#ifdef DEBUG_HEAVY
              SSH_DEBUG_HEXDUMP(12,
                                ("certs from file (len=%d):", tbloblen),
                                tblob, tbloblen);
              SSH_DEBUG_HEXDUMP(12,
                                ("certs from remote (len=%d):", certs_len),
                                certs, certs_len);
#endif /* DEBUG_HEAVY */
              SIGCHECK;
            }
          else
            {
              if (tblob != NULL)
                ssh_xfree(tblob);
              SSH_DEBUG(2, ("unable to read the %s's public key %s", \
                            ssh_user_name(uc), filen));
            }
        }
#ifdef WITH_PGP
      else if (strcasecmp(vars[i], "pgppublickeyfile") == 0)
        {
          SSH_DEBUG(6, ("pgppublickeyfile = %s", vals[i]));
          ssh_xfree(pgp_public_key_file);
          pgp_public_key_file = ssh_xstrdup(vals[i]);
        }
      else if (strcasecmp(vars[i], "pgpkeyid") == 0)
        {
          unsigned long id;
          char *endptr = NULL;

          id = strtoul(vals[i], &endptr, 0);
          if (((*(vals[0])) != '\0') && ((*endptr) == '\0'))
            {
              if (*pgp_public_key_file == '/')
                ssh_snprintf(filen, sizeof(filen), "%s", pgp_public_key_file);
              else
                ssh_snprintf(filen, sizeof(filen), "%s/%s",
                             userdir, pgp_public_key_file);

              SSH_DEBUG(6, ("pgpkey id=0x%lx, %s", (unsigned long)id, filen));
              if (ssh2_find_pgp_public_key_with_id
                  (uc, filen, (SshUInt32)id, &tblob, &tbloblen, NULL))
                {
                  SIGCHECK;
                }
              else
                {
                  SSH_DEBUG(2, ("unable to read the %s's key id "
                                "0x%08lx from keyring %s",
                                ssh_user_name(uc), (unsigned long)id,
                                filen));
                }
            }
          else
            {
              SSH_DEBUG(2, ("invalid pgp key id number \"%s\"", vals[i]));
            }
        }
      else if (strcasecmp(vars[i], "pgpkeyname") == 0)
        {
          if (*pgp_public_key_file == '/')
            ssh_snprintf(filen, sizeof(filen), "%s", pgp_public_key_file);
          else
            ssh_snprintf(filen, sizeof(filen), "%s/%s",
                         userdir, pgp_public_key_file);

          SSH_DEBUG(6, ("pgpkey name=\"%s\", %s", vals[i], filen));
          if (ssh2_find_pgp_public_key_with_name(uc,
                                                 filen,
                                                 vals[i],
                                                 &tblob,
                                                 &tbloblen,
                                                 NULL))
            {
              SIGCHECK;
            }
          else
            {
              SSH_DEBUG(2, ("unable to read the %s's key name "
                            "\"%s\" from keyring %s",
                            ssh_user_name(uc), vals[i], filen));
            }
        }
      else if (strcasecmp(vars[i], "pgpkeyfingerprint") == 0)
        {
          if (*pgp_public_key_file == '/')
            ssh_snprintf(filen, sizeof(filen), "%s", pgp_public_key_file);
          else
            ssh_snprintf(filen, sizeof(filen), "%s/%s",
                         userdir, pgp_public_key_file);
          
          SSH_DEBUG(6, ("pgpkey fingerprint=\"%s\", %s", vals[i], filen));
          if (ssh2_find_pgp_public_key_with_fingerprint(uc,
                                                        filen,
                                                        vals[i],
                                                        &tblob,
                                                        &tbloblen,
                                                        NULL))
            {
              SIGCHECK;
            }
          else
            {
              SSH_DEBUG(2, ("unable to read the %s's key name "
                            "\"%s\" from keyring %s",
                            ssh_user_name(uc), vals[i], filen));
            }
        }
#endif /* WITH_PGP */
#if 0
      else
        {
          /* XXX Put message to log here about malformed authorization
             file keyword. */
        }
#endif /* 0 */
    }

  SSH_DEBUG(6, ("auth_pubkey_verify: the key didn't match."));

  ssh_xfree(userdir);

  ssh_free_varsvals(n, vars, vals);

  goto exit_false;

  /* ok, this public key can be used for authentication .. */
 match:

  SSH_DEBUG(6, ("auth_pubkey_verify: the key matched."));

  ssh_xfree(userdir);

  ssh_free_varsvals(n, vars, vals);

  /* Check whether key usage is ok from remote host. */
  if (server->common->publickey_options != NULL &&
      ssh_server_auth_check_host_generic
      (server->common->remote_host, server->common->remote_ip,
       server->common->publickey_options->use_denied_from,
       server->common->publickey_options->use_allowed_from,
       server->common->config->require_reverse_mapping))
    {
      ssh_log_event(server->config->log_facility,
                    SSH_LOG_NOTICE,
                    "Use of public key %s is not allowed in public key "
                    "options for the host the user %s is logging in from.",
                    filen, ssh_user_name(uc));
      goto exit_false;
    }

  /* decode the public key blob */
  if ((pubkey = ssh_decode_pubkeyblob(certs, tbloblen)) == NULL)
    goto exit_false;

  if (ssh_public_key_get_info(pubkey,
                              SSH_PKF_SIZE, &pk_size,
                              SSH_PKF_END) != SSH_CRYPTO_OK)
    {
      ssh_public_key_free(pubkey);
      ssh_log_event(server->config->log_facility,
                    SSH_LOG_WARNING,
                    "Couldn't get info from public key %s used by user %s.",
                    filen, ssh_user_name(uc));
      goto exit_false;
    }

  SSH_TRACE(2, ("Public key %s, size %d.", filen, pk_size));
  if (server->config->auth_publickey_min_size > 0 &&
      pk_size < server->config->auth_publickey_min_size)
    {
      ssh_log_event(server->config->log_facility,
                    SSH_LOG_WARNING,
                    "Use of public key %s by user %s is not allowed, because "
                    "public key is too small (pk size %d, min size %d.",
                    filen, ssh_user_name(uc), pk_size,
                    server->config->auth_publickey_min_size);
      ssh_public_key_free(pubkey);
      goto exit_false;
    }
  
  if (server->config->auth_publickey_max_size > 0 &&
      pk_size > server->config->auth_publickey_max_size)
    {
      ssh_log_event(server->config->log_facility,
                    SSH_LOG_WARNING,
                    "Use of public key %s by user %s is not allowed, because "
                    "public key is too large (pk size %d, max size %d.",
                    filen, ssh_user_name(uc), pk_size,
                    server->config->auth_publickey_max_size);
      ssh_public_key_free(pubkey);
      goto exit_false;
    }

  if (!check_signatures)
    goto exit_true;

  /* construct a throw-away SSH_MSG_USERAUTH_REQUEST message */

  buf = ssh_xbuffer_allocate();

  if (




      !(*server->common->compat_flags->
        publickey_service_name_draft_incompatibility)

      )
    service = SSH_CONNECTION_SERVICE;
  else
    service = SSH_USERAUTH_SERVICE;

  if (




      !(*server->common->compat_flags->
        publickey_session_id_encoding_draft_incompatibility)

      )
    format_session_id = SSH_FORMAT_UINT32_STR;
  else
    format_session_id = SSH_FORMAT_DATA;

  /* get the public key type. */
  pubkeytype = ssh_pubkeyblob_type(certs, tbloblen);

  if (



      !(*server->common->compat_flags->publickey_draft_incompatibility)

      )
    {

      if (strcmp(pubkeytype, (char *)certs_type))
        {
          SSH_TRACE(2, ("public key was of different type from what the "
                        "client said. (we got: '%s', client gave us: '%s')",
                        pubkeytype, certs_type));
          goto exit_false;
        }

      ssh_encode_buffer(buf,
                        format_session_id, session_id, session_id_len,
                        SSH_FORMAT_CHAR,
                        (unsigned int) SSH_MSG_USERAUTH_REQUEST,
                        SSH_FORMAT_UINT32_STR, ssh_user_name(uc),
                        strlen(ssh_user_name(uc)),
                        SSH_FORMAT_UINT32_STR, service,
                        strlen(service),
                        SSH_FORMAT_UINT32_STR, SSH_AUTH_PUBKEY,
                        strlen(SSH_AUTH_PUBKEY),
                        SSH_FORMAT_BOOLEAN, TRUE,
                        SSH_FORMAT_UINT32_STR, certs_type,
                        strlen((char *)certs_type),
                        SSH_FORMAT_UINT32_STR, certs, tbloblen,
                        SSH_FORMAT_END);
    }
  else
    {
      /* Remote end has publickey draft incompatibility bug. */
      ssh_encode_buffer(buf,
                        format_session_id, session_id, session_id_len,
                        SSH_FORMAT_CHAR,
                        (unsigned int) SSH_MSG_USERAUTH_REQUEST,
                        SSH_FORMAT_UINT32_STR, ssh_user_name(uc),
                        strlen(ssh_user_name(uc)),
                        SSH_FORMAT_UINT32_STR, service,
                        strlen(service),
                        /* against the draft. Here should be 'string
                           "publickey"'*/
                        SSH_FORMAT_BOOLEAN, TRUE,
                        /* against the draft. Here should be 'string
                           public key algorith name'*/
                        SSH_FORMAT_UINT32_STR, certs, tbloblen,
                        SSH_FORMAT_END);
    }

#ifdef DEBUG_HEAVY
  SSH_DEBUG_HEXDUMP(11, ("auth_pubkey_verify: verifying following data"),
                    ssh_buffer_ptr(buf), ssh_buffer_len(buf));
  SSH_DEBUG_HEXDUMP(11, ("auth_pubkey_verify: signature"), sig, sig_len);
#endif /* DEBUG_HEAVY */

  /* verify the signature */
  {
    unsigned char *real_sig;
    char *recv_certs_type;
    size_t real_sig_len, decoded_len;

    if (




        !*(server->common->compat_flags->
           malformed_signatures_draft_incompatibility)

        )
      {
        decoded_len = ssh_decode_array(sig, sig_len,
                                       SSH_FORMAT_UINT32_STR,
                                       &recv_certs_type, NULL,
                                       SSH_FORMAT_UINT32_STR,
                                       &real_sig, &real_sig_len,
                                       SSH_FORMAT_END);

        if (decoded_len == 0 || decoded_len != sig_len)
          {
            SSH_DEBUG(2, ("decoded_len: %ld, sig_len: %ld",
                          decoded_len, sig_len));
            ssh_warning("Received malformed signature during public key "
                        "authentication.");
            goto exit_false;
          }

        if (strcmp(recv_certs_type, (char *)certs_type) != 0)
          {
            ssh_warning("Received malformed signature during public key "
                        "authentication. (public key type doesn't match the "
                        "one in signature.)");
            ssh_xfree(recv_certs_type);
            memset(real_sig, 'F', real_sig_len);
            ssh_xfree(real_sig);
            goto exit_false;
          }
        ssh_xfree(recv_certs_type);
      }
    else
      {
        real_sig = sig;
        real_sig_len = sig_len;
        sig = NULL;
      }

    if (!strncmp(pubkeytype, SSH_SSH_RSA, strlen(SSH_SSH_RSA)) &&
        !ssh_compat_rsa_public_key_change_scheme
        (pubkey,




         !*(server->common->compat_flags->
            rsa_hash_scheme_draft_incompatibility)

         ))
      {
        ssh_public_key_free(pubkey);
        ssh_buffer_free(buf);
        ssh_log_event(server->config->log_facility,
                      SSH_LOG_WARNING,
                      "Crypto operation failed for public key \"%s\" "
                      "(for user %s).",
                      filen, ssh_user_name(uc));
        goto exit_false;
      }

    sig_ok = ssh_public_key_verify_signature(pubkey,
                                             real_sig, real_sig_len,
                                             ssh_buffer_ptr(buf),
                                             ssh_buffer_len(buf));




















    if (




        !*(server->common->compat_flags->
           malformed_signatures_draft_incompatibility)

        )
      {
        memset(real_sig, 'F', real_sig_len);
        ssh_xfree(real_sig);
      }
  }

  ssh_public_key_free(pubkey);
  ssh_buffer_free(buf);

  if (!sig_ok)
    {
      ssh_warning("Public key operation failed for %s.", ssh_user_name(uc));
      goto exit_false;
    }

 exit_true:

#ifdef DEBUG_LIGHT
  ssh_dump_publickey_options(server->common);
#endif /* DEBUG_LIGHT */
  ssh_xfree(pubkeytype);
  if (check_signatures)
    ssh_log_event(server->config->log_facility, SSH_LOG_NOTICE,
                  "Public key %s used.", filen);
#if 0
  /* XXX Don't use config. This should also apply to certificates. */
  config->cert_used_in_pubkey_auth = ssh_xstrdup(filen);
#endif
#ifdef WITH_PGP
  ssh_xfree(pgp_public_key_file);
#endif /* WITH_PGP */
  ssh_userfile_uninit();
  ssh_xfree(metaconfig);
#ifdef HAVE_SIGNAL
  signal(SIGCHLD, old_sigchld_handler);
#endif /* HAVE_SIGNAL */
  return TRUE;

 exit_false:

  ssh_xfree(pubkeytype);
#ifdef WITH_PGP
  ssh_xfree(pgp_public_key_file);
#endif /* WITH_PGP */

  ssh_userfile_uninit();
#ifdef HAVE_SIGNAL
  signal(SIGCHLD, old_sigchld_handler);
#endif /* HAVE_SIGNAL */

  /* Free publickey options, so we don't accidentally use a failed
     keys options. */
  ssh_publickey_options_free(server->common->publickey_options);
  server->common->publickey_options = NULL;
  ssh_xfree(metaconfig);
  return FALSE;
}


/* Public key authentication.  The possession of a private key serves
   as authentication. */

void ssh_server_auth_pubkey(SshAuthServerOperation op,
                            const char *user,
                            SshUser uc,
                            SshBuffer packet,
                            const unsigned char *session_id,
                            size_t session_id_len,
                            void **state_placeholder,
                            void **longtime_placeholder,
                            SshAuthServerCompletionProc completion_proc,
                            void *completion_context,
                            void *method_context)
{
  unsigned char *certs, *data, *sig, *certs_type = NULL;
  size_t certs_len, sig_len, len, bytes;
  SshServer server = (SshServer) method_context;
  SshConfig config = server->config;
  Boolean real_request;

  SSH_DEBUG(6, ("auth_pubkey op = %d  user = %s", op, user));

  switch (op)
    {
    case SSH_AUTH_SERVER_OP_START:
      if (ssh_server_auth_check(uc, user, config, server->common,
                                SSH_AUTH_PUBKEY))
        {
          (*completion_proc) (SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
                              packet,
                              completion_context);
          return;
        }


      /* Parse the publickey authentication request. */

      /* XXX should check permissions on user's home directory, user's
         .ssh2 directory, user's authorization file, user's publickey
         file. (is strictmodes is enabled) */
      data = ssh_buffer_ptr(packet);
      len = ssh_buffer_len(packet);
      sig = NULL;
      sig_len = 0;

      if (



          !(*server->common->compat_flags->publickey_draft_incompatibility)

          )
        {
          bytes = ssh_decode_array(data, len,
                                   SSH_FORMAT_BOOLEAN, &real_request,
                                   SSH_FORMAT_UINT32_STR,
                                   &certs_type, NULL,
                                   SSH_FORMAT_UINT32_STR_NOCOPY,
                                   &certs, &certs_len,
                                   SSH_FORMAT_END);
        }
      else
        {
          bytes = ssh_decode_array(data, len,
                                   SSH_FORMAT_BOOLEAN, &real_request,
                                   /* against the draft. Here should be 'string
                                      public key algorith name'*/
                                   SSH_FORMAT_UINT32_STR_NOCOPY,
                                   &certs, &certs_len,
                                   SSH_FORMAT_END);
          if (bytes > 0)
            certs_type = (unsigned char *)ssh_pubkeyblob_type(certs,
                                                              certs_len);
        }

      if ((bytes == 0) ||
          ((! real_request) && (bytes != len)) ||
          (certs_type == NULL))
        {
          ssh_log_event(config->log_facility,
                        SSH_LOG_WARNING,
                        "got bad packet when verifying user %s's publickey.",
                        ssh_user_name(uc));
          SSH_DEBUG(2, ("bad packet"));
          (*completion_proc) (SSH_AUTH_SERVER_REJECTED, packet,
                              completion_context);
          return;
        }

      if (real_request)
        {
          if (ssh_decode_array(data + bytes, len - bytes,
                               SSH_FORMAT_UINT32_STR, &sig, &sig_len,
                               SSH_FORMAT_END) != len - bytes ||
              len - bytes <= 0)
            {
              ssh_log_event(config->log_facility,
                            SSH_LOG_WARNING,
                            "got bad packet when verifying user "
                            "%s's publickey.",
                            ssh_user_name(uc));
              SSH_DEBUG(2, ("bad packet (real request)"));
              (*completion_proc) (SSH_AUTH_SERVER_REJECTED, packet,
                                  completion_context);
              return;
            }
        }

      SSH_DEBUG(2, ("Public key algorithm is %s", certs_type));
















      /* Check whether the key is authorized for login as the specified
         user.  If real_request if FALSE, this does not need to verify
         signatures on certificates as the result is only advisory. */
      if (ssh_server_auth_pubkey_verify(uc, server->common->remote_ip,
                                        certs, certs_len,
                                        certs_type,
                                        sig, sig_len,
                                        session_id, session_id_len,
                                        server,
                                        real_request)
          == FALSE)
        {
          if (sig != NULL)
            ssh_xfree(sig);
          ssh_xfree(certs_type);
          SSH_DEBUG(6, ("auth_pubkey_verify returned FALSE"));
          (*completion_proc) (SSH_AUTH_SERVER_REJECTED, packet,
                              completion_context);
          return;
        }

      if (real_request)
        {
          /* Free the signature blob. */
          ssh_xfree(sig);
          ssh_xfree(certs_type);

          /* Publickey options. */
          if (server->common->publickey_options != NULL)
            {
              /* Set idle_timeout, if it >0 and smaller than server's
                 own configuration. */
              if ((server->common->config->idle_timeout == 0 ||
                   server->common->config->idle_timeout >
                   server->common->publickey_options->idle_timeout ) &&
                  server->common->publickey_options->idle_timeout > 0)
                server->common->config->idle_timeout =
                  server->common->publickey_options->idle_timeout;

              /* Set whether user can forward ports. */
              if (server->common->publickey_options->no_port_forwarding)
                server->common->config->allow_tcp_forwarding = FALSE;
              /* Set whether user can forward x11. */
              if (server->common->publickey_options->no_x11_forwarding)
                server->common->config->x11_forwarding = FALSE;
              /* Set whether user can forward agent. */
              if (server->common->publickey_options->no_agent_forwarding)
                server->common->config->agent_forwarding = FALSE;
            }

          /* Check for root login and forced commands */
          if(ssh_user_uid(uc) == UID_ROOT &&
             config->permit_root_login == SSH_ROOTLOGIN_FALSE)
            {
              if(!server->common->forced_command)
                {
                  ssh_log_event(config->log_facility,
                                SSH_LOG_NOTICE,
                                "root logins are not permitted.");
                  SSH_TRACE(2, ("root logins are not permitted."));
                  (*completion_proc)
                    (SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED, packet,
                     completion_context);
                  return;
                }
              else
                {
                  ssh_log_event(config->log_facility,
                                SSH_LOG_NOTICE,
                                "root login permitted for forced command.");
                }
            }

          /* Because it is an real request, and it has been verified, the
             authorization is granted. */
          ssh_log_event(config->log_facility,
                        SSH_LOG_NOTICE,
                        "Public key authentication for user %s accepted.",
                        ssh_user_name(uc));

#ifdef TEST_ASYNC_SERVER_METHODS
          {
            SshPublicKeyAuthCtx finalize_ctx;

            finalize_ctx = ssh_xcalloc(1, sizeof(*finalize_ctx));
            finalize_ctx->packet = packet;
            finalize_ctx->result = SSH_AUTH_SERVER_ACCEPTED;
            finalize_ctx->completion_proc = completion_proc;
            finalize_ctx->completion_context = completion_context;

            SSH_TRACE(2, ("Delaying success message for two seconds..."));
            ssh_register_timeout(2L, 0L,
                                 publickey_auth_finalize_callback,
                                 finalize_ctx);
          }
#else /* TEST_ASYNC_SERVER_METHODS */
          (*completion_proc)(SSH_AUTH_SERVER_ACCEPTED, packet,
                             completion_context);
#endif /* TEST_ASYNC_SERVER_METHODS */
          return;
        }

      /* It was just a probe request, return status now. */

      ssh_buffer_clear(packet);
      if (




          !(*server->common->compat_flags->
            publickey_pk_ok_draft_incompatibility)

          )
        ssh_encode_buffer(packet,
                          SSH_FORMAT_CHAR,
                          (unsigned int) SSH_MSG_USERAUTH_PK_OK,
                          SSH_FORMAT_UINT32_STR, certs_type,
                          strlen((char *)certs_type),
                          SSH_FORMAT_UINT32_STR, certs, certs_len,
                          SSH_FORMAT_END);

      else
        ssh_encode_buffer(packet,
                          SSH_FORMAT_CHAR,
                          (unsigned int) SSH_MSG_USERAUTH_PK_OK,
                          SSH_FORMAT_UINT32_STR, certs, certs_len,
                          SSH_FORMAT_END);

      ssh_xfree(certs_type);

      (*completion_proc)(SSH_AUTH_SERVER_REJECTED_WITH_PACKET_BACK, packet,
                         completion_context);
      return;

    case SSH_AUTH_SERVER_OP_ABORT:
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet,
                         completion_context);
      return;

    case SSH_AUTH_SERVER_OP_CONTINUE:
      SSH_DEBUG(2, ("ssh_server_auth_pubkey: unexpected CONTINUE"));
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet,
                         completion_context);
      return;

    case SSH_AUTH_SERVER_OP_CONTINUE_SPECIAL:
      SSH_DEBUG(2, ("ssh_server_auth_pubkey: unexpected CONTINUE_SPECIAL"));
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet,
                         completion_context);
      return;

    case SSH_AUTH_SERVER_OP_UNDO_LONGTIME:
    case SSH_AUTH_SERVER_OP_CLEAR_LONGTIME:
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet,
                         completion_context);
      return;

    default:
      ssh_fatal("ssh_server_auth_pubkey: unknown op %d", (int)op);
    }

  SSH_NOTREACHED;
}
