/*

auths-kerberos-tgt.c

Author: Tatu Ylonen <ylo@ssh.com>

Copyright (C) 2000 SSH Communications Security Corp, Helsinki, Finland
All rights reserved.

Kerberos authentication with TGT (passed to server), server side.

Special acknowledgements to Dug Song <dogsong@umic.edu> and Glenn
Machin (Sandia Natl Labs) for writing the original Kerberos support
code for SSH1.

*/

#include "sshincludes.h"
#include "sshencode.h"
#include "sshauth.h"
#include "sshmsgs.h"
#include "sshuser.h"
#include "sshserver.h"
#include "sshconfig.h"
#include "auths-common.h"
#include "auths-kerberos-tgt.h"

#ifdef KERBEROS

#undef ctime
#undef free

/* The #%@!$(#$ krb5.h header redefines these.  ARRRGH! -ylo */
#undef SIZEOF_INT
#undef SIZEOF_LONG
#undef SIZEOF_SHORT
#undef HAVE_STDARG_H
#undef HAVE_SYS_TYPES_H
#include <krb5.h>

#define SSH_DEBUG_MODULE "Ssh2AuthKerberosTgtServer"

/* Kerberos authentication. */

void ssh_server_auth_kerberos_tgt(SshAuthServerOperation op,
                                  const char *user,
                                  SshUser uc,
                                  SshBuffer packet,
                                  const unsigned char *ses_id,
                                  size_t session_id_len,
                                  void **state_placeholder,
                                  void **longtime_placeholder,
                                  SshAuthServerCompletionProc completion_proc,
                                  void *completion_context,
                                  void *method_context)
{
  SshServer server = (SshServer)method_context;
  SshConfig config = server->config;
  unsigned char *ticket = NULL;
  unsigned char *tgt = NULL;
  size_t ticket_len;
  size_t tgt_len;
  krb5_error_code kerr;
  krb5_context kcontext;
  krb5_auth_context kauth;
  krb5_data auth_data, unenc_tgt;
  krb5_principal sshd_princ;
  krb5_rcache rcache;
  krb5_creds **creds, *local_tgt_creds;
  krb5_ticket *kticket;
  char *server_name, *tkt_client_user;
  int ret = SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED;
  
  SSH_TRACE(6, ("kerberos TGT auth."));
  SSH_DEBUG(6, ("op = %d  user = %s", op, user));

  /* Initialize kerberos data that we use in cleanup. */
  memset(&kcontext, 0, sizeof(kcontext));
  memset(&kauth, 0, sizeof(kauth));
  memset(&auth_data, 0, sizeof(auth_data));
  memset(&unenc_tgt, 0, sizeof(unenc_tgt));
  creds = NULL;
  server_name = NULL;
  tkt_client_user = NULL;
  kticket = NULL;
  sshd_princ = NULL;
  local_tgt_creds = NULL;
  
  switch (op)
    {
    case SSH_AUTH_SERVER_OP_START:
      /* Check that the user is allowed to log in and that this authentication
         method is allowed. */
      if (ssh_server_auth_check(uc, user, config, server->common,
                                SSH_AUTH_KERBEROS))
        goto fail;
      
      /* Parse the password authentication request. */
      if (ssh_decode_buffer(packet,
                            SSH_FORMAT_UINT32_STR, &ticket, &ticket_len,
                            SSH_FORMAT_UINT32_STR, &tgt, &tgt_len,
                            SSH_FORMAT_END) == 0)
        {
          SSH_DEBUG(2, ("bad packet"));
          goto fail;
        }
      
      /* Sanity check: do not pass excessively long tickets, just in case. */
      if (ticket_len > 2048)
        {
          SSH_DEBUG(2, ("ticket too long"));
          ssh_xfree(ticket);
          goto fail;
        }

      /* Initialize the Kerberos application context. */
      kerr = krb5_init_context(&kcontext);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_init_context %d: %s", kerr,
                        error_message(kerr)));
          goto fail;
        }

      /* Initialize the authentication context. */
      kerr = krb5_auth_con_init(kcontext, &kauth);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_auth_con_init %d: %s ", kerr,
                        error_message(kerr)));
          goto fail;
        }

      /* Create principal for "sshd". */
      kerr = krb5_parse_name(kcontext, "sshd", &sshd_princ);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_parse_name (sshd) %d: %s", kerr,
                        error_message(kerr)));
          goto fail;
        }

      /* Allocate an rcache. */
      kerr = krb5_get_server_rcache(kcontext,
                                    krb5_princ_component(kcontext,
                                                         sshd_princ, 0),
                                    &rcache);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_get_server_rcache %d: %s", kerr,
                        error_message(kerr)));
          goto fail;
        }

      /* Start using the rcache for this authentication context. */
      kerr = krb5_auth_con_setrcache(kcontext, kauth, rcache);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_auth_con_setrcache %d: %s", kerr,
                        error_message(kerr)));
          goto fail;
        }      

      /* Prepare the authentication data from the ticket. */
      auth_data.data = (char *) ticket;
      auth_data.length = ticket_len;

      /* Extract remote and local addresses from the socket used for
         communication. */
      kerr =
        krb5_auth_con_genaddrs(kcontext, kauth, config->ssh1_fd,
                               KRB5_AUTH_CONTEXT_GENERATE_REMOTE_FULL_ADDR |
                               KRB5_AUTH_CONTEXT_GENERATE_LOCAL_FULL_ADDR);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_auth_con_genaddrs %d: %s", kerr,
                        error_message(kerr)));
          goto fail;
        }

      /* Parse the client's authentication request. */
      kerr = krb5_rd_req(kcontext, &kauth, &auth_data, NULL, NULL, NULL,
                         &kticket);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_rd_req %d: %s", kerr, error_message(kerr)));
          goto fail;
        }

      /* Parse the server name from the ticket. */
      kerr = krb5_unparse_name(kcontext, kticket->server, &server_name);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_unparse_name %d: %s", kerr,
                        error_message(kerr)));
          goto fail;
        }
      
      /* Make sure the server principal name begins with "host/". */
      if (strncmp(server_name, "host/", 5) != 0)
        {
          SSH_DEBUG(2, ("server principal is not host"));
          goto fail;
        }

      /* Dig out the client user's name from the ticket. */
      kerr = krb5_unparse_name(kcontext, kticket->enc_part2->client,
                               &tkt_client_user);
      if (kerr)
        {
          SSH_DEBUG(2, ("could not unparse client user's name from ticket"));
          goto fail;
        }
      
      /* Check if the client user is allowed to log into the server as the
         specified user. */
      if (!krb5_kuserok(kcontext, kticket->enc_part2->client,
                        ssh_user_name(uc)))
        {
          ssh_log_event(config->log_facility, SSH_LOG_NOTICE,
                        "Kerberos authentication as %.100s denied for %.100s",
                        ssh_user_name(uc), tkt_client_user);
          goto fail;
        }
      
      /* Parse the client's authentication request. */
      /* Unencrypt the forwarded tgt */
      auth_data.data = (char *) tgt;
      auth_data.length = tgt_len;

      kerr = krb5_rd_priv(kcontext, kauth, &auth_data,
                               &unenc_tgt, NULL);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_rd_priv %d: %s", kerr, error_message(kerr)));
          goto fail;
        }

      kerr = krb5_rd_cred(kcontext, kauth, &unenc_tgt, &creds, NULL);
      if (kerr || !creds[0])
        {
          SSH_DEBUG(2, ("krb5_rd_req %d", kerr));
          SSH_DEBUG(1, ("Kerberos V5 TGT rejected for user %s: %s",
                        user, error_message(kerr)));
          ssh_log_event(config->log_facility, SSH_LOG_NOTICE,
                        "Kerberos V5 TGT rejected for user %s: %s",
                        user, error_message(kerr));
          goto fail;
        }

      /* OK we have a tgt but it does not contain all the local system network
         addresses, only the one known to the client. If we attempted to use
         this tgt to authenticate to a system which we connect to via a 
         different network interface, we would get an incorrect network
         address failure.  So even though it slows things down let's get a
         new tgt with all the network addresses in them. */

      kerr = auth_kerberos_newtgt( kcontext, creds[0], &local_tgt_creds);
      if (kerr)
        {
          SSH_DEBUG(2, ("auth_kerberos_newtgt %d: %s", kerr,
                        error_message(kerr)));
          goto fail;
        }
      
      /* Free then reassign creds to new local tgt */
      krb5_free_creds(kcontext, creds[0]); 
      creds[0] = local_tgt_creds;

      /* Log the successful login. */
      ssh_log_event(config->log_facility, SSH_LOG_NOTICE,
           "Kerberos TGT authentication as user %.100s accepted for %.100s.",
                    ssh_user_name(uc), tkt_client_user);

      /* Authentication successful. */
      ret = SSH_AUTH_SERVER_ACCEPTED;

      /* Save the credentials so that we can save them to disk after we have
         forked a shell for the user. */
      if (creds[0])
        ssh_user_kerberos_set_creds(uc, creds[0]);

    fail:
      /* Authentication has failed.  Clear any allocated kerberos data, and
         fail. */
      if (ticket)
        ssh_xfree(ticket);
      if (tgt)
        ssh_xfree(tgt);
      if (server_name)
        free(server_name);
      if (tkt_client_user)
        free(tkt_client_user);
      if (creds)
        krb5_free_tgt_creds(kcontext, creds);
      if (kauth)
        krb5_auth_con_free(kcontext, kauth);
      if (kcontext)
        krb5_free_context(kcontext);
      if (unenc_tgt.data)
        free(unenc_tgt.data);
      if (sshd_princ)
        krb5_free_principal(kcontext, sshd_princ);

      /* Fall to next case. */
      (*completion_proc)(ret, packet, completion_context);
      return;

    case SSH_AUTH_SERVER_OP_ABORT:
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet,
                         completion_context);
      return;
      
    case SSH_AUTH_SERVER_OP_CONTINUE:
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet,
                         completion_context);
      return;
      
    case SSH_AUTH_SERVER_OP_CONTINUE_SPECIAL:
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet,
                         completion_context);
      return;

    case SSH_AUTH_SERVER_OP_UNDO_LONGTIME:
    case SSH_AUTH_SERVER_OP_CLEAR_LONGTIME:
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet,
                         completion_context);
      return;
      
    default:
      ssh_fatal("ssh_server_auth_kerberos: unknown op %d", (int)op);
    }
  
  SSH_NOTREACHED;
}

#define flags2options(flags) (flags & KDC_TKT_COMMON_MASK)

int auth_kerberos_newtgt( context, in_tgt_creds, out_new_tgt_creds)
krb5_context context;
krb5_creds *in_tgt_creds, **out_new_tgt_creds;
{
  krb5_error_code retval;
  krb5_creds creds, *tgt;
  krb5_creds *pcreds;
  krb5_flags kdcoptions;
  krb5_address **addrs = 0;

  memset((char *)&creds, 0, sizeof(creds));
  tgt = in_tgt_creds;


  retval = krb5_os_localaddr(context, &addrs);
  if (retval)
    goto errout;

  if ((retval = krb5_copy_principal(context, tgt->client, &creds.client)))
    goto errout;

  if ((retval = krb5_copy_principal(context, tgt->server, &creds.server)))
    goto errout;

  creds.times = tgt->times;
  creds.times.starttime = 0;

  /* If this is not a forwardable tgt then we are done */
  if (!(flags2options(tgt->ticket_flags) & KDC_OPT_FORWARDABLE)){
    retval=KRB5_TKT_NOT_FORWARDABLE;
    goto errout;
  }

  kdcoptions = flags2options(tgt->ticket_flags)|KDC_OPT_FORWARDED;

  if ((retval = krb5_get_cred_via_tkt(context, tgt, kdcoptions,
                                      addrs, &creds, &pcreds)))
    goto errout;

  if (out_new_tgt_creds)
    *out_new_tgt_creds = pcreds;

 errout:
  if (addrs)
    krb5_free_addresses(context, addrs);
  krb5_free_cred_contents(context, &creds);
  return retval;
}

#endif /* KERBEROS */
