/*

auths-kbd-int-submethods.c

  Author: Sami J. Lehtinen <sjl@ssh.com>

  Created: Fri Feb  1 19:53:16 2002.

  Copyright (C) 2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

*/

#include "sshincludes.h"
#ifdef SSH_SERVER_WITH_KEYBOARD_INTERACTIVE
#include "sshconfig.h"
#include "auths-kbd-int-submethods.h"
#include "auths-kbd-int-passwd.h"
#include "auths-kbd-int-plugin.h"



#ifdef DAEMON_WITH_PAM
#include "auths-kbd-int-pam.h"
#endif /* DAEMON_WITH_PAM */
#ifdef SSH_SERVER_WITH_SECURID
#include "auths-kbd-int-securid.h"
#endif /* SSH_SERVER_WITH_SECURID */

#define SSH_DEBUG_MODULE "Ssh2AuthKbdIntSubmethods"

#define AUTH_KBDINT_SUBMETHOD(name, long_name, init, free)      \
  { (name), (long_name), SSH_KBDINT_SUBMETHOD_NOTUSED,          \
      SSH_KBDINT_SUBMETHOD_UNDEFINED, (init), (free), NULL }

AuthKbdIntSubMethodStruct submethods[] = {
  AUTH_KBDINT_SUBMETHOD("password", "Password authentication",
                        submethod_passwd_init, submethod_passwd_free),
  AUTH_KBDINT_SUBMETHOD("plugin", "Plugin authentication",
                        submethod_plugin_init, submethod_plugin_free),
#ifdef DAEMON_WITH_PAM
  AUTH_KBDINT_SUBMETHOD("pam", "PAM authentication",
                        submethod_pam_init, submethod_pam_free),
#endif /* DAEMON_WITH_PAM */
#ifdef SSH_SERVER_WITH_SECURID
  AUTH_KBDINT_SUBMETHOD("securid", "SecurID authentication",
                        submethod_ace_init, submethod_ace_free),  
#endif /* SSH_SERVER_WITH_SECURID */




  AUTH_KBDINT_SUBMETHOD(NULL, NULL, NULL_FNPTR, NULL_FNPTR)
};

SshAuthKbdIntSubMethods
ssh_server_kbd_int_submethods_init(SshServer server,
                                   const char *user_name,
                                   SshUser uc)
{
  SshADTContainer l;
  SshADTHandle h = SSH_ADT_INVALID;
  SshConfig config = server->config;
  int i;
  char *name;
  SshAuthKbdIntSubMethods methods;
  
  SSH_PRECOND(config != NULL);

  methods = ssh_xcalloc(1, sizeof(*methods));
  
  /* Go through OPTIONAL submethods, and mark them as such. */
  if (config->auth_kbd_int_optional)
    {
      l = config->auth_kbd_int_optional;
      
      for (h = ssh_adt_enumerate_start(l); h != SSH_ADT_INVALID;
           h = ssh_adt_enumerate_next(l, h))
        {
          name = (char *)ssh_adt_get(l, h);

          for (i = 0; submethods[i].name != NULL; i++)
            if (strcmp(submethods[i].name, name) == 0)
              {
                SSH_DEBUG(2, ("Submethod '%s' marked as optional.", name));
                submethods[i].use = SSH_KBDINT_SUBMETHOD_OPTIONAL;
                break;
              }
          
          if (submethods[i].name == NULL)
            ssh_warning("auth-kbd-int: '%s' (optional) not found in "
                        "submethods", name);
        }
    }
  
  /* Go through REQUIRED submethods, and mark them as such. */
  if (config->auth_kbd_int_required)
    {
      l = config->auth_kbd_int_required;
      
      for (h = ssh_adt_enumerate_start(l); h != SSH_ADT_INVALID;
           h = ssh_adt_enumerate_next(l, h))
        {
          name = ssh_adt_get(l, h);

          for (i = 0; submethods[i].name != NULL; i++)
            if (strcmp(submethods[i].name, name) == 0)
              {
                SSH_DEBUG(2, ("Submethod '%s' marked as required.", name));
                submethods[i].use = SSH_KBDINT_SUBMETHOD_REQUIRED;
                break;
              }
          
          if (submethods[i].name == NULL)
            ssh_warning("auth-kbd-int: '%s' (required) not found in "
                        "submethods", name);
        }
    }

  methods->uc = uc;
  methods->user = ssh_xstrdup(user_name);
  methods->server = server;
  methods->submethods = submethods;
  return methods;
}

void ssh_server_kbd_int_submethods_uninit(SshAuthKbdIntSubMethods methods)
{
  int i;

  SSH_PRECOND(methods != NULL);
  SSH_PRECOND(methods->submethods != NULL);
  
  ssh_xfree(methods->user);
  
  for (i = 0; i < sizeof(submethods)/sizeof(*submethods); i++)
    if (methods->submethods[i].method_context)
      {
        (*methods->submethods[i].method_free)
          (methods->submethods[i].method_context);
        methods->submethods[i].method_context = NULL;
        methods->submethods[i].status = SSH_KBDINT_SUBMETHOD_UNDEFINED;
      }
}

void ssh_server_kbd_int_submethod_init(SshAuthKbdIntSubMethods methods,
                                       AuthKbdIntSubMethod method,
                                       SshKbdIntSubMethodConv conv,
                                       void *conv_context)
{
  SSH_PRECOND(method != NULL);
  SSH_PRECOND(method->use != SSH_KBDINT_SUBMETHOD_NOTUSED);
  SSH_PRECOND(method->method_init != NULL_FNPTR);
  SSH_PRECOND(method->method_free != NULL_FNPTR);
  
  (*method->method_init)(&method->method_context, methods, conv, conv_context);

  if (method->method_context == NULL)
    (*conv)(SSH_KBDINT_SUBMETHOD_RESULT_FAILED, NULL, 0, NULL, NULL,
            NULL_FNPTR, NULL, conv_context);
}

const char *ssh_server_kbd_int_submethod_get_name(AuthKbdIntSubMethod method)
{
  SSH_PRECOND(method != NULL);
  return method->long_name;
}

#endif /* SSH_SERVER_WITH_KEYBOARD_INTERACTIVE */
