/*

auths-kbd-int-securid.c

  Author: Sami J. Lehtinen <sjl@ssh.com>

  Created: Wed Feb 27 12:45:49 2002.

  Copyright (C) 2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  SecurID submethod for keyboard-interactive.
  
*/

#include "ssh2includes.h"
#ifdef SSH_SERVER_WITH_KEYBOARD_INTERACTIVE
#ifdef SSH_SERVER_WITH_SECURID
#include "auths-kbd-int-submethods.h"
#include "sshfsm.h"
#include "sshdsprintf.h"

/* Define this, if you wish to try compilation of this file, without the
   actual SecurID libraries. */
/* #define TEST_WITHOUT_SECURID */
#ifndef TEST_WITHOUT_SECURID
#ifndef HAVE_OLD_RSA_ACE_API
/* With RSA ACE 5.0 API, only one header is needed. */
#ifdef HAVE_ACEXPORT_H
#include <acexport.h>
#endif /* HAVE_ACEXPORT_H */
#else /* HAVE_OLD_RSA_ACE_API */
#ifdef HAVE_SDI_ATHD_H
#include <sdi_athd.h>
#endif /* HAVE_SDI_ATHD_H */
#ifdef HAVE_SDI_DEFS_H
#include <sdi_defs.h>
#endif /* HAVE_SDI_DEFS_H */
#ifdef HAVE_SDACMVLS_H
#include <sdacmvls.h>
#endif /* HAVE_SDACMVLS_H */
#ifdef HAVE_SDCONF_H
#include <sdconf.h>
#endif /* HAVE_SDCONF_H */
#endif /* HAVE_OLD_RSA_ACE_API */
#else /* TEST_WITHOUT_SECURID */
#define HAVE_OLD_RSA_ACE_API 1
#include "auths-securid-dummy.h"
#endif /* TEST_WITHOUT_SECURID */

#define SSH_DEBUG_MODULE "Ssh2AuthKbdIntSecurID"

















































































#define MIN_PIN_LEN 4
#define MAX_PIN_LEN 8

#define MAX_REQS 64

#ifdef HAVE_OLD_RSA_ACE_API
#ifdef HAVE_SDCONF_H
union config_record configure;
#endif /* HAVE_SDCONF_H */
#endif /* HAVE_OLD_RSA_ACE_API */

typedef struct SecurIDStateRec {
  SshFSM fsm;
  SshFSMThread main_thread;

  SshAuthKbdIntSubMethods methods;

  SshKbdIntSubMethodConv conv;
  void *conv_context;

  char *instruction;

  char **resps;
  size_t num_resps;

  char *new_pin;
  
  Boolean pin_change_success;
  int user_selectable;
  Boolean alphanumeric;
  char *system_pin;
  int min_pin_len;
  int max_pin_len;
  
  /* handle for the ace server */
#ifndef HAVE_OLD_RSA_ACE_API
  SDI_HANDLE sd;
#else /* HAVE_OLD_RSA_ACE_API */
  struct SD_CLIENT *sd;
#endif /* HAVE_OLD_RSA_ACE_API */  

  /* Whether submethod_ace_finish() should destroy the state
     structure. */
  Boolean destroy_state;
} SecurIDStateStruct, *SecurIDState;

SSH_FSM_STEP(submethod_ace_start);
SSH_FSM_STEP(submethod_ace_passcode_query);
SSH_FSM_STEP(submethod_ace_passcode_check);
SSH_FSM_STEP(submethod_ace_query_next_code);
SSH_FSM_STEP(submethod_ace_next_code_check);
SSH_FSM_STEP(submethod_ace_new_pin_select);
SSH_FSM_STEP(submethod_ace_query_system_pin_usage);
SSH_FSM_STEP(submethod_ace_system_pin_usage_process_reply);
SSH_FSM_STEP(submethod_ace_query_new_pin);
SSH_FSM_STEP(submethod_ace_query_new_pin_real);
SSH_FSM_STEP(submethod_ace_process_new_pin);
SSH_FSM_STEP(submethod_ace_process_new_pin_real);
SSH_FSM_STEP(submethod_ace_query_about_system_pin);
SSH_FSM_STEP(submethod_ace_process_resp_about_system_pin);
SSH_FSM_STEP(submethod_ace_success);
SSH_FSM_STEP(submethod_ace_failed);
SSH_FSM_STEP(submethod_ace_finish);

void submethod_ace_init(void **method_context,
                        SshAuthKbdIntSubMethods methods,
                        SshKbdIntSubMethodConv conv,
                        void *conv_context)
{
  SecurIDState state = ssh_xcalloc(1, sizeof(*state));

  state->fsm = ssh_fsm_create(state);
  SSH_VERIFY(state->fsm);

  state->methods = methods;
  state->conv = conv;
  state->conv_context = conv_context;
  
  state->main_thread = ssh_fsm_thread_create(state->fsm, submethod_ace_start,
                                             NULL_FNPTR, NULL_FNPTR, NULL);
  SSH_VERIFY(state->main_thread);

  SSH_DEBUG(2, ("Starting SecurID submethod."));
  *method_context = state;
  




}

void submethod_ace_free(void *method_context)
{
  SecurIDState state = (SecurIDState) method_context;
  if (state->fsm)
    {
      state->destroy_state = TRUE;
      ssh_fsm_set_next(state->main_thread, submethod_ace_finish);
      ssh_fsm_continue(state->main_thread);
      return;
    }

  ssh_xfree(state);





}

void submethod_ace_free_resps(SecurIDState state)
{
  int i = 0;
  
  if (state->resps == NULL)
    return;

  for (i = 0; i < state->num_resps; i++)
    ssh_xfree(state->resps[i]);
  ssh_xfree(state->resps);
  state->resps = NULL;
}

void submethod_ace_resp_cb(size_t num_resp,
                           char **resps,
                           Boolean cancel,
                           void *context)
{
  SshFSMThread thread = (SshFSMThread)context;
  SecurIDState state = (SecurIDState)ssh_fsm_get_gdata(thread);
  int i = 0;
  
  if (cancel)
    {
      SSH_DEBUG(2, ("Canceling."));
      ssh_fsm_set_next(state->main_thread, submethod_ace_finish);
      SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
      return;
    }

  submethod_ace_free_resps(state);
  state->num_resps = num_resp;
  state->resps = ssh_xcalloc(num_resp, sizeof(char *));
  for (i = 0; i < num_resp; i++)
    state->resps[i] = ssh_xstrdup(resps[i]);

  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_FSM_STEP(submethod_ace_start)
{
  SecurIDState state = (SecurIDState)fsm_context;
  
#ifndef HAVE_OLD_RSA_ACE_API



  if (!AceInitialize())

    {
      SSH_DEBUG(2,("ACE initialization unsuccessful."));
      goto init_error;
    }
  
  if (SD_Init(&state->sd) != ACM_OK)
    {
      SSH_DEBUG(2,("Error initializing connection with SecurID server."));
      goto init_error;
    }

  if (SD_Lock(state->sd, state->methods->user) != ACM_OK)
    {
      SSH_DEBUG(2, ("Unable to lock server."));
      goto init_error;
    }

#else /* HAVE_OLD_RSA_ACE_API */
  state->sd = ssh_xcalloc(1, (sizeof(*state->sd)));

  if (creadcfg())
    {
      SSH_DEBUG(2,("Error reading sdconf.rec"));
      goto init_error;
    }

  if (sd_init(state->sd))
    {
      SSH_DEBUG(2,("Error initializing connection with SecurID server."));
      goto init_error;
    }
#endif /* HAVE_OLD_RSA_ACE_API */

  SSH_FSM_SET_NEXT(submethod_ace_passcode_query);
  return SSH_FSM_CONTINUE;
 init_error:
  SSH_FSM_SET_NEXT(submethod_ace_failed);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(submethod_ace_passcode_query)
{
  SecurIDState state = (SecurIDState)fsm_context;
  size_t num_reqs = 0;
  char *reqs[MAX_REQS];
  Boolean echo[MAX_REQS];

  SSH_PRECOND(state);
  
  if (state->pin_change_success)
    {
      reqs[num_reqs] = "Wait for token to change, and enter PASSCODE: ";
      state->pin_change_success = FALSE;
    }
  else
    {
      reqs[num_reqs] = "Enter PASSCODE: ";
    }
  echo[num_reqs] = FALSE;
  num_reqs++;
  
  SSH_FSM_SET_NEXT(submethod_ace_passcode_check);

  SSH_FSM_ASYNC_CALL((*state->conv)(SSH_KBDINT_SUBMETHOD_RESULT_NONE_YET,
                                   state->instruction ?
                                   state->instruction : "",
                                   num_reqs, reqs, echo,
                                   submethod_ace_resp_cb, thread,
                                   state->conv_context));
}

SSH_FSM_STEP(submethod_ace_passcode_check)
{
  SecurIDState state = (SecurIDState)fsm_context;
  int ret_code;
  
  SSH_ASSERT(state->resps[0] != NULL);
  
#ifndef HAVE_OLD_RSA_ACE_API
  ret_code = SD_Check(state->sd, state->resps[0], state->methods->user);
#else /* HAVE_OLD_RSA_ACE_API */
  /* XXX could check now for valid length of passcode */
  ret_code = sd_check(state->resps[0], state->methods->user, state->sd);
#endif /* HAVE_OLD_RSA_ACE_API */

  switch(ret_code)
    {
    case ACM_OK:

      SSH_DEBUG(2,("User AUTHENTICATED.\r\n"));
      SSH_FSM_SET_NEXT(submethod_ace_success);
      return SSH_FSM_CONTINUE;

    case ACM_ACCESS_DENIED:
      SSH_DEBUG(2,("Access denied from SecurID server"));
      SSH_FSM_SET_NEXT(submethod_ace_failed);
      return SSH_FSM_CONTINUE;

    case ACM_NEXT_CODE_REQUIRED:
      SSH_DEBUG(2,("Next code required"));
      SSH_FSM_SET_NEXT(submethod_ace_query_next_code);
      return SSH_FSM_CONTINUE;

    case ACM_NEW_PIN_REQUIRED:
      SSH_DEBUG(2,("New PIN required."));
      SSH_FSM_SET_NEXT(submethod_ace_new_pin_select);
      return SSH_FSM_CONTINUE;

    default:
      SSH_DEBUG(4,("Unknown return code from SecurID server sd_check"));
      SSH_FSM_SET_NEXT(submethod_ace_failed);
      return SSH_FSM_CONTINUE;
    }
  SSH_NOTREACHED;
}

/*
 * Next code.
 */

SSH_FSM_STEP(submethod_ace_query_next_code)
{
  SecurIDState state = (SecurIDState)fsm_context;
  size_t num_reqs = 0;
  char *reqs[MAX_REQS];
  Boolean echo[MAX_REQS];

  SSH_PRECOND(state);
  
  reqs[num_reqs] = "Wait for token to change and enter PASSCODE: ";
  echo[num_reqs] = FALSE;
  num_reqs++;
  
  SSH_FSM_SET_NEXT(submethod_ace_next_code_check);

  SSH_FSM_ASYNC_CALL((*state->conv)(SSH_KBDINT_SUBMETHOD_RESULT_NONE_YET,
                                    "", num_reqs, reqs, echo,
                                    submethod_ace_resp_cb, thread,
                                    state->conv_context));
}

SSH_FSM_STEP(submethod_ace_next_code_check)
{
  SecurIDState state = (SecurIDState)fsm_context;
  int ret_code;
  
  SSH_ASSERT(state->resps[0] != NULL);
#ifndef HAVE_OLD_RSA_ACE_API
  ret_code = SD_Next(state->sd, state->resps[0]);
#else /* HAVE_OLD_RSA_ACE_API */
  ret_code = sd_next(state->resps[0], state->sd);
#endif /* HAVE_OLD_RSA_ACE_API */

  switch(ret_code)
    {
    case ACM_OK:
      SSH_DEBUG(2,("User AUTHENTICATED by challenge."));
      SSH_FSM_SET_NEXT(submethod_ace_success);
      return SSH_FSM_CONTINUE;

    case ACM_ACCESS_DENIED:
      SSH_DEBUG(2,("User access denied from challenge."));
      SSH_FSM_SET_NEXT(submethod_ace_failed);
      return SSH_FSM_CONTINUE;

    case ACM_NEXT_CODE_REQUIRED:
      SSH_DEBUG(2,("Next code required"));
      SSH_FSM_SET_NEXT(submethod_ace_query_next_code);
      return SSH_FSM_CONTINUE;

    default:
      SSH_DEBUG(4,("Unknown return code from securid server sd_next"));
      SSH_FSM_SET_NEXT(submethod_ace_failed);
      return SSH_FSM_CONTINUE;
    }
  SSH_NOTREACHED;
}

/*
 * New pin.
 */

SSH_FSM_STEP(submethod_ace_new_pin_select)
{
  SecurIDState state = (SecurIDState)fsm_context;
  unsigned int user_selectable;
  Boolean alphanumeric;
  char *system_pin;
  int min_pin_len, max_pin_len;

#ifndef HAVE_OLD_RSA_ACE_API
  SD_PIN pin_params;

  if (AceGetPinParams(state->sd, &pin_params) != ACE_SUCCESS)
    {
      SSH_DEBUG(2,("Error getting PIN params."));
      SSH_FSM_SET_NEXT(submethod_ace_failed);
      return SSH_FSM_CONTINUE;
    }

  user_selectable = pin_params.Selectable;

  if (pin_params.Alphanumeric == 0)
    alphanumeric = FALSE;
  else
    alphanumeric = TRUE;

  /* system PIN stuff. */
  SSH_ASSERT(pin_params.System);
  
  if (user_selectable == MUST_CHOOSE_PIN)
    system_pin = ssh_xstrdup("");
  else
    system_pin = ssh_xstrdup(pin_params.System);

  min_pin_len = pin_params.Min;
  max_pin_len = pin_params.Max;

#else /* HAVE_OLD_RSA_ACE_API */
  user_selectable = state->sd->user_selectable;

  if (state->sd->alphanumeric == 0)
    alphanumeric = FALSE;
  else
    alphanumeric = TRUE;

  if (user_selectable == MUST_CHOOSE_PIN)
    system_pin = ssh_xstrdup("");
  else
    system_pin = ssh_xstrdup(state->sd->system_pin);

  max_pin_len = state->sd->max_pin_len;
  min_pin_len = state->sd->min_pin_len;
#endif /* HAVE_OLD_RSA_ACE_API */

  state->user_selectable = user_selectable;
  state->alphanumeric = alphanumeric;
  state->system_pin = system_pin;
  state->min_pin_len = min_pin_len;
  state->max_pin_len = max_pin_len;
  
  switch(user_selectable)
    {
    case CANNOT_CHOOSE_PIN:
      SSH_FSM_SET_NEXT(submethod_ace_query_about_system_pin);
      break;
    case MUST_CHOOSE_PIN:
      SSH_FSM_SET_NEXT(submethod_ace_query_new_pin);
      break;
    default:
      SSH_FSM_SET_NEXT(submethod_ace_query_system_pin_usage);
      break;
    }
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(submethod_ace_query_system_pin_usage)
{
  SecurIDState state = (SecurIDState)fsm_context;
  size_t num_reqs = 0;
  char *reqs[MAX_REQS];
  Boolean echo[MAX_REQS];

  SSH_PRECOND(state);
  
  reqs[num_reqs] = "Would you like to create your own new PIN (yes/no)? ";
  echo[num_reqs] = TRUE;
  num_reqs++;
  
  SSH_FSM_SET_NEXT(submethod_ace_system_pin_usage_process_reply);

  SSH_FSM_ASYNC_CALL((*state->conv)(SSH_KBDINT_SUBMETHOD_RESULT_NONE_YET,
                                    "You may create your own PIN or accept "
                                    "a server assigned PIN.",
                                    num_reqs, reqs, echo,
                                    submethod_ace_resp_cb, thread,
                                    state->conv_context));
}

SSH_FSM_STEP(submethod_ace_system_pin_usage_process_reply)
{
  SecurIDState state = (SecurIDState)fsm_context;
  size_t num_reqs = 0;
  char *reqs[MAX_REQS];
  Boolean echo[MAX_REQS];

  SSH_PRECOND(state);

  if (strcmp(state->resps[0], "yes") == 0)
    {
      SSH_FSM_SET_NEXT(submethod_ace_query_new_pin);
    }
  else if (strcmp(state->resps[0], "no") == 0)
    {
      SSH_FSM_SET_NEXT(submethod_ace_query_about_system_pin);
    }
  else
    {
      reqs[num_reqs] = "Please answer \"yes\" or \"no\" (without "
        "the quotes): ";
      echo[num_reqs] = TRUE;
      num_reqs++;
      /* We haven't changed state with SET_NEXT, so we get back to this
         state after the query is complete. */
      SSH_FSM_ASYNC_CALL((*state->conv)(SSH_KBDINT_SUBMETHOD_RESULT_NONE_YET,
                                        "",
                                        num_reqs, reqs, echo,
                                        submethod_ace_resp_cb, thread,
                                        state->conv_context));
    }

  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(submethod_ace_query_new_pin)
{
  SecurIDState state = (SecurIDState)fsm_context;
  char *instruction;
  
  SSH_PRECOND(state);

  if (state->alphanumeric)
    instruction = "Enter your new PIN (%d to %d digits or characters) \n"
      "or leave it empty to leave your token in NEW PIN mode.";
  else
    instruction = "Enter your new PIN (%d to %d digits) or leave it \n"
      "empty to leave your token in NEW PIN mode.";

  ssh_dsprintf(&state->instruction, instruction,
               state->min_pin_len, state->max_pin_len);

  SSH_FSM_SET_NEXT(submethod_ace_query_new_pin_real);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(submethod_ace_query_new_pin_real)
{
  SecurIDState state = (SecurIDState)fsm_context;
  size_t num_reqs = 0;
  char *reqs[MAX_REQS];
  Boolean echo[MAX_REQS];

  SSH_PRECOND(state);
  reqs[num_reqs] = "New PIN: ";
  echo[num_reqs] = FALSE;
  num_reqs++;
  reqs[num_reqs] = "Confirm new PIN: ";
  echo[num_reqs] = FALSE;
  num_reqs++;

  SSH_FSM_SET_NEXT(submethod_ace_process_new_pin);
  SSH_FSM_ASYNC_CALL((*state->conv)(SSH_KBDINT_SUBMETHOD_RESULT_NONE_YET,
                                    state->instruction ?
                                    state->instruction : "",
                                    num_reqs, reqs, echo,
                                    submethod_ace_resp_cb, thread,
                                    state->conv_context));
}

SSH_FSM_STEP(submethod_ace_process_new_pin)
{
  SecurIDState state = (SecurIDState)fsm_context;
  int new_pin_len;
  
  SSH_ASSERT(state->num_resps == 2);

  if (strcmp(state->resps[0], state->resps[1]) != 0)
    {
      ssh_xfree(state->instruction);
      state->instruction = ssh_xstrdup("The PIN codes entered do not match. "
                                       "Try again.");
      SSH_FSM_SET_NEXT(submethod_ace_query_new_pin_real);
      return SSH_FSM_CONTINUE;
    }

  new_pin_len = strlen(state->resps[0]);
  
  if (new_pin_len > MAX_PIN_LEN || new_pin_len < MIN_PIN_LEN)
    {
      SSH_DEBUG(2, ("Invalid PIN len %ld.", new_pin_len));
      SSH_FSM_SET_NEXT(submethod_ace_failed);
      return SSH_FSM_CONTINUE;
    }

  state->new_pin = ssh_xstrdup(state->resps[0]);

  ssh_xfree(state->instruction);
  state->instruction = NULL;
  SSH_FSM_SET_NEXT(submethod_ace_process_new_pin_real);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(submethod_ace_process_new_pin_real)
{
  SecurIDState state = (SecurIDState)fsm_context;
  int ret_code;
#ifdef HAVE_OLD_RSA_ACE_API
  int cancel_new_pin;
#endif /* HAVE_OLD_RSA_ACE_API */

#ifndef HAVE_OLD_RSA_ACE_API
  /* New API consideres PIN change canceled if ``new_pin'' is
     zero-length or NULL. */
  ret_code = SD_Pin(state->sd, state->new_pin);
#else /* HAVE_OLD_RSA_ACE_API */
  if (strlen(state->new_pin) == 0)
    cancel_new_pin = 1;
  else
    cancel_new_pin = 0;

  /* verify the new pin with the securid server */
  ret_code = sd_pin(state->new_pin, cancel_new_pin, state->sd);
#endif /* HAVE_OLD_RSA_ACE_API */
  ssh_xfree(state->new_pin);
  state->new_pin = NULL;
  
  if (ret_code == ACM_NEW_PIN_ACCEPTED)
    {
      SSH_DEBUG(2,("New PIN operation successful"));
      /* new pin is ok so send a passphrase next token msg to client */
      state->pin_change_success = TRUE;
      SSH_FSM_SET_NEXT(submethod_ace_passcode_query);
      return SSH_FSM_CONTINUE;
    }

  SSH_DEBUG(2,("New PIN operation failed."));
  SSH_FSM_SET_NEXT(submethod_ace_failed);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(submethod_ace_query_about_system_pin)
{
  SecurIDState state = (SecurIDState)fsm_context;
  size_t num_reqs = 0;
  char *reqs[MAX_REQS];
  Boolean echo[MAX_REQS];
  SSH_PRECOND(state);

  reqs[num_reqs] = "Please answer \"yes\" or \"no\": ";
  echo[num_reqs] = TRUE;
  num_reqs++;

  SSH_FSM_SET_NEXT(submethod_ace_process_resp_about_system_pin);
  
  SSH_FSM_ASYNC_CALL((*state->conv)(SSH_KBDINT_SUBMETHOD_RESULT_NONE_YET,
                                    "Would you like to accept the server "
                                    "assigned PIN?\n"
                                    "Answering \"no\" will leave your token "
                                    "to NEW PIN mode.",
                                    num_reqs, reqs, echo,
                                    submethod_ace_resp_cb, thread,
                                    state->conv_context));
}

SSH_FSM_STEP(submethod_ace_process_resp_about_system_pin)
{
  SecurIDState state = (SecurIDState)fsm_context;
  size_t num_reqs = 0;
  char *reqs[MAX_REQS];
  Boolean echo[MAX_REQS];

  SSH_PRECOND(state);

  if (strcmp(state->resps[0], "yes") == 0)
    {
      ssh_xfree(state->instruction);
      ssh_dsprintf(&state->instruction, "Your new PIN is %s.",
                   state->system_pin);
      state->new_pin = ssh_xstrdup(state->system_pin);
      SSH_FSM_SET_NEXT(submethod_ace_process_new_pin_real);
    }
  else if (strcmp(state->resps[0], "no") == 0)
    {
      state->new_pin = ssh_xstrdup("");
      SSH_FSM_SET_NEXT(submethod_ace_process_new_pin_real);
    }
  else
    {
      reqs[num_reqs] = "Please answer \"yes\" or \"no\" (without "
        "the quotes): ";
      echo[num_reqs] = TRUE;
      num_reqs++;
      /* We haven't changed state with SET_NEXT, so we get back to this
         state after the query is complete. */
      SSH_FSM_ASYNC_CALL((*state->conv)(SSH_KBDINT_SUBMETHOD_RESULT_NONE_YET,
                                        "",
                                        num_reqs, reqs, echo,
                                        submethod_ace_resp_cb, thread,
                                        state->conv_context));
    }

  return SSH_FSM_CONTINUE;
}

/*
 * Terminal (that is, won't lead to continued requests) states.
 */

SSH_FSM_STEP(submethod_ace_success)
{
  SecurIDState state = (SecurIDState)fsm_context;
  SshKbdIntResult result = SSH_KBDINT_SUBMETHOD_RESULT_SUCCESS;
  SSH_FSM_SET_NEXT(submethod_ace_finish);



















  /* Notify main method that this submethod was a success. */
  (*state->conv)(result,
                 NULL, 0, NULL, NULL, NULL_FNPTR, NULL,
                 state->conv_context);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(submethod_ace_failed)
{
  SecurIDState state = (SecurIDState)fsm_context;
  SSH_FSM_SET_NEXT(submethod_ace_finish);
  /* Notify main method that we failed to authenticate. */
  (*state->conv)(SSH_KBDINT_SUBMETHOD_RESULT_FAILED,
                 NULL, 0, NULL, NULL, NULL_FNPTR, NULL,
                 state->conv_context);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(submethod_ace_finish)
{
  SecurIDState state = (SecurIDState)fsm_context;

  SSH_DEBUG(2, ("Freeing state."));
  ssh_fsm_destroy(state->fsm);
  state->fsm = NULL;

  submethod_ace_free_resps(state);

  ssh_xfree(state->new_pin);
  ssh_xfree(state->system_pin);
  ssh_xfree(state->instruction);
#ifndef HAVE_OLD_RSA_ACE_API
  SD_Close(state->sd);
#else /* HAVE_OLD_RSA_ACE_API */
  ssh_xfree(state->sd);
  sd_close();
#endif /* HAVE_OLD_RSA_ACE_API */
  
  if (state->destroy_state)    
    ssh_xfree(state);

  return SSH_FSM_FINISH;
}

#endif /* SSH_SERVER_WITH_SECURID */
#endif /* SSH_SERVER_WITH_KEYBOARD_INTERACTIVE */
