/*

auths-kbd-int-plugin.c

  Author: Sami J. Lehtinen <sjl@ssh.com>

  Created: Sun Apr 21 18:23:04 2002.

  Copyright (C) 2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Plugin authentication for keyboard-interactive. Uses an external
  program, defined by the system's administrator.

*/

#include "ssh2includes.h"
#ifdef SSH_SERVER_WITH_KEYBOARD_INTERACTIVE
#include "auths-kbd-int-submethods.h"
#include "auths-kbd-int-plugin.h"
#include "sshfsm.h"
#include "sshadt_list.h"

#include "sshunixpipestream.h"



#include "sshfsmstreams.h"


#define SSH_DEBUG_MODULE "Ssh2AuthKbdIntPlugin"

typedef struct PluginStateRec {
  SshFSM fsm;
  SshFSMThread thread;
  SshFSMThread stream_handler;
  
  SshAuthKbdIntSubMethods methods;
  
  SshKbdIntSubMethodConv conv;  
  void *conv_context;

  SshADTContainer reqs;
  char *instruction;
  size_t num_reqs;
  char **reqs_array;
  Boolean *echo_array;
  
  size_t num_resps;
  char **resp;

  SshStream stderr_stream;

  SshUInt32 stub_flags;
  SshFSMCondition in_buf_shrunk;
  SshFSMCondition out_buf_grown;
  SshFSMCondition read_more;
  SshBuffer in_buf;
  SshBuffer out_buf;
  SshBuffer stderr_buf;
} PluginStateStruct, *PluginState;

typedef struct PluginReqRec {
  char *req;
  Boolean echo;
} PluginReqStruct, *PluginReq;

SSH_FSM_STEP(plugin_start);
SSH_FSM_STEP(plugin_put_params);
SSH_FSM_STEP(plugin_get);
SSH_FSM_STEP(plugin_put);
SSH_FSM_STEP(plugin_suspended);
SSH_FSM_STEP(plugin_finish);
SSH_FSM_STEP(plugin_finish_finalize);

void plugin_clean_reqs(PluginState state)
{
  SSH_PRECOND(state != NULL);
  ssh_xfree(state->reqs_array);
  ssh_xfree(state->echo_array);
  ssh_xfree(state->instruction);
  state->num_reqs = 0;
  state->reqs_array = NULL;
  state->echo_array = NULL;
  state->instruction = NULL;
  ssh_adt_clear(state->reqs);
}

void plugin_req_destroy(void *obj, void *context)
{
  PluginReq req = (PluginReq) obj;
  ssh_xfree(req->req);
  ssh_xfree(req);
}

void submethod_plugin_method_cb(size_t num_resps,
                                char **resp,
                                Boolean cancel,
                                void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  PluginState state = ssh_fsm_get_gdata(thread);

  SSH_DEBUG(4, ("Cancel = %s, num_resps = %ld.", cancel ? "TRUE" : "FALSE",
                num_resps));
  plugin_clean_reqs(state);
  
  if (cancel)
    {
      /* The main method will call the `free' function shortly after
         sending this. */
      return;
    }
  
  state->num_resps = num_resps;
  state->resp = resp;
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

void submethod_plugin_init(void **method_context,
                          SshAuthKbdIntSubMethods methods,
                          SshKbdIntSubMethodConv conv,
                          void *conv_context)
{
  PluginState state = ssh_xcalloc(1, sizeof(*state));

  state->fsm = ssh_fsm_create(state);

  state->methods = methods;
  state->conv = conv;
  state->conv_context = conv_context;
  
  state->thread = ssh_fsm_thread_create(state->fsm, plugin_start, NULL_FNPTR,
                                        NULL_FNPTR, NULL);
  *method_context = state->thread;
}

void submethod_plugin_free(void *method_context)
{
  SshFSMThread thread = (SshFSMThread) method_context;

  SSH_FSM_SET_NEXT(plugin_finish);
  ssh_fsm_continue(thread);
}

void plugin_stderr_iocb(SshStreamNotification notification,
                        void *context)
{
  PluginState state = (PluginState) context;
  
  if (notification == SSH_STREAM_INPUT_AVAILABLE)
    {
      unsigned char buf[10];
      int ret ;

      do {
        ret = ssh_stream_read(state->stderr_stream, buf, 1);
        if (ret > 0)
          {
            if (buf[0] == '\n')
              {
                ssh_debug("auth-kbd-int: plugin: %.*s",
                          ssh_buffer_len(state->stderr_buf),
                          ssh_buffer_ptr(state->stderr_buf));
                ssh_buffer_consume(state->stderr_buf,
                                   ssh_buffer_len(state->stderr_buf));
              }
            else
              {
                ssh_xbuffer_append(state->stderr_buf, buf, 1);
              }
          }
        else if (ret == 0)
          {
            SSH_DEBUG(2, ("Got EOF from stderr stream."));
          }
      } while (ret > 0);
    }
  else
    {
      SSH_DEBUG(0, ("Got notification %d from stderr stream.", notification));
    }
}

#define SEND_FAILURE(state)                                                 \
do                                                                          \
{                                                                           \
  (*(state)->conv)(SSH_KBDINT_SUBMETHOD_RESULT_FAILED, NULL,                \
                   0, NULL, NULL, NULL_FNPTR, NULL, (state)->conv_context); \
                                                                            \
}                                                                           \
while(0)

SSH_FSM_STEP(plugin_start)
{
  PluginState state = (PluginState) fsm_context;
  char *shell;
  char *argv[10];
  SshStream stdio_stream;

  if (!state->methods->server->config->auth_kbd_int_plugin_prog ||
      strlen(state->methods->server->config->auth_kbd_int_plugin_prog) == 0)
    {
      SSH_DEBUG(2, ("Plugin program not defined."));
      SSH_FSM_SET_NEXT(plugin_suspended);
      SEND_FAILURE(state);
      return SSH_FSM_CONTINUE;
    }

  SSH_FSM_SET_NEXT(plugin_put_params);
  

  switch (ssh_pipe_create_and_fork(&stdio_stream,
                                   &state->stderr_stream))






    {

    case SSH_PIPE_CHILD_OK:
      shell = "/bin/sh";
      argv[0] = shell;
      argv[1] = "-c";
      argv[2] = state->methods->server->config->auth_kbd_int_plugin_prog;
      argv[3] = NULL;
      execv(shell, argv);
      ssh_warning("Failed to execute %s, sys_err `%s'", shell,
                  strerror(errno));
      exit(254);
      break;

    case SSH_PIPE_PARENT_OK:
      SSH_DEBUG(2, ("Starting plugin."));
      state->in_buf = ssh_xbuffer_allocate();
      state->out_buf = ssh_xbuffer_allocate();
      state->stderr_buf = ssh_xbuffer_allocate();
      SSH_ASSERT(state->thread != NULL);
      state->read_more = ssh_fsm_condition_create(state->fsm);
      SSH_ASSERT(state->read_more != NULL);
      state->in_buf_shrunk = ssh_fsm_condition_create(state->fsm);
      SSH_ASSERT(state->in_buf_shrunk != NULL);
      state->out_buf_grown = ssh_fsm_condition_create(state->fsm);
      SSH_ASSERT(state->out_buf_grown != NULL);
      state->stream_handler = ssh_streamstub_spawn(state->fsm,
                                                   stdio_stream,
                                                   state->in_buf,
                                                   state->out_buf,
                                                   1024,
                                                   state->read_more,
                                                   state->in_buf_shrunk,
                                                   NULL,
                                                   state->out_buf_grown,
                                                   NULL,
                                                   &state->stub_flags);
      SSH_ASSERT(state->stream_handler != NULL);
      ssh_stream_set_callback(state->stderr_stream, plugin_stderr_iocb,
                              state);

      state->reqs = ssh_adt_create_generic(SSH_ADT_LIST,
                                           SSH_ADT_DESTROY,
                                           plugin_req_destroy,
                                           SSH_ADT_ARGS_END);
      break;
    case SSH_PIPE_ERROR:
      ssh_warning("Failed to create pipes when trying to execute plugin `%s' "
                  "for keyboard-interactive. Failing submethod for user `%s'.",
                  state->methods->server->config->auth_kbd_int_plugin_prog,
                  state->methods->user);
      SSH_FSM_SET_NEXT(plugin_suspended);
      SEND_FAILURE(state);
      break;
    }

  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(plugin_put_params)
{
  PluginState state = (PluginState) fsm_context;
  char line[1024];
  SSH_DEBUG(4, ("Sending parameters to plugin."));
  ssh_snprintf(line, sizeof(line), "user_name:%s\n", state->methods->user);
  ssh_xbuffer_append(state->out_buf, (unsigned char *)line, strlen(line));
  ssh_snprintf(line, sizeof(line), "host_ip:%s\n",
               state->methods->server->common->local_ip);
  ssh_xbuffer_append(state->out_buf, (unsigned char *)line, strlen(line));
  ssh_snprintf(line, sizeof(line), "host_name:%s\n",
               state->methods->server->common->local_hostname);
  ssh_xbuffer_append(state->out_buf, (unsigned char *)line, strlen(line));
  ssh_snprintf(line, sizeof(line), "remote_host_ip:%s\n",
               state->methods->server->common->remote_ip);
  ssh_xbuffer_append(state->out_buf, (unsigned char *)line, strlen(line));
  ssh_snprintf(line, sizeof(line), "remote_host_name:%s\n",
               state->methods->server->common->remote_host);
  ssh_xbuffer_append(state->out_buf, (unsigned char *)line, strlen(line));
  if (state->methods->server->authenticated_client_user_name)
    {
      ssh_snprintf(line, sizeof(line), "remote_user_name:%s\n",
                   state->methods->server->authenticated_client_user_name);
      ssh_xbuffer_append(state->out_buf, (unsigned char *)line, strlen(line));
    }
  ssh_snprintf(line, sizeof(line), "end_of_params\n");
  ssh_xbuffer_append(state->out_buf, (unsigned char *)line, strlen(line));
  SSH_FSM_CONDITION_SIGNAL(state->out_buf_grown);
  SSH_FSM_SET_NEXT(plugin_get);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(plugin_get)
{
  PluginState state = (PluginState) fsm_context;
  unsigned char *data;
  char line[1024] = "\0";
  size_t len, i;
  Boolean newline_found = FALSE;
  PluginReq req;

  data = ssh_buffer_ptr(state->in_buf);
  len = ssh_buffer_len(state->in_buf);

  if (len == 0)
    goto skip_and_check_for_eof;
  
  for (i = 0; i < len ; i++)
    if (data[i] == '\n')
      break;

  if (i >= sizeof(line))
    {
      ssh_warning("Got too long line from kbd-int plugin "
                  "(len = %d). Submethod \"plugin\" for user %s failed.",
                  i + 1, state->methods->user);
      SSH_FSM_SET_NEXT(plugin_suspended);
      SEND_FAILURE(state);
      return SSH_FSM_CONTINUE;
    }
  if (i < len)
    {
      newline_found = TRUE;
      memmove(line, data, i + 1);
      line[i] = '\0';
      ssh_buffer_consume(state->in_buf, i + 1);
      SSH_FSM_CONDITION_SIGNAL(state->in_buf_shrunk);
    }
  else
    {
      /* No newline found. */
      SSH_DEBUG(6, ("Waiting for newline."));
      SSH_FSM_CONDITION_SIGNAL(state->in_buf_shrunk);
      SSH_FSM_CONDITION_WAIT(state->read_more);
    }

#define REQUEST_ID "req:"
#define REQUEST_WITH_ECHO_ID "req_echo:"
#define INSTRUCTION_ID "instruction:"

  SSH_DEBUG(5, ("Got line `%s' from plugin.", line));
  if (strcasecmp(line, "failure") == 0)
    {
      SSH_DEBUG(2, ("Got failure"));
      SSH_FSM_SET_NEXT(plugin_suspended);
      SEND_FAILURE(state);
      return SSH_FSM_CONTINUE;
    }
  else if (strcasecmp(line, "success") == 0)
    {
      int result = SSH_KBDINT_SUBMETHOD_RESULT_SUCCESS;
      SSH_DEBUG(2, ("Got success"));
      SSH_FSM_SET_NEXT(plugin_suspended);

















      (*state->conv)(result, NULL,
                     0, NULL, NULL, NULL_FNPTR, NULL, state->conv_context);
      return SSH_FSM_CONTINUE;
    }
  else if (strcasecmp(line, "end_of_requests") == 0)
    {
      SshADTHandle h;
      /* Send the requests to the main method */
      state->num_reqs = ssh_adt_num_objects(state->reqs);
      if (state->num_reqs)
        {
          state->reqs_array = ssh_xcalloc(state->num_reqs, sizeof(char *));
          state->echo_array = ssh_xcalloc(state->num_reqs, sizeof(Boolean));
          for (h = ssh_adt_enumerate_start(state->reqs), i = 0;
               h != SSH_ADT_INVALID;
               h = ssh_adt_enumerate_next(state->reqs, h), i++)
            {
              req = ssh_adt_get(state->reqs, h);
              SSH_ASSERT(req != NULL);
              SSH_ASSERT(state->num_reqs > i);
              state->reqs_array[i] = req->req;
              state->echo_array[i] = req->echo;
            }
        }
      else
        {
          state->reqs_array = NULL;
          state->echo_array = NULL;
        }
      SSH_FSM_SET_NEXT(plugin_put);
      SSH_FSM_ASYNC_CALL((*state->conv)(SSH_KBDINT_SUBMETHOD_RESULT_NONE_YET,
                                        state->instruction ?
                                        state->instruction : "",
                                        state->num_reqs, state->reqs_array,
                                        state->echo_array,
                                        submethod_plugin_method_cb,
                                        thread, state->conv_context));
    }
  else if (strncasecmp(line, REQUEST_ID, strlen(REQUEST_ID)) == 0)
    {
      /* Add request (without echo). */
      req = ssh_xcalloc(1, sizeof(*req));
      req->echo = FALSE;
      req->req = ssh_xstrdup(&line[strlen(REQUEST_ID)]);
      ssh_adt_insert_to(state->reqs, SSH_ADT_END, req);
    }
  else if (strncasecmp(line, REQUEST_WITH_ECHO_ID,
                       strlen(REQUEST_WITH_ECHO_ID)) == 0)
    {
      /* Add request (with echo). */
      req = ssh_xcalloc(1, sizeof(*req));
      req->echo = TRUE;
      req->req = ssh_xstrdup(&line[strlen(REQUEST_WITH_ECHO_ID)]);
      ssh_adt_insert_to(state->reqs, SSH_ADT_END, req);
    }
  else if (strncasecmp(line, INSTRUCTION_ID, strlen(INSTRUCTION_ID)) == 0)
    {
      /* Modify instruction id. */
      ssh_xfree(state->instruction);
      state->instruction = ssh_xstrdup(&line[strlen(INSTRUCTION_ID)]);
      SSH_DEBUG(4, ("Got instruction: `%s'.", state->instruction));
    }
  else
    {
      SSH_DEBUG(2, ("Line \"%s\" ignored.", line));
    }
  
 skip_and_check_for_eof:
  if (ssh_buffer_len(state->in_buf) == 0 || !newline_found)
    {
      if (state->stub_flags & SSH_STREAMSTUB_EOF_RECEIVED)
        {
          SSH_DEBUG(4, ("EOF received from plugin."));
          SSH_FSM_SET_NEXT(plugin_suspended);
          SEND_FAILURE(state);
        }
      else
        {
          SSH_DEBUG(4, ("No line yet from plugin."));
          SSH_FSM_CONDITION_WAIT(state->read_more);
        }
    }
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(plugin_put)
{
  PluginState state = (PluginState) fsm_context;
  char line[1024];
  int i;
  
  if (state->stub_flags & SSH_STREAMSTUB_OUTPUT_CLOSED)
    {
      SSH_DEBUG(4, ("EOF on write received."));
      SSH_FSM_SET_NEXT(plugin_suspended);
      SEND_FAILURE(state);
      return SSH_FSM_CONTINUE;
    }

  /* Output the responses to the plugin, and go back to reading the
     replies. */
  SSH_DEBUG(5, ("Sending replies to plugin."));
  for (i = 0; i < state->num_resps; i++)
    {
      ssh_snprintf(line, sizeof(line), "reply:%s\n", state->resp[i]);
      ssh_xbuffer_append(state->out_buf, (unsigned char *)line, strlen(line));
    }
  ssh_snprintf(line, sizeof(line), "end_of_replies\n");
  ssh_xbuffer_append(state->out_buf, (unsigned char *)line, strlen(line));
  SSH_FSM_CONDITION_SIGNAL(state->out_buf_grown);
  SSH_FSM_SET_NEXT(plugin_get);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(plugin_suspended)
{
  return SSH_FSM_SUSPENDED;
}

SSH_FSM_STEP(plugin_finish)
{
  PluginState state = (PluginState) fsm_context;

  SSH_FSM_SET_NEXT(plugin_finish_finalize);

  if (state->stderr_stream)
    ssh_stream_destroy(state->stderr_stream);

  if (state->stream_handler && !(state->stub_flags & SSH_STREAMSTUB_FINISHED))
    SSH_FSM_THROW(state->stream_handler, SSH_STREAMSTUB_ABORT);
  else
    return SSH_FSM_CONTINUE;

  SSH_FSM_WAIT_THREAD(state->stream_handler);
}

SSH_FSM_STEP(plugin_finish_finalize)
{
  PluginState state = (PluginState) fsm_context;
  SSH_DEBUG(2, ("Finishing..."));

  if (state->in_buf)
    ssh_buffer_free(state->in_buf);
  if (state->out_buf)
    ssh_buffer_free(state->out_buf);
  if (state->stderr_buf)  
    ssh_buffer_free(state->stderr_buf);
  if (state->out_buf_grown)
    ssh_fsm_condition_destroy(state->out_buf_grown);
  if (state->read_more)
    ssh_fsm_condition_destroy(state->read_more);
  if (state->in_buf_shrunk)
    ssh_fsm_condition_destroy(state->in_buf_shrunk);
  ssh_fsm_destroy(state->fsm);
  if (state->reqs)
    {
      plugin_clean_reqs(state);
      ssh_adt_destroy(state->reqs);
    }
  state->fsm = NULL;
  state->thread = NULL;
  ssh_xfree(state);
  return SSH_FSM_FINISH;
}

#endif /* SSH_SERVER_WITH_KEYBOARD_INTERACTIVE */
