/*

auths-kbd-int-pam.c

  Author: Sami J. Lehtinen <sjl@ssh.com>

  Created: Wed Feb 13 23:32:28 2002.

  Copyright (C) 2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  PAM submethod for keyboard-interactive. Quite straight rip from
  auths-pam.c. In the future we might be able to get rid of auths-pam.c.
*/

#include "sshincludes.h"
#ifdef SSH_SERVER_WITH_KEYBOARD_INTERACTIVE
#ifdef DAEMON_WITH_PAM
#include "auths-kbd-int-submethods.h"
#include "auths-kbd-int-pam.h"
#include "sshfsm.h"
#include "sshpamserver.h"
#include "sshunixpipestream.h"
#include "sshencode.h"
#include "sshdsprintf.h"
#include "sshmsgs.h"
#include <security/pam_appl.h>

#define SSH_DEBUG_MODULE "Ssh2AuthKbdIntPAM"

typedef enum
{
  /* Nothing special. */
  SENT_BASIC = 0,
  /* Special cases. */
  SENT_PAM_GET_ITEM,
  SENT_PAM_ACCT_MGMT
} PAMAuthSentState;

typedef struct PAMStateRec {
  char *service_name;

  size_t sent_reqs;
  int num_pam_reqs;
  int *pam_req_styles;
  
  SshAuthKbdIntSubMethods methods;
  
  PAMAuthSentState sent_state;
  /* Packetstream to ssh_pam_conv. */
  SshPacketWrapper wrapper;

  SshFSM fsm;
  SshFSMThread main_thread;

  SshKbdIntSubMethodConv conv;
  void *conv_context;

  /* Whether submethod_pam_finish() should destroy the state
     structure. */
  Boolean destroy_state;
} PAMStateStruct, *PAMState;

#define SEND_CODE(state, code)                                          \
do                                                                      \
{                                                                       \
  (*(state)->conv)((code), NULL, 0, NULL, NULL, NULL_FNPTR, NULL,       \
                   (state)->conv_context);                              \
}                                                                       \
while(0)

#define SEND_FAILED(state) SEND_CODE(state, SSH_KBDINT_SUBMETHOD_RESULT_FAILED)
#define SEND_SUCCESS(state) \
        SEND_CODE(state, SSH_KBDINT_SUBMETHOD_RESULT_SUCCESS)

/* Forward declarations. */
SSH_FSM_STEP(submethod_pam_start);
SSH_FSM_STEP(submethod_pam_set_item_rhost);
SSH_FSM_STEP(submethod_pam_set_item_tty);
SSH_FSM_STEP(submethod_pam_set_item_ruser);
SSH_FSM_STEP(submethod_pam_authenticate);
SSH_FSM_STEP(submethod_pam_acct_mgmt);
SSH_FSM_STEP(submethod_pam_chauthtok);
SSH_FSM_STEP(submethod_pam_open_session);
SSH_FSM_STEP(submethod_pam_setcred);
SSH_FSM_STEP(submethod_pam_auth_successful);
SSH_FSM_STEP(submethod_pam_process_client_packet);
SSH_FSM_STEP(submethod_pam_suspend);
SSH_FSM_STEP(submethod_pam_finish);

void submethod_pam_received_packet_cb(SshPacketType packet_type,
                                      const unsigned char *data, size_t len,
                                      void *context);
void submethod_pam_received_eof_cb(void *context);

void submethod_pam_init(void **method_context,
                        SshAuthKbdIntSubMethods methods,
                        SshKbdIntSubMethodConv conv,
                        void *conv_context)
{
  PAMState state;
  SshStream child_stdio;

  switch (ssh_pipe_create_and_fork(&child_stdio, NULL))
    {
    case SSH_PIPE_ERROR:
      ssh_log_event(methods->server->config->log_facility,
                    SSH_LOG_ERROR,
                    "kbd-int:pam: pipe creation failed.");
      *method_context = NULL;
      return;
      break;
    case SSH_PIPE_PARENT_OK:
      SSH_TRACE(2, ("Fork successful."));
      break;
    case SSH_PIPE_CHILD_OK:
      if (ssh_pam_conv())
        {
          ssh_log_event(methods->server->config->log_facility,
                        SSH_LOG_ERROR,
                        "PAM transaction resulted in error.");
          exit(1);
        }
      else
        {
          SSH_DEBUG(2, ("PAM transaction was a success."));
          exit(0);
        }
      SSH_NOTREACHED;
      break;
    }

  state = ssh_xcalloc(1, sizeof(*state));

  state->fsm = ssh_fsm_create(state);
  SSH_VERIFY(state->fsm);
  
  /* Service name defaults to "sshd2". */
  state->service_name = ssh_xstrdup(SSH_PAM_SERVICE_NAME);
  state->sent_state = SENT_BASIC;
  state->methods = methods;
  state->conv = conv;
  state->conv_context = conv_context;
  
  state->wrapper = ssh_packet_wrap(child_stdio,
                                   submethod_pam_received_packet_cb,
                                   submethod_pam_received_eof_cb,
                                   NULL_FNPTR, state);
  SSH_VERIFY(state->wrapper);

  state->main_thread = ssh_fsm_thread_create(state->fsm, submethod_pam_start,
                                             NULL_FNPTR, NULL_FNPTR, NULL);
  SSH_VERIFY(state->main_thread);
  
  *method_context = state; 
}

void submethod_pam_free(void *method_context)
{
  PAMState state = (PAMState) method_context;
  if (state->fsm)
    {
      state->destroy_state = TRUE;
      ssh_fsm_set_next(state->main_thread, submethod_pam_finish);
      ssh_fsm_continue(state->main_thread);
      return;
    }

  ssh_xfree(state);
}

void submethod_pam_resp_cb(size_t num_resp,
                           char **resps,
                           Boolean cancel,
                           void *context)
{
  PAMState gdata = (PAMState) context;
  SshBuffer packet;
  char *resp;
  size_t ret, i;
  size_t cur_resp = 0;
  
  SSH_PRECOND(gdata);
  if (cancel)
    {
      SSH_DEBUG(2, ("Canceling."));
      ssh_fsm_set_next(gdata->main_thread, submethod_pam_finish);
      ssh_fsm_continue(gdata->main_thread);
      return;
    }

  /* Just a sanity check; this is checked by the main method. */
  SSH_ASSERT(gdata->sent_reqs == num_resp);

  packet = ssh_xbuffer_allocate();
  
  ret = ssh_encode_buffer(packet,
                          SSH_FORMAT_UINT32, (SshUInt32)gdata->num_pam_reqs,
                          SSH_FORMAT_END);
  SSH_VERIFY(ret);
  
  for (i = 0; i < gdata->num_pam_reqs; i++)
    {
      if (gdata->pam_req_styles[i] == SSH_PAM_PROMPT_ECHO_OFF ||
          gdata->pam_req_styles[i] == SSH_PAM_PROMPT_ECHO_ON)
        {
          SSH_DEBUG(3, ("Adding response."));
          resp = resps[cur_resp];
          cur_resp++;
        }
      else
        {
          SSH_DEBUG(3, ("Adding empty response."));
          resp = "";
        }

      SSH_ASSERT(cur_resp <= num_resp);
      
      ret = ssh_encode_buffer(packet,
                              SSH_FORMAT_CHAR, (unsigned int) 0,
                              SSH_FORMAT_UINT32_STR, resp, strlen(resp),
                              SSH_FORMAT_END);
      SSH_VERIFY(ret);
    }
  ssh_packet_wrapper_send(gdata->wrapper, SSH_PAM_CONVERSATION_RESP,
                          ssh_buffer_ptr(packet), ssh_buffer_len(packet));
  ssh_buffer_free(packet);
}

void submethod_pam_received_packet_cb(SshPacketType packet_type,
                                      const unsigned char *data, size_t len,
                                      void *context)
{
  PAMState gdata = (PAMState) context;
  SshFSMThread thread;
  unsigned int err_num;
  char *err_msg;
  size_t ret, bytes = 0;
  SshUInt32 num_pam_reqs;
  int *pam_req_styles = NULL;
  char **reqs = NULL, *req = NULL, *to_be_deleted = NULL, *instruction = NULL;
  Boolean *echo = NULL;
  int i, num_reqs = 0;
  unsigned int style;
  
  SSH_PRECOND(gdata);

  thread = gdata->main_thread;
  SSH_PRECOND(thread);
  
  SSH_DEBUG(4, ("Received packet %d.", packet_type));

  if (packet_type == SSH_PAM_OP_ERROR ||
      packet_type == SSH_PAM_ERROR)
    {
      if (ssh_decode_array(data, len,
                           SSH_FORMAT_CHAR, &err_num,
                           SSH_FORMAT_UINT32_STR, &err_msg, NULL,
                           SSH_FORMAT_END) !=
          len || len == 0)
        {
          /* Protocol error. */
          ssh_log_event(gdata->methods->server->config->log_facility,
                        SSH_LOG_ERROR,
                        "kbd-int:pam: PAM subprocess sent a malformed "
                        "packet. (packet type: %d)", packet_type);

          SEND_FAILED(gdata);
          SSH_FSM_SET_NEXT(submethod_pam_finish);
          ssh_fsm_continue(thread);
          gdata->sent_state = SENT_BASIC;
          return;
        }
    }

  switch (packet_type)
    {
    case SSH_PAM_OP_SUCCESS:
      /* Do nothing. */
      break;
    case SSH_PAM_OP_SUCCESS_WITH_PAYLOAD:
      /* Check, that we have sent a message, that expects a success
         message with payload. */
      if (gdata->sent_state == SENT_PAM_GET_ITEM)
        {
          /* XXX Store data (currently this isn't used). */
          SSH_NOTREACHED;
        }
      else
        {
          ssh_log_event(gdata->methods->server->config->log_facility,
                        SSH_LOG_ERROR,
                        "kbd-int:pam: PAM subprocess returned packet "
                        "SSH_PAM_OP_SUCCESS_WITH_PAYLOAD, expecting "
                        "SSH_PAM_OP_SUCCESS.");

          SEND_FAILED(gdata);
          SSH_FSM_SET_NEXT(submethod_pam_finish);
        }
      break;
    case SSH_PAM_OP_ERROR:
      /* Check if operation can fail, and what will we do about
         it. Otherwise, fall through to generic error.*/
      if  (gdata->sent_state == SENT_PAM_ACCT_MGMT &&
           err_num == PAM_NEW_AUTHTOK_REQD)
        {
          SSH_FSM_SET_NEXT(submethod_pam_chauthtok);
          /* XX clean up. */
          ssh_xfree(err_msg);
          ssh_fsm_continue(thread);
          gdata->sent_state = SENT_BASIC;
          return;
        }
      /* FALL THROUGH */
    case SSH_PAM_ERROR:
      SSH_ASSERT(err_msg);
      ssh_log_event(gdata->methods->server->config->log_facility,
                    SSH_LOG_ERROR,
                    "kbd-int:pam: PAM subprocess returned packet "
                    "%s. (err_num: %d, err_msg: %s)",
                    packet_type == SSH_PAM_OP_ERROR ? "SSH_PAM_OP_ERROR" :
                    "SSH_PAM_ERROR", err_num, err_msg);
      ssh_xfree(err_msg);

      SEND_FAILED(gdata);
      SSH_FSM_SET_NEXT(submethod_pam_finish);

      break;
    case SSH_PAM_CONVERSATION_MSG:
      /* send conv message to client. */

      ret = ssh_decode_array(data, len,
                             SSH_FORMAT_UINT32, &num_pam_reqs,
                             SSH_FORMAT_END);
      if (ret == 0)
        {
          SSH_DEBUG(2, ("Error decoding conversation packet from PAM "
                        "subprocess."));
          goto error;
        }

      bytes += ret;

      pam_req_styles = ssh_xcalloc(num_pam_reqs, sizeof(int));
      reqs = ssh_xcalloc(num_pam_reqs, sizeof(char *));
      echo = ssh_xcalloc(num_pam_reqs, sizeof(Boolean));
      
      for (i = 0; i < num_pam_reqs; i++)
        {
          if (bytes > len)
            {
              SSH_DEBUG(2, ("packet too short (num_pam_reqs not satisfied)."));
              goto error;
            }
          
          ret = ssh_decode_array(data + bytes, len - bytes,
                                 SSH_FORMAT_CHAR, &style,
                                 SSH_FORMAT_UINT32_STR, &req, NULL,
                                 SSH_FORMAT_END);
          if (ret == 0)
            {
              SSH_DEBUG(2, ("Error decoding req from conversation packet from "
                            "PAM subprocess."));
              goto error;
            }

          if (!(style == SSH_PAM_PROMPT_ECHO_OFF
#ifdef PAM_BINARY_PROMPT
                || style == SSH_PAM_BINARY_PROMPT
#endif /* PAM_BINARY_PROMPT */
#ifdef PAM_BINARY_MSG
                || style == SSH_PAM_BINARY_PROMPT
#endif /* PAM_BINARY_MSG */
                || style == SSH_PAM_PROMPT_ECHO_ON
                || style == SSH_PAM_ERROR_MSG
                || style == SSH_PAM_TEXT_INFO))
            {
              SSH_DEBUG(2, ("Style %d not supported.", style));
              goto error;
            }

          SSH_DEBUG(3, ("Got req '%s', style %d.", req, style));
          
          if (style == SSH_PAM_PROMPT_ECHO_OFF ||
              style == SSH_PAM_PROMPT_ECHO_ON)
            {
              SSH_ASSERT(req);
              reqs[num_reqs] = req;
              echo[num_reqs] = style == SSH_PAM_PROMPT_ECHO_ON ? TRUE : FALSE;
              num_reqs++;
            }
          else if (style == SSH_PAM_ERROR_MSG || style == SSH_PAM_TEXT_INFO)
            {
              /* Just appended together, and sent in the instruction field.*/
              SSH_ASSERT(req);
              to_be_deleted = instruction;
              if (instruction)
                ssh_xdsprintf(&instruction, "%s\n%s", to_be_deleted, req);
              else
                instruction = ssh_xstrdup(req);

              ssh_xfree(req);
              ssh_xfree(to_be_deleted);
            }
          pam_req_styles[i] = style;
          bytes += ret;
        }

      if (num_reqs == 0)
        {
          ssh_xfree(reqs);
          reqs = NULL;
          ssh_xfree(echo);
          echo = NULL;
        }

      gdata->sent_reqs = num_reqs;
      gdata->num_pam_reqs = num_pam_reqs;
      gdata->pam_req_styles = pam_req_styles;
      pam_req_styles = NULL;

      if (!instruction)
        instruction = ssh_xstrdup("");
      
      SSH_DEBUG(3, ("Calling conv function..."));
      (*gdata->conv)(SSH_KBDINT_SUBMETHOD_RESULT_NONE_YET,
                     instruction,
                     num_reqs,
                     reqs,
                     echo,
                     submethod_pam_resp_cb,
                     gdata,
                     gdata->conv_context);

      /* Don't revive thread; we will come here again (after the
         client sends us back a response, the resp_cb will send it
         to PAM subprocess, and the PAM subprocess will send us the
         acknowledgement for the original message). */
    error:
      for (i = 0; i < num_reqs; i++)
        ssh_xfree(reqs[i]);
      ssh_xfree(reqs);
      ssh_xfree(instruction);
      ssh_xfree(echo);
      ssh_xfree(pam_req_styles);
      return;
    default:
      ssh_log_event(gdata->methods->server->config->log_facility,
                    SSH_LOG_ERROR,
                    "kbd-int:pam: PAM subprocess returned invalid "
                    "packet type %d.", packet_type);

      SEND_FAILED(gdata);
      SSH_FSM_SET_NEXT(submethod_pam_finish);
      break;
    }

  gdata->sent_state = SENT_BASIC;
  ssh_fsm_continue(thread);
}

void submethod_pam_received_eof_cb(void *context)
{
  PAMState gdata = (PAMState) context;
  SSH_TRACE(0, ("Received EOF."));

  SEND_FAILED(gdata);
  ssh_fsm_set_next(gdata->main_thread, submethod_pam_finish);
  ssh_fsm_continue(gdata->main_thread);
}

SSH_FSM_STEP(submethod_pam_start)
{
  PAMState gdata = (PAMState)fsm_context;
  
  SSH_FSM_SET_NEXT(submethod_pam_set_item_rhost);

  SSH_TRACE(4,("Sending service_name:%s, user:%s to PAM subprocess.",
               gdata->service_name, gdata->methods->user));

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_START,
                                 SSH_FORMAT_UINT32_STR, gdata->service_name,
                                 strlen(gdata->service_name),
                                 SSH_FORMAT_UINT32_STR,
                                 gdata->methods->user,
                                 strlen(gdata->methods->user),
                                 SSH_FORMAT_END);
  return SSH_FSM_SUSPENDED;
}

SSH_FSM_STEP(submethod_pam_set_item_rhost)
{
  char *client_host;
  PAMState gdata = (PAMState)fsm_context;
  SSH_FSM_SET_NEXT(submethod_pam_set_item_tty);

  if (gdata->methods->server->authenticated_client_host)
    /* If set, use authenticated remote host name from "hostbased". */
    client_host = gdata->methods->server->authenticated_client_host;
  else
    client_host = gdata->methods->server->common->remote_host;

  SSH_ASSERT(client_host);

  SSH_DEBUG(4, ("Sending a SSH_PAM_SET_ITEM (PAM_RHOST) packet to "
                "PAM subprocess..."));

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_SET_ITEM,
                                 SSH_FORMAT_UINT32, (SshUInt32)PAM_RHOST,
                                 SSH_FORMAT_UINT32_STR,
                                 client_host,
                                 strlen(client_host),
                                 SSH_FORMAT_END);
  return SSH_FSM_SUSPENDED;
}

SSH_FSM_STEP(submethod_pam_set_item_tty)
{
  PAMState gdata = (PAMState)fsm_context;
  SSH_FSM_SET_NEXT(submethod_pam_set_item_ruser);

  SSH_DEBUG(4, ("Sending a SSH_PAM_SET_ITEM (PAM_TTY) packet to "
                "PAM subprocess..."));

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_SET_ITEM,
                                 SSH_FORMAT_UINT32, (SshUInt32)PAM_TTY,
                                 SSH_FORMAT_UINT32_STR,
                                 "ssh", strlen("ssh"),
                                 SSH_FORMAT_END);
  return SSH_FSM_SUSPENDED;
}

SSH_FSM_STEP(submethod_pam_set_item_ruser)
{
  PAMState gdata = (PAMState)fsm_context;
  char *user_name;
  SSH_FSM_SET_NEXT(submethod_pam_authenticate);

  user_name = gdata->methods->server->authenticated_client_user_name;

  if (!user_name)
    {
      return SSH_FSM_CONTINUE;
    }

  SSH_DEBUG(4, ("Sending a SSH_PAM_SET_ITEM (PAM_RUSER) packet to "
                "PAM subprocess..."));

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_SET_ITEM,
                                 SSH_FORMAT_UINT32, (SshUInt32)PAM_RUSER,
                                 SSH_FORMAT_UINT32_STR,
                                 user_name, strlen(user_name),
                                 SSH_FORMAT_END);
  return SSH_FSM_SUSPENDED;
}

SSH_FSM_STEP(submethod_pam_authenticate)
{
  PAMState gdata = (PAMState)fsm_context;
  SSH_FSM_SET_NEXT(submethod_pam_acct_mgmt);

  SSH_DEBUG(4, ("Sending a SSH_PAM_AUTHENTICATE packet to PAM subprocess..."));

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_AUTHENTICATE,
                                 /* XXX PermitEmptyPasswords (etc.) ? */
                                 SSH_FORMAT_UINT32,
                                 (SshUInt32) PAM_DISALLOW_NULL_AUTHTOK,
                                 SSH_FORMAT_END);
  return SSH_FSM_SUSPENDED;
}

SSH_FSM_STEP(submethod_pam_acct_mgmt)
{
  PAMState gdata = (PAMState)fsm_context;
  SSH_FSM_SET_NEXT(submethod_pam_open_session);

  gdata->sent_state = SENT_PAM_ACCT_MGMT;

  SSH_DEBUG(4, ("Sending a SSH_PAM_ACCT_MGMT packet to PAM subprocess..."));

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_ACCT_MGMT,
                                 /* XXX PermitEmptyPasswords (etc.) ? */
                                 SSH_FORMAT_UINT32,
                                 (SshUInt32) PAM_DISALLOW_NULL_AUTHTOK,
                                 SSH_FORMAT_END);
  return SSH_FSM_SUSPENDED;
}

SSH_FSM_STEP(submethod_pam_chauthtok)
{
  PAMState gdata = (PAMState)fsm_context;

  SSH_FSM_SET_NEXT(submethod_pam_acct_mgmt);

  SSH_DEBUG(4, ("Sending a SSH_PAM_CHAUTHTOK packet to PAM subprocess..."));

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_CHAUTHTOK,
                                 SSH_FORMAT_UINT32,
                                 (SshUInt32) PAM_CHANGE_EXPIRED_AUTHTOK,
                                 SSH_FORMAT_END);
  return SSH_FSM_SUSPENDED;
}

SSH_FSM_STEP(submethod_pam_open_session)
{
  PAMState gdata = (PAMState)fsm_context;
  SSH_FSM_SET_NEXT(submethod_pam_setcred);

  SSH_DEBUG(4, ("Sending a SSH_PAM_OPEN_SESSION packet to PAM subprocess..."));

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_OPEN_SESSION,
                                 SSH_FORMAT_UINT32, (SshUInt32) 0,
                                 SSH_FORMAT_END);
  return SSH_FSM_SUSPENDED;
}

SSH_FSM_STEP(submethod_pam_setcred)
{
  PAMState gdata = (PAMState)fsm_context;
  SSH_FSM_SET_NEXT(submethod_pam_auth_successful);

  ssh_packet_wrapper_send_encode(gdata->wrapper,
                                 SSH_PAM_SETCRED,
                                 /* XXX Other flags? */
                                 SSH_FORMAT_UINT32,
                                 (SshUInt32) PAM_ESTABLISH_CRED,
                                 SSH_FORMAT_END);
  return SSH_FSM_SUSPENDED;
}

void submethod_pam_session_close_cb(void *context)
{
  SshPacketWrapper wrapper = (SshPacketWrapper)context;
  SSH_DEBUG(2, ("Closing user's session, and destroying wrapper."));
  SSH_PRECOND(wrapper);
  ssh_packet_wrapper_send_encode(wrapper,
                                 SSH_PAM_CLOSE_SESSION,
                                 /* XXX flags? */
                                 SSH_FORMAT_UINT32, 0L,
                                 SSH_FORMAT_END);
  ssh_packet_wrapper_send_encode(wrapper,
                                 SSH_PAM_END,
                                 /* XXX flags? */
                                 SSH_FORMAT_UINT32, 0L,
                                 SSH_FORMAT_END);
  ssh_packet_wrapper_destroy(wrapper);  
}

SSH_FSM_STEP(submethod_pam_auth_successful)
{
  PAMState gdata = (PAMState)fsm_context;

  ssh_packet_wrapper_can_receive(gdata->wrapper, FALSE);
  ssh_packet_wrapper_set_callbacks(gdata->wrapper,
                                   NULL_FNPTR, NULL_FNPTR, NULL_FNPTR, NULL);

  ssh_common_register_clean_up(gdata->methods->server->common,
                               submethod_pam_session_close_cb, gdata->wrapper);
  
  gdata->wrapper = NULL;

  /* inform userauth-layer of successful authentication. */
  SEND_SUCCESS(gdata);
  SSH_FSM_SET_NEXT(submethod_pam_finish);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(submethod_pam_suspend)
{
  return SSH_FSM_SUSPENDED;
}

SSH_FSM_STEP(submethod_pam_finish)
{
  PAMState gdata = (PAMState)fsm_context;

  ssh_fsm_destroy(gdata->fsm);
  gdata->fsm = NULL;

  ssh_xfree(gdata->service_name);

  if (gdata->wrapper)
    {
      ssh_packet_wrapper_can_receive(gdata->wrapper, FALSE);
      ssh_packet_wrapper_destroy(gdata->wrapper);
      gdata->wrapper = NULL;
    }

  if (gdata->destroy_state)
    {
      SSH_DEBUG(2, ("Destroying state."));
      ssh_xfree(gdata);
    }
  return SSH_FSM_FINISH;
}

#endif /* DAEMON_WITH_PAM */
#endif /* SSH_SERVER_WITH_KEYBOARD_INTERACTIVE */
