/*

  auths-common.h

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.
  
  Functions for checking for user, host in lists. Conveniency functions
  for checking whether user is allowed to log in.
  
*/

#ifndef AUTHS_COMMON_H
#define AUTHS_COMMON_H

#include "ssh2includes.h"
#include "sshuser.h"
#include "sshcommon.h"
#include "sshregex.h"
#include "sshadt.h"

/* XXX Unify API (return TRUE (or FALSE) consistently if match
   found/logging in allowed) */

/* Convenience function to bind together checking of host and
   user. Use this in every authentication method. Authentication methods
   should behave the same whether the host or user is denied connect,
   or the authentication methods fail. Returns FALSE if user and host are
   allowed to connect and authenticate. */
Boolean ssh_server_auth_check(SshUser uc, const char *user, SshConfig config,
                              SshCommon common, char *auth_method_name);

/* Use this to check whether specified user is allowed to
   connect. Returns FALSE if allowed. */
Boolean ssh_server_auth_check_user(SshUser uc, const char *user,
                                   SshCommon common);

/* Use this to check whether specified user matches with given
   lists. The exact policy:

    1) if denied_users != NULL and contains user, return TRUE.
    2) if allowed_users != NULL and doesn't contain user, return TRUE.
    3) if denied_groups != NULL and contains user's group, return TRUE.
    4) if allowed_groups != NULL and doesn't contain user's group, return TRUE.
    5) otherwise return FALSE.
*/
Boolean ssh_server_auth_check_user_generic(SshUser uc, const char *user,
                                           const char *allowed_msg,
                                           const char *denied_msg,
                                           const char *hostname,
                                           const char *host_ip,
                                           SshADTContainer allowed_users,
                                           SshADTContainer denied_users,
                                           SshADTContainer allowed_groups,
                                           SshADTContainer denied_groups);

/* Use this to check whether connects from specified host are
   allowed. Returns FALSE if connects are allowed. */
Boolean ssh_server_auth_check_host(SshCommon common);

/* Use this to check whether specified host matches with given
   lists. The exact policy:

    1) If reverse mapping is required and fails, return TRUE.
    2) If denied != NULL and matches, return TRUE.
    3) If allowed != NULL and matches, return FALSE.
    3) If allowed != NULL and doesn't match, return TRUE.
    4) return TRUE.
*/
Boolean ssh_server_auth_check_host_generic(const char *hostname,
                                           const char *host_ip,
                                           SshADTContainer denied,
                                           SshADTContainer allowed,
                                           Boolean require_reverse_mapping);

/* Helper function to check whether given host name or ip-address
   matches a specified pattern in 'holder'. Uses regex-syntax specified
   in 'holder'. Returns TRUE if a match is found, and FALSE
   otherwise. */
Boolean ssh_match_host_id(const char *host_name, const char *host_ip,
                          SshPatternHolder holder, SshRegexContext rex_ctx);

/* Checks whether given host name or ip-address is found in list. list
   contains SshPatternHolders. Returns TRUE if a match is found, and
   FALSE otherwise. */
Boolean ssh_match_host_in_list(const char *host_name, const char *host_ip,
                               SshADTContainer list);

/* Helper function to check whether user (as user name, uid, remote
   host name and remote ip (as string)) matches with a pattern in
   'holder'. Uses regex-syntax specified in 'holder'. Returns TRUE if
   a match is found, and FALSE otherwise. Matches with pattern
   user[@host]. */
Boolean ssh_match_user(const char *user, const char *uid_str,
                       const char *remote_host,
                       const char *remote_ip,
                       SshPatternHolder holder,
                       SshRegexContext rex_ctx);

/* As ssh_match_user(), but checks also groups. Matches
   user[%group][@host]. */
Boolean ssh_match_user_groups(const char *userp, const char *uid_str,
                              size_t num_groups,
                              char * const *groups,
                              char * const *gid_strs,
                              const char *remote_host,
                              const char *remote_ip,
                              SshPatternHolder holder,
                              SshRegexContext rex_ctx);

/* Helper function to check whether user (as user name, uid, remote
   host name and remote ip (as string)) matches with a pattern in
   list. list contains holders that are interpreted as in
   ssh_match_user_groups. Returns TRUE if a match is found, and FALSE
   otherwise. list must not contain NULL holders. */
Boolean ssh_match_user_in_list(const char *user, uid_t uid, 
                               const char *remote_host, const char *remote_ip,
                               SshADTContainer list,
                               SshRegexContext rex_ctx);

/* Function to check whether given 'string' matches with a
   pattern in 'list'. Uses egrep-like-syntax. Returns TRUE if a match
   is found, and FALSE otherwise. list must not contain NULL holders. */
Boolean ssh_match_string_in_list(const char *string,
                                 SshADTContainer list,
                                 SshRegexContext rex_ctx);

/* Helper function to check whether given 'string' or 'number' matches
   with a pattern in 'list'. Uses egrep-like-syntax. Returns FALSE if
   a match is found, and FALSE otherwise. list must not contain NULL
   holders. */
Boolean ssh_match_string_or_number_in_list(const char *string,
                                           unsigned long number,
                                           SshADTContainer list,
                                           SshRegexContext rex_ctx);

/* Match a given 'string' with the given 'pattern' using
   SshRegex. Return TRUE if a match is found, FALSE otherwise. */
Boolean ssh_match_string(const char *string, const char *pattern,
                         SshRegexSyntax rex_syntax, SshRegexContext rex_ctx);

#endif /* AUTHS_COMMON_H */
