/*

  authc-kerberos-tgt.c

  Author: Tatu Ylonen <ylo@ssh.com>

  Copyright (C) 2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Kerberos authentication, client side.

  This code assumes the presence of Kerberos libraries.
  
*/

#include "ssh2includes.h"
#include "sshencode.h"
#include "sshauth.h"
#include "sshclient.h"
#include "sshconfig.h"
#include "authc-kerberos-tgt.h"

#ifdef KERBEROS

#undef ctime
#undef free

/* The #%@!$(#$ krb5.h header redefines these.  ARRRGH! -ylo */
#undef SIZEOF_INT
#undef SIZEOF_LONG
#undef SIZEOF_SHORT
#undef HAVE_STDARG_H
#undef HAVE_SYS_TYPES_H

#include <krb5.h>

#define SSH_DEBUG_MODULE "Ssh2AuthKerberosTgtClient"

/* Kerberos authentication, client-side. */

void ssh_client_auth_kerberos_tgt(SshAuthClientOperation op,
                                  const char *user,
                                  unsigned int packet_type,
                                  SshBuffer packet_in,
                                  const unsigned char *session_id,
                                  size_t session_id_len,
                                  void **state_placeholder,
                                  SshAuthClientCompletionProc completion,
                                  void *completion_context,
                                  void *method_context)
{
  SshConfig clientconf = ((SshClient)method_context)->config;
  SshBuffer b;
  krb5_error_code kerr;
  krb5_context kcontext;
  krb5_auth_context kauth;
  krb5_principal server;
  krb5_principal client;
  krb5_ccache ccache;
  krb5_data auth_data, tgt_data, tmp_buf;
  char server_name[512];
  krb5_creds cred_specs, *creds;
  krb5_replay_data replay_data;

  switch (op)
    {
    case SSH_AUTH_CLIENT_OP_START:
      /* XXX should really swap this code and START_NONINTERACTIVE code, but
         apparently there is some bug in sshauthc.c that causes NONINTERACTIVE
         to be called even when a method is not allowed. */
      SSH_TRACE(2, ("Starting kerberos tgt auth..."));
      
      /* Initialize data that we test in cleanup. */
      client = NULL;
      server = NULL;
      creds = NULL;
      
      memset(&kcontext, 0, sizeof(kcontext));
      memset(&ccache, 0, sizeof(ccache));
      memset(&cred_specs, 0, sizeof(cred_specs));
      memset(&kauth, 0, sizeof(kauth));
      memset(&auth_data, 0, sizeof(auth_data));
      memset(&tgt_data, 0, sizeof(tgt_data));
      memset(&tmp_buf, 0, sizeof(tmp_buf));
 
      /* Initialize the Kerberos application context. */
      kerr = krb5_init_context(&kcontext);
      if (kerr)
        goto error;
      
      /* Get the name of the default credentials cache (KRB5CCACHE or
         perhaps some fallback). */
      kerr = krb5_cc_default(kcontext, &ccache);
      if (kerr)
        goto error;
      
      /* Build principal name for the server we are connecting to. */
      kerr = krb5_sname_to_principal(kcontext, clientconf->host_to_connect,
                                     "host", KRB5_NT_SRV_HST,
                                     &cred_specs.server);
      if (kerr)
        goto error;

      /* Use our own principal name as the client name. */
      kerr = krb5_cc_get_principal(kcontext, ccache,
                                   &cred_specs.client);
      if (kerr)
        goto error;

      /* Request a key for DES encryption.  I am not sure if anything else
         works with MIT Kerberos. */
      cred_specs.keyblock.enctype = ENCTYPE_DES_CBC_CRC;

      /* Get suitable credentials. */
      kerr = krb5_get_credentials(kcontext, 0, ccache, &cred_specs,
                                  &creds);
      if (kerr)
          goto error;

      /* Initialize the authentication context. */
      kerr = krb5_auth_con_init(kcontext, &kauth);
      if (kerr)
        goto error;

      /* Enable saving timestamps to output structure. */
      krb5_auth_con_setflags(kcontext, kauth, KRB5_AUTH_CONTEXT_RET_TIME);

      /* Obtain a ticket suitable for host-to-host authentication. */
      kerr = krb5_mk_req_extended(kcontext, &kauth, 0, 0, creds,
                                  &auth_data);
      if (kerr)
        goto error;

      /* This hack is from SSH1 (where I think it originates to Dug
         Song and/or Glenn Machin).  To quote: "We need to get the TGT
         for the clients realm.  However if remotehost is in another
         realm krb5_fwd_tgt_creds will try to go to that realm to get
         the TGT, which will fail.  So we create the server principal
         and point it to clients realm.  This way we pass over a TGT
         of the clients realm." */
      kerr = krb5_cc_get_principal(kcontext, ccache, &client);
      if (kerr)
        goto error;
      ssh_snprintf(server_name, sizeof(server_name), "host/%s@%.*s",
                   clientconf->host_to_connect,
                   client->realm.length, client->realm.data);
      kerr = krb5_parse_name(kcontext, server_name, &server);
      if (kerr)
        goto error;
      server->type = KRB5_NT_SRV_HST;
      
      /* Get a ticket granting ticket for use at the remote host. */
      kerr = krb5_fwd_tgt_creds(kcontext, kauth, NULL, client, server,
                                ccache, 1, &tmp_buf);

      if (kerr)
          goto error;
      
      /* Take the tgt and make it private - encypt in the session key of the
         service ticket - established during krb5_mk_req_extended */

      /* Glenn Machin - Why? 
       *      In order for the ASN routines to properly encode the mk_priv
       *      message, we must set the addresses in the authentication context.
       *
       *      Extract remote and local addresses from the socket used for
       *      communication. 
       */
      kerr = krb5_auth_con_genaddrs(kcontext, kauth, clientconf->ssh1_fd,
                                KRB5_AUTH_CONTEXT_GENERATE_REMOTE_FULL_ADDR |
                                KRB5_AUTH_CONTEXT_GENERATE_LOCAL_FULL_ADDR);
      if (kerr)
          goto error;

      kerr = krb5_mk_priv(kcontext, kauth, &tmp_buf, &tgt_data, &replay_data);
      if (kerr)
        goto error;

      /* Construct the method-specific part of the authentication
         request. */
      b = ssh_xbuffer_allocate();
      ssh_encode_buffer(b,
                        SSH_FORMAT_UINT32_STR,
                        auth_data.data, auth_data.length,
                        SSH_FORMAT_UINT32_STR,
                        tgt_data.data, tgt_data.length,
                        SSH_FORMAT_END);
      
      /* Send the authentication request (and complete this operation). */
      (*completion)(SSH_AUTH_CLIENT_SEND, user, b, completion_context);
      
      /* Free the buffer. */
      ssh_buffer_free(b);

    cleanup:
      /* Clean up allocated Kerberos data. */
      if (auth_data.data)
        free(auth_data.data);
      if (tgt_data.data)
        free(tgt_data.data);
      if (tmp_buf.data)
        free(tmp_buf.data);
      krb5_free_cred_contents(kcontext, &cred_specs);
      if (creds != NULL)
        krb5_free_creds(kcontext, creds);
      if (client)
        krb5_free_principal(kcontext, client);
      if (server)
        krb5_free_principal(kcontext, server);
      krb5_cc_close(kcontext, ccache);
      if (kauth)
        krb5_auth_con_free(kcontext, kauth);
      if (kcontext)
        krb5_free_context(kcontext);
      break;
      
    error:
      ssh_debug("Kerberos5 TGT forwarding failed: %s", error_message(kerr));
      /* Indicate that we cannot use this authentication method. */
      (*completion)(SSH_AUTH_CLIENT_FAIL_AND_DISABLE_METHOD, user, NULL,
                    completion_context);
      goto cleanup;
      
    case SSH_AUTH_CLIENT_OP_START_NONINTERACTIVE:
      (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
      break;
      
    case SSH_AUTH_CLIENT_OP_CONTINUE:
      (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
      break;
      
    case SSH_AUTH_CLIENT_OP_ABORT:
      *state_placeholder = NULL;
      break;
      
    default:
      ssh_debug("ssh_client_auth_kerberos: unknown op %d", (int)op);
      (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
      break;
    }
}

#endif /* KERBEROS */
