/*

  authc-kbd-interactive.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 2000, 2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Keyboard interactive authentication, client side. Done according to
  draft-ietf-secsh-auth-kbdinteract-01.txt.

*/
/*
  TODO:
 - `submethods' field not used yet
 */
#include "ssh2includes.h"
#include "sshfsm.h"
#include "sshclient.h"
#include "sshencode.h"
#include "sshmsgs.h"
#include "readpass.h"
#include "sshappcommon.h"

#define SSH_DEBUG_MODULE "Ssh2AuthKbdInteractiveClient"

#ifdef SSH_CLIENT_WITH_KEYBOARD_INTERACTIVE

typedef struct SshClientKbdInteractAuthRec
{
  SshFSM fsm;
  SshFSMThread main_thread;
  SshClient client;
  SshBuffer server_packet;
  SshBuffer response_packet;

  int num_prompts;
  int cur_msg;
  char *user;

  char *response;
  size_t response_len;

  void **state_placeholder;
  SshAuthClientCompletionProc completion_callback;
  void *completion_context;
} *SshClientKbdInteractAuth, SshClientKbdInteractAuthStruct;












SSH_FSM_STEP(ssh_kbd_process_packet);
SSH_FSM_STEP(ssh_kbd_process_next_msg);
SSH_FSM_STEP(ssh_kbd_append_response);
SSH_FSM_STEP(ssh_kbd_send_response_packet);
SSH_FSM_STEP(ssh_kbd_cancel);
SSH_FSM_STEP(ssh_kbd_error);
SSH_FSM_STEP(ssh_kbd_finish);

SSH_FSM_STEP(ssh_kbd_process_packet)
{
  SshUInt32 num_prompts;
  char *req_name = NULL;
  char *req_instruction = NULL;
  SshClientKbdInteractAuth gdata = (SshClientKbdInteractAuth)fsm_context;

  SSH_PRECOND(gdata != NULL);

  /* extract info and the number of interactive question-answer rounds
     from the server packet.  each round will be extracted from the
     same packet individually in set ssh_kbd_process_next_msg. */
  if (!ssh_decode_buffer(gdata->server_packet,
                         /* string name */
                         SSH_FORMAT_UINT32_STR, &req_name, NULL,
                         /* string instruction */
                         SSH_FORMAT_UINT32_STR, &req_instruction, NULL,
                         /* XXX lang tag */
                         SSH_FORMAT_UINT32_STR, NULL, NULL,
                         SSH_FORMAT_UINT32, &num_prompts,
                         SSH_FORMAT_END))
    {
      SSH_FSM_SET_NEXT(ssh_kbd_error);
      return SSH_FSM_CONTINUE;
    }

  ssh_informational("Keyboard-interactive:\r\n");
  if (req_name && strlen(req_name))
    ssh_informational("%s\r\n", req_name);
  if (req_instruction && strlen(req_instruction))
    ssh_informational("%s\r\n", req_instruction);

  gdata->num_prompts = (int) num_prompts;
  SSH_TRACE(4, ("Received %d messages.", num_prompts));

  ssh_encode_buffer(gdata->response_packet,
                    SSH_FORMAT_CHAR, SSH_MSG_USERAUTH_INFO_RESPONSE,
                    SSH_FORMAT_UINT32, gdata->num_prompts,
                    SSH_FORMAT_END);


  SSH_FSM_SET_NEXT(ssh_kbd_process_next_msg);






  
  ssh_xfree(req_name);
  ssh_xfree(req_instruction);

  return SSH_FSM_CONTINUE;
}

void client_kbd_rl_cb(const char *line, void *context)
{
  SshClientKbdInteractAuth gdata;
  SshFSMThread thread = (SshFSMThread)context;
  SSH_PRECOND(thread != NULL);
  gdata = (SshClientKbdInteractAuth) ssh_fsm_get_gdata(thread);

  if (line)
    {
      gdata->response = ssh_xstrdup(line);
      gdata->response_len = strlen(line);
    }
  else
    {
      gdata->response = NULL;
      gdata->response_len = 0;
      SSH_FSM_SET_NEXT(ssh_kbd_cancel);
    }

  fprintf(stderr, "\r\n");
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_FSM_STEP(ssh_kbd_process_next_msg)
{
  SshUInt32 /* (Boolean, but size matters) */ echo;
  char *msg;
  size_t msg_len;

  SshClientKbdInteractAuth gdata = (SshClientKbdInteractAuth)fsm_context;
  SSH_PRECOND(gdata != NULL);

  if (gdata->cur_msg >= gdata->num_prompts)
    {
      /* All messages processed. Send response packet to
         userauth-layer. */
      SSH_FSM_SET_NEXT(ssh_kbd_send_response_packet);
      return SSH_FSM_CONTINUE;
    }

  gdata->cur_msg++;

  /* extract the next question from the server package. */
  if (!ssh_decode_buffer(gdata->server_packet,
                         SSH_FORMAT_UINT32_STR, &msg, &msg_len,
                         SSH_FORMAT_BOOLEAN, &echo,
                         SSH_FORMAT_END))
    {
      SSH_DEBUG_HEXDUMP(6, ("Rest of received info request packet:"),
                        ssh_buffer_ptr(gdata->server_packet),
                        ssh_buffer_len(gdata->server_packet));

      /* Send protocol error to userauth. */
      SSH_TRACE(0, ("Invalid packet."));
      SSH_FSM_SET_NEXT(ssh_kbd_error);
      return SSH_FSM_CONTINUE;
    }

  SSH_FSM_SET_NEXT(ssh_kbd_append_response);

  /* If there is no interaction, assume empty password. */
  if (gdata->client->config->batch_mode)
    {
      SSH_TRACE(2, ("In Batchmode, so we're not asking the "
                    "user anything. (prompt: %s)", msg));

      gdata->response = NULL;
      gdata->response_len = 0;

      return SSH_FSM_CONTINUE;
    }

  /* Read input, with or without echo. */
  if (echo)
    {
      SSH_FSM_ASYNC_CALL(ssh_readline_eloop(msg, "", gdata->client->rl,
                                            client_kbd_rl_cb,
                                            (void *)thread));
    }
  else
    {
      /* XXX async, please! */
      gdata->response = ssh_read_passphrase(msg, FALSE);
      if (gdata->response)
        gdata->response_len = strlen(gdata->response);
      else
        /* User canceled the authentication. */
        SSH_FSM_SET_NEXT(ssh_kbd_cancel);

      return SSH_FSM_CONTINUE;
    }

  SSH_NOTREACHED;
}

SSH_FSM_STEP(ssh_kbd_append_response)
{
  SshClientKbdInteractAuth gdata = (SshClientKbdInteractAuth)fsm_context;

  SSH_FSM_SET_NEXT(ssh_kbd_process_next_msg);

  ssh_encode_buffer(gdata->response_packet,
                    /* XXX UTF-8 */
                    SSH_FORMAT_UINT32_STR,
                    gdata->response_len ? gdata->response : "",
                    gdata->response_len,
                    SSH_FORMAT_END);
  ssh_xfree(gdata->response);
  gdata->response = NULL;
  gdata->response_len = 0;

  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(ssh_kbd_send_response_packet)
{
  SshClientKbdInteractAuth gdata = (SshClientKbdInteractAuth)fsm_context;

  SSH_FSM_SET_NEXT(ssh_kbd_finish);

  SSH_TRACE(2, ("Sending response packet."));

  /* Cleared here, because if we get back to this authentication
     method before ssh_kbd_finish state, it will lead to an
     SSH_ASSERT. */
  *gdata->state_placeholder = NULL;

  (*gdata->completion_callback)(SSH_AUTH_CLIENT_SEND_AND_CONTINUE_SPECIAL,
                                gdata->user, gdata->response_packet,
                                gdata->completion_context);
  ssh_buffer_clear(gdata->response_packet);

  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(ssh_kbd_cancel)
{
  SshClientKbdInteractAuth gdata = (SshClientKbdInteractAuth)fsm_context;

  (*gdata->completion_callback)(SSH_AUTH_CLIENT_CANCEL,
                                gdata->user, NULL, gdata->completion_context);

  SSH_FSM_SET_NEXT(ssh_kbd_finish);

  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(ssh_kbd_error)
{
  SshClientKbdInteractAuth gdata = (SshClientKbdInteractAuth)fsm_context;

  (*gdata->completion_callback)(SSH_AUTH_CLIENT_FAIL_AND_DISABLE_METHOD,
                                gdata->user, NULL, gdata->completion_context);

  SSH_FSM_SET_NEXT(ssh_kbd_finish);

  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(ssh_kbd_finish)
{
  SshClientKbdInteractAuth gdata = (SshClientKbdInteractAuth)fsm_context;
  ssh_fsm_destroy(gdata->fsm);
  gdata->fsm = NULL;

  *gdata->state_placeholder = NULL;

  ssh_buffer_free(gdata->server_packet);
  ssh_buffer_free(gdata->response_packet);

  ssh_xfree(gdata->user);

  ssh_xfree(gdata->response);

  ssh_xfree(gdata);
  
  return SSH_FSM_FINISH;
}

SshFSMStateDebugStruct client_kbd_states[] =
{
  { "ssh_kbd_process_packet", "Process received packet.",
    ssh_kbd_process_packet},
  { "ssh_kbd_process_next_msg", "Process next message in packet",
    ssh_kbd_process_next_msg },
  { "ssh_kbd_append_response", "Append a response to return packet",
    ssh_kbd_append_response },
  { "ssh_kbd_send_response_packet", "Send response packet to userauth-layer",
    ssh_kbd_send_response_packet },

  { "ssh_kbd_error", "Send disable method packet to userauth",
    ssh_kbd_error },
  { "ssh_kbd_finish", "Destroy FSM.", ssh_kbd_finish }
};

void ssh_client_auth_kbd_interact(SshAuthClientOperation op,
                                  const char *user,
                                  unsigned int packet_type,
                                  SshBuffer packet_in,
                                  const unsigned char *session_id,
                                  size_t session_id_len,
                                  void **state_placeholder,
                                  SshAuthClientCompletionProc completion,
                                  void *completion_context,
                                  void *method_context)
{
  SshBuffer buffer = NULL;
  SshClient client = (SshClient) method_context;
  SshClientKbdInteractAuth state;
  char *lang_tag = "en";

  SSH_PRECOND(client != NULL);

  state = (SshClientKbdInteractAuth)*state_placeholder;

  switch (op)
    {
    case SSH_AUTH_CLIENT_OP_START:
      SSH_TRACE(2, ("Starting kbd-int auth..."));
      buffer = ssh_xbuffer_allocate();
      ssh_encode_buffer(buffer,
                        /* language tag */
                        SSH_FORMAT_UINT32_STR, lang_tag, strlen(lang_tag),
                        /* XXX devices, what do we support? */
                        SSH_FORMAT_UINT32_STR, "", 0L,
                        SSH_FORMAT_END);
      (*completion)(SSH_AUTH_CLIENT_SEND_AND_CONTINUE,
                    user, buffer, completion_context);
      ssh_buffer_free(buffer);
      break;

    case SSH_AUTH_CLIENT_OP_START_NONINTERACTIVE:
      (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
      break;

    case SSH_AUTH_CLIENT_OP_CONTINUE:
      if (packet_type != SSH_MSG_USERAUTH_INFO_REQUEST)
        {
          SSH_TRACE(0, ("unknown packet type %d for continuation packet",
                        (int)op));
          (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
          break;
        }
      SSH_ASSERT(state == NULL);

      /* Allocate state. */ 
      state = ssh_xcalloc(1, sizeof(*state));
      state->fsm = ssh_fsm_create(state);
      state->completion_callback = completion;
      state->completion_context = completion_context;
      state->server_packet = ssh_xbuffer_allocate();
      state->response_packet = ssh_xbuffer_allocate();
      state->user = ssh_xstrdup(user);
      state->state_placeholder = state_placeholder;
      state->client = client;

      ssh_fsm_register_debug_names(state->fsm, client_kbd_states,
                                   SSH_FSM_NUM_STATES(client_kbd_states));
      
      /* Copy packet_in (it will be freed after this function
         returns). */
      ssh_xbuffer_append(state->server_packet, ssh_buffer_ptr(packet_in),
                        ssh_buffer_len(packet_in));

      /* Start thread. */
      state->main_thread = ssh_fsm_thread_create(state->fsm,
                                                 ssh_kbd_process_packet,
                                                 NULL_FNPTR, NULL_FNPTR, NULL);

      /* Rest is done in callbacks. */
      break;

    case SSH_AUTH_CLIENT_OP_ABORT:
      /* Clean up state. */
      if (state)
        {
          ssh_fsm_set_next(state->main_thread, ssh_kbd_finish);
          ssh_fsm_continue(state->main_thread);
        }
      
      *state_placeholder = NULL;
      break;

    default:
      SSH_TRACE(0, ("unknown op %d", (int)op));
      (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
      break;
    }
}

































































#endif /* SSH_CLIENT_WITH_KEYBOARD_INTERACTIVE */
