/*
 * File:	sendmsg.c
 *
 * Author:	Ulli Horlacher (framstag@rus.uni-stuttgart.de)
 *
 * History:	
 * 
 *   11 Aug 95   Framstag	initial version
 *   12 Aug 95   Framstag	elm alias support
 *    2 Nov 95   Framstag	added minimal chat mode
 *   14 Nov 95   Framstag	added message receiving modes
 *   21 Dec 95   Framstag	avoid unnecessary error message while 
 *                              configuring the own tty
 *   21 Dec 95   Framstag	better server connect testing
 *   20 Feb 96   Framstag	changed msg-tty file to support NFS
 *    1 Apr 96   Framstag	added multiline mode
 *   17 Apr 96   Framstag	new error handling for open_connection
 *    2 May 96   Framstag	fixed stupid shutdown() programming bug
 *    3 May 96   Framstag	fixed bug with gethostname()
 *   24 May 96   Framstag	sendmsg no longer tries to configure the tty 
 *                              when there is none (sending via pipe)
 *   12 Aug 96   Framstag	no questions asked when in batch mode (no tty)
 *   13 Nov 96   Framstag	faster mesg/tty handling
 *   26 Nov 96   Framstag	added -n option
 *    4 Jan 97	 Framstag	added saft_connect()-call
 *    3 Feb 97	 Framstag	better tty configuration
 * 				sprintf() -> snprintf()
 * 				added -u and -f options
 * 				added -u and -f options
 * 				changed tty permission testing and -m behaviour
 *   23 Feb 97   Framstag	modified str_* function names
 *   19 Jun 97   Framstag	changend tty testing and behaviour
 *   23 Jun 97   Framstag	added readline support
 *   24 Jun 97   Framstag	fixed some prompt bugs
 *
 * The sendmessage client of the sendfile package.
 * Sends a single line text message to the SAFT-server of the destination
 * system to be displayed on the recipients terminal.
 *
 * Copyright  1995-1997 Ulli Horlacher
 * This file is covered by the GNU General Public License
 */

#include "config.h"		/* various #defines */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>

#include "string.h"		/* extended string functions */
#include "net.h"		/* the network routines */
#include "io.h"			/* (socket) read/write */
#include "message.h"		/* information, warning and error messages */
#include "utf7.h"		/* UTF-7 coding */
#include "address.h"		/* address checking */
#include "getline.h"		/* get a line of text from stdin */

#if defined(HAVE_GETOPT_H)
  #include <getopt.h>
#else
  int getopt(int, char * const *, const char *);
  extern int opterr;
  extern int optind;
  extern int optopt;
  extern char *optarg;
#endif

#ifdef SOLARIS2
  int gethostname(char *, size_t);
#endif
 
#ifndef AIX
  #ifndef CONVEXOS
    FILE *popen(const char *, const char *);
  #endif
  int pclose(FILE *);
#else
  #include "bsd.h"
#endif

#ifdef NEXT
  int shutdown(int, int);
#endif

/* print short help usage text */
int usage();


/* global variables */
int 
  client=1,	/* flag to determine client or server */
  quiet=0,	/* quiet mode */
  verbose=0,	/* flag for verbose mode */
  packet_size;	/* only needed for linking */
char 
  *prg;		/* name of the game */


int main(int argc, char *argv[]) {
  int
    sockfd,		/* socket file descriptor */
    chat,		/* flag for chat mode */
    force,		/* flag for force sending */
    receive,		/* receiving flag */
    multiline,		/* flag for sending a multiline message */
    opt;		/* option to test for */
  char
    *cp,		/* simple character pointer */
    *tty,		/* the tty device with path */
    ttyt[FLEN],		/* the tty on which sendfiled will write (see msgcf) */
    recipient[FLEN], 	/* recipient at serverhost */
    user[FLEN], 	/* local user name */
    from[FLEN], 	/* alternative from name */
    host[FLEN], 	/* name of serverhost */
    localhost[FLEN], 	/* name of the local host */
    msgcf[FLEN], 	/* message control file */
    line[MAXLEN],	/* input or output string */
    login[MAXLEN],	/* login user name */
    tmp[3*MAXLEN];	/* temporary string */
  char
    utf_msg[LEN_UTF],	/* msg in UTF-7 format */
    iso_msg[LEN_ISO];	/* msg in ISO Latin-1 format */
  FILE 
    *inf,		/* input file */
    *outf,		/* output file */
    *pipe;		/* pipe file descriptor */
  struct stat finfo;	/* information about a file */

  chat=0;
  force=0;
  sockfd=0;
  verbose=0;
  receive=0;
  multiline=0;
  *user=0;
  *from=0;
  *line=0;
  *iso_msg=0;
  *ttyt=0;
  tty=NULL;
  pipe=NULL;
  
  prg=argv[0];
  if ((cp=strrchr(prg,'/'))) prg=cp+1;

  /* scan the command line on options */
  while ((opt=getopt(argc,argv,"cvVmMlh?fu:")) > 0) {
    switch (opt) {
      case ':':
      case 'h':
      case '?': exit(usage());
      case 'c': chat=1; break;
      case 'v': verbose=1; break;
      case 'm': receive=1 ; break;
      case 'M': receive=2 ; break;
      case 'l': multiline=1; break;
      case 'f': force=1; break;
      case 'u': strcpy(from,optarg); break;
      case 'V': message(prg,'I',"version "VERSION" revision "REVISION); exit(0);
    }
  }

  /* too few arguments? */
  if (argc-optind<1 && receive==0) exit(usage());

  /* get the local host name */
  if (gethostname(localhost,FLEN-1)<0) strcpy(localhost,"localhost");

  /* get own user name, recipient name and host */
  destination(argc,argv,user,recipient,host);
  if (*from) strcpy(user,from);
  strcpy(login,user);
  if ((cp=strchr(login,' '))) *cp=0;

  if (!force) {
    
    /* test the local sendfiled */
    if (verbose) printf("testing local SAFT server:\n");
    sockfd=open_connection("127.0.0.1",SAFT);
    sock_getline(sockfd,line);
    if (verbose && *line) printf("%s\n",line);

    /* no local server? */
    if (!str_beq(line,"220 ") || !strstr(line,"SAFT"))
      message(prg,'W',"there is no local SAFT server - "
	      "you cannot receive messages");
    else {
   
      /* test if you can receive messages */
      snprintf(MAXS(line),"FROM %s",login);
      sock_putline(sockfd,line);
      sock_getline(sockfd,line);
      if (verbose) printf("%s\n",line);
      snprintf(MAXS(line),"TO %s",login);
      sock_putline(sockfd,line);
      sock_getline(sockfd,line);
      if (verbose) printf("%s\n",line);
      if (str_beq(line,"521 ")) {
	errno=0;
	message(prg,'F',"You are not allowed to use the sendmsg service");
      }
      if (!str_beq(line,"200 "))
	message(prg,'W',"local server error - you cannot receive messages");
    
      if (isatty(fileno(stdin))) {
       
	/* get tty name */
	if (!(tty=ttyname(fileno(stderr))) || *tty!='/')
	  message(prg,'F',"cannot determine your tty name");
        else {

	  /* the message tty config file */
	  snprintf(MAXS(msgcf),"%s/%s/config/tty@%s",SPOOL,login,localhost);

	  /* open tty write permissions if necessary*/
	  if (receive) {
	    
	    /* allow receiving of messages on this tty */
	    if (chmod(tty,S_IRUSR|S_IWUSR|S_IWGRP)<0) {
	      receive=0;
	      message(prg,'W',"cannot open your tty for writing");
	    } else {
	      
	      if (receive==1) {
		
		/* mark current tty as active for receiving */
		if ((outf=fopen(msgcf,"w"))) {
		  fprintf(outf,tty);
		  fclose(outf);
		  if (argc-optind<1)
		    message(prg,'I',"receiving messages is now possible only "
			    "on this tty");
		} else {
		  snprintf(MAXS(tmp),"cannot configure your tty "
			   "(no write access to %s)",msgcf);
		  message(prg,'W',tmp);
		}
	      }
	      if (receive==2) {
		unlink(msgcf);
		if (argc-optind<1)
		  message(prg,'I',"receiving messages is now possible "
		                  "on all ttys");
	      }
	      
	    }
	    
	  } else { /* keep tty status and mode */

	    /* is the current tty writable? */
	    if (stat(tty,&finfo)<0 || !(finfo.st_mode&S_IWGRP)) {
	      errno=0;
	      snprintf(MAXS(tmp),"your tty %s is write protected; "
		       "try sendmsg -m",tty);
	      message(prg,'F',tmp);
	    }
	    
	    /* read the receiving tty from the message config file */
	    if ((inf=fopen(msgcf,"r"))) {
	      fgets(MAXS(ttyt),inf);
	      fclose(inf);
	    } else {
	      strcpy(ttyt,tty);
	      
	      /* mark current tty as active for receiving */
	      if ((outf=fopen(msgcf,"w"))) {
		fprintf(outf,tty);
		fclose(outf);
	      }
	      
	    }

	    /* is the current tty active for sendfiled? */
	    if (!str_eq(tty,ttyt)) {
	      errno=0;
	      snprintf(MAXS(tmp),
		       "you cannot receive an answer message on this tty; "
		       "try sendmsg -m or sendmsg -f",ttyt);
	      message(prg,'F',tmp);
	    }
	    
	  }
	}
      }
    }
    
    shutdown(sockfd,2);
  }
	      

  if (argc-optind<1) exit(0);
  
  /* name the local host */
  if (str_eq(host,"127.0.0.1")) strcpy(host,localhost);

  /* look for correct SAFT server and open connection */
  sockfd=saft_connect("msg",recipient,user,host,tmp);
  
  /* send several lines at once? */
  if (multiline) {
    if (isatty(fileno(stdin))) 
      printf("Enter multiline message (max 10 lines! End with Ctrl-D):\n");

    /* read until EOF */
    while (*line=0,getline(line,LEN_ISO-1)) {
      
      if ((cp=strchr(line,'\n'))) *cp=0;

      /* message text too long? */
      if (strlen(iso_msg)+strlen(line)>LEN_ISO*.8) {
        errno=0;
	message(prg,'F',"message line too long");
      }

      /* add a new line if necessary */
      if (*iso_msg || !*line) strcat(iso_msg,"\r\n");
      
      /* add the text line */
      strncat(iso_msg,line,LEN_ISO-1);
      iso_msg[LEN_ISO-1]=0;
      
    }
    
    if (*iso_msg) {

      /* encode to UTF-7 */
      iso2utf(utf_msg,iso_msg);

      /* message text too long? */
      if (strlen(utf_msg)>MAXLEN-10) {
	errno=0;
	message(prg,'F',"message line too long");
      }

      /* send the message */
      snprintf(MAXS(tmp),"MSG %s",utf_msg);
      sendheader(sockfd,tmp);
      
    }
  
  } else {  /* single line or chat mode */
   
    do {
      /* read the message */
      *iso_msg=0;
      if (isatty(fileno(stdin))) strcpy(iso_msg,"message: ");
      if (!getline(iso_msg,LEN_ISO-1)) {
        printf("\n");
	break;
      }
      
      if (!*iso_msg) break;

      /* strip off new line */
      cp=strrchr(iso_msg,'\n');
      if (cp && (cp!=(char *)iso_msg)) *cp=0;

      /* encode to UTF-7 */
      iso2utf(utf_msg,iso_msg);

      /* send the message */
      snprintf(MAXS(tmp),"MSG %s",utf_msg);
      sendheader(sockfd,tmp);

    } while (chat);

  }

  /* close the connection */
  sock_putline(sockfd,"QUIT");
  getreply(sockfd);
  close(sockfd);

  exit(0);
}


/* void function for message() */
void cleanup() { }


/*
 * usage - print short help usage text
 */
int usage() {
  fprintf(stderr,"usage: %s [-vfmM] user[@host]\n",prg);
  fprintf(stderr,"options: -v  verbose mode\n");
  fprintf(stderr,"         -f  force sending of messages and ignore the tty status\n");
  fprintf(stderr,"         -m  allow receiving messages only on this tty\n");
  fprintf(stderr,"         -M  allow receiving messages on other ttys, too\n");
  fprintf(stderr,"example: %s orakel@main01.bb.bawue.de\n",prg);
  return(2);
}
