/*
 * File:	address.c
 *
 * Author:	Ulli Horlacher (framstag@rus.uni-stuttgart.de)
 *
 * History:	
 * 
 *   12 Aug 95   Framstag	initial version
 *    7 Nov 95   Framstag	added URL addressing
 *   15 Nov 95   Framstag	added sendfile alias file
 *   13 Dec 95   Framstag	correct bug when reading alias file
 *   30 Apr 96   Framstag	checking elm alias only if configured
 *    4 Jan 97   Framstag	renamed from destination.c to address.c
 * 				added check_forward()
 *   22 Jan 97   Framstag	added connect-test to generic saft address
 *   23 Feb 97   Framstag	modified str_* function names
 *   24 Feb 97   Framstag	sprintf() -> snprintf()
 *   18 Mar 97   Framstag	better URL parsing
 *
 * Various address routines for sendfile.c and sendmsg.c
 *
 * Copyright  1995-1997 Ulli Horlacher
 * This file is covered by the GNU General Public License
 */

#include "config.h"		/* various #defines */

#include <ctype.h>
#include <stdio.h>
#include <pwd.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>

#include "string.h"		/* Extended string functions */
#include "net.h"		/* the network routines */
#include "utf7.h"		/* UTF-7 coding */
#include "message.h"		/* information, warning and error messages */
#include "address.h"		/* address routines */

/* check an alias file */
int check_alias(char *, char *, char *);


extern int 
  verbose,		/* flag for verbose mode */
  quiet;		/* quiet mode flag */

extern char *prg;	/* name of the game */


/*
 * destination  - get recipient user and host
 *
 * INPUT:  argc		- shell argument count
 *         argv		- the shell arguments
 *
 * OUTPUT: user		- own user login and real name
 *         recipient	- recipient user name
 *         host		- recipient host name
 */
void destination(int argc, char **argv,
		 char *user, char *recipient, char *host) {
  char *cp,			/* simple char pointer */
       *at, 			/* @ character in recepient@host */
       *larg,			/* last argument */
       gecos[FLEN],		/* user real name */
       aliasfile[MAXLEN];	/* the alias file */
  struct passwd *pwe;		/* password entry */

  /* get the own user name */
  if ((pwe=getpwuid(getuid())) == NULL)
    message(prg,'F',"could not determine own user name");

  /* translate the real name to UTF-7 and add it */
  iso2utf(gecos,pwe->pw_gecos);
  if ((cp=strchr(gecos,','))) *cp=0;
  snprintf(user,FLEN-1,"%s %s",pwe->pw_name,gecos);

  /* trick: argc <= 0, when called from quak */
  if (argc<=0)
    larg=argv[-argc];
  else
    larg=argv[argc-1];
  
  /* user@host specified? */
  if ((at=strchr(larg,'@'))) {
   
   /* store recipient name and host */
    *recipient=0;
    strncat(recipient,larg,at-larg);
    strcpy(host,at+1);

  /* URL specified? */
  } else if (str_neq_nocase(larg,"saft://",7)) {
    larg+=7;
    cp=strrchr(larg,'/');
    if (!cp || strchr(larg,'@')) message(prg,'F',"illegal SAFT-URL");
    strcpy(recipient,cp+1);
    *cp=0;
    while ((cp=strchr(larg,'/'))) *cp='.';
    strcpy(host,larg);

  /* local user or alias specified */
  } else {
   
    strcpy(recipient,larg);

    /* check the sendfile alias file */
    snprintf(MAXS(aliasfile),SPOOL"/%s/config/aliases",pwe->pw_name);
    if (check_alias(aliasfile,recipient,host)<0) {
     
#ifdef RESPECT_MAIL_ALIASES
      /* check the sendfile alias file */
      snprintf(MAXS(aliasfile),"%s/.elm/aliases.text",pwe->pw_dir);
      if (check_alias(aliasfile,recipient,host)<0) {
       
#endif
	/* store local recipient name and local host */
	/* trick: argc <= 0, when called from quak */
	if (argc<=0)
	  strcpy(recipient,argv[-argc]);
	else
	  strcpy(recipient,argv[argc-1]);
	strcpy(host,"127.0.0.1");

#ifdef RESPECT_MAIL_ALIASES
      }
#endif
    }
  }
}


/*
 * check_alias  - check an alias file
 *
 * INPUT:  aliasfile	- the alias file
 *         recipient	- recipient alias name
 *
 * OUTPUT: recipient	- recipient user name
 *         host		- recipient host name
 */
int check_alias(char *aliasfile, char *recipient, char *host) {
  char *cp, 			/* a character pointer */
       line[MAXLEN],		/* one line of the alias file */
       address[MAXLEN];		/* address from the alias */
  FILE *inf;			/* input file to read */

  *address=0;

  /* if there is an alias file, open it (what else? :-) ) */
  inf=fopen(aliasfile,"r");
  if (inf==NULL) return(-1);

  /* loop over all lines */
  while (fgets(line,MAXLEN-1,inf)) {
   
    /* trim line */
    if ((cp=strchr(line,'\n'))) *cp=0;
    if ((cp=strchr(line,'#'))) *cp=0;
    str_trim(line);

    /* save the address and check for empty or wrong lines */
    cp=strrchr(line,' ');
    if (cp==NULL) continue;
    strcpy(address,cp+1);
    cp=strchr(line,' ');
    if (cp) *cp=0;

    /* is it the correct alias, we are ready */
    if (str_eq(recipient,line))
      break;
    else
      *address=0;

  }
  fclose(inf);

  /* alias found? */
  if (*address) {
   
    /* store recipient name and host */
    cp=strchr(address,'@');
    if (cp) {
      strcpy(host,cp+1);
      *cp=0;
      strcpy(recipient,address);
      return(0);
    }

  }
  return(-1);
}


/*
 * check_forward  - test if there is a forward address set
 *
 * INPUT:  sockfd	- socket file descriptor
 *         recipient	- recipient user name
 *	   host		- host to connect
 * 	   redirect	- redirect comment
 *
 * OUTPUT: recipient	- new recipient user name
 *	   host		- new host to connect
 * 	   redirect	- redirect comment
 *
 * RETURN: 1 if a forward is set, 0 if not
 */
int check_forward(int sockfd, char *recipient, char *host, char *redirect) {
  char
    *at,		/* simple string pointer */
    *reply,		/* reply string from server */
    tmp[MAXLEN];	/* temporary string */

  /* get reply from server */
  reply=getreply(sockfd);
  str_trim(reply);

  /* forward address set? */
  if (str_beq(reply,"510 ")) {
    at=strchr(reply,'@');

    /* is there a forward address? */
    if (at) {
      if (quiet<2) message(prg,'W',"forward address found");
      if (*redirect)
	snprintf(redirect,MAXLEN-1,
		 "%s\r\nredirected by %s@%s",redirect,recipient,host);
      else
	snprintf(redirect,MAXLEN-1,"redirected by %s@%s",recipient,host);

      /* save new recipient and host name */
      *at=0;
      strcpy(recipient,strrchr(reply,' ')+1);
      strcpy(host,at+1);

      /* close current connection */
      sock_putline(sockfd,"QUIT");
      getreply(sockfd);
      shutdown(sockfd,2);
      return(1);

    }
  }

  if (!str_beq(reply,"200 ")) {
    snprintf(MAXS(tmp),"server error: %s",&reply[4]);
    errno=0;
    message(prg,'F',tmp);
  }

  return(0);
}


/*
 * saft_connect  - look for correct SAFT server and open connection
 *
 * INPUT:  type		- file or message
 *         recipient	- recipient user name
 *	   user		- local user name
 *	   host		- host to connect
 * 	   redirect	- redirect comment
 *
 * OUTPUT: recipient	- new recipient user name
 *	   host		- new host to connect
 * 	   redirect	- redirect comment
 * 
 * RETURN: socket file descriptor
 */
int saft_connect(const char *type, 
		 char *recipient, char *user, char *host, char *redirect) {
  int
    sockfd,     	/* socket file descriptor */
    hopcount;		/* count for forwarding addresses */
  char
    answer[FLEN],	/* answer string */
    tmp[MAXLEN],	/* temporary string */
    ahost[MAXLEN],	/* alternate host */
    line[MAXLEN]; 	/* one line of text */
    
  hopcount=0;
  *answer=0;
  
  /* try to connect to the recipient's server */
  for (hopcount=1; hopcount<11; hopcount++) {
     
    /* tell where to send to */
    snprintf(MAXS(tmp),"opening connection to %s@%s",recipient,host);
    if (quiet<2) message(prg,'I',tmp);

    /* initiate the connection to the server */
    sockfd=open_connection(host,SAFT);
    if (sockfd==-3 && str_eq(type,"file")) {
      snprintf(MAXS(tmp),"%s has no internet-address",host);
      if (quiet<2) message(prg,'W',tmp);
      snprintf(MAXS(ahost),"saft.%s",host);
      if (!quiet) {
	printf("try sending to %s@%s ? ",recipient,ahost);
	fgets(answer,FLEN-1,stdin);
      }
      if (tolower(*answer)!='n') {
	strcpy(host,ahost);
	snprintf(MAXS(tmp),"opening connection to %s@%s",recipient,host);
	if (quiet<2) message(prg,'I',tmp);
	sockfd=open_connection(host,SAFT);
      }
    }
    if (sockfd==-1) snprintf(MAXS(tmp),"cannot create a network socket");
    if (sockfd==-2) snprintf(MAXS(tmp),"cannot open connection to %s",host);
    if (sockfd==-3) snprintf(MAXS(tmp),"%s is unknown",host);
    if (sockfd<0) {
      errno=0;
      message(prg,'F',tmp);
    }

    /* no remote server or protocol error? */
    sock_getline(sockfd,line);
    if (verbose && *line) printf("%s\n",line);
    if (!str_beq(line,"220 ") || !strstr(line,"SAFT")) {
      errno=0;
      snprintf(MAXS(tmp),"No SAFT server on port %d at %s",SAFT,host);
      message(prg,'F',tmp);
    }

    /* send constant header lines */
    snprintf(MAXS(tmp),"FROM %s",user);
    sendheader(sockfd,tmp);
    snprintf(MAXS(tmp),"TO %s",recipient);
    sock_putline(sockfd,tmp);

    /* is there a forward set? */
    if (check_forward(sockfd,recipient,host,redirect)) continue;

    /* sendfile connection? */
    if (str_eq(type,"file")) {

      /* test if server can receive files */
      sock_putline(sockfd,"FILE test");
      if (check_forward(sockfd,recipient,host,redirect)) continue;
      
    }

    /* connection is successfull */
    break;

  }

  if (hopcount>10) {
    errno=0;
    message(prg,'F',"maximum hopcount reached (forward loop?)");
  }
  
  return(sockfd);

}
