/*
 * File:	string.c
 *
 * Author:	Ulli Horlacher (framstag@rus.uni-stuttgart.de)
 *
 * Contrib.:    Ingo Wilken (Ingo.Wilken@Informatik.Uni-Oldenburg.DE)
 *		Christoph 'GNUish' Goern (goern@beuel.rhein.de)
 *
 * History:	
 * 
 *   12 Aug 95   Framstag	initial version
 *    5 Feb 96   Framstag	faster str_trim function
 *    7 May 96   Framstag	added strbeq()
 *   14 Jan 97   GNUish		added strerror()
 *    3 Feb 97   Framstag       included snprintf() from apache-source
 *    4 Feb 97   GNUish         added the #ifdef for snprintf()
 *    7 Feb 97   Ingo		improved simplematch()
 *   23 Feb 97   Framstag       modified str_* function names
 *				added str_beq_nocase and str_neq_nocase
 *
 * Extended string functions for the sendfile package, which are not found
 * in the standard C library.
 * Look at string.h for a list of the functions.
 *
 * Copyright  1995-1997 Ulli Horlacher
 * This file is covered by the GNU General Public License
 */

#include <stdlib.h>
#include <ctype.h>
#include <stdio.h>

#include "string.h"

#if !defined(HAVE_SNPRINTF)
#  include "snprintf.c"
#endif

/*
#ifdef IRIX
  int toupper(char);
  int tolower(char);
#endif
*/

/*
 * str_trim - substitute multiple white spaces with one space
 *            and delete heading and trailing whitespace
 *
 * INPUT:  string  - string to trim
 *
 * OUTPUT: string  - trimmed string
 *
 * RETURN: trimmed string
 */
char *str_trim(char *string) {
  char *rp,	/* reading string pointer */
       *wp;	/* writing string pointer */
  int ws=1;	/* white space flag */

  /* loop over string */
  for (wp=rp=string; *rp; rp++) {
   
    /* is it a white space? */
    if (*rp==' ' || *rp=='\t' || *rp=='\n' || *rp=='\r') {
     
      /* was the last character not a white space? */
      if (ws==0) {
       
	/* store a blank */
	*wp++ = ' ';
	ws = 1;
      }
    } else /* no white space */ {
     
      /* store the character */
      *wp++ = *rp;
      ws = 0;
    }
  }

  /* delete trailing blank */
  if (ws) wp--;
  *wp = 0;

  return(string);
}


/*
 * str_toupper - transform string to upper case
 *
 * INPUT:  string  - string to transform
 *
 * OUTPUT: string  - transformed string
 *
 * RETURN: string  - transformed string
 */
char *str_toupper(char *string) {
  char *cp;

  /* change each character to it's upper case pedant */
  for (cp=string; *cp; cp++) *cp = (char) toupper(*cp);

  return(string);
}


/*
 * str_tolower - transform string to lower case
 *
 * INPUT:  string  - string to transform
 *
 * OUTPUT: string  - transformed string
 *
 * RETURN: string  - transformed string
 */
char *str_tolower(char *string) {
  char *cp;

  /* change each character to it's upper case pedant */
  for (cp=string; *cp; cp++) *cp = (char) tolower(*cp);

  return(string);
}


/*
 * str_beq  - string begin equal test
 *
 * INPUT:  s1  - string 1
 *         s1  - string 1
 *
 * RETURN: 1 if begin is equal, 0 if begin is not equal
 */
int str_beq(const char *s1, const char *s2) {
  int len;
  if (strlen(s1)<strlen(s2)) len=strlen(s1); else len=strlen(s2);
  if (strncmp(s1,s2,len)==0) return(1); else return(0);
}


/*
 * str_beq_nocase  - string begin equal test, ignoring case
 *
 * INPUT:  s1  - string 1
 *         s1  - string 1
 *
 * RETURN: 1 if begin is equal, 0 if begin is not equal
 */
int str_beq_nocase(const char *s1, const char *s2) {
  int len;
  if (strlen(s1)<strlen(s2)) len=strlen(s1); else len=strlen(s2);
  return(str_neq_nocase(s1,s2,len));
}


/*
 * str_neq_nocase  - string equal test until length n, ignoring case
 *
 * INPUT:  s1  - string 1
 *         s1  - string 1
 *         len - max length to compare
 *
 * RETURN: 1 if is equal, 0 if is not equal (until size len)
 */
int str_neq_nocase(const char *s1, const char *s2, int len) {
  int i;
  char a,b;
  for (i=0;i<len;i++) {
    a=s1[i];
    b=s2[i];
    if (tolower(a)!=tolower(b)) return(0);
  }
  return(1);
}


/***************************************************************
 *  simplematch:
 *
 *  Einfaches Wildcard-Matching.
 *
 *  ?           matcht jedes beliebige Zeichen
 *  *           matcht eine beliebig lange Folge von beliebigen Zeichen
 *  [abc]  	matcht 'a', 'b' und 'c'
 *              innerhalb [] gelten '?', '*' und '[' als normale Zeichen
 *  \?          testet auf "echtes" '?'   Alternative:  [?]
 *  \*          testet auf "echtes" '*'   Alternative:  [*]
 *  \[          testet auf "echtes" '['   Alternative:  [[]
 *              ']' gilt ausserhalb [] als normales Zeichen, wenn es 
 *              innerhalb [] verwendet werden soll muss ein \ davor
 *  \a          Bell
 *  \b          BackSpace
 *  \f          FormFeed
 *  \n          NewLine
 *  \r          Return
 *  \t          Tabulator
 *  \v          Vertikaltabulator
 *  \\          Schraegstrich
 *
 *  ACHTUNG:
 *  Matching auf "[]" (leere Klammer) schlaegt immer fehl
 *
 * Copyright (C) 1991 by Ingo Wilken (Ingo.Wilken@informatik.uni-oldenburg.de)
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

/*
#include <stddef.h>
#define NULL        (char *)0
*/

#define SM_MATCH        1
#define SM_NOMATCH      0
#define SM_ERROR       -1        /* Fehler im Pattern */


static char sm_escchar( c )
char c;
{
    switch( c ) {
        case 'a': c = '\a'; break;
        case 'b': c = '\b'; break;
        case 'f': c = '\f'; break;
        case 'n': c = '\n'; break;
        case 'r': c = '\r'; break;
        case 't': c = '\t'; break;
        case 'v': c = '\v'; break;
    }
    return(c);
}


int simplematch( text, pattern, nocase )
char *text, *pattern;
int nocase;
{
    char *retrypat = NULL, *retrytxt;
    register char c;
    int notfound;

    retrytxt = text;
    while( *text || *pattern ) {
        c = *(pattern++);
        if( nocase )
            c = tolower(c);

        switch( c ) {
            case '*' :  
                retrypat = pattern;
                retrytxt = text;
                break;
            case '[' :  
                notfound = 1;
                while( (c = *(pattern++)) != ']' ) {
                    if( c == '\\' )
                        c = sm_escchar(*(pattern++));
                    if( c == '\0' )
                        return(SM_ERROR);
                    if( *text == c || (nocase && tolower(*text) == c) )
                        notfound = 0;
                }
                if( notfound ) {
                    pattern = retrypat;
                    text = ++retrytxt;
                }
                /* fallthrough */
            case '?' :  
                if( *(text++) == '\0' )
                    return(SM_NOMATCH);
                break;
            case '\\':  
                c = sm_escchar(*(pattern++));
                if( c == '\0' )
                    return(SM_ERROR);
                /* fallthrough */
            default  :        
                if( *text == c || (nocase && tolower(*text) == c) ) {
                    if( *text )
                        text++;
                }
                else {
                    if( *text ) {
                        pattern = retrypat;
                        text = ++retrytxt;
                    }
                    else
                        return(SM_NOMATCH);
                }
                break;
        }
        if( pattern == NULL )
            return(SM_NOMATCH);
    }
    return(SM_MATCH);
}


#ifdef SM_TEST
#include <stdio.h>

main()
{
    char s[1000],p[1000];

    while(1) {
        printf("Text   : "); fflush(stdout); scanf("%s",s);
        printf("Pattern: "); fflush(stdout); scanf("%s",p);

        printf("Result case-sensitive  : %d\n",simplematch(s,p,0));   
        printf("Result case-insensitive: %d\n\n",simplematch(s,p,1));
    }
}
#endif


#if !defined(HAVE_STRERROR)
/*
 * strerror - like strerror on many Unices, but not on SunOS 4.1.4
 *
 * INPUT:  int     - error number to look up
 *
 * OUTPUT: none
 *
 * RETURN: string  - string of the error number
 */

char *strerror(int errornumber)
{
  extern int sys_nerr;
  extern char *sys_errlist[];

  static char buffer[40];

  if ((unsigned int) errornumber < sys_nerr) {
    return (sys_errlist[errornumber]);
  }

  (void) sprintf(buffer, "Unknown error: %d", errornumber);
  return (buffer);
}
#endif
