/*
 * File:        sendfile.c
 *
 * Author:      Ulli Horlacher (framstag@rus.uni-stuttgart.de)
 *
 * Contrib.:	Rainer Bawidamann (widi@sol.wohnheim.uni-ulm.de)
 * 		Martin Buck (Martin-2.Buck@student.uni-ulm.de)
 * 
 * History:     11 Aug 95   Framstag	initial version
 *              12 Aug 95   Framstag	elm alias support
 *              10 Sep 95   Framstag	added delete and resend function
 *               6 Feb 96   Framstag	added ATTR EXE
 *               7 Feb 96   Framstag	check for already compressed files
 *              20 Feb 96   Framstag	follow forward address if given
 *              21 Feb 96   widi        better Solaris-2 support
 *              22 Feb 96   Framstag    added bouncing (forwarding) of files
 *	        23 Feb 95   mbuck       bug fixes for getting $HOME
 *	        27 Feb 95   Framstag	added quiet options
 *
 * The sendfile client of the sendfile package.
 * Sends one or more files to the sendfiled of the destination system.
 *
 * Copyright  1995 Ulli Horlacher
 * This file is covered by the GNU General Public License
 */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <pwd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "config.h"		/* various #defines */
#include "string.h"		/* extended string functions */
#include "net.h"		/* the network routines */
#include "io.h"			/* (socket) read/write */
#include "message.h"		/* information, warning and error messages */
#include "utf7.h"		/* UTF-7 coding */
#include "destination.h"	/* check recipient and host */

#ifndef AIX
  #ifndef CONVEXOS
    FILE *popen(const char *, const char *);
  #endif
  int pclose(FILE *);
  #if defined(IRIX) || defined(LINUX)
    #include <getopt.h>
  #else
    int getopt(int, char * const *, const char *);
  #endif
#endif

#if !defined(HPUX)
  int gethostname(char *, int);
#endif

/* print short help usage text */
int usage();

/* whereis - search for program name in PATH */
void whereis(const char *, char *);

/* send file data */
int send_data(int, long, const char*, const char*);

/* delete tmp-file */
void cleanup();


/* global variables */
int  verbose,		/* flag for verbose mode */
     quiet;		/* quiet mode */
char *prg,		/* name of the game */
     tartmp[MAXLEN],	/* name of tar temporary file */
     gziptmp[MAXLEN],	/* name of gzipped temporary file */
     texttmp[MAXLEN];	/* name of text temporary file in NVT telnet format */


int main(int argc, char *argv[]) 
{ extern int 
    optind;		/* number of scanned args with getopt */
  int 
    n,          	/* simple loop count */
    pid,        	/* current proccess id */
    sockfd,     	/* socket file descriptor */
    opt,        	/* option to test for */
    fn,         	/* file number in argv */
    ch,         	/* character to read in from file */
    del,        	/* flag for deleting previous sent files */
    tar,        	/* flag for sending files as tar archive */
    exe,        	/* flag for sending executable */
    text,       	/* flag for sending text file */
    source,     	/* flag for sending source code file */
    bounce,     	/* flag for bouncing files */
    compress,   	/* flag for sending compressed */
    do_compress;	/* flag for really do compressing */
  long
    size,       	/* size of file to send */
    orgsize;    	/* original file size uncompressed */
  char
    *cp,*at,		/* simple string pointer */
    *fnp,		/* file name pointer */
    *type,	 	/* type of file */
    *reply,		/* reply string vom server */
    file[FLEN],		/* name of file to send */
    sdfn[FLEN],		/* spool data file name */
    shfn[FLEN],		/* spool header file name */ 
    archive[FLEN],	/* name of archive file */
    recipient[FLEN], 	/* recipient at serverhost */
    user[FLEN], 	/* local user name */
    home[FLEN], 	/* user home directory */
    date[DLEN], 	/* date of file */
    host[FLEN], 	/* name of serverhost */
    localhost[FLEN], 	/* name of the local host */
    cmd[MAXLEN], 	/* cmd string for system-call */
    line[MAXLEN], 	/* one line of text */
    gzip[MAXLEN],	/* full (path-) name of the gzip command */
    tmp[MAXLEN],	/* temporary string */
    userspool[FLEN],	/* user spool directory */
    filelist[OVERSIZE],	/* list of files for tar command */
    *cft[]=		/* file types which are not compressible */
    { "compress","zip","zoo","frozen","gif","jpg","jpeg","mpg","mpeg","" };
  FILE 
    *shf,		/* spool header file */
    *pipe,		/* pipe input stream */
    *inf,		/* input file descriptor */
    *outf;		/* output file descriptor */
  struct passwd *pwe;	/* password entry */
  struct stat finfo;	/* information about a file */
  unsigned char 
    utf_name[LEN_UNI],	/* name in UTF-7 format */
    iso_name[LEN_ISO];	/* name in ISO Latin-1 format */
  
  
  del=0;
  tar=0;
  exe=0;
  text=0;
  quiet=0;
  source=0;
  bounce=0;
  verbose=0;
  compress=1;
  prg=argv[0];
  *date=0;
  *home=0;
  *filelist=0;
  type="BINARY";
  pid=(int)getpid();

  /* scann the command line on options */
  while ((opt=getopt(argc, argv, "vVtasuqQbdh?")) > 0) 
  { switch (opt) 
    { case ':':
      case 'h':
      case '?': exit(usage());
      case 'd': del=1; break;
      case 'a': tar=1; break;
      case 't': text=1; break;
      case 'q': quiet=1; break;
      case 'Q': quiet=2; break;
      case 'b': bounce=1; break;
      case 's': source=1; break;
      case 'v': verbose=1; break;
      case 'u': compress=0; break;
      case 'V': message(prg,'I',"version "VERSION" revision "REVISION); exit(0);
    }
  }
  
  /* too few arguments? */
  if (argc-optind<2) exit(usage());

  /* incompatible options? */
  if (bounce)
  { if (source||text||tar||del) 
      if (quiet<2) message(prg,'W',"you cannot use any other option "
 		                   "when bouncing a file - ignored");
    text=source=tar=compress=del=0;
  }

  if (del)
  { if (source||text||tar) 
      if (quiet<2) message(prg,'W',"you cannot use any other option "
	                           "when deleting a file - ignored");
    text=source=tar=compress=0;
  }

  if (tar) 
  { if (source) 
      if (quiet<2) message(prg,'W',"option SOURCE is not allowed when "
	                           "sending in archive format - ignored");
    if (text) 
      if (quiet<2) message(prg,'W',"option TEXT is not allowed when "
	                           "sending in archive format - ignored");
    text=source=0;
  }
  
  /* get the local host name */
  if (gethostname(localhost,FLEN-1)<0) strcpy(localhost,"localhost");

  /* get own user name, recipient name and host */
  destination(argc,argv,user,recipient,host); 

  /* get the own user name and home */
  if ((pwe=getpwuid(getuid())) == NULL)
    if (quiet<2) message(prg,'F',"cannot determine own user name");
  if ((cp=getenv("HOME")))
    strcpy(home,cp);
  else
    strcpy(home,pwe->pw_dir);
  if (stat(home,&finfo)==0 || !(finfo.st_mode&S_IRWXU)) strcpy(home,"/tmp");
  
  /* set the tmp file names */
  sprintf(tartmp,"%s/.sendfile.tar.%d",home,pid);
  sprintf(gziptmp,"%s/.sendfile.gzip.%d",home,pid);
  sprintf(texttmp,"%s/.sendfile.text.%d",home,pid);
  
  /* enable simple interrupt handler */
  signal(SIGTERM,cleanup);
  signal(SIGABRT,cleanup);
  signal(SIGQUIT,cleanup);
  signal(SIGHUP,cleanup);
  signal(SIGINT,cleanup);
  
  /* look for the compress command */
  if (compress)
  { whereis("gzip",gzip);
    if (!*gzip)
    { compress=0;
      if (quiet<2) message(prg,'W',"no gzip found - sending uncompressed");
    }
  }
  
  /* determine the file type */
  if (text)   type="TEXT";
  if (source) type="SOURCE";
  
  /* initiate the connection to the server */
  sockfd=open_connection(host,SAFT);

  /* name the local host */
  if (streq(host,"127.0.0.1")) strcpy(host,localhost);

  /* no remote server or protocol error? */
  sock_getline(sockfd,line);
  if (verbose && *line) printf("%s\n",line);
  if (!strneq(line,"220 ",4) || !strstr(line,"SAFT"))
  { sprintf(tmp,"No SAFT server on port %d at %s",SAFT,host);
    if (quiet<2) message(prg,'F',tmp);
  }

  /* test if server can receive files */
  sock_putline(sockfd,"FILE test");
  reply=getreply(sockfd);

  /* is there a forward address? */
  if (strneq(reply,"510 ",4))
  { at=strchr(reply,'@');

    if (at)
    { sprintf(tmp,"server %s cannot receive files",host);
      if (quiet<2) message(prg,'W',tmp);

      /* save new host name */
      str_trim(++at);
      if ((cp=strchr(at,' '))) *cp=0;
      strcpy(host,at);

      /* close current connection and open a new one to the real server */
      sock_putline(sockfd,"QUIT");
      getreply(sockfd);
      sockfd=open_connection(host,SAFT);

      /* no remote server or protocol error? */
      sock_getline(sockfd,line);
      if (verbose && *line) printf("%s\n",line);
      if (!strneq(line,"220 ",4) || !strstr(line," SAFT "))
      { sprintf(tmp,"No SAFT server on port %d at %s",SAFT,host);
	if (quiet<2) message(prg,'F',tmp);
      }

      if (verbose) printf("%s\n",line);

    } else /* no forwarding host ==> error */
    { sprintf(tmp,"server error: %s",&reply[4]);
      errno=0;
      if (quiet<2) message(prg,'F',tmp);
    }

  }
  
  /* tell where to send to */
  sprintf(tmp,"sending to %s@%s",recipient,host);
  if (quiet<2) message(prg,'I',tmp);
     
  /* send constant header lines */
  sprintf(tmp,"FROM %s",user);
  sendheader(sockfd,tmp);
  sprintf(tmp,"TO %s",recipient);
  sendheader(sockfd,tmp);
  if (text) sendheader(sockfd,"CHARSET "CHARSET);
  
  /* bouncing files? */
  if (bounce)
  { 				
    /* does the spool directory exist? */
    sprintf(userspool,SPOOL"/%s",pwe->pw_name);
    if (chdir(userspool)<0) 
    { sprintf(tmp,"cannot access spool directory %s",userspool);
      if (quiet<2) message(prg,'F',tmp);
    }
 
    /* main loop over the spool file names */
    for (fn=optind; fn<argc-1; fn++)
    { sprintf(sdfn,"%s.d",argv[fn]);
      sprintf(shfn,"%s.h",argv[fn]);
      
      /* try to open spool header file */
      if ((shf=fopen(shfn,"r")) == NULL) 
      { sprintf(tmp,"cannot open spool file %s",argv[fn]);
	if (quiet<2) message("",'E',tmp);
        continue;
      }
      
      /* scan through spool header file and forward it */
      while (fgets(line,MAXLEN-1,shf))
      { str_trim(line);
	if (line[4]!=' ' || strneq(line,"FROM",4)) continue;
	if (strneq(line,"SIZE",4)) sscanf(line,"SIZE %ld",&size);
	if (strneq(line,"FILE",4)) utf2iso(0,tmp,file,tmp,&line[5]);
	sendheader(sockfd,line);
      }
      
      fclose(shf);

      /* check the file size */
      stat(sdfn,&finfo);
      if (size!=finfo.st_size)
      { sprintf(tmp,"spool file %s has wrong size count - irgnored",argv[fn]);
	if (quiet<2) message(prg,'E',tmp);
	continue;
      }

      /* forward the spool data file */
      if (send_data(sockfd,size,sdfn,file)==0)
      { unlink(shfn);
	unlink(sdfn);
      }

    }		
  }

  /* sending tar-archive */
  else if (tar)
  { 
    /* enough parameters? */
    if (argc-optind<3)
      if (quiet<2) message(prg,'f',"not enough parameters");
    
    /* translate the archive name to UTF-7 */
    strcpy(archive,argv[optind]);
    iso2utf(utf_name,archive);
    
    /* collect all files */
    for (n=optind+1; n<argc-1; n++)
    { strcat(filelist," ");
      strcat(filelist,argv[n]);
    }
    
    /* do the tar, man! :-) */
    if (verbose)
      sprintf(tmp,"tar cvf %s %s",tartmp,filelist);
    else
    { if (quiet==0) printf("making tar archive...\r");
      fflush(stdout);
      sprintf(tmp,"tar cf %s %s",tartmp,filelist);
    }
    if (system(tmp)<0) 
      if (quiet<2) message(prg,'F',"cannot create tar archive file");
    strcpy(file,tartmp);
    
    /* get the original file size */
    stat(tartmp,&finfo);
    orgsize=finfo.st_size;

    /* compressed mode? */
    if (compress)
    { 
      /* compress tar-archive with gzip */
      if (quiet==0) printf("compressing...       \r");
      fflush(stdout);
      sprintf(cmd,"%s < '%s' > %s",gzip,tartmp,gziptmp);
      if (system(cmd))
      { sprintf(tmp,"cannot compress %s",tartmp);
        if (quiet<2) message(prg,'F',tmp);
      }
      strcpy(file,gziptmp);
      
      /* get the compressed file size */
      stat(file,&finfo);
      size=finfo.st_size;
    } else
      size=orgsize;

    /* send the file name and size */
    sprintf(tmp,"FILE %s",utf_name);
    sendheader(sockfd,tmp);
    if (compress) 
      sendheader(sockfd,"TYPE BINARY COMPRESSED");
    else
      sendheader(sockfd,"TYPE BINARY");
    sprintf(tmp,"SIZE %ld %ld",size,orgsize);
    sendheader(sockfd,tmp);
    sendheader(sockfd,"ATTR TAR");
    
    /* send the file data */
    send_data(sockfd,size,file,archive);

  } else 
  { /* sending or deleting single files */

    /* main loop over the file names */
    for (fn=optind; fn<argc-1; fn++)
    { strcpy(file,argv[fn]);
      
      /* get the file name without path */
      fnp=strrchr(file,'/');
      if (!fnp) fnp=file; else fnp++;
      strcpy(iso_name,fnp);
      
      /* translate the file name into UTF-7 */
      iso2utf(utf_name,iso_name);
  
      /* delete option? */
      if (del)
      { sprintf(tmp,"FILE %s",utf_name);
        sendheader(sockfd,tmp);
        sendheader(sockfd,"DEL");
	sprintf(tmp,"file %s deleted",iso_name);
        if (quiet<2) message(prg,'I',tmp);
        continue;
      } 
      else
      { 
	/* is the file readable? */
        if (access(file,R_OK)!=0)
	{ sprintf(tmp,"cannot access file %s",file);
          if (quiet<2) message(prg,'E',tmp);
          continue;
        } 
      
        stat(file,&finfo);

        /* is it a regular file? */
        if (!S_ISREG(finfo.st_mode)) 
	{ sprintf(tmp,"%s is not a regular file, skipping",file);
          if (quiet<2) message(prg,'e',tmp);
          continue;
        } 
	
        /* is it a executable? */
	if (finfo.st_mode&S_IXUSR) exe=1; else exe=0;
	
        /* get the original file size */
        strftime(date,20,"%Y-%m-%d %H:%M:%S",gmtime(&finfo.st_mtime));
	/* if (date[2]<'9') memcpy(date,"20",2); */
  
        /* text or source mode? */
        if (text || source) { 
        
          /* open and test file to send and tmp-file */
          inf=fopen(file,"r");
          outf=fopen(texttmp,"w");
          if (inf==NULL) 
	  { sprintf(tmp,"cannot open file %s",file);
            if (quiet<2) message(prg,'E',tmp);
            break;
          }
          if (outf==NULL) if (quiet<2) message(prg,'F',"cannot open tmp-file");
        
          /* convert file to NVT format */
          do 
          { ch=fgetc(inf);
            if (ch!=EOF) 
            { if (ch=='\n')
                fprintf(outf,"\r\n");
              else
                fputc(ch,outf);
            }
          } while (!feof(inf));
          fclose(inf);
          fclose(outf);
          strcpy(file,texttmp);
        }
      
        /* get the file size */
        stat(file,&finfo);
        orgsize=finfo.st_size;
	  
	do_compress=compress;

	/* check if file is compressible */
        if (compress)
	{ 
	  /* determine file type */
	  sprintf(cmd,"file '%s'",file);
	  if ((pipe=popen(cmd,"r")) && fgets(line,MAXLEN-1,pipe))
	  { str_tolower(line);
	  
	    /* loop over all non-compressible file type strings */
	    for (n=0;*cft[n];n++)
	    
	      /* is this file a not compressible one? */
	      if (strstr(line,cft[n]))
	      { do_compress=0;
		break;
	      }

	  }
	  pclose(pipe);
	}

        /* compressed mode? */
        if (do_compress) 
	{ 
          /* compress tmp-file with gzip */
          if (quiet==0) printf("compressing...       \r");
          fflush(stdout);
          sprintf(cmd,"%s < '%s' > %s",gzip,file,gziptmp);
          if (system(cmd)) 
	  { sprintf(tmp,"cannot compress %s",file);
            if (quiet<2) message(prg,'E',tmp);
            continue;
          }
          strcpy(file,gziptmp);
        
          /* get the compressed file size */
          stat(file,&finfo);
          size=finfo.st_size;
        } else
          size=orgsize;
    
        /* send the file name, size, date and attribute-flag */
        sprintf(tmp,"FILE %s",utf_name);
	sendheader(sockfd,tmp);
	if (do_compress)
	  sprintf(tmp,"TYPE %s COMPRESSED",type);
	else
	  sprintf(tmp,"TYPE %s",type);
	sendheader(sockfd,tmp);
        sprintf(tmp,"SIZE %ld %ld",size,orgsize);
        sendheader(sockfd,tmp);
        sprintf(tmp,"DATE %s",date);
        sendheader(sockfd,tmp);
	if (exe) 
	  sock_putline(sockfd,"ATTR EXE");
	else
	  sock_putline(sockfd,"ATTR NONE");
	getreply(sockfd);
      
        /* send the file data */
        send_data(sockfd,size,file,iso_name);

      }
    }
  }

  /* close the connection */
  sock_putline(sockfd,"QUIT");
  getreply(sockfd);
  close(sockfd);

  /* delete tmp-files */
  cleanup();
  exit(0);
}


/*
 * whereis - search for program name in PATH
 * 
 * INPUT:  name - program name
 *         prg  - empty string
 * 
 * OUTPUT: prg  - full pathname to program
 * 
 */
void whereis(const char *name, char *prg) 
{ char *pp1,		/* path pointer 1 */
       *pp2, 		/* path pointer 2 */
       path[OVERSIZE]; 	/* path name */

  /* begin of PATH string */
  pp1=strcpy(path,getenv("PATH"));

  /* search PATH for programm */
  do 
  { 
    /* cut path-part */
    pp2=strchr(pp1,':');
    if (pp2) *pp2=0;

    /* test if programm is there */
    sprintf(prg,"%s/%s",pp1,name);
    if (access(prg,X_OK)<0) *prg=0;

    /* go to next path-part */
    pp1=pp2+1;

  } while (pp2 && *prg==0);

}


/*
 * send_data - send file data
 * 
 * INPUT:  sockfd	- socket file descriptor
 *         size		- bytes to send
 *         file		- file to send
 *         iso_name	- name of the original file
 * 
 * RETURN: 0 if ok, 1 if already transfered, -1 if transfer failed
 */
int send_data(int sockfd, long  size, const char *file, const char *iso_name) 
{ int
    n,			/* simple loop count */
    nblocks,		/* number of PACKET length blocks */
    bn,			/* block number to read */
    bytes,		/* bytes which has been sent */
    ffd;		/* file to send file descriptor */
  long
    offset;		/* bytes already sent */
  char 
    packet[PACKET],	/* data packet to send */
    tmp[MAXLEN],	/* temporary string */
    *reply;		/* reply string */


  offset=0;
  if (quiet==0) printf("sending...                        \r");

  /* resend? */
  sock_putline(sockfd,"RESEND");
  reply=getreply(sockfd);
  
  /* correct answer? */
  if (!strneq(reply,"500 ",4) && !strneq(reply,"502 ",4)) 
  { 
    /* error occured? */
    if (!strneq(reply,"230 ",4)) 
    { sprintf(tmp,"server error: %s",&reply[4]);
      if (quiet<2) message(prg,'F',tmp);
    }

    sscanf(&reply[4],"%ld",&offset);
  }

  /* prepare sending of data */
  sock_putline(sockfd,"DATA");
  reply=getreply(sockfd);

  /* file already transmitted? */
  if (strneq(reply,"531 ",4)) 
  { sprintf(tmp,"file %s has been already transmitted - ignored.",iso_name);
    if (quiet<2) message(prg,'W',tmp);
    return(1);
  }

  /* server reply ok? */
  if (!strneq(reply,"302 ",4)) 
  { sprintf(tmp,"server error: %s",&reply[4]);
    if (quiet<2) message(prg,'F',tmp);
  }
  
  /* open file */
  ffd=open(file,O_RDONLY,0);
  if (ffd<0 || lseek(ffd,offset,SEEK_SET)<0) 
  { sprintf(tmp,"error reading %s",iso_name);
    if (quiet<2) message(prg,'E',tmp);
    return(-1);
  }
    
  /* send the file data in PACKET size blocks */
  bytes=0;
  size=size-offset;
  nblocks=size/PACKET;
  for (bn=1; bn<=nblocks; bn++) 
  { if (readn(ffd,packet,PACKET)<PACKET) 
    { if (quiet==0) printf("\n");
      sprintf(tmp,"error reading %s",iso_name);
      if (quiet<2) message(prg,'E',tmp);
      close(ffd);
      return(-1);
    }
    if (writen(sockfd,packet,PACKET)<PACKET) 
    { if (quiet==0) printf("\n");
      if (quiet<2) message(prg,'F',"error sending data");
    }
    
    /* print transaction message */
    bytes+=PACKET;
    if (quiet==0) printf("%s: %d bytes\r",iso_name,bytes);
  }
	
  /* copy the last bytes to the spool file */
  if ((n=size-nblocks*PACKET) > 0) 
  { if (readn(ffd,packet,n)<n) 
    { sprintf(tmp,"error reading %s",iso_name);
      if (quiet<2) message(prg,'E',tmp);
      close(ffd);
      return(-1);
    }
    if (writen(sockfd,packet,n)<n)
      if (quiet<2) message(prg,'F',"error sending data");
    bytes += n;
    if (quiet==0) printf("%s: %d bytes\r",iso_name,bytes);
  }

  if (quiet==0) printf("\n");
  close(ffd);

  /* transfer ok? */
  if (!strneq(getreply(sockfd),"201 ",4)) 
  { sprintf(tmp,"transfer failed for %s",iso_name);
    if (quiet<2) message(prg,'E',tmp);
    return(-1);
  }
  
  return(0);
}


/*
 * cleanup - delete tmp-files
 * 
 * A very simple interrupt handler 
 */
void cleanup() 
{ 
#ifndef DEBUG
  unlink(texttmp);
  unlink(gziptmp);
  unlink(tartmp);
#endif
  exit(0);
}


/*
 * usage - print short help usage text
 */
int usage() 
{ fprintf(stderr,"usage: %s [-stduvqQ] file-name [...] user[@host]\n",prg);
  fprintf(stderr,"   or: %s -a [-uvqQ] archive-name file-or-directory-name "
                              "[...] user[@host]\n",prg);
  fprintf(stderr,"options: -s   send file(s) in source mode\n");
  fprintf(stderr,"         -t   send file(s) in text mode\n");
  fprintf(stderr,"         -u   send file(s) uncompressed\n");
  fprintf(stderr,"         -v   verbose mode\n");
  fprintf(stderr,"         -a   send file(s) as one archive\n");
  fprintf(stderr,"         -d   delete previous sent file(s)\n");
  fprintf(stderr,"         -q   quiet mode 1: no transfer messages\n");
  fprintf(stderr,"         -Q   quiet mode 2: no messages at all\n");
  fprintf(stderr,"         ( default mode: send file(s) in binary mode and "
	                   "compressed )\n");
  fprintf(stderr,"example: %s rabbit.gif beate@juhu.lake.de\n",prg);
  return(2);
}
