/*
 * File:        sendfile.c
 *
 * Author:      Ulli Horlacher (framstag@rus.uni-stuttgart.de)
 *
 * History:     11 Aug 95   Framstag      initial version
 *              12 Aug 95   Framstag      elm alias support
 *              10 Sep 95   Framstag      added delete and resend function
 *
 * The sendfile client of the sendfile package.
 * Sends one or more files to the sendfiled of the destination system.
 *
 * This file is covered by the GNU General Public License
 */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "config.h"		/* various #defines */
#include "net.h"		/* the network routines */
#include "io.h"			/* (socket) read/write */
#include "message.h"		/* information, warning and error messages */
#include "utf7.h"		/* UTF-7 coding */
#include "destination.h"	/* check recipient and host */


#ifndef AIX
  #if defined(IRIX) || defined(LINUX)
    #include <getopt.h>
  #else
    int getopt(int, char * const *, const char *);
  #endif
#endif


/* print short help usage text */
int usage();

/* whereis - search for program name in PATH */
void whereis(const char *, char *);

/* send file data */
void send_data(int, int, const char*, const char*);

/* delete tmp-file */
void cleanup();


/* global variables */
int  verbose;		/* flag for verbose mode */
char *prg,		/* name of the game */
     tartmp[MAXLEN],	/* name of tar temporary file */
     gziptmp[MAXLEN],	/* name of gzipped temporary file */
     texttmp[MAXLEN];	/* name of text temporary file in NVT telnet format */


int main(int argc, char *argv[]) 
{ extern int 
    optind;	/* number of scanned args with getopt */
  int 
    n,			/* simple loop count */
    pid,		/* current proccess id */
    sockfd,		/* socket file descriptor */
    opt,		/* option to test for */
    fn,			/* file number in argv */
    ch,			/* character to read in from file */
    size,		/* size of file to send */
    orgsize,		/* original file size uncompressed */
    del,		/* flag for deleting previous sent files */
    tar,		/* flag for sending files as tar archive */
    source,		/* flag for sending source code file */
    text,		/* flag for sending text file */
    compressed;		/* flag for sending compressed */
  char 
    *fnp,		/* file name pointer */
    file[FLEN],		/* name of file to send */
    archive[FLEN],	/* name of archive file */
    recipient[FLEN], 	/* recipient at serverhost */
    user[FLEN], 	/* local user name */
    home[FLEN], 	/* user home directory */
    date[DLEN], 	/* date of file */
    host[FLEN], 	/* name of serverhost */
    type[FLEN], 	/* type of file */
    cmd[MAXLEN], 	/* cmd string for system-call */
    line[MAXLEN], 	/* socket read line */
    gzip[MAXLEN],	/* full (path-) name of the gzip command */
    msg[OVERSIZE],	/* temporary message string */
    filelist[OVERSIZE];	/* list of files for tar command */
  FILE 
    *inf,		/* input file descriptor */
    *outf;		/* output file descriptor */
  struct stat 
    finfo;		/* information about a file */
  unsigned char 
    utf_name[LEN_UNI],	/* name in UTF-7 format */
    iso_name[LEN_ISO];	/* name in ISO Latin-1 format */
  
  
  del=0;
  tar=0;
  text=0;
  source=0;
  verbose=0;
  compressed=1;
  prg=argv[0];
  *date=0;
  *filelist=0;
  pid=(int)getpid();
  strcpy(home,getenv("HOME"));

  /* scann the command line on options */
  while ((opt=getopt(argc, argv, "vtasudh?")) > 0) 
  { switch (opt) 
    { case ':':
      case 'h':
      case '?': exit(usage());
      case 'd': del=1; break;
      case 'a': tar=1; break;
      case 'v': verbose=1; break;
      case 't': text=1; break;
      case 's': source=1; break;
      case 'u': compressed=0; break;
    }
  }
  
  /* too few arguments? */
  if (argc-optind<2) exit(usage());

  /* incompatible options? */
  if (tar) 
  { if (source) 
    { source=0;
      message(prg,'W',"option SOURCE is not allowed when sending in archive"
	              "format - ignored");
    }
    if (text) 
    { text=0;
      message(prg,'W',"option TEXT is not allowed when sending in archive"
	              "format - ignored");
    }
  }
  
  if (del)
  { if (source||text||tar)
    { text=source=tar=compressed=0;
      message(prg,'W',"you cannot use any other option when deleting a file"
	              "  - ignored");
    }
  }

  /* get own user name, recipient name and host */
  destination(argc,argv,user,recipient,host); 

  /* tell where to send to */
  if (strcmp(host,"127.0.0.1")==0)
    sprintf(msg,"sending to %s@localhost",recipient);
  else
    sprintf(msg,"sending to %s@%s",recipient,host);
  message(prg,'I',msg);
     
  /* set the tmp file names */
  sprintf(tartmp,"%s/.sendfile.tar.%d",home,pid);
  sprintf(gziptmp,"%s/.sendfile.gzip.%d",home,pid);
  sprintf(texttmp,"%s/.sendfile.text.%d",home,pid);
  
  /* enable simple interrupt handler */
  signal(SIGTERM,cleanup);
  signal(SIGABRT,cleanup);
  signal(SIGQUIT,cleanup);
  signal(SIGHUP,cleanup);
  signal(SIGINT,cleanup);
  
  /* look for the compress command */
  if (compressed)
  { whereis("gzip",gzip);
    if (!*gzip)
    { compressed=0;
      message(prg,'W',"no gzip found - sending uncompressed");
    }
  }
  
  /* determine the file type */
  strcpy(type,"BINARY");
  if (source) strcpy(type,"SOURCE");
  if (text) strcpy(type,"TEXT");
  if (compressed) strcat(type," COMPRESSED");
  
  /* initiate the connection to the server */
  sockfd=open_connection(host,SAFT);

  /* no remote server or protocol error? */
  if (sock_getline(sockfd,line)<9 ||
      strncmp(line,"220 ",4)!=0 || strstr(line," SAFT ")==NULL)
    message(prg,'f',"cannot connect to remote server");

  if (verbose) printf("%s\n",line);
  
  /* send constant header lines */
  sprintf(msg,"FROM %s",user);
  if (sendheader(sockfd,msg)<0) exit(1);
  sprintf(msg,"TO %s",recipient);
  if (sendheader(sockfd,msg)<0) exit(1);
  sprintf(msg,"TYPE %s",type);
  if (sendheader(sockfd,msg)<0) exit(1);
  if (text)
  { sprintf(msg,"CHARSET "CHARSET);
    if (sendheader(sockfd,msg)<0) exit(1);
  }
  
  /* sending tar-archive or single files? */
  if (tar)
  { 
    /* enough parameters? */
    if (argc-optind<3)
      message(prg,'f',"not enough parameters");
    
    /* translate the archive name to UTF-7 */
    strcpy(archive,argv[optind]);
    iso2utf(utf_name,archive);
    
    /* collect all files */
    for (n=optind+1; n<argc-1; n++)
    { strcat(filelist," ");
      strcat(filelist,argv[n]);
    }
    
    /* do the tar, man! :-) */
    if (verbose)
      sprintf(msg,"tar cvf %s %s",tartmp,filelist);
    else
    { printf("making tar archive...\r");
      fflush(stdout);
      sprintf(msg,"tar cf %s %s",tartmp,filelist);
    }
    if (system(msg)<0) message(prg,'F',"cannot create tar archive file");
    strcpy(file,tartmp);
    
    /* get the original file size */
    stat(tartmp,&finfo);
    orgsize=finfo.st_size;

    /* compressed mode? */
    if (compressed)
    { 
      /* compress tar-archive with gzip */
      printf("compressing...       \r");
      fflush(stdout);
      sprintf(cmd,"%s < '%s' > %s",gzip,tartmp,gziptmp);
      if (system(cmd))
      { sprintf(msg,"cannot compress %s",tartmp);
        message(prg,'F',msg);
      }
      strcpy(file,gziptmp);
      
      /* get the compressed file size */
      stat(file,&finfo);
      size=finfo.st_size;
    } else
      size=orgsize;

    /* send the file name and size */
    sprintf(msg,"FILE %s",utf_name);
    if (sendheader(sockfd,msg)<0) exit(1);
    sprintf(msg,"SIZE %d %d",size,orgsize);
    if (sendheader(sockfd,msg)<0) exit(1);
    sprintf(msg,"ATTR tar");
    if (sendheader(sockfd,msg)<0) exit(1);
    
    /* send the file data */
    send_data(sockfd,size,file,archive);

  } else 
  { /* sending or deleting single files */

    /* main loop over the file names */
    for (fn=optind; fn<argc-1; fn++)
    { strcpy(file,argv[fn]);
      
      /* get the file name without path */
      fnp=strrchr(file,'/');
      if (!fnp) fnp=file; else fnp++;
      strcpy(iso_name,fnp);
      
      /* translate the file name into UTF-7 */
      iso2utf(utf_name,iso_name);
  
      /* delete option? */
      if (del)
      { sprintf(msg,"FILE %s",utf_name);
        if (sendheader(sockfd,msg)<0) exit(1);
        if (sendheader(sockfd,"DEL")==0)
	{ sprintf(msg,"file %s deleted",file);
          message(prg,'I',msg);
          continue;
        } 
      } 
      else
      { 
	/* is the file readable? */
        if (access(file,R_OK)!=0)
	{ sprintf(msg,"cannot access file %s",file);
          message(prg,'E',msg);
          continue;
        } 
      
        /* is it a regular file? */
        stat(file,&finfo);
        if (!S_ISREG(finfo.st_mode)) 
	{ sprintf(msg,"%s is not a regular file, skipping",file);
          message(prg,'e',msg);
          continue;
        } 
            
        /* get the original file size */
        strftime(date,20,"19%y-%m-%d %H:%M:%S",gmtime(&finfo.st_mtime));
	if (date[2]<'9') memcpy(date,"20",2);
  
        /* text or source mode? */
        if (text || source) { 
        
          /* open and test file to send and tmp-file */
          inf=fopen(file,"r");
          outf=fopen(texttmp,"w");
          if (inf==NULL) 
	  { sprintf(msg,"cannot open file %s",file);
            message(prg,'E',msg);
            break;
          }
          if (outf==NULL) message(prg,'F',"cannot open tmp-file");
        
          /* convert file to NVT telnet format */
          do 
          { ch=fgetc(inf);
            if (ch!=EOF) 
            { if (ch=='\n')
                fprintf(outf,"\r\n");
              else
                fputc(ch,outf);
            }
          } while (!feof(inf));
          fclose(inf);
          fclose(outf);
          strcpy(file,texttmp);
        }
      
        /* get the NVT telnet format file size */
        stat(file,&finfo);
        orgsize=finfo.st_size;

        /* compressed mode? */
        if (compressed) 
	{ 
          /* compress tmp-file with gzip */
          printf("compressing...       \r");
          fflush(stdout);
          sprintf(cmd,"%s < '%s' > %s",gzip,file,gziptmp);
          if (system(cmd)) 
	  { sprintf(msg,"cannot compress %s",file);
            message(prg,'E',msg);
            continue;
          }
          strcpy(file,gziptmp);
        
          /* get the compressed file size */
          stat(file,&finfo);
          size=finfo.st_size;
        } else
          size=orgsize;
    
        /* send the file name, size and date */
        sprintf(msg,"FILE %s",utf_name);
        if (sendheader(sockfd,msg)<0) exit(1);
        sprintf(msg,"SIZE %d %d",size,orgsize);
        if (sendheader(sockfd,msg)<0) exit(1);
        sprintf(msg,"DATE %s",date);
        if (sendheader(sockfd,msg)<0) exit(1);
      
        /* send the file data */
        send_data(sockfd,size,file,iso_name);

      }
    }
  }

  /* close the connection */
  sock_putline(sockfd,"QUIT");
  getreply(sockfd);
  close(sockfd);

  /* delete tmp-files */
  cleanup();
  exit(0);
}


/*
 * whereis - search for program name in PATH
 * 
 * INPUT:  name - program name
 *         prg  - empty string
 * 
 * OUTPUT: prg  - full pathname to program
 * 
 */
void whereis(const char *name, char *prg) 
{ char *pp1,		/* path pointer 1 */
       *pp2, 		/* path pointer 2 */
       path[OVERSIZE]; 	/* path name */

  /* begin of PATH string */
  pp1=strcpy(path,getenv("PATH"));

  /* search PATH for programm */
  do 
  { 
    /* cut path-part */
    pp2=strchr(pp1,':');
    if (pp2) *pp2=0;

    /* test if programm is there */
    sprintf(prg,"%s/%s",pp1,name);
    if (access(prg,X_OK)<0) *prg=0;

    /* go to next path-part */
    pp1=pp2+1;

  } while (pp2 && *prg==0);

}


/*
 * send_data - send file data
 * 
 * INPUT:  fd		- socket file descriptor
 *         size		- bytes to send
 *         file		- file to send
 *         iso_name	- name of the original file
 */
void send_data(int fd, int size, const char *file, const char *iso_name) 
{ int
    n,			/* simple loop count */
    nblocks,		/* number of PACKET length blocks */
    bn,			/* block number to read */
    bytes,		/* bytes which has been sent */
    ffd;		/* file to send file descriptor */
  long int 
    offset=0;		/* bytes already sent */
  char 
    packet[PACKET],	/* data packet to send */
    tmp[MAXLEN],	/* temporary string */
    *reply;		/* reply string */

  printf("                        \r");

  /* resend? */
  sock_putline(fd,"RESEND");
  reply=getreply(fd);
  if (strncmp(reply,"500 ",4) && strncmp(reply,"502 ",4)) 
  { 
    /* error occured? */
    if (strncmp(reply,"230 ",4)) 
    { sprintf(tmp,"server error: %s",&reply[4]);
      message(prg,'f',tmp);
    }

    sscanf(&reply[4],"%ld",&offset);
  }

  /* prepare sending of data */
  sock_putline(fd,"DATA");
  reply=getreply(fd);

  /* file already transmitted? */
  if (strncmp(reply,"531 ",4)==0) 
  { sprintf(tmp,"file %s has been already transmitted - ignored.",iso_name);
    message(prg,'W',tmp);
    return;
  }

  /* server reply ok? */
  if (strncmp(reply,"302 ",4)!=0) 
  { sprintf(tmp,"server error: %s",&reply[4]);
    message(prg,'f',tmp);
  }
  
  /* open file */
  ffd=open(file,O_RDONLY,0);
  if (ffd<0 || lseek(ffd,offset,SEEK_SET)<0) 
  { sprintf(tmp,"error reading %s",iso_name);
    message(prg,'E',tmp);
    return;
  }
    
  /* send the file data in PACKET size blocks */
  bytes=0;
  size=size-offset;
  nblocks=size/PACKET;
  for (bn=1; bn<=nblocks; bn++) 
  { if (readn(ffd,packet,PACKET)<PACKET) 
    { printf("\n");
      sprintf(tmp,"error reading %s",iso_name);
      message(prg,'E',tmp);
      close(ffd);
      return;
    }
    if (writen(fd,packet,PACKET)<PACKET) 
    { printf("\n");
      message(prg,'F',"error sending data");
    }
    
    /* print transaction message */
    bytes+=PACKET;
    printf("%s: %d bytes\r",iso_name,bytes);
  }
	
  /* copy the last bytes to the spool file */
  if ((n=size-nblocks*PACKET) > 0) 
  { if (readn(ffd,packet,n)<n) 
    { sprintf(tmp,"error reading %s",iso_name);
      message(prg,'E',tmp);
      close(ffd);
      return;
    }
    if (writen(fd,packet,n)<n)
      message(prg,'F',"error sending data");
      bytes += n;
      printf("%s: %d bytes\r",iso_name,bytes);
  }

  /* transfer ok? */
  printf("\n");
  if (strncmp(getreply(fd),"201 ",4)!=0) 
  { sprintf(tmp,"transfer failed for %s",iso_name);
    message(prg,'E',tmp);
  }
  
  close(ffd);
}


/*
 * cleanup - delete tmp-files
 * 
 * A very simple interrupt handler 
 */
void cleanup() 
{ 
#ifndef DEBUG
  unlink(texttmp);
  unlink(gziptmp);
  unlink(tartmp);
#endif
  exit(0);
}


/*
 * usage - print short help usage text
 */
int usage() 
{ fprintf(stderr,"usage: %s [-stuvd] file-name [...] user[@host]\n",prg);
  fprintf(stderr,"   or: %s -a [-uv] archive-name file-or-directory-name [...] user[@host]\n",prg);
  fprintf(stderr,"options: -s  send file(s) in source mode\n");
  fprintf(stderr,"         -t  send file(s) in text mode\n");
  fprintf(stderr,"         -u  send file(s) uncompressed\n");
  fprintf(stderr,"         -v  verbose mode\n");
  fprintf(stderr,"         -a  send file(s) as one archive\n");
  fprintf(stderr,"         -d  delete previous sent file(s)\n");
  fprintf(stderr,"         ( default mode: send file(s) in binary mode and compressed )\n");
  fprintf(stderr,"example: %s rabbit.gif beate@juhu.lake.de\n",prg);
  return(2);
}
