/*
 * File:	net.c
 * 
 * Author:	Ulli Horlacher (framstag@rus.uni-stuttgart.de)
 * 
 * History:	11 Aug 95   Framstag	initial version
 *         	10 Sep 95   Framstag	some debugging 
 *         	15 Nov 95   Framstag	improved sock_getline
 *         	21 Dec 95   Framstag	simplified sock_getline and getreply
 * 
 * Network routines for the the sendfile client of the sendfile package.
 * Look at net.h for list of the functions.
 * 
 * This file is covered by the GNU General Public License
 */

/*
#ifdef NEXT
  typedef unsigned char	 u_char;
  typedef unsigned short u_short;
  typedef unsigned int	 u_int;
  typedef unsigned long	 u_long;
  typedef long	daddr_t;
  typedef char *caddr_t;
  typedef long  time_t;
  #define _TIME_T
#endif
*/

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <netdb.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "config.h"	/* various definitions */
#include "message.h"	/* information, warning and error messages */
#include "net.h"	/* network stuff */
#include "io.h"		/* socket read/write */


/* stupid AIX comes with no include files for networking and other stuff */
#if defined(AIX) || defined(ULTRIX)
  #include "bsd.h"
#endif

/*
#ifdef IRIX
  u_short htons(u_short hostshort);
#endif
*/

/*
 * open_connection - open socket and connect to client
 * 
 * INPUT:  adr  - ip address of server to connect to
 *         port - port number to connect to
 * 
 * RETURN: socket file descriptor, -1 if failled
 * 
 * this function is derived from example code from
 * "Unix Networking Programming" by W. R. Stevens
 */
int open_connection(char *adr, int port) 
{ int sockfd,			/* socket file descriptor */
      num=1;			/* flag for numeric ip address */
  struct sockaddr_in serv_addr;	/* internet socket */
  struct in_addr hostaddr;
  struct hostent *hostp;	/* host entity */
  char *cp,			/* character pointer */
       hostname[16],		/* server host name */
       msg[MAXLEN];		/* intermediate information/error message */
  extern char *prg;		/* name of the game */

  /* open socket */
  sockfd=socket(AF_INET,SOCK_STREAM,0);
  if (sockfd<0) message(prg,'F',"cannot open connection");
#ifdef DEBUG
  message(prg,'I',"socket ok");
#endif  

  /* initialisize serv_addr */
  memset((char *) &serv_addr, 0, sizeof(serv_addr));
  
  /* numeric oder symbolic ip address? */
  for (cp=adr; *cp>0; cp++) 
  { if (*cp>'@')
    { num=0;
      break;
    }
  }

  /* look for server host address */
  if (num) 
  { hostaddr.s_addr=inet_addr(adr);
    hostp=gethostbyaddr((char *)&hostaddr,sizeof(hostaddr),AF_INET);
  } else
    hostp=gethostbyname(adr);
  if (hostp==NULL) 
  { errno=0;
    sprintf(msg,"unknown host %s",adr);
    message(prg,'F',msg);
  }
  
  /* convert binary structure to ASCII hostname */
  strcpy(hostname,inet_ntoa(*(struct in_addr *) *hostp->h_addr_list));
#ifdef DEBUG
  printf("host: %s\n",hostname);
#endif  
  
  /* fill out server address descriptor */
  serv_addr.sin_family     =AF_INET;
  serv_addr.sin_addr.s_addr=inet_addr(hostname);
  serv_addr.sin_port       =htons(SAFT);
  
  /* connect to server */
  if (connect(sockfd,(struct sockaddr *) &serv_addr,sizeof(serv_addr)) < 0)
    return(-1);
#ifdef DEBUG
  message(prg,'I',"connect ok");
#endif  

  return(sockfd);
}


/*
 * sock_getline - get a (command) line from the network socket
 * 
 * INPUT:  fd     - socket file descriptor
 *         line   - empty string
 * 
 * OUTPUT: line   - read string
 * 
 * RETURN: number of read bytes, -1 on error
 * 
 * this function is derived from example code from
 * "Unix Networking Programming" by W. R. Stevens
 */
int sock_getline(int fd, char *line) 
{ int n, rc;
  unsigned char c;

  *line=0;

  for (n=0; n+1<MAXLEN; n++) 
  { rc=read(fd,&c,1);
    if (rc==1) 
    { line[n]=c;
      if (c=='\n') break;
    } else 
      return(-1);
  }

  /* too much input? */
  if (n+1==MAXLEN && line[n] != '\n') 
  { errno=0;
    message("",'F',"network socket data overrun");
  }

  /* remove trailing cr or lf */
  line[n]=0;
  if (n>0 && line[n-1]=='\r') line[--n]=0;

  return(n);
}


/*
 * sock_putline - send a line to the network socket
 * 
 * INPUT:  fd     - socket file descriptor
 *         line   - string to send
 * 
 * RETURN: number of send bytes
 */
int sock_putline(int fd, char *line) 
{ int n;		/* number of send bytes */
  char cmd[MAXLEN];	/* command line to send */
  extern int verbose;	/* flag for verbose mode */
  
  /* prepare string */
  strcpy(cmd,line);
  strcat(cmd,"\r\n");
  
  /* on verbose mode show what goes up */
  if (verbose) printf("-> %s\n",line);
  
  /* and up and away :-) */
  n=writen(fd,cmd,strlen(cmd));
  
  return(n);
}


/* 
 * getreply - get the reply on a command from the server
 * 
 * INPUT:  fd - socket file descriptor
 * 
 * RETURN: the reply line string
 */
char *getreply(int fd) 
{ int len;			/* reply message length */
  char msg[MAXLEN];		/* intermediate information/error message */
  static char reply[MAXLEN];	/* reply string from server */
  extern int verbose;		/* flag for verbose mode */
  extern char *prg;		/* name of the game */

  do 
  {
    /* get the next reply line */
    len=sock_getline(fd,reply);

    /* link failure? */
    if (len<0) 
    { errno=0;
      sprintf(msg,"server has closed the connection, last data: \"%s\"",reply);
      message("",'F',msg);
    }

    /* reply message too short? */
    if (len<4) 
    { errno=0;
      sprintf(msg,"corrupt reply: \"%s\"",reply);
      message(prg,'F',msg);
    }
  
    /* on verbose mode show the whole line */
    if (verbose) printf("%s\n",reply);
  
    /* quit if there was a fatal server error */
    if (reply[0]=='4') 
    { errno=0;
      sprintf(msg,"server error: %s",&reply[4]);
      message(prg,'F',msg);
    }

  } while (reply[3]=='-');
  
  return(reply);
}


/*
 * sendheader - send a headerline and check the reply code
 * 
 * INPUT:  fd	- socket file descriptor
 *         line	- header line
 * 
 * RETURN: 0 on sucess, -1 on server error
 */
int sendheader(int fd, char *line) 
{ char msg[MAXLEN],		/* intermediate information/error message */
       *reply;			/* reply string from server */
  extern char *prg;		/* name of the game */
  
  /* send the header line */
  sock_putline(fd,line);
  
  /* server reply ok? */
  reply=getreply(fd);
  if (strncmp(reply,"200 ",4)!=0) 
  { errno=0;
    sprintf(msg,"server error: %s",&reply[4]);
    message(prg,'F',msg);
  } 
  
  return(0);
}
