/*
 * File:	sendmsg.c
 * 
 * Author:	Ulli Horlacher (framstag@rus.uni-stuttgart.de)
 * 
 * History:	11 Aug 95   Framstag	initial version
 *              12 Aug 95   Framstag	elm alias support
 *               2 Nov 95   Framstag	added minimal chat mode
 *              14 Nov 95   Framstag	added message receiving modes
 *              21 Dec 95   Framstag	avoid unnecessary error message while
 *                                      configuring the own tty
 *              21 Dec 95   Framstag	better server connect testing
 * 
 * The sendmessage client of the sendfile package.
 * Sends a single line text message to the SAFT-server of the destination
 * system to be displayed on the recipients terminal.
 * 
 * This file is covered by the GNU General Public License
 */


#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#ifndef CONVEXOS
  #include <sys/stat.h>
#endif

#include "config.h"		/* various #defines */
#include "net.h"		/* the network routines */
#include "io.h"			/* (socket) read/write */
#include "message.h"		/* information, warning and error messages */
#include "utf7.h"		/* UTF-7 coding */
#include "destination.h"	/* check recipient and host */

#ifndef AIX
  #ifndef CONVEXOS
    FILE *popen(const char *, const char *);
  #endif
  int pclose(FILE *);
  #if defined(IRIX) || defined(LINUX)
    #include <getopt.h>
  #else
    int getopt(int, char * const *, const char *);
  #endif
#endif


/* print short help usage text */
int usage();


/* global variables */
int verbose;		/* flag for verbose mode */
char *prg;		/* name of the game */


int main(int argc, char *argv[]) 
{ extern int 
    optind;		/* number of scanned args with getopt */
  int
    sockfd,		/* socket file descriptor */
    chat,		/* flag for chat mode */
    receive,		/* receiving flag */
    opt;		/* option to test for */
  char 
    *cp,		/* simple character pointer */
    recipient[FLEN], 	/* recipient at serverhost */
    user[FLEN], 	/* local user name */
    host[FLEN], 	/* name of serverhost */
    msgcf[FLEN], 	/* message control file */
    line[MAXLEN],	/* input or output string */
    login[MAXLEN],	/* login user name */
    tmp[MAXLEN];	/* temporary string */
  unsigned char 
    utf_msg[LEN_UTF],	/* msg in UTF-7 format */
    iso_msg[LEN_ISO];	/* msg in ISO Latin-1 format */
  FILE *pipe=NULL;	/* pipe file descriptor */

  prg=argv[0];
  chat=0;
  verbose=0;
  receive=0;

  /* scan the command line on options */
  while ((opt=getopt(argc,argv,"cvmMh?")) > 0) 
  { switch (opt) 
    { case ':':
      case 'h':
      case '?': exit(usage());
      case 'v': verbose=1; break;
      case 'm': receive=1 ; break;
      case 'M': receive=2 ; break;
      case 'c': chat=1;
    }
  }

  /* too few arguments? */
  if (argc-optind<1 && receive==0) exit(usage());

  /* get own user name, recipient name and host */
  destination(argc,argv,user,recipient,host);
  strcpy(login,user);
  if ((cp=strchr(login,' '))) *cp=0;

  /* test the local sendfiled */
  if (verbose) printf("testing local SAFT server:\n");
  sockfd=open_connection("127.0.0.1",SAFT);
  sock_getline(sockfd,line);
  if (verbose && *line) printf("%s\n",line);
  
  /* no local server? */
  if (strncmp(line,"220 ",4)!=0 || strstr(line," SAFT ")==NULL)
    message(prg,'W',"there is no local SAFT server - "
	            "you cannot receive messages");
  else 
  { 
    /* test if you can receive messages */
    sprintf(line,"FROM %s",login);
    sock_putline(sockfd,line);
    sock_getline(sockfd,line);
    if (verbose) printf("%s\n",line);
    sprintf(line,"TO %s",login);
    sock_putline(sockfd,line);
    sock_getline(sockfd,line);
    if (verbose) printf("%s\n",line);
    if (strncmp(line,"521 ",4)==0)
      message(prg,'f',"You are not allowed to use the sendmsg service");
    if (strncmp(line,"200 ",4)!=0) 
      message(prg,'W',"local server error - you cannot receive messages");
    close(sockfd);

    /* allow receiving of messages on this tty */
    sprintf(msgcf,"%s/%s/config/msg",SPOOL,login);
    sprintf(tmp,"( mesg y && tty >%s ) 2>&1",msgcf);
    pipe=popen(tmp,"r");
    
    /* error when configuring the tty? */
    if (fgets(line,MAXLEN-1,pipe)) 
    { if ((cp=strrchr(line,'\n'))) *cp=0;
      sprintf(tmp,"error while configuring your tty: %s",line);
      message(prg,'e',tmp);
    } else 
    { 
      if (receive==1 && argc-optind<1)
	message(prg,'I',"receiving messages is now possible only on this tty");
  
      if (receive==2) 
      { unlink(msgcf);
	if (argc-optind<1)
	  message(prg,'I',"receiving messages is now possible on all ttys");
      }

    }
  }

  pclose(pipe);
  if (receive>0 && argc-optind<1) exit(0);

  /* tell where to send to */
  if (strcmp(host,"127.0.0.1")==0)
    sprintf(tmp,"sending to %s@localhost",recipient);
  else
    sprintf(tmp,"sending to %s@%s",recipient,host);
  message(prg,'I',tmp);
  
  /* initiate the connection to the remote server */
  sockfd=open_connection(host,SAFT);

  /* no remote server? */
  if (sock_getline(sockfd,line)<=0)
    message(prg,'f',"cannot connect to remote server");
  if (verbose) printf("%s\n",line);
  
  /* remote server protocol error? */
  if (strncmp(line,"220 ",4)!=0 || strstr(line," SAFT ")==NULL)
    message(prg,'f',"remote protocol error");
  
  /* send FROM and TO headers */
  sprintf(tmp,"FROM %s",user);
  if (sendheader(sockfd,tmp)<0) exit(1);
  sprintf(tmp,"TO %s",recipient);
  if (sendheader(sockfd,tmp)<0) exit(1);

  do 
  { /* read the message */
    printf("message: ");
    *iso_msg=0;
    fgets(iso_msg,LEN_ISO,stdin);

    /* message empty? */
    if (*iso_msg==0) 
    { printf("\n");
      break;
    }
    if (*iso_msg=='\n') continue;

    /* strip off new line */
    cp=strrchr(iso_msg,'\n');
    if (cp && (cp!=(char *)iso_msg)) *cp=0;

    /* encode to UTF-7 */
    iso2utf(utf_msg,iso_msg);
      
    /* send the message */
    sprintf(tmp,"MSG %s",utf_msg);
    sendheader(sockfd,tmp);
  } while (chat);

  /* close the connection */
  sock_putline(sockfd,"QUIT");
  getreply(sockfd);
  close(sockfd);

  exit(0);
}


/*
 * usage - print short help usage text
 */
int usage() 
{ fprintf(stderr,"usage: %s [-v] user[@host]\n",prg);
  fprintf(stderr,"   or: %s [-mM]\n",prg);
  fprintf(stderr,"options: -v  verbose mode\n");
  fprintf(stderr,"         -m  receive messages only on this tty (default)\n");
  fprintf(stderr,"         -M  receive messages on other ttys, too\n");
  fprintf(stderr,"example: %s orakel@main01.bb.bawue.de\n",prg);
  return(2);
}
