/*
 * File:	sendfiled.c
 * 
 * Author:	Ulli Horlacher (framstag@rus.uni-stuttgart.de)
 * 
 * History:	11 Aug 95   Framstag	initial version
 *              10 Sep 95   Framstag	added delete and resend function
 *               1 Nov 95   Framstag	eliminated some security flaws and
 *                                      added pgp signature entry
 *               5 Nov 95   Framstag	added NeXT support
 *              14 Nov 95   Framstag	added user config files
 *              21 Nov 95   Framstag	added chat client support
 * 
 * The sendfile-daemon of the sendfile package.
 * Receive files for a specified recipient, stores them in the sendfile spool
 * directory and inform the recipient.
 * Receive messages for a specified recipient and display them on the 
 * recipients terminal.
 * 
 * This file is covered by the GNU General Public License
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/utsname.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <netdb.h>
#include <fcntl.h>
#include <time.h>
#include <utmp.h>
#include <dirent.h>
#include <pwd.h>
#include <arpa/inet.h>
#include <netinet/in.h>

#include "config.h"     /* various definitions */
#include "reply.h"	/* the 3 digit reply codes with text messages */
#include "peername.h"	/* get the name of the calling host */
#include "string.h"	/* extended string functions */
#include "spool.h"	/* operations on files in the sendfile spool */
#include "io.h"		/* (socket) read/write */
#include "utf7.h"       /* UTF-7 coding */

/* stupid AIX comes with no include files for networking and other stuff */
#if defined(AIX) || defined(ULTRIX)
  #include "bsd.h"
#endif

#ifdef NEXT
  int gethostname(char*, int);
#endif

#ifdef HPUX
  #define seteuid(a) setuid(a)
#endif

#ifndef _PATH_UTMP
  #define _PATH_UTMP "/etc/utmp"
#endif


/* global variables */
char *prg,			/* name of the game (not used) */
     userspool[MAXLEN],		/* user spool directory */
     userconfig[MAXLEN];	/* user config directory */
int client=0;			/* flag to determine client or server */


/* get a command line */
int getline(char *);

/* write one line to header spool file */
void writeheader(int, const char *, const char *);

/* write a message to all ttys of the user */
int msg2tty(const char *, const char *, int);

/* get the next spool id number */
int spoolid();

/* check killfile */
int restricted(const char *, const char *, char);

/* write-lock a file */
int wlock_file(int);

/* test the lock status of a file */
int tlock_file(int);

/* missed in <unistd.h> */
int seteuid(uid_t);


/* 
 * general remarks:
 *   - "string" always means a pointer to a character vector 
 *   - a 4xx reply code (fatal error) closes the connection
 */


int main() 
{ int 
    size,			/* size of the file */
    bytes,			/* number of bytes to receive */
    shfd,			/* spool header file descriptor */
    sdfd,			/* spool data file descriptor */
    lfd,			/* log file descriptor */
    nblocks,			/* number of PACKET length blocks */
    bn,				/* block number to read */
    transmitted,		/* bytes already transmitted */
    notify,			/* flag for sending a message */
    del_success,		/* flag for successfull deleting a file */
    flags,			/* source, text, compress and tar flag */
    id,				/* spool file id */
    n;				/* simple loop variable */
  char 
    *peer,			/* sender host name */
    line[MAXLEN],		/* incoming command line */
    arg[MAXLEN],		/* the argument(s) of the command line */
    cmd[MAXLEN],		/* the command itself */
    type[MAXLEN],		/* file type: binary, source or text */
    sizes[MAXLEN],		/* original and compressed file size */
    charset[MAXLEN],		/* name of the character set */
    attribute[MAXLEN],		/* tar attribute */
    date[MAXLEN],		/* date string */
    currentdate[MAXLEN],	/* current date */
    shfile[MAXLEN],		/* spool header file name */
    sdfile[MAXLEN],		/* spool data file name */
    tmp[3*MAXLEN], 		/* temporary string */
    real[MAXLEN],		/* sender real name */
    sig[MAXLEN],		/* authentification signature */
    sender[MAXLEN],		/* user@senderhost */
    filename[MAXLEN],		/* file name in UTF-7 */
    recipient[MAXLEN],		/* local user */
    user[MAXLEN],		/* check local user */
    host[MAXLEN],		/* host name */
    dummy[MAXLEN],		/* dummy string for utf2iso name conversion */
    packet[PACKET],		/* data packet to read */
    msg[3*PACKET],		/* message to user-tty */
    *cp;			/* simple string pointer */
  struct passwd *pwe;		/* password entry struct */
  struct stat finfo;		/* information about a file */
  struct filelist *flp;		/* file list pointer */
  struct senderlist *sls;	/* sender list start */
#ifndef NEXT
  struct utsname sysname;	/* system information */
#else
  FILE *pp;
#endif
  time_t timetick;		/* unix time */
  FILE *inf;			/* for various files */

  /* set default values */
  notify = 0;
  flags = 0;
  transmitted = 0;
  *sender = 0;
  *recipient = 0;
  *filename = 0;
  *attribute = 0;
  sls = NULL;
  flp = NULL;
  pwe = NULL;
  strcpy(charset,CHARSET);
  strcpy(type,"BINARY");
#ifdef NEXT
  if (gethostname(host,MAXLEN-1)<0) strcpy(host,"UNKNOWN");
#else
  uname(&sysname);
  strcpy(host,sysname.nodename);
#endif
  /* get the actual UTC time */
  timetick = time(0);
  strftime(date,20,"19%y-%m-%d %H:%M:%S",gmtime(&timetick));
  if (date[2]<'9') memcpy(date,"20",2);
  
  /* send the server ready message */
  reply(220);

  /* main loop for command line parsing and getting data */
  while (1) 
  { 
    /* get the next command line from the client */
    if (getline(line)==0) continue;

    /* extract the command name and the argument */
    
    str_trim(line);
    strcpy(cmd,line);
    cp = strchr(cmd,' ');
    if (cp) 
    { *cp = 0;
      strcpy(arg,strchr(line,' ')+1);
    } else 
      *arg = 0;
    str_toupper(cmd);

    /* QUIT command? */
    if (strcmp(cmd,"QUIT")==0) 
    { 
      /* is there a notify message to send? */
      if (notify) 
      { 
#ifdef HAEGAR
        sprintf(msg,"\r\n\n>>> INFO <<< SAFT-Daemon has a new file for you!"
		    "\n\nFrom: %s ( saft://%s/%s )\n\nType \"receive\".\r\n\n",
		    sender,host,recipient);
#else
        sprintf(msg,"\r\n%s has sent you (%s@%s) a file. Type \"receive\".\r\n",
		sender,recipient,host);
#endif	
        msg2tty(recipient,msg,pwe->pw_uid);
      }

      reply(221);
      exit(0);
    }
    
    /* HELP command? */
    if (strcmp(cmd,"HELP")==0 || strcmp(cmd,"?")==0) 
    { reply(214);
      continue;
    }
    
    /* insider joke :-) */
    if (strcmp(cmd,"SLAVE")==0) 
    { reply(203);
      continue;
    }
    
    /* TO command? */
    if (strcmp(cmd,"TO")==0) 
    { 
      /* is there an argument? */
      if (*arg==0) 
      { reply(505);
	continue;
      }
      
      /* is there a notify message to send to the last recipient? */
      if (notify) 
      { 
#ifdef HAEGAR
        sprintf(msg,"\r\n\n>>> INFO <<< SAFT-Daemon has a new file for you!"
		    "\n\nFrom: %s ( saft://%s/%s )\n\nType \"receive\".\r\n\n",
		    sender,host,recipient);
#else
        sprintf(msg,"\r\n%s has sent you (%s@%s) a file. Type \"receive\".\r\n",
		sender,recipient,host);
#endif	
        msg2tty(recipient,msg,pwe->pw_uid);
	notify = 0;
      }

      /* convert recipient name from UTF-7 to ISO Latin 1 */
      utf2iso(0,recipient,dummy,dummy,arg);

      /* get user information */
      pwe = NULL;
#ifdef NEXT
      sprintf(tmp,"( nidump passwd . ; nidump passwd / ) | "
	          " awk -F: '$1==\"%s\"{print $3,$4;exit}'",recipient);
      pp = popen(tmp,"r");
      if (fgets(tmp,MAXLEN-1,pp) && *tmp!='\n') 
      { int uid,gid;
	pwe = (struct passwd *) malloc(sizeof(struct passwd));
	sscanf(tmp,"%d %d",&uid,&gid);
	pwe->pw_uid = uid;
	pwe->pw_gid = gid;
      }
      pclose(pp);
#else
        pwe = getpwnam(recipient);
#endif
      /* does the user exist? */
      if (pwe==NULL) 
      { reply(520);
        *recipient = 0;
	continue;
      }
      
      /* look in the sendfile disallow file */
      if ((inf=fopen(NOSENDFILE,"r")) != NULL) 
      { do 
	{ if (fgets(user,MAXLEN-1,inf)==NULL) break;
	  cp = strchr(user,'\n');
	  if (cp) *cp = 0;
        } while (strcmp(user,recipient)!=0);
        fclose(inf);
	
	/* is the recipient allowed to receive files? */
	if (strcmp(user,recipient)==0) 
	{ reply(521);
	  *recipient = 0;
	  continue;
	}

      }

      /* build user spool string */
      recipient[32] = 0;
      sprintf(userspool,SPOOL"/%s",recipient);
      sprintf(userconfig,"%s/config",userspool);
  
      /* create user spool directory for recipient if necessary */
      if (stat(userspool,&finfo)<0) 
        if (mkdir(userspool,S_IRUSR|S_IWUSR|S_IXUSR)<0) reply(411);
      mkdir(userconfig,S_IRUSR|S_IWUSR|S_IXUSR);
	  
      /* give the ownership of the user spool directory to recipient */
      chown(userspool,pwe->pw_uid,pwe->pw_gid);
      chown(userconfig,pwe->pw_uid,pwe->pw_gid);
      
      reply(200);
      continue;
    }
    
    /* FROM command? */
    if (strcmp(cmd,"FROM")==0) 
    { 
      /* is there an argument? */
      if (*arg==0) 
      { reply(505);
	continue;
      }

      *sig = 0;
      *real = 0;
      
      /* is there a real name? */
      if ((cp=strchr(arg,' '))) 
      { strcpy(tmp,cp+1);
	*cp = 0;
	
	/* is there a signature? */
	if ((cp=strrchr(tmp,' '))) 
	{ strcpy(sig,cp+1);
	  *cp = 0;
	}
	
	utf2iso(0,dummy,real,dummy,tmp);
      }
      
      /* save sender@host */
      peer = peername(0);
      if (strlen(arg)+strlen(peer)+strlen(real)+4<MAXLEN)
        sprintf(sender,"%s@%s (%s)",arg,peer,real);
      else
	sprintf(sender,"???@%s",peer);      

      reply(200);
      continue;
    } 

    /* CHARSET command? */
    if (strcmp(cmd,"CHARSET")==0) 
    { 
      /* is there an argument? */
      if (*arg==0) 
      { reply(505);
	continue;
      }
      
      /* save the charset name */
      strcpy(charset,arg);
      reply(200);
      continue;
    } 

    /* DATE command? */
    if (strcmp(cmd,"DATE")==0) 
    { 
      /* is there an argument? */
      if (*arg==0) 
      { reply(505);
	continue;
      }
      
      /* save the date */
      utf2iso(0,dummy,date,dummy,arg);
      reply(200);
      continue;
    } 

    /* FILE command? */
    if (strcmp(cmd,"FILE")==0) 
    { 
      /* is there an argument? */
      if (*arg==0) 
      { reply(505);
	continue;
      }
      
      /* save the filename */
      strcpy(filename,arg);
      reply(200);
      continue;
    } 

    /* ATTR command? */
    if (strcmp(cmd,"ATTR")==0) 
    { 
      /* is there an argument? */
      if (*arg==0) 
      { reply(505);
	continue;
      }
      
      /* tar attribute? */
      str_toupper(arg);
      if (strcmp(arg,"TAR")==0) 
      { strcpy(attribute,arg);
	flags=flags|F_TAR;
        reply(200);
        continue;
      } 
      
      /* reset attribute? */
      if (strcmp(arg,"NONE")==0) 
      { *attribute=0;
	flags=flags&!F_TAR;
        reply(200);
        continue;
      }
      
      reply(504);
      continue;
    } 

    /* TYPE command? */
    if (strcmp(cmd,"TYPE")==0) 
    { 
      /* parse the type command and check if it is valid */
      str_toupper(arg);
      if (strcmp(arg,"BINARY") && strcmp(arg,"BINARY COMPRESSED") &&
          strcmp(arg,"SOURCE") && strcmp(arg,"SOURCE COMPRESSED") &&
          strcmp(arg,"TEXT")   && strcmp(arg,"TEXT COMPRESSED")) 
      { 	
	/* wrong type format */
	reply(501);
	continue;
      }

      /* save the flags */
      flags=flags&!F_SOURCE&!F_TEXT&!F_COMPRESS;
      if (strstr(arg,"TEXT"))       flags=flags|F_TEXT;
      if (strstr(arg,"SOURCE"))     flags=flags|F_SOURCE;
      if (strstr(arg,"COMPRESSED")) flags=flags|F_COMPRESS;

      /* save the type */
      strcpy(type,arg);
      reply(200);
      continue;
    }

    /* SIZE command? */
    if (strcmp(cmd,"SIZE")==0) 
    { 
      /* is there an argument? */
      if (*arg==0) 
      { reply(505);
	continue;
      }
      
      /* are the arguments correct? */
      
      { int s1,s2;
	if (sscanf(arg,"%d %d",&s1,&s2)!=2) 
	{ reply(501);
	  continue;
	}
      }

      /* save the size(s) */
      sscanf(arg,"%d",&size);
      strcpy(sizes,arg);
      transmitted = 0;

      reply(200);
      continue;
    }

    /* MSG command? */
    if (strcmp(cmd,"MSG")==0) 
    { 
      /* is there an argument? */
      if (*arg==0) 
      { reply(505);
	continue;
      }

      /* sender and recipient already specified? */
      if (*sender==0 || *recipient==0) 
      { reply(503);
	continue;
      }

      /* check killfile */
      if (restricted(sender,recipient,'m')) 
      { reply(523);
	continue;
      }
      
      /* convert message from UTF-7 to ISO Latin 1 and kill control codes */
      utf2iso(0,dummy,tmp,dummy,arg);
      timetick = time(0);
      strftime(currentdate,9,"%H:%M",localtime(&timetick));
      sprintf(msg,"\r\nMessage from %s at %s :\r\n%s\r\n",sender,currentdate,tmp);
      
      /* try to send to recipient ttys */
      if (msg2tty(recipient,msg,pwe->pw_uid)<0)
        reply(522);
      else
        reply(200);
      continue;
    }

    /* DELETE command? */
    if (strcmp(cmd,"DEL")==0) 
    { 
      /* sender, recipient and filename already specified? */
      if (*sender==0 || *recipient==0 || *filename==0) 
      { reply(503);
	continue;
      }

      transmitted = 0;
      del_success = -1;

      /* are there any files in the spool directory? */
      if ((sls=scanspool(sender))) 
      { 
	/* loop over files list */
        for (flp=sls->flist; flp!=NULL; flp=flp->next) 
	{ 
	  /* if file found try to delete spool file */
          if (strcmp(filename,flp->fname)==0) del_success*=delete_sf(flp,0);

	}
      }
      
      if (del_success<0)
      	reply(530);
      else
      	reply(200);
      
      continue;
    }

    /* DATA command? */
    if (strcmp(cmd,"DATA")==0) 
    { 
      /* sender, recipient, sizes and filename already specified? */
      if (*sender==0 || *recipient==0 || *filename==0 || 
	  (*sizes==0 && transmitted==0)) 
      { reply(503);
	continue;
      }
      
      /* does the top level spool directory exist? */
      if (stat(SPOOL,&finfo)<0 || (finfo.st_mode&S_IFMT)!=S_IFDIR) reply(410);
      
      /* change effective uid to recipient */
      seteuid(pwe->pw_uid);

      /* go to the user spool directory */
      if (chdir(userspool)<0) reply(410);
      
      /* create logfile if not there */
      close(open("log",O_CREAT|O_EXCL,S_IRUSR|S_IWUSR));
     
      /* check killfile */
      if (restricted(sender,recipient,'f')) 
      { reply(523);
	continue;
      }
      
      /* resend option? */
      if (transmitted && flp) 
      { if (transmitted==flp->csize) 
	{ transmitted = 0;
          reply(531);
	  continue;
	}

	/* open spool data file for appending */
        sprintf(sdfile,"%d.d",flp->id);
        if ((sdfd=open(sdfile,O_WRONLY|O_APPEND,S_IRUSR|S_IWUSR)) < 0) 
	  reply(412);

      } 
      else 
      { /* new file */
	  
        /* check if the file has been already sent */
        if ((sls=scanspool(sender))) 
	{ 
	  /* loop over files list */
          for (flp=sls->flist; flp!=NULL; flp=flp->next) 
	  { 
	    /* is it the same file and complete? */
            if (strcmp(filename,flp->fname)==0 && 
                strcmp(date,flp->date)==0 && flags==flp->flags &&
                flp->csize==flp->tsize) 
	    { reply(531);
              sls = NULL;
              break;
            }

          }
        
          /* return to main loop if file is already there */
          if (sls==NULL) continue;
        
        }
              
        /* get next spool id */
        id = spoolid();
        if (id==0 || id>=MAXFILES) reply(412);

        /* open spool header and data files */
        sprintf(shfile,"%d.h",id);
        if ((shfd=open(shfile,O_WRONLY,S_IRUSR|S_IWUSR)) < 0) reply(412);
        sprintf(sdfile,"%d.d",id);
        if ((sdfd=open(sdfile,O_WRONLY,S_IRUSR|S_IWUSR)) < 0) reply(412);

      
        /* lock the data file */
        wlock_file(sdfd);
      
        /* write the header lines */
        writeheader(shfd,"FROM",sender);
        writeheader(shfd,"FILE",filename);
        writeheader(shfd,"TYPE",type);
        writeheader(shfd,"SIZE",sizes);
        writeheader(shfd,"DATE",date);
        if (strncmp(type,"TEXT",4)==0) 
          writeheader(shfd,"CHARSET",charset);
        if (*attribute)
          writeheader(shfd,"ATTR",attribute);
        if (*sig)
          writeheader(shfd,"SIGN",sig);
        close(shfd);

      }
      
      /* tell the client to send data */
      reply(302);
      
      /* read the file data in PACKET size blocks */
      /* and write it to the spool file */
      bytes = size-transmitted;
      transmitted = 0;
      nblocks = bytes/PACKET;
      for (bn=1; bn<=nblocks; bn++) 
      { if (readn(0,packet,PACKET)<PACKET) 
	{ close(sdfd);
          unlink(sdfile);
          unlink(shfile);
          reply(415);
        }
        if (writen(sdfd,packet,PACKET)<PACKET) 
	{ close(sdfd);
          unlink(sdfile);
          unlink(shfile);
          reply(452);
        }
      }

      /* copy the last bytes to the spool file */
      if ((n=bytes-nblocks*PACKET)) 
      { if (readn(0,packet,n)<n) 
	{ close(sdfd);
          unlink(sdfile);
          unlink(shfile);
          reply(415);
        }
        if (writen(sdfd,packet,n)<n) 
	{ close(sdfd);
          unlink(sdfile);
          unlink(shfile);
          reply(452);
        }
      }
      close(sdfd);
      
      /* all ok */
      reply(201);
      
      /* get the receive date */
      timetick = time(0);
      strftime(currentdate,20,"19%y-%m-%d %H:%M:%S",localtime(&timetick));
      if (currentdate[2]<'9') memcpy(currentdate,"20",2);

      if (strcmp(attribute,"TAR")==0) strcat(filename," (archive)");

      /* try several times to lock-write the logfile */
      lfd = open("log",O_WRONLY|O_APPEND);
      for (n=1; n<5; n++) 
      { if (wlock_file(lfd) >= 0) 
	{ 
	  /* write to the log file */
          writeheader(lfd,"FROM",sender);
          writeheader(lfd,"FILE",filename);
          writeheader(lfd,"DATE",currentdate);
          writeheader(lfd,"","");
          break;

        }
        sleep(1);
      }
      close(lfd);

      /* reset uid to root */
      seteuid(0);
      
      /* recipient has to be notified at the end */
      notify = 1;
      continue;   
    }

    /* RESEND command? */
    if (strcmp(cmd,"RESEND")==0) 
    { 
      /* sender, recipient and filename already specified? */
      if (*sender==0 || *recipient==0 || *filename==0 || *sizes==0) 
      { reply(503);
        continue;
      }

      /* check, if this file has been already sent */
      if ((sls=scanspool(sender))) 
      { 
	/* loop over files list */
        for (flp=sls->flist; flp!=NULL; flp=flp->next) 
	{ 
	  /* is it the same file ? */
          if (strcmp(filename,flp->fname)==0 && 
	      strcmp(date,flp->date)==0 && flags==flp->flags) 
	  { 
	    /* with same sizes? */
	    sprintf(tmp,"%d %d",flp->csize,flp->osize);
	    if (strcmp(tmp,sizes)==0) 
	    { 
	      /* number of bytes already transmitted */
	      transmitted = flp->tsize;
	      break;

	    }
	  }
	}
      }
      
      /* emulate reply(230) */
      printf("230 %d bytes have already been transmitted.\r\n",transmitted);
      fflush(stdout);
      continue;
    }

    /* unknown command or syntax error */
    reply(500);

  }    
}


/*
 * getline - get a command line until LF
 *
 * INPUT:  ptr  - empty string
 *
 * OUTPUT: ptr  - string containing the command line
 *
 * RETURN: number of read bytes
 */
int getline(char *ptr) 
{ int c,        /* one character */
      n,        /* number of read bytes */
      ctrl=0;	/* flag for non-ASCII character */

  ptr[0]=0;

  /* get max MAXLEN characters */
  for (n=0; n<MAXLEN-1; n++) 
  { 
    /* next char from socket */
    c = getchar();

    /* quit if there is no more a connection to the client */
    if (c==EOF) reply(421);

    /* surpress non-ASCII chars */
    if (c<9 || c==11 || c==12 || (c>13 && c<32) || c>126) 
    { ctrl = 1;
      n--;
    } 
    else 
    { 
      /* copy the char into the command line string */
      ptr[n] = c;

      /* check for EOL */
      if (c=='\n') break;
      if (c=='\r') n--;

    }
  }
  
  /* input line overrun? */
  if (n==MAXLEN-1 && ptr[n]!='\n') 
  {     
    /* read to next lf */
    while (c!='\n') c = getchar();
    n = 0;
    reply(506);
  } else
    if (ctrl) reply(205);
  
  ptr[n] = 0;

  /* trim all white spaces */
  if (n) str_trim(ptr);

  return(strlen(ptr));
}


/* 
 * writeheader - write one line of header information
 * 
 * INPUT: fd		- file descriptor
 *        attribute	- name of the header attribute 
 *        value		- contents of the header attribute
 */
void writeheader(int fd, const char *attribute, const char *value) 
{ int hsize;		/* header string size */
  char header[2*MAXLEN];/* header string */
  
  sprintf(header,"%s\t%s\n",attribute,value);
  hsize = strlen(header);
  if (write(fd,header,hsize)<hsize) reply(412);
}


/* 
 * msg2tty - send a one line message to the recipient (tty(s) or FIFO)
 * 
 * INPUT:  recipient	- recipient of the message
 *         msg		- the message
 *         sender	- message sender
 * 
 * RETURN: 0 if successfull, -1 if failed
*/
int msg2tty(const char *recipient, const char *msg, int uid) 
{ char 
    *cp,		/* simple character pointer */
    tty[FLEN],		/* name of tty */
    user[9],		/* username */
    msgwhere[MAXLEN],	/* where to write message to */
    fifo[MAXLEN],	/* fifo name */
    mcf[MAXLEN];	/* message control file */
  int 
    utmpfd, 		/* file descriptor for utmp */
    mfd, 		/* message file descriptor */
    success;		/* return code */
  struct utmp uinfo;	/* information about a user */
  struct stat finfo;	/* information about a file */
  FILE *inf;		/* input file */
  
  success = 0;
  user[8] = 0;
  *msgwhere = 0;
  sprintf(mcf,"%s/config/msg",userspool);
  sprintf(fifo,"%s/config/msgfifo",userspool);
  
  /* change effective uid to recipient for security reasons */
  if (uid) seteuid(uid);
 
  /* is there a message fifo? */
  if (stat(fifo,&finfo)==0 && S_ISFIFO(finfo.st_mode)) 
  {
    /* is it locked by a chat client? */
    mfd = open(fifo,O_WRONLY|O_NONBLOCK);
    if (mfd>0 && tlock_file(mfd)==1) 
    {
      /* write to fifo */
      close(mfd);
      mfd = open(fifo,O_WRONLY);
      success = write(mfd,msg,strlen(msg));
    }

    close(mfd);
  }
  
  /* is there a message control file? */
  if (success<=0 && (inf=fopen(mcf,"r"))) 
  { 
    /* read the tty name */
    fgets(tty,MAXLEN-1,inf);
    if ((cp=strchr(tty,'\n'))) *cp = 0;
    fclose(inf);
    
    /* belongs the tty to the recipient and is it writable? */
    if (stat(tty,&finfo)==0 && finfo.st_uid==uid &&
	((finfo.st_mode&S_IWOTH) || (finfo.st_mode&S_IWGRP))) 
    { 
      /* write to dedicated tty */
      mfd = open(tty,O_WRONLY);
      success = write(mfd,msg,strlen(msg));
#ifdef BELL
      write(mfd,BELL,1);
#endif BELL
      close(mfd);
    }
  }
  
  /* no write success so far? */
  if (success<=0) 
  { success = 0;
    
    /* search the utmp file (not standarisized, grrrr) */
    utmpfd = open(_PATH_UTMP,O_RDONLY);
    if (utmpfd<0) utmpfd = open("/var/adm/utmp",O_RDONLY);
    if (utmpfd<0) utmpfd = open("/var/run/utmp",O_RDONLY);
    if (utmpfd<0) return(-1);
  
    /* scan through utmp (currently logged in users) */
    while (read(utmpfd,(char *)&uinfo,sizeof(uinfo))>0) 
    {
#if defined(NEXT) || defined(BSD)
      strncpy(user,uinfo.ut_name,8);
      if (strcmp(recipient,user)==0) 
      {
#ifdef JEDPARSE
}
#endif
#else
      strncpy(user,uinfo.ut_user,8);
      if (uinfo.ut_type==USER_PROCESS && strcmp(recipient,user)==0) {
#endif
	/* get the tty */
	sprintf(tty,"/dev/%s",uinfo.ut_line);
	  
	/* is the tty writeable? */
	if (stat(tty,&finfo)==0 && 
	    ((finfo.st_mode&S_IWOTH) || (finfo.st_mode&S_IWGRP))) 
	{ mfd = open(tty,O_WRONLY);
	
	  /* write message to tty */
	  success = success | write(mfd,msg,strlen(msg));
	  write(mfd,"\007",1);
#ifdef BELL
	  write(mfd,BELL,1);
#endif BELL
	  close(mfd);
	  
	}
      }
    }
    close(utmpfd);
  }

  /* reset uid to root */
  if (uid) seteuid(0);
  
  if (success>0) return(0); else return(-1);
}


/*
 * spoolid - find the next spool id
 * 
 * RETURN: spool id, 0 if failed
 */
int spoolid() 
{ int i,		/* simple loop count */
      fd,		/* file descriptor of spool id header file */
      id,		/* id number */
      idmax;		/* biggest id number */
  char *cp, 		/* character pointer to '.' in file name */
       file[MAXLEN];	/* complete file name */
#ifdef NEXT
  char tmp[MAXLEN];	/* tmp string */
  FILE *pp;		/* pipe */
#else
  struct dirent *dire;	/* directory entry */
  DIR *dp;		/* directory pointer */
#endif


  /* try to create next spool files */
  for (i=1; i<5; i++) 
  {
   /* initialisize */
    id = 0;
    fd = 0;
    idmax = 0;

    /* mega stupid NeXT has broken readdir() */
#ifdef NEXT
    /* open spool dir */
    if ((pp=popen("ls . 2>/dev/null","r")) == NULL) return(NULL);
  
    /* scan through spool directory */
    while (fgets(tmp,MAXLEN-1,pp)) 
    {
      if ((cp=strrchr(tmp,'\n'))) *cp = 0;
      cp = strchr(tmp,'.');
      if (cp && strcmp(cp,".h")==0) 
      { *cp = 0;
        id = atoi(tmp);
	if (id>idmax) idmax = id;
      }
    }
    pclose(pp);
  
#else    
    /* open spool dir */
    dp = opendir(".");
  
    /* scan through spool directory and get the highest spool id */
    while ((dire=readdir(dp))) 
    { cp = strchr(dire->d_name,'.');
      if (cp && strcmp(cp,".h")==0) 
      { *cp = 0;
        id = atoi(dire->d_name);
	if (id>idmax) idmax = id;
      }
    }
    closedir(dp);
#endif
    
    id = idmax+1;

    /* try to create header spool file */
    sprintf(file,"%d.h",id);
    fd = open(file,O_CREAT|O_EXCL,S_IRUSR|S_IWUSR);
    close(fd);
    
    /* successfull? */
    if (fd>0) 
    {
      /* set owner and group */
      /* chown(file,uid,gid); */

      /* create data spool file */
      sprintf(file,"%d.d",id);
      close(open(file,O_CREAT,S_IRUSR|S_IWUSR));
      /* chown(file,uid,gid); */

      return(id);
    }
    
    /* wait and test again */
    sleep(1);
  }

  /* failed */
  return(0);
}


/*
 * restricted  - check killfile
 * 
 * INPUT:  sender     - sender name
 *         recipient  - local recipient
 *         type       - type of restriction: m(essage) or f(ile)
 * 
 * RETURN: 1 if not allowed to send, 0 if allowed
 */
int restricted(const char *sender, const char *recipient, char type)
{ int m;
  char *cp,
       killfile[MAXLEN],
       kfu[MAXLEN],
       kfm[MAXLEN],
       line[MAXLEN],
       from[MAXLEN];
  FILE *inf;

  sprintf(killfile,"%s/config/restrictions",userspool);
  *kfm = *kfu = 0;
  
  /* open and check killfile */
  inf = fopen(killfile,"r");
  if (inf==NULL) return(0);

  strcpy(from,sender);
  if ((cp=strchr(from,' '))) *cp = 0;
  
  while (fgets(line,MAXLEN-1,inf)) 
  { sscanf(line,"%s%s",kfu,kfm);
    if (kfm[0]==0) 
      m = 'b';
    else
      m = tolower(kfm[0]);
    kfm[1] = 0;
    if (simplematch(from,kfu,1) && (m==type || m=='b')) 
    { fclose(inf);
      return(1);
    }
  }
  
  fclose(inf);
  return(0);
}



/*
 * wlock_file - write-lock a file (POSIX conform)
 * 
 * INPUT:  file descriptor
 * 
 * RETURN: >= 0 if ok, -1 if error
 */
int wlock_file(int fd) 
{ struct flock lock;	/* file locking structure */

  /* fill out the file locking structure */
  lock.l_type = F_WRLCK;
  lock.l_start = 0;
  lock.l_whence = SEEK_SET;
  lock.l_len = 0;

  /* try to lock the file and return the status */
  return(fcntl(fd,F_SETLK,&lock));
}

  
/*
 * tlock_file - test if a file is write-lock blocked (POSIX conform)
 * 
 * INPUT:  fd  - file descriptor
 * 
 * RETURN: 0 if no lock, 1 if locked, -1 on error
 */
int tlock_file(int fd) 
{ int status;
  struct flock lock;    /* file locking structure */

  /* fill out the file locking structure */
  lock.l_type = F_WRLCK;
  lock.l_start = 0;
  lock.l_whence = SEEK_SET;
  lock.l_len = 0;
  
  /* test the lock status */
  status = fcntl(fd,F_GETLK,&lock);
  if (status>=0) status = lock.l_type!=F_UNLCK;
  return(status);
}
