/*
 * File:	spool.c
 * 
 * Author:	Ulli Horlacher (framstag@rus.uni-stuttgart.de)
 * 
 * History:	 1 Sep 95	Framstag	initial version
 *              10 Oct 95	Framstag	open bug fixed
 *               1 Nov 95	Framstag	added pgp signature
 *               5 Nov 95	Framstag	added NeXT support
 *              15 Nov 95	Framstag	fixed memory leak (oops :-) )
 * 
 * Functions for operations on files in the sendfile spool directory.
 * 
 * This file is covered by the GNU General Public License
 */


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <dirent.h>
#include <sys/stat.h>

#include "config.h"     /* various #defines */
#include "spool.h"	/* operations on files in the sendfile spool */
#include "utf7.h"       /* UTF-7 coding */
#include "message.h"    /* information, warning and error messages */
#include "string.h"     /* extended string functions */
#include "reply.h"	/* the 3 digit reply codes with text messages */


/*
 * scanspool - scan through spool directory and build list-structures
 * 
 * INPUT:   sender  - sender user name ("" if all senders are requested)
 * 
 * RETURN:  start of sender list
 */
struct senderlist *scanspool(const char *sender) 
{ char *cp,			/* simple character pointer */
       *arg,			/* the argument(s) of the header line */
       hline[MAXLEN],		/* header line */
       from[MAXLEN],		/* header from line */
       fname[MAXLEN],		/* header file line */
       msg[MAXLEN],		/* information/error message */
       sig[MAXLEN],		/* pgp signature */
       date[DLEN+1],		/* file date */
       charset[DLEN+1],		/* character set name */
       rdate[DLEN+1],		/* receive date */
       file[FLEN+1];		/* spool file name */
  int id,			/* spool file id number */
      hfc,			/* header file corruption flag */
      osize,			/* original file size */
      csize,			/* compressed file size */
      tsize,			/* true spool data file size */
      flags;			/* source, text, compress and tar flag */
  struct stat finfo;		/* information about a file */
  FILE *hf;			/* header file */
  extern int client;		/* flag to determine client or server */
  extern char userspool[];	/* user spool directory */
#ifdef NEXT
  char tmp[MAXLEN];		/* tmp string */
  FILE *pp;			/* pipe */
#else
  struct dirent *dire;		/* directory entry */
  DIR *dp;			/* directory pointer */
#endif
  struct filelist *flp,		/* file list pointer */
                  *fln;		/* new file list element */
  struct senderlist *sls,	/* sender list start */
                    *sln,	/* sender list next pointer */
                    *slp;	/* sender list pointer */
  static struct senderlist *sll=NULL;	/* last sender list start */

  /* is there already an old senderlist? */
  if (sll)
    
    /* delete old senderlist and free memory */
    for (slp=sll; slp!=NULL; ) 
  { for (flp=slp->flist; flp!=NULL; ) 
    { fln = flp->next;
	free(flp);
	flp = fln;
      }
      sln = slp->next;
      free(slp);
      slp = sln;
    }
  
  sls = sll = NULL;
  
  /* mega stupid NeXT has broken readdir() */
#ifdef NEXT
  /* open spool dir */
  sprintf(tmp,"ls %s 2>/dev/null",userspool);
  if ((pp = popen(tmp,"r")) == NULL) return(NULL);
  
  /* scan through spool directory */
  while (fgets(tmp,MAXLEN-1,pp)) 
  {
    if ((cp=strrchr(tmp,'\n'))) *cp = 0;
    
    /* look for header files */
    if ((cp=strchr(tmp,'.')) && strcmp(cp,".h")==0) 
    { 
      /* extract spool id */
      *cp = 0;
      id=atoi(tmp);
#ifdef JEDPARSE
    }}
#endif  

#else    
  /* open spool dir */
  if ((dp=opendir(userspool)) == NULL) return(NULL);

  /* scan through spool directory */
  while ((dire=readdir(dp))) 
  { 
    /* look for header files */
    if ((cp=strchr(dire->d_name,'.')) && strcmp(cp,".h")==0) 
    { 
      /* extract spool id */
      *cp = 0;
      id=atoi(dire->d_name);
#endif      

      /* open header file */
      sprintf(file,"%s/%d.h",userspool,id);
      if ((hf=fopen(file,"r")) == NULL) 
      { 
	/* called from receive client? */
	if (client) 
	{ sprintf(msg,"cannot open spool file %s",file);
	  message("",'E',msg);
	}
	
        continue;
      }
      
      /* initialisize header entries */
      flags = 0;
      osize = 0;
      csize = 0;
      tsize = 0;
      *from = 0;
      *fname = 0;
      *date = 0;
      *charset = 0;
      *sig = 0;
      sprintf(file,"%s/%d.d",userspool,id);
      
      /* get local time */
      stat(file,&finfo);
      strftime(rdate,21,"19%y-%m-%d %H:%M:%S",localtime(&finfo.st_mtime));
      if (rdate[2]<'9') memcpy(rdate,"20",2);
    
      /* read header file */
      hfc = 0;
      while ((fgets(hline,MAXLEN-1,hf)) && hfc==0) 
      {
	/* prepare the header line */
        if ((cp=strchr(hline,'\n'))) *cp=0;
        cp = strchr(hline,'\t');
	
	/* corrupt header file line? */
	if (cp==NULL)
	{ hfc = 1;
	  continue;
	}
	
        arg = cp+1;
        *cp = 0;
        
        /* extract the header-name and the argument */
        if (strcmp(hline,"FROM")==0) 
	{ strcpy(from,arg);
          continue;
        }
        if (strcmp(hline,"FILE")==0) 
	{ strcpy(fname,arg);
          continue;
        }
        if (strcmp(hline,"DATE")==0) 
	{ strncpy(date,arg,DLEN);
          continue;
        }
        if (strcmp(hline,"CHARSET")==0) 
	{ strncpy(charset,arg,DLEN);
          continue;
        }
        if (strcmp(hline,"SIGN")==0) 
	{ strcpy(sig,arg);
          continue;
        }
        if (strcmp(hline,"SIZE")==0) 
	{ sscanf(arg,"%d %d",&csize,&osize);
	  tsize=finfo.st_size;
          continue;
        }
        str_toupper(arg);
        if (strcmp(hline,"ATTR")==0 && strcmp(arg,"TAR")==0)
          flags = flags|F_TAR;
        if (strcmp(hline,"TYPE")==0) 
	{ if (strstr(arg,"SOURCE"))	flags = flags|F_SOURCE;
          if (strstr(arg,"TEXT"))	flags = flags|F_TEXT;
          if (strstr(arg,"COMPRESSED")) flags = flags|F_COMPRESS;
        }
      }
      
      fclose(hf);
      
      /* bad header file? */
      if (*from==0 || *fname==0 || *date==0 || osize==0) continue;

      /* is it a sender name to look for? */
      if (*sender==0 || strcmp(from,sender)==0) 
      { 
        /* create new file list element */
	if ((fln=(struct filelist *) malloc(sizeof(struct filelist))) == NULL) 
	{ if (client)
      	    message("",'F',"cannot allocate memory");
	  else
	    reply(453);
	}
	
	/* fill in new file list element */
	strcpy(fln->fname,fname);
	strcpy(fln->date,date);
	strcpy(fln->charset,charset);
	strcpy(fln->rdate,rdate);
	fln->id = id;
	fln->next = NULL;
	fln->osize = osize;
	fln->csize = csize;
	fln->tsize = tsize;
	fln->flags = flags;
        
        /* first sender? */
        if (sls==NULL)
          sls = newsle(fln,from,sig);
        else 
	{
	  /* search for sender in senderlist */
          slp = sls;
          while (slp->next) 
	  { if (strcmp(slp->from,from)==0) break;
            slp = slp->next;
          }
        
          /* sender not found in sender list? */
          if (strcmp(slp->from,from)!=0)
          
            /* create new sender list element and append it */
            slp->next = newsle(fln,from,sig);

          else 
	  {
	    /* is it the smallest id? */
            if (id < slp->flist->id) 
	    { fln->next = slp->flist;
              slp->flist = fln;
            } else 
	    {
             /* insert into files list */
              for (flp=slp->flist; flp->next!=NULL; flp=flp->next) 
	      { if (id < flp->next->id) 
		{ fln->next = flp->next;
                  flp->next = fln;
                  break;
                }
              }
            
              /* last element? */
              if (flp->next==NULL) flp->next = fln;

            }
          }
        }
      }
    }
  }

#ifdef NEXT
  pclose(pp);
#else
  closedir(dp);
#endif
  sll = sls;    
  return(sls);
}
  

/*
 * newsle - create new sender list element and fill it out
 * 
 * INPUT:  flp  - first file list element
 *         from - sendername
 * 
 * RETURN: start of sender list
 */
struct senderlist *newsle(struct filelist *flp, 
			  const char *from, const char *sig) 
{ struct senderlist *sln;	/* new sender list element */
  extern int client;		/* flag to determine client or server */

  /* create new sender list element */
  if ((sln=(struct senderlist *)malloc(sizeof(struct senderlist))) == NULL) 
  { if (client)
      message("",'F',"cannot allocate memory");
    else
      reply(453);
  }
  
  /* fill it out */
  sln->next = NULL;
  sln->flist = flp;
  strcpy(sln->from,from);
  strcpy(sln->sig,sig);
  
  return(sln);
}
  

/* 
 * delete_sf - delete a spool file
 * 
 * INPUT:  id   	- spool file id number
 *         verbose	- if set, print success message
 * 
 * RETURN: 0 on success, -1 on fail
 */
int delete_sf(struct filelist *flp, int verbose) 
{ char msg[2*MAXLEN],		/* information/error message */
       file[MAXLEN],		/* spool file */
       dummy[MAXLEN],		/* dummy string for utf2iso */
       fname[MAXLEN];		/* displayble file name */
  extern int client;		/* flag to determine client or server */
  extern char userspool[];	/* user spool directory */

  sprintf(file,"%s/%d.d",userspool,flp->id);
  unlink(file);
  sprintf(file,"%s/%d.h",userspool,flp->id);
  utf2iso(1,dummy,fname,dummy,flp->fname);
  if(unlink(file)<0) 
  { if (client) 
    { sprintf(msg,"cannot delete spoolfile #%d",flp->id);
      message("",'W',msg);
    }
    return(-1);
  } else 
  { if (verbose) 
    { sprintf(msg,"%s deleted",fname);
      message("",'I',msg);
    }
    return(0);
  }
}
