/*
 * File:	sendmsg.c
 * 
 * Author:	Ulli Horlacher (framstag@rus.uni-stuttgart.de)
 * 
 * History:	11 Aug 95   Framstag	initial version
 *              12 Aug 95   Framstag	elm alias support
 *               2 Nov 95   Framstag	added minimal chat mode
 *              14 Nov 95   Framstag	added message receiving modes
 * 
 * The sendmessage client of the sendfile package.
 * Sends a single line text message to the SAFT-server of the destination
 * system to be displayed on the recipients terminal.
 * 
 * This file is covered by the GNU General Public License
 */


#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <sys/stat.h>

#include "config.h"		/* various #defines */
#include "net.h"		/* the network routines */
#include "io.h"			/* (socket) read/write */
#include "message.h"		/* information, warning and error messages */
#include "utf7.h"		/* UTF-7 coding */
#include "destination.h"	/* check recipient and host */

#ifndef AIX
  FILE *popen(const char *, const char *);
  int pclose(FILE *);
  #if defined(IRIX) || defined(LINUX)
    #include <getopt.h>
  #else
    int getopt(int, char * const *, const char *);
  #endif
#endif


/* print short help usage text */
int usage();


/* global variables */
int verbose;		/* flag for verbose mode */
char *prg;		/* name of the game */


int main(int argc, char *argv[]) 
{ extern int optind;	/* number of scanned args with getopt */
  int  sockfd,		/* socket file descriptor */
       chat,		/* flag for chat mode */
       receive,		/* receiving flag */
       opt;		/* option to test for */
  char recipient[FLEN], /* recipient at serverhost */
       user[FLEN], 	/* local user name */
       host[FLEN], 	/* name of serverhost */
       msgcf[FLEN], 	/* message control file */
       *reply,		/* reply string from server */
       *cp,		/* simple character pointer */
       line[MAXLEN],	/* pipe input string */
       login[MAXLEN],	/* login user name */
       tmp[MAXLEN];	/* temporary string */
  FILE *pipe=NULL;	/* pipe file descriptor */
  unsigned char utf_msg[LEN_UTF],	/* msg in UTF-7 format */
                iso_msg[LEN_ISO];	/* msg in ISO Latin-1 format */

  prg = argv[0];
  verbose = 0;
  receive = 0;
  chat = 0;

  /* scan the command line on options */
  while ((opt = getopt(argc,argv,"cvmMh?")) > 0) 
  { switch (opt) 
    { case ':':
      case 'h':
      case '?': exit(usage());
      case 'v': verbose = 1; break;
      case 'm': receive = 1 ; break;
      case 'M': receive = 2 ; break;
      case 'c': chat = 1;
    }
  }

  /* too few arguments? */
  if (argc-optind<1 && receive==0) exit(usage());

  /* get own user name, recipient name and host */
  destination(argc,argv,user,recipient,host);
  strcpy(login,user);
  if ((cp=strchr(login,' '))) *cp = 0;

  /* test local sendfiled */
  if (verbose) 
  { verbose = 0;
    printf("testing local SAFT server...\n");
    sockfd = open_connection("127.0.0.1",SAFT);
    verbose = 1;
  } else
    sockfd = open_connection("127.0.0.1",SAFT);
  
  if (sockfd<0) 
  { message(prg,'W',"there is no local SAFT server - "
	            "you cannot receive messages");
  } else 
  { 
    close(sockfd);

    /* allow receiving of messages on this tty */
    sprintf(msgcf,"%s/%s/config/msg",SPOOL,login);
    sprintf(tmp,"( mesg y; tty >%s ) 2>&1",msgcf);
    pipe = popen(tmp,"r");
    
    /* error when configuring tty? */
    if (fgets(line,MAXLEN-1,pipe)) 
    { if ((cp=strrchr(line,'\n'))) *cp = 0;
      errno = 0;
      sprintf(tmp,"error while configuring tty: %s",line);
      message(prg,'E',tmp);
    } else 
    { 
      if (receive==1 && argc-optind<1)
	message(prg,'I',"receiving messages is now possible only on this tty");
  
      if (receive==2) 
      { unlink(msgcf);
	if (argc-optind<1)
	  message(prg,'I',"receiving messages is now possible on all ttys");
      }

    }
  }

  pclose(pipe);
  if (receive>0 && argc-optind<1) exit(0);

  if (strcmp(host,"127.0.0.1")==0)
    printf("sending to %s@localhost\n",recipient);
  else
    printf("sending to %s@%s\n",recipient,host);
  
  /* initate the connection to the server */
  sockfd = open_connection(host,SAFT);
  if (sockfd<0) message(prg,'F',"cannot connect to server");
  reply = getreply(sockfd);
  
  /* protocol error? */
  if (strncmp(reply,"220 ",4)!=0 || strstr(reply," SAFT ")==NULL) 
  { errno = 0;
    message(prg,'F',"remote protocol error");
  }

  /* send FROM and TO headers */
  sprintf(tmp,"FROM %s",user);
  if (sendheader(sockfd,tmp)<0) exit(1);
  sprintf(tmp,"TO %s",recipient);
  if (sendheader(sockfd,tmp)<0) exit(1);

  do 
  { /* read the message */
    printf("message: ");
    *iso_msg=0;
    fgets(iso_msg,LEN_ISO,stdin);

    /* message empty? */
    if (*iso_msg==0) 
    { printf("\n");
      break;
    }
    if (*iso_msg=='\n') continue;

    /* strip off new line */
    cp=strrchr(iso_msg,'\n');
    if (cp && (cp!=(char *)iso_msg)) *cp=0;

    /* encode to UTF-7 */
    iso2utf(utf_msg,iso_msg);
      
    /* send the message */
    sprintf(tmp,"MSG %s",utf_msg);
    sendheader(sockfd,tmp);
  } while (chat);

  /* close the connection */
  sock_putline(sockfd,"QUIT");
  getreply(sockfd);
  close(sockfd);

  exit(0);
}


/*
 * usage - print short help usage text
 */
int usage() 
{ fprintf(stderr,"usage: %s [-v] user[@host]\n",prg);
  fprintf(stderr,"   or: %s [-mM]\n",prg);
  fprintf(stderr,"options: -v  verbose mode\n");
  fprintf(stderr,"         -m  receive messages only on this tty (default)\n");
  fprintf(stderr,"         -M  receive messages on other ttys, too\n");
  fprintf(stderr,"example: %s orakel@main01.bb.bawue.de\n",prg);
  return(2);
}
