/*
 * File:	receive.c
 * 
 * Author:	Ulli Horlacher (framstag@rus.uni-stuttgart.de)
 * 
 * History:	11 Aug 95   Framstag	initial version
 *              14 Aug 95   Framstag	corrected bug when receiving archives
 *              10 Sep 95   Framstag	extracted functions to spool.c
 *              31 Oct 95   Framstag	fixed security problems
 *               5 Nov 95   Framstag    added NeXT support
 *               7 Nov 95   Framstag    corrected bug when receiving to
 *                                      a non-writable directory
 * 
 * The receive client of the sendfile package.
 * Receives, lists and deletes files from the sendfile spool.
 * 
 * This file is covered by the GNU General Public License
 */


#ifdef ULTRIX
  #define _POSIX_SOURCE
  #define S_IFMT   0170000 /* type of file */
  #define S_IFDIR  0040000 /* directory */
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <pwd.h>
#include <time.h>
#include <utime.h>
#include <ctype.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "config.h"     /* various #defines */
#include "message.h"    /* information, warning and error messages */
#include "utf7.h"       /* UTF-7 coding */
#include "io.h"		/* read/write routines */
#include "string.h"     /* extended string functions */
#include "spool.h"	/* operations on files in the sendfile spool */

#ifndef AIX
  FILE *popen(const char *, const char *);
  int pclose(FILE *);
  #if defined(IRIX) || defined(LINUX)
    #include <getopt.h>
  #else
    int getopt(int, char * const *, const char *);
  #endif
#endif

/*
#ifdef IRIX
  int toupper(char);
  int tolower(char);
#endif
*/

/* global variables */
char *prg,		/* name of the game */
     userspool[FLEN];	/* user spool directory */
int client=1;		/* flag to determine client or server */
mode_t cmask;		/* umask value */
unsigned long blksize;	/* file system block size */
extern int daylight;
#ifdef BSD
  long int timezone = 0;
#else
  extern long int timezone;
#endif


/* print short help usage text */
int usage(void);

/* list all spool files */
void list(struct senderlist *, int);

/* receive a spool file */
void receive_sf(struct filelist *);

/* translate NVT telnet format file to Unix text format file */
void crlf2lf(FILE *, FILE *, const char *);

/* copy a file */
int fcopy(const char *, const char *, unsigned long, mode_t);

/* spawn a subprocess and direct output to a file */
int spawn(char **, const char *, mode_t);

  
int main(int argc, char *argv[]) {
  int i,			/* simple loop count */
      number=0,			/* flag for specifying a spool file number */
      listformat=0,		/* format of listing */
      del=0,			/* flag for deleting */
      all=0,			/* flag for receiving all */
      id,			/* spool id */
      found=0,			/* flag for found spool id */
      opt;			/* option to test for */
  char tmp[MAXLEN],		/* temporary string */
       dummy[MAXLEN],		/* dummy string for utf2iso */
       fname[MAXLEN],		/* displayble file name */
       pattern[MAXLEN];		/* file name pattern to search for */
  struct stat finfo;		/* information about a file */
  struct passwd *pwe;		/* password entry */
  struct filelist *flp;		/* file list pointer */
  struct senderlist *sls,	/* sender list start */
                    *slp;	/* sender list pointer */
#if !(defined(LINUX) || defined(IRIX))
  extern int optind;
#endif
  
  prg = argv[0];
  cmask = umask(0);
  umask(cmask);

  /* no arguments? */
  if (argc==1) exit(usage());

  /* scan the command line */
  while ((opt=getopt(argc,argv,"h?lLnda")) > 0) {
    switch (opt) {
      case ':':
      case 'h':
      case '?': exit(usage());
      case 'l': listformat = 1; break;
      case 'L': listformat = 2; break;
      case 'n': number = 1; break;
      case 'd': del = 1; break;
      case 'a': all = 1; break;
    }
  }

  /* get the own user name */
  if ((pwe=getpwuid(getuid())) == NULL)
    message(prg,'F',"cannot determine own user name");
  
  /* does the spool directory exist? */
  sprintf(userspool,SPOOL"/%s",pwe->pw_name);
  if (stat(userspool,&finfo)<0 || (finfo.st_mode&S_IFMT)!=S_IFDIR) {
    sprintf(tmp,"spool directory %s does not exist",userspool);
    message(prg,'W',tmp);
    exit(1);
  }
 
  /* file system block size for copy operation */  
  blksize = finfo.st_blksize;

  /* are there any files to receive? */
  if ((sls=scanspool("")) == NULL) {
    message(prg,'W',"no files in spool directory");
    exit(1);
  }

  /* list files? */
  if (listformat) {
    list(sls,listformat);
    exit(0);    
  }

  /* number specified? */
  if (number) {
    
    /* loop over all args */
    for (i=optind; i<argc; i++) {
      id = atoi(argv[i]);
      
      /* loop over sender list */
      for (slp=sls, found=0; slp!=NULL && found==0; slp=slp->next) {
        
        /* loop over files list */
        for (flp=slp->flist; flp!=NULL && found==0; flp=flp->next) {
          
          /* spool id found and spool file complete? */
          if (flp->id==id && flp->csize==flp->tsize) {
	      
	    /* delete or receive spool file? */
	    if (del) 
	      delete_sf(flp,1);
	    else
	      receive_sf(flp);
	      
	    found = 1;

	  }
	}
      }
      
      /* not found? */
      if (found==0) {
	sprintf(tmp,"file #%d not found",id);
	message(prg,'W',tmp);
      }
      
    }

  } else { /* file name specified */

    /* loop over all args */
    for (i=optind; i<argc || all; i++) {
      if (all)
        strcpy(pattern,"*");
      else
      	strcpy(pattern,argv[i]);
      
      /* loop over sender list */
      for (slp=sls; slp!=NULL; slp=slp->next) {
          
        /* loop over files list */
        for (flp=slp->flist; flp!=NULL; flp=flp->next) {
	  
      	  /* spool file incomplete? */
          if (flp->csize!=flp->tsize) continue;
	  
	  /* translate UTF-7 name to the displayable file name */
	  utf2iso(1,dummy,fname,dummy,flp->fname);
	  
	  /* match? */
	  if (simplematch(fname,pattern,0)==1) {
	    
	    found = 1;

	    /* delete or receive spool file? */
            if (del) 
              delete_sf(flp,1);
            else
	      receive_sf(flp);
	    
	  }
	}
      }
      
      /* not found? */
      if (found==0) {
	sprintf(tmp,"file %s not found",pattern);
	message(prg,'W',tmp);
      }
      
      if (all) break;
    }
  }

  exit(0);    
}


/* 
 * list - list all spool files with attributes
 * 
 * INPUT:  sls  - sender list start
 */
void list(struct senderlist *sls, int format) {
  char dummy[MAXLEN],		/* dummy string for utf2iso */
       line[MAXLEN],		/* line to read in */
       showtar[MAXLEN],		/* shell command to look inside tar archive */
       show_fname[MAXLEN];	/* file name to display */
  struct senderlist *slp;	/* sender list pointer */
  struct filelist *flp;		/* file list pointer */
  FILE *pipe;			/* pipe input stream */
  
  /* loop over sender list */
  for (slp=sls; slp!=NULL; slp=slp->next) {
    printf("\nFrom %s\n",slp->from);
    
    /* loop over files list */
    for (flp=slp->flist; flp!=NULL; flp=flp->next) {
      
      /* not complete? */
      if (flp->csize!=flp->tsize) continue;
      
      /* print spool file informations */
      utf2iso(1,dummy,show_fname,dummy,flp->fname);
      printf("%3d) %s  %5d KB  %s",
             flp->id,flp->rdate,(flp->osize+999)/1000,show_fname);

      /* tar archive? */
      if (flp->flags&F_TAR) {
        printf(" (archive)\n");
        
        /* on verbose mode look inside tar */
        if (format==2) {
	  
	  /* compressed or not compressed, this is the question! :-) */
          if (flp->flags&F_COMPRESS) 
            sprintf(showtar,"gunzip < %s/%d.d | tar tvf -",userspool,flp->id);
          else
            sprintf(showtar,"tar tvf %s/%d.d",userspool,flp->id);
	  
	  /* sneak inside... */
          if ((pipe=popen(showtar,"r")) == NULL)
            message(prg,'E',"contents of archive is not accessible");
          else
            while (fgets(line,MAXLEN-1,pipe)) printf("     %s",line);
        }

      } else
        printf("\n");

    }
  }
}


/* 
 * find_id - find a spool file in the list-structure
 * 
 * INPUT:  id   - spool file id number
 * 
 * RETURN: pointer to file list element, NULL if not found
 */
struct filelist *find_id(struct senderlist *sls, int id) {
  char msg[MAXLEN];		/* information/error message */
  struct senderlist *slp;	/* sender list pointer */
  struct filelist *flp;		/* file list pointer */
  
  /* loop over sender list */
  for (slp=sls; slp!=NULL; slp=slp->next) {
    
    /* loop over files list */
    for (flp=slp->flist; flp!=NULL; flp=flp->next) {
      
      /* found file? */
      if (flp->id == id) return(flp);
    
    }
  }
  
  sprintf(msg,"cannot find spool file #%d",id);
  errno = 0;
  message(prg,'E',msg);
  
  return(NULL);
}


/* 
 * receive_sf - receive a spool file
 * 
 * INPUT:  flp  - file list element
 */
void receive_sf(struct filelist *flp) {
  int utf,			/* return code from utf2iso */
      tom=0;			/* tar overwrite message flag */
  char *cp,			/* simple character pointer */
       *sad[10],		/* spawn argument descriptor */
       tmp[2*MAXLEN],		/* temporary string */
       cmd[2*MAXLEN],		/* command string for system() */
       sfile[MAXLEN],		/* spool data file */
       fname[MAXLEN],		/* file name to write */
       error[MAXLEN],		/* error log file */
       iso_name[MAXLEN],	/* file name in ISO Latin-1 */
       normal_name[MAXLEN],	/* file name in display format */
       shell_name[MAXLEN];	/* file name in shell handling format */
  static char storeformat='c',	/* file name storing format */
              overwrite='n';	/* overwrite mode */
  struct tm time;		/* broken down time */
  struct utimbuf utb;		/* file date structure */
  struct stat finfo;		/* information about a file */
  time_t dummy;			/* dummy arg for localtime() */
  FILE *pipe,			/* pipe input stream */
       *inf,			/* spool data file to read */
       *outf;			/* file to create */
  
  /* oops? */
  if (flp==NULL) return;

  (void) localtime(&dummy);
  sprintf(error,"%s/.sendfile.error.%d",getenv("HOME"),(int)getpid());

  /* translate UTF-7 file name */
  utf = utf2iso(1,iso_name,normal_name,shell_name,flp->fname);

  /* tar archive? */
  if (flp->flags&F_TAR) {
    
    /* spool file compressed? */
    if (flp->flags&F_COMPRESS)
      sprintf(cmd,"gunzip < %s/%d.d | tar tf -",userspool,flp->id);
    else
      sprintf(cmd,"tar tf %s/%d.d",userspool,flp->id);
    
    /* open pipe to read tar file-info */
    if ((pipe=popen(cmd,"r")) == NULL) {
      message(prg,'E',"cannot open spool file for reading");
      return;
    }
    
    /* loop over all files in tar archive */
    while (fgets(fname,MAXLEN-1,pipe)) {
    
      /* does the file already exist? */
      if ((cp=strchr(fname,'\n'))) *cp = 0;
      if (overwrite!='Y' && stat(fname,&finfo)==0) {
        if (!tom) printf("archive %s will overwrite:\n",normal_name);
        tom = 1;
        printf("  %s\n",fname);
      }
    }
    
    /* ask user */
    if (tom && overwrite!='Y') {
      printf("Overwrite it (yYnN)? ");
      gets(tmp);
      overwrite = tmp[0];
      if (toupper(overwrite)!='Y') return;
    }

    sprintf(tmp,"receiving from archive %s :",normal_name);
    message(prg,'I',tmp);

    /* spool file compressed? */
    if (flp->flags&F_COMPRESS)
      sprintf(cmd,"gunzip < %s/%d.d | tar xvf - 2>%s",userspool,flp->id,error);
    else
      sprintf(cmd,"tar xvf %s/%d.d 2>%s",userspool,flp->id,error);
    
    /* receive tar archive */
    if (system(cmd)!=0) {  
      sprintf(tmp,"cannot receive %s",normal_name);
      message(prg,'E',tmp);
    } 
    
    /* check for errors */
    if (stat(error,&finfo)==0 && finfo.st_size>0) {
      errno = 0;
      sprintf(tmp,"errors while receive %s :",normal_name);
      message(prg,'E',tmp);
      inf = fopen(error,"r");
      while (fgets(tmp,MAXLEN,inf)) printf("%s",tmp);
      fclose(inf);
      sprintf(tmp,"leaving %s in spool intact",normal_name);
      message(prg,'I',tmp);
      unlink(error);
    } else {

      /* delete tar spool file */
      unlink(error);
      delete_sf(flp,0);
    }
    
    return;
  }
  
  /* need user request? */
  if (strchr("CNS",storeformat)==NULL) {
    
    /* found Unicode or 0x0? */
    if (utf==1) {
      printf("The next file name contains characters which are not allowed "
             "in Unix.\nThese characters have been substituted with '_'.\n");
    }
    
    /* found control or meta characters? */
    if (utf&2) {
      printf("The next file name contains characters which may cause problems "
             "with your shell or\nmay do strange things with your terminal.\n"
             "These characters have been substituted with '_'.\n");
      if (utf&1) 
        printf("Non-valid characters for Unix file names have been substituted"
               ", too.\n");
      
      /* let user choose file name format */
/*      if (strcmp(normal_name,shell_name)!=0) */
      printf("(c)omplete file name with control code characters is not "
             "displayable\n");
      printf("(n)ormal file name: >%s<\n",normal_name);
      printf("(s)hell file name:  >%s<\n",shell_name);
      do {
        printf("c or n may cause severe security problems! Kids, don't try this at home!");
        printf("Select one of c n s (or C N S for no more questions): ");
        gets(tmp);
        storeformat=tmp[0];
      } while (strchr("cnsCNS",storeformat)==NULL);

    }
  }

  /* set file name */
  switch (toupper(storeformat)) {
    case 'C': strcpy(fname,iso_name); break;
    case 'N': strcpy(fname,normal_name); break;
    case 'S': strcpy(fname,shell_name); break;
  }

  /* does the file already exist? */
  if (overwrite!='Y' && stat(fname,&finfo)==0) {
    printf("%s already exists. Overwrite it (yYnN)? ",fname);
    gets(tmp);
    overwrite = tmp[0];
    if (toupper(overwrite)!='Y') return;
  }

  /* file writeable? */
  if ((outf=fopen(fname,"w")) == NULL) {
    sprintf(tmp,"cannot open %s for writing",fname);
    message(prg,'E',tmp);
    return;
  }
  fclose(outf);
      
  /* spool file compressed? */
  if (flp->flags&F_COMPRESS) {
    
    /* source or text format? */
    if ((flp->flags&F_SOURCE) || (flp->flags&F_TEXT)) {
      
      /* open pipe to uncompress spool file */
      sprintf(cmd,"gunzip < %s/%d.d",userspool,flp->id);
      if ((pipe=popen(cmd,"r")) == NULL) {
        message(prg,'E',"cannot open spool file for reading");
        return;
     }
        
      /* open output file */
      if ((outf=fopen(fname,"w")) == NULL) {
        sprintf(tmp,"cannot open %s for writing",fname);
        message(prg,'E',tmp);
        fclose(pipe);
        return;
      }
      
      /* translate CR LF to LF and write to output file */
      crlf2lf(pipe,outf,fname);

      /* ready */
      pclose(pipe);
      fclose(outf);

    } else {  /* binary format */
      
      /* uncompress and receive binary file */
      sprintf(sfile,"%s/%d.d",userspool,flp->id);
      sad[0] = "gunzip";
      sad[1] = "-c";
      sad[2] = sfile;
      sad[3] = NULL;
      
      if (spawn(sad,fname,cmask)<0) {
	errno=0;
        sprintf(tmp,"cannot receive %s",fname);
        message(prg,'E',tmp);
      }
    }
  
  } else {  /* not compressed */
    
    /* source or text format? */
    if ((flp->flags&F_SOURCE) || (flp->flags&F_TEXT)) {

      /* open input file */
      sprintf(sfile,"%s/%d.d",userspool,flp->id);
      if ((inf=fopen(sfile,"r")) == NULL) {
        message(prg,'E',"cannot open spool file for reading");
        return;
      }
  
      /* open output file */
      if ((outf=fopen(fname,"w")) == NULL) {
        sprintf(tmp,"cannot open %s for writing",fname);
        message(prg,'E',tmp);
        fclose(inf);
        return;
      }

      /* translate CR LF to LF and write to output file */
      crlf2lf(inf,outf,fname);

      /* ready */
      pclose(inf);
      fclose(outf);

    } else {  /* binary file */

      /* copy file */
      sprintf(tmp,"%s/%d.d",userspool,flp->id);
      if (fcopy(tmp,fname,blksize,cmask)<0) {
        sprintf(tmp,"cannot receive %s",fname);
	errno=0;
        message(prg,'F',tmp);
      }

    }
  }

  /* delete spool file */
  delete_sf(flp,0);
  
  /* foreign character set in text file? */
  if ((flp->flags&F_TEXT) && strcmp(flp->charset,CHARSET)!=0) {
      
    /* call GNU recode */
    sprintf(tmp,"%s:"CHARSET,flp->charset);
    sad[0] = "recode";
    sad[1] = tmp;
    sad[2] = fname;
    sad[3] = NULL;
    if (spawn(sad,NULL,cmask)<0) {
      sprintf(tmp,"cannot translate character set in %s",fname);
      message(prg,'E',tmp);
    }

  }
  
  /* extract date in broken down format */
  sscanf(flp->date,"19%d-%d-%d %d:%d:%d",
	 &time.tm_year,
	 &time.tm_mon,
	 &time.tm_mday,
	 &time.tm_hour,
	 &time.tm_min,
	 &time.tm_sec);
  time.tm_mon--;
  time.tm_isdst = daylight*0; /* ?? */

  /* set the original date */
  utb.actime = mktime(&time)-timezone;
  utb.modtime = mktime(&time)-timezone;
  utime(fname,&utb);

  sprintf(tmp,"%s received",fname);
  message(prg,'I',tmp);
}

    
/* 
 * crlf2lf - translate NVT telnet format file to Unix text format file
 * 
 * INPUT:  inf	  - file to read from
 *         outf   - file to write to
 *         fname  - file name to write
 */
void crlf2lf(FILE *inf, FILE *outf, const char *fname) {
  int c1,c2;		/* characters to read in */
  char tmp[MAXLEN];	/* temporary string */

  /* read first char */
  c1 = fgetc(inf);
  
  /* loop until EOF */
  while ((c2=fgetc(inf)) != EOF) {
    
    /* crlf? */
    if (c1=='\r' && c2=='\n') {
      
      /* write lf */
      if(fputc(c2,outf)==EOF) {
        sprintf(tmp,"cannot write to %s",fname);
        message(prg,'E',tmp);
        return;
      }
      
      /* read next char */
      if ((c2=fgetc(inf)) == EOF) return;

    } else {
      
      /* write char */
      if(fputc(c1,outf)==EOF) {
        sprintf(tmp,"cannot write to %s",fname);
        message(prg,'E',tmp);
        return;
      }
      
    }
    c1 = c2;
  }
  
  /* write last char */
  if(fputc(c1,outf)==EOF) {
    sprintf(tmp,"cannot write to %s",fname);
    message(prg,'E',tmp);
    return;
  }

}


/*
 * fcopy - copy a file
 * 
 * INPUT:  from     - source file
 *         to       - destination file
 *         blksize  - block size
 *         cmask    - umask value
 * 
 * RETURN: 0 on success, -1 on failure
 * 
 */
int fcopy(const char *from, const char *to, 
	  unsigned long blksize, mode_t cmask) {
  int fdin, fdout;	/* file descriptor in/out */
  char tmp[MAXLEN],	/* temporary string */
       buf[blksize];	/* copy buffer */
  int bytes;		/* read bytes */


  /* open source file */
  fdin = open(from,O_RDONLY,0);
  if (fdin<0) {
    sprintf(tmp,"error opening %s",from);
    message(prg,'E',tmp);
    return(-1);
  }
  
  /* open destination file */
  fdout = creat(to,0666&~cmask);
  if (fdout<0) {
    close(fdin);
    sprintf(tmp,"error creating %s",to);
    message(prg,'E',tmp);
    return(-1);
  }
 
  /* read until EOF */
  while ((bytes=readn(fdin,buf,blksize)) > 0) {
    
    /* write to destination file */
    if (writen(fdout,buf,bytes)<0) {
      
      /* write error */
      close(fdin);
      close(fdout);
      sprintf(tmp,"error writing %s",to);
      message(prg,'E',tmp);
      return(-1);
    }
  }
  
  close(fdin);
  close(fdout);

  /* read error? */
  if (bytes<0) {
    sprintf(tmp,"error reading %s",from);
    message(prg,'E',tmp);
    return(-1);
  }

  return(0);
}
  

/* 
 * spawn  - spawn a subprocess and direct output to a file
 * 
 * INPUT:  sad     - spawn argument descriptor
 *         output  - output file
 * 
 * RETURN: 0 on success, -1 on failure
 * 
 */
int spawn(char **sad, const char *output, mode_t cmask) {
  int status,		/* fork status */
      fd;		/* output file descriptor */
  char tmp[MAXLEN];	/* temporary string */
  pid_t pid;		/* process id */
  
  /* spawn subprocess */
  pid = fork();
  
  /* failed? */
  if (pid<0) {
    message(prg,'E',"cannot fork");
    return(-1);
  }

  /* is this the subprocess? */
  if (pid==0) {
    
    /* redirect stdout? */
    if (output) {
      
      /* close stdout */
      close(1);
    
      /* open output file as stdout */
      fd = creat(output,0666&~cmask);
      if (fd!=1) {
	message(prg,'E',"file descriptor mismatch");
	return(-1);
      }
    
    }
    
    /* execute program */
    execvp(sad[0],sad);
    
    /* oops - failed */
    exit(1);
  }
  
  /* wait for termination of subprocess */
#ifdef NEXT
  wait(&status);
#else
  waitpid(pid,&status,0);
#endif
  
  /* error in subprocess? */
  if (status) {
    errno=0;
    sprintf(tmp,"%s failed",sad[0]);
    message(prg,'E',tmp);
    return(-1);
  }
  
  return(0);
}


/*
 * usage - print short help usage text
 */
int usage() {
  fprintf(stderr,"usage: %s [-d] file-name\n",prg);
  fprintf(stderr,"   or: %s -n [-d] file-number\n",prg);
  fprintf(stderr,"   or: %s [-l] [-L]\n",prg);
  fprintf(stderr,"   or: %s -a [-d]\n",prg);
  fprintf(stderr,"   -l  short list of files\n");
  fprintf(stderr,"   -L  full list of files\n");
  fprintf(stderr,"   -n  specify a file number\n");
  fprintf(stderr,"   -a  specify all files\n");
  fprintf(stderr,"   -d  delete\n");
  return(2);
}
