/*
 * File:	net.c
 * 
 * Author:	Ulli Horlacher (framstag@rus.uni-stuttgart.de)
 * 
 * History:	11 Aug 95   Framstag	initial version
 *         	10 Sep 95   Framstag	some debugging 
 * 
 * Network routines for the the sendfile client of the sendfile package.
 * Look at net.h for list of the functions.
 * 
 * This file is covered by the GNU General Public License
 */

/*
#ifdef NEXT
  typedef unsigned char	 u_char;
  typedef unsigned short u_short;
  typedef unsigned int	 u_int;
  typedef unsigned long	 u_long;
  typedef long	daddr_t;
  typedef char *caddr_t;
  typedef long  time_t;
  #define _TIME_T
#endif
*/

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <netdb.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "config.h"	/* various definitions */
#include "message.h"	/* information, warning and error messages */
#include "net.h"	/* network stuff */
#include "io.h"		/* socket read/write */


/* stupid AIX comes with no include files for networking and other stuff */
#if defined(AIX) || defined(ULTRIX)
  #include "bsd.h"
#endif

#ifdef IRIX
  u_short htons(u_short hostshort);
#endif


/*
 * open_connection - open socket and connect to client
 * 
 * INPUT:  adr  - ip address of server to connect to
 *         port - port number to connect to
 * 
 * RETURN: socket file descriptor
 * 
 * this function is derived from example code from
 * "Unix Networking Programming" by W. R. Stevens
 */
int open_connection(char *adr, int port) {
  int sockfd,			/* socket file descriptor */
      num=1;			/* flag for numeric ip address */
  struct sockaddr_in serv_addr;	/* internet socket */
  struct in_addr hostaddr;
  struct hostent *hostp;	/* host entity */
  char *cp,			/* character pointer */
       hostname[16],		/* server host name */
       msg[MAXLEN];		/* intermediate information/error message */
  extern char *prg;		/* name of the game */

  /* open socket */
  sockfd = socket(AF_INET,SOCK_STREAM,0);
  if (sockfd<0) message(prg,'F',"cannot open connection");
#ifdef DEBUG
  message(prg,'I',"socket ok");
#endif  

  /* initialisize serv_addr */
  memset((char *) &serv_addr, 0, sizeof(serv_addr));
  
  /* numeric oder symbolic ip address? */
  for (cp=adr; *cp>0; cp++) {
    if (*cp>'@'){
      num=0;
      break;
    }
  }

  /* look for server host address */
  if (num) {
    hostaddr.s_addr = inet_addr(adr);
    hostp = gethostbyaddr((char *)&hostaddr,sizeof(hostaddr),AF_INET);
  } else
    hostp = gethostbyname(adr);
  if (hostp==NULL) {
    errno = 0;
    sprintf(msg,"unknown host %s",adr);
    message(prg,'F',msg);
  }
  
  /* convert binary structure to ASCII hostname */
  strcpy(hostname,inet_ntoa(*(struct in_addr *) *hostp->h_addr_list));
#ifdef DEBUG
  printf("host: %s\n",hostname);
#endif  
  
  /* fill out server address descriptor */
  serv_addr.sin_family      = AF_INET;
  serv_addr.sin_addr.s_addr = inet_addr(hostname);
  serv_addr.sin_port        = htons(PORT);
  
  /* connect to server */
  if (connect(sockfd,(struct sockaddr *) &serv_addr,sizeof(serv_addr)) < 0)
    message(prg,'F',"cannot connect to server");
#ifdef DEBUG
  message(prg,'I',"connect ok");
#endif  

  return(sockfd);
}


/*
 * sock_getline - get a line from the network socket
 * 
 * INPUT:  fd     - socket file descriptor
 *         ptr    - empty string
 * 
 * RETURN: number of read bytes
 * 
 * this function is derived from example code from
 * "Unix Networking Programming" by W. R. Stevens
 */
int sock_getline(int fd, char *ptr) {
  int n, rc, cr = 0;
  unsigned char c;
  
  ptr[0]=0;
  for (n=0; n-1<MAXLEN; n++) {
    if ((rc = read(fd, &c, 1))==1) {
      ptr[n] = c;
      if (c=='\n' && cr==1) break;
      if (c=='\r') cr = 1; else cr = 0;
    } else {
      if (rc==0) 
        if (n==1) return(0); else break;
      else
      	return(-1);
    }
  }
  ptr[--n] = 0;
  return(n);
}


/*
 * sock_putline - send a line to the network socket
 * 
 * INPUT:  fd     - socket file descriptor
 *         ptr    - string to send
 * 
 * RETURN: number of send bytes
 */
int sock_putline(int fd, char *ptr) {
  int n;		/* number of send bytes */
  char cmd[MAXLEN];	/* command line to send */
  extern int verbose;	/* flag for verbose mode */
  
  /* prepare string */
  strcpy(cmd,ptr);
  strcat(cmd,"\r\n");
  
  /* on verbose mode show what goes up */
  if (verbose) printf("-> %s\n",ptr);
  
  /* and up and away :-) */
  n = writen(fd,cmd,strlen(cmd));
  
  return(n);
}


/* 
 * getreply - get the reply on a command from the server
 * 
 * INPUT:  fd - socket file descriptor
 * 
 * RETURN: the reply line string
 */
char *getreply(int fd) {
  int len;			/* reply message length */
  char replycode[5],		/* 3 digit reply code */
       msg[MAXLEN];		/* intermediate information/error message */
  static char reply[MAXLEN];	/* reply string from server */
  extern int verbose;		/* flag for verbose mode */
  extern char *prg;		/* name of the game */

  /* get the answer */
  len = sock_getline(fd,reply);

  /* on verbose mode show the whole line */
  if (verbose) printf("%s\n",reply);
  
  /* reply message too short? */
  if (len<4) message(prg,'F',"corrupt reply, connection closed");
  
  /* cut the reply code */
  strncpy(replycode,reply,3);
  replycode[3]=' ';
  replycode[4]=0;
#ifdef DEBUG
  printf("replycode: >%s<\n",replycode);
#endif
  
  /* reply text continues on the next line? */
  if (reply[3]=='-')
    
    /* repeat the game as above */
    do {
      if (sock_getline(fd,reply)<4) message(prg,'F',"connection closed");
      if (verbose) printf("%s\n",reply);
    } while (strncmp(reply,replycode,4)!=0);

  /* quit if there was a fatal server error */
  if (reply[0]=='4') {
    errno = 0;
    sprintf(msg,"server error: %s",&reply[4]);
    message(prg,'F',msg);
  }
  
  return(reply);
}


/*
 * sendheader - send a headerline and check the reply code
 * 
 * INPUT:  fd	- socket file descriptor
 *         line	- header line
 * 
 * RETURN: 0 on sucess, -1 on server error
 */
int sendheader(int fd, char *line) {
  char msg[MAXLEN],		/* intermediate information/error message */
       *reply,			/* reply string from server */
       severity;		/* severity of server error */
  extern char *prg;		/* name of the game */
  
  /* send the header line */
  sock_putline(fd,line);
  
  /* server reply ok? */
  reply = getreply(fd);
  if (strncmp(reply,"200 ",4)!=0) {
    errno = 0;
    
    /* save error text */
    sprintf(msg,"server error: %s",&reply[4]);

    /* kind of error? */
    if (*reply=='4') 
      severity = 'F';
    else
      severity = 'E';
    message(prg,severity,msg);
    return(-1);

  } else
    return(0);

}
