/*  *********************************************************************
    File: sslsess.c

    SSLRef 3.0 Final -- 11/19/96

    Copyright (c)1996 by Netscape Communications Corp.

    By retrieving this software you are bound by the licensing terms
    disclosed in the file "LICENSE.txt". Please read it, and if you don't
    accept the terms, delete this software.

    SSLRef 3.0 was developed by Netscape Communications Corp. of Mountain
    View, California <http://home.netscape.com/> and Consensus Development
    Corporation of Berkeley, California <http://www.consensus.com/>.

    *********************************************************************

    File: sslsess.c    SSL Session DB interface

    This file contains functions which interact with the session database
    to store and restore sessions and retrieve information from packed
    session records.

    ****************************************************************** */

#ifndef _SSL_H_
#include "ssl.h"
#endif

#ifndef _SSLCTX_H_
#include "sslctx.h"
#endif

#ifndef _SSLSESS_H_
#include "sslsess.h"
#endif

#ifndef _SSLALLOC_H_
#include "sslalloc.h"
#endif

#ifndef _X509_H_
#include "x509.h"
#endif

#include <string.h>
#include <stddef.h>

typedef struct
{   int                 sessionIDLen;
    uint8               sessionID[32];
    SSLProtocolVersion  protocolVersion;
    uint16              cipherSuite;
    uint8               masterSecret[48];
    int                 certCount;
    uint8               certs[1];   /* Actually, variable length */
} ResumableSession;

SSLErr
SSLAddSessionID(SSLContext *ctx)
{   SSLErr              err;
    uint32              sessionIDLen;
    SSLBuffer           sessionID;
    ResumableSession    *session;
    int                 certCount;
    SSLCertificate      *cert;
    uint8               *certDest;
    
    /* If we don't know who the peer is, we can't store a session */
    if (ctx->peerID.data == 0)
        return SSLSessionNotFoundErr;
    
    sessionIDLen = offsetof(ResumableSession, certs);
    cert = ctx->peerCert;
    certCount = 0;
    while (cert)
    {   ++certCount;
        sessionIDLen += 4 + cert->derCert.length;
        cert = cert->next;
    }
    
    if ((err = SSLAllocBuffer(&sessionID, sessionIDLen, &ctx->sysCtx)) != 0)
        return err;
    
    session = (ResumableSession*)sessionID.data;
    
    session->sessionIDLen = ctx->sessionID.length;
    memcpy(session->sessionID, ctx->sessionID.data, session->sessionIDLen);
    session->protocolVersion = ctx->protocolVersion;
    session->cipherSuite = ctx->selectedCipher;
    memcpy(session->masterSecret, ctx->masterSecret, 48);
    session->certCount = certCount;
    
    certDest = session->certs;
    cert = ctx->peerCert;
    while (cert)
    {   certDest = SSLEncodeInt(certDest, cert->derCert.length, 4);
        memcpy(certDest, cert->derCert.data, cert->derCert.length);
        certDest += cert->derCert.length;
        cert = cert->next;
    }
    
    err = ctx->sessionCtx.addSession(ctx->peerID, sessionID, ctx->sessionCtx.sessionRef);
    SSLFreeBuffer(&sessionID, &ctx->sysCtx);
    
    return err;
}

SSLErr
SSLGetSessionID(SSLBuffer *sessionID, SSLContext *ctx)
{   SSLErr      err;
    
    if (ctx->peerID.data == 0)
        return ERR(SSLSessionNotFoundErr);
    
    sessionID->data = 0;
    
    ERR(err = ctx->sessionCtx.getSession(ctx->peerID, sessionID, ctx->sessionCtx.sessionRef));
    
    if (sessionID->data == 0)
        return ERR(SSLSessionNotFoundErr);
    
    return err;
}

SSLErr
SSLDeleteSessionID(SSLContext *ctx)
{   SSLErr      err;
    
    if (ctx->peerID.data == 0)
        return SSLSessionNotFoundErr;
    
    err = ctx->sessionCtx.deleteSession(ctx->peerID, ctx->sessionCtx.sessionRef);
    
    return err;
}

SSLErr
SSLRetrieveSessionIDIdentifier(SSLBuffer sessionID, SSLBuffer *identifier, SSLContext *ctx)
{   SSLErr              err;
    ResumableSession    *session;
    
    session = (ResumableSession*) sessionID.data;
    if ((err = SSLAllocBuffer(identifier, session->sessionIDLen, &ctx->sysCtx)) != 0)
        return err;
    memcpy(identifier->data, session->sessionID, session->sessionIDLen);
    return SSLNoErr;
}

SSLErr
SSLRetrieveSessionIDProtocolVersion(SSLBuffer sessionID, SSLProtocolVersion *version, SSLContext *ctx)
{   ResumableSession    *session;
    
    session = (ResumableSession*) sessionID.data;
    *version = session->protocolVersion;
    return SSLNoErr;
}

SSLErr
SSLInstallSessionID(SSLBuffer sessionID, SSLContext *ctx)
{   SSLErr              err;
    ResumableSession    *session;
    uint8               *storedCertProgress;
    SSLCertificate      *cert, *lastCert;
    SSLBuffer           certAlloc;
    int                 certCount;
    uint32              certLen;
    
    session = (ResumableSession*)sessionID.data;
    
    ASSERT(ctx->protocolVersion == session->protocolVersion);
    
    ctx->selectedCipher = session->cipherSuite;
    if ((err = FindCipherSpec(ctx->selectedCipher, &ctx->selectedCipherSpec)) != 0)
        return err;
    memcpy(ctx->masterSecret, session->masterSecret, 48);
    
    lastCert = 0;
    storedCertProgress = session->certs;
    certCount = session->certCount;
    while (certCount--)
    {   if ((err = SSLAllocBuffer(&certAlloc, sizeof(SSLCertificate), &ctx->sysCtx)) != 0)
            return err;
        cert = (SSLCertificate*)certAlloc.data;
        cert->next = 0;
        certLen = SSLDecodeInt(storedCertProgress, 4);
        storedCertProgress += 4;
        if ((err = SSLAllocBuffer(&cert->derCert, certLen, &ctx->sysCtx)) != 0)
        {   SSLFreeBuffer(&certAlloc,&ctx->sysCtx);
            return err;
        }
        memcpy(cert->derCert.data, storedCertProgress, certLen);
        storedCertProgress += certLen;
        if ((err = ASNParseX509Certificate(cert->derCert, &cert->cert, ctx)) != 0)
        {   SSLFreeBuffer(&cert->derCert,&ctx->sysCtx);
            SSLFreeBuffer(&certAlloc,&ctx->sysCtx);
            return err;
        }
        if (lastCert == 0)
            ctx->peerCert = cert;
        else
            lastCert->next = cert;
        lastCert = cert;
    }
    
    return SSLNoErr;
}
