/*  *********************************************************************
    File: digests.c

    SSLRef 3.0 Final -- 11/19/96

    Copyright (c)1996 by Netscape Communications Corp.

    By retrieving this software you are bound by the licensing terms
    disclosed in the file "LICENSE.txt". Please read it, and if you don't
    accept the terms, delete this software.

    SSLRef 3.0 was developed by Netscape Communications Corp. of Mountain
    View, California <http://home.netscape.com/> and Consensus Development
    Corporation of Berkeley, California <http://www.consensus.com/>.

    *********************************************************************

    File: digests.c    Hashing support functions and data structures

    Contains interface functions which generalize hashing support for MD5
    and SHA1 and a dummy null hash implementation (used before MACing is
    turned on). Also, utility functions for using the hashes.

    ****************************************************************** */

#ifndef _CRYPTYPE_H_
#include "cryptype.h"
#endif

#ifndef _SSLCRYPT_H_
#include "sslcrypt.h"
#endif

#ifndef SHA_H
#include <stdio.h>      /* sha.h has a prototype with a FILE* */
#include "sha.h"
#endif

#ifndef _SSLALLOC_H_
#include "sslalloc.h"
#endif

#ifndef _SSLCTX_H_
#include "sslctx.h"
#endif

#include <string.h>

typedef struct
{   SHA_INFO    sha;
    int         bufferPos;
    uint8       dataBuffer[SHA_BLOCKSIZE];
} SSL_SHA_INFO;

uint8   SSLMACPad1[MAX_MAC_PADDING], SSLMACPad2[MAX_MAC_PADDING];

void
SSLInitMACPads(void)
{   int     i;
    
    for (i = 0; i < MAX_MAC_PADDING; i++)
    {   SSLMACPad1[i] = 0x36;
        SSLMACPad2[i] = 0x5C;
    }
}

SSLErr
CloneHashState(HashReference *ref, SSLBuffer state, SSLBuffer *newState, SSLContext *ctx)
{   SSLErr      err;
    if ((err = SSLAllocBuffer(newState, state.length, &ctx->sysCtx)) != 0)
        return err;
    memcpy(newState->data, state.data, state.length);
#if DEBUGDATA
    if (ref == &SSLHashSHA1)
        DEBUGVAL2("SHA1 cloned from %08X to %08X", state.data, newState->data);
#endif /* DEBUGDATA */
    return SSLNoErr;
}

SSLErr
ReadyHash(HashReference *ref, SSLBuffer *state, SSLContext *ctx)
{   SSLErr      err;
    if ((err = SSLAllocBuffer(state, ref->contextSize, &ctx->sysCtx)) != 0)
        return err;
    if ((err = ref->init(*state)) != 0)
        return err;
    return SSLNoErr;
}

SSLErr HashNullInit(SSLBuffer);
SSLErr HashNullUpdate(SSLBuffer,SSLBuffer);
SSLErr HashNullFinal(SSLBuffer,SSLBuffer);
SSLErr HashNullClone(SSLBuffer,SSLBuffer);

SSLErr HashMD2Init(SSLBuffer digestCtx);
SSLErr HashMD2Update(SSLBuffer digestCtx, SSLBuffer data);
SSLErr HashMD2Final(SSLBuffer digestCtx, SSLBuffer digest);
SSLErr HashMD2Clone(SSLBuffer src, SSLBuffer dest);

SSLErr HashMD5Init(SSLBuffer digestCtx);
SSLErr HashMD5Update(SSLBuffer digestCtx, SSLBuffer data);
SSLErr HashMD5Final(SSLBuffer digestCtx, SSLBuffer digest);
SSLErr HashMD5Clone(SSLBuffer src, SSLBuffer dest);

SSLErr HashSHA1Init(SSLBuffer digestCtx);
SSLErr HashSHA1Update(SSLBuffer digestCtx, SSLBuffer data);
SSLErr HashSHA1Final(SSLBuffer digestCtx, SSLBuffer digest);
SSLErr HashSHA1Clone(SSLBuffer src, SSLBuffer dest);

HashReference SSLHashNull = { 0, 0, 0, HashNullInit, HashNullUpdate, HashNullFinal, HashNullClone };
#if RSAREF
HashReference SSLHashMD5 = { sizeof(R_DIGEST_CTX), 16, 48, HashMD5Init, HashMD5Update, HashMD5Final, HashMD5Clone };
#elif BSAFE
HashReference SSLHashMD5 = { sizeof(MD5_CTX), 16, 48, HashMD5Init, HashMD5Update, HashMD5Final, HashMD5Clone };
#endif
HashReference SSLHashSHA1 = { sizeof(SSL_SHA_INFO), 20, 40, HashSHA1Init, HashSHA1Update, HashSHA1Final, HashSHA1Clone };

/*** NULL ***/
SSLErr HashNullInit(SSLBuffer digestCtx) { return SSLNoErr; }
SSLErr HashNullUpdate(SSLBuffer digestCtx, SSLBuffer data) { return SSLNoErr; }
SSLErr HashNullFinal(SSLBuffer digestCtx, SSLBuffer digest) { return SSLNoErr; }
SSLErr HashNullClone(SSLBuffer src, SSLBuffer dest) { return SSLNoErr; }

/*** MD5 ***/
#if RSAREF
    SSLErr HashMD5Init(SSLBuffer digestCtx)
    {   ASSERT(digestCtx.length >= sizeof(R_DIGEST_CTX));
        R_DigestInit((R_DIGEST_CTX*)digestCtx.data, DA_MD5);
        return SSLNoErr;
    }

    SSLErr HashMD5Update(SSLBuffer digestCtx, SSLBuffer data)
    {   ASSERT(digestCtx.length >= sizeof(R_DIGEST_CTX));
        R_DigestUpdate((R_DIGEST_CTX*)digestCtx.data, data.data, data.length);
        return SSLNoErr;
    }

    SSLErr HashMD5Final(SSLBuffer digestCtx, SSLBuffer digest)
    {   unsigned int    digestLen;
        ASSERT(digestCtx.length >= sizeof(R_DIGEST_CTX));
        ASSERT(digest.length >= 16);
        R_DigestFinal((R_DIGEST_CTX*)digestCtx.data, digest.data, &digestLen);
        ASSERT(digestLen == 16);
        return SSLNoErr;
    }

    SSLErr HashMD5Clone(SSLBuffer src, SSLBuffer dest)
    {   if (src.length != dest.length)
            return SSLProtocolErr;
        memcpy(dest.data, src.data, src.length);
        return SSLNoErr;
    }
#elif BSAFE
/* I can't use a BSAFE implementation of MD5 or SHA beacuse
    SSLRef requires clonability from its MD5 and SHA hashes
    and it can't be done with BSAFE, as there isn't any way
    to duplicate an ALGORITHM_OBJ */
    SSLErr HashMD5Init(SSLBuffer digestCtx)
    {   ASSERT(digestCtx.length >= sizeof(MD5_CTX));
        MD5Init((MD5_CTX*)digestCtx.data);
        return SSLNoErr;
    }

    SSLErr HashMD5Update(SSLBuffer digestCtx, SSLBuffer data)
    {   ASSERT(digestCtx.length >= sizeof(MD5_CTX));
        MD5Update((MD5_CTX*)digestCtx.data, data.data, data.length);
        return SSLNoErr;
    }

    SSLErr HashMD5Final(SSLBuffer digestCtx, SSLBuffer digest)
    {   ASSERT(digestCtx.length >= sizeof(MD5_CTX));
        ASSERT(digest.length >= 16);
        MD5Final(digest.data, (MD5_CTX*)digestCtx.data);
        digest.length = 16;
        return SSLNoErr;
    }

    SSLErr HashMD5Clone(SSLBuffer src, SSLBuffer dest)
    {   if (src.length != dest.length)
            return SSLProtocolErr;
        memcpy(dest.data, src.data, src.length);
        return SSLNoErr;
    }   
#endif /* RSAREF / BSAFE */

/*** SHA ***/
SSLErr HashSHA1Init(SSLBuffer digestCtx)
{   SSL_SHA_INFO    *ctx = (SSL_SHA_INFO*)digestCtx.data;
    ASSERT(digestCtx.length >= sizeof(SSL_SHA_INFO));
    sha_init(&ctx->sha);
    ctx->bufferPos = 0;
#if DEBUGDATA
    DEBUGVAL1("SHA1 initialized %08X", digestCtx.data);
#endif
    return SSLNoErr;
}

SSLErr HashSHA1Update(SSLBuffer digestCtx, SSLBuffer data)
{   SSL_SHA_INFO    *ctx = (SSL_SHA_INFO*)digestCtx.data;
    uint32          dataRemaining, processed;
    uint8           *dataPos;
    
    ASSERT(digestCtx.length >= sizeof(SSL_SHA_INFO));
    dataRemaining = data.length;
    dataPos = data.data;
    while (dataRemaining > 0)
    {   processed = SHA_BLOCKSIZE - ctx->bufferPos;
        if (dataRemaining < processed)
            processed = dataRemaining;
        memcpy(ctx->dataBuffer+ctx->bufferPos, dataPos, processed);
        ctx->bufferPos += processed;
        if (ctx->bufferPos == SHA_BLOCKSIZE)
        {   sha_update(&ctx->sha, ctx->dataBuffer, ctx->bufferPos);
            ctx->bufferPos = 0;
        }
        dataRemaining -= processed;
        dataPos += processed;
    }
    DUMP_BUFFER_PTR("SHA1 data", digestCtx.data, data);
    return SSLNoErr;
}

SSLErr HashSHA1Final(SSLBuffer digestCtx, SSLBuffer digest)
{   SSL_SHA_INFO    *ctx = (SSL_SHA_INFO*)digestCtx.data;
    ASSERT(digestCtx.length >= sizeof(SSL_SHA_INFO));
    ASSERT(digest.length >= SHA_DIGESTSIZE);
    if (ctx->bufferPos > 0)
        sha_update(&ctx->sha, ctx->dataBuffer, ctx->bufferPos);
    sha_final((SHA_INFO*)digestCtx.data);
    memcpy(digest.data, ((SHA_INFO*)digestCtx.data)->digest, 20);
    DUMP_BUFFER_PTR("SHA1 final", digestCtx.data, digest);
    return SSLNoErr;
}

SSLErr HashSHA1Clone(SSLBuffer src, SSLBuffer dest)
{   if (src.length != dest.length)
        return SSLProtocolErr;
    memcpy(dest.data, src.data, src.length);
    return SSLNoErr;
}   
