/*  *********************************************************************
    File: ciphers.c

    SSLRef 3.0 Final -- 11/19/96

    Copyright (c)1996 by Netscape Communications Corp.

    By retrieving this software you are bound by the licensing terms
    disclosed in the file "LICENSE.txt". Please read it, and if you don't
    accept the terms, delete this software.

    SSLRef 3.0 was developed by Netscape Communications Corp. of Mountain
    View, California <http://home.netscape.com/> and Consensus Development
    Corporation of Berkeley, California <http://www.consensus.com/>.

    *********************************************************************

    File: ciphers.c    Data structures for handling supported ciphers

    Contains a table mapping cipherSuite values to the ciphers, MAC
    algorithms, key exchange procedures and so on that are used for that
    algorithm, in order of preference.

    ****************************************************************** */

#ifndef _CRYPTYPE_H_
#include "cryptype.h"
#endif

#ifndef _SSLCTX_H_
#include "sslctx.h"
#endif

#include <string.h>

extern SSLSymmetricCipher SSLCipherNull;
extern SSLSymmetricCipher SSLCipherDES_CBC;
#if BSAFE || HAS_RSAREF_RC4
extern SSLSymmetricCipher SSLCipherRC4_40;
extern SSLSymmetricCipher SSLCipherRC4_128;
#endif

/* Even if we don't support NULL_WITH_NULL_NULL for transport, we need a reference for startup */
SSLCipherSpec SSL_NULL_WITH_NULL_NULL_CipherSpec =
{   SSL_NULL_WITH_NULL_NULL,
    Exportable,
    SSL_NULL_auth,
    &SSLHashNull,
    &SSLCipherNull
};

/* Order by preference */
SSLCipherSpec KnownCipherSpecs[] =
{
    {   SSL_RSA_WITH_DES_CBC_SHA, NotExportable, SSL_RSA, &SSLHashSHA1, &SSLCipherDES_CBC },
#if BSAFE || HAS_RSAREF_RC4
    {   SSL_RSA_EXPORT_WITH_RC4_40_MD5, Exportable, SSL_RSA_EXPORT, &SSLHashMD5, &SSLCipherRC4_40 },
    {   SSL_DH_anon_WITH_RC4_128_MD5, NotExportable, SSL_DH_anon, &SSLHashMD5, &SSLCipherRC4_128 },
#endif
    {   SSL_RSA_WITH_NULL_MD5, Exportable, SSL_RSA, &SSLHashMD5, &SSLCipherNull }
};

int CipherSpecCount = sizeof(KnownCipherSpecs) / sizeof(SSLCipherSpec);

SSLErr
FindCipherSpec(uint16 specID, SSLCipherSpec* *spec)
{   int i;
    
    *spec = 0;
    for (i = 0; i < CipherSpecCount; i++)
    {   if (KnownCipherSpecs[i].cipherSpec == specID)
        {   *spec = &KnownCipherSpecs[i];
            break;
        }
    }
    
    if (*spec == 0)         /* Not found */
        return SSLNegotiationErr;
    return SSLNoErr;
}

SSLErr SSLDESInit(uint8 *key, uint8* iv, void **cipherRef, SSLContext *ctx);
SSLErr SSLDESEncrypt(SSLBuffer src, SSLBuffer dest, void *cipherRef, SSLContext *ctx);
SSLErr SSLDESDecrypt(SSLBuffer src, SSLBuffer dest, void *cipherRef, SSLContext *ctx);
SSLErr SSLDESFinish(void *cipherRef, SSLContext *ctx);

SSLSymmetricCipher SSLCipherDES_CBC = {
    8,      /* Key size in bytes */
    8,      /* Secret key size = 64 bits */
    8,      /* IV size */
    8,      /* Block size */
    SSLDESInit,
    SSLDESEncrypt,
    SSLDESDecrypt,
    SSLDESFinish
};

#if BSAFE
SSLErr
SSLDESInit(uint8 *key, uint8* iv, void **cipherRef, SSLContext *ctx)
{   SSLBuffer                   desState;
    B_ALGORITHM_OBJ             *des;
    static B_ALGORITHM_METHOD  *chooser[] = { &AM_DES_CBC_ENCRYPT, &AM_DES_CBC_DECRYPT, 0 };
    B_KEY_OBJ                   desKey;
    ITEM                        keyData;
    SSLErr                      err;
    int                         rsaErr;

    if ((err = SSLAllocBuffer(&desState, sizeof(B_ALGORITHM_OBJ), &ctx->sysCtx)) != 0)
        return err;
    des = (B_ALGORITHM_OBJ*)desState.data;
    if ((rsaErr = B_CreateAlgorithmObject(des)) != 0)
        return SSLUnknownErr;
    if ((rsaErr = B_SetAlgorithmInfo(*des, AI_DES_CBC_IV8, iv)) != 0)
        return SSLUnknownErr;
    if ((rsaErr = B_CreateKeyObject(&desKey)) != 0)
        return SSLUnknownErr;
    keyData.data = key;
    keyData.len = 8;
    if ((rsaErr = B_SetKeyInfo(desKey, KI_DES8, key)) != 0)
    {   B_DestroyKeyObject(&desKey);
        return SSLUnknownErr;
    }
    if (cipherRef == (void**)&(ctx->writePending.symCipherState))
    {   if ((rsaErr = B_EncryptInit(*des, desKey, chooser, NO_SURR)) != 0)
        {   B_DestroyKeyObject(&desKey);
            return SSLUnknownErr;
        }
    }
    else if (cipherRef == (void**)&(ctx->readPending.symCipherState))
    {   if ((rsaErr = B_DecryptInit(*des, desKey, chooser, NO_SURR)) != 0)
        {   B_DestroyKeyObject(&desKey);
            return SSLUnknownErr;
        }
    }
    else
        ASSERTMSG("Couldn't determine read/writeness");

    B_DestroyKeyObject(&desKey);
    *cipherRef = (void*)desState.data;
    return SSLNoErr;
}

SSLErr
SSLDESEncrypt(SSLBuffer src, SSLBuffer dest, void *cipherRef, SSLContext *ctx)
{   B_ALGORITHM_OBJ     *des = (B_ALGORITHM_OBJ*)cipherRef;
    int                 rsaErr;
    unsigned int        outputLen;
    SSLBuffer           temp;
    SSLErr              err;
    
    ASSERT(src.length == dest.length);
    ASSERT(src.length % 8 == 0);
    
    if (src.data == dest.data)
/* BSAFE won't let you encrypt in place */
    {   if (ERR(err = SSLAllocBuffer(&temp, src.length, &ctx->sysCtx)) != 0)
            return err;
        memcpy(temp.data, src.data, src.length);
    }
    else
        temp = src;
    
    if ((rsaErr = B_EncryptUpdate(*des, dest.data, &outputLen, dest.length, temp.data, temp.length,
                    (B_ALGORITHM_OBJ) 0, NO_SURR)) != 0)
    {   if (src.data == dest.data)
            SSLFreeBuffer(&temp, &ctx->sysCtx);
        return SSLUnknownErr;
    }
    
    ASSERT(outputLen == src.length);
    
    if (src.data == dest.data)
        SSLFreeBuffer(&temp, &ctx->sysCtx);
    
    if (outputLen != src.length)
        return SSLUnknownErr;
    
    return SSLNoErr;
}

SSLErr
SSLDESDecrypt(SSLBuffer src, SSLBuffer dest, void *cipherRef, SSLContext *ctx)
{   B_ALGORITHM_OBJ     *des = (B_ALGORITHM_OBJ*)cipherRef;
    int                 rsaErr;
    unsigned int        outputLen;
    SSLBuffer           temp;
    SSLErr              err;
    
    ASSERT(src.length == dest.length);
    ASSERT(src.length % 8 == 0);
    
    if (src.data == dest.data)
/* BSAFE won't let you encrypt in place */
    {   if (ERR(err = SSLAllocBuffer(&temp, src.length, &ctx->sysCtx)) != 0)
            return err;
        memcpy(temp.data, src.data, src.length);
    }
    else
        temp = src;
    
    if ((rsaErr = B_DecryptUpdate(*des, dest.data, &outputLen, dest.length, temp.data, temp.length,
                    (B_ALGORITHM_OBJ) 0, NO_SURR)) != 0)
    {   if (src.data == dest.data)
            SSLFreeBuffer(&temp, &ctx->sysCtx);
        return SSLUnknownErr;
    }
    
    ASSERT(outputLen == src.length);
    
    if (src.data == dest.data)
        SSLFreeBuffer(&temp, &ctx->sysCtx);
    
    if (outputLen != src.length)
        return SSLUnknownErr;
    
    return SSLNoErr;
}

SSLErr
SSLDESFinish(void *cipherRef, SSLContext *ctx)
{   B_ALGORITHM_OBJ     *des = (B_ALGORITHM_OBJ*)cipherRef;
    SSLBuffer           desState;
    SSLErr              err;
    
    B_DestroyAlgorithmObject(des);
    desState.data = (unsigned char*)cipherRef;
    desState.length = sizeof(B_ALGORITHM_OBJ);
    err = SSLFreeBuffer(&desState, &ctx->sysCtx);
    return err;
}

#else

#include "deseay.h"

struct DESState
{   des_key_schedule    key;
    uint8               iv[8];
};

SSLErr
SSLDESInit(uint8 *key, uint8* iv, void **cipherRef, SSLContext *ctx)
{   SSLBuffer           desState;
    struct DESState     *des;
    SSLErr              err;
    
    if ((err = SSLAllocBuffer(&desState, sizeof(struct DESState), &ctx->sysCtx)) != 0)
        return err;
    des = (struct DESState*)desState.data;
    
    des_set_odd_parity((des_cblock*)key);
    des_key_sched((des_cblock*)key,des->key);
    memcpy(des->iv, iv, 8);

    *cipherRef = (void*)desState.data;
    return SSLNoErr;
}

SSLErr
SSLDESEncrypt(SSLBuffer src, SSLBuffer dest, void *cipherRef, SSLContext *ctx)
{   struct DESState     *des = (struct DESState*)cipherRef;
    SSLErr              err;
    
    ASSERT(src.length == dest.length);
    ASSERT(src.length % 8 == 0);
    
    if (src.length > 0)
    {   des_cbc_encrypt((des_cblock*)src.data,(des_cblock*)dest.data,
                        src.length,des->key,(des_cblock*)(des->iv),1);
        memcpy(des->iv, dest.data + dest.length - 8, 8);    /* Save the last block as new iv */
    }
    
    return SSLNoErr;
}

SSLErr
SSLDESDecrypt(SSLBuffer src, SSLBuffer dest, void *cipherRef, SSLContext *ctx)
{   struct DESState     *des = (struct DESState*)cipherRef;
    SSLErr              err;
    uint8               savedIV[8];
    
    ASSERT(src.length == dest.length);
    ASSERT(src.length % 8 == 0);
    
    if (src.data == dest.data)
        memcpy(savedIV, dest.data + dest.length - 8, 8);    /* Save the last block as future iv */
    
    if (src.length > 0)
    {   des_cbc_encrypt((des_cblock*)src.data,(des_cblock*)dest.data,
                        src.length,des->key,(des_cblock*)(des->iv),0);
    }
    
    if (src.data == dest.data)
        memcpy(des->iv, savedIV, 8);
    else
        memcpy(des->iv, dest.data + dest.length - 8, 8);    /* Save the last block as new iv */
    
    return SSLNoErr;
}

SSLErr
SSLDESFinish(void *cipherRef, SSLContext *ctx)
{   SSLBuffer           desState;
    SSLErr              err;
    
    desState.data = (unsigned char*)cipherRef;
    desState.length = sizeof(struct DESState);
    err = SSLFreeBuffer(&desState, &ctx->sysCtx);
    return err;
}
#endif /* BSAFE */
