/* x509_vrf.c */
/* Copyright (C) 1995 Eric Young (eay@mincom.oz.au).
 * All rights reserved.
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.
 * See the COPYRIGHT file in the SSLeay distribution for more details.
 */

#include <stdio.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "crypto.h"
#include "bn.h"
#include "md2.h"
#include "md5.h"
#include "der.h"
#include "x509.h"
#include "x509_obj.h"
#include "lhash.h"
#include "buffer.h"
#include "pem.h"

#ifdef PROTO
static int verify(X509 *xs, int (*cb)(), int depth);
static int null_callback(int e, X509 *a, X509 *b, int c, int d);
static int time_cmp(char *str);
static int certs_cmp(X509 *a, X509 *b);
static int init_certs(void);
#else
static int verify();
static int null_callback();
static int time_cmp();
static int certs_cmp();
static int init_certs();
#endif

#ifdef DEBUG
void RSA_print_bytes();
#endif

static int num_dirs=0;
static int num_dirs_alloced=0;
static char **dirs=NULL;/**/
static int *dirs_type=NULL;/**/
static LHASH *certs=NULL;
static int init=1;

static int certs_cmp(a, b)
X509 *a;
X509 *b;
	{
	return(X509_name_cmp(a->cert_info->subject,b->cert_info->subject));
	}

static int init_certs()
	{
	certs=lh_new(X509_subject_name_hash,certs_cmp);
	if (certs == NULL)
		{
		X509err(X509_F_INIT_CERTS,ERR_R_MALLOC_FAILURE);
		return(0);
		}
	init=0;
	return(1);
	}

X509 *X509_get_cert(name)
X509_NAME *name;
	{
	static X509 *xs=NULL;
	X509 *x;
	int i,j,k;
	unsigned long h;
	BUFFER *b=NULL;
	struct stat st;

	if (init) if (!init_certs()) return(NULL);
	if (name == NULL) return(NULL);
	if (xs == NULL) xs=(X509 *)X509_new();
	if (xs == NULL)
		{
		X509err(X509_F_X509_GET_CERT,ERR_R_MALLOC_FAILURE);
		return(NULL);
		}
	xs->cert_info->subject=name;
	x=(X509 *)lh_retrieve(certs,(char *)xs);
	if (x != NULL) return(x);

	b=buffer_new();
	if ((b == NULL) || !buffer_grow(b,256))
		{
		X509err(X509_F_X509_GET_CERT,ERR_R_MALLOC_FAILURE);
		return(NULL);
		}
	h=X509_subject_name_hash(xs);
	for (i=0; i<num_dirs; i++)
		{
		j=strlen(dirs[i])+1+8+6+1;
		if (!buffer_grow(b,j))
			{
			X509err(X509_F_X509_GET_CERT,ERR_R_MALLOC_FAILURE);
			return(NULL);
			}
		k=0;
		for (;;)
			{
			sprintf(b->data,"%s/%08lx.%d",dirs[i],h,k);
			k++;
			if (stat(b->data,&st) < 0)
				break;
			/* found one. */
			if ((X509_add_cert_file(b->data,dirs_type[i])) == 0)
				break;
			}
		x=(X509 *)lh_retrieve(certs,(char *)xs);
		if (x != NULL) return(x);
		}
	return(NULL);
	}

int X509_add_cert(x)
X509 *x;
	{
	X509 *r;

	if (init) if (!init_certs()) return(0);
	if (x == NULL) return(0);
	r=(X509 *)lh_insert(certs,(char *)x);
	if (r != NULL)
		{ /* oops, put it back */
		lh_delete(certs,(char *)x);
		lh_insert(certs,(char *)r);
		X509err(X509_F_X509_ADD_CERT,X509_R_CERT_ALREADY_IN_HASH_TABLE);
	/*	X509_free(x); */
		return(0);
		}
	return(1);	
	}

int X509_add_cert_file(file, type)
char *file;
int type;
	{
	FILE *in;
	int i,count=0;
	X509 *x;

	in=fopen(file,"r");
	if (in == NULL)
		{
		SYSerr(ERR_F_FOPEN,errno);
		X509err(X509_F_X509_ADD_CERT_FILE,ERR_R_SYS_LIB);
		return(0);
		}

	if (type == X509_FILETYPE_PEM)
		{
		for (;;)
			{
			x=X509_new();
			if (x == NULL) return(0);
			i=PEM_read_X509(in,x);
			if (!i)
				{

				if ((ERR_peek_error() ==
					PEM_R_NO_START_LINE) && (count > 0))
					{
					ERR_clear_error();
					break;
					}
				else
					{
					X509err(X509_F_X509_ADD_CERT_FILE,
						ERR_R_PEM_LIB);
					X509_free(x);
					return(0);
					}
				}
			i=X509_add_cert(x);
			if (!i) return(0);
			count++;
			}
		fclose(in);
		return(count);
		}
	else if (type == X509_FILETYPE_TEXT)
		{
		x=X509_new();
		if (x == NULL) return(0);
		i=f2i_X509(in,x);
		fclose(in);
		if (!i)
			{
			X509err(X509_F_X509_ADD_CERT_FILE,ERR_R_DER_LIB);
			return(0);
			}
		i=X509_add_cert(x);
		return(i);
		}
	else if (type == X509_FILETYPE_DER)
		{
		x=X509_new();
		if (x == NULL) return(0);
		i=D2i_X509_fp(in,x);
		fclose(in);
		if (!i)
			{
			X509err(X509_F_X509_ADD_CERT_FILE,ERR_R_DER_LIB);
			return(0);
			}
		i=X509_add_cert(x);
		return(i);
		}
	else
		{
		X509err(X509_F_X509_ADD_CERT_FILE,X509_R_BAD_X509_FILETYPE);
		return(0);
		}
	}

int X509_add_cert_dir(dir, type)
char *dir;
int type;
	{
	int j,len;
	int *ip;
	char *s,*ss,*p;
	char **pp;

	if (dir == NULL) return(0);

	s=dir;
	p=s;
	for (;;)
		{
		if ((*p == ':') || (*p == '\0'))
			{
			ss=s;
			s=p+1;
			len=(p-ss);
			if (len == 0) continue;
			for (j=0; j<num_dirs; j++)
				if (strncmp(dirs[j],ss,(unsigned int)len) == 0)
					continue;
			if (num_dirs_alloced < (num_dirs+1))
				{
				num_dirs_alloced+=10;
				pp=(char **)malloc(num_dirs_alloced*
					sizeof(char *));
				ip=(int *)malloc(num_dirs_alloced*
					sizeof(int));
				if ((pp == NULL) || (ip == NULL))
					{
					X509err(X509_F_X509_ADD_CERT_DIR,
						ERR_R_MALLOC_FAILURE);
					return(0);
					}
				memcpy(pp,dirs,(num_dirs_alloced-10)*
					sizeof(char *));
				memcpy(ip,dirs_type,(num_dirs_alloced-10)*
					sizeof(int));
				if (dirs != NULL) free(dirs);
				if (dirs_type != NULL) free(dirs_type);
				dirs=pp;
				dirs_type=ip;
				}
			dirs_type[num_dirs]=type;
			dirs[num_dirs]=(char *)malloc((unsigned int)len+1);
			if (dirs[num_dirs] == NULL) return(0);
			strncpy(dirs[num_dirs],ss,(unsigned int)len);
			dirs[num_dirs][len]='\0';
			num_dirs++;
			}
		if (*p == '\0') break;
		p++;
		}
	return(1);
	}

X509_NAME *X509_get_issuer_name(a)
X509 *a;
	{
	return(a->cert_info->issuer);
	}

X509_NAME *X509_get_subject_name(a)
X509 *a;
	{
	return(a->cert_info->subject);
	}

unsigned long X509_subject_name_hash(x)
X509 *x;
	{
	return X509_name_hash(x->cert_info->subject);
	}

/* I should do a DER encoding of the name and then hash it. */
unsigned long X509_name_hash(x)
X509_NAME *x;
	{
	unsigned long ret=0;
	unsigned char md[16];
	char *str;

	str=X509_oneline_X509_NAME(x);
	if (str == NULL) return(0);
	ret=strlen(str);
	MD5(ret,(unsigned char *)str,&(md[0]));
	free(str);
	ret=(md[0]|(md[1]<<8)|(md[2]<<16)|(md[3]<<24))&0xffffffff;
	return(ret);
	}

#define MD_LEN ((MD2_DIGEST_LENGTH>MD5_DIGEST_LENGTH)\
		?MD2_DIGEST_LENGTH\
		:MD5_DIGEST_LENGTH)

static int null_callback(e,a,b,c,d)
int e;
X509 *a;
X509 *b;
int c;
int d;
	{
#ifdef LINT
	a=b; b=a; c=d; d=c;
#endif
	return(e);
	}

int X509_verify(xs, cb)
X509 *xs;
int (*cb)();
	{
	int tobs=buffer_get_tos();
	int ret;

	if (cb == NULL)
		ret=verify(xs,null_callback,0);
	else	ret=verify(xs,cb,0);

	buffer_set_tos(tobs);
	buffer_clean_up();
	return(ret);
	}

static int verify(xs, cb, depth)
X509 *xs;
int (*cb)();
int depth;
	{
	X509 *xi;
	PEM_CTX ctx;
	BUFFER *buf;
	unsigned char *p;
	RSA *rsa;
	unsigned char *sig;
	int ret,i,alg;
	int btos=buffer_get_tos();

	xi=X509_get_cert(X509_get_issuer_name(xs));
	if (xi == NULL)
		return((*cb)(0,xs,xi,depth,
			VERIFY_ERR_UNABLE_TO_GET_ISSUER));

	rsa=RSA_new();
	if (rsa == NULL) return((*cb)(0,xs,xi,depth, VERIFY_ERR_OUT_OF_MEM));
	p=(unsigned char *)xi->cert_info->key->public_key->data;
	if (!D2i_RSAPublicKey(rsa,&p))
		{
		ret=(*cb)(0,xs,xi,depth,
			VERIFY_ERR_UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY);
		goto err;
		}

	if (!PEM_VerifyInit(&ctx,X509_obj2nid(xs->sig_alg->algorithm)))
		{
		ret=(*cb)(0,xs,xi,depth,VERIFY_ERR_UNKNOWN_X509_SIG_ALGORITHM);
		goto err;
		}

	buf=buffer_get_buf();
	if (buf == NULL)
		{ ret=(*cb)(0,xs,xi,depth,VERIFY_ERR_OUT_OF_MEM); goto err; }

	i=i2D_X509_CINF(xs->cert_info,NULL);
	if (!buffer_grow(buf,i))
		{ ret=(*cb)(0,xs,xi,depth,VERIFY_ERR_OUT_OF_MEM); goto err; }

	p=(unsigned char *)buf->data;
	i2D_X509_CINF(xs->cert_info,&p);
	PEM_VerifyUpdate(&ctx,(unsigned char *)buf->data,i);
	i=PEM_VerifyFinal(&ctx,xs->signature->data,
		(unsigned int)xs->signature->length,rsa);
	if (!i)
		{
		ret=(*cb)(0,xs,xi,depth,VERIFY_ERR_SIGNATURE_FAILURE);
		goto err;
		}

	i=time_cmp(xs->cert_info->validity->notBefore);
	if (i == 0)
		{
		ret=(*cb)(0,xs,xi,depth,VERIFY_ERR_ERROR_IN_NOT_BEFORE_FIELD);
		goto err;
		}
	if (i > 0)
		{
		ret=(*cb)(0,xs,xi,depth,VERIFY_ERR_CERT_NOT_YET_VALID);
		goto err;
		}

	i=time_cmp(xs->cert_info->validity->notAfter);
	if (i == 0)
		{
		ret=((*cb)(0,xs,xi,depth,
			VERIFY_ERR_ERROR_IN_NOT_AFTER_FIELD));
		goto err;
		}

	if (i < 0)
		return((*cb)(0,xs,xi,depth,VERIFY_ERR_CERT_HAS_EXPIRED));
	
	if (X509_name_cmp(xs->cert_info->subject,xs->cert_info->issuer) != 0)
		{
		/* say this one is ok and check the parent */
		i=(*cb)(1,xs,xi,depth,VERIFY_OK);
		if (!i) return(0);
		return(verify(xi,cb,depth+1));
		}
	else
		{
		/* hmm... a self signed cert was passed, same as not
		 * being able to lookup parent - bad. */
		if (depth == 0)
			return((*cb)(0,xs,xi,depth,
				VERIFY_ERR_DEPTH_ZERO_SELF_SIGNED_CERT));
		else
			{
			/* self signed CA, we are happy with this */
			return((*cb)(1,xs,xi,depth,VERIFY_OK));
			}
		}
err:
	buffer_set_tos(btos);
	RSA_free(rsa);
	return(ret);
	}

int X509_name_cmp(a, b)
X509_NAME *a;
X509_NAME *b;
	{
	int i,j;

	if (a->num != b->num)
		return(a->num-b->num);
	for (i=0; i<a->num; i++)
		{
		j=(a->values[i]->length-b->values[i]->length);
		if (j) return(j);
		j=memcmp(a->values[i]->data,b->values[i]->data,
			(unsigned int)a->values[i]->length);
		if (j != 0) return(j);
		}
	for (i=0; i<a->num; i++)
		{
		j=DER_OBJECT_eq(a->objects[i],b->objects[i]);
		if (!j) return(1);
		}
	return(0);
	}

static int time_cmp(str)
char *str;
	{
	time_t offset;
	char buff1[100],buff2[100],*p;
	int i,j;

	p=buff1;
	i=strlen(str);
	if ((i < 11) || (i > 15)) return(0);
	memcpy(p,str,10);
	p+=10;
	str+=10;

	if ((*str == 'Z') || (*str == '-') || (*str == '+'))
		{ *(p++)='0'; *(p++)='0'; }
	else	{ *(p++)= *(str++); *(p++)= *(str++); }
	*(p++)='Z';
	*(p++)='\0';

	if (*str == 'Z')
		offset=0;
	else
		{
		if ((*str != '+') && (str[5] != '-'))
			return(0);
		offset=((str[1]-'0')*10+(str[2]-'0'))*60;
		offset+=(str[3]-'0')*10+(str[4]-'0');
		if (*str == '-')
			offset-=offset;
		}
	X509_gmtime(buff2,offset);

	i=(buff1[0]-'0')*10+(buff1[1]-'0');
	if (i < 70) i+=100;
	j=(buff2[0]-'0')*10+(buff2[1]-'0');
	if (j < 70) j+=100;

	if (i < j) return (-1);
	if (i > j) return (1);
	i=strcmp(buff1,buff2);
	if (i == 0) /* wait a second then return younger :-) */
		return(-1);
	else
		return(i);
	}

char *X509_gmtime(s, adj)
char *s;
long adj;
	{
	time_t t;
	struct tm *ts;

	time(&t);
	t+=adj;
	ts=gmtime(&t);
	sprintf(s,"%02d%02d%02d%02d%02d%02dZ",ts->tm_year%100,
		ts->tm_mon+1,ts->tm_mday,ts->tm_hour,ts->tm_min,ts->tm_sec);
	return(s);
	}

#ifdef DEBUG
void RSA_print_bytes(f,n,b)
FILE *f;
int n;
char *b;
	{
	int i;
	static char *h="0123456789abcdef";

	for (i=0; i<n; i++)
		{
		fputc(h[(b[i]>>4)&0x0f],f);
		fputc(h[(b[i]   )&0x0f],f);
		fputc(' ',f);
		}
	}
#endif

char *X509_verify_error_string(n)
int n;
	{
	static char buf[100];

	switch (n)
		{
	case VERIFY_OK:
		return("ok");
	case VERIFY_ERR_UNABLE_TO_GET_ISSUER:
		return("unable to get issuer certificate");
	case VERIFY_ERR_UNABLE_TO_DECRYPT_SIGNATURE:
		return("unable to decrypt certificate's signature");
	case VERIFY_ERR_UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY:
		return("unable to decode issuer public key");
	case VERIFY_ERR_UNABLE_TO_UNPACK_SIGNATURE:
		return("unable to unpack certificate's signature");
	case VERIFY_ERR_UNKNOWN_X509_SIG_ALGORITHM:
		return("unknown signature algorithm");
	case VERIFY_ERR_SIG_DIGEST_LENGTH_WRONG:
		return("signature digest length has come out wrong");
	case VERIFY_ERR_SIGNATURE_FAILURE:
		return("signature is wrong");
	case VERIFY_ERR_CERT_NOT_YET_VALID:
		return("certificate is not yet valid");
	case VERIFY_ERR_CERT_HAS_EXPIRED:
		return("certificate has expired");
	case VERIFY_ERR_ERROR_IN_NOT_BEFORE_FIELD:
		return("format error in certificate's notBefore field");
	case VERIFY_ERR_ERROR_IN_NOT_AFTER_FIELD:
		return("format error in certificate's notAfter field");
	case VERIFY_ERR_DEPTH_ZERO_SELF_SIGNED_CERT:
		return("self signed certificate");
	default:
		sprintf(buf,"error number %d",n);
		return(buf);
		}
	}
