/* pem.c */
/* Copyright (C) 1995 Eric Young (eay@mincom.oz.au).
 * All rights reserved.
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.
 * See the COPYRIGHT file in the SSLeay distribution for more details.
 */

#include <stdio.h>
#include "crypto.h"
#include "der.h"
#include "x509.h"
#include "pem.h"
#include "buffer.h"
#include "des.h"
#ifndef NOIDEA
#include "idea.h"
#endif
#include "md2.h"
#include "md5.h"

char *PEM_version="\0PEM part of SSLeay v 0.4.4 17/07/95";

#define MIN_LENGTH	4
#define DES_COUNT	1
#define conv_bin2ascii(a)       (data_bin2ascii[(a)&0x3f])
#define conv_ascii2bin(a)       (data_ascii2bin[(a)&0x7f])

/* 64 char lines
 * pad input with 0
 * left over chars are set to =
 * 1 byte  => xx==
 * 2 bytes => xxx=
 * 3 bytes => xxxx
 */
#define BIN_PER_LINE    (64/4*3)
#define CHUNKS_PER_LINE (64/4)
#define CHAR_PER_LINE   (64+1)

#ifdef PROTO
static void PEM_proc_type(char *buf, int type);
static void PEM_dek_info(char *buf, int type, int len, char *str);
static void PEM_make_key_des_md5(int strl, char *str, int count, des_cblock (*iv), des_cblock (*key1), des_cblock (*key2));
#ifdef undef
static void PEM_make_key_des_md2(int strl, char *str, int count, des_cblock (*iv), des_cblock (*key1), des_cblock (*key2));
#endif
static int PEM_des_cbc_encrypt(unsigned char *input, unsigned char *output, long length, des_key_schedule ks, des_cblock (*iv), int enc);
static int PEM_des_ede2_encrypt(unsigned char *input, unsigned char *output, long length, des_key_schedule ks1, des_key_schedule ks2, des_cblock (*iv), int enc);
#ifndef NOIDEA
static int PEM_idea_cbc_encrypt(unsigned char *input, unsigned char *output, long length, IDEA_KEY_SCHEDULE *ks, unsigned char *iv, int enc);
#endif
static int def_callback(char *buf, int num, int w);
static int load_iv(unsigned char **fromp,unsigned char *to, int num);
static int DEK_str2num(char **str);
static int get_key_bytes(unsigned char *buf, unsigned char *iv, int num);
#else
static void PEM_proc_type();
static void PEM_dek_info();
static void PEM_make_key_des_md5();
#ifdef undef
static void PEM_make_key_des_md2();
#endif
static int PEM_des_cbc_encrypt();
static int PEM_des_ede2_encrypt();
#ifndef NOIDEA
static int PEM_idea_cbc_encrypt();
#endif
static int def_callback();
static int load_iv();
static int DEK_str2num();
static int get_key_bytes();
#endif

static int (*callback)()=def_callback;
static unsigned char data_bin2ascii[65]="ABCDEFGHIJKLMNOPQRSTUVWXYZ\
abcdefghijklmnopqrstuvwxyz0123456789+/";
static unsigned char data_ascii2bin[128]={
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
	0xFF,0xFF,0xFF,0x3E,0xFF,0xFF,0xFF,0x3F,
	0x34,0x35,0x36,0x37,0x38,0x39,0x3A,0x3B,
	0x3C,0x3D,0xFF,0xFF,0xFF,0x00,0xFF,0xFF,
	0xFF,0x00,0x01,0x02,0x03,0x04,0x05,0x06,
	0x07,0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,
	0x0F,0x10,0x11,0x12,0x13,0x14,0x15,0x16,
	0x17,0x18,0x19,0xFF,0xFF,0xFF,0xFF,0xFF,
	0xFF,0x1A,0x1B,0x1C,0x1D,0x1E,0x1F,0x20,
	0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,
	0x29,0x2A,0x2B,0x2C,0x2D,0x2E,0x2F,0x30,
	0x31,0x32,0x33,0xFF,0xFF,0xFF,0xFF,0xFF,
	};

static int def_callback(buf, num, w)
char *buf;
int num;
int w;
	{
	int i,j;

#ifdef _Windows
	/* We should not ever call the default callback routine from
	 * windows. */
	PEMerr(PEM_F_DEF_CALLBACK,ERR_R_SHOULD_NOT_HAVE_BEEN_CALLED);
	return(-1);
#else

	for (;;)
		{
		i=des_read_pw_string(buf,num,"Enter password:",w);
		if (i != 0)
			{
			memset(buf,0,(unsigned int)num);
			return(-1);
			}
		j=strlen(buf);
		if (j < MIN_LENGTH)
			{
			fprintf(stderr,"password too short, needs to be at least %d chars\n",MIN_LENGTH);
			}
		else
			break;
		}
	return(j);
#endif
	}

void PEM_set_getkey_callback(a)
int (*a)();
	{
	callback=a;
	}

static void PEM_proc_type(buf, type)
char *buf;
int type;
	{
	char *str;

	if (type == PEM_TYPE_ENCRYPTED)
		str="ENCRYPTED";
	else if (type == PEM_TYPE_ENCRYPTED)
		str="MIC-ONLY";
	else if (type == PEM_TYPE_MIC_ONLY)
		str="MIC-CLEAR";
	else
		str="BAD-TYPE";
		
	strcat(buf,"Proc-Type: 4,");
	strcat(buf,str);
	strcat(buf,"\n");
	}

static void PEM_dek_info(buf, type, len, str)
char *buf;
int type;
int len;
char *str;
	{
	static unsigned char map[17]="0123456789ABCDEF";
	char *s;
	long i;
	int j;

	switch (type)
		{
	case PEM_DEK_DES_CBC:
		s=PEM_STRING_DES_CBC; break;
	case PEM_DEK_IDEA_CBC:
		s=PEM_STRING_IDEA_CBC; break;
	case PEM_DEK_DES_EDE:
		s=PEM_STRING_DES_EDE; break;
	case PEM_DEK_DES_ECB:
		s=PEM_STRING_DES_ECB; break;
	case PEM_DEK_RSA:
		s=PEM_STRING_RSA; break;
	case PEM_DEK_RSA_MD2:
		s=PEM_STRING_RSA_MD2; break;
	case PEM_DEK_RSA_MD5:
		s=PEM_STRING_RSA_MD5; break;
	default:
		s=PEM_STRING_BAD_DAK; break;
		}
	strcat(buf,"DEK-Info: ");
	strcat(buf,s);
	strcat(buf,",");
	j=strlen(buf);
	for (i=0; i<len; i++)
		{
		buf[j+i*2]  =map[(str[i]>>4)&0x0f];
		buf[j+i*2+1]=map[(str[i]   )&0x0f];
		}
	buf[j+i*2]='\n';
	buf[j+i*2+1]='\0';
	}

int PEM_write_X509(fp, x)
FILE *fp;
X509 *x;
	{
	int i;
	unsigned char *p,*buf;

	i=i2D_X509(x,NULL);
	if (i <= 0)
		{
		PEMerr(PEM_F_PEM_WRITE_X509,ERR_R_X509_LIB);
		return(0);
		}
	buf=(unsigned char *)malloc((unsigned int)i);
	if (buf == NULL)
		{
		PEMerr(PEM_F_PEM_WRITE_X509,ERR_R_MALLOC_FAILURE);
		return(0);
		}
	p=buf;
	i=i2D_X509(x,&p);
	i=PEM_write(fp,"X509 CERTIFICATE","",i,buf);
	free(buf);
	return(i);
	}

int PEM_write_X509_REQ(fp, x)
FILE *fp;
X509_REQ *x;
	{
	int i;
	unsigned char *p,*buf;

	i=i2D_X509_REQ(x,NULL);
	if (i <= 0)
		{
		PEMerr(PEM_F_PEM_WRITE_X509_REQ,ERR_R_X509_LIB);
		return(0);
		}
	buf=(unsigned char *)malloc((unsigned int)i);
	if (buf == NULL)
		{
		PEMerr(PEM_F_PEM_WRITE_X509_REQ,ERR_R_MALLOC_FAILURE);
		return(0);
		}
	p=buf;
	i=i2D_X509_REQ(x,&p);
	i=PEM_write(fp,"X509 CERTIFICATE REQUEST","",i,buf);
	free(buf);
	return(i);
	}

int PEM_read_X509(fp, x)
FILE *fp;
X509 *x;
	{
	int i;
	char *name,*header;
	unsigned char *data;
	long len;

	if (x == NULL)
		{
		x=X509_new();
		if (x == NULL)
			{
			PEMerr(PEM_F_PEM_READ_X509,ERR_R_X509_LIB);
			return(0);
			}
		}

	for (;;)
		{
		if (!PEM_read(fp,&name,&header,&len,&data))
			return(0);
		if (strcmp(name,"X509 CERTIFICATE") == 0) break;
		free(name);
		free(header);
		free(data);
		}
	if (!PEM_do_header(header,len,data)) return(0);
	i=D2i_X509(x,len,data);
	free(name);
	free(header);
	free(data);
	if (i <= 0)
		{
		PEMerr(ERR_R_X509_LIB,ERR_R_DER_LIB);
		return(0);
		}
	return(1);
	}

int PEM_write_X509_CRL(fp, x)
FILE *fp;
X509_CRL *x;
	{
	int i;
	unsigned char *p,*buf;

	i=i2D_X509_CRL(x,NULL);
	if (i <= 0)
		{
		PEMerr(PEM_F_PEM_WRITE_CRL,ERR_R_X509_LIB);
		return(0);
		}
	buf=(unsigned char *)malloc((unsigned int)i);
	if (buf == NULL)
		{
		PEMerr(PEM_F_PEM_WRITE_CRL,ERR_R_MALLOC_FAILURE);
		return(0);
		}
	p=buf;
	i=i2D_X509_CRL(x,&p);
	i=PEM_write(fp,"X509 CRL","",i,buf);
	free(buf);
	return(i);
	}

int PEM_read_X509_CRL(fp, x)
FILE *fp;
X509_CRL *x;
	{
	int i;
	char *name,*header;
	unsigned char *data;
	long len;

	if (x == NULL)
		{
		x=X509_CRL_new();
		if (x == NULL)
			{
			PEMerr(PEM_F_PEM_READ_CRL,ERR_R_X509_LIB);
			return(0);
			}
		}

	for (;;)
		{
		if (!PEM_read(fp,&name,&header,&len,&data))
			return(0);
		if (strcmp(name,"X509 CRL") == 0) break;
		free(name);
		free(header);
		free(data);
		}
	if (!PEM_do_header(header,len,data)) return(0);
	i=D2i_X509_CRL(x,len,data);
	free(name);
	free(header);
	free(data);
	if (i <= 0)
		{
		PEMerr(PEM_F_PEM_READ_CRL,ERR_R_DER_LIB);
		return(0);
		}
	return(1);
	}

int PEM_read_RSA(fp, x)
FILE *fp;
RSA *x;
	{
	int i;
	char *name,*header;
	unsigned char *data;
	unsigned char *p;
	long len;

	if (x == NULL)
		{
		x=RSA_new();
		if (x == NULL)
			{
			PEMerr(PEM_F_PEM_READ_RSA,ERR_R_RSA_LIB);
			return(0);
			}
		}

	for (;;)
		{
		if (!PEM_read(fp,&name,&header,&len,&data))
			return(0);
		if (strcmp(name,"RSA PRIVATE KEY") == 0) break;
		free(name);
		free(header);
		free(data);
		}
	if (!PEM_do_header(header,len,data)) return(0);
	p=(unsigned char *)data;
	i=D2i_RSAPrivateKey(x,&p);
	free(name);
	free(header);
	free(data);
	if (i <= 0)
		{
		PEMerr(PEM_F_PEM_READ_RSA,ERR_R_DER_LIB);
		return(0);
		}
	return(1);
	}

int PEM_write_RSA(fp, r, enc, klen, kstr)
FILE *fp;
RSA *r;
int enc;
int klen;
char *kstr;
	{
	unsigned char *str;
	unsigned char *p;
	int size=0,i;
	char buf[1024],buff[512],*pwd,salt[8];
	C_Block key[2],iv;
	des_key_schedule ks;
#ifndef NOIDEA
	IDEA_KEY_SCHEDULE iks;
#endif
	
	size=i2D_RSAPrivateKey(r,NULL);
	if (size <= 0)
		{ PEMerr(PEM_F_PEM_WRITE_RSA,ERR_R_X509_LIB); return(0); }
	str=(unsigned char *)malloc((unsigned int)size+8);
	if (str == NULL) return(0);
	p=str;
	i=i2D_RSAPrivateKey(r,&p);
	if (!i) { PEMerr(PEM_F_PEM_WRITE_RSA,ERR_R_X509_LIB); return(0); }

	if (enc && (kstr == NULL))
		{
		klen=(*callback)(buf,1024,1);
		if (klen <= 0)
			{
			memset(buf,0,1024);
			memset(str,0,(unsigned int)size);
			free(str);
			PEMerr(PEM_F_PEM_WRITE_RSA,PEM_R_READ_KEY);
			return(0);
			}
		pwd=buf;
		}
	else
		pwd=kstr;

#ifndef NOIDEA
	if (enc == PEM_DEK_IDEA_CBC)
		{
		MD5_rand_seed(size,str); /**/
		MD5_rand(8,iv);
		PEM_make_key_des_md5(klen,pwd,DES_COUNT,
			(des_cblock *)&(iv[0]),
			(des_cblock *)&(key[0][0]),(des_cblock *)&(key[1][0]));
		if (pwd == buf) memset(buf,0,1024);
		idea_encrypt_key(&(key[0][0]),&iks);

		buff[0]='\0';
		PEM_proc_type(buff,PEM_TYPE_ENCRYPTED);
		PEM_dek_info(buff,PEM_DEK_IDEA_CBC,8,(char *)iv);
		i=PEM_idea_cbc_encrypt(str,str,size,&iks,iv,1);
		}
	else
#endif
	      if (enc == PEM_DEK_DES_CBC)
		{
		MD5_rand_seed(size,str); /**/
		MD5_rand(8,iv);
		PEM_make_key_des_md5(klen,pwd,DES_COUNT,
			(des_cblock *)&(iv[0]),
			(des_cblock *)&(key[0][0]),NULL);
		if (pwd == buf) memset(buf,0,1024);
		set_key((des_cblock *)&(key[0][0]),ks);

		buff[0]='\0';
		PEM_proc_type(buff,PEM_TYPE_ENCRYPTED);
		PEM_dek_info(buff,PEM_DEK_DES_CBC,8,(char *)iv);
	
		i=PEM_des_cbc_encrypt(str,str,
			size,ks,(C_Block *)&(iv[0]),DES_ENCRYPT);
		}
	else
		buff[0]='\0';
	i=PEM_write(fp,"RSA PRIVATE KEY",buff,i,str);

	if (str != NULL) memset(str,0,(unsigned int)size);
	free(str);
	memset(salt,0,8);
	memset(key,0,sizeof(key));
	memset(ks,0,sizeof(ks));
#ifndef NOIDEA
	memset(&iks,0,sizeof(iks));
#endif
	memset(iv,0,8);
	return(i);
	}

int PEM_do_header(header, len, data)
char *header;
int len;
unsigned char *data;
	{
	int i;
	C_Block key[2],iv[2];
	des_key_schedule ks[2];
#ifndef NOIDEA
	IDEA_KEY_SCHEDULE iks,iks2;
#endif

	if ((header == NULL) || (*header == '\0') || (*header == '\n'))
		return(1);
	if (strncmp(header,"Proc-Type: ",11) != 0)
		{
		PEMerr(PEM_F_PEM_DO_HEADER,PEM_R_NOT_PROC_TYPE);
		return(0);
		}
	header+=11;
	if (*header != '4') return(0); header++;
	if (*header != ',') return(0); header++;
	if (strncmp(header,"ENCRYPTED",9) != 0)
		{
		PEMerr(PEM_F_PEM_DO_HEADER,PEM_R_NOT_ENCRYPTED);
		return(0);
		}
	for (; (*header != '\n') && (*header != '\0'); header++)
		;
	if (*header == '\0')
		{
		PEMerr(PEM_F_PEM_DO_HEADER,PEM_R_SHORT_HEADER);
		return(0);
		}
	header++;
	if (strncmp(header,"DEK-Info: ",10) != 0)
		{
		PEMerr(PEM_F_PEM_DO_HEADER,PEM_R_NOT_DEK_INFO);
		return(0);
		}
	header+=10;

	i=DEK_str2num(&header);
	header++;
	switch (i)
		{
	case PEM_DEK_DES_CBC:
		if (!load_iv((unsigned char **)&header,&(iv[0][0]),8))
			return(0);
		if (!get_key_bytes(&(key[0][0]),&(iv[0][0]),8)) return(0);
		set_key((des_cblock *)&(key[0][0]),ks[0]);
		i=PEM_des_cbc_encrypt(data,data,
			(long)len,ks[0],(C_Block *)&(iv[0][0]),DES_DECRYPT);
		break;
	case PEM_DEK_DES_EDE:
		if (!load_iv((unsigned char **)&header,&(iv[0][0]),8))
			return(0);
		if (!get_key_bytes(&(key[0][0]),&(iv[0][0]),16)) return(0);
		set_key((des_cblock *)&(key[0][0]),ks[0]);
		set_key((des_cblock *)&(key[1][0]),ks[1]);
		i=PEM_des_ede2_encrypt(data,data,
			(long)len,ks[0],ks[1],(C_Block *)&(iv[0][0]),
			DES_DECRYPT);
		break;
#ifndef NOIDEA
	case PEM_DEK_IDEA_CBC:
		if (!load_iv((unsigned char **)&header,&(iv[0][0]),8))
			return(0);
		if (!get_key_bytes(&(key[0][0]),&(iv[0][0]),16)) return(0);
		idea_encrypt_key(&(key[0][0]),&iks);
		idea_decrypt_key(&iks,&iks2);
		i=PEM_idea_cbc_encrypt(data,data,
			(long)len,&iks2,(unsigned char *)&(iv[0][0]),0);
		break;
#endif
	default:
		PEMerr(PEM_F_PEM_DO_HEADER,PEM_R_UNSUPPORTED_ENCRYPTION);
		return(0);
		}
	memset(key,0,sizeof(key));
	memset(iv,0,sizeof(iv));
	memset(ks,0,sizeof(ks));
#ifndef NOIDEA
	memset(&iks,0,sizeof(iks));
	memset(&iks2,0,sizeof(iks2));
#endif
	if (i == -1)
		{
		PEMerr(PEM_F_PEM_DO_HEADER,PEM_R_BAD_DECRYPT);
		return(0);
		}
	return(1);
	}

static int get_key_bytes(key,iv,num)
unsigned char *key;
unsigned char *iv;
int num;
	{
	char buf[1024];
	unsigned char *k1=NULL,*k2=NULL;

	if ((*callback)(buf,1024,0) <= 0)
		{
		PEMerr(PEM_F_GET_KEY_BYTES,PEM_R_BAD_READ_PW_STRING);
		return(0);
		}

	if (num >= 8)	k1=key;
	if (num >= 16)	k2= &(key[8]);
	PEM_make_key_des_md5((int)strlen(buf),buf,DES_COUNT,
		(C_Block *)&(iv[0]),
		(C_Block *)k1,(C_Block *)k2);
	memset(buf,0,1024);
	return(1);
	}

static int DEK_str2num(str)
char **str;
	{
	char *s;
	int ret=0;

	s= *str;
	if (strncmp(s,PEM_STRING_DES_CBC,PEM_STRING_DES_CBC_LEN) == 0)
		{
		s+=PEM_STRING_DES_CBC_LEN;
		ret=PEM_DEK_DES_CBC;
		}
	else if (strncmp(s,PEM_STRING_IDEA_CBC,PEM_STRING_IDEA_CBC_LEN) == 0)
		{
		s+=PEM_STRING_IDEA_CBC_LEN;
		ret=PEM_DEK_IDEA_CBC;
		}
	else if (strncmp(s,PEM_STRING_RSA,PEM_STRING_RSA_LEN) == 0)
		{
		s+=PEM_STRING_RSA_LEN;
		ret=PEM_DEK_RSA;
		}
	else if (strncmp(s,PEM_STRING_RSA_MD2,PEM_STRING_RSA_MD2_LEN) == 0)
		{
		s+=PEM_STRING_RSA_MD2_LEN;
		ret=PEM_DEK_RSA_MD2;
		}
	else if (strncmp(s,PEM_STRING_RSA_MD5,PEM_STRING_RSA_MD5_LEN) == 0)
		{
		s+=PEM_STRING_RSA_MD5_LEN;
		ret=PEM_DEK_RSA_MD5;
		}
	else if (strncmp(s,PEM_STRING_DES_ECB,PEM_STRING_DES_ECB_LEN) == 0)
		{
		s+=PEM_STRING_DES_ECB_LEN;
		ret=PEM_DEK_DES_ECB;
		}
	else if (strncmp(s,PEM_STRING_DES_EDE,PEM_STRING_DES_EDE_LEN) == 0)
		{
		s+=PEM_STRING_DES_EDE_LEN;
		ret=PEM_DEK_DES_EDE;
		}
	else	ret=PEM_ERROR;

	*str=s;
	return(ret);
	}

static int load_iv(fromp,to,num)
unsigned char **fromp,*to;
int num;
	{
	int v,i;
	unsigned char *from;

	from= *fromp;
	for (i=0; i<num; i++) to[i]=0;
	num*=2;
	for (i=0; i<num; i++)
		{
		if ((*from >= '0') && (*from <= '9'))
			v= *from-'0';
		else if ((*from >= 'A') && (*from <= 'F'))
			v= *from-'A'+10;
		else if ((*from >= 'a') && (*from <= 'f'))
			v= *from-'a'+10;
		else
			{
			PEMerr(PEM_F_LOAD_IV,PEM_R_BAD_IV_CHARS);
			return(0);
			}
		from++;
		to[i/2]|=v<<(long)((!(i&1))*4);
		}

	*fromp=from;
	return(1);
	}

int PEM_bin2ascii(len, f, t)
int len;
unsigned char *f;
unsigned char *t;
	{
	int r,i,j,chunks;
	unsigned long l;

	i=len/BIN_PER_LINE;
	j=len%BIN_PER_LINE;
	if (j != 0) j=((j+2)/3)*4+1;
	r=(i*CHAR_PER_LINE+j);
	if (t == NULL) return(r);

	chunks=0;
	for (;;)
		{
		if (len >= 3)
			{
			l=	((unsigned long)f[0]<<16L)|
				((unsigned long)f[1]<< 8L)|f[2];
			*(t++)=conv_bin2ascii(l>>18L);
			*(t++)=conv_bin2ascii(l>>12L);
			*(t++)=conv_bin2ascii(l>> 6L);
			*(t++)=conv_bin2ascii(l    );
			chunks++;
			f+=3;
			}
		else
			{
			l=0;
			l=(f[0]<<16L);
			if (len == 2) l|=(f[1]<<8L);

			*(t++)=conv_bin2ascii(l>>18L);
			*(t++)=conv_bin2ascii(l>>12L);
			*(t++)=(len == 1)?'=':conv_bin2ascii(l>> 6L);
			*(t++)='=';
			chunks++;
			}
		if (chunks == CHUNKS_PER_LINE)
			{
			*(t++)='\n';
			chunks=0;
			}
		if (len <= 3) break;
		len-=3;
		}
	if (chunks != 0) *(t++)='\n';
	*(t++)='\0';
	return(r);
	}

int PEM_ascii2bin(f, t)
unsigned char *f;
unsigned char *t;
	{
	int i,j,chunks,err,c,sub;
	unsigned long l,len;

	err=len=0;
	chunks=0;
	for (;;)
		{
		i=0;
		sub=0;
		l=0;
		for (i=0; i<4; i++) /* get 4 chars */
			{
			c= *(f++);
			j=conv_ascii2bin(c);
			if (j == 0xFF)
				{
				err=1;
				PEMerr(PEM_F_PEM_ASCII2BIN,
					PEM_R_BASE64_BAD_INPUT_CHAR);
				break;
				}
			if (c == '=') sub++;
			l<<=6L;
			l|=j;
			}
		if (err) break;
		if (++chunks > CHUNKS_PER_LINE)
			{
			err=1;
			PEMerr(PEM_F_PEM_ASCII2BIN,
				PEM_R_BASE64_LINE_TOO_LONG);
			break;
			}
		if (t != NULL)
			{
			*(t++)=(unsigned char)(l>>16L)&0xff;
			*(t++)=(unsigned char)(l>> 8L)&0xff;
			*(t++)=(unsigned char)(l     )&0xff;
			len+=3;
			}
		if (sub) break;
		if (*f == '\n')
			{
			if (chunks == CHUNKS_PER_LINE)
				{
				f++;
				chunks=0;
				}
			else	break;
			}
		if (*f == '\0') break;
		}
	if (err) return(0);
	return(len-sub);
	}

int PEM_write(fp, name, header, len, data)
FILE *fp;
char *name;
char *header;
long len;
unsigned char *data;
	{
	int i;
	unsigned char *buf;
	
	Fprintf(fp,"-----BEGIN %s-----\n",name);
	Fwrite(header,strlen(header),1,fp);
	Fprintf(fp,"\n");

	i=PEM_bin2ascii((int)len,data,(unsigned char *)NULL);
	buf=(unsigned char *)malloc((unsigned int)i+1);
	if (buf == NULL) return(0);
	i=PEM_bin2ascii((int)len,data,buf);
	i=Fwrite(buf,(unsigned int)i,1,fp);
	free(buf);
	Fprintf(fp,"-----END %s-----\n",name);
	return(i);
	}

int PEM_read(fp, name, header, len, data)
FILE *fp;
char **name;
char **header;
long *len;
unsigned char **data;
	{
	int end=0,i,j,hl,nohead=0;
	char buf[256];
	BUFFER *nameB;
	BUFFER *headerB;
	BUFFER *dataB,*tmpB;
	
	nameB=buffer_new();
	headerB=buffer_new();
	dataB=buffer_new();
	if ((nameB == NULL) || (headerB == NULL) || (dataB == NULL))
		{
		PEMerr(PEM_F_PEM_READ,ERR_R_MALLOC_FAILURE);
		return(0);
		}

	for (;;)
		{
		buf[0]='\0';
		Fgets(buf,256,fp);
		if (buf[0] == '\0')
			{
			PEMerr(PEM_F_PEM_READ,PEM_R_NO_START_LINE);
			goto err;
			}
		if (strncmp(buf,"-----BEGIN ",11) == 0)
			{
			i=strlen(&(buf[11]));
			if (strncmp(&(buf[11+i-6]),"-----\n",
				(unsigned int)i-6) != 0)
				continue;
			if (!buffer_grow(nameB,i+9))
				{
				PEMerr(PEM_F_PEM_READ,ERR_R_MALLOC_FAILURE);
				goto err;
				}
			strncpy(nameB->data,&(buf[11]),(unsigned int)i-6);
			nameB->data[i-6]='\0';
			break;
			}
		}
	hl=0;
	if (!buffer_grow(headerB,256))
		{ PEMerr(PEM_F_PEM_READ,ERR_R_MALLOC_FAILURE); goto err; }
	headerB->data[0]='\0';
	for (;;)
		{
		buf[0]='\0';
		Fgets(buf,256,fp);
		if (buf[0] == '\0') break;
		if (buf[0] == '\n') break;
		i=strlen(buf);
		if (!buffer_grow(headerB,hl+i+9))
			{ PEMerr(PEM_F_PEM_READ,ERR_R_MALLOC_FAILURE); goto err; }
		if (strncmp(buf,"-----END ",9) == 0)
			{
			nohead=1;
			break;
			}
		strncpy(&(headerB->data[hl]),buf,(unsigned int)i);
		headerB->data[hl+i]='\0';
		hl+=i;
		}

	j=0;
	if (!buffer_grow(dataB,1024))
		{ PEMerr(PEM_F_PEM_READ,ERR_R_MALLOC_FAILURE); goto err; }
	dataB->data[0]='\0';
	if (!nohead)
		{
		for (;;)
			{
			buf[0]='\0';
			Fgets(buf,256,fp);
			if (buf[0] == '\0') break;
			i=strlen(buf);
			if (i != 65) end=1;
			if (strncmp(buf,"-----END ",9) == 0)
				break;
			if (i > 65) break;
			if (!buffer_grow(dataB,i+j+9))
				{
				PEMerr(PEM_F_PEM_READ,ERR_R_MALLOC_FAILURE);
				goto err;
				}
			strncpy(&(dataB->data[j]),buf,(unsigned int)i);
			dataB->data[j+i]='\0';
			j+=i;
			if (end)
				{
				buf[0]='\0';
				Fgets(buf,256,fp);
				break;
				}
			}
		}
	else
		{
		tmpB=headerB;
		headerB=dataB;
		dataB=tmpB;
		}
	i=strlen(nameB->data);
	if (	(strncmp(buf,"-----END ",9) != 0) ||
		(strncmp(nameB->data,&(buf[9]),(unsigned int)i) != 0) ||
		(strncmp(&(buf[9+i]),"-----\n",6) != 0))
		{
		PEMerr(PEM_F_PEM_READ,PEM_R_BAD_END_LINE);
		goto err;
		}

	i=PEM_ascii2bin((unsigned char *)dataB->data,
		(unsigned char *)dataB->data);
	dataB->data[i]='\0';

	if (i == 0) goto err;
	*name=nameB->data;
	*header=headerB->data;
	*data=(unsigned char *)dataB->data;
	*len=i;
	free(nameB);
	free(headerB);
	free(dataB);
	return(1);
err:
	buffer_free(nameB);
	buffer_free(headerB);
	buffer_free(dataB);
	return(0);
	}

/* pkcs-5 pbeWithMD5AndDESCBC { 1 2 840 113549 1 5 1 } */
static void PEM_make_key_des_md5(strl, str, count, iv, key1,key2)
int strl;
char *str;
int count;
des_cblock (*iv);
des_cblock (*key1);
des_cblock (*key2);
	{
	MD5_CTX c;
	unsigned char md[MD5_DIGEST_LENGTH];
	int i;

	MD5Init(&c);
	MD5Update(&c,(unsigned char *)str,(unsigned int)strl);
	MD5Update(&c,(unsigned char *)iv,8);
	MD5Final(&(md[0]),&c);
	for (i=1; i<count; i++)
		MD5(MD5_DIGEST_LENGTH,&(md[0]),&(md[0]));
	if (key1 != NULL)
		{
		memcpy(key1,&(md[0]),DES_KEY_SZ);
		des_set_odd_parity(key1);
		}
	if (key2 != NULL)
		{
		memcpy(key2,&(md[8]),DES_KEY_SZ);
		des_set_odd_parity(key2);
		}
	memset(&(md[0]),0,MD5_DIGEST_LENGTH);
	}

#ifdef undef
/* pkcs-5 pbeWithMD2AndDESCBC { 1 2 840 113549 1 5 3 } */
static void PEM_make_key_des_md2(strl, str, count, iv, key1, key2)
int strl;
char *str;
int count;
des_cblock (*iv);
des_cblock (*key1);
des_cblock (*key2);
	{
	MD2_CTX c;
	unsigned char md[MD2_DIGEST_LENGTH];
	int i;

	MD2Init(&c);
	MD2Update(&c,(unsigned char *)str,(unsigned int)strl);
	MD2Update(&c,(unsigned char *)iv,8);
	MD2Final(&(md[0]),&c);
	for (i=1; i<count; i++)
		MD2(MD5_DIGEST_LENGTH,&(md[0]),&(md[0]));
	if (key1 != NULL)
		{
		memcpy(key1,&(md[0]),DES_KEY_SZ);
		des_set_odd_parity(key1);
		}
	if (key2 != NULL)
		{
		memcpy(key2,&(md[8]),DES_KEY_SZ);
		des_set_odd_parity(key2);
		}
	memset(&(md[0]),0,MD2_DIGEST_LENGTH);
	}
#endif

/* must have room for padding of upto 8 bytes */
static int PEM_des_cbc_encrypt(input, output, length, ks, iv, enc)
unsigned char *input;
unsigned char *output;
long length;
des_key_schedule ks;
des_cblock (*iv);
int enc;
	{
	int i,v;

	if (enc)
		{
		v=8-length%8;
		for (i=0; i < v; i++)
			input[length+i]=(unsigned char)v;
		des_cbc_encrypt((des_cblock*)input,(des_cblock *)output,
			length+v,ks,iv,enc);
		return(length+v);
		}
	else
		{
		des_cbc_encrypt((des_cblock *)input,(des_cblock *)output,
			length,ks,iv,enc);
		v=output[length-1];
		if (v > 8) return(-1);
		for (i=0; i<v; i++)
			if (output[length-1-i] != v)
				{
				PEMerr(PEM_F_PEM_DES_CBC_ENCRYPT,
					PEM_R_BAD_DECRYPT);
				return(-1);
				}
		return(length-v);
		}
	}

/* must have room for padding of upto 8 bytes */
static int PEM_des_ede2_encrypt(input, output, length, ks1, ks2, iv, enc)
unsigned char *input;
unsigned char *output;
long length;
des_key_schedule ks1,ks2;
des_cblock (*iv);
int enc;
	{
	int i,v;

	if (enc)
		{
		v=8-length%8;
		for (i=0; i < v; i++)
			input[length+i]=(unsigned char)v;
		des_ede2_cbc_encrypt((des_cblock*)input,(des_cblock *)output,
			length+v,ks1,ks2,iv,enc);
		return(length+v);
		}
	else
		{
		des_ede2_cbc_encrypt((des_cblock *)input,(des_cblock *)output,
			length,ks1,ks2,iv,enc);
		v=output[length-1];
		if (v > 8) return(-1);
		for (i=0; i<v; i++)
			if (output[length-1-i] != v)
				{
				PEMerr(PEM_F_PEM_DES_EDE2_ENCRYPT,
					PEM_R_BAD_DECRYPT);
				return(-1);
				}
		return(length-v);
		}
	}

#ifndef NOIDEA
/* must have room for padding of upto 8 bytes */
static int PEM_idea_cbc_encrypt(input, output, length, ks, iv, enc)
unsigned char *input;
unsigned char *output;
long length;
IDEA_KEY_SCHEDULE *ks;
unsigned char *iv;
int enc;
	{
	int i,v;

	if (enc)
		{
		v=8-length%8;
		for (i=0; i < v; i++)
			input[length+i]=(unsigned char)v;
		idea_cbc_encrypt(input,output,
			(unsigned int)length+v,ks,iv,enc);
		return(length+v);
		}
	else
		{
		idea_cbc_encrypt(input,output,
			(unsigned int)length,ks,iv,enc);
		v=output[length-1];
		if (v > 8) return(-1);
		for (i=0; i<v; i++)
			if (output[length-1-i] != v)
				{
				PEMerr(PEM_F_PEM_IDEA_CBC_ENCRYPT,
					PEM_R_BAD_DECRYPT);
				return(-1);
				}
		return(length-v);
		}
	}

#endif
